"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateLoggingConfiguration = validateLoggingConfiguration;
exports.validateCloudWatchLogDestination = validateCloudWatchLogDestination;
exports.validateLoggingFilter = validateLoggingFilter;
exports.validateLoggingDestination = validateLoggingDestination;
exports.validateQueryLoggingConfiguration = validateQueryLoggingConfiguration;
exports.validateLimitsPerLabelSetEntry = validateLimitsPerLabelSetEntry;
exports.validateLabel = validateLabel;
exports.validateLimitsPerLabelSet = validateLimitsPerLabelSet;
exports.validateWorkspaceConfiguration = validateWorkspaceConfiguration;
exports.validateAlertManagerDefinition = validateAlertManagerDefinition;
exports.validateAlias = validateAlias;
const utils_1 = require("@robhan-cdk-lib/utils");
function validateLoggingConfiguration(loggingConfiguration) {
    const errors = [];
    if (typeof loggingConfiguration !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const loggingConfigurationObject = loggingConfiguration;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: loggingConfigurationObject,
        mandatoryAttributes: [],
        optionalAttributes: ['logGroup'],
    }));
    return errors;
}
function validateCloudWatchLogDestination(cloudWatchLogDestination) {
    const errors = [];
    if (typeof cloudWatchLogDestination !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const cloudWatchLogDestinationObject = cloudWatchLogDestination;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: cloudWatchLogDestinationObject,
        mandatoryAttributes: ['logGroup'],
        optionalAttributes: [],
    }));
    return errors;
}
function validateLoggingFilter(loggingFilter) {
    const errors = [];
    if (typeof loggingFilter !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const loggingFilterObject = loggingFilter;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: loggingFilterObject,
        mandatoryAttributes: ['qspThreshold'],
        optionalAttributes: [],
    }));
    if (loggingFilterObject.qspThreshold !== undefined) {
        if (typeof loggingFilterObject.qspThreshold !== 'number') {
            errors.push('qspThreshold: must be a number');
        }
        else {
            errors.push(...(0, utils_1.validateNumberRange)({
                value: loggingFilterObject.qspThreshold,
                min: 0,
                messagePrefix: 'qspThreshold: ',
            }));
        }
    }
    return errors;
}
function validateLoggingDestination(loggingDestination) {
    const errors = [];
    if (typeof loggingDestination !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const loggingDestinationObject = loggingDestination;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: loggingDestinationObject,
        mandatoryAttributes: ['cloudWatchLogs', 'filters'],
        optionalAttributes: [],
    }));
    if (loggingDestinationObject.cloudWatchLogs !== undefined) {
        errors.push(...validateCloudWatchLogDestination(loggingDestinationObject.cloudWatchLogs).map((err) => `cloudWatchLogs: ${err}`));
    }
    if (loggingDestinationObject.filters !== undefined) {
        errors.push(...validateLoggingFilter(loggingDestinationObject.filters).map((err) => `filters: ${err}`));
    }
    return errors;
}
function validateQueryLoggingConfiguration(queryLoggingConfiguration) {
    const errors = [];
    if (typeof queryLoggingConfiguration !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const queryLoggingConfigurationObject = queryLoggingConfiguration;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: queryLoggingConfigurationObject,
        mandatoryAttributes: ['destinations'],
        optionalAttributes: [],
    }));
    if (queryLoggingConfigurationObject.destinations !== undefined) {
        errors.push(...queryLoggingConfigurationObject.destinations.map((d, i) => {
            return validateLoggingDestination(d).map((err) => `destinations[${i}]: ${err}`);
        }).flat());
    }
    return errors;
}
function validateLimitsPerLabelSetEntry(limitsPerLabelSetEntry) {
    const errors = [];
    if (typeof limitsPerLabelSetEntry !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const limitsPerLabelSetEntryObject = limitsPerLabelSetEntry;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: limitsPerLabelSetEntryObject,
        mandatoryAttributes: [],
        optionalAttributes: ['maxSeries'],
    }));
    if (limitsPerLabelSetEntryObject.maxSeries !== undefined) {
        if (typeof limitsPerLabelSetEntryObject.maxSeries !== 'number') {
            errors.push('maxSeries: must be a number');
        }
        else {
            errors.push(...(0, utils_1.validateNumberRange)({
                value: limitsPerLabelSetEntryObject.maxSeries,
                min: 0,
                messagePrefix: 'maxSeries: ',
            }));
        }
    }
    return errors;
}
function validateLabel(label) {
    const errors = [];
    if (typeof label !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const labelObject = label;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: labelObject,
        mandatoryAttributes: ['name', 'value'],
        optionalAttributes: [],
    }));
    if (labelObject.name !== undefined) {
        if (typeof labelObject.name !== 'string') {
            errors.push('name: must be a string');
        }
        else {
            errors.push(...(0, utils_1.validateRegExp)({
                regExp: /^[a-zA-Z_][a-zA-Z0-9_]*$/,
                value: labelObject.name,
                messagePrefix: 'name: ',
            }));
            errors.push(...(0, utils_1.validateStringLength)({
                value: labelObject.name,
                min: 1,
            }));
        }
    }
    if (labelObject.value !== undefined) {
        if (typeof labelObject.value !== 'string') {
            errors.push('value: must be a string');
        }
        else {
            errors.push(...(0, utils_1.validateStringLength)({
                value: labelObject.value,
                min: 1,
                messagePrefix: 'value: ',
            }));
        }
    }
    return errors;
}
function validateLimitsPerLabelSet(limitsPerLabelSet) {
    const errors = [];
    if (typeof limitsPerLabelSet !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const limitsPerLabelSetObject = limitsPerLabelSet;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: limitsPerLabelSetObject,
        mandatoryAttributes: ['labelSet', 'limits'],
        optionalAttributes: [],
    }));
    if (limitsPerLabelSetObject.labelSet !== undefined) {
        errors.push(...limitsPerLabelSetObject.labelSet.map((l, i) => {
            return validateLabel(l).map((err) => `labelSet[${i}]: ${err}`);
        }).flat());
    }
    if (limitsPerLabelSetObject.limits !== undefined) {
        errors.push(...validateLimitsPerLabelSetEntry(limitsPerLabelSetObject.limits).map((err) => `limits: ${err}`));
    }
    return errors;
}
function validateWorkspaceConfiguration(workspaceConfiguration) {
    const errors = [];
    if (typeof workspaceConfiguration !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const workspaceConfigurationObject = workspaceConfiguration;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: workspaceConfigurationObject,
        mandatoryAttributes: [],
        optionalAttributes: ['limitsPerLabelSets', 'retentionPeriodInDays'],
    }));
    if (workspaceConfigurationObject.limitsPerLabelSets !== undefined) {
        if (!Array.isArray(workspaceConfigurationObject.limitsPerLabelSets)) {
            errors.push('limitsPerLabelSets: must be an array');
        }
        else {
            errors.push(...(0, utils_1.validateArrayLength)({
                value: workspaceConfigurationObject.limitsPerLabelSets,
                min: 0,
                messagePrefix: 'limitsPerLabelSets: ',
            }));
            errors.push(...workspaceConfigurationObject.limitsPerLabelSets.map((l, i) => {
                return validateLimitsPerLabelSet(l).map((err) => `limitsPerLabelSets[${i}]: ${err}`);
            }).flat());
        }
    }
    if (workspaceConfigurationObject.retentionPeriodInDays !== undefined) {
        if (typeof workspaceConfigurationObject.retentionPeriodInDays !== 'number') {
            errors.push('retentionPeriodInDays: must be a number');
        }
        else {
            errors.push(...(0, utils_1.validateNumberRange)({
                value: workspaceConfigurationObject.retentionPeriodInDays,
                min: 1,
                messagePrefix: 'retentionPeriodInDays: ',
            }));
        }
    }
    return errors;
}
function validateAlertManagerDefinition(alertManagerDefinition) {
    const errors = [];
    if (typeof alertManagerDefinition !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    return errors;
}
function validateAlias(alias) {
    const errors = [];
    if (typeof alias !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    errors.push(...(0, utils_1.validateStringLength)({
        value: alias,
        min: 0,
        max: 100,
    }));
    return errors;
}
//# sourceMappingURL=data:application/json;base64,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