"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateAmpConfiguration = validateAmpConfiguration;
exports.validateDestination = validateDestination;
exports.validateRoleConfiguration = validateRoleConfiguration;
exports.validateScrapeConfiguration = validateScrapeConfiguration;
exports.validateEksConfiguration = validateEksConfiguration;
exports.validateSource = validateSource;
exports.validateAlias = validateAlias;
const utils_1 = require("@robhan-cdk-lib/utils");
function validateAmpConfiguration(ampConfiguration) {
    const errors = [];
    if (typeof ampConfiguration !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const ampConfigurationObject = ampConfiguration;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: ampConfigurationObject,
        mandatoryAttributes: ['workspace'],
        optionalAttributes: [],
    }));
    return errors;
}
function validateDestination(destination) {
    const errors = [];
    if (typeof destination !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const destinationObject = destination;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: destinationObject,
        mandatoryAttributes: ['ampConfiguration'],
        optionalAttributes: [],
    }));
    if (destinationObject.ampConfiguration !== undefined) {
        errors.push(...validateAmpConfiguration(destinationObject.ampConfiguration).map((err) => `ampConfiguration: ${err}`));
    }
    return errors;
}
function validateRoleConfiguration(roleConfiguration) {
    const errors = [];
    if (typeof roleConfiguration !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const roleConfigurationObject = roleConfiguration;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: roleConfigurationObject,
        mandatoryAttributes: [],
        optionalAttributes: ['sourceRole', 'targetRole'],
    }));
    return errors;
}
function validateScrapeConfiguration(scrapeConfiguration) {
    const errors = [];
    if (typeof scrapeConfiguration !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const scrapeConfigurationObject = scrapeConfiguration;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: scrapeConfigurationObject,
        mandatoryAttributes: ['configurationBlob'],
        optionalAttributes: [],
    }));
    if (scrapeConfigurationObject.configurationBlob !== undefined && typeof scrapeConfigurationObject.configurationBlob !== 'string') {
        errors.push('configurationBlob: must be a string');
    }
    return errors;
}
function validateEksConfiguration(eksConfiguration) {
    const errors = [];
    if (typeof eksConfiguration !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const eksConfigurationObject = eksConfiguration;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: eksConfigurationObject,
        mandatoryAttributes: ['cluster', 'subnets'],
        optionalAttributes: ['securityGroups'],
    }));
    if (eksConfigurationObject.securityGroups !== undefined) {
        if (!Array.isArray(eksConfigurationObject.securityGroups)) {
            errors.push('securityGroups: must be an array');
        }
        else {
            errors.push(...(0, utils_1.validateArrayLength)({
                value: eksConfigurationObject.securityGroups,
                min: 1,
                max: 5,
                messagePrefix: 'securityGroups: ',
            }));
        }
    }
    if (eksConfigurationObject.subnets !== undefined) {
        if (!Array.isArray(eksConfigurationObject.subnets)) {
            errors.push('subnets: must be an array');
        }
        else {
            errors.push(...(0, utils_1.validateArrayLength)({
                value: eksConfigurationObject.subnets,
                min: 1,
                max: 5,
                messagePrefix: 'subnets: ',
            }));
        }
    }
    return errors;
}
function validateSource(source) {
    const errors = [];
    if (typeof source !== 'object') {
        errors.push('must be an object');
        return errors; // No need to check further if not an object
    }
    const sourceObject = source;
    errors.push(...(0, utils_1.validateObjectAttributes)({
        inputObject: sourceObject,
        mandatoryAttributes: ['eksConfiguration'],
        optionalAttributes: [],
    }));
    if (sourceObject.eksConfiguration !== undefined) {
        errors.push(...validateEksConfiguration(sourceObject.eksConfiguration).map((err) => `eksConfiguration: ${err}`));
    }
    return errors;
}
function validateAlias(alias) {
    const errors = [];
    if (typeof alias !== 'string') {
        errors.push('must be a string');
        return errors; // No need to check further if not a string
    }
    errors.push(...(0, utils_1.validateStringLength)({
        value: alias,
        min: 1,
        max: 100,
    }));
    errors.push(...(0, utils_1.validateRegExp)({
        regExp: /^[0-9A-Za-z][-.0-9A-Z_a-z]*$/,
        value: alias,
    }));
    return errors;
}
//# sourceMappingURL=data:application/json;base64,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