"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KongDataPlane = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
class KongDataPlane extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // TODO : Use IRSA , Scope down the secret values and use new external secrets helm
        props.nodegroup.role.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:DescribeSecret',
                'secretsmanager:ListSecretVersionIds',
            ],
        }));
        const kong_namespace = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongNamespace', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'v1',
                    kind: 'Namespace',
                    metadata: { name: props.namespace },
                },
            ],
        });
        const kong_certificate = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'KongCert', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'cert-manager.io/v1',
                    kind: 'Certificate',
                    metadata: {
                        name: props.cacertname,
                        namespace: props.namespace,
                    },
                    spec: {
                        commonName: props.hostedZoneName,
                        dnsNames: props.dnsNames,
                        duration: '2160h0m0s',
                        issuerRef: {
                            group: 'awspca.cert-manager.io',
                            kind: 'AWSPCAClusterIssuer',
                            name: props.clusterIssuerName,
                        },
                        renewBefore: '360h0m0s',
                        secretName: `${props.clusterIssuerName}-secret`,
                        usages: [
                            'server auth',
                            'client auth',
                        ],
                        privateKey: {
                            algorithm: 'RSA',
                            size: 2048,
                        },
                    },
                },
            ],
        });
        kong_certificate.node.addDependency(kong_namespace);
        const deploy_external_secrets_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'ExternalSecrets', {
            cluster: props.cluster,
            repository: 'https://external-secrets.github.io/kubernetes-external-secrets/',
            chart: 'kubernetes-external-secrets',
            release: 'kubernetes-external-secrets',
            namespace: props.namespace,
            wait: true,
            values: {
                securityContext: {
                    fsGroup: 65534,
                },
                env: {
                    AWS_REGION: aws_cdk_lib_1.Stack.of(this).region,
                },
            },
        });
        deploy_external_secrets_helm.node.addDependency(kong_namespace);
        const secrets_crd_mount = new aws_cdk_lib_1.aws_eks.KubernetesManifest(this, 'SecretsMount', {
            cluster: props.cluster,
            overwrite: true,
            manifest: [
                {
                    apiVersion: 'kubernetes-client.io/v1',
                    kind: 'ExternalSecret',
                    metadata: {
                        name: 'kong-license',
                        namespace: props.namespace,
                    },
                    spec: {
                        backendType: 'secretsManager',
                        data: [
                            {
                                key: props.license_secret_name,
                                name: 'license',
                            },
                        ],
                    },
                },
            ],
        });
        secrets_crd_mount.node.addDependency(deploy_external_secrets_helm);
        const deploy_kong_dp_helm = new aws_cdk_lib_1.aws_eks.HelmChart(this, 'KongDpHelm', {
            cluster: props.cluster,
            repository: 'https://charts.konghq.com',
            chart: 'kong',
            release: 'kong',
            namespace: props.namespace,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            wait: true,
            createNamespace: true,
            values: {
                resources: {
                    requests: {
                        cpu: '300m',
                        memory: '300Mi',
                    },
                    limits: {
                        cpu: '1200m',
                        memory: '800Mi',
                    },
                },
                autoscaling: {
                    enabled: true,
                    minReplicas: 1,
                    maxReplicas: 20,
                    metrics: [
                        {
                            type: 'Resource',
                            resource: {
                                name: 'cpu',
                                target: {
                                    type: 'Utilization',
                                    averageUtilization: 75,
                                },
                            },
                        },
                    ],
                },
                ingressController: {
                    enabled: false,
                },
                image: {
                    repository: 'kong/kong-gateway',
                    tag: '2.5.0.0-alpine',
                },
                env: {
                    database: 'off',
                    role: 'data_plane',
                    cluster_mtls: 'pki',
                    cluster_cert: '/etc/secrets/kong-dp-cluster-issuer-secret/tls.crt',
                    cluster_cert_key: '/etc/secrets/kong-dp-cluster-issuer-secret/tls.key',
                    cluster_ca_cert: '/etc/secrets/kong-dp-cluster-issuer-secret/ca.crt',
                    cluster_server_name: 'kong-cp.internal',
                    cluster_control_plane: `${props.cluster_dns}:8005`,
                    cluster_telemetry_endpoint: `${props.telemetry_dns}:8006`,
                },
                proxy: {
                    annotations: {
                        'service.beta.kubernetes.io/aws-load-balancer-type': 'nlb',
                        'service.beta.kubernetes.io/aws-load-balancer-additional-resource-tags': 'Type=proxy',
                        'external-dns.alpha.kubernetes.io/hostname': 'proxy.kong-dp.internal',
                    },
                },
                enterprise: {
                    enabled: true,
                    license_secret: 'kong-license',
                },
                manager: {
                    enabled: false,
                },
                portal: {
                    enabled: false,
                },
                portalapi: {
                    enabled: false,
                },
                secretVolumes: [
                    'kong-dp-cluster-issuer-secret',
                ],
            },
        });
        deploy_kong_dp_helm.node.addDependency(secrets_crd_mount);
    }
}
exports.KongDataPlane = KongDataPlane;
//# sourceMappingURL=data:application/json;base64,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