# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['gcloud', 'gcloud.rest', 'gcloud.rest.datastore']

package_data = \
{'': ['*']}

install_requires = \
['gcloud-rest-auth>=3.1.0,<5.0.0']

setup_kwargs = {
    'name': 'gcloud-rest-datastore',
    'version': '7.0.0',
    'description': 'Python Client for Google Cloud Datastore',
    'long_description': '(Asyncio OR Threadsafe) Python Client for Google Cloud Datastore\n================================================================\n\n    This is a shared codebase for ``gcloud-rest-datastore`` and\n    ``gcloud-rest-datastore``\n\n|pypi| |pythons-aio| |pythons-rest|\n\nInstallation\n------------\n\n.. code-block:: console\n\n    $ pip install --upgrade gcloud-{aio,rest}-datastore\n\nUsage\n-----\n\nWe\'re still working on documentation; for now, this should help get you\nstarted:\n\n.. code-block:: python\n\n    from gcloud.rest.datastore import Datastore\n    from gcloud.rest.datastore import Direction\n    from gcloud.rest.datastore import Filter\n    from gcloud.rest.datastore import GQLQuery\n    from gcloud.rest.datastore import Key\n    from gcloud.rest.datastore import PathElement\n    from gcloud.rest.datastore import PropertyFilter\n    from gcloud.rest.datastore import PropertyFilterOperator\n    from gcloud.rest.datastore import PropertyOrder\n    from gcloud.rest.datastore import Query\n    from gcloud.rest.datastore import Value\n\n    ds = Datastore(\'my-gcloud-project\', \'/path/to/creds.json\')\n    key1 = Key(\'my-gcloud-project\', [PathElement(\'Kind\', \'entityname\')])\n    key2 = Key(\'my-gcloud-project\', [PathElement(\'Kind\', \'entityname2\')])\n\n    # batched lookups\n    entities = await ds.lookup([key1, key2])\n\n    # convenience functions for any datastore mutations\n    await ds.insert(key1, {\'a_boolean\': True, \'meaning_of_life\': 41})\n    await ds.update(key1, {\'a_boolean\': True, \'meaning_of_life\': 42})\n    await ds.upsert(key1, {\'animal\': \'aardvark\'})\n    await ds.delete(key1)\n\n    # or build your own mutation sequences with full transaction support\n    transaction = await ds.beginTransaction()\n    try:\n        mutations = [\n            ds.make_mutation(Operation.INSERT, key1, properties={\'animal\': \'sloth\'}),\n            ds.make_mutation(Operation.UPSERT, key1, properties={\'animal\': \'aardvark\'}),\n            ds.make_mutation(Operation.INSERT, key2, properties={\'animal\': \'aardvark\'}),\n        ]\n        await ds.commit(transaction, mutations=mutations)\n    except Exception:\n        await ds.rollback(transaction)\n\n    # support for partial keys\n    partial_key = Key(\'my-gcloud-project\', [PathElement(\'Kind\')])\n    # and ID allocation or reservation\n    allocated_keys = await ds.allocateIds([partial_key])\n    await ds.reserveIds(allocated_keys)\n\n    # query support\n    property_filter = PropertyFilter(prop=\'answer\',\n                                     operator=PropertyFilterOperator.EQUAL,\n                                     value=Value(42))\n    property_order = PropertyOrder(prop=\'length\',\n                                   direction=Direction.DESCENDING)\n    query = Query(kind=\'the_meaning_of_life\',\n                  query_filter=Filter(property_filter),\n                  order=property_order)\n    results = await ds.runQuery(query, session=s)\n\n    # alternatively, query support using GQL\n    gql_query = GQLQuery(\'SELECT * FROM the_meaning_of_life WHERE answer = @answer\',\n                         named_bindings={\'answer\': 42})\n    results = await ds.runQuery(gql_query, session=s)\n\n    # close the HTTP session\n    # Note that other options include:\n    # * providing your own session: ``Datastore(.., session=session)``\n    # * using a context manager: ``async with Datastore(..) as ds:``\n    await ds.close()\n\nCustom Subclasses\n~~~~~~~~~~~~~~~~~\n\n``gcloud-rest-datastore`` provides class interfaces mirroring all official\nGoogle API types, ie. ``Key`` and ``PathElement``, ``Entity`` and\n``EntityResult``, ``QueryResultBatch``, and ``Value``. These types will be\nreturned from arbitrary Datastore operations, for example\n``Datastore.allocateIds(...)`` will return a list of ``Key`` entities.\n\nFor advanced usage, all of these datatypes may be overloaded. A common use-case\nmay be to deserialize entities into more specific classes. For example, given a\ncustom entity class such as:\n\n.. code-block:: python\n\n    class MyEntityKind(gcloud.rest.datastore.Entity):\n        def __init__(self, key, properties = None) -> None:\n            self.key = key\n            self.is_an_aardvark = (properties or {}).get(\'aardvark\', False)\n\n        def __repr__(self):\n            return "I\'m an aardvark!" if self.is_an_aardvark else "Sorry, nope"\n\nWe can then configure ``gcloud-rest-datastore`` to serialize/deserialize from\nthis custom entity class with:\n\n.. code-block:: python\n\n    class MyCustomDatastore(gcloud.rest.datastore.Datastore):\n        entity_result_kind.entity_kind = MyEntityKind\n\nThe full list of classes which may be overridden in this way is:\n\n.. code-block:: python\n\n    class MyVeryCustomDatastore(gcloud.rest.datastore.Datastore):\n        datastore_operation_kind = DatastoreOperation\n        entity_result_kind = EntityResult\n        entity_result_kind.entity_kind = Entity\n        entity_result_kind.entity_kind.key_kind = Key\n        key_kind = Key\n        key_kind.path_element_kind = PathElement\n        mutation_result_kind = MutationResult\n        mutation_result_kind.key_kind = Key\n        query_result_batch_kind = QueryResultBatch\n        query_result_batch_kind.entity_result_kind = EntityResult\n        value_kind = Value\n        value_kind.key_kind = Key\n\n    class MyVeryCustomQuery(gcloud.rest.datastore.Query):\n        value_kind = Value\n\n    class MyVeryCustomGQLQuery(gcloud.rest.datastore.GQLQuery):\n        value_kind = Value\n\nYou can then drop-in the ``MyVeryCustomDatastore`` class anywhere where you\npreviously used ``Datastore`` and do the same for ``Query`` and ``GQLQuery``.\n\nTo override any sub-key, you\'ll need to override any parents which use it. For\nexample, if you want to use a custom Key kind and be able to use queries with\nit, you will need to implement your own ``Value``, ``Query``, and ``GQLQuery``\nclasses and wire them up to the rest of the custom classes:\n\n.. code-block:: python\n\n    class MyKey(gcloud.rest.datastore.Key):\n        pass\n\n    class MyValue(gcloud.rest.datastore.Value):\n        key_kind = MyKey\n\n    class MyEntity(gcloud.rest.datastore.Entity):\n        key_kind = MyKey\n        value_kind = MyValue\n\n    class MyEntityResult(gcloud.rest.datastore.EntityResult):\n        entity_kind = MyEntity\n\n    class MyQueryResultBatch(gcloud.rest.datastore.QueryResultBatch):\n        entity_result_kind = MyEntityResult\n\n    class MyDatastore(gcloud.rest.datastore.Datastore):\n        key_kind = MyKey\n        entity_result_kind = MyEntityResult\n        query_result_batch = MyQueryResultBatch\n        value_kind = MyValue\n\n    class MyQuery(gcloud.rest.datastore.Query):\n        value_kind = MyValue\n\n    class MyGQLQuery(gcloud.rest.datastore.GQLQuery):\n        value_kind = MyValue\n\nContributing\n------------\n\nPlease see our `contributing guide`_.\n\n.. _contributing guide: https://github.com/talkiq/gcloud-rest/blob/master/.github/CONTRIBUTING.rst\n\n.. |pypi| image:: https://img.shields.io/pypi/v/gcloud-rest-datastore.svg?style=flat-square\n    :alt: Latest PyPI Version (gcloud-rest-datastore)\n    :target: https://pypi.org/project/gcloud-rest-datastore/\n\n.. |pythons-aio| image:: https://img.shields.io/pypi/pyversions/gcloud-rest-datastore.svg?style=flat-square&label=python (aio)\n    :alt: Python Version Support (gcloud-rest-datastore)\n    :target: https://pypi.org/project/gcloud-rest-datastore/\n\n.. |pythons-rest| image:: https://img.shields.io/pypi/pyversions/gcloud-rest-datastore.svg?style=flat-square&label=python (rest)\n    :alt: Python Version Support (gcloud-rest-datastore)\n    :target: https://pypi.org/project/gcloud-rest-datastore/\n',
    'author': 'Vi Engineering',
    'author_email': 'voiceai-eng@dialpad.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/talkiq/gcloud-aio',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*',
}


setup(**setup_kwargs)
