"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationMultipleTargetGroupsServiceBase = void 0;
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const core_1 = require("@aws-cdk/core");
/**
 * The base class for ApplicationMultipleTargetGroupsEc2Service and ApplicationMultipleTargetGroupsFargateService classes.
 */
class ApplicationMultipleTargetGroupsServiceBase extends core_1.Construct {
    /**
     * Constructs a new instance of the ApplicationMultipleTargetGroupsServiceBase class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.listeners = new Array();
        this.targetGroups = new Array();
        this.loadBalancers = new Array();
        this.validateInput(props);
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        this.desiredCount = props.desiredCount || 1;
        if (props.taskImageOptions) {
            this.logDriver = this.createLogDriver(props.taskImageOptions.enableLogging, props.taskImageOptions.logDriver);
        }
        if (props.loadBalancers) {
            for (const lbProps of props.loadBalancers) {
                const lb = this.createLoadBalancer(lbProps.name, lbProps.publicLoadBalancer);
                this.loadBalancers.push(lb);
                const protocolType = new Set();
                for (const listenerProps of lbProps.listeners) {
                    const protocol = this.createListenerProtocol(listenerProps.protocol, listenerProps.certificate);
                    if (listenerProps.certificate !== undefined && protocol !== undefined && protocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                        throw new Error('The HTTPS protocol must be used when a certificate is given');
                    }
                    protocolType.add(protocol);
                    const listener = this.configListener(protocol, {
                        certificate: listenerProps.certificate,
                        domainName: lbProps.domainName,
                        domainZone: lbProps.domainZone,
                        listenerName: listenerProps.name,
                        loadBalancer: lb,
                        port: listenerProps.port,
                    });
                    this.listeners.push(listener);
                }
                const domainName = this.createDomainName(lb, lbProps.domainName, lbProps.domainZone);
                new core_1.CfnOutput(this, `LoadBalancerDNS${lb.node.id}`, { value: lb.loadBalancerDnsName });
                for (const protocol of protocolType) {
                    new core_1.CfnOutput(this, `ServiceURL${lb.node.id}${protocol.toLowerCase()}`, { value: protocol.toLowerCase() + '://' + domainName });
                }
            }
            // set up default load balancer and listener.
            this.loadBalancer = this.loadBalancers[0];
            this.listener = this.listeners[0];
        }
        else {
            this.loadBalancer = this.createLoadBalancer('LB');
            const protocol = this.createListenerProtocol();
            this.listener = this.configListener(protocol, {
                listenerName: 'PublicListener',
                loadBalancer: this.loadBalancer,
            });
            const domainName = this.createDomainName(this.loadBalancer);
            new core_1.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
            new core_1.CfnOutput(this, 'ServiceURL', { value: protocol.toLowerCase() + '://' + domainName });
        }
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs.
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
    findListener(name) {
        if (!name) {
            return this.listener;
        }
        for (const listener of this.listeners) {
            if (listener.node.id === name) {
                return listener;
            }
        }
        throw new Error(`Listener ${name} is not defined. Did you define listener with name ${name}?`);
    }
    registerECSTargets(service, container, targets) {
        for (const targetProps of targets) {
            const targetGroup = this.findListener(targetProps.listener).addTargets(`ECSTargetGroup${container.containerName}${targetProps.containerPort}`, {
                port: 80,
                targets: [
                    service.loadBalancerTarget({
                        containerName: container.containerName,
                        containerPort: targetProps.containerPort,
                        protocol: targetProps.protocol,
                    }),
                ],
                hostHeader: targetProps.hostHeader,
                pathPattern: targetProps.pathPattern,
                priority: targetProps.priority,
            });
            this.targetGroups.push(targetGroup);
        }
        if (this.targetGroups.length === 0) {
            throw new Error('At least one target group should be specified.');
        }
        return this.targetGroups[0];
    }
    addPortMappingForTargets(container, targets) {
        for (const target of targets) {
            if (!container.findPortMapping(target.containerPort, target.protocol || aws_ecs_1.Protocol.TCP)) {
                container.addPortMappings({
                    containerPort: target.containerPort,
                    protocol: target.protocol,
                });
            }
        }
    }
    /**
     * Create log driver if logging is enabled.
     */
    createLogDriver(enableLoggingProp, logDriverProp) {
        const enableLogging = enableLoggingProp !== undefined ? enableLoggingProp : true;
        const logDriver = logDriverProp !== undefined
            ? logDriverProp : enableLogging
            ? this.createAWSLogDriver(this.node.id) : undefined;
        return logDriver;
    }
    configListener(protocol, props) {
        const listener = this.createListener(props.listenerName, props.loadBalancer, protocol, props.port);
        let certificate;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            certificate = this.createListenerCertificate(props.listenerName, props.certificate, props.domainName, props.domainZone);
        }
        else {
            certificate = undefined;
        }
        if (certificate !== undefined) {
            listener.addCertificateArns(`Arns${props.listenerName}`, [certificate.certificateArn]);
        }
        return listener;
    }
    validateInput(props) {
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        if (props.loadBalancers) {
            if (props.loadBalancers.length === 0) {
                throw new Error('At least one load balancer must be specified');
            }
            for (const lbProps of props.loadBalancers) {
                if (lbProps.listeners.length === 0) {
                    throw new Error('At least one listener must be specified');
                }
            }
        }
    }
    createLoadBalancer(name, publicLoadBalancer) {
        const internetFacing = publicLoadBalancer !== undefined ? publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        return new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, name, lbProps);
    }
    createListenerProtocol(listenerProtocol, certificate) {
        return listenerProtocol !== undefined ? listenerProtocol : (certificate ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP);
    }
    createListenerCertificate(listenerName, certificate, domainName, domainZone) {
        if (typeof domainName === 'undefined' || typeof domainZone === 'undefined') {
            throw new Error('A domain name and zone is required when using the HTTPS protocol');
        }
        if (certificate !== undefined) {
            return certificate;
        }
        else {
            return new aws_certificatemanager_1.Certificate(this, `Certificate${listenerName}`, {
                domainName,
                validation: aws_certificatemanager_1.CertificateValidation.fromDns(domainZone),
            });
        }
    }
    createListener(name, lb, protocol, port) {
        return lb.addListener(name, {
            protocol,
            open: true,
            port,
        });
    }
    createDomainName(loadBalancer, name, zone) {
        let domainName = loadBalancer.loadBalancerDnsName;
        if (typeof name !== 'undefined') {
            if (typeof zone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            const record = new aws_route53_1.ARecord(this, `DNS${loadBalancer.node.id}`, {
                zone,
                recordName: name,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
            domainName = record.domainName;
        }
        return domainName;
    }
}
exports.ApplicationMultipleTargetGroupsServiceBase = ApplicationMultipleTargetGroupsServiceBase;
//# sourceMappingURL=data:application/json;base64,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