#! /usr/bin/env python
# -*- coding: utf-8 -*-
#
# Copyright 2019 Spotify AB

import codecs
import os
import re

from setuptools import find_packages
from setuptools import setup


HERE = os.path.abspath(os.path.dirname(__file__))


#####
# Helper functions
#####
def read(*filenames, **kwargs):
    """
    Build an absolute path from ``*filenames``, and  return contents of
    resulting file.  Defaults to UTF-8 encoding.
    """
    encoding = kwargs.get("encoding", "utf-8")
    sep = kwargs.get("sep", "\n")
    buf = []
    for fl in filenames:
        with codecs.open(os.path.join(HERE, fl), "rb", encoding) as f:
            buf.append(f.read())
    return sep.join(buf)


def find_meta(meta):
    """Extract __*meta*__ from META_FILE."""
    re_str = r"^__{meta}__ = ['\"]([^'\"]*)['\"]".format(meta=meta)
    meta_match = re.search(re_str, META_FILE, re.M)
    if meta_match:
        return meta_match.group(1)
    raise RuntimeError("Unable to find __{meta}__ string.".format(meta=meta))


#####
# Project-specific constants
#####
NAME = "spotify-klio-exec"
PACKAGE_NAME = "spotify_klio_exec"
PACKAGES = find_packages(where="src")
META_PATH = os.path.join("src", PACKAGE_NAME, "__init__.py")
CLASSIFIERS = [
    "Development Status :: 3 - Alpha",
    "Natural Language :: English",
    "Operating System :: POSIX :: Linux",
    "Operating System :: MacOS :: MacOS X",
    "Programming Language :: Python",
    "Programming Language :: Python :: 3.6",
    "Programming Language :: Python :: 3.7",
    "Programming Language :: Python :: 3.8",
    "Programming Language :: Python :: Implementation :: CPython",
]
META_FILE = read(META_PATH)
INSTALL_REQUIRES = [
    "click",
    "spotify-klio>=21.8.0",
    "klio-core>=21.8.0",
    "klio-exec>=21.8.0",
    "packaging",
]
EXTRAS_REQUIRE = {
    "docs": ["mkdocs", "interrogate"],
    "tests": ["coverage", "pytest>=4.3.0", "pytest-cov", "pytest-mock"],
    "debug": ["klio-exec[debug]"],
}
EXTRAS_REQUIRE["dev"] = (
    EXTRAS_REQUIRE["docs"]
    + EXTRAS_REQUIRE["tests"]
    + EXTRAS_REQUIRE["debug"]
    + ["bumpversion", "wheel"]
)
PLUGIN_PATH = "spotify_klio_exec.plugins.audit."
AUDIT_PLUGINS = [
    "multithreaded_tf=" + PLUGIN_PATH + "multithreaded_tf:_init",
    "numpy_broken_blas=" + PLUGIN_PATH + "numpy_broken_blas:_init",
]
# support 3.6, 3.7, & 3.8, matching Beam's support
PYTHON_REQUIRES = ">=3.6, <3.9"


setup(
    name=NAME,
    version=find_meta("version"),
    description=find_meta("description"),
    url=find_meta("uri"),
    author=find_meta("author"),
    author_email=find_meta("email"),
    maintainer=find_meta("author"),
    maintainer_email=find_meta("email"),
    packages=PACKAGES,
    package_dir={"": "src"},
    include_package_data=True,
    classifiers=CLASSIFIERS,
    zip_safe=False,
    install_requires=INSTALL_REQUIRES,
    extras_require=EXTRAS_REQUIRE,
    python_requires=PYTHON_REQUIRES,
    entry_points={
        "console_scripts": [
            "klioexec = spotify_klio_exec.cli:main",
            # maintain backwards compatibility until we can get all
            # of spotify-klio-cli and klio-cli updated
            "klio-runner = spotify_klio_exec.cli:main",
        ],
        "klio.plugins.audit": AUDIT_PLUGINS,
    },
)
