import periodictable
from enum import Enum

class CalcType(Enum):
    UNDEFINED = 0
    SP = 1
    OPT = 2
    CONSTR_OPT = 3
    FREQ = 4
    TS = 5
    MEP = 6

    NMR = 10
    UVVIS = 11
    UVVIS_TDA = 12
    MO = 13

    OPTFREQ = 20

ATOMIC_NUMBER = {}
ATOMIC_SYMBOL = {}
LOWERCASE_ATOMIC_SYMBOLS = {}

for el in periodictable.elements:
    ATOMIC_NUMBER[el.symbol] = el.number
    ATOMIC_SYMBOL[el.number] = el.symbol
    LOWERCASE_ATOMIC_SYMBOLS[el.symbol.lower()] = el.symbol

STR_TYPES = {#TODO: more synonyms
        "sp": CalcType.SP,
        "single-point": CalcType.SP,
        "single point": CalcType.SP,
        "single point energy": CalcType.SP,
        "single-point energy": CalcType.SP,
        "opt": CalcType.OPT,
        "optimisation": CalcType.OPT,
        "optimization": CalcType.OPT,
        "geometrical optimisation": CalcType.OPT,
        "geometrical optimization": CalcType.OPT,
        "constr_opt": CalcType.CONSTR_OPT,
        "constrained optimisation": CalcType.CONSTR_OPT,
        "constrained optimization": CalcType.CONSTR_OPT,
        "freq": CalcType.FREQ,
        "frequency calculation": CalcType.FREQ,
        "ts": CalcType.TS,
        "ts optimisation": CalcType.TS,
        "opt ts": CalcType.TS,
        "optts": CalcType.TS,
        "nmr": CalcType.NMR,
        "nmr prediction": CalcType.NMR,
        "nmr calculation": CalcType.NMR,
        "uvvis": CalcType.UVVIS,
        "uvvis prediction": CalcType.UVVIS,
        "uv-vis prediction": CalcType.UVVIS,
        "uvvis calculation": CalcType.UVVIS,
        "uv-vis calculation": CalcType.UVVIS,
        "td": CalcType.UVVIS,
        "tddft": CalcType.UVVIS,
        "td-dft": CalcType.UVVIS,
        "tda": CalcType.UVVIS_TDA,
        "mep": CalcType.MEP,
        "minimum energy path": CalcType.MEP,
        "minimum-energy path": CalcType.MEP,
        "minimum-energy-path": CalcType.MEP,
        "neb": CalcType.MEP,
        "nudged elastic band": CalcType.MEP,
        "mo": CalcType.MO,
        "mo calc": CalcType.MO,
        "mo calculation": CalcType.MO,
        "optfreq": CalcType.OPTFREQ,
        "opt_freq": CalcType.OPTFREQ,
        "opt-freq": CalcType.OPTFREQ,
        "opt+freq": CalcType.OPTFREQ,
        }

INV_STR_TYPES = {}
for cmd, t in STR_TYPES.items():
    if t in INV_STR_TYPES.keys():
        INV_STR_TYPES[t].append(cmd)
    else:
        INV_STR_TYPES[t] = [cmd]

SPECIFICATIONS = {
    'gaussian': {
        'general': {
            'nosymm': 0,
            '5d': 0,
            '6d': 0,
        },
        'scf': {
            'xqc': 0,
            'yqc': 0,
            'tight': 0,
        },
        'pop': {
            'nbo': 0,
            'npa': 0,
            'hirshfeld': 0,
            'esp': 0,
            'hly': 0,
            'hlygat': 0,
        },
        'opt': {
            'maxstep': 1,
            'maxcycles': 1,
            'calcfc': 0,
            'calcall': 0,
            'recalcfc': 1,
            'loose': 0,
            'tight': 0,
            'verytight': 0,
        },
        'freq': {
            'noraman': 0,
        },
    },
    'orca': {
        'general': {
            'sloppyscf': 0,
            'loosescf': 0,
            'normalscf': 0,
            'strongscf': 0,
            'tightscf': 0,
            'verytightscf': 0,
            'extremescf': 0,
            'grid': 1,
        },
        'opt': {
            'looseopt': 0,
            'normalopt': 0,
            'tightopt': 0,
            'verytightopt': 0,
        }

    },
    'xtb': {
        'general': {
            'acc': 1,
            'iterations': 1,
            'gfn2-xtb': 0,
            'gfn1-xtb': 0,
            'gfn0-xtb': 0,
            'gfn-ff': 0,
        },
        'Geometrical Optimisation': {
            'opt(crude)': 0,
            'opt(sloppy)': 0,
            'opt(loose)': 0,
            'opt(lax)': 0,
            'opt(normal)': 0,
            'opt(tight)': 0,
            'opt(vtight)': 0,
            'opt(extreme)': 0,
        },
        'Conformational Search': {
            'gfn2-xtb//gfn-ff': 0,
            'rthr': 1,
            'ewin': 1,
        },
    },
}

THEORY_LEVELS = {
    'hartree-fock': ['hf', 'hartree-fock'],
    # Some old exotic semi-empirical methods in ORCA are missing
    'semi-empirical': ['am1', 'pm3', 'pm6', 'pm7', 'mndo'],
    'xtb': ['gfn2-xtb', 'gfn1-xtb', 'gfn0-xtb', 'gfn-ff'],
    'special': ['hf-3c', 'pbeh-3c', 'r2scan-3c', 'b97-3c'],
    'mp2': ['mp2', 'ri-mp2'],
    'cc': ['ccsd', 'ccsd(t)', 'qcisd', 'qcisd(t)', 'lpno-ccsd', \
            'lpno-ccsd(t)', 'lpno-qcisd', 'lpno-qcisd(t)', \
            'dlpno-ccsd', 'dlpno-ccsd(t)', 'dlpno-qcisd', \
            'dlpno-qcisd(t)'],
    # Everything else is assumed to be DFT, because it most likely is
}

SYN_SOFTWARE = {
    'gaussian': ['g16', 'gaussian16', 'gaussian 16'],
    'orca': ['orca 5', 'orca5'],
        }

SYN_SOLVENTS = {
    'acetone': ['acetone'],
    'acetonitrile': ['acetonitrile', 'acn', 'ch3cn', 'mecn'],
    'benzene': ['benzene', 'c6h6'],
    'dichloromethane': ['ch2cl2', 'dcm'],
    'chloroform': ['chcl3'],
    'carbon disulfide': ['cs2'],
    'dimethylformamide': ['dmf'],
    'dimethylsulfoxide': ['dmso'],
    'diethylether': ['diethyl ether', 'ether', 'et2o'],
    'water': ['h2o'],
    'methanol': ['meoh'],
    'n-hexane': ['hexane', 'hex'],
    'tetrahydrofuran': ['thf'],
    'toluene': ['phme', 'phch3'],

    'ethanol': [],
    'isoquinoline': [],
    'quinoline': [],
    'dichloroethane': [],
    'carbontetrachloride': ['ccl4', 'carbon tetrachloride'],
    'chlorobenzene': ['phcl'],
    'nitromethane': ['meno2'],
    'heptane': [],
    'cyclohexane': [],
    'aniline': [],
    'argon': [],
    'krypton': [],
    'xenon': [],
    'n-octanol': ['1-octanol', 'octanol'],
    '1,1,1-trichloroethane': [],
    '1,1,2-trichloroethane': [],
    '1,2,4-trimethylbenzene': [],
    '1,2-dibromoethane': [],
    '1,2-ethanediol': [],
    '1,4-dioxane': [],
    '1-bromo-2-methylpropane': [],
    '1-bromooctane': [],
    '1-bromopentane': [],
    '1-bromopropane': [],
    '1-butanol': [],
    '1-chlorohexane': [],
    '1-chloropentane': [],
    '1-chloropropane': [],
    '1-decanol': [],
    '1-fluorooctane': [],
    '1-heptanol': [],
    '1-hexanol': [],
    '1-hexene': [],
    '1-hexyne': [],
    '1-iodobutane': [],
    '1-iodohexadecane': [],
    '1-iodopentane': [],
    '1-iodopropane': [],
    '1-nitropropane': [],
    '1-nonanol': [],
    '1-pentanol': [],
    '1-pentene': [],
    '1-propanol': [],
    '2,2,2-trifluoroethanol': [],
    '2,2,4-trimethylpentane': [],
    '2,4-dimethylpentane': [],
    '2,4-dimethylpyridine': [],
    '2,6-dimethylpyridine': [],
    '2-bromopropane': [],
    '2-butanol': [],
    '2-chlorobutane': [],
    '2-heptanone': [],
    '2-hexanone': [],
    '2-methoxyethanol': [],
    '2-methyl-1-propanol': [],
    '2-methyl-2-propanol': [],
    '2-methylpentane': [],
    '2-methylpyridine': [],
    '2-nitropropane': [],
    '2-octanone': [],
    '2-pentanone': [],
    '2-propanol': [],
    '2-propen-1-ol': [],
    '3-methylpyridine': [],
    '3-pentanone': [],
    '4-heptanone': [],
    '4-methyl-2-pentanone': [],
    '4-methylpyridine': [],
    '5-nonanone': [],
    'aceticacid': [],
    'acetophenone': [],
    'a-chlorotoluene': [],
    'anisole': [],
    'benzaldehyde': [],
    'benzonitrile': [],
    'benzylalcohol': [],
    'bromobenzene': [],
    'bromoethane': [],
    'bromoform': [],
    'butanal': [],
    'butanoicacid': [],
    'butanone': [],
    'butanonitrile': [],
    'butylamine': [],
    'butylethanoate': [],
    'carbondisulfide': [],
    'cis-1,2-dimethylcyclohexane': [],
    'cis-decalin': [],
    'cyclohexanone': [],
    'cyclopentane': [],
    'cyclopentanol': [],
    'cyclopentanone': [],
    'decalin-mixture': [],
    'dibromomethane': [],
    'dibutylether': [],
    'diethylamine': [],
    'diethylsulfide': [],
    'diiodomethane': [],
    'diisopropylether': [],
    'dimethyldisulfide': [],
    'diphenylether': [],
    'dipropylamine': [],
    'e-1,2-dichloroethene': [],
    'e-2-pentene': [],
    'ethanethiol': [],
    'ethylbenzene': [],
    'ethylethanoate': [],
    'ethylmethanoate': [],
    'ethylphenylether': [],
    'fluorobenzene': [],
    'formamide': [],
    'formicacid': [],
    'hexanoicacid': [],
    'iodobenzene': [],
    'iodoethane': [],
    'iodomethane': [],
    'isopropylbenzene': [],
    'm-cresol': [],
    'mesitylene': [],
    'methylbenzoate': [],
    'methylbutanoate': [],
    'methylcyclohexane': [],
    'methylethanoate': [],
    'methylmethanoate': [],
    'methylpropanoate': [],
    'm-xylene': [],
    'n-butylbenzene': [],
    'n-decane': [],
    'n-dodecane': [],
    'n-hexadecane': [],
    'nitrobenzene': ['phno2'],
    'nitroethane': [],
    'n-methylaniline': [],
    'n-methylformamide-mixture': [],
    'n,n-dimethylacetamide': [],
    'n,n-dimethylformamide': ['n,n-dimethylformamide', 'dmf'],
    'n-nonane': [],
    'n-octane': [],
    'n-pentadecane': [],
    'n-pentane': [],
    'n-undecane': [],
    'o-chlorotoluene': [],
    'o-cresol': [],
    'o-dichlorobenzene': [],
    'o-nitrotoluene': [],
    'o-xylene': [],
    'pentanal': [],
    'pentanoicacid': [],
    'pentylamine': [],
    'pentylethanoate': [],
    'perfluorobenzene': [],
    'p-isopropyltoluene': [],
    'propanal': [],
    'propanoicacid': [],
    'propanonitrile': [],
    'propylamine': [],
    'propylethanoate': [],
    'p-xylene': [],
    'pyridine': [],
    'sec-butylbenzene': [],
    'tert-butylbenzene': [],
    'tetrachloroethene': [],
    'tetrahydrothiophene-s,s-dioxide': [],
    'tetralin': [],
    'thiophene': [],
    'thiophenol': [],
    'trans-decalin': [],
    'tributylphosphate': [],
    'trichloroethene': [],
    'triethylamine': [],
    'xylene-mixture': [],
    'z-1,2-dichloroethene': [],
}


SYN_METHODS = {
        'hfs': [],
        'xalpha': [],
        'lsd': [],
        'vwn5': [],
        'vwn3': [],
        'bnull': [],
        'bvwn': [],
        'bp': [],
        'blyp': [],
        'gp': [],
        'glyp': [],
        'pbe': [],
        'revpbe': [],
        'rpbe': [],
        'pwp': [],
        'olyp': [],
        'opbe': [],
        'xlyp': [],

        'm05' : [],
        'm052x' : ['m05-2x'],
        'm06' : [],
        'm062x' : ['m06-2x'],
        'm06hf' : ['m06-hf'],
        'm06l': ['m06-l'],
        'm08hx': [],

        'tpss': [],
        'revtpss': [],
        'scanfunc': [],

        'b1lyp' : [],
        'b3lyp' : [],
        'bhandhlyp' : [],
        'cam-b3lyp ' : ['camb3lyp'],
        'lc-blyp' : ['lcblyp'],
        'o3lyp' : [],
        'x3lyp' : [],

        'wb2plyp' : [],
        'wb2gp-plyp ' : ['wb2gpplyp'],

        'b1p' : [],
        'b3p' : [],
        'g1lyp' : [],
        'g3lyp' : [],
        'g1p' : [],
        'g3p' : [],
        'pbe0' : ['pbe1pbe'],
        'pwp1' : [],

        'mpwpw': [],
        'mpwlyp': [],
        'mpw1pw' : [],
        'mpw1lyp' : [],
        'mpw1pw91' : [],
        'mpw1lyp' : [],
        'mpw1pbe' : [],
        'mpw3pbe' : [],
        'pw6b95d3' : ['pw6b95-d3'],
        'pw86pbe': [],
        'rpw86pbe': [],
        'pw91_0' : ['pw91-0', 'pw910'],
        'pw6b95' : [],
        'pwlda': [],
        'pw91': [],

        'b97' : [],
        'b97-d': ['b97d'],
        'b97-d3': ['b97d3'],
        'wb97' : [],
        'wb97x' : [],
        'wb97x-d' : ['wb97xd'],
        'wb97x-d3' : ['wb97xd3'],
        'wb97x-v' : ['wb97xv'],
        'wb97x-d3bj ' : ['wb97xd3bj'],

        'tpssh' : [],
        'tpss0' : [],

        'b2plyp' : [],
        'mpw2plyp ' : [],
        'b2gp-plyp ' : ['b2gpplyp'],
        'b2k-plyp ' : ['b2kplyp'],
        'b2t-plyp ' : ['b2tplyp'],
        'pwpb95' : [],

        'b3p86' : [],
        'hse06' : ['hse1pbe'],
        'ohse2pbe' : [],
        'ohse1pbe' : [],
        'pbeh1pbe' : [],
        'b1b95' : [],
        'apfd' : [],
        'lc-whpbe' : ['lcwhpbe'],
        'thcthhyb' : [],
        'bmk' : [],
        'hissbpbe' : [],
        'mn15' : [],
        'mn15l' : ['mn15-l'],
        'm11' : [],
        'm11l' : ['m11-l'],
        'sogga11x' : [],
        'sogga11' : [],
        'n12sx' : [],
        'n12' : [],
        'mn12sx' : [],
        'mn12l' : ['mn12-l'],
        'vsxc' : [],
        'hcth407' : ['hcth/407', 'hcth-407'],
        'hcth93' : ['hcth/93', 'hcth-93'],
        'hcth147' : ['hcth/147', 'hcth-147'],
        'thcth' : [],

        'hf': ['hartree-fock'],

        'mndo': [],
        'am1': [],
        'pm3': [],
        'pm6': [],
        'pm7': ['pm7r6'],

        'gfn2-xtb': ['xtb2', 'gfn2', 'gfn2xtb'],
        'gfn1-xtb': ['xtb1', 'gfn1', 'gfn1xtb', 'gfn-xtb', 'gfnxtb'],
        'gfn0-xtb': ['xtb0', 'gfn0', 'gfn0xtb'],
        'gfn-ff': ['gfnff'],

        'mp2': [],
        'ri-mp2': ['rimp2'],

        'hf-3c': ['hf3c'],
        'pbeh-3c': ['pbeh3c'],
        'r2scan-3c': ['r2scan3c'],
        'b97-3c': ['b973c'],

        'ccsd': [],
        'ccsd(t)': ['ccsd(t)', 'ccsdt'],
        'qcisd': ['qcisd'],
        'qcisd(t)': ['qcisd(t)', 'qcisdt'],

        'lpno-ccsd': ['lpnoccsd'],
        'lpno-ccsd(t)': ['lpnoccsd(t)', 'lpno-ccsdt', 'lpnoccsdt'],
        'lpno-qcisd': ['lpnoqcisd'],
        'lpno-qcisd(t)': ['lpno-qcisdt', 'lpnoqcisd(t)', 'lpnoqcisdt'],
        'dlpno-ccsd': [],
        'dlpno-ccsd(t)': ['dlpno-ccsdt', 'dlpnoccsd(t)', 'dlpnoccsdt'],
        'dlpno-qcisd': ['dlpnoqcisd'],
        'dlpno-qcisd(t)': ['dlpno-qcisdt', 'dlpnoqcisdt'],
    }



# Gaussian allows the combination of exchange and correlation functionals
# without using any special keyword.

EXCHANGE_FUNCTIONALS = {
        's': 'S',
        'xa': 'XA',
        'b': 'B',
        'pw91': 'PW91',
        'mpw': 'mPW',
        'g96': 'G96',
        'pbe': 'PBE',
        'o': 'O',
        'tpss': 'TPSS',
        'revtpss': 'RevTPSS',
        'brx': 'BRx',
        'pkzb': 'PKZB',
        'wpbeh': 'wPBEh',
        'pbeh': 'PBEh',
}

CORRELATION_FUNCTIONALS = {
        'vwn': 'VWN',
        'vwn5': 'VWN5',
        'lyp': 'LYP',
        'pl': 'PL',
        'p86': 'P86',
        'pw91': 'PW91',
        'b95': 'B95',
        'pbe': 'PBE',
        'tpss': 'TPSS',
        'revtpss': 'RevTPSS',
        'kcis': 'KCIS',
        'brc': 'BRC',
        'pkzb': 'PKZB',
        'vp86': 'VP86',
        'v5lyp': 'V5LYP',
}

SYN_BASIS_SETS = {
        'sto-3g': [],

        'mini': [],
        'minis': [],
        'minix': [],
        'midi': [],
        'midix': ['midi-x'],

        '3-21g': [],
        '3-21gsp': [],
        '4-22gsp': [],
        '4-31g': [],
        '6-21g': [],

        '6-31g': [],
        '6-31g(d)': ['6-31g*'],

        'm6-31g': [],
        'm6-31g(d)': ['m6-31g*'],

        '6-31g(d,p)': ['6-31g**'],
        '6-31g(2d)': [],
        '6-31g(2d,p)': [],
        '6-31g(2d,2p)': [],
        '6-31g(2df)': [],
        '6-31g(2df,2p)': [],
        '6-31g(2df,2pd)': [],

        '6-31+g(d)': ['6-31+g*'],
        '6-31+g(d,p)': ['6-31+g**'],

        '6-31+g(2d)': [],
        '6-31+g(2d,p)': [],
        '6-31+g(2d,2p)': [],
        '6-31+g(2df)': [],
        '6-31+g(2df,2p)': [],
        '6-31+g(2df,2pd)': [],

        '6-31++g(d,p)': ['6-31++g**'],
        '6-31++g(2d,p)': [],
        '6-31++g(2d,2p)': [],
        '6-31++g(2df)': [],
        '6-31++g(2df,2p)': [],
        '6-31++g(2df,2pd)': [],

        '6-311g(d)': ['6-311g*'],
        '6-311g(d,p)': ['6-311g**'],
        '6-311g(2d)': [],
        '6-311g(2d,p)': [],
        '6-311g(2d,2p)': [],
        '6-311g(2df)': [],
        '6-311g(2df,2p)': [],
        '6-311g(2df,2pd)': [],
        '6-311g(3df)': [],
        '6-311g(3df,3pd)': [],

        '6-311+g(d)': ['6-311+g*'],
        '6-311+g(d,p)': ['6-311+g**'],
        '6-311+g(2d)': [],
        '6-311+g(2d,p)': [],
        '6-311+g(2d,2p)': [],
        '6-311+g(2df)': [],
        '6-311+g(2df,2p)': [],
        '6-311+g(2df,2pd)': [],
        '6-311+g(3df)': [],
        '6-311+g(3df,2p)': [],
        '6-311+g(3df,3pd)': [],

        '6-311++g(d,p)': ['6-311++g**'],
        '6-311++g(2d)': [],
        '6-311++g(2d,p)': [],
        '6-311++g(2d,2p)': [],
        '6-311++g(2df)': [],
        '6-311++g(2df,2p)': [],
        '6-311++g(2df,2pd)': [],
        '6-311++g(3df)': [],
        '6-311++g(3df,3pd)': [],

        'hav(t+d)z': [],
        'hav(q+d)z': [],
        'hav(5+d)z': [],

        'w1-dz': [],
        'w1-mtsmall': [],
        'w1-dz': [],

        'sv': [],
        'svp': ['sv(p)'],

        'tzv': [],
        'tzvp': ['tzv(p)'],
        'tzvpp': ['tzv-pp'],

        'qzvp': ['qzv(p)'],
        'qzvpp': ['qzv-pp'],

        'dkh-svp': ['dkh-sv(p)'],
        'dkh-tzvp': ['dkh-tzv(p)'],
        'dkh-tzv-pp': ['dkh-tzvpp'],
        'dkh-qzvp': ['dkh-qzv(p)'],
        'dkh-qzv-pp': ['dkh-qzvpp'],

        'zora-svp': ['zora-sv(p)'],
        'zora-tzvp': ['zora-tzv(p)'],
        'zora-tzv-pp': ['zora-tzvpp'],
        'zora-qzvp': ['zora-qzv(p)'],
        'zora-qzv-pp': ['zora-qzvpp'],

        'def2-msvp': ['def2msvp'],
        'def2-mtzvp': ['def2mtzvp'],

        'def2-svp': ['def2-sv(p)', 'def2sv(p)', 'def2svp'],
        'def2-tzvp(-f)': ['def2tzvp(-f)'],
        'def2-tzvp': ['def2tzvp'],
        'def2-tzvpp': ['def2tzvpp', 'def2tzv-pp', 'def2-tzv-pp'],
        'def2-qzvp': ['def2qzvp'],
        'def2-qzvpp': ['def2qzvpp', 'def2qzv-pp', 'def2-qzv-pp'],

        'def2-svpd': ['def2svpd'],
        'def2-tzvpd': ['def2tzvpd'],
        'def2-tzvppd': ['def2tzvppd'],
        'def2-qzvpd': ['def2qzvpd'],
        'def2-qzvppd': ['def2qzvppd'],

        'def2-sv': ['def2sv'],
        'def2-svpp': ['def2svpp'],
        'def2-tzv': ['def2tzv'],
        'def2-qzv': ['def2qzv'],

        'dkh-def2-svp': ['dkh-def2-sv(p)', 'dkh-def2sv(p)', 'dkh-def2svp'],
        'dkh-def2-tzvp(-f)': ['dkh-def2tzvp(-f)'],
        'dkh-def2-tzvp': ['dkh-def2tzvp'],
        'dkh-def2-tzvpp': ['dkh-def2tzvpp', 'dkh-def2tzv-pp', 'dkh-def2-tzv-pp'],
        'dkh-def2-qzvp': ['dkh-def2qzvp'],
        'dkh-def2-qzvpp': ['dkh-def2qzvpp', 'dkh-def2qzv-pp', 'dkh-def2-qzv-pp'],

        'zora-def2-svp': ['zora-def2-sv(p)', 'zora-def2sv(p)', 'zora-def2svp'],
        'zora-def2-tzvp(-f)': ['zora-def2tzvp(-f)'],
        'zora-def2-tzvp': ['zora-def2tzvp'],
        'zora-def2-tzvpp': ['zora-def2tzvpp', 'zora-def2tzv-pp', 'zora-def2-tzv-pp'],
        'zora-def2-qzvp': ['zora-def2qzvp'],
        'zora-def2-qzvpp': ['zora-def2qzvpp', 'zora-def2qzv-pp', 'zora-def2-qzv-pp'],

        'ma-def2-msvp': ['ma-def2-msv(p)', 'ma-def2msv(p)', 'ma-def2msvp'],
        'ma-def2-svp': ['ma-def2-sv(p)', 'ma-def2sv(p)', 'ma-def2svp'],
        'ma-def2-tzvp(-f)': ['ma-def2tzvp(-f)'],
        'ma-def2-tzvp': ['ma-def2tzvp'],
        'ma-def2-tzvpp': ['ma-def2tzvpp', 'ma-def2tzv-pp', 'ma-def2-tzv-pp'],
        'ma-def2-qzvp': ['ma-def2qzvp'],
        'ma-def2-qzvpp': ['ma-def2qzvpp', 'ma-def2qzv-pp', 'ma-def2-qzv-pp'],

        'ma-dkh-def2-svp': ['ma-dkh-def2-sv(p)', 'ma-dkh-def2sv(p)', 'ma-dkh-def2svp'],
        'ma-dkh-def2-tzvp(-f)': ['ma-dkh-def2tzvp(-f)'],
        'ma-dkh-def2-tzvp': ['ma-dkh-def2tzvp'],
        'ma-dkh-def2-tzvpp': ['ma-dkh-def2tzvpp', 'ma-dkh-def2tzv-pp', 'ma-dkh-def2-tzv-pp'],
        'ma-dkh-def2-qzvp': ['ma-dkh-def2qzvp'],
        'ma-dkh-def2-qzvpp': ['ma-dkh-def2qzvpp', 'ma-dkh-def2qzv-pp', 'ma-dkh-def2-qzv-pp'],

        'ma-zora-def2-svp': ['ma-zora-def2-sv(p)', 'ma-zora-def2sv(p)', 'ma-zora-def2svp'],
        'ma-zora-def2-tzvp(-f)': ['ma-zora-def2tzvp(-f)'],
        'ma-zora-def2-tzvp': ['ma-zora-def2tzvp'],
        'ma-zora-def2-tzvpp': ['ma-zora-def2tzvpp', 'ma-zora-def2tzv-pp', 'ma-zora-def2-tzv-pp'],
        'ma-zora-def2-qzvpp': ['ma-zora-def2qzvpp', 'ma-zora-def2qzv-pp', 'ma-zora-def2-qzv-pp'],

        'ano-pvdz': [],
        'ano-pvtz': [],
        'ano-pvqz': [],
        'ano-pv5z': [],
        'ano-pv6z': [],

        'aug-ano-pvdz': [],
        'aug-ano-pvtz': [],
        'aug-ano-pvqz': [],
        'aug-ano-pv5z': [],

        'saug-ano-pvdz': [],
        'saug-ano-pvtz': [],
        'saug-ano-pvqz': [],
        'saug-ano-pv5z': [],

        'ano-rcc-dzp': [],
        'ano-rcc-tzp': [],
        'ano-rcc-qzp': [],
        'ano-rcc-full': [],

        'pc-0': ['pc0'],
        'pc-1': ['pc1'],
        'pc-2': ['pc2'],
        'pc-3': ['pc3'],
        'pc-4': ['pc4'],

        'aug-pc-0': ['aug-pc0'],
        'aug-pc-1': ['aug-pc1'],
        'aug-pc-2': ['aug-pc2'],
        'aug-pc-3': ['aug-pc3'],
        'aug-pc-4': ['aug-pc4'],

        'pcj-0': ['pcj0'],
        'pcj-1': ['pcj1'],
        'pcj-2': ['pcj2'],
        'pcj-3': ['pcj3'],
        'pcj-4': ['pcj4'],

        'aug-pcj-0': ['aug-pcj0'],
        'aug-pcj-1': ['aug-pcj1'],
        'aug-pcj-2': ['aug-pcj2'],
        'aug-pcj-3': ['aug-pcj3'],
        'aug-pcj-4': ['aug-pcj4'],

        'pcseg-0': ['pcseg0'],
        'pcseg-1': ['pcseg1'],
        'pcseg-2': ['pcseg2'],
        'pcseg-3': ['pcseg3'],
        'pcseg-4': ['pcseg4'],

        'aug-pcseg-0': ['aug-pcseg0'],
        'aug-pcseg-1': ['aug-pcseg1'],
        'aug-pcseg-2': ['aug-pcseg2'],
        'aug-pcseg-3': ['aug-pcseg3'],
        'aug-pcseg-4': ['aug-pcseg4'],

        'pcsseg-0': ['pcsseg0'],
        'pcsseg-1': ['pcsseg1'],
        'pcsseg-2': ['pcsseg2'],
        'pcsseg-3': ['pcsseg3'],
        'pcsseg-4': ['pcsseg4'],

        'aug-pcsseg-0': ['aug-pcsseg0'],
        'aug-pcsseg-1': ['aug-pcsseg1'],
        'aug-pcsseg-2': ['aug-pcsseg2'],
        'aug-pcsseg-3': ['aug-pcsseg3'],
        'aug-pcsseg-4': ['aug-pcsseg4'],

        'cc-pvdz': [],
        'cc-pvtz': [],
        'cc-pvqz': [],
        'cc-pv5z': [],
        'cc-pv6z': [],

        'aug-cc-pvdz': [],
        'aug-cc-pvtz': [],
        'aug-cc-pvqz': [],
        'aug-cc-pv5z': [],
        'aug-cc-pv6z': [],

        'cc-pvd(+d)z': [],
        'cc-pvt(+d)z': [],
        'cc-pvq(+d)z': [],
        'cc-pv5(+d)z': [],

        'aug-cc-pvtz-j': [],

        'cc-pcvdz': [],
        'cc-pcvtz': [],
        'cc-pcvqz': [],
        'cc-pcv5z': [],
        'cc-pcv6z': [],

        'aug-cc-pcvdz': [],
        'aug-cc-pcvtz': [],
        'aug-cc-pcvqz': [],
        'aug-cc-pcv5z': [],
        'aug-cc-pcv6z': [],

        'cc-pwcvdz': [],
        'cc-pwcvtz': [],
        'cc-pwcvqz': [],
        'cc-pwcv5z': [],

        'aug-cc-pwcvdz': [],
        'aug-cc-pwcvtz': [],
        'aug-cc-pwcvqz': [],
        'aug-cc-pwcv5z': [],

        'cc-pvdz-pp': ['cc-pvdzpp'],
        'cc-pvtz-pp': ['cc-pvtzpp'],
        'cc-pvqz-pp': ['cc-pvqzpp'],
        'cc-pv5z-pp': ['cc-pv5zpp'],

        'aug-cc-pvdz-pp': ['aug-cc-pvdzpp'],
        'aug-cc-pvtz-pp': ['aug-cc-pvtzpp'],
        'aug-cc-pvqz-pp': ['aug-cc-pvqzpp'],
        'aug-cc-pv5z-pp': ['aug-cc-pv5zpp'],

        'cc-pcvdz-pp': ['cc-pcvdzpp'],
        'cc-pcvtz-pp': ['cc-pcvtzpp'],
        'cc-pcvqz-pp': ['cc-pcvqzpp'],
        'cc-pcv5z-pp': ['cc-pcv5zpp'],

        'aug-cc-pcvdz-pp': ['aug-cc-pcvdzpp'],
        'aug-cc-pcvtz-pp': ['aug-cc-pcvtzpp'],
        'aug-cc-pcvqz-pp': ['aug-cc-pcvqzpp'],
        'aug-cc-pcv5z-pp': ['aug-cc-pcv5zpp'],

        'cc-pwvdz-pp': ['cc-pvdzpp'],
        'cc-pwvtz-pp': ['cc-pvtzpp'],
        'cc-pwvqz-pp': ['cc-pvqzpp'],
        'cc-pwv5z-pp': ['cc-pv5zpp'],

        'aug-cc-pwvdz-pp': ['aug-cc-pvdzpp'],
        'aug-cc-pwvtz-pp': ['aug-cc-pvtzpp'],
        'aug-cc-pwvqz-pp': ['aug-cc-pvqzpp'],
        'aug-cc-pwv5z-pp': ['aug-cc-pv5zpp'],

        'cc-pvdz-dk': ['cc-pvdzdk'],
        'cc-pvtz-dk': ['cc-pvtzdk'],
        'cc-pvqz-dk': ['cc-pvqzdk'],
        'cc-pv5z-dk': ['cc-pv5zdk'],

        'aug-cc-pvdz-dk': ['auc-cc-pvdzdk'],
        'aug-cc-pvtz-dk': ['auc-cc-pvtzdk'],
        'aug-cc-pvqz-dk': ['auc-cc-pvqzdk'],
        'aug-cc-pv5z-dk': ['auc-cc-pv5zdk'],

        'cc-pwcvdz-dk': ['cc-pwcvdzdk'],
        'cc-pwcvtz-dk': ['cc-pwcvtzdk'],
        'cc-pwcvqz-dk': ['cc-pwcvqzdk'],
        'cc-pwcv5z-dk': ['cc-pwcv5zdk'],

        'aug-cc-pv(d+d)z': [],
        'aug-cc-pv(t+d)z': [],
        'aug-cc-pv(q+d)z': [],
        'aug-cc-pv(5+d)z': [],
        'aug-cc-pv(6+d)z': [],

        'w1-opt': [],
        'w1-qz': [],

        'aug-cc-pwvdz-dk': ['aug-cc-pvdzdk'],
        'aug-cc-pwvtz-dk': ['aug-cc-pvtzdk'],
        'aug-cc-pwvqz-dk': ['aug-cc-pvqzdk'],
        'aug-cc-pwv5z-dk': ['aug-cc-pv5zdk'],

        'partridge-1': ['partridge1'],
        'partridge-2': ['partridge2'],
        'partridge-3': ['partridge3'],
        'partridge-4': ['partridge4'],

        'ano-sz': [],

        'lanl08': [],
        'lanl08(f)': ['lanl08f'],
        'lanl2dz': [],
        'lanl2tz': [],
        'lanl2tz(f)': ['lanl2tzf'],

        'lanl2mb': [],

        'sapporo-dzp-2012': [],
        'sapporo-tzp-2012': [],
        'sapporo-qzp-2012': [],

        'sapporo-dkh3-dzp-2012': [],
        'sapporo-dkh3-tzp-2012': [],
        'sapporo-dkh3-qzp-2012': [],

        'sarc-dkh-svp': ['sarc-dkh-sv(p)'],
        'sarc-dkh-tzvp': ['sarc-dkh-tzv(p)'],
        'sarc-dkh-tzvpp': ['sarc-dkh-tzv-pp'],

        'sarc2-dkh-qzv': [],
        'sarc2-dkh-qzvp': ['sarc2-dkh-qzv(p)'],
        'sarc2-zora-qzv': [],
        'sarc2-zora-qzvp': ['sarc2-zora-qzv(p)'],

        'd95': [],
        'd95p': ['d95(p)'],

        'epr-ii': ['eprii'],
        'epr-iii': ['epriii'],

        'iglo-ii': ['igloii'],
        'iglo-iii': ['igloiii'],

        'ugbs': [],
        'cp': [],
        'cp(ppp)': ['cp-ppp'],

        'wachters+f': [],

        'cc-pvdz-f12': [],
        'cc-pvtz-f12': [],
        'cc-pvqz-f12': [],

        'cc-pvdz-pp-f12': ['cc-pvdzpp-f12'],
        'cc-pvtz-pp-f12': ['cc-pvtzpp-f12'],
        'cc-pvqz-pp-f12': ['cc-pvqzpp-f12'],

        'cc-pcvdz-f12': [],
        'cc-pcvtz-f12': [],
        'cc-pcvqz-f12': [],

        'd95v': [],
        'shc': [],
        'cep-4g': ['cep4g'],
        'cep-31g': ['cep31g'],
        'cep-121g': ['cep121g'],

        'sdd': [],
        'sddall': [],

        'dgdzvp': [],
        'dgdzvp2': [],
        'gdtzvp': [],
        'cbsb7': [],

    }

SOFTWARE_METHODS = {
        'orca':
                {
                    'hfs': 'HFS',
                    'xalpha': 'XAlpha',
                    'lsd': 'LSD',
                    'vwn5': 'VWN5',
                    'vwn3': 'VWN3',
                    'pwlda': 'PWLDA',
                    'bnull': 'BNULL',
                    'bvwn': 'BVWN',
                    'bp': 'BP',
                    'pw91': 'PW91',
                    'mpwpw': 'mPWPW',
                    'mpwlyp': 'mPWLYP',
                    'blyp': 'BLYP',
                    'gp': 'GP',
                    'glyp': 'GLYP',
                    'pbe': 'PBE',
                    'revpbe': 'revPBE',
                    'rpbe': 'RPBE',
                    'pwp': 'PWP',
                    'olyp': 'OLYP',
                    'opbe': 'OPBE',
                    'xlyp': 'XLYP',
                    'b97-d': 'B97-D',
                    'b97-d3': 'B97-D3',
                    'pw86pbe': 'PW86PBE',
                    'rpw86pbe': 'RPW86PBE',
                    'm06l': 'M06L',
                    'tpss': 'TPSS',
                    'revtpss': 'revTPSS',
                    'scanfunc': 'SCANfunc',
                    'b1lyp' : 'B1LYP',
                    'b3lyp' : 'B3LYP',
                    'b1p' : 'B1P',
                    'b3p' : 'B3P',
                    'g1lyp' : 'G1LYP',
                    'g3lyp' : 'G3LYP',
                    'g1p' : 'G1P',
                    'g3p' : 'G3P',
                    'pbe0' : 'PBE0',
                    'pwp1' : 'PWP1',
                    'm06' : 'M06',
                    'm062x' : 'M062X',
                    'mpw1pw' : 'mPW1PW',
                    'mpw1lyp' : 'mPW1LYP',
                    'pw91_0' : 'PW91_0',
                    'o3lyp' : 'O3LYP',
                    'x3lyp' : 'X3LYP',
                    'pw6b95' : 'PW6B95',
                    'b97' : 'B97',
                    'bhandhlyp' : 'BHANDHLYP',
                    'wb97' : 'wB97',
                    'wb97x' : 'wB97X',
                    'wb97x-d3' : 'wB97X-D3',
                    'wb97x-v' : 'wB97X-V',
                    'wb97x-d3bj' : 'wB97X-D3BJ',
                    'cam-b3lyp' : 'CAM-B3LYP',
                    'lc-blyp' : 'LC-BLYP',
                    'tpssh' : 'TPSSH',
                    'tpss0' : 'TPSS0',
                    'b2plyp' : 'B2PLYP',
                    'mpw2plyp' : 'mPW2PLYP',
                    'b2gp-plyp' : 'B2GP-PLYP',
                    'b2k-plyp' : 'B2K-PLYP',
                    'b2t-plyp' : 'B2T-PLYP',
                    'pwpb95' : 'PWPB95',
                    'wb2plyp' : 'wB2PLYP',
                    'wb2gp-plyp' : 'wB2GP-PLYP',

                    'am1': 'AM1',
                    'pm3': 'PM3',
                    'mndo': 'MNDO',

                    'gfn2-xtb': 'xtb2',
                    'gfn1-xtb': 'xtb1',
                    #'gfn0-xtb': 'xtb0', # Not available in ORCA 5.0.2
                    'gfn-ff': 'xtbff',

                    'hf': 'HF',

                    'mp2': 'MP2',
                    'ri-mp2': 'RI-MP2',

                    'hf-3c': 'HF-3c',
                    'pbeh-3c': 'PBEh-3c',
                    'r2scan-3c': 'r2SCAN-3c',
                    'b97-3c': 'B97-3c',

                    'ccsd': 'CCSD',
                    'ccsd(t)': 'CCSD(T)',
                    'qcisd': 'QCISD',
                    'qcisd(t)': 'QCISD(T)',

                    'lpno-ccsd': 'LPNO-CCSD',
                    'lpno-ccsd(t)': 'LPNOCCSD(T)',
                    'lpno-qcisd': 'LPNO-QCISD',
                    'lpno-qcisd(t)': 'LPNO-QCISD(T)',
                    'dlpno-ccsd': 'DLPNO-CCSD',
                    'dlpno-ccsd(t)': 'DLPNO-CCSD(T)',
                    'dlpno-qcisd': 'DLPNO-QCISD',
                    'dlpno-qcisd(t)': 'DLPNO-QCISD(T)',

                },
       'gaussian':
                {
                    'b97-d': 'B97D',
                    'b97-d3': 'B97D3',
                    'm06l': 'M06L',
                    'm08hx': 'M08HX',
                    'b1lyp' : 'B1LYP',
                    'b3lyp' : 'B3LYP',
                    'b3p86' : 'B3P86',
                    'pbe0' : 'PBE1PBE',
                    'hse06' : 'HSE1PBE',
                    'ohse2pbe' : 'OHSE2PBE',
                    'ohse1pbe' : 'OHSE1PBE',
                    'pbeh1pbe' : 'PBEh1PBE',
                    'b1b95' : 'B1B95',
                    'm05' : 'M05',
                    'm052x' : 'M052X',
                    'm06' : 'M06',
                    'm06hf' : 'M06HF',
                    'm062x' : 'M062X',
                    'o3lyp' : 'O3LYP',
                    'apfd' : 'APFD',
                    'x3lyp' : 'X3LYP',
                    'pw6b95' : 'PW6B95',
                    'pw6b95d3' : 'PW6B95D3',
                    'mpw1pw91' : 'mPW1PW91',
                    'mpw1lyp' : 'mPW1LYP',
                    'mpw1pbe' : 'mPW1PBE',
                    'mpw3pbe' : 'mPW3PBE',
                    'wb97' : 'wB97',
                    'wb97x' : 'wB97X',
                    'wb97x-d' : 'wB97XD',#Gaussian says dispersion model D2, not D3
                    'cam-b3lyp' : 'CAM-B3LYP',
                    'lc-whpbe' : 'LC-wHPBE',
                    'tpssh' : 'TPSSH',
                    'thcthhyb' : 'tHCTHhyb',
                    'bmk' : 'BMK',
                    'hissbpbe' : 'HISSbPBE',
                    'mn15' : 'MN15',
                    'mn15l' : 'MN15L',
                    'm11' : 'M11',
                    'm11l' : 'M11L',
                    'sogga11x' : 'SOGGA11X',
                    'sogga11' : 'SOGGA11',
                    'n12sx' : 'N12SX',
                    'n12' : 'N12',
                    'mn12sx' : 'MN12SX',
                    'mn12l' : 'MN12L',
                    'vsxc' : 'VSXC',
                    'hcth407' : 'HCTH/407',
                    'hcth93' : 'HCTH/93',
                    'hcth147' : 'HCTH/147',
                    'thcth' : 'tHCTH',

                    'am1': 'AM1',
                    'pm3': 'PM3',
                    'pm6': 'PM6',
                    'pm7': 'PM7',

                    'hf': 'HF',
                }

    }

SOFTWARE_BASIS_SETS = {
        'orca':
                {
                    'sto-3g': 'STO-3G',
                    'mini': 'MINI',
                    'minis': 'MINIS',
                    'minix': 'MINIX',
                    'midi': 'MIDI',
                    '3-21g': '3-21G',
                    '3-21gsp': '3-21GSP',
                    '4-22gsp': '4-22GSP',
                    '6-31g': '6-31G',
                    '6-31g(d)': '6-31G(d)',
                    'm6-31g': 'm6-31G',
                    'm6-31g(d)': 'm6-31G(d)',
                    '6-31g(d,p)': '6-31G(d,p)',
                    '6-31g(2d)': '6-31G(2d)',
                    '6-31g(2d,p)': '6-31G(2d,p)',
                    '6-31g(2d,2p)': '6-31G(2d,2p)',
                    '6-31g(2df)': '6-31G(2df)',
                    '6-31g(2df,2p)': '6-31G(2df,2p)',
                    '6-31g(2df,2pd)': '6-31G(2df,2pd)',
                    '6-31+g(d)': '6-31+G(d)',
                    '6-31+g(d,p)': '6-31+G(d,p)',
                    '6-31+g(2d)': '6-31+G(2d)',
                    '6-31+g(2d,p)': '6-31+G(2d,p)',
                    '6-31+g(2d,2p)': '6-31+G(2d,2p)',
                    '6-31+g(2df)': '6-31+G(2df)',
                    '6-31+g(2df,2p)': '6-31+G(2df,2p)',
                    '6-31+g(2df,2pd)': '6-31+G(2df,2pd)',
                    '6-31++g(d,p)': '6-31++G(d,p)',
                    '6-31++g(2d,p)': '6-31++G(2d,p)',
                    '6-31++g(2d,2p)': '6-31++G(2d,2p)',
                    '6-31++g(2df)': '6-31++G(2df)',
                    '6-31++g(2df,2p)': '6-31++G(2df,2p)',
                    '6-31++g(2df,2pd)': '6-31++G(2df,2pd)',
                    '6-311g(d)': '6-311G(d)',
                    '6-311g(d,p)': '6-311G(d,p)',
                    '6-311g(2d)': '6-311G(2d)',
                    '6-311g(2d,p)': '6-311G(2d,p)',
                    '6-311g(2d,2p)': '6-311G(2d,2p)',
                    '6-311g(2df)': '6-311G(2df)',
                    '6-311g(2df,2p)': '6-311G(2df,2p)',
                    '6-311g(2df,2pd)': '6-311G(2df,2pd)',
                    '6-311g(3df)': '6-311G(3df)',
                    '6-311g(3df,3pd)': '6-311G(3df,3pd)',
                    '6-311+g(d)': '6-311+G(d)',
                    '6-311+g(d,p)': '6-311+G(d,p)',
                    '6-311+g(2d)': '6-311+G(2d)',
                    '6-311+g(2d,p)': '6-311+G(2d,p)',
                    '6-311+g(2d,2p)': '6-311+G(2d,2p)',
                    '6-311+g(2df)': '6-311+G(2df)',
                    '6-311+g(2df,2p)': '6-311+G(2df,2p)',
                    '6-311+g(2df,2pd)': '6-311+G(2df,2pd)',
                    '6-311+g(3df)': '6-311+G(3df)',
                    '6-311+g(3df,2p)': '6-311+G(3df,2p)',
                    '6-311+g(3df,3pd)': '6-311+G(3df,3pd)',
                    '6-311++g(d,p)': '6-311++G(d,p)',
                    '6-311++g(2d)': '6-311++G(2d)',
                    '6-311++g(2d,p)': '6-311++G(2d,p)',
                    '6-311++g(2d,2p)': '6-311++G(2d,2p)',
                    '6-311++g(2df)': '6-311++G(2df)',
                    '6-311++g(2df,2p)': '6-311++G(2df,2p)',
                    '6-311++g(2df,2pd)': '6-311++G(2df,2pd)',
                    '6-311++g(3df)': '6-311++G(3df)',
                    '6-311++g(3df,3pd)': '6-311++G(3df,3pd)',
                    'hav(t+d)z': 'haV(t+d)z',
                    'hav(q+d)z': 'haV(q+d)z',
                    'hav(5+d)z': 'haV(5+d)z',
                    'w1-dz': 'W1-DZ',
                    'w1-mtsmall': 'w1-mtsmall',
                    'w1-dz': 'W1-TZ',
                    'sv': 'SV',
                    'svp': 'SVP',
                    'tzv': 'TZV',
                    'tzvp': 'TZVP',
                    'tzvpp': 'TZVPP',
                    'qzvp': 'QZVP',
                    'qzvpp': 'QZVPP',
                    'dkh-svp': 'DKH-SVP',
                    'dkh-tzvp': 'DKH-TZVP',
                    'dkh-tzv-pp': 'DKH-TZV-PP',
                    'dkh-qzvp': 'DKH-QZVP',
                    'dkh-qzv-pp': 'DKH-QZV-PP',
                    'zora-svp': 'ZORA-SVP',
                    'zora-tzvp': 'ZORA-TZVP',
                    'zora-tzv-pp': 'ZORA-TZV-PP',
                    'zora-qzvp': 'ZORA-QZVP',
                    'zora-qzv-pp': 'ZORA-QZV-PP',
                    'def2-msvp': 'Def2-mSVP',
                    'def2-mtzvp': 'Def2-mTZVP',
                    'def2-svp': 'Def2-SVP',
                    'def2-tzvp(-f)': 'Def2-TZVP(-f)',
                    'def2-tzvp': 'Def2-TZVP',
                    'def2-tzvpp': 'Def2-TZVPP',
                    'def2-qzvp': 'Def2-QZVP',
                    'def2-qzvpp': 'Def2-QZVPP',
                    'def2-svpd': 'Def2-SVPD',
                    'def2-tzvpd': 'Def2-TZVPD',
                    'def2-tzvppd': 'Def2-TZVPPD',
                    'def2-qzvpd': 'Def2-QZVPD',
                    'def2-qzvppd': 'Def2-QZVPPD',
                    'dkh-def2-svp': 'DKH-Def2-SVP',
                    'dkh-def2-tzvp(-f)': 'DKH-Def2-TZVP(-f)',
                    'dkh-def2-tzvp': 'DKH-Def2-TZVP',
                    'dkh-def2-tzvpp': 'DKH-Def2-TZVPP',
                    'dkh-def2-qzvp': 'DKH-Def2-QZVP',
                    'dkh-def2-qzvpp': 'DKH-Def2-QZVPP',
                    'zora-def2-svp': 'ZORA-Def2-SVP',
                    'zora-def2-tzvp(-f)': 'ZORA-Def2-TZVP(-f)',
                    'zora-def2-tzvp': 'ZORA-Def2-TZVP',
                    'zora-def2-tzvpp': 'ZORA-Def2-TZVPP',
                    'zora-def2-qzvp': 'ZORA-Def2-QZVP',
                    'zora-def2-qzvpp': 'ZORA-Def2-QZVPP',
                    'ma-def2-msvp': 'ma-Def2-mSVP',
                    'ma-def2-svp': 'ma-Def2-SVP',
                    'ma-def2-tzvp(-f)': 'ma-Def2-TZVP(-f)',
                    'ma-def2-tzvp': 'ma-Def2-TZVP',
                    'ma-def2-tzvpp': 'ma-Def2-TZVPP',
                    'ma-def2-qzvp': 'ma-Def2-QZVP',
                    'ma-def2-qzvpp': 'ma-Def2-QZVPP',
                    'ma-dkh-def2-svp': 'ma-DKH-Def2-SVP',
                    'ma-dkh-def2-tzvp(-f)': 'ma-DKH-Def2-TZVP(-f)',
                    'ma-dkh-def2-tzvp': 'ma-DKH-Def2-TZVP',
                    'ma-dkh-def2-tzvpp': 'ma-DKH-Def2-TZVPP',
                    'ma-dkh-def2-qzvp': 'ma-DKH-def2-QZVP',
                    'ma-dkh-def2-qzvpp': 'ma-DKH-def2-QZVPP',
                    'ma-zora-def2-svp': 'ma-ZORA-def2-SVP',
                    'ma-zora-def2-tzvp(-f)': 'ma-ZORA-Def2-TZVP(-f)',
                    'ma-zora-def2-tzvp': 'ma-ZORA-Def2-TZVP',
                    'ma-zora-def2-tzvpp': 'ma-ZORA-Def2-TZVPP',
                    'ma-zora-def2-qzvpp': 'ma-ZORA-Def2-QZVPP',
                    'ano-pvdz': 'ANO-pVDZ',
                    'ano-pvtz': 'ANO-pVTZ',
                    'ano-pvqz': 'ANO-pVQZ',
                    'ano-pv5z': 'ANO-pV5Z',
                    'ano-pv6z': 'ANO-pV6Z',
                    'aug-ano-pvdz': 'aug-ANO-pVDZ',
                    'aug-ano-pvtz': 'aug-ANO-pVTZ',
                    'aug-ano-pvqz': 'aug-ANO-pVQZ',
                    'aug-ano-pv5z': 'aug-ANO-pV5Z',
                    'saug-ano-pvdz': 'saug-ANO-pVDZ',
                    'saug-ano-pvtz': 'saug-ANO-pVTZ',
                    'saug-ano-pvqz': 'saug-ANO-pVQZ',
                    'saug-ano-pv5z': 'saug-ANO-pV5Z',
                    'ano-rcc-dzp': 'ANO-RCC-DZP',
                    'ano-rcc-tzp': 'ANO-RCC-TZP',
                    'ano-rcc-qzp': 'ANO-RCC-QZP',
                    'ano-rcc-full': 'ANO-RCC-Full',
                    'pc-0': 'pc-0',
                    'pc-1': 'pc-1',
                    'pc-2': 'pc-2',
                    'pc-3': 'pc-3',
                    'pc-4': 'pc-4',
                    'aug-pc-0': 'aug-pc-0',
                    'aug-pc-1': 'aug-pc-1',
                    'aug-pc-2': 'aug-pc-2',
                    'aug-pc-3': 'aug-pc-3',
                    'aug-pc-4': 'aug-pc-4',
                    'pcj-0': 'pcJ-0',
                    'pcj-1': 'pcJ-1',
                    'pcj-2': 'pcJ-2',
                    'pcj-3': 'pcJ-3',
                    'pcj-4': 'pcJ-4',
                    'aug-pcj-0': 'aug-pcJ-0',
                    'aug-pcj-1': 'aug-pcJ-1',
                    'aug-pcj-2': 'aug-pcJ-2',
                    'aug-pcj-3': 'aug-pcJ-3',
                    'aug-pcj-4': 'aug-pcJ-4',
                    'pcseg-0': 'pcseg-0',
                    'pcseg-1': 'pcseg-1',
                    'pcseg-2': 'pcseg-2',
                    'pcseg-3': 'pcseg-3',
                    'pcseg-4': 'pcseg-4',
                    'aug-pcseg-0': 'aug-pcseg-0',
                    'aug-pcseg-1': 'aug-pcseg-1',
                    'aug-pcseg-2': 'aug-pcseg-2',
                    'aug-pcseg-3': 'aug-pcseg-3',
                    'aug-pcseg-4': 'aug-pcseg-4',
                    'pcsseg-0': 'pcSseg-0',
                    'pcsseg-1': 'pcSseg-1',
                    'pcsseg-2': 'pcSseg-2',
                    'pcsseg-3': 'pcSseg-3',
                    'pcsseg-4': 'pcSseg-4',
                    'aug-pcsseg-0': 'aug-pcSseg-0',
                    'aug-pcsseg-1': 'aug-pcSseg-1',
                    'aug-pcsseg-2': 'aug-pcSseg-2',
                    'aug-pcsseg-3': 'aug-pcSseg-3',
                    'aug-pcsseg-4': 'aug-pcSseg-4',
                    'cc-pvdz': 'cc-pVDZ',
                    'cc-pvtz': 'cc-pVTZ',
                    'cc-pvqz': 'cc-pVQZ',
                    'cc-pv5z': 'cc-pV5Z',
                    'cc-pv6z': 'cc-pV6Z',
                    'aug-cc-pvdz': 'aug-cc-pVDZ',
                    'aug-cc-pvtz': 'aug-cc-pVTZ',
                    'aug-cc-pvqz': 'aug-cc-pVQZ',
                    'aug-cc-pv5z': 'aug-cc-pV5Z',
                    'aug-cc-pv6z': 'aug-cc-pV6Z',
                    'cc-pvd(+d)z': 'cc-pVD(+d)Z',
                    'cc-pvt(+d)z': 'cc-pVT(+d)Z',
                    'cc-pvq(+d)z': 'cc-pVQ(+d)Z',
                    'cc-pv5(+d)z': 'cc-pV5(+d)Z',
                    'aug-cc-pvtz-j': 'aug-cc-pVTZ-J',
                    'cc-pcvdz': 'cc-pCVDZ',
                    'cc-pcvtz': 'cc-pCVTZ',
                    'cc-pcvqz': 'cc-pCVQZ',
                    'cc-pcv5z': 'cc-pCV5Z',
                    'cc-pcv6z': 'cc-pCV6Z',
                    'aug-cc-pcvdz': 'aug-cc-pCVDZ',
                    'aug-cc-pcvtz': 'aug-cc-pCVTZ',
                    'aug-cc-pcvqz': 'aug-cc-pCVQZ',
                    'aug-cc-pcv5z': 'aug-cc-pCV5Z',
                    'aug-cc-pcv6z': 'aug-cc-pCV6Z',
                    'cc-pwcvdz': 'cc-pwCVDZ',
                    'cc-pwcvtz': 'cc-pwCVTZ',
                    'cc-pwcvqz': 'cc-pwCVQZ',
                    'cc-pwcv5z': 'cc-pwCV5Z',
                    'aug-cc-pwcvdz': 'aug-cc-pwCVDZ',
                    'aug-cc-pwcvtz': 'aug-cc-pwCVTZ',
                    'aug-cc-pwcvqz': 'aug-cc-pwCVQZ',
                    'aug-cc-pwcv5z': 'aug-cc-pwCV5Z',
                    'cc-pvdz-pp': 'cc-pVDZ-PP',
                    'cc-pvtz-pp': 'cc-pVTZ-PP',
                    'cc-pvqz-pp': 'cc-pVQZ-PP',
                    'cc-pv5z-pp': 'cc-pV5Z-PP',
                    'aug-cc-pvdz-pp': 'aug-cc-pVDZ-PP',
                    'aug-cc-pvtz-pp': 'aug-cc-pVTZ-PP',
                    'aug-cc-pvqz-pp': 'aug-cc-pVQZ-PP',
                    'aug-cc-pv5z-pp': 'aug-cc-pV5Z-PP',
                    'cc-pcvdz-pp': 'cc-pCVDZ-PP',
                    'cc-pcvtz-pp': 'cc-pCVTZ-PP',
                    'cc-pcvqz-pp': 'cc-pCVQZ-PP',
                    'cc-pcv5z-pp': 'cc-pCV5Z-PP',
                    'aug-cc-pcvdz-pp': 'aug-cc-pCVDZ-PP',
                    'aug-cc-pcvtz-pp': 'aug-cc-pCVTZ-PP',
                    'aug-cc-pcvqz-pp': 'aug-cc-pCVQZ-PP',
                    'aug-cc-pcv5z-pp': 'aug-cc-pCV5Z-PP',
                    'cc-pwvdz-pp': 'cc-pwVDZ-PP',
                    'cc-pwvtz-pp': 'cc-pwVTZ-PP',
                    'cc-pwvqz-pp': 'cc-pwVQZ-PP',
                    'cc-pwv5z-pp': 'cc-pwV5Z-PP',
                    'aug-cc-pwvdz-pp': 'aug-cc-pwVDZ-PP',
                    'aug-cc-pwvtz-pp': 'aug-cc-pwVTZ-PP',
                    'aug-cc-pwvqz-pp': 'aug-cc-pwVQZ-PP',
                    'aug-cc-pwv5z-pp': 'aug-cc-pwV5Z-PP',
                    'cc-pvdz-dk': 'cc-pVDZ-DK',
                    'cc-pvtz-dk': 'cc-pVTZ-DK',
                    'cc-pvqz-dk': 'cc-pVQZ-DK',
                    'cc-pv5z-dk': 'cc-pV5Z-DK',
                    'aug-cc-pvdz-dk': 'aug-cc-pVDZ-DK',
                    'aug-cc-pvtz-dk': 'aug-cc-pVTZ-DK',
                    'aug-cc-pvqz-dk': 'aug-cc-pVQZ-DK',
                    'aug-cc-pv5z-dk': 'aug-cc-pV5Z-DK',
                    'cc-pwcvdz-dk': 'cc-pwCVDZ-DK',
                    'cc-pwcvtz-dk': 'cc-pwCVTZ-DK',
                    'cc-pwcvqz-dk': 'cc-pwCVQZ-DK',
                    'cc-pwcv5z-dk': 'cc-pwCV5Z-DK',
                    'aug-cc-pv(d+d)z': 'aug-cc-pV(D+d)Z',
                    'aug-cc-pv(t+d)z': 'aug-cc-pV(T+d)Z',
                    'aug-cc-pv(q+d)z': 'aug-cc-pV(Q+d)Z',
                    'aug-cc-pv(5+d)z': 'aug-cc-pV(5+d)Z',
                    'aug-cc-pv(6+d)z': 'aug-cc-pV(6+d)Z',
                    'w1-opt': 'W1-Opt',
                    'w1-qz': 'W1-QZ',
                    'aug-cc-pwvdz-dk': 'aug-cc-pwVDZ-DK',
                    'aug-cc-pwvtz-dk': 'aug-cc-pwVTZ-DK',
                    'aug-cc-pwvqz-dk': 'aug-cc-pwVQZ-DK',
                    'aug-cc-pwv5z-dk': 'aug-cc-pwV5Z-DK',
                    'partridge-1': 'Partridge-1',
                    'partridge-2': 'Partridge-2',
                    'partridge-3': 'Partridge-3',
                    'partridge-4': 'Partridge-4',
                    'ano-sz': 'ANO-SZ',
                    'lanl08': 'LANL08',
                    'lanl08(f)': 'LANL08(f)',
                    'lanl2dz': 'LANL2DZ',
                    'lanl2tz': 'LANL2TZ',
                    'lanl2tz(f)': 'LANL2TZ(f)',
                    'sapporo-dzp-2012': 'Sapporo-DZP-2012',
                    'sapporo-tzp-2012': 'Sapporo-TZP-2012',
                    'sapporo-qzp-2012': 'Sapporo-QZP-2012',
                    'sapporo-dkh3-dzp-2012': 'Sapporo-DKH3-DZP-2012',
                    'sapporo-dkh3-tzp-2012': 'Sapporo-DKH3-TZP-2012',
                    'sapporo-dkh3-qzp-2012': 'Sapporo-DKH3-QZP-2012',
                    'sarc-dkh-svp': 'SARC-DKH-SVP',
                    'sarc-dkh-tzvp': 'SARC-DKH-TZVP',
                    'sarc-dkh-tzvpp': 'SARC-DKH-TZVPP',
                    'sarc2-dkh-qzv': 'SARC2-DKH-QZV',
                    'sarc2-dkh-qzvp': 'SARC2-DKH-QZVP',
                    'sarc2-zora-qzv': 'SARC2-ZORA-QZV',
                    'sarc2-zora-qzvp': 'SARC2-ZORA-QZVP',
                    'd95': 'D95',
                    'd95p': 'D95p',
                    'epr-ii': 'EPR-II',
                    'epr-iii': 'EPR-III',
                    'iglo-ii': 'IGLO-II',
                    'iglo-iii': 'IGLO-III',
                    'ugbs': 'UGBS',
                    'cp': 'CP',
                    'cp(ppp)': 'CP(PPP)',
                    'wachters+f': 'Wachters+f',
                    'cc-pvdz-f12': 'cc-pVDZ-F12',
                    'cc-pvtz-f12': 'cc-pVTZ-F12',
                    'cc-pvqz-f12': 'cc-pVQZ-F12',
                    'cc-pvdz-pp-f12': 'cc-pVDZ-PP-F12',
                    'cc-pvtz-pp-f12': 'cc-pVTZ-PP-F12',
                    'cc-pvqz-pp-f12': 'cc-pVQZ-PP-F12',
                    'cc-pcvdz-f12': 'cc-pCVDZ-F12',
                    'cc-pcvtz-f12': 'cc-pCVTZ-F12',
                    'cc-pcvqz-f12': 'cc-pCVQZ-F12',
                },
        'gaussian':
                {
                    'sto-3g': 'STO-3G',
                    'midix': 'MidiX',
                    '3-21g': '3-21G',
                    '4-31g': '4-31G',
                    '6-21g': '6-21G',
                    '6-31g': '6-31G',
                    #Missing: 6-31G dagger from Petersson
                    '6-31g(d)': '6-31G(d)',
                    '6-31g(d,p)': '6-31G(d,p)',
                    '6-31g(2d)': '6-31G(2d)',
                    '6-31g(2d,p)': '6-31G(2d,p)',
                    '6-31g(2d,2p)': '6-31G(2d,2p)',
                    '6-31g(2df)': '6-31G(2df)',
                    '6-31g(2df,2p)': '6-31G(2df,2p)',
                    '6-31g(2df,2pd)': '6-31G(2df,2pd)',
                    '6-31+g(d)': '6-31+G(d)',
                    '6-31+g(d,p)': '6-31+G(d,p)',
                    '6-31+g(2d)': '6-31+G(2d)',
                    '6-31+g(2d,p)': '6-31+G(2d,p)',
                    '6-31+g(2d,2p)': '6-31+G(2d,2p)',
                    '6-31+g(2df)': '6-31+G(2df)',
                    '6-31+g(2df,2p)': '6-31+G(2df,2p)',
                    '6-31+g(2df,2pd)': '6-31+G(2df,2pd)',
                    '6-31++g(d,p)': '6-31++G(d,p)',
                    '6-31++g(2d,p)': '6-31++G(2d,p)',
                    '6-31++g(2d,2p)': '6-31++G(2d,2p)',
                    '6-31++g(2df)': '6-31++G(2df)',
                    '6-31++g(2df,2p)': '6-31++G(2df,2p)',
                    '6-31++g(2df,2pd)': '6-31++G(2df,2pd)',
                    '6-311g(d)': '6-311G(d)',
                    '6-311g(d,p)': '6-311G(d,p)',
                    '6-311g(2d)': '6-311G(2d)',
                    '6-311g(2d,p)': '6-311G(2d,p)',
                    '6-311g(2d,2p)': '6-311G(2d,2p)',
                    '6-311g(2df)': '6-311G(2df)',
                    '6-311g(2df,2p)': '6-311G(2df,2p)',
                    '6-311g(2df,2pd)': '6-311G(2df,2pd)',
                    '6-311g(3df)': '6-311G(3df)',
                    '6-311g(3df,3pd)': '6-311G(3df,3pd)',
                    '6-311+g(d)': '6-311+G(d)',
                    '6-311+g(d,p)': '6-311+G(d,p)',
                    '6-311+g(2d)': '6-311+G(2d)',
                    '6-311+g(2d,p)': '6-311+G(2d,p)',
                    '6-311+g(2d,2p)': '6-311+G(2d,2p)',
                    '6-311+g(2df)': '6-311+G(2df)',
                    '6-311+g(2df,2p)': '6-311+G(2df,2p)',
                    '6-311+g(2df,2pd)': '6-311+G(2df,2pd)',
                    '6-311+g(3df)': '6-311+G(3df)',
                    '6-311+g(3df,2p)': '6-311+G(3df,2p)',
                    '6-311+g(3df,3pd)': '6-311+G(3df,3pd)',
                    '6-311++g(d,p)': '6-311++G(d,p)',
                    '6-311++g(2d)': '6-311++G(2d)',
                    '6-311++g(2d,p)': '6-311++G(2d,p)',
                    '6-311++g(2d,2p)': '6-311++G(2d,2p)',
                    '6-311++g(2df)': '6-311++G(2df)',
                    '6-311++g(2df,2p)': '6-311++G(2df,2p)',
                    '6-311++g(2df,2pd)': '6-311++G(2df,2pd)',
                    '6-311++g(3df)': '6-311++G(3df)',
                    '6-311++g(3df,3pd)': '6-311++G(3df,3pd)',
                    'w1-mtsmall': 'MTSmall',
                    'sv': 'SV',
                    'svp': 'SVP',
                    'tzv': 'TZV',
                    'tzvp': 'TZVP',
                    'qzvp': 'QZVP',
                    'def2-sv': 'Def2SV',
                    'def2-svp': 'Def2SVP',
                    'def2-svpp': 'Def2SVPP',
                    'def2-tzv': 'Def2TZV',
                    'def2-tzvp': 'Def2TZVP',
                    'def2-tzvpp': 'Def2TZVPP',
                    'def2-qzv': 'Def2QZV',
                    'def2-qzvp': 'Def2QZVP',
                    'def2-qzvpp': 'Def2QZVPP',
                    'cc-pvdz': 'cc-pVDZ',
                    'cc-pvtz': 'cc-pVTZ',
                    'cc-pvqz': 'cc-pVQZ',
                    'cc-pv5z': 'cc-pV5Z',
                    'cc-pv6z': 'cc-pV6Z',
                    'aug-cc-pvdz': 'aug-cc-pVDZ',
                    'aug-cc-pvtz': 'aug-cc-pVTZ',
                    'aug-cc-pvqz': 'aug-cc-pVQZ',
                    'aug-cc-pv5z': 'aug-cc-pV5Z',
                    'aug-cc-pv6z': 'aug-cc-pV6Z',
                    'aug-cc-pv(d+d)z': 'dAug-cc-pVDZ',#to verify
                    'aug-cc-pv(t+d)z': 'dAug-cc-pVTZ',#to verify
                    'aug-cc-pv(q+d)z': 'dAug-cc-pVQZ',#to verify
                    'aug-cc-pv(5+d)z': 'dAug-cc-pV5Z',#to verify
                    'lanl2mb': 'LANL2MB',
                    'lanl2dz': 'LANL2DZ',
                    'd95': 'D95',
                    'd95v': 'D95V',
                    'epr-ii': 'EPR-II',
                    'epr-iii': 'EPR-III',
                    'ugbs': 'UGBS',#polarization missing
                    'shc': 'SHC',
                    'cep-4g': 'CEP-4G',
                    'cep-31g': 'CEP-31G',
                    'cep-121g': 'CEP-121G',
                    'sdd': 'SDD',
                    'sddall': 'SDDAll',
                    'dgdzvp': 'DGDZVP',
                    'dgdzvp2': 'DGDZVP2',
                    'gdtzvp': 'DGTZVP',
                    'cbsb7': 'cbsb7',
                },
    }

SOFTWARE_SOLVENTS = {
         'gaussian':
                {
                    'water': 'water',
                    'acetonitrile': 'acetonitrile',
                    'methanol': 'methanol',
                    'ethanol': 'ethanol',
                    'isoquinoline': 'isoquinoline',
                    'quinoline': 'quinoline',
                    'chloroform': 'chloroform',
                    'diethylether': 'diethylether',
                    'dichloromethane': 'dichloromethane',
                    'dichloroethane': 'dichloroethane',
                    'carbontetrachloride': 'carbontetrachloride',
                    'benzene': 'benzene',
                    'toluene': 'toluene',
                    'chlorobenzene': 'chlorobenzene',
                    'nitromethane': 'nitromethane',
                    'heptane': 'heptane',
                    'cyclohexane': 'cyclohexane',
                    'aniline': 'aniline',
                    'acetone': 'acetone',
                    'tetrahydrofuran': 'tetrahydrofuran',
                    'dimethylsulfoxide': 'dimethylsulfoxide',
                    'argon': 'argon',
                    'krypton': 'krypton',
                    'xenon': 'xenon',
                    'n-octanol': 'n-octanol',
                    '1,1,1-trichloroethane': '1,1,1-trichloroethane',
                    '1,1,2-trichloroethane': '1,1,2-trichloroethane',
                    '1,2,4-trimethylbenzene': '1,2,4-trimethylbenzene',
                    '1,2-dibromoethane': '1,2-dibromoethane',
                    '1,2-ethanediol': '1,2-ethanediol',
                    '1,4-dioxane': '1,4-dioxane',
                    '1-bromo-2-methylpropane': '1-bromo-2-methylpropane',
                    '1-bromooctane': '1-bromooctane',
                    '1-bromopentane': '1-bromopentane',
                    '1-bromopropane': '1-bromopropane',
                    '1-butanol': '1-butanol',
                    '1-chlorohexane': '1-chlorohexane',
                    '1-chloropentane': '1-chloropentane',
                    '1-chloropropane': '1-chloropropane',
                    '1-decanol': '1-decanol',
                    '1-fluorooctane': '1-fluorooctane',
                    '1-heptanol': '1-heptanol',
                    '1-hexanol': '1-hexanol',
                    '1-hexene': '1-hexene',
                    '1-hexyne': '1-hexyne',
                    '1-iodobutane': '1-iodobutane',
                    '1-iodohexadecane': '1-iodohexadecane',
                    '1-iodopentane': '1-iodopentane',
                    '1-iodopropane': '1-iodopropane',
                    '1-nitropropane': '1-nitropropane',
                    '1-nonanol': '1-nonanol',
                    '1-pentanol': '1-pentanol',
                    '1-pentene': '1-pentene',
                    '1-propanol': '1-propanol',
                    '2,2,2-trifluoroethanol': '2,2,2-trifluoroethanol',
                    '2,2,4-trimethylpentane': '2,2,4-trimethylpentane',
                    '2,4-dimethylpentane': '2,4-dimethylpentane',
                    '2,4-dimethylpyridine': '2,4-dimethylpyridine',
                    '2,6-dimethylpyridine': '2,6-dimethylpyridine',
                    '2-bromopropane': '2-bromopropane',
                    '2-butanol': '2-butanol',
                    '2-chlorobutane': '2-chlorobutane',
                    '2-heptanone': '2-heptanone',
                    '2-hexanone': '2-hexanone',
                    '2-methoxyethanol': '2-methoxyethanol',
                    '2-methyl-1-propanol': '2-methyl-1-propanol',
                    '2-methyl-2-propanol': '2-methyl-2-propanol',
                    '2-methylpentane': '2-methylpentane',
                    '2-methylpyridine': '2-methylpyridine',
                    '2-nitropropane': '2-nitropropane',
                    '2-octanone': '2-octanone',
                    '2-pentanone': '2-pentanone',
                    '2-propanol': '2-propanol',
                    '2-propen-1-ol': '2-propen-1-ol',
                    '3-methylpyridine': '3-methylpyridine',
                    '3-pentanone': '3-pentanone',
                    '4-heptanone': '4-heptanone',
                    '4-methyl-2-pentanone': '4-methyl-2-pentanone',
                    '4-methylpyridine': '4-methylpyridine',
                    '5-nonanone': '5-nonanone',
                    'aceticacid': 'aceticacid',
                    'acetophenone': 'acetophenone',
                    'a-chlorotoluene': 'a-chlorotoluene',
                    'anisole': 'anisole',
                    'benzaldehyde': 'benzaldehyde',
                    'benzonitrile': 'benzonitrile',
                    'benzylalcohol': 'benzylalcohol',
                    'bromobenzene': 'bromobenzene',
                    'bromoethane': 'bromoethane',
                    'bromoform': 'bromoform',
                    'butanal': 'butanal',
                    'butanoicacid': 'butanoicacid',
                    'butanone': 'butanone',
                    'butanonitrile': 'butanonitrile',
                    'butylamine': 'butylamine',
                    'butylethanoate': 'butylethanoate',
                    'carbondisulfide': 'carbondisulfide',
                    'cis-1,2-dimethylcyclohexane': 'cis-1,2-dimethylcyclohexane',
                    'cis-decalin': 'cis-decalin',
                    'cyclohexanone': 'cyclohexanone',
                    'cyclopentane': 'cyclopentane',
                    'cyclopentanol': 'cyclopentanol',
                    'cyclopentanone': 'cyclopentanone',
                    'decalin-mixture': 'decalin-mixture',
                    'dibromomethane': 'dibromomethane',
                    'dibutylether': 'dibutylether',
                    'diethylamine': 'diethylamine',
                    'diethylsulfide': 'diethylsulfide',
                    'diiodomethane': 'diiodomethane',
                    'diisopropylether': 'diisopropylether',
                    'dimethyldisulfide': 'dimethyldisulfide',
                    'diphenylether': 'diphenylether',
                    'dipropylamine': 'dipropylamine',
                    'e-1,2-dichloroethene': 'e-1,2-dichloroethene',
                    'e-2-pentene': 'e-2-pentene',
                    'ethanethiol': 'ethanethiol',
                    'ethylbenzene': 'ethylbenzene',
                    'ethylethanoate': 'ethylethanoate',
                    'ethylmethanoate': 'ethylmethanoate',
                    'ethylphenylether': 'ethylphenylether',
                    'fluorobenzene': 'fluorobenzene',
                    'formamide': 'formamide',
                    'formicacid': 'formicacid',
                    'hexanoicacid': 'hexanoicacid',
                    'iodobenzene': 'iodobenzene',
                    'iodoethane': 'iodoethane',
                    'iodomethane': 'iodomethane',
                    'isopropylbenzene': 'isopropylbenzene',
                    'm-cresol': 'm-cresol',
                    'mesitylene': 'mesitylene',
                    'methylbenzoate': 'methylbenzoate',
                    'methylbutanoate': 'methylbutanoate',
                    'methylcyclohexane': 'methylcyclohexane',
                    'methylethanoate': 'methylethanoate',
                    'methylmethanoate': 'methylmethanoate',
                    'methylpropanoate': 'methylpropanoate',
                    'm-xylene': 'm-xylene',
                    'n-butylbenzene': 'n-butylbenzene',
                    'n-decane': 'n-decane',
                    'n-dodecane': 'n-dodecane',
                    'n-hexadecane': 'n-hexadecane',
                    'n-hexane': 'n-hexane',
                    'nitrobenzene': 'nitrobenzene',
                    'nitroethane': 'nitroethane',
                    'n-methylaniline': 'n-methylaniline',
                    'n-methylformamide-mixture': 'n-methylformamide-mixture',
                    'n,n-dimethylacetamide': 'n,n-dimethylacetamide',
                    'n,n-dimethylformamide': 'n,n-dimethylformamide',
                    'n-nonane': 'n-nonane',
                    'n-octane': 'n-octane',
                    'n-pentadecane': 'n-pentadecane',
                    'n-pentane': 'n-pentane',
                    'n-undecane': 'n-undecane',
                    'o-chlorotoluene': 'o-chlorotoluene',
                    'o-cresol': 'o-cresol',
                    'o-dichlorobenzene': 'o-dichlorobenzene',
                    'o-nitrotoluene': 'o-nitrotoluene',
                    'o-xylene': 'o-xylene',
                    'pentanal': 'pentanal',
                    'pentanoicacid': 'pentanoicacid',
                    'pentylamine': 'pentylamine',
                    'pentylethanoate': 'pentylethanoate',
                    'perfluorobenzene': 'perfluorobenzene',
                    'p-isopropyltoluene': 'p-isopropyltoluene',
                    'propanal': 'propanal',
                    'propanoicacid': 'propanoicacid',
                    'propanonitrile': 'propanonitrile',
                    'propylamine': 'propylamine',
                    'propylethanoate': 'propylethanoate',
                    'p-xylene': 'p-xylene',
                    'pyridine': 'pyridine',
                    'sec-butylbenzene': 'sec-butylbenzene',
                    'tert-butylbenzene': 'tert-butylbenzene',
                    'tetrachloroethene': 'tetrachloroethene',
                    'tetrahydrothiophene-s,s-dioxide': 'tetrahydrothiophene-s,s-dioxide',
                    'tetralin': 'tetralin',
                    'thiophene': 'thiophene',
                    'thiophenol': 'thiophenol',
                    'trans-decalin': 'trans-decalin',
                    'tributylphosphate': 'tributylphosphate',
                    'trichloroethene': 'trichloroethene',
                    'triethylamine': 'triethylamine',
                    'xylene-mixture': 'xylene-mixture',
                    'z-1,2-dichloroethene': 'z-1,2-dichloroethene',
                },
         'orca':
                {
                    'water': 'water',
                    'acetone': 'acetone',
                    'acetonitrile': 'acetonitrile',
                    'ammonia': 'ammonia',
                    'benzene': 'benzene',
                    'carbontetrachloride': 'ccl4',
                    'dichloromethane': 'ch2cl2',
                    'chloroform': 'chloroform',
                    'cyclohexane': 'cyclohexane',
                    'dimethylformamide': 'dmf',
                    'dimethylsulfoxide': 'dmso',
                    'ethanol': 'ethanol',
                    'n-hexane': 'hexane',
                    'methanol': 'methanol',
                    'n-octanol': 'octanol',#CPCM keyword, see calculation_helper.py
                    'pyridine': 'pyridine',
                    'tetrahydrofuran': 'thf',
                    'toluene': 'toluene',

                    '1,1,1-trichloroethane': '1,1,1-trichloroethane',
                    '1,1,2-trichloroethane': '1,1,2-trichloroethane',
                    '1,2,4-trimethylbenzene': '1,2,4-trimethylbenzene',
                    '1,2-dibromoethane': '1,2-dibromoethane',
                    '1,2-dichloroethane': '1,2-dichloroethane',
                    '1,2-ethanediol': '1,2-ethanediol',
                    '1,4-dioxane': '1,4-dioxane',
                    '1-bromo-2-methylpropane': '1-bromo-2-methylpropane',
                    '1-bromooctane': '1-bromooctane',
                    '1-bromopentane': '1-bromopentane',
                    '1-bromopropane': '1-bromopropane',
                    '1-butanol': '1-butanol',
                    '1-chlorohexane': '1-chlorohexane',
                    '1-chloropentane': '1-chloropentane',
                    '1-chloropropane': '1-chloropropane',
                    '1-decanol': '1-decanol',
                    '1-fluorooctane': '1-fluorooctane',
                    '1-heptanol': '1-heptanol',
                    '1-hexanol': '1-hexanol',
                    '1-hexene': '1-hexene',
                    '1-hexyne': '1-hexyne',
                    '1-iodobutane': '1-iodobutane',
                    '1-iodohexadecane': '1-iodohexadecane',
                    '1-iodopentane': '1-iodopentane',
                    '1-iodopropane': '1-iodopropane',
                    '1-nitropropane': '1-nitropropane',
                    '1-nonanol': '1-nonanol',
                    '1-octanol': '1-octanol',
                    '1-pentanol': '1-pentanol',
                    '1-pentene': '1-pentene',
                    '1-propanol': '1-propanol',
                    '2,2,2-trifluoroethanol': '2,2,2-trifluoroethanol',
                    '2,2,4-trimethylpentane': '2,2,4-trimethylpentane',
                    '2,4-dimethylpentane': '2,4-dimethylpentane',
                    '2,4-dimethylpyridine': '2,4-dimethylpyridine',
                    '2,6-dimethylpyridine': '2,6-dimethylpyridine',
                    '2-bromopropane': '2-bromopropane',
                    '2-butanol': '2-butanol',
                    'cyclopentane': 'cyclopentane',
                    'cyclopentanol': 'cyclopentanol',
                    'cyclopentanone': 'cyclopentanone',
                    'decalin (cis/trans mixture)': 'decalin (cis/trans mixture)',
                    'cis-decalin': 'cis-decalin',
                    'n-decane': 'n-decane',
                    'dibromomethane': 'dibromomethane',
                    'dibutylether': 'dibutylether',
                    'o-dichlorobenzene': 'o-dichlorobenzene',
                    'e-1,2-dichloroethene': 'e-1,2-dichloroethene',
                    'z-1,2-dichloroethene': 'z-1,2-dichloroethene',
                    'diethyl sulfide': 'diethyl sulfide',
                    'diethylamine': 'diethylamine',
                    'diiodomethane': 'diiodomethane',
                    'diisopropyl ether': 'diisopropyl ether',
                    'cis-1,2-dimethylcyclohexane': 'cis-1,2-dimethylcyclohexane',
                    'dimethyl disulfide': 'dimethyl disulfide',
                    'n,n-dimethylacetamide': 'n,n-dimethylacetamide',
                    'n,n-dimethylformamide': 'n,n-dimethylformamide',
                    'dimethylsulfoxide': 'dimethylsulfoxide',
                    'diphenylether': 'diphenylether',
                    'dipropylamine': 'dipropylamine',
                    'n-dodecane': 'n-dodecane',
                    'ethanethiol': 'ethanethiol',
                    'ethanol': 'ethanol',
                    'ethyl ethanoate': 'ethyl ethanoate',
                    'ethyl methanoate': 'ethyl methanoate',
                    'ethyl phenyl ether': 'ethyl phenyl ether',
                    'ethylbenzene': 'ethylbenzene',
                    'fluorobenzene': 'fluorobenzene',
                    'formamide': 'formamide',
                    'formic acid': 'formic acid',
                    'n-heptane': 'n-heptane',
                    'n-hexadecane': 'n-hexadecane',
                    'hexanoic acid': 'hexanoic acid',
                    '2-chlorobutane': '2-chlorobutane',
                    '2-heptanone': '2-heptanone',
                    '2-hexanone': '2-hexanone',
                    '2-methoxyethanol': '2-methoxyethanol',
                    '2-methyl-1-propanol': '2-methyl-1-propanol',
                    '2-methyl-2-propanol': '2-methyl-2-propanol',
                    '2-methylpentane': '2-methylpentane',
                    '2-methylpyridine': '2-methylpyridine',
                    '2-nitropropane': '2-nitropropane',
                    '2-octanone': '2-octanone',
                    '2-pentanone': '2-pentanone',
                    '2-propanol': '2-propanol',
                    '2-propen-1-ol': '2-propen-1-ol',
                    'e-2-pentene': 'e-2-pentene',
                    '3-methylpyridine': '3-methylpyridine',
                    '3-pentanone': '3-pentanone',
                    '4-heptanone': '4-heptanone',
                    '4-methyl-2-pentanone': '4-methyl-2-pentanone',
                    '4-methylpyridine': '4-methylpyridine',
                    '5-nonanone': '5-nonanone',
                    'acetic acid': 'acetic acid',
                    ' acetophenone': ' acetophenone',
                    'aniline': 'aniline',
                    'anisole': 'anisole',
                    'benzaldehyde': 'benzaldehyde',
                    'benzonitrile': 'benzonitrile',
                    'benzyl alcohol': 'benzyl alcohol',
                    'bromobenzene': 'bromobenzene',
                    'bromoethane': 'bromoethane',
                    'bromoform': 'bromoform',
                    'butanal': 'butanal',
                    'butanoic acid': 'butanoic acid',
                    'butanone': 'butanone',
                    'butanonitrile': 'butanonitrile',
                    'butyl ethanoate': 'butyl ethanoate',
                    'butylamine': 'butylamine',
                    'n-butylbenzene': 'n-butylbenzene',
                    'sec-butylbenzene': 'sec-butylbenzene',
                    'tert-butylbenzene': 'tert-butylbenzene',
                    'carbon disulfide': 'carbon disulfide',
                    'carbon tetrachloride': 'carbon tetrachloride',
                    ' chlorobenzene': ' chlorobenzene',
                    'chloroform': 'chloroform',
                    'a-chlorotoluene': 'a-chlorotoluene',
                    'o-chlorotoluene': 'o-chlorotoluene',
                    'm-cresol': 'm-cresol',
                    'o-cresol': 'o-cresol',
                    'cyclohexane': 'cyclohexane',
                    'cyclohexanone': 'cyclohexanone',
                    'iodobenzene': 'iodobenzene',
                    'iodoethane': 'iodoethane',
                    'iodomethane': 'iodomethane',
                    'isopropylbenzene': 'isopropylbenzene',
                    'p-isopropyltoluene': 'p-isopropyltoluene',
                    'mesitylene': 'mesitylene',
                    'methyl benzoate': 'methyl benzoate',
                    'methyl butanoate': 'methyl butanoate',
                    'methyl ethanoate': 'methyl ethanoate',
                    'methyl methanoate': 'methyl methanoate',
                    'methyl propanoate': 'methyl propanoate',
                    'n-methylaniline': 'n-methylaniline',
                    'methylcyclohexane': 'methylcyclohexane',
                    'n-methylformamide (e/z mixture)': 'n-methylformamide (e/z mixture)',
                    'nitrobenzene': 'nitrobenzene',
                    'nitroethane': 'nitroethane',
                    'nitromethane': 'nitromethane',
                    'o-nitrotoluene': 'o-nitrotoluene',
                    'n-nonane': 'n-nonane',
                    'n-octane': 'n-octane',
                    'n-pentadecane': 'n-pentadecane',
                    'pentanal': 'pentanal',
                    'n-pentane': 'n-pentane',
                    'pentanoic acid': 'pentanoic acid',
                    'pentyl ethanoate': 'pentyl ethanoate',
                    'pentylamine': 'pentylamine',
                    'perfluorobenzene': 'perfluorobenzene',
                    'propanal': 'propanal',
                    'propanoic acid': 'propanoic acid',
                    'propanonitrile': 'propanonitrile',
                    'propyl ethanoate': 'propyl ethanoate',
                    'propylamine': 'propylamine',
                    'tetrachloroethene': 'tetrachloroethene',
                    'tetrahydrothiophene-s,s-dioxide': 'tetrahydrothiophene-s,s-dioxide',
                    'tetralin': 'tetralin',
                    'thiophene': 'thiophene',
                    'thiophenol': 'thiophenol',
                    'trans-decalin': 'trans-decalin',
                    'tributylphosphate': 'tributylphosphate',
                    'trichloroethene': 'trichloroethene',
                    'triethylamine': 'triethylamine',
                    'n-undecane': 'n-undecane',
                    'xylene (mixture)': 'xylene (mixture)',
                    'm-xylene': 'm-xylene',
                    'o-xylene': 'o-xylene',
                    'p-xylene': 'p-xylene',
                },
        'xtb':
                {
                    'acetone': 'acetone',
                    'acetonitrile': 'acetonitrile',
                    'benzene': 'benzene',
                    'dichloromethane': 'ch2cl2',
                    'chloroform': 'chcl3',
                    'carbon disulfide': 'cs2',
                    'dimethylformamide': 'dmf',
                    'dimethylsulfoxide': 'dmso',
                    'diethylether': 'ether',
                    'water': 'h2o',
                    'methanol': 'methanol',
                    'n-hexane': 'n-hexane',
                    'tetrahydrofuran': 'thf',
                    'toluene': 'toluene',
                },
}

# The two lists below use the "absolute" method identifications.
# See SYN_METHODS for the full list of names and their synonyms.

# These lists contain all the functionals with D3 parameters developed
# by the Grimme group, which are not software specific. As such, the
# parameters might not be available in every QM package (although it could).

# Some functionals were not found in the list of known functionals or synonyms.
# These are indicated with the question mark.
FUNCTIONALS_WITH_DISPERSION_PARAMETERS = {
        'd3': [
                'b1b95',
                'b2gp-plyp',
                'b3lyp',
                'b97-d',
                'bhlyp', # BHandHLYP?
                'blyp',
                'bp86', # ?
                'bpbe', # HISSbPBE?
                'mpwlyp',
                'pbe',
                'pbe0',
                'pw6b95',
                'pwb6k', # ?
                'revpbe',
                'tpss',
                'tpss0',
                'tpssh',
                'bop', # ?
                'mpw1b95', # ?
                'mpwb1k', # ?
                'olyp',
                'opbe',
                'otpss', # ?
                'pbe38', # ?
                'pbesol', # ?
                'revssb', # ?
                'ssb', # ?
                'b3pw91', # ?
                'bmk',
                'cam-b3lyp',
                'lcwpbe', # ?
                'm052x',
                'm05',
                'm062x',
                'm06hf',
                'm06l',
                'm06',
                'hcth120', # ?
                'b2plyp',
                'dsdblyp', # ?
                'ptpss', # ?
                'pwpb95',
                'revpbe0', # ?
                'revpbe38', # ?
                'rpw86pbe',
            ],
        'd3bj': [
                '1b95',
                'b2gp-plyp',
                'b3pw91', # ?
                'bhlyp', # ?
                'bmk',
                'bop', # ?
                'bpbe', # ?
                'cam-b3lyp',
                'lcwpbe', # ?
                'mpw1b95', # ?
                'mpwb1k', # ?
                'mpwlyp',
                'olyp',
                'opbe',
                'otpss', # ?
                'pbe38', # ?
                'pbesol', # ?
                'ptpss', # ?
                'pwb6k', # ?
                'revssb', # ?
                'ssb', # ?
                'tpssh',
                'hcth120', # ?
                'b2plyp',
                'b3lyp',
                'b97-d',
                'blyp',
                'bp86', # ?
                'dsdblyp', # ?
                'pbe0',
                'pbe',
                'pw6b95',
                'pwpb95',
                'revpbe0', # ?
                'revpbe38', # ?
                'revpbe',
                'rpw86pbe',
                'tpss0',
                'tpss',
            ],
}

