"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const os = require("os");
const path = require("path");
const vm = require("vm");
const fs = require("fs-extra");
const inquirer = require("inquirer");
const yargs = require("yargs");
const common_1 = require("../../common");
const inventory = require("../../inventory");
const logging = require("../../logging");
const util_1 = require("../../util");
const macros_1 = require("../macros");
class Command {
    constructor() {
        this.command = 'new [PROJECT-TYPE] [OPTIONS]';
        this.describe = 'Creates a new projen project';
    }
    builder(args) {
        var _a;
        args.positional('PROJECT-TYPE', { describe: 'optional only when --from is used and there is a single project type in the external module', type: 'string' });
        args.option('synth', { type: 'boolean', default: true, desc: 'Synthesize after creating .projenrc.js' });
        args.option('comments', { type: 'boolean', default: true, desc: 'Include commented out options in .projenrc.js (use --no-comments to disable)' });
        args.option('from', { type: 'string', alias: 'f', desc: 'External jsii npm module to create project from. Supports any package spec supported by yarn (such as "my-pack@^2.0")' });
        args.example('projen new awscdk-app-ts', 'Creates a new project of built-in type "awscdk-app-ts"');
        args.example('projen new --from projen-vue@^2', 'Creates a new project from an external module "projen-vue" with the specified version');
        for (const type of inventory.discover()) {
            args.command(type.pjid, (_a = type.docs) !== null && _a !== void 0 ? _a : '', {
                builder: cargs => {
                    var _a, _b, _c;
                    cargs.showHelpOnFail(true);
                    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
                        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean') {
                            continue; // we don't support non-primitive fields as command line options
                        }
                        let desc = [(_c = (_b = option.docs) === null || _b === void 0 ? void 0 : _b.replace(/\ *\.$/, '')) !== null && _c !== void 0 ? _c : ''];
                        const required = !option.optional;
                        let defaultValue;
                        if (option.default && option.default !== 'undefined') {
                            if (!required) {
                                // if the field is not required, just describe the default but don't actually assign a value
                                desc.push(`[default: ${option.default.replace(/^\ *-/, '').replace(/\.$/, '').trim()}]`);
                            }
                            else {
                                // if the field is required and we have a @default, then assign
                                // the value here so it appears in `--help`
                                defaultValue = renderDefault(option.default);
                            }
                        }
                        cargs.option(option.switch, {
                            group: required ? 'Required:' : 'Optional:',
                            type: option.type,
                            description: desc.join(' '),
                            default: defaultValue,
                            required,
                        });
                    }
                    return cargs;
                },
                handler: argv => newProject(process.cwd(), type, argv),
            });
        }
        return args;
    }
    async handler(args) {
        // handle --from which means we want to first install a jsii module and then
        // create a project defined within this module.
        if (args.from) {
            return newProjectFromModule(process.cwd(), args.from, args);
        }
        // project type is defined but was not matched by yargs, so print the list of supported types
        if (args.projectType) {
            console.log(`Invalid project type ${args.projectType}. Supported types:`);
            for (const pjid of inventory.discover().map(x => x.pjid)) {
                console.log(`  ${pjid}`);
            }
            return;
        }
        // Handles the use case that nothing was specified since PROJECT-TYPE is now an optional positional parameter
        yargs.showHelp();
    }
}
/**
 * Creates a new project with defaults.
 *
 * This function creates the project type in-process (with in VM) and calls
 * `.synth()` on it (if `options.synth` is not `false`).
 *
 * At the moment, it also generates a `.projenrc.js` file with the same code
 * that was just executed. In the future, this will also be done by the project
 * type, so we can easily support multiple languages of projenrc.
 */
function createProject(opts) {
    // Default project resolution location
    let mod = '../../index';
    // External projects need to load the module from the modules directory
    if (opts.type.moduleName !== 'projen') {
        try {
            mod = path.dirname(require.resolve(path.join(opts.type.moduleName, 'package.json'), { paths: [process.cwd()] }));
        }
        catch (err) {
            throw new Error(`External project module '${opts.type.moduleName}' could not be resolved.`);
        }
    }
    // pass the FQN of the project type to the project initializer so it can
    // generate the projenrc file.
    opts.params.jsiiFqn = JSON.stringify(opts.type.fqn);
    const newProjectCode = `const project = new ${opts.type.typename}(${renderParams(opts)});`;
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const module = require(mod);
    const ctx = vm.createContext(module);
    process.env.PROJEN_DISABLE_POST = (!opts.post).toString();
    vm.runInContext([
        newProjectCode,
        opts.synth ? 'project.synth();' : '',
    ].join('\n'), ctx);
    const configPath = path.join(opts.dir, common_1.PROJEN_RC);
    if (fs.existsSync(configPath)) {
        logging.error(`Directory ${opts.dir} already contains ${common_1.PROJEN_RC}`);
        process.exit(1);
    }
    const [importName] = opts.type.typename.split('.');
    const lines = [
        `const { ${importName} } = require('${opts.type.moduleName}');`,
        '',
        newProjectCode,
        '',
        'project.synth();',
        '',
    ];
    fs.writeFileSync(configPath, lines.join('\n'));
    logging.info(`Created ${common_1.PROJEN_RC} for ${opts.type.typename}`);
}
function makePadding(paddingLength) {
    return ' '.repeat(paddingLength);
}
/**
 * Prints all parameters that can be used in a project type, alongside their descriptions.
 *
 * Parameters in `params` that aren't undefined are rendered as defaults,
 * while all other parameters are rendered as commented out.
 *
 * @param type Project type
 * @param params Object with parameter default values
 * @param comments Whether to include optional parameters in commented out form
 */
function renderParams(opts) {
    var _a, _b, _c;
    // preprocessing
    const renders = {};
    const optionsWithDefaults = [];
    const optionsByModule = {}; // only options without defaults
    for (const option of opts.type.options) {
        if (option.deprecated) {
            continue;
        }
        const optionName = option.name;
        // skip the JSII FQN option
        if (optionName === 'jsiiFqn') {
            continue;
        }
        let paramRender;
        if (opts.params[optionName] !== undefined) {
            paramRender = `${optionName}: ${opts.params[optionName]},`;
            optionsWithDefaults.push(optionName);
        }
        else {
            const defaultValue = ((_a = option.default) === null || _a === void 0 ? void 0 : _a.startsWith('-')) ? undefined : ((_b = option.default) !== null && _b !== void 0 ? _b : undefined);
            paramRender = `// ${optionName}: ${defaultValue === null || defaultValue === void 0 ? void 0 : defaultValue.replace(/"(.+)"/, '\'$1\'')},`; // single quotes
            const parentModule = option.parent;
            optionsByModule[parentModule] = (_c = optionsByModule[parentModule]) !== null && _c !== void 0 ? _c : [];
            optionsByModule[parentModule].push(option);
        }
        renders[optionName] = paramRender;
    }
    // alphabetize
    const marginSize = Math.max(...Object.values(renders).map(str => str.length));
    optionsWithDefaults.sort();
    for (const parentModule in optionsByModule) {
        optionsByModule[parentModule].sort((o1, o2) => o1.name.localeCompare(o2.name));
    }
    // generate rendering
    const tab = makePadding(2);
    const result = [];
    result.push('{');
    // render options with defaults
    for (const optionName of optionsWithDefaults) {
        result.push(`${tab}${renders[optionName]}`);
    }
    if (result.length > 1) {
        result.push('');
    }
    // render options without defaults
    if (opts.comments) {
        for (const [moduleName, options] of Object.entries(optionsByModule).sort()) {
            result.push(`${tab}/* ${moduleName} */`);
            for (const option of options) {
                const paramRender = renders[option.name];
                result.push(`${tab}${paramRender}${makePadding(marginSize - paramRender.length + 2)}/* ${option.docs} */`);
            }
            result.push('');
        }
    }
    if (result[result.length - 1] === '') {
        result.pop();
    }
    result.push('}');
    return result.join('\n');
}
/**
 * Given a value from "@default", processes macros and returns a stringified
 * (quoted) result.
 *
 * @returns a javascript primitive (could be a string, number or boolean)
 */
function renderDefault(value) {
    var _a;
    return (_a = macros_1.tryProcessMacro(value)) !== null && _a !== void 0 ? _a : JSON.parse(value);
}
/**
 * Converts yargs command line switches to project type props.
 * @param type Project type
 * @param argv Command line switches
 */
function commandLineToProps(type, argv) {
    var _a;
    const props = {};
    // initialize props with default values
    for (const prop of type.options) {
        if (prop.default && prop.default !== 'undefined' && !prop.optional) {
            props[prop.name] = renderDefault(prop.default);
        }
    }
    for (const [arg, value] of Object.entries(argv)) {
        for (const prop of type.options) {
            if (prop.switch === arg) {
                let curr = props;
                const queue = [...prop.path];
                while (true) {
                    const p = queue.shift();
                    if (!p) {
                        break;
                    }
                    if (queue.length === 0) {
                        let val = value;
                        // if this is a string, then single quote it
                        if (val && typeof (val) === 'string') {
                            val = JSON.stringify(val).replace(/"(.+)"/, '\'$1\'');
                        }
                        curr[p] = val;
                    }
                    else {
                        curr[p] = (_a = curr[p]) !== null && _a !== void 0 ? _a : {};
                        curr = curr[p];
                    }
                }
            }
        }
    }
    return props;
}
/**
 * Generates a new project from an external module.
 *
 * @param spec The name of the external module to load
 * @param args Command line arguments (incl. project type)
 */
async function newProjectFromModule(baseDir, spec, args) {
    var _a;
    const specDependencyInfo = yarnAdd(baseDir, spec);
    // collect projects by looking up all .jsii modules in `node_modules`.
    const modulesDir = path.join(baseDir, 'node_modules');
    const modules = fs.readdirSync(modulesDir).map(file => path.join(modulesDir, file));
    const projects = inventory
        .discover(...modules)
        .filter(x => x.moduleName !== 'projen'); // filter built-in project types
    if (projects.length < 1) {
        throw new Error(`No projects found after installing ${spec}. The module must export at least one class which extends projen.Project`);
    }
    const requested = args.projectType;
    const types = projects.map(p => p.pjid);
    // if user did not specify a project type but the module has more than one, we need them to tell us which one...
    if (!requested && projects.length > 1) {
        throw new Error(`Multiple projects found after installing ${spec}: ${types.join(',')}. Please specify a project name.\nExample: npx projen new --from ${spec} ${types[0]}`);
    }
    // if user did not specify a type (and we know we have only one), the select it. otherwise, search by pjid.
    const type = !requested ? projects[0] : projects.find(p => p.pjid === requested);
    if (!type) {
        throw new Error(`Project type ${requested} not found. Found ${types.join(',')}`);
    }
    for (const option of (_a = type.options) !== null && _a !== void 0 ? _a : []) {
        if (option.type !== 'string' && option.type !== 'number' && option.type !== 'boolean') {
            continue; // we don't support non-primitive fields as command line options
        }
        if (option.default && option.default !== 'undefined') {
            if (!option.optional) {
                const defaultValue = renderDefault(option.default);
                args[option.name] = defaultValue;
                args[option.switch] = defaultValue;
            }
        }
    }
    // include a dev dependency for the external module
    await newProject(baseDir, type, args, {
        devDeps: JSON.stringify([specDependencyInfo]),
    });
}
/**
 * Generates a new project.
 * @param type Project type
 * @param args Command line arguments
 * @param additionalProps Additional parameters to include in .projenrc.js
 */
async function newProject(baseDir, type, args, additionalProps) {
    // convert command line arguments to project props using type information
    const props = commandLineToProps(type, args);
    // merge in additional props if specified
    for (const [k, v] of Object.entries(additionalProps !== null && additionalProps !== void 0 ? additionalProps : {})) {
        props[k] = v;
    }
    createProject({
        dir: baseDir,
        type,
        params: props,
        comments: args.comments,
        synth: args.synth,
        post: args.post,
    });
    // interactive git and github setup
    const gitFolder = path.resolve(baseDir, '.git');
    let pushInitialToGithub = false;
    if (!fs.existsSync(gitFolder)) {
        pushInitialToGithub = await askAboutGit(baseDir);
    }
    if (pushInitialToGithub) {
        util_1.exec('git add .', { cwd: baseDir });
        util_1.exec('git commit -m \'Initial commit generated by projen\'', { cwd: baseDir });
        util_1.exec('git branch -M main', { cwd: baseDir });
        util_1.exec('git push --set-upstream origin main', { cwd: baseDir });
    }
}
/**
 * Installs the npm module (through `yarn add`) to node_modules under `projectDir`.
 * @param spec The npm package spec (e.g. foo@^1.2)
 * @returns String info for the project devDeps (e.g. foo@^1.2 or foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
 */
function yarnAdd(baseDir, spec) {
    const packageJsonPath = path.join(baseDir, 'package.json');
    const packageJsonExisted = fs.existsSync(packageJsonPath);
    let dependencyInfo = spec;
    // workaround: yarn fails to extract tgz if it contains '@' in the name, so we
    // create a temp copy called pkg.tgz and install from there.
    // see: https://github.com/yarnpkg/yarn/issues/6339
    if (spec.endsWith('.tgz') && spec.includes('@')) {
        // if user passes in a file spec then we have to specify the project name and the package location
        // (e.g foo@/var/folders/8k/qcw0ls5pv_ph0000gn/T/projen-RYurCw/pkg.tgz)
        const moduleName = spec.split('/').slice(-1)[0].trim().split('@')[0].trim(); // Example: ./cdk-project/dist/js/cdk-project@1.0.0.jsii.tgz
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'projen-'));
        const copy = path.join(tmpdir, 'pkg.tgz');
        fs.copyFileSync(spec, copy);
        spec = copy;
        dependencyInfo = `${moduleName}@${spec}`;
    }
    logging.info(`installing external module ${spec}...`);
    util_1.exec(`yarn add --modules-folder=${baseDir}/node_modules --silent --no-lockfile --dev ${spec}`, { cwd: baseDir });
    // if package.json did not exist before calling yarn add, we should remove it
    // so we can start off clean.
    if (!packageJsonExisted) {
        fs.removeSync(packageJsonPath);
    }
    return dependencyInfo;
}
/**
 * Returns the last path element for use as a repository name default.
 */
function repoName() {
    return path.basename(path.basename(process.cwd()));
}
async function askAboutGit(cwd) {
    logging.info('We notice that you do not have a local git repository.');
    const { setUpGit } = await inquirer.prompt([
        {
            type: 'confirm',
            name: 'setUpGit',
            message: 'Do you want to set that up now?',
        },
    ]);
    if (setUpGit) {
        const { plan } = await inquirer.prompt(githubPlanOptions);
        const { gh, git } = plan;
        if (!git && !gh) {
            util_1.exec('git init', { cwd });
            logging.info('Ok, we have run \'git init\' for you! Have a great day.');
        }
        if (git) {
            const { gitRepoURL } = await inquirer.prompt([
                {
                    type: 'input',
                    name: 'gitRepoURL',
                    message: 'What is the repo? (example: https://github.com/projen/projen)',
                },
            ]);
            util_1.exec('git init', { cwd });
            let formattedGitRepoURL = gitRepoURL;
            if (!gitRepoURL.includes('https')) {
                formattedGitRepoURL = `https://github.com/${gitRepoURL}`;
            }
            util_1.exec(`git remote add origin ${formattedGitRepoURL}`, { cwd });
            logging.info(`Great! We have run 'git init' for you and set the remote to ${formattedGitRepoURL}`);
        }
        if (!git && gh) {
            logging.info('Ok! We will make you a repository on GitHub.');
            const ghCLIPath = util_1.execOrUndefined(`${os.platform() === 'win32' ? 'where' : 'which'} gh`, { cwd });
            if (!ghCLIPath) {
                logging.warn('Looks like you do not have the GitHub CLI installed. Please go to https://cli.github.com/ to install and try again.');
            }
            else {
                const { gitProjectName } = await inquirer.prompt([
                    {
                        type: 'input',
                        name: 'gitProjectName',
                        message: 'What would you like to name it?',
                        default: repoName(),
                    },
                ]);
                logging.info(`Wow! ${gitProjectName} is such a great name!`);
                util_1.exec('git init', { cwd });
                util_1.exec(`gh repo create ${gitProjectName}`, { cwd });
                return true;
            }
        }
    }
    return false;
}
const githubPlanOptions = [
    {
        type: 'list',
        name: 'plan',
        message: 'We\'ll need some more info. Please choose one:',
        choices: [
            {
                value: {
                    git: true,
                },
                name: 'I already have a git repository',
            },
            {
                value: {
                    gh: true,
                    git: false,
                },
                name: 'I don\'t have a git repository and want to make one on GitHub',
            },
            {
                value: {
                    gh: false,
                    git: false,
                },
                name: 'I don\'t have a git repository and I don\'t want to use GitHub',
            },
        ],
    },
];
module.exports = new Command();
//# sourceMappingURL=data:application/json;base64,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