# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/00_core.ipynb (unless otherwise specified).

__all__ = ['xla_imported', 'xla_available_config', 'xla_module_exist', '__getstate__', '__setstate__', 'XLAOptimProxy',
           'DeviceMoverTransform', 'has_affinecoord_tfm', 'has_devicemover_tfm', 'get_last_affinecoord_tfm_idx',
           'insert_batch_tfm', 'setup_input_device_mover', 'XLAOptCallback']

# Internal Cell
#colab
IN_COLAB = True

# Internal Cell
DEBUG = False # set to false for prod release
TRACE = False # set to false for prod release

# Internal Cell
if DEBUG:
    from pdb import set_trace
else:
    from fastcore.imports import noop
    set_trace = noop

# Cell
#hide_output
import importlib
import os
import sys

def xla_imported(): return 'torch_xla' in sys.modules
# currently unused, might be deleted later?
def xla_available_config(): return os.environ.get("XRT_DEVICE_MAP", False) and os.environ.get("XRT_WORKERS", False)
def xla_module_exist(): return importlib.util.find_spec('torch_xla')

# Internal Cell
import warnings
try:
    import torch_xla
except ImportError as e:
    if DEBUG: warnings.warn('TPU environment not available')


# Internal Cell
if not xla_imported():
    from types import SimpleNamespace
    from typing import Union,BinaryIO
    import os
    import pickle
    import torch.cuda
    def fake_opt_step(opt,barrier=False):
        opt.step()
    def fake_device(n=None, devkind=None):
        gpu_available = torch.cuda.is_available()
        return torch.device(torch.cuda.current_device()) if gpu_available else torch.device('cpu')
    def fake_save(obj, f: Union[str, os.PathLike, BinaryIO], master_only=True, global_master=False):
        return torch.save(obj,f,pickle_module=pickle,
                          pickle_protocol=2,
                          _use_new_zipfile_serialization=True)
    xm = SimpleNamespace(
        optimizer_step = fake_opt_step,
        xla_device = fake_device,
        save = fake_save
    )
else:
    import torch_xla.core.xla_model as xm

# Cell
# from fastcore.foundation import GetAttr
# from fastai.optimizer import Optimizer
# from copy import deepcopy

# Right now deciding to patch BaseOptimizer instead of add with a PickableOpt(Optimizer) class like in previous versions
from fastcore.basics import patch_to
from fastai.optimizer import _BaseOptimizer

@patch_to(_BaseOptimizer)
def __getstate__(self):
    # https://github.com/pytorch/pytorch/blob/46b252b83a97bba0926cead050d76fcef129cb6b/torch/optim/optimizer.py#L54
    d = {
            'defaults': self.defaults,
            'state': self.state_dict(),
            'param_groups': self.param_groups,
        }
    return d

@patch_to(_BaseOptimizer)
def __setstate__(self, data):
    # https://github.com/pytorch/pytorch/blob/46b252b83a97bba0926cead050d76fcef129cb6b/torch/optim/optimizer.py#L61
    self.defaults = data['defaults']
    self.load_state_dict(data['state'])
    self.param_groups = data['param_groups']

# Cell
#colab
# import torch_xla.core.xla_model as xm

# Cell
from fastcore.foundation import GetAttr

class XLAOptimProxy(GetAttr):
    _default='opt'
    "Proxy optimizer to override `opt.step` with Pytorch XLA sync method `xm.optimizer_step` "
    def __init__(self,opt, barrier):
        self.opt = opt # because not using PickableOpt(opt) for the moment
        self._barrier = barrier

    def step(self):
        xm.optimizer_step(self.opt,barrier=self._barrier)

    @property
    def barrier(self): return self._barrier
    @barrier.setter
    def barrier(self,v): self._barrier = v

# Cell
from fastcore.transform import DisplayedTransform
from fastcore.basics import store_attr
from torch import Tensor
import torch
class DeviceMoverTransform(DisplayedTransform):
    "Transform to move input to new device and reverse to cpu"
    def __init__(self, device_to, device_from=torch.device('cpu')):
        store_attr('device_to,device_from')
    def encodes(self, o:Tensor):
        return o.to(self.device_to)
    def decodes(self, o:Tensor):
        return o.to(self.device_from)


# Cell
from fastcore.transform import Transform
from fastai.vision.augment import AffineCoordTfm, RandomResizedCropGPU
from fastai.data.core import DataLoaders,DataLoader

def _isAffineCoordTfm(o:Transform):
    return isinstance(o,(AffineCoordTfm,RandomResizedCropGPU))

def _isDeviceMoverTransform(o:Transform):
    return isinstance(o,DeviceMoverTransform)

def has_affinecoord_tfm(dls: DataLoaders) -> bool:
    "returns true if train dataloader has an AffineCoordTfm in the batch_tfms"
    idxs = dls.train.after_batch.fs.argwhere(_isAffineCoordTfm)
    return len(idxs) > 0
def has_devicemover_tfm(dl: DataLoader) -> bool:
    "returns true if train dataloader has a DeviceMoverTransform in the batch_tfms"
    idxs = dl.after_batch.fs.argwhere(_isDeviceMoverTransform)
    return len(idxs) > 0

def get_last_affinecoord_tfm_idx(dl:DataLoader)-> int: # -1 if none
    "returns index of last AffineCoordTfm if it exists, otherwise returns -1"
    idxs = dl.after_batch.fs.argwhere(_isAffineCoordTfm)
    return -1 if len(idxs) == 0 else idxs[-1]


# Cell
def insert_batch_tfm(dl:DataLoader, batch_tfm:Transform, idx:int):
    "adds a batch_tfm in the batch_tfms for the dataloader at idx location"
    dl.after_batch.fs.insert(idx, batch_tfm)


# Cell
from fastai.learner import Learner

def setup_input_device_mover(learn: Learner, new_device):
    "setup batch_tfms to use cpu if dataloader batch_tfms has AffineCoordTfms"
    if not has_affinecoord_tfm(learn.dls):
        learn.dls.device = new_device
        return
    learn.dls.device = None
    if has_devicemover_tfm(learn.dls.train):
        return # skip adding device mover if already added
    dm_tfm = DeviceMoverTransform(new_device)
    for dl in learn.dls.loaders:
        if not has_devicemover_tfm(dl):
            idx = get_last_affinecoord_tfm_idx(dl)
            if DEBUG: print(f'setup device mover dl: {dl} idx: {idx}')
            if idx != -1:
                insert_batch_tfm(dl, dm_tfm, idx+1)


# Cell
from fastai.callback.core import Callback
from fastai.data.core import DataLoaders
from fastai.vision.all import to_device
from fastai.callback.core import TrainEvalCallback
from fastai.learner import Recorder

class XLAOptCallback(Callback):
    'Callback to replace `opt.step` with `xm.optimizer_step(opt)` as required to run on TPU'
    run_after,run_before = TrainEvalCallback,Recorder
    def __init__(self, barrier=True):
        self._barrier = barrier

    def before_fit(self):
        'replace opt with proxy which calls `xm.optimizer_step` instead of `opt.step` and set `dls.device` and model to `xla_device`'
        # set dls device to none so prevent trigger of moving to batch input to XLA device
        # as this move will be done by the DeviceMoverTransform which has been added to the dls after_batch tfms
        if has_affinecoord_tfm(self.dls):
            self.dls.device = None

        if self.learn.opt is not None:
            if not isinstance(self.learn.opt,XLAOptimProxy):
                # force opt to reinitialize its parameters and make sure its parameters
                opt = self.learn.opt
                self.learn.opt = XLAOptimProxy(opt, barrier=self._barrier)

    def after_fit(self):
        'restore original opt '
        if isinstance(self.learn.opt, XLAOptimProxy):
            opt = self.learn.opt.opt
            self.learn.opt = opt
    @property
    def barrier(self): return self._barrier
    @barrier.setter
    def barrier(self,v): self._barrier = v

# Cell
from fastcore.foundation import patch
@patch
def to_xla(self:Learner, new_device=None):
    self.add_cb(XLAOptCallback())
    if new_device is None:
        new_device = xm.xla_device()
    self.model.to(new_device)
    setup_input_device_mover(self, new_device)
    self.opt = None
    return self


# Cell
@patch
def detach_xla(self:Learner):
    self.remove_cb(XLAOptCallback)
    self.dls.device = torch.device('cpu')
    self.model.to(self.dls.device)
    self.opt = None
    return self