# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['lethargy']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'lethargy',
    'version': '2.1.0',
    'description': 'A minimal library to make your option-parsing easier.',
    'long_description': '# Lethargy: Terse & tiny command-line option library\n\n**Lethargy was born out of frustration**, like most of my projects. It gets out of your way as soon as possible to let you get on with the actual logic. No bullshit, no magic, no objects to understand, you just call a function.\n\nI write a lot of small scripts to get my job done faster, and manually working with options is a pain. Existing libraries are extremely verbose or just don\'t feel good to use. _Lethargy is designed to make writing scripts easier and faster, and to reduce effort to maintain them_.\n\n<!-- Note that the spaces here are U+2000 (\'\u2002\') EN QUAD -->\n<!--                 v                                  -->\n- **No boilerplate.**\u2002Headaches are directly proportional to lines of code.\n- **No bloat.**\u2002Small API surface area, very little to learn.\n- **No ambiguity.**\u2002Lethargy raises exceptions instead of getting your code into bad state.\n- **Clear errors.**\u2002Great error messages and context managers for dealing with them.\n- **Flexible.**\u2002You\'re not locked in to any styles or paradigms.\n\nLethargy is completely imperative and is **not** a framework. If you _are_ building a complete CLI or want automatic help commands, you\'re better off using **[Click]** — a fantastic, declarative CLI framework.\n\n[Click]: https://click.palletsprojects.com/en/7.x/\n\n## Installation\n\nYou can use pip to install lethargy. It\'s tiny and only depends on the standard library.\n\n```console\npip install lethargy\n```\n\n## Usage\n\n```python\nimport lethargy\n\n# Accepts the option \'--bytes <int>\'. Show the error nicely if it goes wrong.\nwith lethargy.show_errors():\n    n_bytes = lethargy.take_opt(\'bytes\', 1, int) or 8\n\n# Now the option and value have been removed from lethargy.argv\nwith lethargy.expect(IndexError, reason="Missing required argument: [DIR]"):\n    directory = lethargy.argv[1]\n\n...\n```\n\n## Getting Started\n\nThis is both a tutorial and the documentation. All examples assume you\'ve got `import lethargy` at the top.\n\n###### FLAGS\n\n**Options can be flags.** `True` if present, `False` if not.\n\n```python\n# --debug\ndebug = lethargy.take_opt(\'debug\')\n\nprint(debug)\n```\n\n```console\n$ python example.py --debug\nTrue\n$ python example.py\nFalse\n```\n\n<br>\n\n###### NAMES\n\n**Options can have more than one name.** Instead of a string, use a list of strings. Names are case-sensitive.\n\n```python\n# -v|--verbose\nverbose = lethargy.take_opt([\'v\', \'verbose\'])\n\nprint(verbose)\n```\n\n```console\n$ python example.py -v\nTrue\n$ python example.py --verbose\nTrue\n```\n\n<details>\n<summary align="right">Learn more about option names</summary>\n<br>\n\nOption names are automatically generated. `"use headers"` becomes `--use-headers`, and `"I"` becomes `-I`.\n\nIf you provide an explicit name (starting with a non-alphanumeric character, such as `-`, `/` or `+`), the name is stripped and treated as literal.\n\n```python\n# -Enable\nenabled = lethargy.take_opt(\'-Enable\')\nprint(enabled)\n```\n\n```console\n$ python example.py -Enable\nTrue\n$ python example.py\nFalse\n```\n\nNames are _always_ case sensitive. `-Enable` **≠** `-enable`\n\n```console\n$ python example.py -enable\nFalse\n```\n\n<hr>\n</details>\n\n###### ARGUMENTS\n\n**Options can take arguments, too.** They can take any amount.\n\n```python\n# -o|--output <value>\noutput = lethargy.take_opt([\'o\', \'output\'], 1)\n\nprint(output)\n```\n\n```console\n$ python example.py -o out.txt\nout.txt\n$ python example.py\nNone\n```\n\n<details>\n<summary align="right">Learn more about arguments</summary>\n<br>\n\nIf there are fewer values for the option than the number given, `lethargy.ArgsError` will be raised. See [Error Handling](#error-handling) for how to present this nicely.\n\n```console\n$ python example.py --output\nTraceback (most recent call last):\n  [...]\nlethargy.errors.ArgsError: expected 1 argument for \'-o|--output <value>\', found none\n```\n\n<hr>\n</details>\n\n###### GREEDINESS\n\n**Options can be variadic (greedy).** Use `...` instead of a number to take every value following the option.\n\n```python\n# -i|--ignore [value]...\nignored = lethargy.take_opt([\'i\', \'ignore\'], ...)\n\nfor pattern in ignored:\n    print(pattern)\n```\n\n```console\n$ python example.py --ignore .git .vscode .DS_Store\n.git\n.vscode\n.DS_Store\n$ python example.py --ignore experiments\nexperiments\n$ python example.py\n$ ▏\n```\n\n<details>\n<summary align="right">Learn more about variadic options</summary>\n<br>\n\nBecause variadic options will take every argument, including values that look like other options, you should try and take these last (_after_ taking the fixed-count options).\n\n```console\n$ python example.py --ignore "*.pyc" --exceptions some.pyc\n*.pyc\n--exceptions\nsome.pyc\n```\n\n<hr>\n</details>\n\n###### UNPACKING\n\n**Unpack multiple values into separate variables.** If the option wasn\'t present, they\'ll all be `None`.\n\n```python\n# --name <value> <value> <value>\nfirst, middle, last = lethargy.take_opt(\'name\', 3)\n\nprint(f\'Hi, {first}!\')\n```\n\n```console\n$ python example.py --name Dwight Kurt Schrute\nHi, Dwight!\n$ python example.py\nHi, None!\n```\n\n<br>\n\n###### DEFAULTS\n\n**Set sensible defaults.** Use the `or` keyword and your default value(s).\n\n```python\n# -h|--set-hours <value> <value>\nstart, finish = lethargy.take_opt([\'set hours\', \'h\'], 2) or "9AM", "5PM"\n\nprint(f\'Employee now works {start} to {finish}\')\n```\n\n```console\n$ python example.py\nEmployee works 9AM to 5PM\n$ python example.py --set-hours 8AM 4PM\nEmployee works 8AM to 4PM\n```\n\n<br>\n\n###### TYPES & CONVERSION\n\n**Convert your option\'s values.** Use a function or type as the final argument. Defaults aren\'t converted.\n\n```python\n# --date-ymd <int> <int> <int>\ny, m, d = lethargy.take_opt(\'date ymd\', 3, int) or 1970, 1, 1\n\nfrom datetime import datetime\ndate = datetime(y, m, d)\ndelta = datetime.today() - date\nprint(f\'it has been {delta.days} days since {date}\')\n```\n\n```console\n$ python example.py --date-ymd 1999 10 9\nit has been 7500 days since 1999-10-09 00:00:00\n```\n\n<br>\n\n###### ERROR HANDLING\n\n**Give clear error messages.** Lucky for you, lethargy\'s errors are extremely descriptive.\n\n```python\nwith lethargy.show_errors():\n    n_bytes = lethargy.take_opt(\'bytes\', 1, int) or 8\n    start, end = lethargy.take_opt([\'r\', \'range\'], 2, int) or 0, 10\n```\n\n```console\n$ python example.py --range 20\nExpected 2 arguments for option \'-r|--range <int> <int>\', but found 1 (\'20\')\n$ python example.py --bytes\nExpected 1 argument for option \'--bytes <int>\', but found none\n$ python example.py --bytes wrong\nOption \'--bytes <int>\' received an invalid value: \'wrong\'\n```\n\n<details>\n<summary align="right">Learn more about error handling</summary>\n<br>\n\nCalling `fail()` will exit with status code 1. You can optionally use a message.\n\nLethargy provides two context managers for easier error handling. These share similar behaviour, but are separate to make intent clearer.\n\n> <i>with</i> <code><i>lethargy.</i><b>expect(</b><i>*errors: Exception</i>, <i>reason: Optional[str] = None</i><b>)</b></code>\n\nWhen one of the given exceptions is raised, it calls `fail()` to exit and print the message.\n\n> <i>with</i> <code><i>lethargy.</i><b>show_errors()</b></code>\n\nSame behaviour as `expect`, but specifically for exceptions from lethargy.\n\nNote that exceptions raised during value conversions will be caught by `show_errors()`.\n\nYou can access the original exception that caused a `TransformError` with the `__cause__` attribute (see the Python [Built-in Exceptions] docs).\n\n[Built-in Exceptions]: https://docs.python.org/3/library/exceptions.html\n\n<hr>\n</details>\n\n## Contributing\n\nAny and all contributions are absolutely welcome. Feel free to open an issue or just jump straight to a PR. Let\'s discuss and make this the best it can be! 😄\n\n## License\n\nLethargy is released under the [MIT license](https://github.com/SeparateRecords/lethargy/blob/master/LICENSE).\n',
    'author': 'SeparateRecords',
    'author_email': 'me@rob.ac',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/SeparateRecords/lethargy',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
