"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Absolute class name of the Hadoop `InputFormat` to use when reading table files.
 */
class InputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.InputFormat = InputFormat;
/**
 * InputFormat for Avro files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerInputFormat.html
 */
InputFormat.AVRO = new InputFormat('org.apache.hadoop.hive.ql.io.avro.AvroContainerInputFormat');
/**
 * InputFormat for Cloudtrail Logs.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
InputFormat.CLOUDTRAIL = new InputFormat('com.amazon.emr.cloudtrail.CloudTrailInputFormat');
/**
 * InputFormat for Orc files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcInputFormat.html
 */
InputFormat.ORC = new InputFormat('org.apache.hadoop.hive.ql.io.orc.OrcInputFormat');
/**
 * InputFormat for Parquet files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetInputFormat.html
 */
InputFormat.PARQUET = new InputFormat('org.apache.hadoop.hive.ql.io.parquet.MapredParquetInputFormat');
/**
 * An InputFormat for plain text files. Files are broken into lines. Either linefeed or
 * carriage-return are used to signal end of line. Keys are the position in the file, and
 * values are the line of text.
 * JSON & CSV files are examples of this InputFormat
 *
 * @see https://hadoop.apache.org/docs/stable/api/org/apache/hadoop/mapred/TextInputFormat.html
 */
InputFormat.TEXT = new InputFormat('org.apache.hadoop.mapred.TextInputFormat');
/**
 * Absolute class name of the Hadoop `OutputFormat` to use when writing table files.
 */
class OutputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.OutputFormat = OutputFormat;
/**
 * Writes text data with a null key (value only).
 *
 * @see https://hive.apache.org/javadocs/r2.2.0/api/org/apache/hadoop/hive/ql/io/HiveIgnoreKeyTextOutputFormat.html
 */
OutputFormat.HIVE_IGNORE_KEY_TEXT = new OutputFormat('org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat');
/**
 * OutputFormat for Avro files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/avro/AvroContainerOutputFormat.html
 */
OutputFormat.AVRO = new InputFormat('org.apache.hadoop.hive.ql.io.avro.AvroContainerOutputFormat');
/**
 * OutputFormat for Orc files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcOutputFormat.html
 */
OutputFormat.ORC = new InputFormat('org.apache.hadoop.hive.ql.io.orc.OrcOutputFormat');
/**
 * OutputFormat for Parquet files.
 *
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/MapredParquetOutputFormat.html
 */
OutputFormat.PARQUET = new OutputFormat('org.apache.hadoop.hive.ql.io.parquet.MapredParquetOutputFormat');
/**
 * Serialization library to use when serializing/deserializing (SerDe) table records.
 *
 * @see https://cwiki.apache.org/confluence/display/Hive/SerDe
 */
class SerializationLibrary {
    constructor(className) {
        this.className = className;
    }
}
exports.SerializationLibrary = SerializationLibrary;
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/avro/AvroSerDe.html
 */
SerializationLibrary.AVRO = new SerializationLibrary('org.apache.hadoop.hive.serde2.avro.AvroSerDe');
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
SerializationLibrary.CLOUDTRAIL = new SerializationLibrary('com.amazon.emr.hive.serde.CloudTrailSerde');
/**
 * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
 */
SerializationLibrary.GROK = new SerializationLibrary('com.amazonaws.glue.serde.GrokSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hive/hcatalog/data/JsonSerDe.html
 */
SerializationLibrary.HIVE_JSON = new SerializationLibrary('org.apache.hive.hcatalog.data.JsonSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/lazy/LazySimpleSerDe.html
 */
SerializationLibrary.LAZY_SIMPLE = new SerializationLibrary('org.apache.hadoop.hive.serde2.lazy.LazySimpleSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/OpenCSVSerde.html
 */
SerializationLibrary.OPEN_CSV = new SerializationLibrary('org.apache.hadoop.hive.serde2.OpenCSVSerde');
/**
 * @see https://github.com/rcongiu/Hive-JSON-Serde
 */
SerializationLibrary.OPENX_JSON = new SerializationLibrary('org.openx.data.jsonserde.JsonSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/orc/OrcSerde.html
 */
SerializationLibrary.ORC = new SerializationLibrary('org.apache.hadoop.hive.ql.io.orc.OrcSerde');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/ql/io/parquet/serde/ParquetHiveSerDe.html
 */
SerializationLibrary.PARQUET = new SerializationLibrary('org.apache.hadoop.hive.ql.io.parquet.serde.ParquetHiveSerDe');
/**
 * @see https://hive.apache.org/javadocs/r1.2.2/api/org/apache/hadoop/hive/serde2/RegexSerDe.html
 */
SerializationLibrary.REGEXP = new SerializationLibrary('org.apache.hadoop.hive.serde2.RegexSerDe');
/**
 * Defines the input/output formats and ser/de for a single DataFormat.
 */
class DataFormat {
    constructor(props) {
        this.inputFormat = props.inputFormat;
        this.outputFormat = props.outputFormat;
        this.serializationLibrary = props.serializationLibrary;
    }
}
exports.DataFormat = DataFormat;
/**
 * DataFormat for Apache Web Server Logs. Also works for CloudFront logs
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/apache.html
 */
DataFormat.APACHE_LOGS = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.REGEXP,
});
/**
 * DataFormat for Apache Avro
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/avro.html
 */
DataFormat.AVRO = new DataFormat({
    inputFormat: InputFormat.AVRO,
    outputFormat: OutputFormat.AVRO,
    serializationLibrary: SerializationLibrary.AVRO,
});
/**
 * DataFormat for CloudTrail logs stored on S3
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/cloudtrail.html
 */
DataFormat.CLOUDTRAIL_LOGS = new DataFormat({
    inputFormat: InputFormat.CLOUDTRAIL,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.CLOUDTRAIL,
});
/**
 * DataFormat for CSV Files
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/csv.html
 */
DataFormat.CSV = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.OPEN_CSV,
});
/**
 * Stored as plain text files in JSON format.
 * Uses OpenX Json SerDe for serialization and deseralization.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/json.html
 */
DataFormat.JSON = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.OPENX_JSON,
});
/**
 * DataFormat for Logstash Logs, using the GROK SerDe
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/grok.html
 */
DataFormat.LOGSTASH = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.GROK,
});
/**
 * DataFormat for Apache ORC (Optimized Row Columnar)
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/orc.html
 */
DataFormat.ORC = new DataFormat({
    inputFormat: InputFormat.ORC,
    outputFormat: OutputFormat.ORC,
    serializationLibrary: SerializationLibrary.ORC,
});
/**
 * DataFormat for Apache Parquet
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/parquet.html
 */
DataFormat.PARQUET = new DataFormat({
    inputFormat: InputFormat.PARQUET,
    outputFormat: OutputFormat.PARQUET,
    serializationLibrary: SerializationLibrary.PARQUET,
});
/**
 * DataFormat for TSV (Tab-Separated Values)
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/lazy-simple-serde.html
 */
DataFormat.TSV = new DataFormat({
    inputFormat: InputFormat.TEXT,
    outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT,
    serializationLibrary: SerializationLibrary.LAZY_SIMPLE,
});
//# sourceMappingURL=data:application/json;base64,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