# 🧠 Mem-LLM

[![PyPI version](https://badge.fury.io/py/mem-llm.svg)](https://badge.fury.io/py/mem-llm)
[![Python 3.8+](https://img.shields.io/badge/python-3.8+-blue.svg)](https://www.python.org/downloads/)
[![License: MIT](https://img.shields.io/badge/License-MIT-yellow.svg)](https://opensource.org/licenses/MIT)

**Memory-enabled AI assistant with local LLM support**

Mem-LLM is a powerful Python library that brings persistent memory capabilities to local Large Language Models. Build AI assistants that remember user interactions, manage knowledge bases, and work completely offline with Ollama.

## ✨ Key Features

- 🧠 **Persistent Memory** - Remembers conversations across sessions
- 🤖 **Universal Ollama Support** - Works with ALL Ollama models (Qwen3, DeepSeek, Llama3, Granite, etc.)
- 💾 **Dual Storage Modes** - JSON (simple) or SQLite (advanced) memory backends
- 📚 **Knowledge Base** - Built-in FAQ/support system with categorized entries
- 🎯 **Dynamic Prompts** - Context-aware system prompts that adapt to active features
- 👥 **Multi-User Support** - Separate memory spaces for different users
- 🔧 **Memory Tools** - Search, export, and manage stored memories
- 🎨 **Flexible Configuration** - Personal or business usage modes
- 📊 **Production Ready** - Comprehensive test suite with 34+ automated tests
- 🔒 **100% Local & Private** - No cloud dependencies, your data stays yours

## 🚀 Quick Start

### Installation

```bash
pip install mem-llm
```

### Prerequisites

Install and start [Ollama](https://ollama.ai):

```bash
# Install Ollama (visit https://ollama.ai)
# Then pull a model
ollama pull granite4:tiny-h

# Start Ollama service
ollama serve
```

### Basic Usage

```python
from mem_llm import MemAgent

# Create an agent
agent = MemAgent(model="granite4:tiny-h")

# Set user and chat
agent.set_user("alice")
response = agent.chat("My name is Alice and I love Python!")
print(response)

# Memory persists across sessions
response = agent.chat("What's my name and what do I love?")
print(response)  # Agent remembers: "Your name is Alice and you love Python!"
```

That's it! Just 5 lines of code to get started.

## 📖 Usage Examples

### Multi-User Conversations

```python
from mem_llm import MemAgent

agent = MemAgent()

# User 1
agent.set_user("alice")
agent.chat("I'm a Python developer")

# User 2
agent.set_user("bob")
agent.chat("I'm a JavaScript developer")

# Each user has separate memory
agent.set_user("alice")
response = agent.chat("What do I do?")  # "You're a Python developer"
```

### Advanced Configuration

```python
from mem_llm import MemAgent

# Use SQL database with knowledge base
agent = MemAgent(
    model="qwen3:8b",
    use_sql=True,
    load_knowledge_base=True,
    config_file="config.yaml"
)

# Add knowledge base entry
agent.add_kb_entry(
    category="FAQ",
    question="What are your hours?",
    answer="We're open 9 AM - 5 PM EST, Monday-Friday"
)

# Agent will use KB to answer
response = agent.chat("When are you open?")
```

### Memory Tools

```python
from mem_llm import MemAgent

agent = MemAgent(use_sql=True)
agent.set_user("alice")

# Chat with memory
agent.chat("I live in New York")
agent.chat("I work as a data scientist")

# Search memories
results = agent.search_memories("location")
print(results)  # Finds "New York" memory

# Export all data
data = agent.export_user_data()
print(f"Total memories: {len(data['memories'])}")

# Get statistics
stats = agent.get_memory_stats()
print(f"Users: {stats['total_users']}, Memories: {stats['total_memories']}")
```

### CLI Interface

```bash
# Interactive chat
mem-llm chat

# With specific model
mem-llm chat --model llama3:8b

# Customer service mode
mem-llm customer-service

# Knowledge base management
mem-llm kb add --category "FAQ" --question "How to install?" --answer "Run: pip install mem-llm"
mem-llm kb list
mem-llm kb search "install"
```

## 🎯 Usage Modes

### Personal Mode (Default)
- Single user with JSON storage
- Simple and lightweight
- Perfect for personal projects
- No configuration needed

```python
agent = MemAgent()  # Automatically uses personal mode
```

### Business Mode
- Multi-user with SQL database
- Knowledge base support
- Advanced memory tools
- Requires configuration file

```python
agent = MemAgent(
    config_file="config.yaml",
    use_sql=True,
    load_knowledge_base=True
)
```

## 🔧 Configuration

Create a `config.yaml` file for advanced features:

```yaml
# Usage mode: 'personal' or 'business'
usage_mode: business

# LLM settings
llm:
  model: granite4:tiny-h
  base_url: http://localhost:11434
  temperature: 0.7
  max_tokens: 2000

# Memory settings
memory:
  type: sql  # or 'json'
  db_path: ./data/memory.db
  
# Knowledge base
knowledge_base:
  enabled: true
  kb_path: ./data/knowledge_base.db

# Logging
logging:
  level: INFO
  file: logs/mem_llm.log
```

## 🧪 Supported Models

Mem-LLM works with **ALL Ollama models**, including:

- ✅ **Thinking Models**: Qwen3, DeepSeek, QwQ
- ✅ **Standard Models**: Llama3, Granite, Phi, Mistral
- ✅ **Specialized Models**: CodeLlama, Vicuna, Neural-Chat
- ✅ **Any Custom Model** in your Ollama library

### Model Compatibility Features
- 🔄 Automatic thinking mode detection
- 🎯 Dynamic prompt adaptation
- ⚡ Token limit optimization (2000 tokens)
- 🔧 Automatic retry on empty responses

## 📚 Architecture

```
mem-llm/
├── mem_llm/
│   ├── mem_agent.py           # Main agent class
│   ├── memory_manager.py      # JSON memory backend
│   ├── memory_db.py           # SQL memory backend
│   ├── llm_client.py          # Ollama API client
│   ├── knowledge_loader.py    # Knowledge base system
│   ├── dynamic_prompt.py      # Context-aware prompts
│   ├── memory_tools.py        # Memory management tools
│   ├── config_manager.py      # Configuration handler
│   └── cli.py                 # Command-line interface
└── examples/                  # Usage examples
```

## 🔥 Advanced Features

### Dynamic Prompt System
Prevents hallucinations by only including instructions for enabled features:

```python
agent = MemAgent(use_sql=True, load_knowledge_base=True)
# Agent automatically knows:
# ✅ Knowledge Base is available
# ✅ Memory tools are available
# ✅ SQL storage is active
```

### Knowledge Base Categories
Organize knowledge by category:

```python
agent.add_kb_entry(category="FAQ", question="...", answer="...")
agent.add_kb_entry(category="Technical", question="...", answer="...")
agent.add_kb_entry(category="Billing", question="...", answer="...")
```

### Memory Search & Export
Powerful memory management:

```python
# Search across all memories
results = agent.search_memories("python", limit=5)

# Export everything
data = agent.export_user_data()

# Get insights
stats = agent.get_memory_stats()
```

## 📦 Project Structure

### Core Components
- **MemAgent**: Main interface for building AI assistants
- **MemoryManager**: JSON-based memory storage (simple)
- **SQLMemoryManager**: SQLite-based storage (advanced)
- **OllamaClient**: LLM communication handler
- **KnowledgeLoader**: Knowledge base management

### Optional Features
- **MemoryTools**: Search, export, statistics
- **ConfigManager**: YAML configuration
- **CLI**: Command-line interface

## 🧪 Testing

Run the comprehensive test suite:

```bash
# Install dev dependencies
pip install -r requirements-dev.txt

# Run all tests (34+ automated tests)
cd tests
python run_all_tests.py

# Run specific test
python -m pytest test_mem_agent.py -v
```

### Test Coverage
- ✅ Core imports and dependencies
- ✅ CLI functionality
- ✅ Ollama connection and models
- ✅ JSON memory operations
- ✅ SQL memory operations
- ✅ MemAgent features
- ✅ Configuration management
- ✅ Multi-user scenarios
- ✅ Hallucination detection

## 📝 Examples

The `examples/` directory contains ready-to-run demonstrations:

1. **01_hello_world.py** - Simplest possible example (5 lines)
2. **02_basic_memory.py** - Memory persistence basics
3. **03_multi_user.py** - Multiple users with separate memories
4. **04_customer_service.py** - Real-world customer service scenario
5. **05_knowledge_base.py** - FAQ/support system
6. **06_cli_demo.py** - Command-line interface examples
7. **07_document_config.py** - Configuration from documents

## 🛠️ Development

### Setup Development Environment

```bash
git clone https://github.com/emredeveloper/Mem-LLM.git
cd Mem-LLM
pip install -e .
pip install -r requirements-dev.txt
```

### Running Tests

```bash
pytest tests/ -v --cov=mem_llm
```

### Building Package

```bash
python -m build
twine upload dist/*
```

## 📋 Requirements

### Core Dependencies
- Python 3.8+
- requests>=2.31.0
- pyyaml>=6.0.1
- click>=8.1.0

### Optional Dependencies
- pytest>=7.4.0 (for testing)
- flask>=3.0.0 (for web interface)
- fastapi>=0.104.0 (for API server)

## 🤝 Contributing

Contributions are welcome! Please feel free to submit a Pull Request.

1. Fork the repository
2. Create your feature branch (`git checkout -b feature/AmazingFeature`)
3. Commit your changes (`git commit -m 'Add some AmazingFeature'`)
4. Push to the branch (`git push origin feature/AmazingFeature`)
5. Open a Pull Request

## 📄 License

This project is licensed under the MIT License - see the [LICENSE](LICENSE) file for details.

## 👤 Author

**C. Emre Karataş**
- Email: karatasqemre@gmail.com
- GitHub: [@emredeveloper](https://github.com/emredeveloper)

## 🙏 Acknowledgments

- Built with [Ollama](https://ollama.ai) for local LLM support
- Inspired by the need for privacy-focused AI assistants
- Thanks to all contributors and users

## 📊 Project Status

- **Version**: 1.0.10
- **Status**: Beta (Production Ready)
- **Last Updated**: October 20, 2025

## 🔗 Links

- **PyPI**: https://pypi.org/project/mem-llm/
- **GitHub**: https://github.com/emredeveloper/Mem-LLM
- **Issues**: https://github.com/emredeveloper/Mem-LLM/issues
- **Documentation**: See examples/ directory

## 📈 Roadmap

- [ ] Web UI dashboard
- [ ] REST API server
- [ ] Vector database integration
- [ ] Multi-language support
- [ ] Cloud backup options
- [ ] Advanced analytics

---

**⭐ If you find this project useful, please give it a star on GitHub!**
