"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretTargetAttachment = exports.AttachmentTargetType = exports.Secret = exports.SecretStringValueBeta1 = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * An experimental class used to specify an initial secret value for a Secret.
 * The class wraps a simple string (or JSON representation) in order to provide some safety checks and warnings
 * about the dangers of using plaintext strings as initial secret seed values via CDK/CloudFormation.
 */
class SecretStringValueBeta1 {
    constructor(_secretValue) {
        this._secretValue = _secretValue;
    }
    /**
     * Creates a `SecretStringValueBeta1` from a plaintext value.
     * This approach is inherently unsafe, as the secret value may be visible in your source control repository
     * and will also appear in plaintext in the resulting CloudFormation template, including in the AWS Console or APIs.
     * Usage of this method is discouraged, especially for production workloads.
     */
    static fromUnsafePlaintext(secretValue) { return new SecretStringValueBeta1(secretValue); }
    /**
     * Creates a `SecretValueValueBeta1` from a string value coming from a Token.
     * The intent is to enable creating secrets from references (e.g., `Ref`, `Fn::GetAtt`) from other resources.
     * This might be the direct output of another Construct, or the output of a Custom Resource.
     * This method throws if it determines the input is an unsafe plaintext string.
     *
     * For example:
     * ```ts
     *     // Creates a new IAM user, access and secret keys, and stores the secret access key in a Secret.
     *     const user = new iam.User(this, 'User');
     *     const accessKey = new iam.AccessKey(this, 'AccessKey', { user });
     *     const secretValue = secretsmanager.SecretStringValueBeta1.fromToken(accessKey.secretAccessKey.toString());
     *     new secretsmanager.Secret(this, 'Secret', {
     *       secretStringBeta1: secretValue,
     *     });
     * ```
     *
     * The secret may also be embedded in a string representation of a JSON structure:
     *     const secretValue = secretsmanager.SecretStringValueBeta1.fromToken(JSON.stringify({
     *       username: user.userName,
     *       database: 'foo',
     *       password: accessKey.secretAccessKey.toString(),
     *     }));
     *
     * Note that the value being a Token does *not* guarantee safety. For example, a Lazy-evaluated string
     * (e.g., `Lazy.string({ produce: () => 'myInsecurePassword' }))`) is a Token, but as the output is
     * ultimately a plaintext string, and so insecure.
     *
     * @param secretValueFromToken a secret value coming from a Construct attribute or Custom Resource output
     */
    static fromToken(secretValueFromToken) {
        if (!core_1.Token.isUnresolved(secretValueFromToken)) {
            throw new Error('SecretStringValueBeta1 appears to be plaintext (unsafe) string (or resolved Token); use fromUnsafePlaintext if this is intentional');
        }
        return new SecretStringValueBeta1(secretValueFromToken);
    }
    /** Returns the secret value */
    secretValue() { return this._secretValue; }
}
exports.SecretStringValueBeta1 = SecretStringValueBeta1;
_a = JSII_RTTI_SYMBOL_1;
SecretStringValueBeta1[_a] = { fqn: "@aws-cdk/aws-secretsmanager.SecretStringValueBeta1", version: "1.145.0" };
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    get secretFullArn() { return this.secretArn; }
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'],
            resourceArns: [this.arnForPolicies],
            resource: this,
        });
        const statement = result.principalStatement || result.resourceStatement;
        if (versionStages != null && statement) {
            statement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        const crossAccount = core_1.Token.compareStrings(core_1.Stack.of(this).account, grantee.grantPrincipal.principalAccount || '');
        // Throw if secret is not imported and it's shared cross account and no KMS key is provided
        if (this instanceof Secret && result.resourceStatement && (!this.encryptionKey && crossAccount === core_1.TokenComparison.DIFFERENT)) {
            throw new Error('KMS Key must be provided for cross account access to Secret');
        }
        return result;
    }
    grantWrite(grantee) {
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['secretsmanager:PutSecretValue', 'secretsmanager:UpdateSecret'],
            resourceArns: [this.arnForPolicies],
            resource: this,
        });
        if (this.encryptionKey) {
            // See https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantEncrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        // Throw if secret is not imported and it's shared cross account and no KMS key is provided
        if (this instanceof Secret && result.resourceStatement && !this.encryptionKey) {
            throw new Error('KMS Key must be provided for cross account access to Secret');
        }
        return result;
    }
    get secretValue() {
        return this.secretValueFromJson('');
    }
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    validate() {
        var _d;
        const errors = super.validate();
        errors.push(...((_d = this.policy) === null || _d === void 0 ? void 0 : _d.document.validateForResourcePolicy()) || []);
        return errors;
    }
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
    /**
     * Provides an identifier for this secret for use in IAM policies.
     * If there is a full ARN, this is just the ARN;
     * if we have a partial ARN -- due to either importing by secret name or partial ARN --
     * then we need to add a suffix to capture the full ARN's format.
     */
    get arnForPolicies() {
        return this.secretFullArn ? this.secretFullArn : `${this.secretArn}-??????`;
    }
    /**
     * Attach a target to this secret
     *
     * @param target The target to attach
     * @returns An attached secret
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    constructor(scope, id, props = {}) {
        var _d, _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.secretName,
        });
        this.replicaRegions = [];
        this.autoCreatePolicy = true;
        jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SecretProps(props);
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        if (props.generateSecretString && props.secretStringBeta1) {
            throw new Error('Cannot specify both `generateSecretString` and `secretStringBeta1`.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: (_d = props.generateSecretString) !== null && _d !== void 0 ? _d : (props.secretStringBeta1 ? undefined : {}),
            secretString: (_e = props.secretStringBeta1) === null || _e === void 0 ? void 0 : _e.secretValue(),
            name: this.physicalName,
            replicaRegions: core_1.Lazy.any({ produce: () => this.replicaRegions }, { omitEmptyArray: true }),
        });
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core_1.RemovalPolicy.DESTROY,
        });
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.encryptionKey = props.encryptionKey;
        const parseOwnedSecretName = core_1.FeatureFlags.of(this).isEnabled(cxapi.SECRETS_MANAGER_PARSE_OWNED_SECRET_NAME);
        this.secretName = parseOwnedSecretName
            ? parseSecretNameForOwnedSecret(this, this.secretArn, props.secretName)
            : parseSecretName(this, this.secretArn);
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principal = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        (_f = this.encryptionKey) === null || _f === void 0 ? void 0 : _f.grantEncryptDecrypt(principal);
        (_g = this.encryptionKey) === null || _g === void 0 ? void 0 : _g.grant(principal, 'kms:CreateGrant', 'kms:DescribeKey');
        for (const replica of (_h = props.replicaRegions) !== null && _h !== void 0 ? _h : []) {
            this.addReplicaRegion(replica.region, replica.encryptionKey);
        }
    }
    /** @deprecated use `fromSecretCompleteArn` or `fromSecretPartialArn` */
    static fromSecretArn(scope, id, secretArn) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-secretsmanager.Secret#fromSecretArn", "use `fromSecretCompleteArn` or `fromSecretPartialArn`");
        const attrs = arnIsComplete(secretArn) ? { secretCompleteArn: secretArn } : { secretPartialArn: secretArn };
        return Secret.fromSecretAttributes(scope, id, attrs);
    }
    /** Imports a secret by complete ARN. The complete ARN is the ARN with the Secrets Manager-supplied suffix. */
    static fromSecretCompleteArn(scope, id, secretCompleteArn) {
        return Secret.fromSecretAttributes(scope, id, { secretCompleteArn });
    }
    /** Imports a secret by partial ARN. The partial ARN is the ARN without the Secrets Manager-supplied suffix. */
    static fromSecretPartialArn(scope, id, secretPartialArn) {
        return Secret.fromSecretAttributes(scope, id, { secretPartialArn });
    }
    /**
     * Imports a secret by secret name; the ARN of the Secret will be set to the secret name.
     * A secret with this name must exist in the same account & region.
     * @deprecated use `fromSecretNameV2`
     */
    static fromSecretName(scope, id, secretName) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-secretsmanager.Secret#fromSecretName", "use `fromSecretNameV2`");
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretArn = secretName;
                this.secretName = secretName;
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return undefined; }
            // Overrides the secretArn for grant* methods, where the secretArn must be in ARN format.
            // Also adds a wildcard to the resource name to support the SecretsManager-provided suffix.
            get arnForPolicies() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: this.secretName + '*',
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }(scope, id);
    }
    /**
     * Imports a secret by secret name.
     * A secret with this name must exist in the same account & region.
     * Replaces the deprecated `fromSecretName`.
     */
    static fromSecretNameV2(scope, id, secretName) {
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = undefined;
                this.secretName = secretName;
                this.secretArn = this.partialArn;
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return undefined; }
            // Creates a "partial" ARN from the secret name. The "full" ARN would include the SecretsManager-provided suffix.
            get partialArn() {
                return core_1.Stack.of(this).formatArn({
                    service: 'secretsmanager',
                    resource: 'secret',
                    resourceName: secretName,
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                });
            }
        }(scope, id);
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SecretAttributes(attrs);
        let secretArn;
        let secretArnIsPartial;
        if (attrs.secretArn) {
            if (attrs.secretCompleteArn || attrs.secretPartialArn) {
                throw new Error('cannot use `secretArn` with `secretCompleteArn` or `secretPartialArn`');
            }
            secretArn = attrs.secretArn;
            secretArnIsPartial = false;
        }
        else {
            if ((attrs.secretCompleteArn && attrs.secretPartialArn) ||
                (!attrs.secretCompleteArn && !attrs.secretPartialArn)) {
                throw new Error('must use only one of `secretCompleteArn` or `secretPartialArn`');
            }
            if (attrs.secretCompleteArn && !arnIsComplete(attrs.secretCompleteArn)) {
                throw new Error('`secretCompleteArn` does not appear to be complete; missing 6-character suffix');
            }
            [secretArn, secretArnIsPartial] = attrs.secretCompleteArn ? [attrs.secretCompleteArn, false] : [attrs.secretPartialArn, true];
        }
        return new class extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = secretArn;
                this.secretName = parseSecretName(scope, secretArn);
                this.autoCreatePolicy = false;
            }
            get secretFullArn() { return secretArnIsPartial ? undefined : secretArn; }
        }(scope, id, { environmentFromArn: secretArn });
    }
    /**
     * Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     *
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        jsiiDeprecationWarnings.print("@aws-cdk/aws-secretsmanager.Secret#addTargetAttachment", "use `attach()` instead");
        jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_AttachedSecretOptions(options);
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Adds a replica region for the secret
     *
     * @param region The name of the region
     * @param encryptionKey The customer-managed encryption key to use for encrypting the secret value.
     */
    addReplicaRegion(region, encryptionKey) {
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && !core_1.Token.isUnresolved(region) && region === stack.region) {
            throw new Error('Cannot add the region where this stack is deployed as a replica region.');
        }
        this.replicaRegions.push({
            region,
            kmsKeyId: encryptionKey === null || encryptionKey === void 0 ? void 0 : encryptionKey.keyArn,
        });
    }
}
exports.Secret = Secret;
_b = JSII_RTTI_SYMBOL_1;
Secret[_b] = { fqn: "@aws-cdk/aws-secretsmanager.Secret", version: "1.145.0" };
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    /**
     * A database instance
     *
     * @deprecated use RDS_DB_INSTANCE instead
     */
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * A database cluster
     *
     * @deprecated use RDS_DB_CLUSTER instead
     */
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBProxy
     */
    AttachmentTargetType["RDS_DB_PROXY"] = "AWS::RDS::DBProxy";
    /**
     * AWS::Redshift::Cluster
     */
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    /**
     * AWS::DocDB::DBInstance
     */
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    /**
     * AWS::DocDB::DBCluster
     */
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.autoCreatePolicy = true;
        jsiiDeprecationWarnings._aws_cdk_aws_secretsmanager_SecretTargetAttachmentProps(props);
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType,
        });
        this.encryptionKey = props.secret.encryptionKey;
        this.secretName = props.secret.secretName;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.secretName = parseSecretName(scope, secretTargetAttachmentSecretArn);
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
_c = JSII_RTTI_SYMBOL_1;
SecretTargetAttachment[_c] = { fqn: "@aws-cdk/aws-secretsmanager.SecretTargetAttachment", version: "1.145.0" };
/** Parses the secret name from the ARN. */
function parseSecretName(construct, secretArn) {
    const resourceName = core_1.Stack.of(construct).splitArn(secretArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName;
    if (resourceName) {
        // Can't operate on the token to remove the SecretsManager suffix, so just return the full secret name
        if (core_1.Token.isUnresolved(resourceName)) {
            return resourceName;
        }
        // Secret resource names are in the format `${secretName}-${6-character SecretsManager suffix}`
        // If there is no hyphen (or 6-character suffix) assume no suffix was provided, and return the whole name.
        const lastHyphenIndex = resourceName.lastIndexOf('-');
        const hasSecretsSuffix = lastHyphenIndex !== -1 && resourceName.substr(lastHyphenIndex + 1).length === 6;
        return hasSecretsSuffix ? resourceName.substr(0, lastHyphenIndex) : resourceName;
    }
    throw new Error('invalid ARN format; no secret name provided');
}
/**
 * Parses the secret name from the ARN of an owned secret. With owned secrets we know a few things we don't with imported secrets:
 * - The ARN is guaranteed to be a full ARN, with suffix.
 * - The name -- if provided -- will tell us how many hyphens to expect in the final secret name.
 * - If the name is not provided, we know the format used by CloudFormation for auto-generated names.
 *
 * Note: This is done rather than just returning the secret name passed in by the user to keep the relationship
 * explicit between the Secret and wherever the secretName might be used (i.e., using Tokens).
 */
function parseSecretNameForOwnedSecret(construct, secretArn, secretName) {
    const resourceName = core_1.Stack.of(construct).splitArn(secretArn, core_1.ArnFormat.COLON_RESOURCE_NAME).resourceName;
    if (!resourceName) {
        throw new Error('invalid ARN format; no secret name provided');
    }
    // Secret name was explicitly provided, but is unresolved; best option is to use it directly.
    // If it came from another Secret, it should (hopefully) already be properly formatted.
    if (secretName && core_1.Token.isUnresolved(secretName)) {
        return secretName;
    }
    // If no secretName was provided, the name will be automatically generated by CloudFormation.
    // The autogenerated names have the form of `${logicalID}-${random}`.
    // Otherwise, we can use the existing secretName to determine how to parse the resulting resourceName.
    const secretNameHyphenatedSegments = secretName ? secretName.split('-').length : 2;
    // 2 => [0, 1]
    const segmentIndexes = [...new Array(secretNameHyphenatedSegments)].map((_, i) => i);
    // Create the secret name from the resource name by joining all the known segments together.
    // This should have the effect of stripping the final hyphen and SecretManager suffix.
    return core_1.Fn.join('-', segmentIndexes.map(i => core_1.Fn.select(i, core_1.Fn.split('-', resourceName))));
}
/** Performs a best guess if an ARN is complete, based on if it ends with a 6-character suffix. */
function arnIsComplete(secretArn) {
    return core_1.Token.isUnresolved(secretArn) || /-[a-z0-9]{6}$/i.test(secretArn);
}
//# sourceMappingURL=data:application/json;base64,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