# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pigeonXT_jupyter']

package_data = \
{'': ['*']}

install_requires = \
['ipywidgets>=8.0,<9.0',
 'jupyterlab>=3.3,<4.0',
 'numpy>=1.23,<2.0',
 'pandas>=1.3,<2.0']

setup_kwargs = {
    'name': 'pigeonxt-jupyter',
    'version': '0.7.1',
    'description': 'Quickly annotate data in Jupyter notebooks.',
    'long_description': "# 🐦 pigeonXT - Quickly annotate data in Jupyter Lab\nPigeonXT is an extention to the original [Pigeon](https://github.com/agermanidis/pigeon), created by [Anastasis Germanidis](https://pypi.org/user/agermanidis/).\nPigeonXT is a simple widget that lets you quickly annotate a dataset of\nunlabeled examples from the comfort of your Jupyter notebook.\n\nPigeonXT currently support the following annotation tasks:\n- binary / multi-class classification\n- multi-label classification\n- regression tasks\n- captioning tasks\n\nAnything that can be displayed on Jupyter\n(text, images, audio, graphs, etc.) can be displayed by pigeon\nby providing the appropriate `display_fn` argument.\n\nAdditionally, custom hooks can be attached to each row update (`example_process_fn`),\nor when the annotating task is complete(`final_process_fn`).\n\nThere is a full blog post on the usage of PigeonXT on [Towards Data Science](https://towardsdatascience.com/quickly-label-data-in-jupyter-lab-999e7e455e9e).\n\n### Contributors\n- Anastasis Germanidis\n- Dennis Bakhuis\n- Ritesh Agrawal\n- Deepak Tunuguntla\n- Bram van Es\n\n## Installation\nPigeonXT obviously needs a Jupyter Lab environment. Futhermore, it requires ipywidgets.\nThe widget itself can be installed using pip:\n```bash\n    pip install pigeonXT-jupyter\n```\n\nCurrently, it is much easier to install due to Jupyterlab 3:\nTo run the provided examples in a new environment using Conda:\n```bash\n    conda create --name pigeon python=3.9\n    conda activate pigeon\n    pip install numpy pandas jupyterlab ipywidgets pigeonXT-jupyter\n```\n\nFor an older Jupyterlab or any other trouble, please try the old method:\n```bash\n    conda create --name pigeon python=3.7\n    conda activate pigeon\n    conda install nodejs\n    pip install numpy pandas jupyterlab ipywidgets\n    jupyter nbextension enable --py widgetsnbextension\n    jupyter labextension install @jupyter-widgets/jupyterlab-manager\n\n    pip install pigeonXT-jupyter\n```\n\nStarting Jupyter Lab environment:\n```bash\n    jupyter lab\n```\n\n### Development environment\nI have moved the development environment to Poetry. To create an identical environment use:\n```bash\nconda env create -f environment.yml\nconda activate pigeonxt\npoetry install\npre-commit install\n```\n\n## Examples\nExamples are also provided in the accompanying notebook.\n\n### Binary or multi-class text classification\nCode:\n```python\n    import pandas as pd\n    import pigeonXT as pixt\n\n    annotations = pixt.annotate(\n        ['I love this movie', 'I was really disappointed by the book'],\n        options=['positive', 'negative', 'inbetween']\n    )\n```\n\nPreview:\n![Jupyter notebook multi-class classification](/assets/multiclassexample.png)\n\n### Multi-label text classification\nCode:\n```python\n    import pandas as pd\n    import pigeonXT as pixt\n\n    df = pd.DataFrame([\n        {'example': 'Star wars'},\n        {'example': 'The Positively True Adventures of the Alleged Texas Cheerleader-Murdering Mom'},\n        {'example': 'Eternal Sunshine of the Spotless Mind'},\n        {'example': 'Dr. Strangelove or: How I Learned to Stop Worrying and Love the Bomb'},\n        {'example': 'Killer klowns from outer space'},\n    ])\n\n    labels = ['Adventure', 'Romance', 'Fantasy', 'Science fiction', 'Horror', 'Thriller']\n\n    annotations = pixt.annotate(\n        df,\n        options=labels,\n        task_type='multilabel-classification',\n        buttons_in_a_row=3,\n        reset_buttons_after_click=True,\n        include_next=True,\n        include_back=True,\n    )\n```\n\nPreview:\n![Jupyter notebook multi-label classification](/assets/multilabelexample.png)\n\n### Image classification\nCode:\n```python\n    import pandas as pd\n    import pigeonXT as pixt\n\n    from IPython.display import display, Image\n\n    annotations = pixt.annotate(\n      ['assets/img_example1.jpg', 'assets/img_example2.jpg'],\n      options=['cat', 'dog', 'horse'],\n      display_fn=lambda filename: display(Image(filename))\n    )\n```\n\nPreview:\n![Jupyter notebook multi-label classification](/assets/imagelabelexample.png)\n\n\n### Audio classification\nCode:\n```python\n    import pandas as pd\n    import pigeonXT as pixt\n\n    from IPython.display import Audio\n\n    annotations = pixt.annotate(\n        ['assets/audio_1.mp3', 'assets/audio_2.mp3'],\n        task_type='regression',\n        options=(1,5,1),\n        display_fn=lambda filename: display(Audio(filename, autoplay=True))\n    )\n\n    annotations\n```\n\nPreview:\n![Jupyter notebook multi-label classification](/assets/audiolabelexample.png)\n\n### multi-label text classification with custom hooks\nCode:\n```python\n    import pandas as pd\n    import numpy as np\n\n    from pathlib import Path\n    from pigeonXT import annotate\n\n    df = pd.DataFrame([\n        {'example': 'Star wars'},\n        {'example': 'The Positively True Adventures of the Alleged Texas Cheerleader-Murdering Mom'},\n        {'example': 'Eternal Sunshine of the Spotless Mind'},\n        {'example': 'Dr. Strangelove or: How I Learned to Stop Worrying and Love the Bomb'},\n        {'example': 'Killer klowns from outer space'},\n    ])\n\n    labels = ['Adventure', 'Romance', 'Fantasy', 'Science fiction', 'Horror', 'Thriller']\n    shortLabels = ['A', 'R', 'F', 'SF', 'H', 'T']\n\n    df.to_csv('inputtestdata.csv', index=False)\n\n\n    def setLabels(labels, numClasses):\n        row = np.zeros([numClasses], dtype=np.uint8)\n        row[labels] = 1\n        return row\n\n    def labelPortion(\n        inputFile,\n        labels = ['yes', 'no'],\n        outputFile='output.csv',\n        portionSize=2,\n        textColumn='example',\n        shortLabels=None,\n    ):\n        if shortLabels == None:\n            shortLabels = labels\n\n        out = Path(outputFile)\n        if out.exists():\n            outdf = pd.read_csv(out)\n            currentId = outdf.index.max() + 1\n        else:\n            currentId = 0\n\n        indf = pd.read_csv(inputFile)\n        examplesInFile = len(indf)\n        indf = indf.loc[currentId:currentId + portionSize - 1]\n        actualPortionSize = len(indf)\n        print(f'{currentId + 1} - {currentId + actualPortionSize} of {examplesInFile}')\n        sentences = indf[textColumn].tolist()\n\n        for label in shortLabels:\n            indf[label] = None\n\n        def updateRow(example, selectedLabels):\n            print(example, selectedLabels)\n            labs = setLabels([labels.index(y) for y in selectedLabels], len(labels))\n            indf.loc[indf[textColumn] == example, shortLabels] = labs\n\n        def finalProcessing(annotations):\n            if out.exists():\n                prevdata = pd.read_csv(out)\n                outdata = pd.concat([prevdata, indf]).reset_index(drop=True)\n            else:\n                outdata = indf.copy()\n            outdata.to_csv(out, index=False)\n\n        annotated = annotate(\n            sentences,\n            options=labels,\n            task_type='multilabel-classification',\n            buttons_in_a_row=3,\n            reset_buttons_after_click=True,\n            include_next=False,\n            example_process_fn=updateRow,\n            final_process_fn=finalProcessing\n        )\n        return indf\n\n    def getAnnotationsCountPerlabel(annotations, shortLabels):\n\n        countPerLabel = pd.DataFrame(columns=shortLabels, index=['count'])\n\n        for label in shortLabels:\n            countPerLabel.loc['count', label] = len(annotations.loc[annotations[label] == 1.0])\n\n        return countPerLabel\n\n    def getAnnotationsCountPerlabel(annotations, shortLabels):\n\n        countPerLabel = pd.DataFrame(columns=shortLabels, index=['count'])\n\n        for label in shortLabels:\n            countPerLabel.loc['count', label] = len(annotations.loc[annotations[label] == 1.0])\n\n        return countPerLabel\n\n\n    annotations = labelPortion('inputtestdata.csv',\n                               labels=labels,\n                               shortLabels= shortLabels)\n\n    # counts per label\n    getAnnotationsCountPerlabel(annotations, shortLabels)\n```\n\nPreview:\n![Jupyter notebook multi-label classification](/assets/pigeonhookfunctions.png)\n\n\nThe complete and runnable examples are available in the provided Notebook.\n",
    'author': 'Dennis Bakhuis',
    'author_email': 'pypi@bakhuis.nu',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/dennisbakhuis/pigeonXT',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
