from typing import Dict
from xml.etree import Element, ElementTree

def parse_coeff(element, prefix: str, indices: range) -> str:
    """ helper function"""
    return ' '.join([element.find("%s_%s" % (prefix, str(x))).text for x in indices])

def read_rpc_xml_pleiades(tree: Element) -> Dict[str, str]:
    """Read RPC fields from a parsed XML tree assuming the pleiades, spot-6 XML format
    Also reads the inverse model parameters

    Parameters
    ----------
        tree: parsed XML tree

    Returns
    -------
        dictionary read from the RPC file, or empty dict in case of failure
    """
    m: Dict[str, str] = {}

    # direct model (LOCALIZATION)
    d = tree.find('Rational_Function_Model/Global_RFM/Direct_Model')
    m['LON_NUM_COEFF'] = parse_coeff(d, "SAMP_NUM_COEFF", range(1, 21))
    m['LON_DEN_COEFF'] = parse_coeff(d, "SAMP_DEN_COEFF", range(1, 21))
    m['LAT_NUM_COEFF'] = parse_coeff(d, "LINE_NUM_COEFF", range(1, 21))
    m['LAT_DEN_COEFF'] = parse_coeff(d, "LINE_DEN_COEFF", range(1, 21))
    #m['ERR_BIAS']       = parse_coeff(d, "ERR_BIAS", ['X', 'Y'])


    ## inverse model (PROJECTION)
    i = tree.find('Rational_Function_Model/Global_RFM/Inverse_Model')
    m['SAMP_NUM_COEFF']  = parse_coeff(i, "SAMP_NUM_COEFF", range(1, 21))
    m['SAMP_DEN_COEFF']  = parse_coeff(i, "SAMP_DEN_COEFF", range(1, 21))
    m['LINE_NUM_COEFF']  = parse_coeff(i, "LINE_NUM_COEFF", range(1, 21))
    m['LINE_DEN_COEFF']  = parse_coeff(i, "LINE_DEN_COEFF", range(1, 21))
    m['ERR_BIAS']        = parse_coeff(i, "ERR_BIAS", ['ROW', 'COL'])

    # validity domains
    v = tree.find('Rational_Function_Model/Global_RFM/RFM_Validity')
    #vd = v.find('Direct_Model_Validity_Domain')
    #m.firstRow = float(vd.find('FIRST_ROW').text)
    #m.firstCol = float(vd.find('FIRST_COL').text)
    #m.lastRow  = float(vd.find('LAST_ROW').text)
    #m.lastCol  = float(vd.find('LAST_COL').text)

    #vi = v.find('Inverse_Model_Validity_Domain')
    #m.firstLon = float(vi.find('FIRST_LON').text)
    #m.firstLat = float(vi.find('FIRST_LAT').text)
    #m.lastLon  = float(vi.find('LAST_LON').text)
    #m.lastLat  = float(vi.find('LAST_LAT').text)

    # scale and offset
    # the -1 in line and column offsets is due to Pleiades RPC convention
    # that states that the top-left pixel of an image has coordinates
    # (1, 1)
    m['LINE_OFF'    ] = float(v.find('LINE_OFF').text) - 1
    m['SAMP_OFF'    ] = float(v.find('SAMP_OFF').text) - 1
    m['LAT_OFF'     ] = float(v.find('LAT_OFF').text)
    m['LONG_OFF'    ] = float(v.find('LONG_OFF').text)
    m['HEIGHT_OFF'  ] = float(v.find('HEIGHT_OFF').text)
    m['LINE_SCALE'  ] = float(v.find('LINE_SCALE').text)
    m['SAMP_SCALE'  ] = float(v.find('SAMP_SCALE').text)
    m['LAT_SCALE'   ] = float(v.find('LAT_SCALE').text)
    m['LONG_SCALE'  ] = float(v.find('LONG_SCALE').text)
    m['HEIGHT_SCALE'] = float(v.find('HEIGHT_SCALE').text)

    return m

def read_rpc_xml_pleiades_neo(tree: Element) -> Dict[str, str]:
    """Read RPC fields from a parsed XML tree assuming the pleiades NEO XML format
    Also reads the inverse model parameters

    Parameters
    ----------
        tree: parsed XML tree

    Returns
    -------
        dictionary read from the RPC file, or empty dict in case of failure
    """
    m: Dict[str, str] = {}

    # direct model (LOCALIZATION)
    d = tree.find('Rational_Function_Model/Global_RFM/ImagetoGround_Values')
    m['LON_NUM_COEFF'] = parse_coeff(d, "LON_NUM_COEFF", range(1, 21))
    m['LON_DEN_COEFF'] = parse_coeff(d, "LON_DEN_COEFF", range(1, 21))
    m['LAT_NUM_COEFF'] = parse_coeff(d, "LAT_NUM_COEFF", range(1, 21))
    m['LAT_DEN_COEFF'] = parse_coeff(d, "LAT_DEN_COEFF", range(1, 21))
    #m['ERR_BIAS']       = parse_coeff(d, "ERR_BIAS", ['X', 'Y'])


    ## inverse model (PROJECTION)
    i = tree.find('Rational_Function_Model/Global_RFM/GroundtoImage_Values')
    m['SAMP_NUM_COEFF']  = parse_coeff(i, "SAMP_NUM_COEFF", range(1, 21))
    m['SAMP_DEN_COEFF']  = parse_coeff(i, "SAMP_DEN_COEFF", range(1, 21))
    m['LINE_NUM_COEFF']  = parse_coeff(i, "LINE_NUM_COEFF", range(1, 21))
    m['LINE_DEN_COEFF']  = parse_coeff(i, "LINE_DEN_COEFF", range(1, 21))
    m['ERR_BIAS']        = parse_coeff(i, "ERR_BIAS", ['ROW', 'COL'])

    # validity domains
    v = tree.find('Rational_Function_Model/Global_RFM/RFM_Validity')
    #vd = v.find('Direct_Model_Validity_Domain')
    #m.firstRow = float(vd.find('FIRST_ROW').text)
    #m.firstCol = float(vd.find('FIRST_COL').text)
    #m.lastRow  = float(vd.find('LAST_ROW').text)
    #m.lastCol  = float(vd.find('LAST_COL').text)

    #vi = v.find('Inverse_Model_Validity_Domain')
    #m.firstLon = float(vi.find('FIRST_LON').text)
    #m.firstLat = float(vi.find('FIRST_LAT').text)
    #m.lastLon  = float(vi.find('LAST_LON').text)
    #m.lastLat  = float(vi.find('LAST_LAT').text)

    # scale and offset
    # the -1 in line and column offsets is due to Pleiades RPC convention
    # that states that the top-left pixel of an image has coordinates
    # (1, 1)
    m['LINE_OFF'    ] = float(v.find('LINE_OFF').text) - 1
    m['SAMP_OFF'    ] = float(v.find('SAMP_OFF').text) - 1
    m['LAT_OFF'     ] = float(v.find('LAT_OFF').text)
    m['LONG_OFF'    ] = float(v.find('LONG_OFF').text)
    m['HEIGHT_OFF'  ] = float(v.find('HEIGHT_OFF').text)
    m['LINE_SCALE'  ] = float(v.find('LINE_SCALE').text)
    m['SAMP_SCALE'  ] = float(v.find('SAMP_SCALE').text)
    m['LAT_SCALE'   ] = float(v.find('LAT_SCALE').text)
    m['LONG_SCALE'  ] = float(v.find('LONG_SCALE').text)
    m['HEIGHT_SCALE'] = float(v.find('HEIGHT_SCALE').text)

    return m


def read_rpc_xml_worldview(tree: Element) -> Dict[str, str]:
    """Read RPC fields from a parsed XML tree assuming the worldview XML format

    Parameters
    ----------
        tree: parsed XML tree

    Returns
    -------
        dictionary read from the RPC file, or empty dict in case of failure
    """
    m: Dict[str, str] = {}

    # inverse model (PROJECTION)
    im = tree.find('RPB/IMAGE')
    l = im.find('LINENUMCOEFList/LINENUMCOEF')
    m['LINE_NUM_COEFF'] =  l.text
    l = im.find('LINEDENCOEFList/LINEDENCOEF')
    m['LINE_DEN_COEFF'] =  l.text
    l = im.find('SAMPNUMCOEFList/SAMPNUMCOEF')
    m['SAMP_NUM_COEFF'] =  l.text
    l = im.find('SAMPDENCOEFList/SAMPDENCOEF')
    m['SAMP_DEN_COEFF'] =  l.text
    m['ERR_BIAS'] = float(im.find('ERRBIAS').text)

    # scale and offset
    m['LINE_OFF'    ] = float(im.find('LINEOFFSET').text)
    m['SAMP_OFF'    ] = float(im.find('SAMPOFFSET').text)
    m['LAT_OFF'     ] = float(im.find('LATOFFSET').text)
    m['LONG_OFF'    ] = float(im.find('LONGOFFSET').text)
    m['HEIGHT_OFF'  ] = float(im.find('HEIGHTOFFSET').text)

    m['LINE_SCALE'  ] = float(im.find('LINESCALE').text)
    m['SAMP_SCALE'  ] = float(im.find('SAMPSCALE').text)
    m['LAT_SCALE'   ] = float(im.find('LATSCALE').text)
    m['LONG_SCALE'  ] = float(im.find('LONGSCALE').text)
    m['HEIGHT_SCALE'] = float(im.find('HEIGHTSCALE').text)

#    # image dimensions
#    m.lastRow = int(tree.find('IMD/NUMROWS').text)
#    m.lastCol = int(tree.find('IMD/NUMCOLUMNS').text)

    return m

def read_rpc_xml(rpc_content: str) -> Dict[str, str]:
    """Read RPC file assuming the XML format and determine whether it's a pleiades, spot-6 or worldview image

    Parameters
    ----------
        rpc_content -- content of RPC sidecar file path read as a string (XML format)
    Returns:
        dictionary read from the RPC file
    Raises:
        NotImplementedError: if the file format is not handled (the expected keys are not found)
    """

    # parse the xml file content
    tree = ElementTree.fromstring(rpc_content)

    # determine wether it's a pleiades, spot-6 or worldview image
    a = tree.find('Metadata_Identification/METADATA_PROFILE') # PHR_SENSOR
    b = tree.find('IMD/IMAGE/SATID') # WorldView
    parsed_rpc = None
    if a is not None:
        if a.text in ['PHR_SENSOR', 'S6_SENSOR', 'S7_SENSOR']:
            parsed_rpc = read_rpc_xml_pleiades(tree)
        elif a.text in ['PNEO_SENSOR']:
            parsed_rpc = read_rpc_xml_pleiades_neo(tree)
    elif b is not None:
        if b.text == 'WV02' or b.text == 'WV01' or b.text == 'WV03':
            parsed_rpc = read_rpc_xml_worldview(tree)

    if not parsed_rpc:
        raise NotImplementedError()

    return parsed_rpc

def parse_coeff(dic: Dict[str, str], prefix: str, indices: range) -> str:
    """ helper function"""
    return ' '.join([dic["%s_%s" % (prefix, str(x))] for x in indices])

def read_rpc_ikonos(rpc_content: str) -> Dict[str, str]:
    """Read RPC file assuming the ikonos format

    Parameters
    ----------
        rpc_content -- content of RPC sidecar file path read as a string

    Returns
    -------
        dictionary read from the RPC file
    """
    import re

    lines = rpc_content.split('\n')

    d: Dict[str, str] = {}
    for l in lines:
        ll = l.split()
        if len(ll) > 1:
            k = re.sub(r"[^a-zA-Z0-9_]","",ll[0])
            d[k] = ll[1]

    d['SAMP_NUM_COEFF']  = parse_coeff(d, "SAMP_NUM_COEFF", range(1, 21))
    d['SAMP_DEN_COEFF']  = parse_coeff(d, "SAMP_DEN_COEFF", range(1, 21))
    d['LINE_NUM_COEFF']  = parse_coeff(d, "LINE_NUM_COEFF", range(1, 21))
    d['LINE_DEN_COEFF']  = parse_coeff(d, "LINE_DEN_COEFF", range(1, 21))

    return d

def read_rpc_file(rpc_file: str):
    """Read RPC from a file deciding the format from the extension of the filename.
      xml          : spot6, pleiades, worldview
      txt (others) : ikonos

    Parameters
    ----------
        rpc_file -- RPC sidecar file path

    Returns
    -------
        dictionary read from the RPC file, or an empty dict if fail
    """

    with open(rpc_file) as f:
        rpc_content = f.read()

    if rpc_file.lower().endswith('xml'):
        try:
            rpc = read_rpc_xml(rpc_content)
        except NotImplementedError:
            raise NotImplementedError('XML file {} not supported'.format(rpc_file))
    else:
        # we assume that non xml rpc files follow the ikonos convention
        rpc = read_rpc_ikonos(rpc_content)

    return rpc