"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const os = require("os");
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const ec2 = require("../lib");
let app;
let stack;
let instanceRole;
let resource;
let linuxUserData;
function resetState() {
    resetStateWithSynthesizer();
}
function resetStateWithSynthesizer(customSynthesizer) {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
        synthesizer: customSynthesizer,
    });
    instanceRole = new iam.Role(stack, 'InstanceRole', {
        assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
    });
    resource = new core_1.CfnResource(stack, 'Resource', {
        type: 'CDK::Test::Resource',
    });
    linuxUserData = ec2.UserData.forLinux();
}
;
beforeEach(resetState);
test('whole config with restart handles', () => {
    // WHEN
    const handle = new ec2.InitServiceRestartHandle();
    const config = new ec2.InitConfig([
        ec2.InitFile.fromString('/etc/my.cnf', '[mysql]\ngo_fast=true', { serviceRestartHandles: [handle] }),
        ec2.InitSource.fromUrl('/tmp/foo', 'https://amazon.com/foo.zip', { serviceRestartHandles: [handle] }),
        ec2.InitPackage.yum('httpd', { serviceRestartHandles: [handle] }),
        ec2.InitCommand.argvCommand(['/bin/true'], { serviceRestartHandles: [handle] }),
        ec2.InitService.enable('httpd', { serviceRestartHandle: handle }),
    ]);
    // THEN
    expect(config._bind(stack, linuxOptions()).config).toEqual(expect.objectContaining({
        services: {
            sysvinit: {
                httpd: {
                    enabled: true,
                    ensureRunning: true,
                    commands: ['000'],
                    files: ['/etc/my.cnf'],
                    packages: {
                        yum: ['httpd'],
                    },
                    sources: ['/tmp/foo'],
                },
            },
        },
    }));
});
test('CloudFormationInit can be added to after instantiation', () => {
    // GIVEN
    const config = new ec2.InitConfig([]);
    const init = ec2.CloudFormationInit.fromConfig(config);
    // WHEN
    config.add(ec2.InitCommand.argvCommand(['/bin/true']));
    init._attach(resource, linuxOptions());
    // THEN
    expectMetadataLike({
        'AWS::CloudFormation::Init': {
            config: {
                commands: {
                    '000': { command: ['/bin/true'] },
                },
            },
        },
    });
});
test('empty configs are not rendered', () => {
    // GIVEN
    const config1 = new ec2.InitConfig([]);
    const config2 = new ec2.InitConfig([
        ec2.InitCommand.argvCommand(['/bin/true']),
    ]);
    // WHEN
    const init = ec2.CloudFormationInit.fromConfigSets({
        configSets: { default: ['config2', 'config1'] },
        configs: { config1, config2 },
    });
    init._attach(resource, linuxOptions());
    // THEN
    expectMetadataLike({
        'AWS::CloudFormation::Init': {
            configSets: {
                default: ['config2'],
            },
            config2: {
                commands: {
                    '000': { command: ['/bin/true'] },
                },
            },
        },
    });
});
describe('userdata', () => {
    let simpleInit;
    beforeEach(() => {
        simpleInit = ec2.CloudFormationInit.fromElements(ec2.InitCommand.argvCommand(['/bin/true']));
    });
    test('linux userdata contains right commands', () => {
        // WHEN
        simpleInit._attach(resource, linuxOptions());
        // THEN
        const lines = linuxUserData.render().split('\n');
        expectLine(lines, cmdArg('cfn-init', `--region ${core_1.Aws.REGION}`));
        expectLine(lines, cmdArg('cfn-init', `--stack ${core_1.Aws.STACK_NAME}`));
        expectLine(lines, cmdArg('cfn-init', `--resource ${resource.logicalId}`));
        expectLine(lines, cmdArg('cfn-init', '-c default'));
        expectLine(lines, cmdArg('cfn-signal', `--region ${core_1.Aws.REGION}`));
        expectLine(lines, cmdArg('cfn-signal', `--stack ${core_1.Aws.STACK_NAME}`));
        expectLine(lines, cmdArg('cfn-signal', `--resource ${resource.logicalId}`));
        expectLine(lines, cmdArg('cfn-signal', '-e $?'));
        expectLine(lines, cmdArg('cat', 'cfn-init.log'));
        expectLine(lines, /fingerprint/);
    });
    test('Windows userdata contains right commands', () => {
        // WHEN
        const windowsUserData = ec2.UserData.forWindows();
        simpleInit._attach(resource, {
            platform: ec2.OperatingSystemType.WINDOWS,
            instanceRole,
            userData: windowsUserData,
        });
        // THEN
        const lines = windowsUserData.render().split('\n');
        expectLine(lines, cmdArg('cfn-init', `--region ${core_1.Aws.REGION}`));
        expectLine(lines, cmdArg('cfn-init', `--stack ${core_1.Aws.STACK_NAME}`));
        expectLine(lines, cmdArg('cfn-init', `--resource ${resource.logicalId}`));
        expectLine(lines, cmdArg('cfn-init', '-c default'));
        expectLine(lines, cmdArg('cfn-signal', `--region ${core_1.Aws.REGION}`));
        expectLine(lines, cmdArg('cfn-signal', `--stack ${core_1.Aws.STACK_NAME}`));
        expectLine(lines, cmdArg('cfn-signal', `--resource ${resource.logicalId}`));
        expectLine(lines, cmdArg('cfn-signal', '-e $LASTEXITCODE'));
        expectLine(lines, cmdArg('type', 'cfn-init.log'));
        expectLine(lines, /fingerprint/);
    });
    test('ignoreFailures disables result code reporting', () => {
        // WHEN
        simpleInit._attach(resource, {
            ...linuxOptions(),
            ignoreFailures: true,
        });
        // THEN
        const lines = linuxUserData.render().split('\n');
        dontExpectLine(lines, cmdArg('cfn-signal', '-e $?'));
        expectLine(lines, cmdArg('cfn-signal', '-e 0'));
    });
    test('can disable log printing', () => {
        // WHEN
        simpleInit._attach(resource, {
            ...linuxOptions(),
            printLog: false,
        });
        // THEN
        const lines = linuxUserData.render().split('\n');
        dontExpectLine(lines, cmdArg('cat', 'cfn-init.log'));
    });
    test('can disable fingerprinting', () => {
        // WHEN
        simpleInit._attach(resource, {
            ...linuxOptions(),
            embedFingerprint: false,
        });
        // THEN
        const lines = linuxUserData.render().split('\n');
        dontExpectLine(lines, /fingerprint/);
    });
    test('can request multiple different configsets to be used', () => {
        // WHEN
        simpleInit._attach(resource, {
            ...linuxOptions(),
            configSets: ['banana', 'peach'],
        });
        // THEN
        const lines = linuxUserData.render().split('\n');
        expectLine(lines, cmdArg('cfn-init', '-c banana,peach'));
    });
});
const ASSET_STATEMENT = {
    Action: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
    Effect: 'Allow',
    Resource: [
        {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':s3:::',
                    { Ref: assert_1.stringLike('AssetParameter*S3Bucket*') },
                ]],
        },
        {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':s3:::',
                    { Ref: assert_1.stringLike('AssetParameter*S3Bucket*') },
                    '/*',
                ]],
        },
    ],
};
describe('assets n buckets', () => {
    test.each([
        ['Existing'],
        [''],
    ])('InitFile.from%sAsset', (existing) => {
        // GIVEN
        const asset = new aws_s3_assets_1.Asset(stack, 'Asset', { path: __filename });
        const init = ec2.CloudFormationInit.fromElements(existing
            ? ec2.InitFile.fromExistingAsset('/etc/fun.js', asset)
            : ec2.InitFile.fromAsset('/etc/fun.js', __filename));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assert_1.arrayWith(ASSET_STATEMENT),
                Version: '2012-10-17',
            },
        });
        expectMetadataLike({
            'AWS::CloudFormation::Init': {
                config: {
                    files: {
                        '/etc/fun.js': {
                            source: {
                                'Fn::Join': ['', [
                                        'https://s3.testregion.',
                                        { Ref: 'AWS::URLSuffix' },
                                        '/',
                                        { Ref: assert_1.stringLike('AssetParameters*') },
                                        '/',
                                        { 'Fn::Select': [0, { 'Fn::Split': ['||', { Ref: assert_1.stringLike('AssetParameters*') }] }] },
                                        { 'Fn::Select': [1, { 'Fn::Split': ['||', { Ref: assert_1.stringLike('AssetParameters*') }] }] },
                                    ]],
                            },
                        },
                    },
                },
            },
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: [
                        { Ref: assert_1.stringLike('AssetParameters*S3Bucket*') },
                    ],
                },
            },
        });
    });
    test.each([
        ['Existing'],
        [''],
    ])('InitSource.from%sAsset', (existing) => {
        // GIVEN
        const asset = new aws_s3_assets_1.Asset(stack, 'Asset', { path: path.join(__dirname, 'asset-fixture') });
        const init = ec2.CloudFormationInit.fromElements(existing
            ? ec2.InitSource.fromExistingAsset('/etc/fun', asset)
            : ec2.InitSource.fromAsset('/etc/fun', path.join(__dirname, 'asset-fixture')));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assert_1.arrayWith(ASSET_STATEMENT),
                Version: '2012-10-17',
            },
        });
        expectMetadataLike({
            'AWS::CloudFormation::Init': {
                config: {
                    sources: {
                        '/etc/fun': {
                            'Fn::Join': ['', [
                                    'https://s3.testregion.',
                                    { Ref: 'AWS::URLSuffix' },
                                    '/',
                                    { Ref: assert_1.stringLike('AssetParameters*') },
                                    '/',
                                    { 'Fn::Select': [0, { 'Fn::Split': ['||', { Ref: assert_1.stringLike('AssetParameters*') }] }] },
                                    { 'Fn::Select': [1, { 'Fn::Split': ['||', { Ref: assert_1.stringLike('AssetParameters*') }] }] },
                                ]],
                        },
                    },
                },
            },
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: [
                        { Ref: assert_1.stringLike('AssetParameters*S3Bucket*') },
                    ],
                },
            },
        });
    });
    test('InitFile.fromS3Object', () => {
        const bucket = s3.Bucket.fromBucketName(stack, 'Bucket', 'my-bucket');
        const init = ec2.CloudFormationInit.fromElements(ec2.InitFile.fromS3Object('/etc/fun.js', bucket, 'file.js'));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assert_1.arrayWith({
                    Action: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                    Effect: 'Allow',
                    Resource: [
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket']] },
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket/file.js']] },
                    ],
                }),
                Version: '2012-10-17',
            },
        });
        expectMetadataLike({
            'AWS::CloudFormation::Init': {
                config: {
                    files: {
                        '/etc/fun.js': {
                            source: { 'Fn::Join': ['', ['https://s3.testregion.', { Ref: 'AWS::URLSuffix' }, '/my-bucket/file.js']] },
                        },
                    },
                },
            },
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: ['my-bucket'],
                },
            },
        });
    });
    test('InitSource.fromS3Object', () => {
        const bucket = s3.Bucket.fromBucketName(stack, 'Bucket', 'my-bucket');
        const init = ec2.CloudFormationInit.fromElements(ec2.InitSource.fromS3Object('/etc/fun', bucket, 'file.zip'));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assert_1.arrayWith({
                    Action: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                    Effect: 'Allow',
                    Resource: [
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket']] },
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket/file.zip']] },
                    ],
                }),
                Version: '2012-10-17',
            },
        });
        expectMetadataLike({
            'AWS::CloudFormation::Init': {
                config: {
                    sources: {
                        '/etc/fun': { 'Fn::Join': ['', ['https://s3.testregion.', { Ref: 'AWS::URLSuffix' }, '/my-bucket/file.zip']] },
                    },
                },
            },
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: ['my-bucket'],
                },
            },
        });
    });
    test('no duplication of bucket names when using multiple assets', () => {
        // GIVEN
        const init = ec2.CloudFormationInit.fromElements(ec2.InitFile.fromAsset('/etc/fun.js', __filename), ec2.InitSource.fromAsset('/etc/fun', path.join(__dirname, 'asset-fixture')));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expectMetadataLike({
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: [
                        { Ref: assert_1.stringLike('AssetParameters*S3Bucket*') },
                    ],
                },
            },
        });
    });
    test('multiple buckets appear in the same auth block', () => {
        // GIVEN
        const bucket = s3.Bucket.fromBucketName(stack, 'Bucket', 'my-bucket');
        const init = ec2.CloudFormationInit.fromElements(ec2.InitFile.fromS3Object('/etc/fun.js', bucket, 'file.js'), ec2.InitSource.fromAsset('/etc/fun', path.join(__dirname, 'asset-fixture')));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expectMetadataLike({
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: assert_1.arrayWith({ Ref: assert_1.stringLike('AssetParameters*S3Bucket*') }, 'my-bucket'),
                },
            },
        });
    });
    test('fingerprint data changes on asset hash update', () => {
        function calculateFingerprint(assetFilePath) {
            resetState(); // Needed so the same resources/assets/filenames can be used.
            const init = ec2.CloudFormationInit.fromElements(ec2.InitFile.fromAsset('/etc/myFile', assetFilePath));
            init._attach(resource, linuxOptions());
            return linuxUserData.render().split('\n').find(line => line.match(/# fingerprint:/));
        }
        // Setup initial asset file
        const assetFileDir = fs.mkdtempSync(path.join(os.tmpdir(), 'cfn-init-test'));
        const assetFilePath = path.join(assetFileDir, 'fingerprint-test');
        fs.writeFileSync(assetFilePath, 'hello');
        const fingerprintOne = calculateFingerprint(assetFilePath);
        const fingerprintOneAgain = calculateFingerprint(assetFilePath);
        // Consistent without changes.
        expect(fingerprintOneAgain).toEqual(fingerprintOne);
        // Change asset file content/hash
        fs.writeFileSync(assetFilePath, ' world');
        const fingerprintTwo = calculateFingerprint(assetFilePath);
        expect(fingerprintTwo).not.toEqual(fingerprintOne);
    });
    test('fingerprint data changes on existing asset update, even for assets with unchanging URLs', () => {
        function calculateFingerprint(assetFilePath) {
            resetStateWithSynthesizer(new SingletonLocationSythesizer());
            const init = ec2.CloudFormationInit.fromElements(ec2.InitFile.fromExistingAsset('/etc/myFile', new aws_s3_assets_1.Asset(stack, 'FileAsset', { path: assetFilePath })));
            init._attach(resource, linuxOptions());
            return linuxUserData.render().split('\n').find(line => line.match(/# fingerprint:/));
        }
        // Setup initial asset file
        const assetFileDir = fs.mkdtempSync(path.join(os.tmpdir(), 'cfn-init-test'));
        const assetFilePath = path.join(assetFileDir, 'fingerprint-test');
        fs.writeFileSync(assetFilePath, 'hello');
        const fingerprintOne = calculateFingerprint(assetFilePath);
        const fingerprintOneAgain = calculateFingerprint(assetFilePath);
        // Consistent without changes.
        expect(fingerprintOneAgain).toEqual(fingerprintOne);
        // Change asset file content/hash
        fs.writeFileSync(assetFilePath, ' world');
        const fingerprintTwo = calculateFingerprint(assetFilePath);
        expect(fingerprintTwo).not.toEqual(fingerprintOne);
    });
});
function linuxOptions() {
    return {
        platform: ec2.OperatingSystemType.LINUX,
        instanceRole,
        userData: linuxUserData,
    };
}
function expectMetadataLike(pattern) {
    expect(stack).toHaveResourceLike('CDK::Test::Resource', {
        Metadata: pattern,
    }, assert_1.ResourcePart.CompleteDefinition);
}
function expectLine(lines, re) {
    for (const line of lines) {
        if (re.test(line)) {
            return;
        }
    }
    throw new Error(`None of the lines matched '${re}': ${lines.join('\n')}`);
}
function dontExpectLine(lines, re) {
    try {
        expectLine(lines, re);
    }
    catch (e) {
        return;
    }
    throw new Error(`Found unexpected line matching '${re}': ${lines.join('\n')}`);
}
function cmdArg(command, argument) {
    return new RegExp(`${escapeRegex(command)}(\.exe)? .*${escapeRegex(argument)}`);
}
function escapeRegex(s) {
    return s.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'); // $& means the whole matched string
}
/** Creates file assets that have a hard-coded asset url, rather than the default based on asset hash */
class SingletonLocationSythesizer extends core_1.DefaultStackSynthesizer {
    addFileAsset(_asset) {
        const httpUrl = 'https://MyBucket.s3.amazonaws.com/MyAsset';
        return {
            bucketName: 'MyAssetBucket',
            objectKey: 'MyAssetFile',
            httpUrl,
            s3ObjectUrl: httpUrl,
            s3Url: httpUrl,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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