"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateSubnet = exports.PublicSubnet = exports.RouterType = exports.Subnet = exports.Vpc = exports.DefaultInstanceTenancy = exports.SubnetType = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const ec2_generated_1 = require("./ec2.generated");
const nat_1 = require("./nat");
const network_acl_1 = require("./network-acl");
const network_util_1 = require("./network-util");
const subnet_1 = require("./subnet");
const util_1 = require("./util");
const vpc_endpoint_1 = require("./vpc-endpoint");
const vpc_flow_logs_1 = require("./vpc-flow-logs");
const vpn_1 = require("./vpn");
const VPC_SUBNET_SYMBOL = Symbol.for('@aws-cdk/aws-ec2.VpcSubnet');
/**
 * The type of Subnet
 */
var SubnetType;
(function (SubnetType) {
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC).
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     */
    SubnetType["ISOLATED"] = "Isolated";
    /**
     * Subnet that routes to the internet, but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. Internet traffic will
     * be routed via a NAT Gateway.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     */
    SubnetType["PRIVATE"] = "Private";
    /**
     * Subnet connected to the Internet
     *
     * Instances in a Public subnet can connect to the Internet and can be
     * connected to from the Internet as long as they are launched with public
     * IPs (controlled on the AutoScalingGroup or other constructs that launch
     * instances).
     *
     * Public subnets route outbound traffic via an Internet Gateway.
     */
    SubnetType["PUBLIC"] = "Public";
})(SubnetType = exports.SubnetType || (exports.SubnetType = {}));
/**
 * A new or imported VPC
 */
class VpcBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Dependencies for NAT connectivity
         *
         * @deprecated - This value is no longer used.
         */
        this.natDependencies = new Array();
        /**
         * If this is set to true, don't error out on trying to select subnets
         */
        this.incompleteSubnetDefinition = false;
    }
    /**
     * Returns IDs of selected subnets
     */
    selectSubnets(selection = {}) {
        const subnets = this.selectSubnetObjects(selection);
        const pubs = new Set(this.publicSubnets);
        return {
            subnetIds: subnets.map(s => s.subnetId),
            get availabilityZones() { return subnets.map(s => s.availabilityZone); },
            internetConnectivityEstablished: tap(new CompositeDependable(), d => subnets.forEach(s => d.add(s.internetConnectivityEstablished))),
            subnets,
            hasPublic: subnets.some(s => pubs.has(s)),
        };
    }
    /**
     * Adds a VPN Gateway to this VPC
     */
    enableVpnGateway(options) {
        var _a;
        if (this.vpnGatewayId) {
            throw new Error('The VPN Gateway has already been enabled.');
        }
        const vpnGateway = new vpn_1.VpnGateway(this, 'VpnGateway', {
            amazonSideAsn: options.amazonSideAsn,
            type: vpn_1.VpnConnectionType.IPSEC_1,
        });
        this._vpnGatewayId = vpnGateway.gatewayId;
        const attachment = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCVPNGW', {
            vpcId: this.vpcId,
            vpnGatewayId: this._vpnGatewayId,
        });
        // Propagate routes on route tables associated with the right subnets
        const vpnRoutePropagation = (_a = options.vpnRoutePropagation) !== null && _a !== void 0 ? _a : [{}];
        const routeTableIds = util_1.allRouteTableIds(util_1.flatten(vpnRoutePropagation.map(s => this.selectSubnets(s).subnets)));
        if (routeTableIds.length === 0) {
            core_1.Annotations.of(this).addError(`enableVpnGateway: no subnets matching selection: '${JSON.stringify(vpnRoutePropagation)}'. Select other subnets to add routes to.`);
        }
        const routePropagation = new ec2_generated_1.CfnVPNGatewayRoutePropagation(this, 'RoutePropagation', {
            routeTableIds,
            vpnGatewayId: this._vpnGatewayId,
        });
        // The AWS::EC2::VPNGatewayRoutePropagation resource cannot use the VPN gateway
        // until it has successfully attached to the VPC.
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpn-gatewayrouteprop.html
        routePropagation.node.addDependency(attachment);
    }
    /**
     * Adds a new VPN connection to this VPC
     */
    addVpnConnection(id, options) {
        return new vpn_1.VpnConnection(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new interface endpoint to this VPC
     */
    addInterfaceEndpoint(id, options) {
        return new vpc_endpoint_1.InterfaceVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new gateway endpoint to this VPC
     */
    addGatewayEndpoint(id, options) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new flow log to this VPC
     */
    addFlowLog(id, options) {
        return new vpc_flow_logs_1.FlowLog(this, id, {
            resourceType: vpc_flow_logs_1.FlowLogResourceType.fromVpc(this),
            ...options,
        });
    }
    /**
     * Returns the id of the VPN Gateway (if enabled)
     */
    get vpnGatewayId() {
        return this._vpnGatewayId;
    }
    /**
     * Return the subnets appropriate for the placement strategy
     */
    selectSubnetObjects(selection = {}) {
        var _a;
        selection = this.reifySelectionDefaults(selection);
        if (selection.subnets !== undefined) {
            return selection.subnets;
        }
        let subnets;
        if (selection.subnetGroupName !== undefined) { // Select by name
            subnets = this.selectSubnetObjectsByName(selection.subnetGroupName);
        }
        else { // Or specify by type
            const type = selection.subnetType || SubnetType.PRIVATE;
            subnets = this.selectSubnetObjectsByType(type);
        }
        // Apply all the filters
        subnets = this.applySubnetFilters(subnets, (_a = selection.subnetFilters) !== null && _a !== void 0 ? _a : []);
        return subnets;
    }
    applySubnetFilters(subnets, filters) {
        let filtered = subnets;
        // Apply each filter in sequence
        for (const filter of filters) {
            filtered = filter.selectSubnets(filtered);
        }
        return filtered;
    }
    selectSubnetObjectsByName(groupName) {
        const allSubnets = [...this.publicSubnets, ...this.privateSubnets, ...this.isolatedSubnets];
        const subnets = allSubnets.filter(s => util_1.subnetGroupNameFromConstructId(s) === groupName);
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const names = Array.from(new Set(allSubnets.map(util_1.subnetGroupNameFromConstructId)));
            throw new Error(`There are no subnet groups with name '${groupName}' in this VPC. Available names: ${names}`);
        }
        return subnets;
    }
    selectSubnetObjectsByType(subnetType) {
        const allSubnets = {
            [SubnetType.ISOLATED]: this.isolatedSubnets,
            [SubnetType.PRIVATE]: this.privateSubnets,
            [SubnetType.PUBLIC]: this.publicSubnets,
        };
        const subnets = allSubnets[subnetType];
        // Force merge conflict here with https://github.com/aws/aws-cdk/pull/4089
        // see ImportedVpc
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const availableTypes = Object.entries(allSubnets).filter(([_, subs]) => subs.length > 0).map(([typeName, _]) => typeName);
            throw new Error(`There are no '${subnetType}' subnet groups in this VPC. Available types: ${availableTypes}`);
        }
        return subnets;
    }
    /**
     * Validate the fields in a SubnetSelection object, and reify defaults if necessary
     *
     * In case of default selection, select the first type of PRIVATE, ISOLATED,
     * PUBLIC (in that order) that has any subnets.
     */
    reifySelectionDefaults(placement) {
        var _a;
        if (placement.subnetName !== undefined) {
            if (placement.subnetGroupName !== undefined) {
                throw new Error('Please use only \'subnetGroupName\' (\'subnetName\' is deprecated and has the same behavior)');
            }
            else {
                core_1.Annotations.of(this).addWarning('Usage of \'subnetName\' in SubnetSelection is deprecated, use \'subnetGroupName\' instead');
            }
            placement = { ...placement, subnetGroupName: placement.subnetName };
        }
        const exclusiveSelections = ['subnets', 'subnetType', 'subnetGroupName'];
        const providedSelections = exclusiveSelections.filter(key => placement[key] !== undefined);
        if (providedSelections.length > 1) {
            throw new Error(`Only one of '${providedSelections}' can be supplied to subnet selection.`);
        }
        if (placement.subnetType === undefined && placement.subnetGroupName === undefined && placement.subnets === undefined) {
            // Return default subnet type based on subnets that actually exist
            let subnetType = this.privateSubnets.length ? SubnetType.PRIVATE : this.isolatedSubnets.length ? SubnetType.ISOLATED : SubnetType.PUBLIC;
            placement = { ...placement, subnetType: subnetType };
        }
        // Establish which subnet filters are going to be used
        let subnetFilters = (_a = placement.subnetFilters) !== null && _a !== void 0 ? _a : [];
        // Backwards compatibility with existing `availabilityZones` and `onePerAz` functionality
        if (placement.availabilityZones !== undefined) { // Filter by AZs, if specified
            subnetFilters.push(subnet_1.SubnetFilter.availabilityZones(placement.availabilityZones));
        }
        if (!!placement.onePerAz) { // Ensure one per AZ if specified
            subnetFilters.push(subnet_1.SubnetFilter.onePerAz());
        }
        // Overwrite the provided placement filters and remove the availabilityZones and onePerAz properties
        placement = { ...placement, subnetFilters: subnetFilters, availabilityZones: undefined, onePerAz: undefined };
        const { availabilityZones, onePerAz, ...rest } = placement;
        return rest;
    }
}
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The default tenancy of instances launched into the VPC.
 */
var DefaultInstanceTenancy;
(function (DefaultInstanceTenancy) {
    /**
     * Instances can be launched with any tenancy.
     */
    DefaultInstanceTenancy["DEFAULT"] = "default";
    /**
     * Any instance launched into the VPC automatically has dedicated tenancy, unless you launch it with the default tenancy.
     */
    DefaultInstanceTenancy["DEDICATED"] = "dedicated";
})(DefaultInstanceTenancy = exports.DefaultInstanceTenancy || (exports.DefaultInstanceTenancy = {}));
/**
 * Define an AWS Virtual Private Cloud
 *
 * See the package-level documentation of this package for an overview
 * of the various dimensions in which you can configure your VPC.
 *
 * For example:
 *
 * ```ts
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   cidr: "10.0.0.0/16"
 * })
 *
 * // Iterate the private subnets
 * const selection = vpc.selectSubnets({
 *   subnetType: ec2.SubnetType.PRIVATE
 * });
 *
 * for (const subnet of selection.subnets) {
 *   // ...
 * }
 * ```
 *
 * @resource AWS::EC2::VPC
 */
class Vpc extends VpcBase {
    /**
     * Vpc creates a VPC that spans a whole region.
     * It will automatically divide the provided VPC CIDR range, and create public and private subnets per Availability Zone.
     * Network routing for the public subnets will be configured to allow outbound access directly via an Internet Gateway.
     * Network routing for the private subnets will be configured to allow outbound access via a set of resilient NAT Gateways (one per AZ).
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * List of public subnets in this VPC
         */
        this.publicSubnets = [];
        /**
         * List of private subnets in this VPC
         */
        this.privateSubnets = [];
        /**
         * List of isolated subnets in this VPC
         */
        this.isolatedSubnets = [];
        /**
         * Subnet configurations for this VPC
         */
        this.subnetConfiguration = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        const stack = core_1.Stack.of(this);
        // Can't have enabledDnsHostnames without enableDnsSupport
        if (props.enableDnsHostnames && !props.enableDnsSupport) {
            throw new Error('To use DNS Hostnames, DNS Support must be enabled, however, it was explicitly disabled.');
        }
        const cidrBlock = ifUndefined(props.cidr, Vpc.DEFAULT_CIDR_RANGE);
        if (core_1.Token.isUnresolved(cidrBlock)) {
            throw new Error('\'cidr\' property must be a concrete CIDR string, got a Token (we need to parse it for automatic subdivision)');
        }
        this.networkBuilder = new network_util_1.NetworkBuilder(cidrBlock);
        this.dnsHostnamesEnabled = props.enableDnsHostnames == null ? true : props.enableDnsHostnames;
        this.dnsSupportEnabled = props.enableDnsSupport == null ? true : props.enableDnsSupport;
        const instanceTenancy = props.defaultInstanceTenancy || 'default';
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
        // Define a VPC using the provided CIDR range
        this.resource = new ec2_generated_1.CfnVPC(this, 'Resource', {
            cidrBlock,
            enableDnsHostnames: this.dnsHostnamesEnabled,
            enableDnsSupport: this.dnsSupportEnabled,
            instanceTenancy,
        });
        this.vpcDefaultNetworkAcl = this.resource.attrDefaultNetworkAcl;
        this.vpcCidrBlockAssociations = this.resource.attrCidrBlockAssociations;
        this.vpcCidrBlock = this.resource.attrCidrBlock;
        this.vpcDefaultSecurityGroup = this.resource.attrDefaultSecurityGroup;
        this.vpcIpv6CidrBlocks = this.resource.attrIpv6CidrBlocks;
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.availabilityZones = stack.availabilityZones;
        const maxAZs = props.maxAzs !== undefined ? props.maxAzs : 3;
        this.availabilityZones = this.availabilityZones.slice(0, maxAZs);
        this.vpcId = this.resource.ref;
        const defaultSubnet = props.natGateways === 0 ? Vpc.DEFAULT_SUBNETS_NO_NAT : Vpc.DEFAULT_SUBNETS;
        this.subnetConfiguration = ifUndefined(props.subnetConfiguration, defaultSubnet);
        const natGatewayPlacement = props.natGatewaySubnets || { subnetType: SubnetType.PUBLIC };
        const natGatewayCount = determineNatGatewayCount(props.natGateways, this.subnetConfiguration, this.availabilityZones.length);
        // subnetConfiguration must be set before calling createSubnets
        this.createSubnets();
        const allowOutbound = this.subnetConfiguration.filter(subnet => (subnet.subnetType !== SubnetType.ISOLATED)).length > 0;
        // Create an Internet Gateway and attach it if necessary
        if (allowOutbound) {
            const igw = new ec2_generated_1.CfnInternetGateway(this, 'IGW', {});
            this.internetGatewayId = igw.ref;
            this._internetConnectivityEstablished.add(igw);
            const att = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCGW', {
                internetGatewayId: igw.ref,
                vpcId: this.resource.ref,
            });
            this.publicSubnets.forEach(publicSubnet => {
                publicSubnet.addDefaultInternetRoute(igw.ref, att);
            });
            // if gateways are needed create them
            if (natGatewayCount > 0) {
                const provider = props.natGatewayProvider || nat_1.NatProvider.gateway();
                this.createNatGateways(provider, natGatewayCount, natGatewayPlacement);
            }
        }
        if (props.vpnGateway && this.publicSubnets.length === 0 && this.privateSubnets.length === 0 && this.isolatedSubnets.length === 0) {
            throw new Error('Can not enable the VPN gateway while the VPC has no subnets at all');
        }
        if ((props.vpnConnections || props.vpnGatewayAsn) && props.vpnGateway === false) {
            throw new Error('Cannot specify `vpnConnections` or `vpnGatewayAsn` when `vpnGateway` is set to false.');
        }
        if (props.vpnGateway || props.vpnConnections || props.vpnGatewayAsn) {
            this.enableVpnGateway({
                amazonSideAsn: props.vpnGatewayAsn,
                type: vpn_1.VpnConnectionType.IPSEC_1,
                vpnRoutePropagation: props.vpnRoutePropagation,
            });
            const vpnConnections = props.vpnConnections || {};
            for (const [connectionId, connection] of Object.entries(vpnConnections)) {
                this.addVpnConnection(connectionId, connection);
            }
        }
        // Allow creation of gateway endpoints on VPC instantiation as those can be
        // immediately functional without further configuration. This is not the case
        // for interface endpoints where the security group must be configured.
        if (props.gatewayEndpoints) {
            const gatewayEndpoints = props.gatewayEndpoints || {};
            for (const [endpointId, endpoint] of Object.entries(gatewayEndpoints)) {
                this.addGatewayEndpoint(endpointId, endpoint);
            }
        }
        // Add flow logs to the VPC
        if (props.flowLogs) {
            const flowLogs = props.flowLogs || {};
            for (const [flowLogId, flowLog] of Object.entries(flowLogs)) {
                this.addFlowLog(flowLogId, flowLog);
            }
        }
    }
    /**
     * Import an exported VPC
     */
    static fromVpcAttributes(scope, id, attrs) {
        return new ImportedVpc(scope, id, attrs, false);
    }
    /**
     * Import an existing VPC from by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use VPCs not defined in your CDK
     * application. If you are looking to share a VPC between stacks, you can
     * pass the `Vpc` object between stacks and use it as normal.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     *
     * The VPC information will be cached in `cdk.context.json` and the same VPC
     * will be used on future runs. To refresh the lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static fromLookup(scope, id, options) {
        if (core_1.Token.isUnresolved(options.vpcId)
            || core_1.Token.isUnresolved(options.vpcName)
            || Object.values(options.tags || {}).some(core_1.Token.isUnresolved)
            || Object.keys(options.tags || {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to Vpc.fromLookup() must be concrete (no Tokens)');
        }
        const filter = makeTagFilter(options.tags);
        // We give special treatment to some tags
        if (options.vpcId) {
            filter['vpc-id'] = options.vpcId;
        }
        if (options.vpcName) {
            filter['tag:Name'] = options.vpcName;
        }
        if (options.isDefault !== undefined) {
            filter.isDefault = options.isDefault ? 'true' : 'false';
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.VPC_PROVIDER,
            props: {
                filter,
                returnAsymmetricSubnets: true,
                subnetGroupNameTag: options.subnetGroupNameTag,
            },
            dummyValue: undefined,
        }).value;
        return new LookedUpVpc(scope, id, attributes || DUMMY_VPC_PROPS, attributes === undefined);
        /**
         * Prefixes all keys in the argument with `tag:`.`
         */
        function makeTagFilter(tags) {
            const result = {};
            for (const [name, value] of Object.entries(tags || {})) {
                result[`tag:${name}`] = value;
            }
            return result;
        }
    }
    /**
     * Adds a new S3 gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addS3Endpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.S3,
            vpc: this,
            subnets,
        });
    }
    /**
     * Adds a new DynamoDB gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addDynamoDbEndpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.DYNAMODB,
            vpc: this,
            subnets,
        });
    }
    createNatGateways(provider, natCount, placement) {
        const natSubnets = this.selectSubnetObjects(placement);
        for (const sub of natSubnets) {
            if (this.publicSubnets.indexOf(sub) === -1) {
                throw new Error(`natGatewayPlacement ${placement} contains non public subnet ${sub}`);
            }
        }
        provider.configureNat({
            vpc: this,
            natSubnets: natSubnets.slice(0, natCount),
            privateSubnets: this.privateSubnets,
        });
    }
    /**
     * createSubnets creates the subnets specified by the subnet configuration
     * array or creates the `DEFAULT_SUBNETS` configuration
     */
    createSubnets() {
        const remainingSpaceSubnets = [];
        for (const subnet of this.subnetConfiguration) {
            if (subnet.cidrMask === undefined) {
                remainingSpaceSubnets.push(subnet);
                continue;
            }
            this.createSubnetResources(subnet, subnet.cidrMask);
        }
        const totalRemaining = remainingSpaceSubnets.length * this.availabilityZones.length;
        const cidrMaskForRemaing = this.networkBuilder.maskForRemainingSubnets(totalRemaining);
        for (const subnet of remainingSpaceSubnets) {
            this.createSubnetResources(subnet, cidrMaskForRemaing);
        }
    }
    createSubnetResources(subnetConfig, cidrMask) {
        this.availabilityZones.forEach((zone, index) => {
            if (subnetConfig.reserved === true) {
                // For reserved subnets, just allocate ip space but do not create any resources
                this.networkBuilder.addSubnet(cidrMask);
                return;
            }
            const name = util_1.subnetId(subnetConfig.name, index);
            const subnetProps = {
                availabilityZone: zone,
                vpcId: this.vpcId,
                cidrBlock: this.networkBuilder.addSubnet(cidrMask),
                mapPublicIpOnLaunch: (subnetConfig.subnetType === SubnetType.PUBLIC),
            };
            let subnet;
            switch (subnetConfig.subnetType) {
                case SubnetType.PUBLIC:
                    const publicSubnet = new PublicSubnet(this, name, subnetProps);
                    this.publicSubnets.push(publicSubnet);
                    subnet = publicSubnet;
                    break;
                case SubnetType.PRIVATE:
                    const privateSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.privateSubnets.push(privateSubnet);
                    subnet = privateSubnet;
                    break;
                case SubnetType.ISOLATED:
                    const isolatedSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.isolatedSubnets.push(isolatedSubnet);
                    subnet = isolatedSubnet;
                    break;
                default:
                    throw new Error(`Unrecognized subnet type: ${subnetConfig.subnetType}`);
            }
            // These values will be used to recover the config upon provider import
            const includeResourceTypes = [ec2_generated_1.CfnSubnet.CFN_RESOURCE_TYPE_NAME];
            core_1.Tags.of(subnet).add(SUBNETNAME_TAG, subnetConfig.name, { includeResourceTypes });
            core_1.Tags.of(subnet).add(SUBNETTYPE_TAG, subnetTypeTagValue(subnetConfig.subnetType), { includeResourceTypes });
        });
    }
}
exports.Vpc = Vpc;
/**
 * The default CIDR range used when creating VPCs.
 * This can be overridden using VpcProps when creating a VPCNetwork resource.
 * e.g. new VpcResource(this, { cidr: '192.168.0.0./16' })
 */
Vpc.DEFAULT_CIDR_RANGE = '10.0.0.0/16';
/**
 * The default subnet configuration
 *
 * 1 Public and 1 Private subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE),
    },
];
/**
 * The default subnet configuration if natGateways specified to be 0
 *
 * 1 Public and 1 Isolated Subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS_NO_NAT = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.ISOLATED,
        name: util_1.defaultSubnetName(SubnetType.ISOLATED),
    },
];
const SUBNETTYPE_TAG = 'aws-cdk:subnet-type';
const SUBNETNAME_TAG = 'aws-cdk:subnet-name';
function subnetTypeTagValue(type) {
    switch (type) {
        case SubnetType.PUBLIC: return 'Public';
        case SubnetType.PRIVATE: return 'Private';
        case SubnetType.ISOLATED: return 'Isolated';
    }
}
/**
 * Represents a new VPC subnet resource
 *
 * @resource AWS::EC2::Subnet
 */
class Subnet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Parts of this VPC subnet
         */
        this.dependencyElements = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        Object.defineProperty(this, VPC_SUBNET_SYMBOL, { value: true });
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.availabilityZone = props.availabilityZone;
        this.ipv4CidrBlock = props.cidrBlock;
        const subnet = new ec2_generated_1.CfnSubnet(this, 'Subnet', {
            vpcId: props.vpcId,
            cidrBlock: props.cidrBlock,
            availabilityZone: props.availabilityZone,
            mapPublicIpOnLaunch: props.mapPublicIpOnLaunch,
        });
        this.subnetId = subnet.ref;
        this.subnetVpcId = subnet.attrVpcId;
        this.subnetAvailabilityZone = subnet.attrAvailabilityZone;
        this.subnetIpv6CidrBlocks = subnet.attrIpv6CidrBlocks;
        // subnet.attrNetworkAclAssociationId is the default ACL after the subnet
        // was just created. However, the ACL can be replaced at a later time.
        this._networkAcl = network_acl_1.NetworkAcl.fromNetworkAclId(this, 'Acl', subnet.attrNetworkAclAssociationId);
        this.subnetNetworkAclAssociationId = core_1.Lazy.stringValue({ produce: () => this._networkAcl.networkAclId });
        this.node.defaultChild = subnet;
        const table = new ec2_generated_1.CfnRouteTable(this, 'RouteTable', {
            vpcId: props.vpcId,
        });
        this.routeTable = { routeTableId: table.ref };
        // Associate the public route table for this subnet, to this subnet
        new ec2_generated_1.CfnSubnetRouteTableAssociation(this, 'RouteTableAssociation', {
            subnetId: this.subnetId,
            routeTableId: table.ref,
        });
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
    }
    static isVpcSubnet(x) {
        return VPC_SUBNET_SYMBOL in x;
    }
    static fromSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * Import existing subnet from id.
     */
    // eslint-disable-next-line @typescript-eslint/no-shadow
    static fromSubnetId(scope, id, subnetId) {
        return this.fromSubnetAttributes(scope, id, { subnetId });
    }
    /**
     * Create a default route that points to a passed IGW, with a dependency
     * on the IGW's attachment to the VPC.
     *
     * @param gatewayId the logical ID (ref) of the gateway attached to your VPC
     * @param gatewayAttachment the gateway attachment construct to be added as a dependency
     */
    addDefaultInternetRoute(gatewayId, gatewayAttachment) {
        const route = new ec2_generated_1.CfnRoute(this, 'DefaultRoute', {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: '0.0.0.0/0',
            gatewayId,
        });
        route.node.addDependency(gatewayAttachment);
        // Since the 'route' depends on the gateway attachment, just
        // depending on the route is enough.
        this._internetConnectivityEstablished.add(route);
    }
    /**
     * Network ACL associated with this Subnet
     *
     * Upon creation, this is the default ACL which allows all traffic, except
     * explicit DENY entries that you add.
     *
     * You can replace it with a custom ACL which denies all traffic except
     * the explic it ALLOW entries that you add by creating a `NetworkAcl`
     * object and calling `associateNetworkAcl()`.
     */
    get networkAcl() {
        return this._networkAcl;
    }
    /**
     * Adds an entry to this subnets route table that points to the passed NATGatwayId
     * @param natGatewayId The ID of the NAT gateway
     */
    addDefaultNatRoute(natGatewayId) {
        this.addRoute('DefaultRoute', {
            routerType: RouterType.NAT_GATEWAY,
            routerId: natGatewayId,
            enablesInternetConnectivity: true,
        });
    }
    /**
     * Adds an entry to this subnets route table
     */
    addRoute(id, options) {
        if (options.destinationCidrBlock && options.destinationIpv6CidrBlock) {
            throw new Error('Cannot specify both \'destinationCidrBlock\' and \'destinationIpv6CidrBlock\'');
        }
        const route = new ec2_generated_1.CfnRoute(this, id, {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: options.destinationCidrBlock || (options.destinationIpv6CidrBlock === undefined ? '0.0.0.0/0' : undefined),
            destinationIpv6CidrBlock: options.destinationIpv6CidrBlock,
            [routerTypeToPropName(options.routerType)]: options.routerId,
        });
        if (options.enablesInternetConnectivity) {
            this._internetConnectivityEstablished.add(route);
        }
    }
    associateNetworkAcl(id, networkAcl) {
        this._networkAcl = networkAcl;
        const scope = core_1.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_1.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + other.node.uniqueId, {
            networkAcl,
            subnet: this,
        });
    }
}
exports.Subnet = Subnet;
/**
 * Type of router used in route
 */
var RouterType;
(function (RouterType) {
    /**
     * Egress-only Internet Gateway
     */
    RouterType["EGRESS_ONLY_INTERNET_GATEWAY"] = "EgressOnlyInternetGateway";
    /**
     * Internet Gateway
     */
    RouterType["GATEWAY"] = "Gateway";
    /**
     * Instance
     */
    RouterType["INSTANCE"] = "Instance";
    /**
     * NAT Gateway
     */
    RouterType["NAT_GATEWAY"] = "NatGateway";
    /**
     * Network Interface
     */
    RouterType["NETWORK_INTERFACE"] = "NetworkInterface";
    /**
     * VPC peering connection
     */
    RouterType["VPC_PEERING_CONNECTION"] = "VpcPeeringConnection";
})(RouterType = exports.RouterType || (exports.RouterType = {}));
function routerTypeToPropName(routerType) {
    return ({
        [RouterType.EGRESS_ONLY_INTERNET_GATEWAY]: 'egressOnlyInternetGatewayId',
        [RouterType.GATEWAY]: 'gatewayId',
        [RouterType.INSTANCE]: 'instanceId',
        [RouterType.NAT_GATEWAY]: 'natGatewayId',
        [RouterType.NETWORK_INTERFACE]: 'networkInterfaceId',
        [RouterType.VPC_PEERING_CONNECTION]: 'vpcPeeringConnectionId',
    })[routerType];
}
/**
 * Represents a public VPC subnet resource
 */
class PublicSubnet extends Subnet {
    static fromPublicSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * Creates a new managed NAT gateway attached to this public subnet.
     * Also adds the EIP for the managed NAT.
     * @returns A ref to the the NAT Gateway ID
     */
    addNatGateway() {
        // Create a NAT Gateway in this public subnet
        const ngw = new ec2_generated_1.CfnNatGateway(this, 'NATGateway', {
            subnetId: this.subnetId,
            allocationId: new ec2_generated_1.CfnEIP(this, 'EIP', {
                domain: 'vpc',
            }).attrAllocationId,
        });
        return ngw;
    }
}
exports.PublicSubnet = PublicSubnet;
/**
 * Represents a private VPC subnet resource
 */
class PrivateSubnet extends Subnet {
    static fromPrivateSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
    }
}
exports.PrivateSubnet = PrivateSubnet;
function ifUndefined(value, defaultValue) {
    return value !== undefined ? value : defaultValue;
}
class ImportedVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.cidr = props.vpcCidrBlock;
        this.availabilityZones = props.availabilityZones;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        /* eslint-disable max-len */
        const pub = new util_1.ImportSubnetGroup(props.publicSubnetIds, props.publicSubnetNames, props.publicSubnetRouteTableIds, SubnetType.PUBLIC, this.availabilityZones, 'publicSubnetIds', 'publicSubnetNames', 'publicSubnetRouteTableIds');
        const priv = new util_1.ImportSubnetGroup(props.privateSubnetIds, props.privateSubnetNames, props.privateSubnetRouteTableIds, SubnetType.PRIVATE, this.availabilityZones, 'privateSubnetIds', 'privateSubnetNames', 'privateSubnetRouteTableIds');
        const iso = new util_1.ImportSubnetGroup(props.isolatedSubnetIds, props.isolatedSubnetNames, props.isolatedSubnetRouteTableIds, SubnetType.ISOLATED, this.availabilityZones, 'isolatedSubnetIds', 'isolatedSubnetNames', 'isolatedSubnetRouteTableIds');
        /* eslint-enable max-len */
        this.publicSubnets = pub.import(this);
        this.privateSubnets = priv.import(this);
        this.isolatedSubnets = iso.import(this);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not supplied when creating this VPC');
        }
        return this.cidr;
    }
}
class LookedUpVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.cidr = props.vpcCidrBlock;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        const subnetGroups = props.subnetGroups || [];
        const availabilityZones = Array.from(new Set(flatMap(subnetGroups, subnetGroup => {
            return subnetGroup.subnets.map(subnet => subnet.availabilityZone);
        })));
        availabilityZones.sort((az1, az2) => az1.localeCompare(az2));
        this.availabilityZones = availabilityZones;
        this.publicSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PUBLIC);
        this.privateSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PRIVATE);
        this.isolatedSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.ISOLATED);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            // Value might be cached from an old CLI version, so bumping the CX API protocol to
            // force the value to exist would not have helped.
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not found when looking up this VPC. Use a newer version of the CDK CLI and clear the old context value.');
        }
        return this.cidr;
    }
    extractSubnetsOfType(subnetGroups, subnetGroupType) {
        return flatMap(subnetGroups.filter(subnetGroup => subnetGroup.type === subnetGroupType), subnetGroup => this.subnetGroupToSubnets(subnetGroup));
    }
    subnetGroupToSubnets(subnetGroup) {
        const ret = new Array();
        for (let i = 0; i < subnetGroup.subnets.length; i++) {
            const vpcSubnet = subnetGroup.subnets[i];
            ret.push(Subnet.fromSubnetAttributes(this, `${subnetGroup.name}Subnet${i + 1}`, {
                availabilityZone: vpcSubnet.availabilityZone,
                subnetId: vpcSubnet.subnetId,
                routeTableId: vpcSubnet.routeTableId,
            }));
        }
        return ret;
    }
}
function flatMap(xs, fn) {
    const ret = new Array();
    for (const x of xs) {
        ret.push(...fn(x));
    }
    return ret;
}
class CompositeDependable {
    constructor() {
        this.dependables = new Array();
        const self = this;
        core_1.DependableTrait.implement(this, {
            get dependencyRoots() {
                const ret = new Array();
                for (const dep of self.dependables) {
                    ret.push(...core_1.DependableTrait.get(dep).dependencyRoots);
                }
                return ret;
            },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(dep) {
        this.dependables.push(dep);
    }
}
/**
 * Invoke a function on a value (for its side effect) and return the value
 */
function tap(x, fn) {
    fn(x);
    return x;
}
class ImportedSubnet extends core_1.Resource {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        if (!attrs.routeTableId) {
            const ref = core_1.Token.isUnresolved(attrs.subnetId)
                ? `at '${scope.node.path}/${id}'`
                : `'${attrs.subnetId}'`;
            // eslint-disable-next-line max-len
            core_1.Annotations.of(this).addWarning(`No routeTableId was provided to the subnet ${ref}. Attempting to read its .routeTable.routeTableId will return null/undefined. (More info: https://github.com/aws/aws-cdk/pull/3171)`);
        }
        this._ipv4CidrBlock = attrs.ipv4CidrBlock;
        this._availabilityZone = attrs.availabilityZone;
        this.subnetId = attrs.subnetId;
        this.routeTable = {
            // Forcing routeTableId to pretend non-null to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            routeTableId: attrs.routeTableId,
        };
    }
    get availabilityZone() {
        if (!this._availabilityZone) {
            // eslint-disable-next-line max-len
            throw new Error('You cannot reference a Subnet\'s availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()');
        }
        return this._availabilityZone;
    }
    get ipv4CidrBlock() {
        if (!this._ipv4CidrBlock) {
            // tslint:disable-next-line: max-line-length
            throw new Error('You cannot reference an imported Subnet\'s IPv4 CIDR if it was not supplied. Add the ipv4CidrBlock when importing using Subnet.fromSubnetAttributes()');
        }
        return this._ipv4CidrBlock;
    }
    associateNetworkAcl(id, networkAcl) {
        const scope = core_1.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_1.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + other.node.uniqueId, {
            networkAcl,
            subnet: this,
        });
    }
}
/**
 * Determine (and validate) the NAT gateway count w.r.t. the rest of the subnet configuration
 *
 * We have the following requirements:
 *
 * - NatGatewayCount = 0 ==> there are no private subnets
 * - NatGatewayCount > 0 ==> there must be public subnets
 *
 * Do we want to require that there are private subnets if there are NatGateways?
 * They seem pointless but I see no reason to prevent it.
 */
function determineNatGatewayCount(requestedCount, subnetConfig, azCount) {
    const hasPrivateSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PRIVATE && !c.reserved);
    const hasPublicSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PUBLIC);
    const count = requestedCount !== undefined ? Math.min(requestedCount, azCount) : (hasPrivateSubnets ? azCount : 0);
    if (count === 0 && hasPrivateSubnets) {
        // eslint-disable-next-line max-len
        throw new Error('If you do not want NAT gateways (natGateways=0), make sure you don\'t configure any PRIVATE subnets in \'subnetConfiguration\' (make them PUBLIC or ISOLATED instead)');
    }
    if (count > 0 && !hasPublicSubnets) {
        // eslint-disable-next-line max-len
        throw new Error(`If you configure PRIVATE subnets in 'subnetConfiguration', you must also configure PUBLIC subnets to put the NAT gateways into (got ${JSON.stringify(subnetConfig)}.`);
    }
    return count;
}
/**
 * There are returned when the provider has not supplied props yet
 *
 * It's only used for testing and on the first run-through.
 */
const DUMMY_VPC_PROPS = {
    availabilityZones: [],
    vpcCidrBlock: '1.2.3.4/5',
    isolatedSubnetIds: undefined,
    isolatedSubnetNames: undefined,
    isolatedSubnetRouteTableIds: undefined,
    privateSubnetIds: undefined,
    privateSubnetNames: undefined,
    privateSubnetRouteTableIds: undefined,
    publicSubnetIds: undefined,
    publicSubnetNames: undefined,
    publicSubnetRouteTableIds: undefined,
    subnetGroups: [
        {
            name: 'Public',
            type: cxapi.VpcSubnetGroupType.PUBLIC,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 's-12345',
                    routeTableId: 'rtb-12345s',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 's-67890',
                    routeTableId: 'rtb-67890s',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Private',
            type: cxapi.VpcSubnetGroupType.PRIVATE,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
    ],
    vpcId: 'vpc-12345',
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidnBjLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidnBjLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDJEQUEyRDtBQUMzRCx3Q0FHdUI7QUFDdkIseUNBQXlDO0FBQ3pDLG1EQUd5QjtBQUN6QiwrQkFBb0M7QUFDcEMsK0NBQXFGO0FBQ3JGLGlEQUFnRDtBQUNoRCxxQ0FBd0M7QUFDeEMsaUNBQW1JO0FBQ25JLGlEQUFnSztBQUNoSyxtREFBK0U7QUFFL0UsK0JBQW9IO0FBRXBILE1BQU0saUJBQWlCLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0FBNkhuRTs7R0FFRztBQUNILElBQVksVUFpQ1g7QUFqQ0QsV0FBWSxVQUFVO0lBQ3BCOzs7OztPQUtHO0lBQ0gsbUNBQXFCLENBQUE7SUFFckI7Ozs7Ozs7Ozs7T0FVRztJQUNILGlDQUFtQixDQUFBO0lBRW5COzs7Ozs7Ozs7T0FTRztJQUNILCtCQUFpQixDQUFBO0FBQ25CLENBQUMsRUFqQ1csVUFBVSxHQUFWLGtCQUFVLEtBQVYsa0JBQVUsUUFpQ3JCO0FBa0hEOztHQUVHO0FBQ0gsTUFBZSxPQUFRLFNBQVEsZUFBUTtJQUF2Qzs7UUFxQ0U7Ozs7V0FJRztRQUNnQixvQkFBZSxHQUFHLElBQUksS0FBSyxFQUFjLENBQUM7UUFFN0Q7O1dBRUc7UUFDTywrQkFBMEIsR0FBWSxLQUFLLENBQUM7SUErTnhELENBQUM7SUF0TkM7O09BRUc7SUFDSSxhQUFhLENBQUMsWUFBNkIsRUFBRTtRQUNsRCxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDcEQsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRXpDLE9BQU87WUFDTCxTQUFTLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLENBQUM7WUFDdkMsSUFBSSxpQkFBaUIsS0FBZSxPQUFPLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDbEYsK0JBQStCLEVBQUUsR0FBRyxDQUFDLElBQUksbUJBQW1CLEVBQUUsRUFBRSxDQUFDLENBQUMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDLENBQUM7WUFDcEksT0FBTztZQUNQLFNBQVMsRUFBRSxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUMxQyxDQUFDO0lBQ0osQ0FBQztJQUVEOztPQUVHO0lBQ0ksZ0JBQWdCLENBQUMsT0FBZ0M7O1FBQ3RELElBQUksSUFBSSxDQUFDLFlBQVksRUFBRTtZQUNyQixNQUFNLElBQUksS0FBSyxDQUFDLDJDQUEyQyxDQUFDLENBQUM7U0FDOUQ7UUFFRCxNQUFNLFVBQVUsR0FBRyxJQUFJLGdCQUFVLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtZQUNwRCxhQUFhLEVBQUUsT0FBTyxDQUFDLGFBQWE7WUFDcEMsSUFBSSxFQUFFLHVCQUFpQixDQUFDLE9BQU87U0FDaEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGFBQWEsR0FBRyxVQUFVLENBQUMsU0FBUyxDQUFDO1FBRTFDLE1BQU0sVUFBVSxHQUFHLElBQUksdUNBQXVCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMvRCxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUs7WUFDakIsWUFBWSxFQUFFLElBQUksQ0FBQyxhQUFhO1NBQ2pDLENBQUMsQ0FBQztRQUVILHFFQUFxRTtRQUNyRSxNQUFNLG1CQUFtQixTQUFHLE9BQU8sQ0FBQyxtQkFBbUIsbUNBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUNoRSxNQUFNLGFBQWEsR0FBRyx1QkFBZ0IsQ0FBQyxjQUFPLENBQUMsbUJBQW1CLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFN0csSUFBSSxhQUFhLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUM5QixrQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQUMscURBQXFELElBQUksQ0FBQyxTQUFTLENBQUMsbUJBQW1CLENBQUMsMkNBQTJDLENBQUMsQ0FBQztTQUNwSztRQUVELE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSw2Q0FBNkIsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDbkYsYUFBYTtZQUNiLFlBQVksRUFBRSxJQUFJLENBQUMsYUFBYTtTQUNqQyxDQUFDLENBQUM7UUFDSCwrRUFBK0U7UUFDL0UsaURBQWlEO1FBQ2pELGdIQUFnSDtRQUNoSCxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7T0FFRztJQUNJLGdCQUFnQixDQUFDLEVBQVUsRUFBRSxPQUE2QjtRQUMvRCxPQUFPLElBQUksbUJBQWEsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ2pDLEdBQUcsRUFBRSxJQUFJO1lBQ1QsR0FBRyxPQUFPO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksb0JBQW9CLENBQUMsRUFBVSxFQUFFLE9BQW9DO1FBQzFFLE9BQU8sSUFBSSxtQ0FBb0IsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ3hDLEdBQUcsRUFBRSxJQUFJO1lBQ1QsR0FBRyxPQUFPO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksa0JBQWtCLENBQUMsRUFBVSxFQUFFLE9BQWtDO1FBQ3RFLE9BQU8sSUFBSSxpQ0FBa0IsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ3RDLEdBQUcsRUFBRSxJQUFJO1lBQ1QsR0FBRyxPQUFPO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksVUFBVSxDQUFDLEVBQVUsRUFBRSxPQUF3QjtRQUNwRCxPQUFPLElBQUksdUJBQU8sQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQzNCLFlBQVksRUFBRSxtQ0FBbUIsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO1lBQy9DLEdBQUcsT0FBTztTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsWUFBWTtRQUNyQixPQUFPLElBQUksQ0FBQyxhQUFhLENBQUM7SUFDNUIsQ0FBQztJQUVEOztPQUVHO0lBQ08sbUJBQW1CLENBQUMsWUFBNkIsRUFBRTs7UUFDM0QsU0FBUyxHQUFHLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUVuRCxJQUFJLFNBQVMsQ0FBQyxPQUFPLEtBQUssU0FBUyxFQUFFO1lBQ25DLE9BQU8sU0FBUyxDQUFDLE9BQU8sQ0FBQztTQUMxQjtRQUVELElBQUksT0FBTyxDQUFDO1FBRVosSUFBSSxTQUFTLENBQUMsZUFBZSxLQUFLLFNBQVMsRUFBRSxFQUFFLGlCQUFpQjtZQUM5RCxPQUFPLEdBQUcsSUFBSSxDQUFDLHlCQUF5QixDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsQ0FBQztTQUVyRTthQUFNLEVBQUUscUJBQXFCO1lBQzVCLE1BQU0sSUFBSSxHQUFHLFNBQVMsQ0FBQyxVQUFVLElBQUksVUFBVSxDQUFDLE9BQU8sQ0FBQztZQUN4RCxPQUFPLEdBQUcsSUFBSSxDQUFDLHlCQUF5QixDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ2hEO1FBRUQsd0JBQXdCO1FBQ3hCLE9BQU8sR0FBRyxJQUFJLENBQUMsa0JBQWtCLENBQUMsT0FBTyxRQUFFLFNBQVMsQ0FBQyxhQUFhLG1DQUFJLEVBQUUsQ0FBQyxDQUFDO1FBRTFFLE9BQU8sT0FBTyxDQUFDO0lBQ2pCLENBQUM7SUFFTyxrQkFBa0IsQ0FBQyxPQUFrQixFQUFFLE9BQXVCO1FBQ3BFLElBQUksUUFBUSxHQUFHLE9BQU8sQ0FBQztRQUN2QixnQ0FBZ0M7UUFDaEMsS0FBSyxNQUFNLE1BQU0sSUFBSSxPQUFPLEVBQUU7WUFDNUIsUUFBUSxHQUFHLE1BQU0sQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDM0M7UUFDRCxPQUFPLFFBQVEsQ0FBQztJQUNsQixDQUFDO0lBRU8seUJBQXlCLENBQUMsU0FBaUI7UUFDakQsTUFBTSxVQUFVLEdBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQyxhQUFhLEVBQUUsR0FBRyxJQUFJLENBQUMsY0FBYyxFQUFFLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQzVGLE1BQU0sT0FBTyxHQUFHLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQ0FBOEIsQ0FBQyxDQUFDLENBQUMsS0FBSyxTQUFTLENBQUMsQ0FBQztRQUV4RixJQUFJLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLDBCQUEwQixFQUFFO1lBQzVELE1BQU0sS0FBSyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxxQ0FBOEIsQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUNsRixNQUFNLElBQUksS0FBSyxDQUFDLHlDQUF5QyxTQUFTLG1DQUFtQyxLQUFLLEVBQUUsQ0FBQyxDQUFDO1NBQy9HO1FBRUQsT0FBTyxPQUFPLENBQUM7SUFDakIsQ0FBQztJQUVPLHlCQUF5QixDQUFDLFVBQXNCO1FBQ3RELE1BQU0sVUFBVSxHQUFHO1lBQ2pCLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxFQUFFLElBQUksQ0FBQyxlQUFlO1lBQzNDLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxFQUFFLElBQUksQ0FBQyxjQUFjO1lBQ3pDLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxFQUFFLElBQUksQ0FBQyxhQUFhO1NBQ3hDLENBQUM7UUFFRixNQUFNLE9BQU8sR0FBRyxVQUFVLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFdkMsMEVBQTBFO1FBQzFFLGtCQUFrQjtRQUVsQixJQUFJLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLDBCQUEwQixFQUFFO1lBQzVELE1BQU0sY0FBYyxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsSUFBSSxDQUFDLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQzFILE1BQU0sSUFBSSxLQUFLLENBQUMsaUJBQWlCLFVBQVUsaURBQWlELGNBQWMsRUFBRSxDQUFDLENBQUM7U0FDL0c7UUFFRCxPQUFPLE9BQU8sQ0FBQztJQUNqQixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSyxzQkFBc0IsQ0FBQyxTQUEwQjs7UUFFdkQsSUFBSSxTQUFTLENBQUMsVUFBVSxLQUFLLFNBQVMsRUFBRTtZQUN0QyxJQUFJLFNBQVMsQ0FBQyxlQUFlLEtBQUssU0FBUyxFQUFFO2dCQUMzQyxNQUFNLElBQUksS0FBSyxDQUFDLDhGQUE4RixDQUFDLENBQUM7YUFDakg7aUJBQU07Z0JBQ0wsa0JBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUFDLDJGQUEyRixDQUFDLENBQUM7YUFDOUg7WUFDRCxTQUFTLEdBQUcsRUFBRSxHQUFHLFNBQVMsRUFBRSxlQUFlLEVBQUUsU0FBUyxDQUFDLFVBQVUsRUFBRSxDQUFDO1NBQ3JFO1FBRUQsTUFBTSxtQkFBbUIsR0FBaUMsQ0FBQyxTQUFTLEVBQUUsWUFBWSxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFDdkcsTUFBTSxrQkFBa0IsR0FBRyxtQkFBbUIsQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLEtBQUssU0FBUyxDQUFDLENBQUM7UUFDM0YsSUFBSSxrQkFBa0IsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ2pDLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0JBQWdCLGtCQUFrQix3Q0FBd0MsQ0FBQyxDQUFDO1NBQzdGO1FBRUQsSUFBSSxTQUFTLENBQUMsVUFBVSxLQUFLLFNBQVMsSUFBSSxTQUFTLENBQUMsZUFBZSxLQUFLLFNBQVMsSUFBSSxTQUFTLENBQUMsT0FBTyxLQUFLLFNBQVMsRUFBRTtZQUNwSCxrRUFBa0U7WUFDbEUsSUFBSSxVQUFVLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDO1lBQ3pJLFNBQVMsR0FBRyxFQUFFLEdBQUcsU0FBUyxFQUFFLFVBQVUsRUFBRSxVQUFVLEVBQUUsQ0FBQztTQUN0RDtRQUVELHNEQUFzRDtRQUN0RCxJQUFJLGFBQWEsU0FBRyxTQUFTLENBQUMsYUFBYSxtQ0FBSSxFQUFFLENBQUM7UUFFbEQseUZBQXlGO1FBQ3pGLElBQUksU0FBUyxDQUFDLGlCQUFpQixLQUFLLFNBQVMsRUFBRSxFQUFFLDhCQUE4QjtZQUM3RSxhQUFhLENBQUMsSUFBSSxDQUFDLHFCQUFZLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQztTQUNqRjtRQUNELElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQyxRQUFRLEVBQUUsRUFBRSxpQ0FBaUM7WUFDM0QsYUFBYSxDQUFDLElBQUksQ0FBQyxxQkFBWSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7U0FDN0M7UUFFRCxvR0FBb0c7UUFDcEcsU0FBUyxHQUFHLEVBQUUsR0FBRyxTQUFTLEVBQUUsYUFBYSxFQUFFLGFBQWEsRUFBRSxpQkFBaUIsRUFBRSxTQUFTLEVBQUUsUUFBUSxFQUFFLFNBQVMsRUFBRSxDQUFDO1FBQzlHLE1BQU0sRUFBRSxpQkFBaUIsRUFBRSxRQUFRLEVBQUUsR0FBRyxJQUFJLEVBQUUsR0FBRyxTQUFTLENBQUM7UUFFM0QsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0NBQ0Y7QUF5SEQ7O0dBRUc7QUFDSCxNQUFNLFFBQVEsR0FBVyxNQUFNLENBQUM7QUE2TGhDOztHQUVHO0FBQ0gsSUFBWSxzQkFVWDtBQVZELFdBQVksc0JBQXNCO0lBQ2hDOztPQUVHO0lBQ0gsNkNBQW1CLENBQUE7SUFFbkI7O09BRUc7SUFDSCxpREFBdUIsQ0FBQTtBQUN6QixDQUFDLEVBVlcsc0JBQXNCLEdBQXRCLDhCQUFzQixLQUF0Qiw4QkFBc0IsUUFVakM7QUErQ0Q7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztHQXdCRztBQUNILE1BQWEsR0FBSSxTQUFRLE9BQU87SUE2TDlCOzs7OztPQUtHO0lBQ0gsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFrQixFQUFFO1FBQzVELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUE5RG5COztXQUVHO1FBQ2Esa0JBQWEsR0FBYyxFQUFFLENBQUM7UUFFOUM7O1dBRUc7UUFDYSxtQkFBYyxHQUFjLEVBQUUsQ0FBQztRQUUvQzs7V0FFRztRQUNhLG9CQUFlLEdBQWMsRUFBRSxDQUFDO1FBbUNoRDs7V0FFRztRQUNLLHdCQUFtQixHQUEwQixFQUFFLENBQUM7UUFFdkMscUNBQWdDLEdBQUcsSUFBSSx5QkFBa0IsRUFBRSxDQUFDO1FBVzNFLE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFN0IsMERBQTBEO1FBQzFELElBQUksS0FBSyxDQUFDLGtCQUFrQixJQUFJLENBQUMsS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQ3ZELE1BQU0sSUFBSSxLQUFLLENBQUMseUZBQXlGLENBQUMsQ0FBQztTQUM1RztRQUVELE1BQU0sU0FBUyxHQUFHLFdBQVcsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQ2xFLElBQUksWUFBSyxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUNqQyxNQUFNLElBQUksS0FBSyxDQUFDLCtHQUErRyxDQUFDLENBQUM7U0FDbEk7UUFFRCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksNkJBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUVwRCxJQUFJLENBQUMsbUJBQW1CLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLElBQUksQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUM7UUFDOUYsSUFBSSxDQUFDLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDO1FBQ3hGLE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxzQkFBc0IsSUFBSSxTQUFTLENBQUM7UUFDbEUsSUFBSSxDQUFDLCtCQUErQixHQUFHLElBQUksQ0FBQyxnQ0FBZ0MsQ0FBQztRQUU3RSw2Q0FBNkM7UUFDN0MsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLHNCQUFNLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMzQyxTQUFTO1lBQ1Qsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLG1CQUFtQjtZQUM1QyxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsaUJBQWlCO1lBQ3hDLGVBQWU7U0FDaEIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLG9CQUFvQixHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMscUJBQXFCLENBQUM7UUFDaEUsSUFBSSxDQUFDLHdCQUF3QixHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMseUJBQXlCLENBQUM7UUFDeEUsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQztRQUNoRCxJQUFJLENBQUMsdUJBQXVCLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyx3QkFBd0IsQ0FBQztRQUN0RSxJQUFJLENBQUMsaUJBQWlCLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQztRQUUxRCxXQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU1QyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1FBRWpELE1BQU0sTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDN0QsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRWpFLElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUM7UUFFL0IsTUFBTSxhQUFhLEdBQUcsS0FBSyxDQUFDLFdBQVcsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQztRQUNqRyxJQUFJLENBQUMsbUJBQW1CLEdBQUcsV0FBVyxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUVqRixNQUFNLG1CQUFtQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDekYsTUFBTSxlQUFlLEdBQUcsd0JBQXdCLENBQUMsS0FBSyxDQUFDLFdBQVcsRUFBRSxJQUFJLENBQUMsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBRTdILCtEQUErRDtRQUMvRCxJQUFJLENBQUMsYUFBYSxFQUFFLENBQUM7UUFFckIsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLE1BQU0sQ0FDbkQsTUFBTSxDQUFDLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxVQUFVLEtBQUssVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztRQUVwRSx3REFBd0Q7UUFDeEQsSUFBSSxhQUFhLEVBQUU7WUFDakIsTUFBTSxHQUFHLEdBQUcsSUFBSSxrQ0FBa0IsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLEVBQy9DLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxpQkFBaUIsR0FBRyxHQUFHLENBQUMsR0FBRyxDQUFDO1lBRWpDLElBQUksQ0FBQyxnQ0FBZ0MsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDL0MsTUFBTSxHQUFHLEdBQUcsSUFBSSx1Q0FBdUIsQ0FBQyxJQUFJLEVBQUUsT0FBTyxFQUFFO2dCQUNyRCxpQkFBaUIsRUFBRSxHQUFHLENBQUMsR0FBRztnQkFDMUIsS0FBSyxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRzthQUN6QixDQUFDLENBQUM7WUFFRixJQUFJLENBQUMsYUFBZ0MsQ0FBQyxPQUFPLENBQUMsWUFBWSxDQUFDLEVBQUU7Z0JBQzVELFlBQVksQ0FBQyx1QkFBdUIsQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1lBQ3JELENBQUMsQ0FBQyxDQUFDO1lBRUgscUNBQXFDO1lBQ3JDLElBQUksZUFBZSxHQUFHLENBQUMsRUFBRTtnQkFDdkIsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixJQUFJLGlCQUFXLENBQUMsT0FBTyxFQUFFLENBQUM7Z0JBQ25FLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxRQUFRLEVBQUUsZUFBZSxFQUFFLG1CQUFtQixDQUFDLENBQUM7YUFDeEU7U0FDRjtRQUVELElBQUksS0FBSyxDQUFDLFVBQVUsSUFBSSxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksSUFBSSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLElBQUksQ0FBQyxlQUFlLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNoSSxNQUFNLElBQUksS0FBSyxDQUFDLG9FQUFvRSxDQUFDLENBQUM7U0FDdkY7UUFFRCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLElBQUksS0FBSyxDQUFDLFVBQVUsS0FBSyxLQUFLLEVBQUU7WUFDL0UsTUFBTSxJQUFJLEtBQUssQ0FBQyx1RkFBdUYsQ0FBQyxDQUFDO1NBQzFHO1FBRUQsSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQyxjQUFjLElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUNuRSxJQUFJLENBQUMsZ0JBQWdCLENBQUM7Z0JBQ3BCLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtnQkFDbEMsSUFBSSxFQUFFLHVCQUFpQixDQUFDLE9BQU87Z0JBQy9CLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxtQkFBbUI7YUFDL0MsQ0FBQyxDQUFDO1lBRUgsTUFBTSxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSSxFQUFFLENBQUM7WUFDbEQsS0FBSyxNQUFNLENBQUMsWUFBWSxFQUFFLFVBQVUsQ0FBQyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLEVBQUU7Z0JBQ3ZFLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxZQUFZLEVBQUUsVUFBVSxDQUFDLENBQUM7YUFDakQ7U0FDRjtRQUVELDJFQUEyRTtRQUMzRSw2RUFBNkU7UUFDN0UsdUVBQXVFO1FBQ3ZFLElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQzFCLE1BQU0sZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixJQUFJLEVBQUUsQ0FBQztZQUN0RCxLQUFLLE1BQU0sQ0FBQyxVQUFVLEVBQUUsUUFBUSxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFO2dCQUNyRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsVUFBVSxFQUFFLFFBQVEsQ0FBQyxDQUFDO2FBQy9DO1NBQ0Y7UUFFRCwyQkFBMkI7UUFDM0IsSUFBSSxLQUFLLENBQUMsUUFBUSxFQUFFO1lBQ2xCLE1BQU0sUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLElBQUksRUFBRSxDQUFDO1lBQ3RDLEtBQUssTUFBTSxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFO2dCQUMzRCxJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsQ0FBQzthQUNyQztTQUNGO0lBQ0gsQ0FBQztJQWxSRDs7T0FFRztJQUNJLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFvQjtRQUNoRixPQUFPLElBQUksV0FBVyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFRDs7Ozs7Ozs7Ozs7Ozs7O09BZUc7SUFDSSxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLE9BQXlCO1FBQzlFLElBQUksWUFBSyxDQUFDLFlBQVksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDO2VBQ2hDLFlBQUssQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQztlQUNuQyxNQUFNLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLElBQUksRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQUssQ0FBQyxZQUFZLENBQUM7ZUFDMUQsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxJQUFJLEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxZQUFLLENBQUMsWUFBWSxDQUFDLEVBQUU7WUFDN0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxnRUFBZ0UsQ0FBQyxDQUFDO1NBQ25GO1FBRUQsTUFBTSxNQUFNLEdBQTRCLGFBQWEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFcEUseUNBQXlDO1FBQ3pDLElBQUksT0FBTyxDQUFDLEtBQUssRUFBRTtZQUFFLE1BQU0sQ0FBQyxRQUFRLENBQUMsR0FBRyxPQUFPLENBQUMsS0FBSyxDQUFDO1NBQUU7UUFDeEQsSUFBSSxPQUFPLENBQUMsT0FBTyxFQUFFO1lBQUUsTUFBTSxDQUFDLFVBQVUsQ0FBQyxHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUM7U0FBRTtRQUM5RCxJQUFJLE9BQU8sQ0FBQyxTQUFTLEtBQUssU0FBUyxFQUFFO1lBQ25DLE1BQU0sQ0FBQyxTQUFTLEdBQUcsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUM7U0FDekQ7UUFFRCxNQUFNLFVBQVUsR0FBNkIsc0JBQWUsQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFO1lBQzNFLFFBQVEsRUFBRSxRQUFRLENBQUMsZUFBZSxDQUFDLFlBQVk7WUFDL0MsS0FBSyxFQUFFO2dCQUNMLE1BQU07Z0JBQ04sdUJBQXVCLEVBQUUsSUFBSTtnQkFDN0Isa0JBQWtCLEVBQUUsT0FBTyxDQUFDLGtCQUFrQjthQUNuQjtZQUM3QixVQUFVLEVBQUUsU0FBUztTQUN0QixDQUFDLENBQUMsS0FBSyxDQUFDO1FBRVQsT0FBTyxJQUFJLFdBQVcsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLFVBQVUsSUFBSSxlQUFlLEVBQUUsVUFBVSxLQUFLLFNBQVMsQ0FBQyxDQUFDO1FBRTNGOztXQUVHO1FBQ0gsU0FBUyxhQUFhLENBQUMsSUFBNEM7WUFDakUsTUFBTSxNQUFNLEdBQStCLEVBQUUsQ0FBQztZQUM5QyxLQUFLLE1BQU0sQ0FBQyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLElBQUksRUFBRSxDQUFDLEVBQUU7Z0JBQ3RELE1BQU0sQ0FBQyxPQUFPLElBQUksRUFBRSxDQUFDLEdBQUcsS0FBSyxDQUFDO2FBQy9CO1lBQ0QsT0FBTyxNQUFNLENBQUM7UUFDaEIsQ0FBQztJQUNILENBQUM7SUFzTkQ7Ozs7T0FJRztJQUNJLGFBQWEsQ0FBQyxFQUFVLEVBQUUsT0FBMkI7UUFDMUQsT0FBTyxJQUFJLGlDQUFrQixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDdEMsT0FBTyxFQUFFLDJDQUE0QixDQUFDLEVBQUU7WUFDeEMsR0FBRyxFQUFFLElBQUk7WUFDVCxPQUFPO1NBQ1IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxtQkFBbUIsQ0FBQyxFQUFVLEVBQUUsT0FBMkI7UUFDaEUsT0FBTyxJQUFJLGlDQUFrQixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDdEMsT0FBTyxFQUFFLDJDQUE0QixDQUFDLFFBQVE7WUFDOUMsR0FBRyxFQUFFLElBQUk7WUFDVCxPQUFPO1NBQ1IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLGlCQUFpQixDQUFDLFFBQXFCLEVBQUUsUUFBZ0IsRUFBRSxTQUEwQjtRQUMzRixNQUFNLFVBQVUsR0FBbUIsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFNBQVMsQ0FBbUIsQ0FBQztRQUN6RixLQUFLLE1BQU0sR0FBRyxJQUFJLFVBQVUsRUFBRTtZQUM1QixJQUFJLElBQUksQ0FBQyxhQUFhLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO2dCQUMxQyxNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixTQUFTLCtCQUErQixHQUFHLEVBQUUsQ0FBQyxDQUFDO2FBQ3ZGO1NBQ0Y7UUFFRCxRQUFRLENBQUMsWUFBWSxDQUFDO1lBQ3BCLEdBQUcsRUFBRSxJQUFJO1lBQ1QsVUFBVSxFQUFFLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLFFBQVEsQ0FBQztZQUN6QyxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWlDO1NBQ3ZELENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7O09BR0c7SUFDSyxhQUFhO1FBQ25CLE1BQU0scUJBQXFCLEdBQTBCLEVBQUUsQ0FBQztRQUV4RCxLQUFLLE1BQU0sTUFBTSxJQUFJLElBQUksQ0FBQyxtQkFBbUIsRUFBRTtZQUM3QyxJQUFJLE1BQU0sQ0FBQyxRQUFRLEtBQUssU0FBUyxFQUFFO2dCQUNqQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7Z0JBQ25DLFNBQVM7YUFDVjtZQUNELElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsTUFBTSxjQUFjLEdBQUcscUJBQXFCLENBQUMsTUFBTSxHQUFHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxNQUFNLENBQUM7UUFDcEYsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLHVCQUF1QixDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ3ZGLEtBQUssTUFBTSxNQUFNLElBQUkscUJBQXFCLEVBQUU7WUFDMUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLE1BQU0sRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO1NBQ3hEO0lBQ0gsQ0FBQztJQUVPLHFCQUFxQixDQUFDLFlBQWlDLEVBQUUsUUFBZ0I7UUFDL0UsSUFBSSxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUUsRUFBRTtZQUM3QyxJQUFJLFlBQVksQ0FBQyxRQUFRLEtBQUssSUFBSSxFQUFFO2dCQUNsQywrRUFBK0U7Z0JBQy9FLElBQUksQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDO2dCQUN4QyxPQUFPO2FBQ1I7WUFFRCxNQUFNLElBQUksR0FBRyxlQUFRLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztZQUNoRCxNQUFNLFdBQVcsR0FBZ0I7Z0JBQy9CLGdCQUFnQixFQUFFLElBQUk7Z0JBQ3RCLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSztnQkFDakIsU0FBUyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQztnQkFDbEQsbUJBQW1CLEVBQUUsQ0FBQyxZQUFZLENBQUMsVUFBVSxLQUFLLFVBQVUsQ0FBQyxNQUFNLENBQUM7YUFDckUsQ0FBQztZQUVGLElBQUksTUFBYyxDQUFDO1lBQ25CLFFBQVEsWUFBWSxDQUFDLFVBQVUsRUFBRTtnQkFDL0IsS0FBSyxVQUFVLENBQUMsTUFBTTtvQkFDcEIsTUFBTSxZQUFZLEdBQUcsSUFBSSxZQUFZLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FBQztvQkFDL0QsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7b0JBQ3RDLE1BQU0sR0FBRyxZQUFZLENBQUM7b0JBQ3RCLE1BQU07Z0JBQ1IsS0FBSyxVQUFVLENBQUMsT0FBTztvQkFDckIsTUFBTSxhQUFhLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FBQztvQkFDakUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUM7b0JBQ3hDLE1BQU0sR0FBRyxhQUFhLENBQUM7b0JBQ3ZCLE1BQU07Z0JBQ1IsS0FBSyxVQUFVLENBQUMsUUFBUTtvQkFDdEIsTUFBTSxjQUFjLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxXQUFXLENBQUMsQ0FBQztvQkFDbEUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7b0JBQzFDLE1BQU0sR0FBRyxjQUFjLENBQUM7b0JBQ3hCLE1BQU07Z0JBQ1I7b0JBQ0UsTUFBTSxJQUFJLEtBQUssQ0FBQyw2QkFBNkIsWUFBWSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7YUFDM0U7WUFFRCx1RUFBdUU7WUFDdkUsTUFBTSxvQkFBb0IsR0FBRyxDQUFDLHlCQUFTLENBQUMsc0JBQXNCLENBQUMsQ0FBQztZQUNoRSxXQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsQ0FBQyxjQUFjLEVBQUUsWUFBWSxDQUFDLElBQUksRUFBRSxFQUFFLG9CQUFvQixFQUFFLENBQUMsQ0FBQztZQUNqRixXQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEdBQUcsQ0FBQyxjQUFjLEVBQUUsa0JBQWtCLENBQUMsWUFBWSxDQUFDLFVBQVUsQ0FBQyxFQUFFLEVBQUUsb0JBQW9CLEVBQUUsQ0FBQyxDQUFDO1FBQzdHLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7QUFyYUgsa0JBc2FDO0FBcmFDOzs7O0dBSUc7QUFDb0Isc0JBQWtCLEdBQVcsYUFBYSxDQUFDO0FBRWxFOzs7O0dBSUc7QUFDb0IsbUJBQWUsR0FBMEI7SUFDOUQ7UUFDRSxVQUFVLEVBQUUsVUFBVSxDQUFDLE1BQU07UUFDN0IsSUFBSSxFQUFFLHdCQUFpQixDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUM7S0FDM0M7SUFDRDtRQUNFLFVBQVUsRUFBRSxVQUFVLENBQUMsT0FBTztRQUM5QixJQUFJLEVBQUUsd0JBQWlCLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQztLQUM1QztDQUNGLENBQUM7QUFFRjs7OztHQUlHO0FBQ29CLDBCQUFzQixHQUEwQjtJQUNyRTtRQUNFLFVBQVUsRUFBRSxVQUFVLENBQUMsTUFBTTtRQUM3QixJQUFJLEVBQUUsd0JBQWlCLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQztLQUMzQztJQUNEO1FBQ0UsVUFBVSxFQUFFLFVBQVUsQ0FBQyxRQUFRO1FBQy9CLElBQUksRUFBRSx3QkFBaUIsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDO0tBQzdDO0NBQ0YsQ0FBQztBQWtZSixNQUFNLGNBQWMsR0FBRyxxQkFBcUIsQ0FBQztBQUM3QyxNQUFNLGNBQWMsR0FBRyxxQkFBcUIsQ0FBQztBQUU3QyxTQUFTLGtCQUFrQixDQUFDLElBQWdCO0lBQzFDLFFBQVEsSUFBSSxFQUFFO1FBQ1osS0FBSyxVQUFVLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxRQUFRLENBQUM7UUFDeEMsS0FBSyxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxTQUFTLENBQUM7UUFDMUMsS0FBSyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxVQUFVLENBQUM7S0FDN0M7QUFDSCxDQUFDO0FBOEJEOzs7O0dBSUc7QUFDSCxNQUFhLE1BQU8sU0FBUSxlQUFRO0lBcUVsQyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQWtCO1FBQzFELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFqQm5COztXQUVHO1FBQ2EsdUJBQWtCLEdBQWtCLEVBQUUsQ0FBQztRQVN0QyxxQ0FBZ0MsR0FBRyxJQUFJLHlCQUFrQixFQUFFLENBQUM7UUFPM0UsTUFBTSxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUUsRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUVoRSxXQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUU1QyxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixDQUFDO1FBQy9DLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztRQUNyQyxNQUFNLE1BQU0sR0FBRyxJQUFJLHlCQUFTLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtZQUMzQyxLQUFLLEVBQUUsS0FBSyxDQUFDLEtBQUs7WUFDbEIsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQzFCLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7WUFDeEMsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLG1CQUFtQjtTQUMvQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsUUFBUSxHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUM7UUFDM0IsSUFBSSxDQUFDLFdBQVcsR0FBRyxNQUFNLENBQUMsU0FBUyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxzQkFBc0IsR0FBRyxNQUFNLENBQUMsb0JBQW9CLENBQUM7UUFDMUQsSUFBSSxDQUFDLG9CQUFvQixHQUFHLE1BQU0sQ0FBQyxrQkFBa0IsQ0FBQztRQUV0RCx5RUFBeUU7UUFDekUsc0VBQXNFO1FBQ3RFLElBQUksQ0FBQyxXQUFXLEdBQUcsd0JBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ2hHLElBQUksQ0FBQyw2QkFBNkIsR0FBRyxXQUFJLENBQUMsV0FBVyxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsWUFBWSxFQUFFLENBQUMsQ0FBQztRQUN4RyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksR0FBRyxNQUFNLENBQUM7UUFFaEMsTUFBTSxLQUFLLEdBQUcsSUFBSSw2QkFBYSxDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7WUFDbEQsS0FBSyxFQUFFLEtBQUssQ0FBQyxLQUFLO1NBQ25CLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxVQUFVLEdBQUcsRUFBRSxZQUFZLEVBQUUsS0FBSyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBRTlDLG1FQUFtRTtRQUNuRSxJQUFJLDhDQUE4QixDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtZQUNoRSxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7WUFDdkIsWUFBWSxFQUFFLEtBQUssQ0FBQyxHQUFHO1NBQ3hCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQywrQkFBK0IsR0FBRyxJQUFJLENBQUMsZ0NBQWdDLENBQUM7SUFDL0UsQ0FBQztJQXpHTSxNQUFNLENBQUMsV0FBVyxDQUFDLENBQU07UUFDOUIsT0FBTyxpQkFBaUIsSUFBSSxDQUFDLENBQUM7SUFDaEMsQ0FBQztJQUVNLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QjtRQUN0RixPQUFPLElBQUksY0FBYyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVEOztPQUVHO0lBQ0gsd0RBQXdEO0lBQ2pELE1BQU0sQ0FBQyxZQUFZLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBZ0I7UUFDdkUsT0FBTyxJQUFJLENBQUMsb0JBQW9CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7SUFDNUQsQ0FBQztJQTZGRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUIsQ0FBQyxTQUFpQixFQUFFLGlCQUE4QjtRQUM5RSxNQUFNLEtBQUssR0FBRyxJQUFJLHdCQUFRLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUMvQyxZQUFZLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxZQUFZO1lBQzFDLG9CQUFvQixFQUFFLFdBQVc7WUFDakMsU0FBUztTQUNWLENBQUMsQ0FBQztRQUNILEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFFNUMsNERBQTREO1FBQzVELG9DQUFvQztRQUNwQyxJQUFJLENBQUMsZ0NBQWdDLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSCxJQUFXLFVBQVU7UUFDbkIsT0FBTyxJQUFJLENBQUMsV0FBVyxDQUFDO0lBQzFCLENBQUM7SUFFRDs7O09BR0c7SUFDSSxrQkFBa0IsQ0FBQyxZQUFvQjtRQUM1QyxJQUFJLENBQUMsUUFBUSxDQUFDLGNBQWMsRUFBRTtZQUM1QixVQUFVLEVBQUUsVUFBVSxDQUFDLFdBQVc7WUFDbEMsUUFBUSxFQUFFLFlBQVk7WUFDdEIsMkJBQTJCLEVBQUUsSUFBSTtTQUNsQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxRQUFRLENBQUMsRUFBVSxFQUFFLE9BQXdCO1FBQ2xELElBQUksT0FBTyxDQUFDLG9CQUFvQixJQUFJLE9BQU8sQ0FBQyx3QkFBd0IsRUFBRTtZQUNwRSxNQUFNLElBQUksS0FBSyxDQUFDLCtFQUErRSxDQUFDLENBQUM7U0FDbEc7UUFFRCxNQUFNLEtBQUssR0FBRyxJQUFJLHdCQUFRLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUNuQyxZQUFZLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxZQUFZO1lBQzFDLG9CQUFvQixFQUFFLE9BQU8sQ0FBQyxvQkFBb0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyx3QkFBd0IsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ2hJLHdCQUF3QixFQUFFLE9BQU8sQ0FBQyx3QkFBd0I7WUFDMUQsQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUMsRUFBRSxPQUFPLENBQUMsUUFBUTtTQUM3RCxDQUFDLENBQUM7UUFFSCxJQUFJLE9BQU8sQ0FBQywyQkFBMkIsRUFBRTtZQUN2QyxJQUFJLENBQUMsZ0NBQWdDLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ2xEO0lBQ0gsQ0FBQztJQUVNLG1CQUFtQixDQUFDLEVBQVUsRUFBRSxVQUF1QjtRQUM1RCxJQUFJLENBQUMsV0FBVyxHQUFHLFVBQVUsQ0FBQztRQUU5QixNQUFNLEtBQUssR0FBRyxnQkFBUyxDQUFDLFdBQVcsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7UUFDcEUsTUFBTSxLQUFLLEdBQUcsZ0JBQVMsQ0FBQyxXQUFXLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDO1FBQ3BFLElBQUkseUNBQTJCLENBQUMsS0FBSyxFQUFFLEVBQUUsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRTtZQUMvRCxVQUFVO1lBQ1YsTUFBTSxFQUFFLElBQUk7U0FDYixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUF6TEQsd0JBeUxDO0FBMkNEOztHQUVHO0FBQ0gsSUFBWSxVQThCWDtBQTlCRCxXQUFZLFVBQVU7SUFDcEI7O09BRUc7SUFDSCx3RUFBMEQsQ0FBQTtJQUUxRDs7T0FFRztJQUNILGlDQUFtQixDQUFBO0lBRW5COztPQUVHO0lBQ0gsbUNBQXFCLENBQUE7SUFFckI7O09BRUc7SUFDSCx3Q0FBMEIsQ0FBQTtJQUUxQjs7T0FFRztJQUNILG9EQUFzQyxDQUFBO0lBRXRDOztPQUVHO0lBQ0gsNkRBQStDLENBQUE7QUFDakQsQ0FBQyxFQTlCVyxVQUFVLEdBQVYsa0JBQVUsS0FBVixrQkFBVSxRQThCckI7QUFFRCxTQUFTLG9CQUFvQixDQUFDLFVBQXNCO0lBQ2xELE9BQU8sQ0FBQztRQUNOLENBQUMsVUFBVSxDQUFDLDRCQUE0QixDQUFDLEVBQUUsNkJBQTZCO1FBQ3hFLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxFQUFFLFdBQVc7UUFDakMsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLEVBQUUsWUFBWTtRQUNuQyxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxjQUFjO1FBQ3hDLENBQUMsVUFBVSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsb0JBQW9CO1FBQ3BELENBQUMsVUFBVSxDQUFDLHNCQUFzQixDQUFDLEVBQUUsd0JBQXdCO0tBQzlELENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQztBQUNqQixDQUFDO0FBVUQ7O0dBRUc7QUFDSCxNQUFhLFlBQWEsU0FBUSxNQUFNO0lBRS9CLE1BQU0sQ0FBQywwQkFBMEIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE2QjtRQUNsRyxPQUFPLElBQUksY0FBYyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBd0I7UUFDaEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDMUIsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxhQUFhO1FBQ2xCLDZDQUE2QztRQUM3QyxNQUFNLEdBQUcsR0FBRyxJQUFJLDZCQUFhLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtZQUNoRCxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7WUFDdkIsWUFBWSxFQUFFLElBQUksc0JBQU0sQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO2dCQUNwQyxNQUFNLEVBQUUsS0FBSzthQUNkLENBQUMsQ0FBQyxnQkFBZ0I7U0FDcEIsQ0FBQyxDQUFDO1FBQ0gsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDO0NBQ0Y7QUF6QkQsb0NBeUJDO0FBVUQ7O0dBRUc7QUFDSCxNQUFhLGFBQWMsU0FBUSxNQUFNO0lBRWhDLE1BQU0sQ0FBQywyQkFBMkIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE4QjtRQUNwRyxPQUFPLElBQUksY0FBYyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDOUMsQ0FBQztJQUVELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDMUIsQ0FBQztDQUNGO0FBVEQsc0NBU0M7QUFFRCxTQUFTLFdBQVcsQ0FBSSxLQUFvQixFQUFFLFlBQWU7SUFDM0QsT0FBTyxLQUFLLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQztBQUNwRCxDQUFDO0FBRUQsTUFBTSxXQUFZLFNBQVEsT0FBTztJQVMvQixZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW9CLEVBQUUsWUFBcUI7UUFDbkYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUpILG9DQUErQixHQUFnQixJQUFJLHlCQUFrQixFQUFFLENBQUM7UUFNdEYsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDO1FBQ3pCLElBQUksQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUMvQixJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixDQUFDO1FBQ2pELElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUN4QyxJQUFJLENBQUMsMEJBQTBCLEdBQUcsWUFBWSxDQUFDO1FBRS9DLDRCQUE0QjtRQUM1QixNQUFNLEdBQUcsR0FBRyxJQUFJLHdCQUFpQixDQUFDLEtBQUssQ0FBQyxlQUFlLEVBQUUsS0FBSyxDQUFDLGlCQUFpQixFQUFFLEtBQUssQ0FBQyx5QkFBeUIsRUFBRSxVQUFVLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxpQkFBaUIsRUFBRSxpQkFBaUIsRUFBRSxtQkFBbUIsRUFBRSwyQkFBMkIsQ0FBQyxDQUFDO1FBQ25PLE1BQU0sSUFBSSxHQUFHLElBQUksd0JBQWlCLENBQUMsS0FBSyxDQUFDLGdCQUFnQixFQUFFLEtBQUssQ0FBQyxrQkFBa0IsRUFBRSxLQUFLLENBQUMsMEJBQTBCLEVBQUUsVUFBVSxDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsaUJBQWlCLEVBQUUsa0JBQWtCLEVBQUUsb0JBQW9CLEVBQUUsNEJBQTRCLENBQUMsQ0FBQztRQUMzTyxNQUFNLEdBQUcsR0FBRyxJQUFJLHdCQUFpQixDQUFDLEtBQUssQ0FBQyxpQkFBaUIsRUFBRSxLQUFLLENBQUMsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLDJCQUEyQixFQUFFLFVBQVUsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixFQUFFLG1CQUFtQixFQUFFLHFCQUFxQixFQUFFLDZCQUE2QixDQUFDLENBQUM7UUFDalAsMkJBQTJCO1FBRTNCLElBQUksQ0FBQyxhQUFhLEdBQUcsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN0QyxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDeEMsSUFBSSxDQUFDLGVBQWUsR0FBRyxHQUFHLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRCxJQUFXLFlBQVk7UUFDckIsSUFBSSxJQUFJLENBQUMsSUFBSSxLQUFLLFNBQVMsRUFBRTtZQUMzQixNQUFNLElBQUksS0FBSyxDQUFDLHlGQUF5RixDQUFDLENBQUM7U0FDNUc7UUFDRCxPQUFPLElBQUksQ0FBQyxJQUFJLENBQUM7SUFDbkIsQ0FBQztDQUNGO0FBRUQsTUFBTSxXQUFZLFNBQVEsT0FBTztJQVMvQixZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQStCLEVBQUUsWUFBcUI7UUFDOUYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQVJILG9DQUErQixHQUFnQixJQUFJLHlCQUFrQixFQUFFLENBQUM7UUFVdEYsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDO1FBQ3pCLElBQUksQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUMvQixJQUFJLENBQUMsYUFBYSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUM7UUFDeEMsSUFBSSxDQUFDLDBCQUEwQixHQUFHLFlBQVksQ0FBQztRQUUvQyxNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJLEVBQUUsQ0FBQztRQUM5QyxNQUFNLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxHQUFHLENBQVMsT0FBTyxDQUFDLFlBQVksRUFBRSxXQUFXLENBQUMsRUFBRTtZQUN2RixPQUFPLFdBQVcsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDcEUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ0wsaUJBQWlCLENBQUMsSUFBSSxDQUFDLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRSxFQUFFLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQzdELElBQUksQ0FBQyxpQkFBaUIsR0FBRyxpQkFBaUIsQ0FBQztRQUUzQyxJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxZQUFZLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQyxDQUFDO1FBQzlGLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFlBQVksRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDaEcsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUMsb0JBQW9CLENBQUMsWUFBWSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztJQUNwRyxDQUFDO0lBRUQsSUFBVyxZQUFZO1FBQ3JCLElBQUksSUFBSSxDQUFDLElBQUksS0FBSyxTQUFTLEVBQUU7WUFDM0IsbUZBQW1GO1lBQ25GLGtEQUFrRDtZQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLDZKQUE2SixDQUFDLENBQUM7U0FDaEw7UUFDRCxPQUFPLElBQUksQ0FBQyxJQUFJLENBQUM7SUFDbkIsQ0FBQztJQUVPLG9CQUFvQixDQUFDLFlBQW9DLEVBQUUsZUFBeUM7UUFDMUcsT0FBTyxPQUFPLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEtBQUssZUFBZSxDQUFDLEVBQ3JGLFdBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLG9CQUFvQixDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUM7SUFDM0QsQ0FBQztJQUVPLG9CQUFvQixDQUFDLFdBQWlDO1FBQzVELE1BQU0sR0FBRyxHQUFHLElBQUksS0FBSyxFQUFXLENBQUM7UUFDakMsS0FBSyxJQUFJLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxHQUFHLFdBQVcsQ0FBQyxPQUFPLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFO1lBQ25ELE1BQU0sU0FBUyxHQUFHLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDekMsR0FBRyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsb0JBQW9CLENBQUMsSUFBSSxFQUFFLEdBQUcsV0FBVyxDQUFDLElBQUksU0FBUyxDQUFDLEdBQUcsQ0FBQyxFQUFFLEVBQUU7Z0JBQzlFLGdCQUFnQixFQUFFLFNBQVMsQ0FBQyxnQkFBZ0I7Z0JBQzVDLFFBQVEsRUFBRSxTQUFTLENBQUMsUUFBUTtnQkFDNUIsWUFBWSxFQUFFLFNBQVMsQ0FBQyxZQUFZO2FBQ3JDLENBQUMsQ0FBQyxDQUFDO1NBQ0w7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNiLENBQUM7Q0FDRjtBQUVELFNBQVMsT0FBTyxDQUFPLEVBQU8sRUFBRSxFQUFpQjtJQUMvQyxNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssRUFBSyxDQUFDO0lBQzNCLEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxFQUFFO1FBQ2xCLEdBQUcsQ0FBQyxJQUFJLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztLQUNwQjtJQUNELE9BQU8sR0FBRyxDQUFDO0FBQ2IsQ0FBQztBQUVELE1BQU0sbUJBQW1CO0lBR3ZCO1FBRmlCLGdCQUFXLEdBQUcsSUFBSSxLQUFLLEVBQWUsQ0FBQztRQUd0RCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsc0JBQWUsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFO1lBQzlCLElBQUksZUFBZTtnQkFDakIsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLEVBQWMsQ0FBQztnQkFDcEMsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLENBQUMsV0FBVyxFQUFFO29CQUNsQyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsc0JBQWUsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUMsZUFBZSxDQUFDLENBQUM7aUJBQ3ZEO2dCQUNELE9BQU8sR0FBRyxDQUFDO1lBQ2IsQ0FBQztTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNJLEdBQUcsQ0FBQyxHQUFnQjtRQUN6QixJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUM3QixDQUFDO0NBQ0Y7QUFFRDs7R0FFRztBQUNILFNBQVMsR0FBRyxDQUFJLENBQUksRUFBRSxFQUFrQjtJQUN0QyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDTixPQUFPLENBQUMsQ0FBQztBQUNYLENBQUM7QUFFRCxNQUFNLGNBQWUsU0FBUSxlQUFRO0lBT25DLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBdUI7UUFDL0QsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQVBILG9DQUErQixHQUFnQixJQUFJLHlCQUFrQixFQUFFLENBQUM7UUFTdEYsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDdkIsTUFBTSxHQUFHLEdBQUcsWUFBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDO2dCQUM1QyxDQUFDLENBQUMsT0FBTyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksSUFBSSxFQUFFLEdBQUc7Z0JBQ2pDLENBQUMsQ0FBQyxJQUFJLEtBQUssQ0FBQyxRQUFRLEdBQUcsQ0FBQztZQUMxQixtQ0FBbUM7WUFDbkMsa0JBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUFDLDhDQUE4QyxHQUFHLHFJQUFxSSxDQUFDLENBQUM7U0FDek47UUFFRCxJQUFJLENBQUMsY0FBYyxHQUFHLEtBQUssQ0FBQyxhQUFhLENBQUM7UUFDMUMsSUFBSSxDQUFDLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztRQUNoRCxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLFVBQVUsR0FBRztZQUNoQiw2SEFBNkg7WUFDN0gsWUFBWSxFQUFFLEtBQUssQ0FBQyxZQUFhO1NBQ2xDLENBQUM7SUFDSixDQUFDO0lBRUQsSUFBVyxnQkFBZ0I7UUFDekIsSUFBSSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsRUFBRTtZQUMzQixtQ0FBbUM7WUFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3SkFBd0osQ0FBQyxDQUFDO1NBQzNLO1FBQ0QsT0FBTyxJQUFJLENBQUMsaUJBQWlCLENBQUM7SUFDaEMsQ0FBQztJQUVELElBQVcsYUFBYTtRQUN0QixJQUFJLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRTtZQUN4Qiw0Q0FBNEM7WUFDNUMsTUFBTSxJQUFJLEtBQUssQ0FBQyx1SkFBdUosQ0FBQyxDQUFDO1NBQzFLO1FBQ0QsT0FBTyxJQUFJLENBQUMsY0FBYyxDQUFDO0lBQzdCLENBQUM7SUFFTSxtQkFBbUIsQ0FBQyxFQUFVLEVBQUUsVUFBdUI7UUFDNUQsTUFBTSxLQUFLLEdBQUcsZ0JBQVMsQ0FBQyxXQUFXLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDO1FBQ3BFLE1BQU0sS0FBSyxHQUFHLGdCQUFTLENBQUMsV0FBVyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQztRQUNwRSxJQUFJLHlDQUEyQixDQUFDLEtBQUssRUFBRSxFQUFFLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUU7WUFDL0QsVUFBVTtZQUNWLE1BQU0sRUFBRSxJQUFJO1NBQ2IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztDQUNGO0FBRUQ7Ozs7Ozs7Ozs7R0FVRztBQUNILFNBQVMsd0JBQXdCLENBQUMsY0FBa0MsRUFBRSxZQUFtQyxFQUFFLE9BQWU7SUFDeEgsTUFBTSxpQkFBaUIsR0FBRyxZQUFZLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFVBQVUsS0FBSyxVQUFVLENBQUMsT0FBTyxJQUFJLENBQUMsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3JHLE1BQU0sZ0JBQWdCLEdBQUcsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxVQUFVLEtBQUssVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBRXBGLE1BQU0sS0FBSyxHQUFHLGNBQWMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsY0FBYyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBRW5ILElBQUksS0FBSyxLQUFLLENBQUMsSUFBSSxpQkFBaUIsRUFBRTtRQUNwQyxtQ0FBbUM7UUFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyx1S0FBdUssQ0FBQyxDQUFDO0tBQzFMO0lBRUQsSUFBSSxLQUFLLEdBQUcsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7UUFDbEMsbUNBQW1DO1FBQ25DLE1BQU0sSUFBSSxLQUFLLENBQUMsdUlBQXVJLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0tBQ3pMO0lBRUQsT0FBTyxLQUFLLENBQUM7QUFDZixDQUFDO0FBRUQ7Ozs7R0FJRztBQUNILE1BQU0sZUFBZSxHQUE2QjtJQUNoRCxpQkFBaUIsRUFBRSxFQUFFO0lBQ3JCLFlBQVksRUFBRSxXQUFXO0lBQ3pCLGlCQUFpQixFQUFFLFNBQVM7SUFDNUIsbUJBQW1CLEVBQUUsU0FBUztJQUM5QiwyQkFBMkIsRUFBRSxTQUFTO0lBQ3RDLGdCQUFnQixFQUFFLFNBQVM7SUFDM0Isa0JBQWtCLEVBQUUsU0FBUztJQUM3QiwwQkFBMEIsRUFBRSxTQUFTO0lBQ3JDLGVBQWUsRUFBRSxTQUFTO0lBQzFCLGlCQUFpQixFQUFFLFNBQVM7SUFDNUIseUJBQXlCLEVBQUUsU0FBUztJQUNwQyxZQUFZLEVBQUU7UUFDWjtZQUNFLElBQUksRUFBRSxRQUFRO1lBQ2QsSUFBSSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxNQUFNO1lBQ3JDLE9BQU8sRUFBRTtnQkFDUDtvQkFDRSxnQkFBZ0IsRUFBRSxTQUFTO29CQUMzQixRQUFRLEVBQUUsU0FBUztvQkFDbkIsWUFBWSxFQUFFLFlBQVk7b0JBQzFCLElBQUksRUFBRSxXQUFXO2lCQUNsQjtnQkFDRDtvQkFDRSxnQkFBZ0IsRUFBRSxTQUFTO29CQUMzQixRQUFRLEVBQUUsU0FBUztvQkFDbkIsWUFBWSxFQUFFLFlBQVk7b0JBQzFCLElBQUksRUFBRSxXQUFXO2lCQUNsQjthQUNGO1NBQ0Y7UUFDRDtZQUNFLElBQUksRUFBRSxTQUFTO1lBQ2YsSUFBSSxFQUFFLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPO1lBQ3RDLE9BQU8sRUFBRTtnQkFDUDtvQkFDRSxnQkFBZ0IsRUFBRSxTQUFTO29CQUMzQixRQUFRLEVBQUUsU0FBUztvQkFDbkIsWUFBWSxFQUFFLFlBQVk7b0JBQzFCLElBQUksRUFBRSxXQUFXO2lCQUNsQjtnQkFDRDtvQkFDRSxnQkFBZ0IsRUFBRSxTQUFTO29CQUMzQixRQUFRLEVBQUUsU0FBUztvQkFDbkIsWUFBWSxFQUFFLFlBQVk7b0JBQzFCLElBQUksRUFBRSxXQUFXO2lCQUNsQjthQUNGO1NBQ0Y7S0FDRjtJQUNELEtBQUssRUFBRSxXQUFXO0NBQ25CLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjeHNjaGVtYSBmcm9tICdAYXdzLWNkay9jbG91ZC1hc3NlbWJseS1zY2hlbWEnO1xuaW1wb3J0IHtcbiAgQW5ub3RhdGlvbnMsIENvbmNyZXRlRGVwZW5kYWJsZSwgQ29uc3RydWN0LCBDb250ZXh0UHJvdmlkZXIsIERlcGVuZGFibGVUcmFpdCwgSUNvbnN0cnVjdCxcbiAgSURlcGVuZGFibGUsIElSZXNvdXJjZSwgTGF6eSwgUmVzb3VyY2UsIFN0YWNrLCBUb2tlbiwgVGFncyxcbn0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBjeGFwaSBmcm9tICdAYXdzLWNkay9jeC1hcGknO1xuaW1wb3J0IHtcbiAgQ2ZuRUlQLCBDZm5JbnRlcm5ldEdhdGV3YXksIENmbk5hdEdhdGV3YXksIENmblJvdXRlLCBDZm5Sb3V0ZVRhYmxlLCBDZm5TdWJuZXQsXG4gIENmblN1Ym5ldFJvdXRlVGFibGVBc3NvY2lhdGlvbiwgQ2ZuVlBDLCBDZm5WUENHYXRld2F5QXR0YWNobWVudCwgQ2ZuVlBOR2F0ZXdheVJvdXRlUHJvcGFnYXRpb24sXG59IGZyb20gJy4vZWMyLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBOYXRQcm92aWRlciB9IGZyb20gJy4vbmF0JztcbmltcG9ydCB7IElOZXR3b3JrQWNsLCBOZXR3b3JrQWNsLCBTdWJuZXROZXR3b3JrQWNsQXNzb2NpYXRpb24gfSBmcm9tICcuL25ldHdvcmstYWNsJztcbmltcG9ydCB7IE5ldHdvcmtCdWlsZGVyIH0gZnJvbSAnLi9uZXR3b3JrLXV0aWwnO1xuaW1wb3J0IHsgU3VibmV0RmlsdGVyIH0gZnJvbSAnLi9zdWJuZXQnO1xuaW1wb3J0IHsgYWxsUm91dGVUYWJsZUlkcywgZGVmYXVsdFN1Ym5ldE5hbWUsIGZsYXR0ZW4sIEltcG9ydFN1Ym5ldEdyb3VwLCBzdWJuZXRHcm91cE5hbWVGcm9tQ29uc3RydWN0SWQsIHN1Ym5ldElkIH0gZnJvbSAnLi91dGlsJztcbmltcG9ydCB7IEdhdGV3YXlWcGNFbmRwb2ludCwgR2F0ZXdheVZwY0VuZHBvaW50QXdzU2VydmljZSwgR2F0ZXdheVZwY0VuZHBvaW50T3B0aW9ucywgSW50ZXJmYWNlVnBjRW5kcG9pbnQsIEludGVyZmFjZVZwY0VuZHBvaW50T3B0aW9ucyB9IGZyb20gJy4vdnBjLWVuZHBvaW50JztcbmltcG9ydCB7IEZsb3dMb2csIEZsb3dMb2dPcHRpb25zLCBGbG93TG9nUmVzb3VyY2VUeXBlIH0gZnJvbSAnLi92cGMtZmxvdy1sb2dzJztcbmltcG9ydCB7IFZwY0xvb2t1cE9wdGlvbnMgfSBmcm9tICcuL3ZwYy1sb29rdXAnO1xuaW1wb3J0IHsgRW5hYmxlVnBuR2F0ZXdheU9wdGlvbnMsIFZwbkNvbm5lY3Rpb24sIFZwbkNvbm5lY3Rpb25PcHRpb25zLCBWcG5Db25uZWN0aW9uVHlwZSwgVnBuR2F0ZXdheSB9IGZyb20gJy4vdnBuJztcblxuY29uc3QgVlBDX1NVQk5FVF9TWU1CT0wgPSBTeW1ib2wuZm9yKCdAYXdzLWNkay9hd3MtZWMyLlZwY1N1Ym5ldCcpO1xuXG5leHBvcnQgaW50ZXJmYWNlIElTdWJuZXQgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAvKipcbiAgICogVGhlIEF2YWlsYWJpbGl0eSBab25lIHRoZSBzdWJuZXQgaXMgbG9jYXRlZCBpblxuICAgKi9cbiAgcmVhZG9ubHkgYXZhaWxhYmlsaXR5Wm9uZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgc3VibmV0SWQgZm9yIHRoaXMgcGFydGljdWxhciBzdWJuZXRcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0SWQ6IHN0cmluZztcblxuICAvKipcbiAgICogRGVwZW5kYWJsZSB0aGF0IGNhbiBiZSBkZXBlbmRlZCB1cG9uIHRvIGZvcmNlIGludGVybmV0IGNvbm5lY3Rpdml0eSBlc3RhYmxpc2hlZCBvbiB0aGUgVlBDXG4gICAqL1xuICByZWFkb25seSBpbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkOiBJRGVwZW5kYWJsZTtcblxuICAvKipcbiAgICogVGhlIElQdjQgQ0lEUiBibG9jayBmb3IgdGhpcyBzdWJuZXRcbiAgICovXG4gIHJlYWRvbmx5IGlwdjRDaWRyQmxvY2s6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHJvdXRlIHRhYmxlIGZvciB0aGlzIHN1Ym5ldFxuICAgKi9cbiAgcmVhZG9ubHkgcm91dGVUYWJsZTogSVJvdXRlVGFibGU7XG5cbiAgLyoqXG4gICAqIEFzc29jaWF0ZSBhIE5ldHdvcmsgQUNMIHdpdGggdGhpcyBzdWJuZXRcbiAgICpcbiAgICogQHBhcmFtIGFjbCBUaGUgTmV0d29yayBBQ0wgdG8gYXNzb2NpYXRlXG4gICAqL1xuICBhc3NvY2lhdGVOZXR3b3JrQWNsKGlkOiBzdHJpbmcsIGFjbDogSU5ldHdvcmtBY2wpOiB2b2lkO1xufVxuXG4vKipcbiAqIEFuIGFic3RyYWN0IHJvdXRlIHRhYmxlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSVJvdXRlVGFibGUge1xuICAvKipcbiAgICogUm91dGUgdGFibGUgSURcbiAgICovXG4gIHJlYWRvbmx5IHJvdXRlVGFibGVJZDogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIElWcGMgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAvKipcbiAgICogSWRlbnRpZmllciBmb3IgdGhpcyBWUENcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdnBjSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogQ0lEUiByYW5nZSBmb3IgdGhpcyBWUENcbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgdnBjQ2lkckJsb2NrOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgcHVibGljIHN1Ym5ldHMgaW4gdGhpcyBWUENcbiAgICovXG4gIHJlYWRvbmx5IHB1YmxpY1N1Ym5ldHM6IElTdWJuZXRbXTtcblxuICAvKipcbiAgICogTGlzdCBvZiBwcml2YXRlIHN1Ym5ldHMgaW4gdGhpcyBWUENcbiAgICovXG4gIHJlYWRvbmx5IHByaXZhdGVTdWJuZXRzOiBJU3VibmV0W107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgaXNvbGF0ZWQgc3VibmV0cyBpbiB0aGlzIFZQQ1xuICAgKi9cbiAgcmVhZG9ubHkgaXNvbGF0ZWRTdWJuZXRzOiBJU3VibmV0W107XG5cbiAgLyoqXG4gICAqIEFacyBmb3IgdGhpcyBWUENcbiAgICovXG4gIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmVzOiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogSWRlbnRpZmllciBmb3IgdGhlIFZQTiBnYXRld2F5XG4gICAqL1xuICByZWFkb25seSB2cG5HYXRld2F5SWQ/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIERlcGVuZGFibGUgdGhhdCBjYW4gYmUgZGVwZW5kZWQgdXBvbiB0byBmb3JjZSBpbnRlcm5ldCBjb25uZWN0aXZpdHkgZXN0YWJsaXNoZWQgb24gdGhlIFZQQ1xuICAgKi9cbiAgcmVhZG9ubHkgaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZDogSURlcGVuZGFibGU7XG5cbiAgLyoqXG4gICAqIFJldHVybiBpbmZvcm1hdGlvbiBvbiB0aGUgc3VibmV0cyBhcHByb3ByaWF0ZSBmb3IgdGhlIGdpdmVuIHNlbGVjdGlvbiBzdHJhdGVneVxuICAgKlxuICAgKiBSZXF1aXJlcyB0aGF0IGF0IGxlYXN0IG9uZSBzdWJuZXQgaXMgbWF0Y2hlZCwgdGhyb3dzIGEgZGVzY3JpcHRpdmVcbiAgICogZXJyb3IgbWVzc2FnZSBvdGhlcndpc2UuXG4gICAqL1xuICBzZWxlY3RTdWJuZXRzKHNlbGVjdGlvbj86IFN1Ym5ldFNlbGVjdGlvbik6IFNlbGVjdGVkU3VibmV0cztcblxuICAvKipcbiAgICogQWRkcyBhIFZQTiBHYXRld2F5IHRvIHRoaXMgVlBDXG4gICAqL1xuICBlbmFibGVWcG5HYXRld2F5KG9wdGlvbnM6IEVuYWJsZVZwbkdhdGV3YXlPcHRpb25zKTogdm9pZDtcblxuICAvKipcbiAgICogQWRkcyBhIG5ldyBWUE4gY29ubmVjdGlvbiB0byB0aGlzIFZQQ1xuICAgKi9cbiAgYWRkVnBuQ29ubmVjdGlvbihpZDogc3RyaW5nLCBvcHRpb25zOiBWcG5Db25uZWN0aW9uT3B0aW9ucyk6IFZwbkNvbm5lY3Rpb247XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgZ2F0ZXdheSBlbmRwb2ludCB0byB0aGlzIFZQQ1xuICAgKi9cbiAgYWRkR2F0ZXdheUVuZHBvaW50KGlkOiBzdHJpbmcsIG9wdGlvbnM6IEdhdGV3YXlWcGNFbmRwb2ludE9wdGlvbnMpOiBHYXRld2F5VnBjRW5kcG9pbnRcblxuICAvKipcbiAgICogQWRkcyBhIG5ldyBpbnRlcmZhY2UgZW5kcG9pbnQgdG8gdGhpcyBWUENcbiAgICovXG4gIGFkZEludGVyZmFjZUVuZHBvaW50KGlkOiBzdHJpbmcsIG9wdGlvbnM6IEludGVyZmFjZVZwY0VuZHBvaW50T3B0aW9ucyk6IEludGVyZmFjZVZwY0VuZHBvaW50XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgRmxvdyBMb2cgdG8gdGhpcyBWUENcbiAgICovXG4gIGFkZEZsb3dMb2coaWQ6IHN0cmluZywgb3B0aW9ucz86IEZsb3dMb2dPcHRpb25zKTogRmxvd0xvZ1xufVxuXG4vKipcbiAqIFRoZSB0eXBlIG9mIFN1Ym5ldFxuICovXG5leHBvcnQgZW51bSBTdWJuZXRUeXBlIHtcbiAgLyoqXG4gICAqIElzb2xhdGVkIFN1Ym5ldHMgZG8gbm90IHJvdXRlIHRyYWZmaWMgdG8gdGhlIEludGVybmV0IChpbiB0aGlzIFZQQykuXG4gICAqXG4gICAqIFRoaXMgY2FuIGJlIGdvb2QgZm9yIHN1Ym5ldHMgd2l0aCBSRFMgb3IgRWxhc3RpY2FjaGUgaW5zdGFuY2VzLFxuICAgKiBvciB3aGljaCByb3V0ZSBJbnRlcm5ldCB0cmFmZmljIHRocm91Z2ggYSBwZWVyIFZQQy5cbiAgICovXG4gIElTT0xBVEVEID0gJ0lzb2xhdGVkJyxcblxuICAvKipcbiAgICogU3VibmV0IHRoYXQgcm91dGVzIHRvIHRoZSBpbnRlcm5ldCwgYnV0IG5vdCB2aWNlIHZlcnNhLlxuICAgKlxuICAgKiBJbnN0YW5jZXMgaW4gYSBwcml2YXRlIHN1Ym5ldCBjYW4gY29ubmVjdCB0byB0aGUgSW50ZXJuZXQsIGJ1dCB3aWxsIG5vdFxuICAgKiBhbGxvdyBjb25uZWN0aW9ucyB0byBiZSBpbml0aWF0ZWQgZnJvbSB0aGUgSW50ZXJuZXQuIEludGVybmV0IHRyYWZmaWMgd2lsbFxuICAgKiBiZSByb3V0ZWQgdmlhIGEgTkFUIEdhdGV3YXkuXG4gICAqXG4gICAqIE5vcm1hbGx5IGEgUHJpdmF0ZSBzdWJuZXQgd2lsbCB1c2UgYSBOQVQgZ2F0ZXdheSBpbiB0aGUgc2FtZSBBWiwgYnV0XG4gICAqIGlmIGBuYXRHYXRld2F5c2AgaXMgdXNlZCB0byByZWR1Y2UgdGhlIG51bWJlciBvZiBOQVQgZ2F0ZXdheXMsIGEgTkFUXG4gICAqIGdhdGV3YXkgZnJvbSBhbm90aGVyIEFaIHdpbGwgYmUgdXNlZCBpbnN0ZWFkLlxuICAgKi9cbiAgUFJJVkFURSA9ICdQcml2YXRlJyxcblxuICAvKipcbiAgICogU3VibmV0IGNvbm5lY3RlZCB0byB0aGUgSW50ZXJuZXRcbiAgICpcbiAgICogSW5zdGFuY2VzIGluIGEgUHVibGljIHN1Ym5ldCBjYW4gY29ubmVjdCB0byB0aGUgSW50ZXJuZXQgYW5kIGNhbiBiZVxuICAgKiBjb25uZWN0ZWQgdG8gZnJvbSB0aGUgSW50ZXJuZXQgYXMgbG9uZyBhcyB0aGV5IGFyZSBsYXVuY2hlZCB3aXRoIHB1YmxpY1xuICAgKiBJUHMgKGNvbnRyb2xsZWQgb24gdGhlIEF1dG9TY2FsaW5nR3JvdXAgb3Igb3RoZXIgY29uc3RydWN0cyB0aGF0IGxhdW5jaFxuICAgKiBpbnN0YW5jZXMpLlxuICAgKlxuICAgKiBQdWJsaWMgc3VibmV0cyByb3V0ZSBvdXRib3VuZCB0cmFmZmljIHZpYSBhbiBJbnRlcm5ldCBHYXRld2F5LlxuICAgKi9cbiAgUFVCTElDID0gJ1B1YmxpYydcbn1cblxuLyoqXG4gKiBDdXN0b21pemUgc3VibmV0cyB0aGF0IGFyZSBzZWxlY3RlZCBmb3IgcGxhY2VtZW50IG9mIEVOSXNcbiAqXG4gKiBDb25zdHJ1Y3RzIHRoYXQgYWxsb3cgY3VzdG9taXphdGlvbiBvZiBWUEMgcGxhY2VtZW50IHVzZSBwYXJhbWV0ZXJzIG9mIHRoaXNcbiAqIHR5cGUgdG8gcHJvdmlkZSBwbGFjZW1lbnQgc2V0dGluZ3MuXG4gKlxuICogQnkgZGVmYXVsdCwgdGhlIGluc3RhbmNlcyBhcmUgcGxhY2VkIGluIHRoZSBwcml2YXRlIHN1Ym5ldHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU3VibmV0U2VsZWN0aW9uIHtcbiAgLyoqXG4gICAqIFNlbGVjdCBhbGwgc3VibmV0cyBvZiB0aGUgZ2l2ZW4gdHlwZVxuICAgKlxuICAgKiBBdCBtb3N0IG9uZSBvZiBgc3VibmV0VHlwZWAgYW5kIGBzdWJuZXRHcm91cE5hbWVgIGNhbiBiZSBzdXBwbGllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgU3VibmV0VHlwZS5QUklWQVRFIChvciBJU09MQVRFRCBvciBQVUJMSUMgaWYgdGhlcmUgYXJlIG5vIFBSSVZBVEUgc3VibmV0cylcbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldFR5cGU/OiBTdWJuZXRUeXBlO1xuXG4gIC8qKlxuICAgKiBTZWxlY3Qgc3VibmV0cyBvbmx5IGluIHRoZSBnaXZlbiBBWnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG5vIGZpbHRlcmluZyBvbiBBWnMgaXMgZG9uZVxuICAgKi9cbiAgcmVhZG9ubHkgYXZhaWxhYmlsaXR5Wm9uZXM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogU2VsZWN0IHRoZSBzdWJuZXQgZ3JvdXAgd2l0aCB0aGUgZ2l2ZW4gbmFtZVxuICAgKlxuICAgKiBTZWxlY3QgdGhlIHN1Ym5ldCBncm91cCB3aXRoIHRoZSBnaXZlbiBuYW1lLiBUaGlzIG9ubHkgbmVlZHNcbiAgICogdG8gYmUgdXNlZCBpZiB5b3UgaGF2ZSBtdWx0aXBsZSBzdWJuZXQgZ3JvdXBzIG9mIHRoZSBzYW1lIHR5cGVcbiAgICogYW5kIHlvdSBuZWVkIHRvIGRpc3Rpbmd1aXNoIGJldHdlZW4gdGhlbS4gT3RoZXJ3aXNlLCBwcmVmZXJcbiAgICogYHN1Ym5ldFR5cGVgLlxuICAgKlxuICAgKiBUaGlzIGZpZWxkIGRvZXMgbm90IHNlbGVjdCBpbmRpdmlkdWFsIHN1Ym5ldHMsIGl0IHNlbGVjdHMgYWxsIHN1Ym5ldHMgdGhhdFxuICAgKiBzaGFyZSB0aGUgZ2l2ZW4gc3VibmV0IGdyb3VwIG5hbWUuIFRoaXMgaXMgdGhlIG5hbWUgc3VwcGxpZWQgaW5cbiAgICogYHN1Ym5ldENvbmZpZ3VyYXRpb25gLlxuICAgKlxuICAgKiBBdCBtb3N0IG9uZSBvZiBgc3VibmV0VHlwZWAgYW5kIGBzdWJuZXRHcm91cE5hbWVgIGNhbiBiZSBzdXBwbGllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBTZWxlY3Rpb24gYnkgdHlwZSBpbnN0ZWFkIG9mIGJ5IG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldEdyb3VwTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogQWxpYXMgZm9yIGBzdWJuZXRHcm91cE5hbWVgXG4gICAqXG4gICAqIFNlbGVjdCB0aGUgc3VibmV0IGdyb3VwIHdpdGggdGhlIGdpdmVuIG5hbWUuIFRoaXMgb25seSBuZWVkc1xuICAgKiB0byBiZSB1c2VkIGlmIHlvdSBoYXZlIG11bHRpcGxlIHN1Ym5ldCBncm91cHMgb2YgdGhlIHNhbWUgdHlwZVxuICAgKiBhbmQgeW91IG5lZWQgdG8gZGlzdGluZ3Vpc2ggYmV0d2VlbiB0aGVtLlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCBVc2UgYHN1Ym5ldEdyb3VwTmFtZWAgaW5zdGVhZFxuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0TmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogSWYgdHJ1ZSwgcmV0dXJuIGF0IG1vc3Qgb25lIHN1Ym5ldCBwZXIgQVpcbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IG9uZVBlckF6PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogTGlzdCBvZiBwcm92aWRlZCBzdWJuZXQgZmlsdGVycy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBub25lXG4gICAqL1xuICByZWFkb25seSBzdWJuZXRGaWx0ZXJzPzogU3VibmV0RmlsdGVyW107XG5cbiAgLyoqXG4gICAqIEV4cGxpY2l0bHkgc2VsZWN0IGluZGl2aWR1YWwgc3VibmV0c1xuICAgKlxuICAgKiBVc2UgdGhpcyBpZiB5b3UgZG9uJ3Qgd2FudCB0byBhdXRvbWF0aWNhbGx5IHVzZSBhbGwgc3VibmV0cyBpblxuICAgKiBhIGdyb3VwLCBidXQgaGF2ZSBhIG5lZWQgdG8gY29udHJvbCBzZWxlY3Rpb24gZG93biB0b1xuICAgKiBpbmRpdmlkdWFsIHN1Ym5ldHMuXG4gICAqXG4gICAqIENhbm5vdCBiZSBzcGVjaWZpZWQgdG9nZXRoZXIgd2l0aCBgc3VibmV0VHlwZWAgb3IgYHN1Ym5ldEdyb3VwTmFtZWAuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gVXNlIGFsbCBzdWJuZXRzIGluIGEgc2VsZWN0ZWQgZ3JvdXAgKGFsbCBwcml2YXRlIHN1Ym5ldHMgYnkgZGVmYXVsdClcbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldHM/OiBJU3VibmV0W11cbn1cblxuLyoqXG4gKiBSZXN1bHQgb2Ygc2VsZWN0aW5nIGEgc3Vic2V0IG9mIHN1Ym5ldHMgZnJvbSBhIFZQQ1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFNlbGVjdGVkU3VibmV0cyB7XG4gIC8qKlxuICAgKiBUaGUgc3VibmV0IElEc1xuICAgKi9cbiAgcmVhZG9ubHkgc3VibmV0SWRzOiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogVGhlIHJlc3BlY3RpdmUgQVpzIG9mIGVhY2ggc3VibmV0XG4gICAqL1xuICByZWFkb25seSBhdmFpbGFiaWxpdHlab25lczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIERlcGVuZGVuY3kgcmVwcmVzZW50aW5nIGludGVybmV0IGNvbm5lY3Rpdml0eSBmb3IgdGhlc2Ugc3VibmV0c1xuICAgKi9cbiAgcmVhZG9ubHkgaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZDogSURlcGVuZGFibGU7XG5cbiAgLyoqXG4gICAqIFNlbGVjdGVkIHN1Ym5ldCBvYmplY3RzXG4gICAqL1xuICByZWFkb25seSBzdWJuZXRzOiBJU3VibmV0W107XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgYW55IG9mIHRoZSBnaXZlbiBzdWJuZXRzIGFyZSBmcm9tIHRoZSBWUEMncyBwdWJsaWMgc3VibmV0cy5cbiAgICovXG4gIHJlYWRvbmx5IGhhc1B1YmxpYzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBBIG5ldyBvciBpbXBvcnRlZCBWUENcbiAqL1xuYWJzdHJhY3QgY2xhc3MgVnBjQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVZwYyB7XG5cbiAgLyoqXG4gICAqIElkZW50aWZpZXIgZm9yIHRoaXMgVlBDXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgdnBjSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogQ0lEUiByYW5nZSBmb3IgdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSB2cGNDaWRyQmxvY2s6IHN0cmluZztcblxuICAvKipcbiAgICogTGlzdCBvZiBwdWJsaWMgc3VibmV0cyBpbiB0aGlzIFZQQ1xuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHB1YmxpY1N1Ym5ldHM6IElTdWJuZXRbXTtcblxuICAvKipcbiAgICogTGlzdCBvZiBwcml2YXRlIHN1Ym5ldHMgaW4gdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBwcml2YXRlU3VibmV0czogSVN1Ym5ldFtdO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIGlzb2xhdGVkIHN1Ym5ldHMgaW4gdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBpc29sYXRlZFN1Ym5ldHM6IElTdWJuZXRbXTtcblxuICAvKipcbiAgICogQVpzIGZvciB0aGlzIFZQQ1xuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmVzOiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogRGVwZW5kZW5jaWVzIGZvciBpbnRlcm5ldCBjb25uZWN0aXZpdHlcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBpbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkOiBJRGVwZW5kYWJsZTtcblxuICAvKipcbiAgICogRGVwZW5kZW5jaWVzIGZvciBOQVQgY29ubmVjdGl2aXR5XG4gICAqXG4gICAqIEBkZXByZWNhdGVkIC0gVGhpcyB2YWx1ZSBpcyBubyBsb25nZXIgdXNlZC5cbiAgICovXG4gIHByb3RlY3RlZCByZWFkb25seSBuYXREZXBlbmRlbmNpZXMgPSBuZXcgQXJyYXk8SUNvbnN0cnVjdD4oKTtcblxuICAvKipcbiAgICogSWYgdGhpcyBpcyBzZXQgdG8gdHJ1ZSwgZG9uJ3QgZXJyb3Igb3V0IG9uIHRyeWluZyB0byBzZWxlY3Qgc3VibmV0c1xuICAgKi9cbiAgcHJvdGVjdGVkIGluY29tcGxldGVTdWJuZXREZWZpbml0aW9uOiBib29sZWFuID0gZmFsc2U7XG5cbiAgLyoqXG4gICAqIE11dGFibGUgcHJpdmF0ZSBmaWVsZCBmb3IgdGhlIHZwbkdhdGV3YXlJZFxuICAgKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHByb3RlY3RlZCBfdnBuR2F0ZXdheUlkPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSZXR1cm5zIElEcyBvZiBzZWxlY3RlZCBzdWJuZXRzXG4gICAqL1xuICBwdWJsaWMgc2VsZWN0U3VibmV0cyhzZWxlY3Rpb246IFN1Ym5ldFNlbGVjdGlvbiA9IHt9KTogU2VsZWN0ZWRTdWJuZXRzIHtcbiAgICBjb25zdCBzdWJuZXRzID0gdGhpcy5zZWxlY3RTdWJuZXRPYmplY3RzKHNlbGVjdGlvbik7XG4gICAgY29uc3QgcHVicyA9IG5ldyBTZXQodGhpcy5wdWJsaWNTdWJuZXRzKTtcblxuICAgIHJldHVybiB7XG4gICAgICBzdWJuZXRJZHM6IHN1Ym5ldHMubWFwKHMgPT4gcy5zdWJuZXRJZCksXG4gICAgICBnZXQgYXZhaWxhYmlsaXR5Wm9uZXMoKTogc3RyaW5nW10geyByZXR1cm4gc3VibmV0cy5tYXAocyA9PiBzLmF2YWlsYWJpbGl0eVpvbmUpOyB9LFxuICAgICAgaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZDogdGFwKG5ldyBDb21wb3NpdGVEZXBlbmRhYmxlKCksIGQgPT4gc3VibmV0cy5mb3JFYWNoKHMgPT4gZC5hZGQocy5pbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkKSkpLFxuICAgICAgc3VibmV0cyxcbiAgICAgIGhhc1B1YmxpYzogc3VibmV0cy5zb21lKHMgPT4gcHVicy5oYXMocykpLFxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIFZQTiBHYXRld2F5IHRvIHRoaXMgVlBDXG4gICAqL1xuICBwdWJsaWMgZW5hYmxlVnBuR2F0ZXdheShvcHRpb25zOiBFbmFibGVWcG5HYXRld2F5T3B0aW9ucyk6IHZvaWQge1xuICAgIGlmICh0aGlzLnZwbkdhdGV3YXlJZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgVlBOIEdhdGV3YXkgaGFzIGFscmVhZHkgYmVlbiBlbmFibGVkLicpO1xuICAgIH1cblxuICAgIGNvbnN0IHZwbkdhdGV3YXkgPSBuZXcgVnBuR2F0ZXdheSh0aGlzLCAnVnBuR2F0ZXdheScsIHtcbiAgICAgIGFtYXpvblNpZGVBc246IG9wdGlvbnMuYW1hem9uU2lkZUFzbixcbiAgICAgIHR5cGU6IFZwbkNvbm5lY3Rpb25UeXBlLklQU0VDXzEsXG4gICAgfSk7XG5cbiAgICB0aGlzLl92cG5HYXRld2F5SWQgPSB2cG5HYXRld2F5LmdhdGV3YXlJZDtcblxuICAgIGNvbnN0IGF0dGFjaG1lbnQgPSBuZXcgQ2ZuVlBDR2F0ZXdheUF0dGFjaG1lbnQodGhpcywgJ1ZQQ1ZQTkdXJywge1xuICAgICAgdnBjSWQ6IHRoaXMudnBjSWQsXG4gICAgICB2cG5HYXRld2F5SWQ6IHRoaXMuX3ZwbkdhdGV3YXlJZCxcbiAgICB9KTtcblxuICAgIC8vIFByb3BhZ2F0ZSByb3V0ZXMgb24gcm91dGUgdGFibGVzIGFzc29jaWF0ZWQgd2l0aCB0aGUgcmlnaHQgc3VibmV0c1xuICAgIGNvbnN0IHZwblJvdXRlUHJvcGFnYXRpb24gPSBvcHRpb25zLnZwblJvdXRlUHJvcGFnYXRpb24gPz8gW3t9XTtcbiAgICBjb25zdCByb3V0ZVRhYmxlSWRzID0gYWxsUm91dGVUYWJsZUlkcyhmbGF0dGVuKHZwblJvdXRlUHJvcGFnYXRpb24ubWFwKHMgPT4gdGhpcy5zZWxlY3RTdWJuZXRzKHMpLnN1Ym5ldHMpKSk7XG5cbiAgICBpZiAocm91dGVUYWJsZUlkcy5sZW5ndGggPT09IDApIHtcbiAgICAgIEFubm90YXRpb25zLm9mKHRoaXMpLmFkZEVycm9yKGBlbmFibGVWcG5HYXRld2F5OiBubyBzdWJuZXRzIG1hdGNoaW5nIHNlbGVjdGlvbjogJyR7SlNPTi5zdHJpbmdpZnkodnBuUm91dGVQcm9wYWdhdGlvbil9Jy4gU2VsZWN0IG90aGVyIHN1Ym5ldHMgdG8gYWRkIHJvdXRlcyB0by5gKTtcbiAgICB9XG5cbiAgICBjb25zdCByb3V0ZVByb3BhZ2F0aW9uID0gbmV3IENmblZQTkdhdGV3YXlSb3V0ZVByb3BhZ2F0aW9uKHRoaXMsICdSb3V0ZVByb3BhZ2F0aW9uJywge1xuICAgICAgcm91dGVUYWJsZUlkcyxcbiAgICAgIHZwbkdhdGV3YXlJZDogdGhpcy5fdnBuR2F0ZXdheUlkLFxuICAgIH0pO1xuICAgIC8vIFRoZSBBV1M6OkVDMjo6VlBOR2F0ZXdheVJvdXRlUHJvcGFnYXRpb24gcmVzb3VyY2UgY2Fubm90IHVzZSB0aGUgVlBOIGdhdGV3YXlcbiAgICAvLyB1bnRpbCBpdCBoYXMgc3VjY2Vzc2Z1bGx5IGF0dGFjaGVkIHRvIHRoZSBWUEMuXG4gICAgLy8gU2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NDbG91ZEZvcm1hdGlvbi9sYXRlc3QvVXNlckd1aWRlL2F3cy1yZXNvdXJjZS1lYzItdnBuLWdhdGV3YXlyb3V0ZXByb3AuaHRtbFxuICAgIHJvdXRlUHJvcGFnYXRpb24ubm9kZS5hZGREZXBlbmRlbmN5KGF0dGFjaG1lbnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgVlBOIGNvbm5lY3Rpb24gdG8gdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyBhZGRWcG5Db25uZWN0aW9uKGlkOiBzdHJpbmcsIG9wdGlvbnM6IFZwbkNvbm5lY3Rpb25PcHRpb25zKTogVnBuQ29ubmVjdGlvbiB7XG4gICAgcmV0dXJuIG5ldyBWcG5Db25uZWN0aW9uKHRoaXMsIGlkLCB7XG4gICAgICB2cGM6IHRoaXMsXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgaW50ZXJmYWNlIGVuZHBvaW50IHRvIHRoaXMgVlBDXG4gICAqL1xuICBwdWJsaWMgYWRkSW50ZXJmYWNlRW5kcG9pbnQoaWQ6IHN0cmluZywgb3B0aW9uczogSW50ZXJmYWNlVnBjRW5kcG9pbnRPcHRpb25zKTogSW50ZXJmYWNlVnBjRW5kcG9pbnQge1xuICAgIHJldHVybiBuZXcgSW50ZXJmYWNlVnBjRW5kcG9pbnQodGhpcywgaWQsIHtcbiAgICAgIHZwYzogdGhpcyxcbiAgICAgIC4uLm9wdGlvbnMsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIG5ldyBnYXRld2F5IGVuZHBvaW50IHRvIHRoaXMgVlBDXG4gICAqL1xuICBwdWJsaWMgYWRkR2F0ZXdheUVuZHBvaW50KGlkOiBzdHJpbmcsIG9wdGlvbnM6IEdhdGV3YXlWcGNFbmRwb2ludE9wdGlvbnMpOiBHYXRld2F5VnBjRW5kcG9pbnQge1xuICAgIHJldHVybiBuZXcgR2F0ZXdheVZwY0VuZHBvaW50KHRoaXMsIGlkLCB7XG4gICAgICB2cGM6IHRoaXMsXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgZmxvdyBsb2cgdG8gdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyBhZGRGbG93TG9nKGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBGbG93TG9nT3B0aW9ucyk6IEZsb3dMb2cge1xuICAgIHJldHVybiBuZXcgRmxvd0xvZyh0aGlzLCBpZCwge1xuICAgICAgcmVzb3VyY2VUeXBlOiBGbG93TG9nUmVzb3VyY2VUeXBlLmZyb21WcGModGhpcyksXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIGlkIG9mIHRoZSBWUE4gR2F0ZXdheSAoaWYgZW5hYmxlZClcbiAgICovXG4gIHB1YmxpYyBnZXQgdnBuR2F0ZXdheUlkKCk6IHN0cmluZyB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMuX3ZwbkdhdGV3YXlJZDtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIHN1Ym5ldHMgYXBwcm9wcmlhdGUgZm9yIHRoZSBwbGFjZW1lbnQgc3RyYXRlZ3lcbiAgICovXG4gIHByb3RlY3RlZCBzZWxlY3RTdWJuZXRPYmplY3RzKHNlbGVjdGlvbjogU3VibmV0U2VsZWN0aW9uID0ge30pOiBJU3VibmV0W10ge1xuICAgIHNlbGVjdGlvbiA9IHRoaXMucmVpZnlTZWxlY3Rpb25EZWZhdWx0cyhzZWxlY3Rpb24pO1xuXG4gICAgaWYgKHNlbGVjdGlvbi5zdWJuZXRzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldHVybiBzZWxlY3Rpb24uc3VibmV0cztcbiAgICB9XG5cbiAgICBsZXQgc3VibmV0cztcblxuICAgIGlmIChzZWxlY3Rpb24uc3VibmV0R3JvdXBOYW1lICE9PSB1bmRlZmluZWQpIHsgLy8gU2VsZWN0IGJ5IG5hbWVcbiAgICAgIHN1Ym5ldHMgPSB0aGlzLnNlbGVjdFN1Ym5ldE9iamVjdHNCeU5hbWUoc2VsZWN0aW9uLnN1Ym5ldEdyb3VwTmFtZSk7XG5cbiAgICB9IGVsc2UgeyAvLyBPciBzcGVjaWZ5IGJ5IHR5cGVcbiAgICAgIGNvbnN0IHR5cGUgPSBzZWxlY3Rpb24uc3VibmV0VHlwZSB8fCBTdWJuZXRUeXBlLlBSSVZBVEU7XG4gICAgICBzdWJuZXRzID0gdGhpcy5zZWxlY3RTdWJuZXRPYmplY3RzQnlUeXBlKHR5cGUpO1xuICAgIH1cblxuICAgIC8vIEFwcGx5IGFsbCB0aGUgZmlsdGVyc1xuICAgIHN1Ym5ldHMgPSB0aGlzLmFwcGx5U3VibmV0RmlsdGVycyhzdWJuZXRzLCBzZWxlY3Rpb24uc3VibmV0RmlsdGVycyA/PyBbXSk7XG5cbiAgICByZXR1cm4gc3VibmV0cztcbiAgfVxuXG4gIHByaXZhdGUgYXBwbHlTdWJuZXRGaWx0ZXJzKHN1Ym5ldHM6IElTdWJuZXRbXSwgZmlsdGVyczogU3VibmV0RmlsdGVyW10pOiBJU3VibmV0W10ge1xuICAgIGxldCBmaWx0ZXJlZCA9IHN1Ym5ldHM7XG4gICAgLy8gQXBwbHkgZWFjaCBmaWx0ZXIgaW4gc2VxdWVuY2VcbiAgICBmb3IgKGNvbnN0IGZpbHRlciBvZiBmaWx0ZXJzKSB7XG4gICAgICBmaWx0ZXJlZCA9IGZpbHRlci5zZWxlY3RTdWJuZXRzKGZpbHRlcmVkKTtcbiAgICB9XG4gICAgcmV0dXJuIGZpbHRlcmVkO1xuICB9XG5cbiAgcHJpdmF0ZSBzZWxlY3RTdWJuZXRPYmplY3RzQnlOYW1lKGdyb3VwTmFtZTogc3RyaW5nKSB7XG4gICAgY29uc3QgYWxsU3VibmV0cyA9IFsuLi50aGlzLnB1YmxpY1N1Ym5ldHMsIC4uLnRoaXMucHJpdmF0ZVN1Ym5ldHMsIC4uLnRoaXMuaXNvbGF0ZWRTdWJuZXRzXTtcbiAgICBjb25zdCBzdWJuZXRzID0gYWxsU3VibmV0cy5maWx0ZXIocyA9PiBzdWJuZXRHcm91cE5hbWVGcm9tQ29uc3RydWN0SWQocykgPT09IGdyb3VwTmFtZSk7XG5cbiAgICBpZiAoc3VibmV0cy5sZW5ndGggPT09IDAgJiYgIXRoaXMuaW5jb21wbGV0ZVN1Ym5ldERlZmluaXRpb24pIHtcbiAgICAgIGNvbnN0IG5hbWVzID0gQXJyYXkuZnJvbShuZXcgU2V0KGFsbFN1Ym5ldHMubWFwKHN1Ym5ldEdyb3VwTmFtZUZyb21Db25zdHJ1Y3RJZCkpKTtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVGhlcmUgYXJlIG5vIHN1Ym5ldCBncm91cHMgd2l0aCBuYW1lICcke2dyb3VwTmFtZX0nIGluIHRoaXMgVlBDLiBBdmFpbGFibGUgbmFtZXM6ICR7bmFtZXN9YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHN1Ym5ldHM7XG4gIH1cblxuICBwcml2YXRlIHNlbGVjdFN1Ym5ldE9iamVjdHNCeVR5cGUoc3VibmV0VHlwZTogU3VibmV0VHlwZSkge1xuICAgIGNvbnN0IGFsbFN1Ym5ldHMgPSB7XG4gICAgICBbU3VibmV0VHlwZS5JU09MQVRFRF06IHRoaXMuaXNvbGF0ZWRTdWJuZXRzLFxuICAgICAgW1N1Ym5ldFR5cGUuUFJJVkFURV06IHRoaXMucHJpdmF0ZVN1Ym5ldHMsXG4gICAgICBbU3VibmV0VHlwZS5QVUJMSUNdOiB0aGlzLnB1YmxpY1N1Ym5ldHMsXG4gICAgfTtcblxuICAgIGNvbnN0IHN1Ym5ldHMgPSBhbGxTdWJuZXRzW3N1Ym5ldFR5cGVdO1xuXG4gICAgLy8gRm9yY2UgbWVyZ2UgY29uZmxpY3QgaGVyZSB3aXRoIGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9wdWxsLzQwODlcbiAgICAvLyBzZWUgSW1wb3J0ZWRWcGNcblxuICAgIGlmIChzdWJuZXRzLmxlbmd0aCA9PT0gMCAmJiAhdGhpcy5pbmNvbXBsZXRlU3VibmV0RGVmaW5pdGlvbikge1xuICAgICAgY29uc3QgYXZhaWxhYmxlVHlwZXMgPSBPYmplY3QuZW50cmllcyhhbGxTdWJuZXRzKS5maWx0ZXIoKFtfLCBzdWJzXSkgPT4gc3Vicy5sZW5ndGggPiAwKS5tYXAoKFt0eXBlTmFtZSwgX10pID0+IHR5cGVOYW1lKTtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgVGhlcmUgYXJlIG5vICcke3N1Ym5ldFR5cGV9JyBzdWJuZXQgZ3JvdXBzIGluIHRoaXMgVlBDLiBBdmFpbGFibGUgdHlwZXM6ICR7YXZhaWxhYmxlVHlwZXN9YCk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHN1Ym5ldHM7XG4gIH1cblxuICAvKipcbiAgICogVmFsaWRhdGUgdGhlIGZpZWxkcyBpbiBhIFN1Ym5ldFNlbGVjdGlvbiBvYmplY3QsIGFuZCByZWlmeSBkZWZhdWx0cyBpZiBuZWNlc3NhcnlcbiAgICpcbiAgICogSW4gY2FzZSBvZiBkZWZhdWx0IHNlbGVjdGlvbiwgc2VsZWN0IHRoZSBmaXJzdCB0eXBlIG9mIFBSSVZBVEUsIElTT0xBVEVELFxuICAgKiBQVUJMSUMgKGluIHRoYXQgb3JkZXIpIHRoYXQgaGFzIGFueSBzdWJuZXRzLlxuICAgKi9cbiAgcHJpdmF0ZSByZWlmeVNlbGVjdGlvbkRlZmF1bHRzKHBsYWNlbWVudDogU3VibmV0U2VsZWN0aW9uKTogU3VibmV0U2VsZWN0aW9uIHtcblxuICAgIGlmIChwbGFjZW1lbnQuc3VibmV0TmFtZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBpZiAocGxhY2VtZW50LnN1Ym5ldEdyb3VwTmFtZSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignUGxlYXNlIHVzZSBvbmx5IFxcJ3N1Ym5ldEdyb3VwTmFtZVxcJyAoXFwnc3VibmV0TmFtZVxcJyBpcyBkZXByZWNhdGVkIGFuZCBoYXMgdGhlIHNhbWUgYmVoYXZpb3IpJyk7XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBBbm5vdGF0aW9ucy5vZih0aGlzKS5hZGRXYXJuaW5nKCdVc2FnZSBvZiBcXCdzdWJuZXROYW1lXFwnIGluIFN1Ym5ldFNlbGVjdGlvbiBpcyBkZXByZWNhdGVkLCB1c2UgXFwnc3VibmV0R3JvdXBOYW1lXFwnIGluc3RlYWQnKTtcbiAgICAgIH1cbiAgICAgIHBsYWNlbWVudCA9IHsgLi4ucGxhY2VtZW50LCBzdWJuZXRHcm91cE5hbWU6IHBsYWNlbWVudC5zdWJuZXROYW1lIH07XG4gICAgfVxuXG4gICAgY29uc3QgZXhjbHVzaXZlU2VsZWN0aW9uczogQXJyYXk8a2V5b2YgU3VibmV0U2VsZWN0aW9uPiA9IFsnc3VibmV0cycsICdzdWJuZXRUeXBlJywgJ3N1Ym5ldEdyb3VwTmFtZSddO1xuICAgIGNvbnN0IHByb3ZpZGVkU2VsZWN0aW9ucyA9IGV4Y2x1c2l2ZVNlbGVjdGlvbnMuZmlsdGVyKGtleSA9PiBwbGFjZW1lbnRba2V5XSAhPT0gdW5kZWZpbmVkKTtcbiAgICBpZiAocHJvdmlkZWRTZWxlY3Rpb25zLmxlbmd0aCA+IDEpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgT25seSBvbmUgb2YgJyR7cHJvdmlkZWRTZWxlY3Rpb25zfScgY2FuIGJlIHN1cHBsaWVkIHRvIHN1Ym5ldCBzZWxlY3Rpb24uYCk7XG4gICAgfVxuXG4gICAgaWYgKHBsYWNlbWVudC5zdWJuZXRUeXBlID09PSB1bmRlZmluZWQgJiYgcGxhY2VtZW50LnN1Ym5ldEdyb3VwTmFtZSA9PT0gdW5kZWZpbmVkICYmIHBsYWNlbWVudC5zdWJuZXRzID09PSB1bmRlZmluZWQpIHtcbiAgICAgIC8vIFJldHVybiBkZWZhdWx0IHN1Ym5ldCB0eXBlIGJhc2VkIG9uIHN1Ym5ldHMgdGhhdCBhY3R1YWxseSBleGlzdFxuICAgICAgbGV0IHN1Ym5ldFR5cGUgPSB0aGlzLnByaXZhdGVTdWJuZXRzLmxlbmd0aCA/IFN1Ym5ldFR5cGUuUFJJVkFURSA6IHRoaXMuaXNvbGF0ZWRTdWJuZXRzLmxlbmd0aCA/IFN1Ym5ldFR5cGUuSVNPTEFURUQgOiBTdWJuZXRUeXBlLlBVQkxJQztcbiAgICAgIHBsYWNlbWVudCA9IHsgLi4ucGxhY2VtZW50LCBzdWJuZXRUeXBlOiBzdWJuZXRUeXBlIH07XG4gICAgfVxuXG4gICAgLy8gRXN0YWJsaXNoIHdoaWNoIHN1Ym5ldCBmaWx0ZXJzIGFyZSBnb2luZyB0byBiZSB1c2VkXG4gICAgbGV0IHN1Ym5ldEZpbHRlcnMgPSBwbGFjZW1lbnQuc3VibmV0RmlsdGVycyA/PyBbXTtcblxuICAgIC8vIEJhY2t3YXJkcyBjb21wYXRpYmlsaXR5IHdpdGggZXhpc3RpbmcgYGF2YWlsYWJpbGl0eVpvbmVzYCBhbmQgYG9uZVBlckF6YCBmdW5jdGlvbmFsaXR5XG4gICAgaWYgKHBsYWNlbWVudC5hdmFpbGFiaWxpdHlab25lcyAhPT0gdW5kZWZpbmVkKSB7IC8vIEZpbHRlciBieSBBWnMsIGlmIHNwZWNpZmllZFxuICAgICAgc3VibmV0RmlsdGVycy5wdXNoKFN1Ym5ldEZpbHRlci5hdmFpbGFiaWxpdHlab25lcyhwbGFjZW1lbnQuYXZhaWxhYmlsaXR5Wm9uZXMpKTtcbiAgICB9XG4gICAgaWYgKCEhcGxhY2VtZW50Lm9uZVBlckF6KSB7IC8vIEVuc3VyZSBvbmUgcGVyIEFaIGlmIHNwZWNpZmllZFxuICAgICAgc3VibmV0RmlsdGVycy5wdXNoKFN1Ym5ldEZpbHRlci5vbmVQZXJBeigpKTtcbiAgICB9XG5cbiAgICAvLyBPdmVyd3JpdGUgdGhlIHByb3ZpZGVkIHBsYWNlbWVudCBmaWx0ZXJzIGFuZCByZW1vdmUgdGhlIGF2YWlsYWJpbGl0eVpvbmVzIGFuZCBvbmVQZXJBeiBwcm9wZXJ0aWVzXG4gICAgcGxhY2VtZW50ID0geyAuLi5wbGFjZW1lbnQsIHN1Ym5ldEZpbHRlcnM6IHN1Ym5ldEZpbHRlcnMsIGF2YWlsYWJpbGl0eVpvbmVzOiB1bmRlZmluZWQsIG9uZVBlckF6OiB1bmRlZmluZWQgfTtcbiAgICBjb25zdCB7IGF2YWlsYWJpbGl0eVpvbmVzLCBvbmVQZXJBeiwgLi4ucmVzdCB9ID0gcGxhY2VtZW50O1xuXG4gICAgcmV0dXJuIHJlc3Q7XG4gIH1cbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIHRoYXQgcmVmZXJlbmNlIGFuIGV4dGVybmFsIFZwY1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFZwY0F0dHJpYnV0ZXMge1xuICAvKipcbiAgICogVlBDJ3MgaWRlbnRpZmllclxuICAgKi9cbiAgcmVhZG9ubHkgdnBjSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogVlBDJ3MgQ0lEUiByYW5nZVxuICAgKlxuICAgKiBAZGVmYXVsdCAtIFJldHJpZXZpbmcgdGhlIENJRFIgZnJvbSB0aGUgVlBDIHdpbGwgZmFpbFxuICAgKi9cbiAgcmVhZG9ubHkgdnBjQ2lkckJsb2NrPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIGF2YWlsYWJpbGl0eSB6b25lcyBmb3IgdGhlIHN1Ym5ldHMgaW4gdGhpcyBWUEMuXG4gICAqL1xuICByZWFkb25seSBhdmFpbGFiaWxpdHlab25lczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgcHVibGljIHN1Ym5ldCBJRHNcbiAgICpcbiAgICogTXVzdCBiZSB1bmRlZmluZWQgb3IgbWF0Y2ggdGhlIGF2YWlsYWJpbGl0eSB6b25lcyBpbiBsZW5ndGggYW5kIG9yZGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgcHVibGljU3VibmV0SWRzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgbmFtZXMgZm9yIHRoZSBwdWJsaWMgc3VibmV0c1xuICAgKlxuICAgKiBNdXN0IGJlIHVuZGVmaW5lZCBvciBoYXZlIGEgbmFtZSBmb3IgZXZlcnkgcHVibGljIHN1Ym5ldCBncm91cC5cbiAgICovXG4gIHJlYWRvbmx5IHB1YmxpY1N1Ym5ldE5hbWVzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgSURzIG9mIHJvdXRpbmcgdGFibGVzIGZvciB0aGUgcHVibGljIHN1Ym5ldHMuXG4gICAqXG4gICAqIE11c3QgYmUgdW5kZWZpbmVkIG9yIGhhdmUgYSBuYW1lIGZvciBldmVyeSBwdWJsaWMgc3VibmV0IGdyb3VwLlxuICAgKi9cbiAgcmVhZG9ubHkgcHVibGljU3VibmV0Um91dGVUYWJsZUlkcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIHByaXZhdGUgc3VibmV0IElEc1xuICAgKlxuICAgKiBNdXN0IGJlIHVuZGVmaW5lZCBvciBtYXRjaCB0aGUgYXZhaWxhYmlsaXR5IHpvbmVzIGluIGxlbmd0aCBhbmQgb3JkZXIuXG4gICAqL1xuICByZWFkb25seSBwcml2YXRlU3VibmV0SWRzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgbmFtZXMgZm9yIHRoZSBwcml2YXRlIHN1Ym5ldHNcbiAgICpcbiAgICogTXVzdCBiZSB1bmRlZmluZWQgb3IgaGF2ZSBhIG5hbWUgZm9yIGV2ZXJ5IHByaXZhdGUgc3VibmV0IGdyb3VwLlxuICAgKi9cbiAgcmVhZG9ubHkgcHJpdmF0ZVN1Ym5ldE5hbWVzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgSURzIG9mIHJvdXRpbmcgdGFibGVzIGZvciB0aGUgcHJpdmF0ZSBzdWJuZXRzLlxuICAgKlxuICAgKiBNdXN0IGJlIHVuZGVmaW5lZCBvciBoYXZlIGEgbmFtZSBmb3IgZXZlcnkgcHJpdmF0ZSBzdWJuZXQgZ3JvdXAuXG4gICAqL1xuICByZWFkb25seSBwcml2YXRlU3VibmV0Um91dGVUYWJsZUlkcz86IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBMaXN0IG9mIGlzb2xhdGVkIHN1Ym5ldCBJRHNcbiAgICpcbiAgICogTXVzdCBiZSB1bmRlZmluZWQgb3IgbWF0Y2ggdGhlIGF2YWlsYWJpbGl0eSB6b25lcyBpbiBsZW5ndGggYW5kIG9yZGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgaXNvbGF0ZWRTdWJuZXRJZHM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogTGlzdCBvZiBuYW1lcyBmb3IgdGhlIGlzb2xhdGVkIHN1Ym5ldHNcbiAgICpcbiAgICogTXVzdCBiZSB1bmRlZmluZWQgb3IgaGF2ZSBhIG5hbWUgZm9yIGV2ZXJ5IGlzb2xhdGVkIHN1Ym5ldCBncm91cC5cbiAgICovXG4gIHJlYWRvbmx5IGlzb2xhdGVkU3VibmV0TmFtZXM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogTGlzdCBvZiBJRHMgb2Ygcm91dGluZyB0YWJsZXMgZm9yIHRoZSBpc29sYXRlZCBzdWJuZXRzLlxuICAgKlxuICAgKiBNdXN0IGJlIHVuZGVmaW5lZCBvciBoYXZlIGEgbmFtZSBmb3IgZXZlcnkgaXNvbGF0ZWQgc3VibmV0IGdyb3VwLlxuICAgKi9cbiAgcmVhZG9ubHkgaXNvbGF0ZWRTdWJuZXRSb3V0ZVRhYmxlSWRzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFZQTiBnYXRld2F5J3MgaWRlbnRpZmllclxuICAgKi9cbiAgcmVhZG9ubHkgdnBuR2F0ZXdheUlkPzogc3RyaW5nO1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFN1Ym5ldEF0dHJpYnV0ZXMge1xuXG4gIC8qKlxuICAgKiBUaGUgQXZhaWxhYmlsaXR5IFpvbmUgdGhlIHN1Ym5ldCBpcyBsb2NhdGVkIGluXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gQVogaW5mb3JtYXRpb24sIGNhbm5vdCB1c2UgQVogc2VsZWN0aW9uIGZlYXR1cmVzXG4gICAqL1xuICByZWFkb25seSBhdmFpbGFiaWxpdHlab25lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgSVB2NCBDSURSIGJsb2NrIGFzc29jaWF0ZWQgd2l0aCB0aGUgc3VibmV0XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gQ0lEUiBpbmZvcm1hdGlvbiwgY2Fubm90IHVzZSBDSURSIGZpbHRlciBmZWF0dXJlc1xuICAgKi9cbiAgcmVhZG9ubHkgaXB2NENpZHJCbG9jaz86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIElEIG9mIHRoZSByb3V0ZSB0YWJsZSBmb3IgdGhpcyBwYXJ0aWN1bGFyIHN1Ym5ldFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHJvdXRlIHRhYmxlIGluZm9ybWF0aW9uLCBjYW5ub3QgY3JlYXRlIFZQQyBlbmRwb2ludHNcbiAgICovXG4gIHJlYWRvbmx5IHJvdXRlVGFibGVJZD86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHN1Ym5ldElkIGZvciB0aGlzIHBhcnRpY3VsYXIgc3VibmV0XG4gICAqL1xuICByZWFkb25seSBzdWJuZXRJZDogc3RyaW5nO1xufVxuXG4vKipcbiAqIE5hbWUgdGFnIGNvbnN0YW50XG4gKi9cbmNvbnN0IE5BTUVfVEFHOiBzdHJpbmcgPSAnTmFtZSc7XG5cbi8qKlxuICogQ29uZmlndXJhdGlvbiBmb3IgVnBjXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVnBjUHJvcHMge1xuXG4gIC8qKlxuICAgKiBUaGUgQ0lEUiByYW5nZSB0byB1c2UgZm9yIHRoZSBWUEMsIGUuZy4gJzEwLjAuMC4wLzE2Jy5cbiAgICpcbiAgICogU2hvdWxkIGJlIGEgbWluaW11bSBvZiAvMjggYW5kIG1heGltdW0gc2l6ZSBvZiAvMTYuIFRoZSByYW5nZSB3aWxsIGJlXG4gICAqIHNwbGl0IGFjcm9zcyBhbGwgc3VibmV0cyBwZXIgQXZhaWxhYmlsaXR5IFpvbmUuXG4gICAqXG4gICAqIEBkZWZhdWx0IFZwYy5ERUZBVUxUX0NJRFJfUkFOR0VcbiAgICovXG4gIHJlYWRvbmx5IGNpZHI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIHRoZSBpbnN0YW5jZXMgbGF1bmNoZWQgaW4gdGhlIFZQQyBnZXQgcHVibGljIEROUyBob3N0bmFtZXMuXG4gICAqXG4gICAqIElmIHRoaXMgYXR0cmlidXRlIGlzIHRydWUsIGluc3RhbmNlcyBpbiB0aGUgVlBDIGdldCBwdWJsaWMgRE5TIGhvc3RuYW1lcyxcbiAgICogYnV0IG9ubHkgaWYgdGhlIGVuYWJsZURuc1N1cHBvcnQgYXR0cmlidXRlIGlzIGFsc28gc2V0IHRvIHRydWUuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZURuc0hvc3RuYW1lcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIHRoZSBETlMgcmVzb2x1dGlvbiBpcyBzdXBwb3J0ZWQgZm9yIHRoZSBWUEMuXG4gICAqXG4gICAqIElmIHRoaXMgYXR0cmlidXRlIGlzIGZhbHNlLCB0aGUgQW1hem9uLXByb3ZpZGVkIEROUyBzZXJ2ZXIgaW4gdGhlIFZQQyB0aGF0XG4gICAqIHJlc29sdmVzIHB1YmxpYyBETlMgaG9zdG5hbWVzIHRvIElQIGFkZHJlc3NlcyBpcyBub3QgZW5hYmxlZC4gSWYgdGhpc1xuICAgKiBhdHRyaWJ1dGUgaXMgdHJ1ZSwgcXVlcmllcyB0byB0aGUgQW1hem9uIHByb3ZpZGVkIEROUyBzZXJ2ZXIgYXQgdGhlXG4gICAqIDE2OS4yNTQuMTY5LjI1MyBJUCBhZGRyZXNzLCBvciB0aGUgcmVzZXJ2ZWQgSVAgYWRkcmVzcyBhdCB0aGUgYmFzZSBvZiB0aGVcbiAgICogVlBDIElQdjQgbmV0d29yayByYW5nZSBwbHVzIHR3byB3aWxsIHN1Y2NlZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZURuc1N1cHBvcnQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgZGVmYXVsdCB0ZW5hbmN5IG9mIGluc3RhbmNlcyBsYXVuY2hlZCBpbnRvIHRoZSBWUEMuXG4gICAqXG4gICAqIEJ5IHNldHRpbmcgdGhpcyB0byBkZWRpY2F0ZWQgdGVuYW5jeSwgaW5zdGFuY2VzIHdpbGwgYmUgbGF1bmNoZWQgb25cbiAgICogaGFyZHdhcmUgZGVkaWNhdGVkIHRvIGEgc2luZ2xlIEFXUyBjdXN0b21lciwgdW5sZXNzIHNwZWNpZmljYWxseSBzcGVjaWZpZWRcbiAgICogYXQgaW5zdGFuY2UgbGF1bmNoIHRpbWUuIFBsZWFzZSBub3RlLCBub3QgYWxsIGluc3RhbmNlIHR5cGVzIGFyZSB1c2FibGVcbiAgICogd2l0aCBEZWRpY2F0ZWQgdGVuYW5jeS5cbiAgICpcbiAgICogQGRlZmF1bHQgRGVmYXVsdEluc3RhbmNlVGVuYW5jeS5EZWZhdWx0IChzaGFyZWQpIHRlbmFuY3lcbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRJbnN0YW5jZVRlbmFuY3k/OiBEZWZhdWx0SW5zdGFuY2VUZW5hbmN5O1xuXG4gIC8qKlxuICAgKiBEZWZpbmUgdGhlIG1heGltdW0gbnVtYmVyIG9mIEFacyB0byB1c2UgaW4gdGhpcyByZWdpb25cbiAgICpcbiAgICogSWYgdGhlIHJlZ2lvbiBoYXMgbW9yZSBBWnMgdGhhbiB5b3Ugd2FudCB0byB1c2UgKGZvciBleGFtcGxlLCBiZWNhdXNlIG9mXG4gICAqIEVJUCBsaW1pdHMpLCBwaWNrIGEgbG93ZXIgbnVtYmVyIGhlcmUuIFRoZSBBWnMgd2lsbCBiZSBzb3J0ZWQgYW5kIHBpY2tlZFxuICAgKiBmcm9tIHRoZSBzdGFydCBvZiB0aGUgbGlzdC5cbiAgICpcbiAgICogSWYgeW91IHBpY2sgYSBoaWdoZXIgbnVtYmVyIHRoYW4gdGhlIG51bWJlciBvZiBBWnMgaW4gdGhlIHJlZ2lvbiwgYWxsIEFac1xuICAgKiBpbiB0aGUgcmVnaW9uIHdpbGwgYmUgc2VsZWN0ZWQuIFRvIHVzZSBcImFsbCBBWnNcIiBhdmFpbGFibGUgdG8geW91clxuICAgKiBhY2NvdW50LCB1c2UgYSBoaWdoIG51bWJlciAoc3VjaCBhcyA5OSkuXG4gICAqXG4gICAqIEJlIGF3YXJlIHRoYXQgZW52aXJvbm1lbnQtYWdub3N0aWMgc3RhY2tzIHdpbGwgYmUgY3JlYXRlZCB3aXRoIGFjY2VzcyB0b1xuICAgKiBvbmx5IDIgQVpzLCBzbyB0byB1c2UgbW9yZSB0aGFuIDIgQVpzLCBiZSBzdXJlIHRvIHNwZWNpZnkgdGhlIGFjY291bnQgYW5kXG4gICAqIHJlZ2lvbiBvbiB5b3VyIHN0YWNrLlxuICAgKlxuICAgKiBAZGVmYXVsdCAzXG4gICAqL1xuICByZWFkb25seSBtYXhBenM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgTkFUIEdhdGV3YXlzL0luc3RhbmNlcyB0byBjcmVhdGUuXG4gICAqXG4gICAqIFRoZSB0eXBlIG9mIE5BVCBnYXRld2F5IG9yIGluc3RhbmNlIHdpbGwgYmUgZGV0ZXJtaW5lZCBieSB0aGVcbiAgICogYG5hdEdhdGV3YXlQcm92aWRlcmAgcGFyYW1ldGVyLlxuICAgKlxuICAgKiBZb3UgY2FuIHNldCB0aGlzIG51bWJlciBsb3dlciB0aGFuIHRoZSBudW1iZXIgb2YgQXZhaWxhYmlsaXR5IFpvbmVzIGluIHlvdXJcbiAgICogVlBDIGluIG9yZGVyIHRvIHNhdmUgb24gTkFUIGNvc3QuIEJlIGF3YXJlIHlvdSBtYXkgYmUgY2hhcmdlZCBmb3JcbiAgICogY3Jvc3MtQVogZGF0YSB0cmFmZmljIGluc3RlYWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gT25lIE5BVCBnYXRld2F5L2luc3RhbmNlIHBlciBBdmFpbGFiaWxpdHkgWm9uZVxuICAgKi9cbiAgcmVhZG9ubHkgbmF0R2F0ZXdheXM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyZXMgdGhlIHN1Ym5ldHMgd2hpY2ggd2lsbCBoYXZlIE5BVCBHYXRld2F5cy9JbnN0YW5jZXNcbiAgICpcbiAgICogWW91IGNhbiBwaWNrIGEgc3BlY2lmaWMgZ3JvdXAgb2Ygc3VibmV0cyBieSBzcGVjaWZ5aW5nIHRoZSBncm91cCBuYW1lO1xuICAgKiB0aGUgcGlja2VkIHN1Ym5ldHMgbXVzdCBiZSBwdWJsaWMgc3VibmV0cy5cbiAgICpcbiAgICogT25seSBuZWNlc3NhcnkgaWYgeW91IGhhdmUgbW9yZSB0aGFuIG9uZSBwdWJsaWMgc3VibmV0IGdyb3VwLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFsbCBwdWJsaWMgc3VibmV0cy5cbiAgICovXG4gIHJlYWRvbmx5IG5hdEdhdGV3YXlTdWJuZXRzPzogU3VibmV0U2VsZWN0aW9uO1xuXG4gIC8qKlxuICAgKiBXaGF0IHR5cGUgb2YgTkFUIHByb3ZpZGVyIHRvIHVzZVxuICAgKlxuICAgKiBTZWxlY3QgYmV0d2VlbiBOQVQgZ2F0ZXdheXMgb3IgTkFUIGluc3RhbmNlcy4gTkFUIGdhdGV3YXlzXG4gICAqIG1heSBub3QgYmUgYXZhaWxhYmxlIGluIGFsbCBBV1MgcmVnaW9ucy5cbiAgICpcbiAgICogQGRlZmF1bHQgTmF0UHJvdmlkZXIuZ2F0ZXdheSgpXG4gICAqIEBleHBlcmltZW50YWxcbiAgICovXG4gIHJlYWRvbmx5IG5hdEdhdGV3YXlQcm92aWRlcj86IE5hdFByb3ZpZGVyO1xuXG4gIC8qKlxuICAgKiBDb25maWd1cmUgdGhlIHN1Ym5ldHMgdG8gYnVpbGQgZm9yIGVhY2ggQVpcbiAgICpcbiAgICogRWFjaCBlbnRyeSBpbiB0aGlzIGxpc3QgY29uZmlndXJlcyBhIFN1Ym5ldCBHcm91cDsgZWFjaCBncm91cCB3aWxsIGNvbnRhaW4gYVxuICAgKiBzdWJuZXQgZm9yIGVhY2ggQXZhaWxhYmlsaXR5IFpvbmUuXG4gICAqXG4gICAqIEZvciBleGFtcGxlLCBpZiB5b3Ugd2FudCAxIHB1YmxpYyBzdWJuZXQsIDEgcHJpdmF0ZSBzdWJuZXQsIGFuZCAxIGlzb2xhdGVkXG4gICAqIHN1Ym5ldCBpbiBlYWNoIEFaIHByb3ZpZGUgdGhlIGZvbGxvd2luZzpcbiAgICpcbiAgICogYGBgdHNcbiAgICogbmV3IGVjMi5WcGModGhpcywgJ1ZQQycsIHtcbiAgICogICBzdWJuZXRDb25maWd1cmF0aW9uOiBbXG4gICAqICAgICAge1xuICAgKiAgICAgICAgY2lkck1hc2s6IDI0LFxuICAgKiAgICAgICAgbmFtZTogJ2luZ3Jlc3MnLFxuICAgKiAgICAgICAgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUFVCTElDLFxuICAgKiAgICAgIH0sXG4gICAqICAgICAge1xuICAgKiAgICAgICAgY2lkck1hc2s6IDI0LFxuICAgKiAgICAgICAgbmFtZTogJ2FwcGxpY2F0aW9uJyxcbiAgICogICAgICAgIHN1Ym5ldFR5cGU6IGVjMi5TdWJuZXRUeXBlLlBSSVZBVEUsXG4gICAqICAgICAgfSxcbiAgICogICAgICB7XG4gICAqICAgICAgICBjaWRyTWFzazogMjgsXG4gICAqICAgICAgICBuYW1lOiAncmRzJyxcbiAgICogICAgICAgIHN1Ym5ldFR5cGU6IGVjMi5TdWJuZXRUeXBlLklTT0xBVEVELFxuICAgKiAgICAgIH1cbiAgICogICBdXG4gICAqIH0pO1xuICAgKiBgYGBcbiAgICpcbiAgICogQGRlZmF1bHQgLSBUaGUgVlBDIENJRFIgd2lsbCBiZSBldmVubHkgZGl2aWRlZCBiZXR3ZWVuIDEgcHVibGljIGFuZCAxXG4gICAqIHByaXZhdGUgc3VibmV0IHBlciBBWi5cbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldENvbmZpZ3VyYXRpb24/OiBTdWJuZXRDb25maWd1cmF0aW9uW107XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIGEgVlBOIGdhdGV3YXkgc2hvdWxkIGJlIGNyZWF0ZWQgYW5kIGF0dGFjaGVkIHRvIHRoaXMgVlBDLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHRydWUgd2hlbiB2cG5HYXRld2F5QXNuIG9yIHZwbkNvbm5lY3Rpb25zIGlzIHNwZWNpZmllZFxuICAgKi9cbiAgcmVhZG9ubHkgdnBuR2F0ZXdheT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBwcml2YXRlIEF1dG9ub21vdXMgU3lzdGVtIE51bWJlciAoQVNOKSBmb3IgdGhlIFZQTiBnYXRld2F5LlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFtYXpvbiBkZWZhdWx0IEFTTi5cbiAgICovXG4gIHJlYWRvbmx5IHZwbkdhdGV3YXlBc24/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFZQTiBjb25uZWN0aW9ucyB0byB0aGlzIFZQQy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBjb25uZWN0aW9ucy5cbiAgICovXG4gIHJlYWRvbmx5IHZwbkNvbm5lY3Rpb25zPzogeyBbaWQ6IHN0cmluZ106IFZwbkNvbm5lY3Rpb25PcHRpb25zIH1cblxuICAvKipcbiAgICogV2hlcmUgdG8gcHJvcGFnYXRlIFZQTiByb3V0ZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gT24gdGhlIHJvdXRlIHRhYmxlcyBhc3NvY2lhdGVkIHdpdGggcHJpdmF0ZSBzdWJuZXRzLiBJZiBub1xuICAgKiBwcml2YXRlIHN1Ym5ldHMgZXhpc3RzLCBpc29sYXRlZCBzdWJuZXRzIGFyZSB1c2VkLiBJZiBubyBpc29sYXRlZCBzdWJuZXRzXG4gICAqIGV4aXN0cywgcHVibGljIHN1Ym5ldHMgYXJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSB2cG5Sb3V0ZVByb3BhZ2F0aW9uPzogU3VibmV0U2VsZWN0aW9uW11cblxuICAvKipcbiAgICogR2F0ZXdheSBlbmRwb2ludHMgdG8gYWRkIHRvIHRoaXMgVlBDLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmUuXG4gICAqL1xuICByZWFkb25seSBnYXRld2F5RW5kcG9pbnRzPzogeyBbaWQ6IHN0cmluZ106IEdhdGV3YXlWcGNFbmRwb2ludE9wdGlvbnMgfVxuXG4gIC8qKlxuICAgKiBGbG93IGxvZ3MgdG8gYWRkIHRvIHRoaXMgVlBDLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGZsb3cgbG9ncy5cbiAgICovXG4gIHJlYWRvbmx5IGZsb3dMb2dzPzogeyBbaWQ6IHN0cmluZ106IEZsb3dMb2dPcHRpb25zIH1cbn1cblxuLyoqXG4gKiBUaGUgZGVmYXVsdCB0ZW5hbmN5IG9mIGluc3RhbmNlcyBsYXVuY2hlZCBpbnRvIHRoZSBWUEMuXG4gKi9cbmV4cG9ydCBlbnVtIERlZmF1bHRJbnN0YW5jZVRlbmFuY3kge1xuICAvKipcbiAgICogSW5zdGFuY2VzIGNhbiBiZSBsYXVuY2hlZCB3aXRoIGFueSB0ZW5hbmN5LlxuICAgKi9cbiAgREVGQVVMVCA9ICdkZWZhdWx0JyxcblxuICAvKipcbiAgICogQW55IGluc3RhbmNlIGxhdW5jaGVkIGludG8gdGhlIFZQQyBhdXRvbWF0aWNhbGx5IGhhcyBkZWRpY2F0ZWQgdGVuYW5jeSwgdW5sZXNzIHlvdSBsYXVuY2ggaXQgd2l0aCB0aGUgZGVmYXVsdCB0ZW5hbmN5LlxuICAgKi9cbiAgREVESUNBVEVEID0gJ2RlZGljYXRlZCdcbn1cblxuLyoqXG4gKiBTcGVjaWZ5IGNvbmZpZ3VyYXRpb24gcGFyYW1ldGVycyBmb3IgYSBzaW5nbGUgc3VibmV0IGdyb3VwIGluIGEgVlBDLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFN1Ym5ldENvbmZpZ3VyYXRpb24ge1xuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBsZWFkaW5nIDEgYml0cyBpbiB0aGUgcm91dGluZyBtYXNrLlxuICAgKlxuICAgKiBUaGUgbnVtYmVyIG9mIGF2YWlsYWJsZSBJUCBhZGRyZXNzZXMgaW4gZWFjaCBzdWJuZXQgb2YgdGhpcyBncm91cFxuICAgKiB3aWxsIGJlIGVxdWFsIHRvIGAyXigzMiAtIGNpZHJNYXNrKSAtIDJgLlxuICAgKlxuICAgKiBWYWxpZCB2YWx1ZXMgYXJlIGAxNi0tMjhgLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEF2YWlsYWJsZSBJUCBzcGFjZSBpcyBldmVubHkgZGl2aWRlZCBhY3Jvc3Mgc3VibmV0cy5cbiAgICovXG4gIHJlYWRvbmx5IGNpZHJNYXNrPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgdHlwZSBvZiBTdWJuZXQgdG8gY29uZmlndXJlLlxuICAgKlxuICAgKiBUaGUgU3VibmV0IHR5cGUgd2lsbCBjb250cm9sIHRoZSBhYmlsaXR5IHRvIHJvdXRlIGFuZCBjb25uZWN0IHRvIHRoZVxuICAgKiBJbnRlcm5ldC5cbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGU7XG5cbiAgLyoqXG4gICAqIExvZ2ljYWwgbmFtZSBmb3IgdGhlIHN1Ym5ldCBncm91cC5cbiAgICpcbiAgICogVGhpcyBuYW1lIGNhbiBiZSB1c2VkIHdoZW4gc2VsZWN0aW5nIFZQQyBzdWJuZXRzIHRvIGRpc3Rpbmd1aXNoXG4gICAqIGJldHdlZW4gZGlmZmVyZW50IHN1Ym5ldCBncm91cHMgb2YgdGhlIHNhbWUgdHlwZS5cbiAgICovXG4gIHJlYWRvbmx5IG5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogQ29udHJvbHMgaWYgc3VibmV0IElQIHNwYWNlIG5lZWRzIHRvIGJlIHJlc2VydmVkLlxuICAgKlxuICAgKiBXaGVuIHRydWUsIHRoZSBJUCBzcGFjZSBmb3IgdGhlIHN1Ym5ldCBpcyByZXNlcnZlZCBidXQgbm8gYWN0dWFsXG4gICAqIHJlc291cmNlcyBhcmUgcHJvdmlzaW9uZWQuIFRoaXMgc3BhY2UgaXMgb25seSBkZXBlbmRlbnQgb24gdGhlXG4gICAqIG51bWJlciBvZiBhdmFpbGliaWxpdHkgem9uZXMgYW5kIG9uIGBjaWRyTWFza2AgLSBhbGwgb3RoZXIgc3VibmV0XG4gICAqIHByb3BlcnRpZXMgYXJlIGlnbm9yZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSByZXNlcnZlZD86IGJvb2xlYW47XG59XG5cbi8qKlxuICogRGVmaW5lIGFuIEFXUyBWaXJ0dWFsIFByaXZhdGUgQ2xvdWRcbiAqXG4gKiBTZWUgdGhlIHBhY2thZ2UtbGV2ZWwgZG9jdW1lbnRhdGlvbiBvZiB0aGlzIHBhY2thZ2UgZm9yIGFuIG92ZXJ2aWV3XG4gKiBvZiB0aGUgdmFyaW91cyBkaW1lbnNpb25zIGluIHdoaWNoIHlvdSBjYW4gY29uZmlndXJlIHlvdXIgVlBDLlxuICpcbiAqIEZvciBleGFtcGxlOlxuICpcbiAqIGBgYHRzXG4gKiBjb25zdCB2cGMgPSBuZXcgZWMyLlZwYyh0aGlzLCAnVGhlVlBDJywge1xuICogICBjaWRyOiBcIjEwLjAuMC4wLzE2XCJcbiAqIH0pXG4gKlxuICogLy8gSXRlcmF0ZSB0aGUgcHJpdmF0ZSBzdWJuZXRzXG4gKiBjb25zdCBzZWxlY3Rpb24gPSB2cGMuc2VsZWN0U3VibmV0cyh7XG4gKiAgIHN1Ym5ldFR5cGU6IGVjMi5TdWJuZXRUeXBlLlBSSVZBVEVcbiAqIH0pO1xuICpcbiAqIGZvciAoY29uc3Qgc3VibmV0IG9mIHNlbGVjdGlvbi5zdWJuZXRzKSB7XG4gKiAgIC8vIC4uLlxuICogfVxuICogYGBgXG4gKlxuICogQHJlc291cmNlIEFXUzo6RUMyOjpWUENcbiAqL1xuZXhwb3J0IGNsYXNzIFZwYyBleHRlbmRzIFZwY0Jhc2Uge1xuICAvKipcbiAgICogVGhlIGRlZmF1bHQgQ0lEUiByYW5nZSB1c2VkIHdoZW4gY3JlYXRpbmcgVlBDcy5cbiAgICogVGhpcyBjYW4gYmUgb3ZlcnJpZGRlbiB1c2luZyBWcGNQcm9wcyB3aGVuIGNyZWF0aW5nIGEgVlBDTmV0d29yayByZXNvdXJjZS5cbiAgICogZS5nLiBuZXcgVnBjUmVzb3VyY2UodGhpcywgeyBjaWRyOiAnMTkyLjE2OC4wLjAuLzE2JyB9KVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBERUZBVUxUX0NJRFJfUkFOR0U6IHN0cmluZyA9ICcxMC4wLjAuMC8xNic7XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IHN1Ym5ldCBjb25maWd1cmF0aW9uXG4gICAqXG4gICAqIDEgUHVibGljIGFuZCAxIFByaXZhdGUgc3VibmV0IHBlciBBWiBldmVubHkgc3BsaXRcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgREVGQVVMVF9TVUJORVRTOiBTdWJuZXRDb25maWd1cmF0aW9uW10gPSBbXG4gICAge1xuICAgICAgc3VibmV0VHlwZTogU3VibmV0VHlwZS5QVUJMSUMsXG4gICAgICBuYW1lOiBkZWZhdWx0U3VibmV0TmFtZShTdWJuZXRUeXBlLlBVQkxJQyksXG4gICAgfSxcbiAgICB7XG4gICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBSSVZBVEUsXG4gICAgICBuYW1lOiBkZWZhdWx0U3VibmV0TmFtZShTdWJuZXRUeXBlLlBSSVZBVEUpLFxuICAgIH0sXG4gIF07XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IHN1Ym5ldCBjb25maWd1cmF0aW9uIGlmIG5hdEdhdGV3YXlzIHNwZWNpZmllZCB0byBiZSAwXG4gICAqXG4gICAqIDEgUHVibGljIGFuZCAxIElzb2xhdGVkIFN1Ym5ldCBwZXIgQVogZXZlbmx5IHNwbGl0XG4gICAqL1xuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IERFRkFVTFRfU1VCTkVUU19OT19OQVQ6IFN1Ym5ldENvbmZpZ3VyYXRpb25bXSA9IFtcbiAgICB7XG4gICAgICBzdWJuZXRUeXBlOiBTdWJuZXRUeXBlLlBVQkxJQyxcbiAgICAgIG5hbWU6IGRlZmF1bHRTdWJuZXROYW1lKFN1Ym5ldFR5cGUuUFVCTElDKSxcbiAgICB9LFxuICAgIHtcbiAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuSVNPTEFURUQsXG4gICAgICBuYW1lOiBkZWZhdWx0U3VibmV0TmFtZShTdWJuZXRUeXBlLklTT0xBVEVEKSxcbiAgICB9LFxuICBdO1xuXG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXhwb3J0ZWQgVlBDXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21WcGNBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBWcGNBdHRyaWJ1dGVzKTogSVZwYyB7XG4gICAgcmV0dXJuIG5ldyBJbXBvcnRlZFZwYyhzY29wZSwgaWQsIGF0dHJzLCBmYWxzZSk7XG4gIH1cblxuICAvKipcbiAgICogSW1wb3J0IGFuIGV4aXN0aW5nIFZQQyBmcm9tIGJ5IHF1ZXJ5aW5nIHRoZSBBV1MgZW52aXJvbm1lbnQgdGhpcyBzdGFjayBpcyBkZXBsb3llZCB0by5cbiAgICpcbiAgICogVGhpcyBmdW5jdGlvbiBvbmx5IG5lZWRzIHRvIGJlIHVzZWQgdG8gdXNlIFZQQ3Mgbm90IGRlZmluZWQgaW4geW91ciBDREtcbiAgICogYXBwbGljYXRpb24uIElmIHlvdSBhcmUgbG9va2luZyB0byBzaGFyZSBhIFZQQyBiZXR3ZWVuIHN0YWNrcywgeW91IGNhblxuICAgKiBwYXNzIHRoZSBgVnBjYCBvYmplY3QgYmV0d2VlbiBzdGFja3MgYW5kIHVzZSBpdCBhcyBub3JtYWwuXG4gICAqXG4gICAqIENhbGxpbmcgdGhpcyBtZXRob2Qgd2lsbCBsZWFkIHRvIGEgbG9va3VwIHdoZW4gdGhlIENESyBDTEkgaXMgZXhlY3V0ZWQuXG4gICAqIFlvdSBjYW4gdGhlcmVmb3JlIG5vdCB1c2UgYW55IHZhbHVlcyB0aGF0IHdpbGwgb25seSBiZSBhdmFpbGFibGUgYXRcbiAgICogQ2xvdWRGb3JtYXRpb24gZXhlY3V0aW9uIHRpbWUgKGkuZS4sIFRva2VucykuXG4gICAqXG4gICAqIFRoZSBWUEMgaW5mb3JtYXRpb24gd2lsbCBiZSBjYWNoZWQgaW4gYGNkay5jb250ZXh0Lmpzb25gIGFuZCB0aGUgc2FtZSBWUENcbiAgICogd2lsbCBiZSB1c2VkIG9uIGZ1dHVyZSBydW5zLiBUbyByZWZyZXNoIHRoZSBsb29rdXAsIHlvdSB3aWxsIGhhdmUgdG9cbiAgICogZXZpY3QgdGhlIHZhbHVlIGZyb20gdGhlIGNhY2hlIHVzaW5nIHRoZSBgY2RrIGNvbnRleHRgIGNvbW1hbmQuIFNlZVxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY2RrL2xhdGVzdC9ndWlkZS9jb250ZXh0Lmh0bWwgZm9yIG1vcmUgaW5mb3JtYXRpb24uXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Mb29rdXAoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgb3B0aW9uczogVnBjTG9va3VwT3B0aW9ucyk6IElWcGMge1xuICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQob3B0aW9ucy52cGNJZClcbiAgICAgIHx8IFRva2VuLmlzVW5yZXNvbHZlZChvcHRpb25zLnZwY05hbWUpXG4gICAgICB8fCBPYmplY3QudmFsdWVzKG9wdGlvbnMudGFncyB8fCB7fSkuc29tZShUb2tlbi5pc1VucmVzb2x2ZWQpXG4gICAgICB8fCBPYmplY3Qua2V5cyhvcHRpb25zLnRhZ3MgfHwge30pLnNvbWUoVG9rZW4uaXNVbnJlc29sdmVkKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBbGwgYXJndW1lbnRzIHRvIFZwYy5mcm9tTG9va3VwKCkgbXVzdCBiZSBjb25jcmV0ZSAobm8gVG9rZW5zKScpO1xuICAgIH1cblxuICAgIGNvbnN0IGZpbHRlcjoge1trZXk6IHN0cmluZ106IHN0cmluZ30gPSBtYWtlVGFnRmlsdGVyKG9wdGlvbnMudGFncyk7XG5cbiAgICAvLyBXZSBnaXZlIHNwZWNpYWwgdHJlYXRtZW50IHRvIHNvbWUgdGFnc1xuICAgIGlmIChvcHRpb25zLnZwY0lkKSB7IGZpbHRlclsndnBjLWlkJ10gPSBvcHRpb25zLnZwY0lkOyB9XG4gICAgaWYgKG9wdGlvbnMudnBjTmFtZSkgeyBmaWx0ZXJbJ3RhZzpOYW1lJ10gPSBvcHRpb25zLnZwY05hbWU7IH1cbiAgICBpZiAob3B0aW9ucy5pc0RlZmF1bHQgIT09IHVuZGVmaW5lZCkge1xuICAgICAgZmlsdGVyLmlzRGVmYXVsdCA9IG9wdGlvbnMuaXNEZWZhdWx0ID8gJ3RydWUnIDogJ2ZhbHNlJztcbiAgICB9XG5cbiAgICBjb25zdCBhdHRyaWJ1dGVzOiBjeGFwaS5WcGNDb250ZXh0UmVzcG9uc2UgPSBDb250ZXh0UHJvdmlkZXIuZ2V0VmFsdWUoc2NvcGUsIHtcbiAgICAgIHByb3ZpZGVyOiBjeHNjaGVtYS5Db250ZXh0UHJvdmlkZXIuVlBDX1BST1ZJREVSLFxuICAgICAgcHJvcHM6IHtcbiAgICAgICAgZmlsdGVyLFxuICAgICAgICByZXR1cm5Bc3ltbWV0cmljU3VibmV0czogdHJ1ZSxcbiAgICAgICAgc3VibmV0R3JvdXBOYW1lVGFnOiBvcHRpb25zLnN1Ym5ldEdyb3VwTmFtZVRhZyxcbiAgICAgIH0gYXMgY3hzY2hlbWEuVnBjQ29udGV4dFF1ZXJ5LFxuICAgICAgZHVtbXlWYWx1ZTogdW5kZWZpbmVkLFxuICAgIH0pLnZhbHVlO1xuXG4gICAgcmV0dXJuIG5ldyBMb29rZWRVcFZwYyhzY29wZSwgaWQsIGF0dHJpYnV0ZXMgfHwgRFVNTVlfVlBDX1BST1BTLCBhdHRyaWJ1dGVzID09PSB1bmRlZmluZWQpO1xuXG4gICAgLyoqXG4gICAgICogUHJlZml4ZXMgYWxsIGtleXMgaW4gdGhlIGFyZ3VtZW50IHdpdGggYHRhZzpgLmBcbiAgICAgKi9cbiAgICBmdW5jdGlvbiBtYWtlVGFnRmlsdGVyKHRhZ3M6IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9IHwgdW5kZWZpbmVkKTogeyBbbmFtZTogc3RyaW5nXTogc3RyaW5nIH0ge1xuICAgICAgY29uc3QgcmVzdWx0OiB7IFtuYW1lOiBzdHJpbmddOiBzdHJpbmcgfSA9IHt9O1xuICAgICAgZm9yIChjb25zdCBbbmFtZSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKHRhZ3MgfHwge30pKSB7XG4gICAgICAgIHJlc3VsdFtgdGFnOiR7bmFtZX1gXSA9IHZhbHVlO1xuICAgICAgfVxuICAgICAgcmV0dXJuIHJlc3VsdDtcbiAgICB9XG4gIH1cblxuICAvKipcbiAgICogSWRlbnRpZmllciBmb3IgdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB2cGNJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdnBjQ2lkckJsb2NrOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB2cGNEZWZhdWx0TmV0d29ya0FjbDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgdnBjQ2lkckJsb2NrQXNzb2NpYXRpb25zOiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHZwY0RlZmF1bHRTZWN1cml0eUdyb3VwOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSB2cGNJcHY2Q2lkckJsb2Nrczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgcHVibGljIHN1Ym5ldHMgaW4gdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwdWJsaWNTdWJuZXRzOiBJU3VibmV0W10gPSBbXTtcblxuICAvKipcbiAgICogTGlzdCBvZiBwcml2YXRlIHN1Ym5ldHMgaW4gdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwcml2YXRlU3VibmV0czogSVN1Ym5ldFtdID0gW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgaXNvbGF0ZWQgc3VibmV0cyBpbiB0aGlzIFZQQ1xuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGlzb2xhdGVkU3VibmV0czogSVN1Ym5ldFtdID0gW107XG5cbiAgLyoqXG4gICAqIEFacyBmb3IgdGhpcyBWUENcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhdmFpbGFiaWxpdHlab25lczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIEludGVybmV0IEdhdGV3YXkgZm9yIHRoZSBWUEMuIE5vdGUgdGhhdCBpbiBjYXNlIHRoZSBWUEMgaXMgY29uZmlndXJlZCBvbmx5XG4gICAqIHdpdGggSVNPTEFURUQgc3VibmV0cywgdGhpcyBhdHRyaWJ1dGUgd2lsbCBiZSBgdW5kZWZpbmVkYC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBpbnRlcm5ldEdhdGV3YXlJZD86IHN0cmluZztcblxuICBwdWJsaWMgcmVhZG9ubHkgaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZDogSURlcGVuZGFibGU7XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyBpZiBpbnN0YW5jZXMgbGF1bmNoZWQgaW4gdGhpcyBWUEMgd2lsbCBoYXZlIHB1YmxpYyBETlMgaG9zdG5hbWVzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRuc0hvc3RuYW1lc0VuYWJsZWQ6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyBpZiBETlMgc3VwcG9ydCBpcyBlbmFibGVkIGZvciB0aGlzIFZQQy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkbnNTdXBwb3J0RW5hYmxlZDogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIFZQQyByZXNvdXJjZVxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSByZXNvdXJjZTogQ2ZuVlBDO1xuXG4gIC8qKlxuICAgKiBUaGUgTmV0d29ya0J1aWxkZXJcbiAgICovXG4gIHByaXZhdGUgbmV0d29ya0J1aWxkZXI6IE5ldHdvcmtCdWlsZGVyO1xuXG4gIC8qKlxuICAgKiBTdWJuZXQgY29uZmlndXJhdGlvbnMgZm9yIHRoaXMgVlBDXG4gICAqL1xuICBwcml2YXRlIHN1Ym5ldENvbmZpZ3VyYXRpb246IFN1Ym5ldENvbmZpZ3VyYXRpb25bXSA9IFtdO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX2ludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQgPSBuZXcgQ29uY3JldGVEZXBlbmRhYmxlKCk7XG5cbiAgLyoqXG4gICAqIFZwYyBjcmVhdGVzIGEgVlBDIHRoYXQgc3BhbnMgYSB3aG9sZSByZWdpb24uXG4gICAqIEl0IHdpbGwgYXV0b21hdGljYWxseSBkaXZpZGUgdGhlIHByb3ZpZGVkIFZQQyBDSURSIHJhbmdlLCBhbmQgY3JlYXRlIHB1YmxpYyBhbmQgcHJpdmF0ZSBzdWJuZXRzIHBlciBBdmFpbGFiaWxpdHkgWm9uZS5cbiAgICogTmV0d29yayByb3V0aW5nIGZvciB0aGUgcHVibGljIHN1Ym5ldHMgd2lsbCBiZSBjb25maWd1cmVkIHRvIGFsbG93IG91dGJvdW5kIGFjY2VzcyBkaXJlY3RseSB2aWEgYW4gSW50ZXJuZXQgR2F0ZXdheS5cbiAgICogTmV0d29yayByb3V0aW5nIGZvciB0aGUgcHJpdmF0ZSBzdWJuZXRzIHdpbGwgYmUgY29uZmlndXJlZCB0byBhbGxvdyBvdXRib3VuZCBhY2Nlc3MgdmlhIGEgc2V0IG9mIHJlc2lsaWVudCBOQVQgR2F0ZXdheXMgKG9uZSBwZXIgQVopLlxuICAgKi9cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFZwY1Byb3BzID0ge30pIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcblxuICAgIC8vIENhbid0IGhhdmUgZW5hYmxlZERuc0hvc3RuYW1lcyB3aXRob3V0IGVuYWJsZURuc1N1cHBvcnRcbiAgICBpZiAocHJvcHMuZW5hYmxlRG5zSG9zdG5hbWVzICYmICFwcm9wcy5lbmFibGVEbnNTdXBwb3J0KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1RvIHVzZSBETlMgSG9zdG5hbWVzLCBETlMgU3VwcG9ydCBtdXN0IGJlIGVuYWJsZWQsIGhvd2V2ZXIsIGl0IHdhcyBleHBsaWNpdGx5IGRpc2FibGVkLicpO1xuICAgIH1cblxuICAgIGNvbnN0IGNpZHJCbG9jayA9IGlmVW5kZWZpbmVkKHByb3BzLmNpZHIsIFZwYy5ERUZBVUxUX0NJRFJfUkFOR0UpO1xuICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQoY2lkckJsb2NrKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdcXCdjaWRyXFwnIHByb3BlcnR5IG11c3QgYmUgYSBjb25jcmV0ZSBDSURSIHN0cmluZywgZ290IGEgVG9rZW4gKHdlIG5lZWQgdG8gcGFyc2UgaXQgZm9yIGF1dG9tYXRpYyBzdWJkaXZpc2lvbiknKTtcbiAgICB9XG5cbiAgICB0aGlzLm5ldHdvcmtCdWlsZGVyID0gbmV3IE5ldHdvcmtCdWlsZGVyKGNpZHJCbG9jayk7XG5cbiAgICB0aGlzLmRuc0hvc3RuYW1lc0VuYWJsZWQgPSBwcm9wcy5lbmFibGVEbnNIb3N0bmFtZXMgPT0gbnVsbCA/IHRydWUgOiBwcm9wcy5lbmFibGVEbnNIb3N0bmFtZXM7XG4gICAgdGhpcy5kbnNTdXBwb3J0RW5hYmxlZCA9IHByb3BzLmVuYWJsZURuc1N1cHBvcnQgPT0gbnVsbCA/IHRydWUgOiBwcm9wcy5lbmFibGVEbnNTdXBwb3J0O1xuICAgIGNvbnN0IGluc3RhbmNlVGVuYW5jeSA9IHByb3BzLmRlZmF1bHRJbnN0YW5jZVRlbmFuY3kgfHwgJ2RlZmF1bHQnO1xuICAgIHRoaXMuaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZCA9IHRoaXMuX2ludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQ7XG5cbiAgICAvLyBEZWZpbmUgYSBWUEMgdXNpbmcgdGhlIHByb3ZpZGVkIENJRFIgcmFuZ2VcbiAgICB0aGlzLnJlc291cmNlID0gbmV3IENmblZQQyh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBjaWRyQmxvY2ssXG4gICAgICBlbmFibGVEbnNIb3N0bmFtZXM6IHRoaXMuZG5zSG9zdG5hbWVzRW5hYmxlZCxcbiAgICAgIGVuYWJsZURuc1N1cHBvcnQ6IHRoaXMuZG5zU3VwcG9ydEVuYWJsZWQsXG4gICAgICBpbnN0YW5jZVRlbmFuY3ksXG4gICAgfSk7XG5cbiAgICB0aGlzLnZwY0RlZmF1bHROZXR3b3JrQWNsID0gdGhpcy5yZXNvdXJjZS5hdHRyRGVmYXVsdE5ldHdvcmtBY2w7XG4gICAgdGhpcy52cGNDaWRyQmxvY2tBc3NvY2lhdGlvbnMgPSB0aGlzLnJlc291cmNlLmF0dHJDaWRyQmxvY2tBc3NvY2lhdGlvbnM7XG4gICAgdGhpcy52cGNDaWRyQmxvY2sgPSB0aGlzLnJlc291cmNlLmF0dHJDaWRyQmxvY2s7XG4gICAgdGhpcy52cGNEZWZhdWx0U2VjdXJpdHlHcm91cCA9IHRoaXMucmVzb3VyY2UuYXR0ckRlZmF1bHRTZWN1cml0eUdyb3VwO1xuICAgIHRoaXMudnBjSXB2NkNpZHJCbG9ja3MgPSB0aGlzLnJlc291cmNlLmF0dHJJcHY2Q2lkckJsb2NrcztcblxuICAgIFRhZ3Mub2YodGhpcykuYWRkKE5BTUVfVEFHLCB0aGlzLm5vZGUucGF0aCk7XG5cbiAgICB0aGlzLmF2YWlsYWJpbGl0eVpvbmVzID0gc3RhY2suYXZhaWxhYmlsaXR5Wm9uZXM7XG5cbiAgICBjb25zdCBtYXhBWnMgPSBwcm9wcy5tYXhBenMgIT09IHVuZGVmaW5lZCA/IHByb3BzLm1heEF6cyA6IDM7XG4gICAgdGhpcy5hdmFpbGFiaWxpdHlab25lcyA9IHRoaXMuYXZhaWxhYmlsaXR5Wm9uZXMuc2xpY2UoMCwgbWF4QVpzKTtcblxuICAgIHRoaXMudnBjSWQgPSB0aGlzLnJlc291cmNlLnJlZjtcblxuICAgIGNvbnN0IGRlZmF1bHRTdWJuZXQgPSBwcm9wcy5uYXRHYXRld2F5cyA9PT0gMCA/IFZwYy5ERUZBVUxUX1NVQk5FVFNfTk9fTkFUIDogVnBjLkRFRkFVTFRfU1VCTkVUUztcbiAgICB0aGlzLnN1Ym5ldENvbmZpZ3VyYXRpb24gPSBpZlVuZGVmaW5lZChwcm9wcy5zdWJuZXRDb25maWd1cmF0aW9uLCBkZWZhdWx0U3VibmV0KTtcblxuICAgIGNvbnN0IG5hdEdhdGV3YXlQbGFjZW1lbnQgPSBwcm9wcy5uYXRHYXRld2F5U3VibmV0cyB8fCB7IHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFVCTElDIH07XG4gICAgY29uc3QgbmF0R2F0ZXdheUNvdW50ID0gZGV0ZXJtaW5lTmF0R2F0ZXdheUNvdW50KHByb3BzLm5hdEdhdGV3YXlzLCB0aGlzLnN1Ym5ldENvbmZpZ3VyYXRpb24sIHRoaXMuYXZhaWxhYmlsaXR5Wm9uZXMubGVuZ3RoKTtcblxuICAgIC8vIHN1Ym5ldENvbmZpZ3VyYXRpb24gbXVzdCBiZSBzZXQgYmVmb3JlIGNhbGxpbmcgY3JlYXRlU3VibmV0c1xuICAgIHRoaXMuY3JlYXRlU3VibmV0cygpO1xuXG4gICAgY29uc3QgYWxsb3dPdXRib3VuZCA9IHRoaXMuc3VibmV0Q29uZmlndXJhdGlvbi5maWx0ZXIoXG4gICAgICBzdWJuZXQgPT4gKHN1Ym5ldC5zdWJuZXRUeXBlICE9PSBTdWJuZXRUeXBlLklTT0xBVEVEKSkubGVuZ3RoID4gMDtcblxuICAgIC8vIENyZWF0ZSBhbiBJbnRlcm5ldCBHYXRld2F5IGFuZCBhdHRhY2ggaXQgaWYgbmVjZXNzYXJ5XG4gICAgaWYgKGFsbG93T3V0Ym91bmQpIHtcbiAgICAgIGNvbnN0IGlndyA9IG5ldyBDZm5JbnRlcm5ldEdhdGV3YXkodGhpcywgJ0lHVycsIHtcbiAgICAgIH0pO1xuXG4gICAgICB0aGlzLmludGVybmV0R2F0ZXdheUlkID0gaWd3LnJlZjtcblxuICAgICAgdGhpcy5faW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZC5hZGQoaWd3KTtcbiAgICAgIGNvbnN0IGF0dCA9IG5ldyBDZm5WUENHYXRld2F5QXR0YWNobWVudCh0aGlzLCAnVlBDR1cnLCB7XG4gICAgICAgIGludGVybmV0R2F0ZXdheUlkOiBpZ3cucmVmLFxuICAgICAgICB2cGNJZDogdGhpcy5yZXNvdXJjZS5yZWYsXG4gICAgICB9KTtcblxuICAgICAgKHRoaXMucHVibGljU3VibmV0cyBhcyBQdWJsaWNTdWJuZXRbXSkuZm9yRWFjaChwdWJsaWNTdWJuZXQgPT4ge1xuICAgICAgICBwdWJsaWNTdWJuZXQuYWRkRGVmYXVsdEludGVybmV0Um91dGUoaWd3LnJlZiwgYXR0KTtcbiAgICAgIH0pO1xuXG4gICAgICAvLyBpZiBnYXRld2F5cyBhcmUgbmVlZGVkIGNyZWF0ZSB0aGVtXG4gICAgICBpZiAobmF0R2F0ZXdheUNvdW50ID4gMCkge1xuICAgICAgICBjb25zdCBwcm92aWRlciA9IHByb3BzLm5hdEdhdGV3YXlQcm92aWRlciB8fCBOYXRQcm92aWRlci5nYXRld2F5KCk7XG4gICAgICAgIHRoaXMuY3JlYXRlTmF0R2F0ZXdheXMocHJvdmlkZXIsIG5hdEdhdGV3YXlDb3VudCwgbmF0R2F0ZXdheVBsYWNlbWVudCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnZwbkdhdGV3YXkgJiYgdGhpcy5wdWJsaWNTdWJuZXRzLmxlbmd0aCA9PT0gMCAmJiB0aGlzLnByaXZhdGVTdWJuZXRzLmxlbmd0aCA9PT0gMCAmJiB0aGlzLmlzb2xhdGVkU3VibmV0cy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2FuIG5vdCBlbmFibGUgdGhlIFZQTiBnYXRld2F5IHdoaWxlIHRoZSBWUEMgaGFzIG5vIHN1Ym5ldHMgYXQgYWxsJyk7XG4gICAgfVxuXG4gICAgaWYgKChwcm9wcy52cG5Db25uZWN0aW9ucyB8fCBwcm9wcy52cG5HYXRld2F5QXNuKSAmJiBwcm9wcy52cG5HYXRld2F5ID09PSBmYWxzZSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3Qgc3BlY2lmeSBgdnBuQ29ubmVjdGlvbnNgIG9yIGB2cG5HYXRld2F5QXNuYCB3aGVuIGB2cG5HYXRld2F5YCBpcyBzZXQgdG8gZmFsc2UuJyk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLnZwbkdhdGV3YXkgfHwgcHJvcHMudnBuQ29ubmVjdGlvbnMgfHwgcHJvcHMudnBuR2F0ZXdheUFzbikge1xuICAgICAgdGhpcy5lbmFibGVWcG5HYXRld2F5KHtcbiAgICAgICAgYW1hem9uU2lkZUFzbjogcHJvcHMudnBuR2F0ZXdheUFzbixcbiAgICAgICAgdHlwZTogVnBuQ29ubmVjdGlvblR5cGUuSVBTRUNfMSxcbiAgICAgICAgdnBuUm91dGVQcm9wYWdhdGlvbjogcHJvcHMudnBuUm91dGVQcm9wYWdhdGlvbixcbiAgICAgIH0pO1xuXG4gICAgICBjb25zdCB2cG5Db25uZWN0aW9ucyA9IHByb3BzLnZwbkNvbm5lY3Rpb25zIHx8IHt9O1xuICAgICAgZm9yIChjb25zdCBbY29ubmVjdGlvbklkLCBjb25uZWN0aW9uXSBvZiBPYmplY3QuZW50cmllcyh2cG5Db25uZWN0aW9ucykpIHtcbiAgICAgICAgdGhpcy5hZGRWcG5Db25uZWN0aW9uKGNvbm5lY3Rpb25JZCwgY29ubmVjdGlvbik7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gQWxsb3cgY3JlYXRpb24gb2YgZ2F0ZXdheSBlbmRwb2ludHMgb24gVlBDIGluc3RhbnRpYXRpb24gYXMgdGhvc2UgY2FuIGJlXG4gICAgLy8gaW1tZWRpYXRlbHkgZnVuY3Rpb25hbCB3aXRob3V0IGZ1cnRoZXIgY29uZmlndXJhdGlvbi4gVGhpcyBpcyBub3QgdGhlIGNhc2VcbiAgICAvLyBmb3IgaW50ZXJmYWNlIGVuZHBvaW50cyB3aGVyZSB0aGUgc2VjdXJpdHkgZ3JvdXAgbXVzdCBiZSBjb25maWd1cmVkLlxuICAgIGlmIChwcm9wcy5nYXRld2F5RW5kcG9pbnRzKSB7XG4gICAgICBjb25zdCBnYXRld2F5RW5kcG9pbnRzID0gcHJvcHMuZ2F0ZXdheUVuZHBvaW50cyB8fCB7fTtcbiAgICAgIGZvciAoY29uc3QgW2VuZHBvaW50SWQsIGVuZHBvaW50XSBvZiBPYmplY3QuZW50cmllcyhnYXRld2F5RW5kcG9pbnRzKSkge1xuICAgICAgICB0aGlzLmFkZEdhdGV3YXlFbmRwb2ludChlbmRwb2ludElkLCBlbmRwb2ludCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gQWRkIGZsb3cgbG9ncyB0byB0aGUgVlBDXG4gICAgaWYgKHByb3BzLmZsb3dMb2dzKSB7XG4gICAgICBjb25zdCBmbG93TG9ncyA9IHByb3BzLmZsb3dMb2dzIHx8IHt9O1xuICAgICAgZm9yIChjb25zdCBbZmxvd0xvZ0lkLCBmbG93TG9nXSBvZiBPYmplY3QuZW50cmllcyhmbG93TG9ncykpIHtcbiAgICAgICAgdGhpcy5hZGRGbG93TG9nKGZsb3dMb2dJZCwgZmxvd0xvZyk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgUzMgZ2F0ZXdheSBlbmRwb2ludCB0byB0aGlzIFZQQ1xuICAgKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYGFkZEdhdGV3YXlFbmRwb2ludCgpYCBpbnN0ZWFkXG4gICAqL1xuICBwdWJsaWMgYWRkUzNFbmRwb2ludChpZDogc3RyaW5nLCBzdWJuZXRzPzogU3VibmV0U2VsZWN0aW9uW10pOiBHYXRld2F5VnBjRW5kcG9pbnQge1xuICAgIHJldHVybiBuZXcgR2F0ZXdheVZwY0VuZHBvaW50KHRoaXMsIGlkLCB7XG4gICAgICBzZXJ2aWNlOiBHYXRld2F5VnBjRW5kcG9pbnRBd3NTZXJ2aWNlLlMzLFxuICAgICAgdnBjOiB0aGlzLFxuICAgICAgc3VibmV0cyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgbmV3IER5bmFtb0RCIGdhdGV3YXkgZW5kcG9pbnQgdG8gdGhpcyBWUENcbiAgICpcbiAgICogQGRlcHJlY2F0ZWQgdXNlIGBhZGRHYXRld2F5RW5kcG9pbnQoKWAgaW5zdGVhZFxuICAgKi9cbiAgcHVibGljIGFkZER5bmFtb0RiRW5kcG9pbnQoaWQ6IHN0cmluZywgc3VibmV0cz86IFN1Ym5ldFNlbGVjdGlvbltdKTogR2F0ZXdheVZwY0VuZHBvaW50IHtcbiAgICByZXR1cm4gbmV3IEdhdGV3YXlWcGNFbmRwb2ludCh0aGlzLCBpZCwge1xuICAgICAgc2VydmljZTogR2F0ZXdheVZwY0VuZHBvaW50QXdzU2VydmljZS5EWU5BTU9EQixcbiAgICAgIHZwYzogdGhpcyxcbiAgICAgIHN1Ym5ldHMsXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZU5hdEdhdGV3YXlzKHByb3ZpZGVyOiBOYXRQcm92aWRlciwgbmF0Q291bnQ6IG51bWJlciwgcGxhY2VtZW50OiBTdWJuZXRTZWxlY3Rpb24pOiB2b2lkIHtcbiAgICBjb25zdCBuYXRTdWJuZXRzOiBQdWJsaWNTdWJuZXRbXSA9IHRoaXMuc2VsZWN0U3VibmV0T2JqZWN0cyhwbGFjZW1lbnQpIGFzIFB1YmxpY1N1Ym5ldFtdO1xuICAgIGZvciAoY29uc3Qgc3ViIG9mIG5hdFN1Ym5ldHMpIHtcbiAgICAgIGlmICh0aGlzLnB1YmxpY1N1Ym5ldHMuaW5kZXhPZihzdWIpID09PSAtMSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYG5hdEdhdGV3YXlQbGFjZW1lbnQgJHtwbGFjZW1lbnR9IGNvbnRhaW5zIG5vbiBwdWJsaWMgc3VibmV0ICR7c3VifWApO1xuICAgICAgfVxuICAgIH1cblxuICAgIHByb3ZpZGVyLmNvbmZpZ3VyZU5hdCh7XG4gICAgICB2cGM6IHRoaXMsXG4gICAgICBuYXRTdWJuZXRzOiBuYXRTdWJuZXRzLnNsaWNlKDAsIG5hdENvdW50KSxcbiAgICAgIHByaXZhdGVTdWJuZXRzOiB0aGlzLnByaXZhdGVTdWJuZXRzIGFzIFByaXZhdGVTdWJuZXRbXSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBjcmVhdGVTdWJuZXRzIGNyZWF0ZXMgdGhlIHN1Ym5ldHMgc3BlY2lmaWVkIGJ5IHRoZSBzdWJuZXQgY29uZmlndXJhdGlvblxuICAgKiBhcnJheSBvciBjcmVhdGVzIHRoZSBgREVGQVVMVF9TVUJORVRTYCBjb25maWd1cmF0aW9uXG4gICAqL1xuICBwcml2YXRlIGNyZWF0ZVN1Ym5ldHMoKSB7XG4gICAgY29uc3QgcmVtYWluaW5nU3BhY2VTdWJuZXRzOiBTdWJuZXRDb25maWd1cmF0aW9uW10gPSBbXTtcblxuICAgIGZvciAoY29uc3Qgc3VibmV0IG9mIHRoaXMuc3VibmV0Q29uZmlndXJhdGlvbikge1xuICAgICAgaWYgKHN1Ym5ldC5jaWRyTWFzayA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIHJlbWFpbmluZ1NwYWNlU3VibmV0cy5wdXNoKHN1Ym5ldCk7XG4gICAgICAgIGNvbnRpbnVlO1xuICAgICAgfVxuICAgICAgdGhpcy5jcmVhdGVTdWJuZXRSZXNvdXJjZXMoc3VibmV0LCBzdWJuZXQuY2lkck1hc2spO1xuICAgIH1cblxuICAgIGNvbnN0IHRvdGFsUmVtYWluaW5nID0gcmVtYWluaW5nU3BhY2VTdWJuZXRzLmxlbmd0aCAqIHRoaXMuYXZhaWxhYmlsaXR5Wm9uZXMubGVuZ3RoO1xuICAgIGNvbnN0IGNpZHJNYXNrRm9yUmVtYWluZyA9IHRoaXMubmV0d29ya0J1aWxkZXIubWFza0ZvclJlbWFpbmluZ1N1Ym5ldHModG90YWxSZW1haW5pbmcpO1xuICAgIGZvciAoY29uc3Qgc3VibmV0IG9mIHJlbWFpbmluZ1NwYWNlU3VibmV0cykge1xuICAgICAgdGhpcy5jcmVhdGVTdWJuZXRSZXNvdXJjZXMoc3VibmV0LCBjaWRyTWFza0ZvclJlbWFpbmcpO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlU3VibmV0UmVzb3VyY2VzKHN1Ym5ldENvbmZpZzogU3VibmV0Q29uZmlndXJhdGlvbiwgY2lkck1hc2s6IG51bWJlcikge1xuICAgIHRoaXMuYXZhaWxhYmlsaXR5Wm9uZXMuZm9yRWFjaCgoem9uZSwgaW5kZXgpID0+IHtcbiAgICAgIGlmIChzdWJuZXRDb25maWcucmVzZXJ2ZWQgPT09IHRydWUpIHtcbiAgICAgICAgLy8gRm9yIHJlc2VydmVkIHN1Ym5ldHMsIGp1c3QgYWxsb2NhdGUgaXAgc3BhY2UgYnV0IGRvIG5vdCBjcmVhdGUgYW55IHJlc291cmNlc1xuICAgICAgICB0aGlzLm5ldHdvcmtCdWlsZGVyLmFkZFN1Ym5ldChjaWRyTWFzayk7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH1cblxuICAgICAgY29uc3QgbmFtZSA9IHN1Ym5ldElkKHN1Ym5ldENvbmZpZy5uYW1lLCBpbmRleCk7XG4gICAgICBjb25zdCBzdWJuZXRQcm9wczogU3VibmV0UHJvcHMgPSB7XG4gICAgICAgIGF2YWlsYWJpbGl0eVpvbmU6IHpvbmUsXG4gICAgICAgIHZwY0lkOiB0aGlzLnZwY0lkLFxuICAgICAgICBjaWRyQmxvY2s6IHRoaXMubmV0d29ya0J1aWxkZXIuYWRkU3VibmV0KGNpZHJNYXNrKSxcbiAgICAgICAgbWFwUHVibGljSXBPbkxhdW5jaDogKHN1Ym5ldENvbmZpZy5zdWJuZXRUeXBlID09PSBTdWJuZXRUeXBlLlBVQkxJQyksXG4gICAgICB9O1xuXG4gICAgICBsZXQgc3VibmV0OiBTdWJuZXQ7XG4gICAgICBzd2l0Y2ggKHN1Ym5ldENvbmZpZy5zdWJuZXRUeXBlKSB7XG4gICAgICAgIGNhc2UgU3VibmV0VHlwZS5QVUJMSUM6XG4gICAgICAgICAgY29uc3QgcHVibGljU3VibmV0ID0gbmV3IFB1YmxpY1N1Ym5ldCh0aGlzLCBuYW1lLCBzdWJuZXRQcm9wcyk7XG4gICAgICAgICAgdGhpcy5wdWJsaWNTdWJuZXRzLnB1c2gocHVibGljU3VibmV0KTtcbiAgICAgICAgICBzdWJuZXQgPSBwdWJsaWNTdWJuZXQ7XG4gICAgICAgICAgYnJlYWs7XG4gICAgICAgIGNhc2UgU3VibmV0VHlwZS5QUklWQVRFOlxuICAgICAgICAgIGNvbnN0IHByaXZhdGVTdWJuZXQgPSBuZXcgUHJpdmF0ZVN1Ym5ldCh0aGlzLCBuYW1lLCBzdWJuZXRQcm9wcyk7XG4gICAgICAgICAgdGhpcy5wcml2YXRlU3VibmV0cy5wdXNoKHByaXZhdGVTdWJuZXQpO1xuICAgICAgICAgIHN1Ym5ldCA9IHByaXZhdGVTdWJuZXQ7XG4gICAgICAgICAgYnJlYWs7XG4gICAgICAgIGNhc2UgU3VibmV0VHlwZS5JU09MQVRFRDpcbiAgICAgICAgICBjb25zdCBpc29sYXRlZFN1Ym5ldCA9IG5ldyBQcml2YXRlU3VibmV0KHRoaXMsIG5hbWUsIHN1Ym5ldFByb3BzKTtcbiAgICAgICAgICB0aGlzLmlzb2xhdGVkU3VibmV0cy5wdXNoKGlzb2xhdGVkU3VibmV0KTtcbiAgICAgICAgICBzdWJuZXQgPSBpc29sYXRlZFN1Ym5ldDtcbiAgICAgICAgICBicmVhaztcbiAgICAgICAgZGVmYXVsdDpcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVucmVjb2duaXplZCBzdWJuZXQgdHlwZTogJHtzdWJuZXRDb25maWcuc3VibmV0VHlwZX1gKTtcbiAgICAgIH1cblxuICAgICAgLy8gVGhlc2UgdmFsdWVzIHdpbGwgYmUgdXNlZCB0byByZWNvdmVyIHRoZSBjb25maWcgdXBvbiBwcm92aWRlciBpbXBvcnRcbiAgICAgIGNvbnN0IGluY2x1ZGVSZXNvdXJjZVR5cGVzID0gW0NmblN1Ym5ldC5DRk5fUkVTT1VSQ0VfVFlQRV9OQU1FXTtcbiAgICAgIFRhZ3Mub2Yoc3VibmV0KS5hZGQoU1VCTkVUTkFNRV9UQUcsIHN1Ym5ldENvbmZpZy5uYW1lLCB7IGluY2x1ZGVSZXNvdXJjZVR5cGVzIH0pO1xuICAgICAgVGFncy5vZihzdWJuZXQpLmFkZChTVUJORVRUWVBFX1RBRywgc3VibmV0VHlwZVRhZ1ZhbHVlKHN1Ym5ldENvbmZpZy5zdWJuZXRUeXBlKSwgeyBpbmNsdWRlUmVzb3VyY2VUeXBlcyB9KTtcbiAgICB9KTtcbiAgfVxufVxuXG5jb25zdCBTVUJORVRUWVBFX1RBRyA9ICdhd3MtY2RrOnN1Ym5ldC10eXBlJztcbmNvbnN0IFNVQk5FVE5BTUVfVEFHID0gJ2F3cy1jZGs6c3VibmV0LW5hbWUnO1xuXG5mdW5jdGlvbiBzdWJuZXRUeXBlVGFnVmFsdWUodHlwZTogU3VibmV0VHlwZSkge1xuICBzd2l0Y2ggKHR5cGUpIHtcbiAgICBjYXNlIFN1Ym5ldFR5cGUuUFVCTElDOiByZXR1cm4gJ1B1YmxpYyc7XG4gICAgY2FzZSBTdWJuZXRUeXBlLlBSSVZBVEU6IHJldHVybiAnUHJpdmF0ZSc7XG4gICAgY2FzZSBTdWJuZXRUeXBlLklTT0xBVEVEOiByZXR1cm4gJ0lzb2xhdGVkJztcbiAgfVxufVxuXG4vKipcbiAqIFNwZWNpZnkgY29uZmlndXJhdGlvbiBwYXJhbWV0ZXJzIGZvciBhIFZQQyBzdWJuZXRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTdWJuZXRQcm9wcyB7XG5cbiAgLyoqXG4gICAqIFRoZSBhdmFpbGFiaWxpdHkgem9uZSBmb3IgdGhlIHN1Ym5ldFxuICAgKi9cbiAgcmVhZG9ubHkgYXZhaWxhYmlsaXR5Wm9uZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgVlBDIHdoaWNoIHRoaXMgc3VibmV0IGlzIHBhcnQgb2ZcbiAgICovXG4gIHJlYWRvbmx5IHZwY0lkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBDSURSIG5vdGF0aW9uIGZvciB0aGlzIHN1Ym5ldFxuICAgKi9cbiAgcmVhZG9ubHkgY2lkckJsb2NrOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIENvbnRyb2xzIGlmIGEgcHVibGljIElQIGlzIGFzc29jaWF0ZWQgdG8gYW4gaW5zdGFuY2UgYXQgbGF1bmNoXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWUgaW4gU3VibmV0LlB1YmxpYywgZmFsc2UgaW4gU3VibmV0LlByaXZhdGUgb3IgU3VibmV0Lklzb2xhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgbWFwUHVibGljSXBPbkxhdW5jaD86IGJvb2xlYW47XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhIG5ldyBWUEMgc3VibmV0IHJlc291cmNlXG4gKlxuICogQHJlc291cmNlIEFXUzo6RUMyOjpTdWJuZXRcbiAqL1xuZXhwb3J0IGNsYXNzIFN1Ym5ldCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVN1Ym5ldCB7XG5cbiAgcHVibGljIHN0YXRpYyBpc1ZwY1N1Ym5ldCh4OiBhbnkpOiB4IGlzIFN1Ym5ldCB7XG4gICAgcmV0dXJuIFZQQ19TVUJORVRfU1lNQk9MIGluIHg7XG4gIH1cblxuICBwdWJsaWMgc3RhdGljIGZyb21TdWJuZXRBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBTdWJuZXRBdHRyaWJ1dGVzKTogSVN1Ym5ldCB7XG4gICAgcmV0dXJuIG5ldyBJbXBvcnRlZFN1Ym5ldChzY29wZSwgaWQsIGF0dHJzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbXBvcnQgZXhpc3Rpbmcgc3VibmV0IGZyb20gaWQuXG4gICAqL1xuICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25vLXNoYWRvd1xuICBwdWJsaWMgc3RhdGljIGZyb21TdWJuZXRJZChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBzdWJuZXRJZDogc3RyaW5nKTogSVN1Ym5ldCB7XG4gICAgcmV0dXJuIHRoaXMuZnJvbVN1Ym5ldEF0dHJpYnV0ZXMoc2NvcGUsIGlkLCB7IHN1Ym5ldElkIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBBdmFpbGFiaWxpdHkgWm9uZSB0aGUgc3VibmV0IGlzIGxvY2F0ZWQgaW5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhdmFpbGFiaWxpdHlab25lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBpcHY0Q2lkckJsb2NrOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBzdWJuZXRJZCBmb3IgdGhpcyBwYXJ0aWN1bGFyIHN1Ym5ldFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHN1Ym5ldElkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzdWJuZXRWcGNJZDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3VibmV0QXZhaWxhYmlsaXR5Wm9uZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3VibmV0SXB2NkNpZHJCbG9ja3M6IHN0cmluZ1tdO1xuXG4gIC8qKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgc3VibmV0TmV0d29ya0FjbEFzc29jaWF0aW9uSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogUGFydHMgb2YgdGhpcyBWUEMgc3VibmV0XG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGVwZW5kZW5jeUVsZW1lbnRzOiBJRGVwZW5kYWJsZVtdID0gW107XG5cbiAgLyoqXG4gICAqIFRoZSByb3V0ZVRhYmxlSWQgYXR0YWNoZWQgdG8gdGhpcyBzdWJuZXQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcm91dGVUYWJsZTogSVJvdXRlVGFibGU7XG5cbiAgcHVibGljIHJlYWRvbmx5IGludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQ6IElEZXBlbmRhYmxlO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgX2ludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQgPSBuZXcgQ29uY3JldGVEZXBlbmRhYmxlKCk7XG5cbiAgcHJpdmF0ZSBfbmV0d29ya0FjbDogSU5ldHdvcmtBY2w7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFN1Ym5ldFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIE9iamVjdC5kZWZpbmVQcm9wZXJ0eSh0aGlzLCBWUENfU1VCTkVUX1NZTUJPTCwgeyB2YWx1ZTogdHJ1ZSB9KTtcblxuICAgIFRhZ3Mub2YodGhpcykuYWRkKE5BTUVfVEFHLCB0aGlzLm5vZGUucGF0aCk7XG5cbiAgICB0aGlzLmF2YWlsYWJpbGl0eVpvbmUgPSBwcm9wcy5hdmFpbGFiaWxpdHlab25lO1xuICAgIHRoaXMuaXB2NENpZHJCbG9jayA9IHByb3BzLmNpZHJCbG9jaztcbiAgICBjb25zdCBzdWJuZXQgPSBuZXcgQ2ZuU3VibmV0KHRoaXMsICdTdWJuZXQnLCB7XG4gICAgICB2cGNJZDogcHJvcHMudnBjSWQsXG4gICAgICBjaWRyQmxvY2s6IHByb3BzLmNpZHJCbG9jayxcbiAgICAgIGF2YWlsYWJpbGl0eVpvbmU6IHByb3BzLmF2YWlsYWJpbGl0eVpvbmUsXG4gICAgICBtYXBQdWJsaWNJcE9uTGF1bmNoOiBwcm9wcy5tYXBQdWJsaWNJcE9uTGF1bmNoLFxuICAgIH0pO1xuICAgIHRoaXMuc3VibmV0SWQgPSBzdWJuZXQucmVmO1xuICAgIHRoaXMuc3VibmV0VnBjSWQgPSBzdWJuZXQuYXR0clZwY0lkO1xuICAgIHRoaXMuc3VibmV0QXZhaWxhYmlsaXR5Wm9uZSA9IHN1Ym5ldC5hdHRyQXZhaWxhYmlsaXR5Wm9uZTtcbiAgICB0aGlzLnN1Ym5ldElwdjZDaWRyQmxvY2tzID0gc3VibmV0LmF0dHJJcHY2Q2lkckJsb2NrcztcblxuICAgIC8vIHN1Ym5ldC5hdHRyTmV0d29ya0FjbEFzc29jaWF0aW9uSWQgaXMgdGhlIGRlZmF1bHQgQUNMIGFmdGVyIHRoZSBzdWJuZXRcbiAgICAvLyB3YXMganVzdCBjcmVhdGVkLiBIb3dldmVyLCB0aGUgQUNMIGNhbiBiZSByZXBsYWNlZCBhdCBhIGxhdGVyIHRpbWUuXG4gICAgdGhpcy5fbmV0d29ya0FjbCA9IE5ldHdvcmtBY2wuZnJvbU5ldHdvcmtBY2xJZCh0aGlzLCAnQWNsJywgc3VibmV0LmF0dHJOZXR3b3JrQWNsQXNzb2NpYXRpb25JZCk7XG4gICAgdGhpcy5zdWJuZXROZXR3b3JrQWNsQXNzb2NpYXRpb25JZCA9IExhenkuc3RyaW5nVmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLl9uZXR3b3JrQWNsLm5ldHdvcmtBY2xJZCB9KTtcbiAgICB0aGlzLm5vZGUuZGVmYXVsdENoaWxkID0gc3VibmV0O1xuXG4gICAgY29uc3QgdGFibGUgPSBuZXcgQ2ZuUm91dGVUYWJsZSh0aGlzLCAnUm91dGVUYWJsZScsIHtcbiAgICAgIHZwY0lkOiBwcm9wcy52cGNJZCxcbiAgICB9KTtcbiAgICB0aGlzLnJvdXRlVGFibGUgPSB7IHJvdXRlVGFibGVJZDogdGFibGUucmVmIH07XG5cbiAgICAvLyBBc3NvY2lhdGUgdGhlIHB1YmxpYyByb3V0ZSB0YWJsZSBmb3IgdGhpcyBzdWJuZXQsIHRvIHRoaXMgc3VibmV0XG4gICAgbmV3IENmblN1Ym5ldFJvdXRlVGFibGVBc3NvY2lhdGlvbih0aGlzLCAnUm91dGVUYWJsZUFzc29jaWF0aW9uJywge1xuICAgICAgc3VibmV0SWQ6IHRoaXMuc3VibmV0SWQsXG4gICAgICByb3V0ZVRhYmxlSWQ6IHRhYmxlLnJlZixcbiAgICB9KTtcblxuICAgIHRoaXMuaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZCA9IHRoaXMuX2ludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQ7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgZGVmYXVsdCByb3V0ZSB0aGF0IHBvaW50cyB0byBhIHBhc3NlZCBJR1csIHdpdGggYSBkZXBlbmRlbmN5XG4gICAqIG9uIHRoZSBJR1cncyBhdHRhY2htZW50IHRvIHRoZSBWUEMuXG4gICAqXG4gICAqIEBwYXJhbSBnYXRld2F5SWQgdGhlIGxvZ2ljYWwgSUQgKHJlZikgb2YgdGhlIGdhdGV3YXkgYXR0YWNoZWQgdG8geW91ciBWUENcbiAgICogQHBhcmFtIGdhdGV3YXlBdHRhY2htZW50IHRoZSBnYXRld2F5IGF0dGFjaG1lbnQgY29uc3RydWN0IHRvIGJlIGFkZGVkIGFzIGEgZGVwZW5kZW5jeVxuICAgKi9cbiAgcHVibGljIGFkZERlZmF1bHRJbnRlcm5ldFJvdXRlKGdhdGV3YXlJZDogc3RyaW5nLCBnYXRld2F5QXR0YWNobWVudDogSURlcGVuZGFibGUpIHtcbiAgICBjb25zdCByb3V0ZSA9IG5ldyBDZm5Sb3V0ZSh0aGlzLCAnRGVmYXVsdFJvdXRlJywge1xuICAgICAgcm91dGVUYWJsZUlkOiB0aGlzLnJvdXRlVGFibGUucm91dGVUYWJsZUlkLFxuICAgICAgZGVzdGluYXRpb25DaWRyQmxvY2s6ICcwLjAuMC4wLzAnLFxuICAgICAgZ2F0ZXdheUlkLFxuICAgIH0pO1xuICAgIHJvdXRlLm5vZGUuYWRkRGVwZW5kZW5jeShnYXRld2F5QXR0YWNobWVudCk7XG5cbiAgICAvLyBTaW5jZSB0aGUgJ3JvdXRlJyBkZXBlbmRzIG9uIHRoZSBnYXRld2F5IGF0dGFjaG1lbnQsIGp1c3RcbiAgICAvLyBkZXBlbmRpbmcgb24gdGhlIHJvdXRlIGlzIGVub3VnaC5cbiAgICB0aGlzLl9pbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkLmFkZChyb3V0ZSk7XG4gIH1cblxuICAvKipcbiAgICogTmV0d29yayBBQ0wgYXNzb2NpYXRlZCB3aXRoIHRoaXMgU3VibmV0XG4gICAqXG4gICAqIFVwb24gY3JlYXRpb24sIHRoaXMgaXMgdGhlIGRlZmF1bHQgQUNMIHdoaWNoIGFsbG93cyBhbGwgdHJhZmZpYywgZXhjZXB0XG4gICAqIGV4cGxpY2l0IERFTlkgZW50cmllcyB0aGF0IHlvdSBhZGQuXG4gICAqXG4gICAqIFlvdSBjYW4gcmVwbGFjZSBpdCB3aXRoIGEgY3VzdG9tIEFDTCB3aGljaCBkZW5pZXMgYWxsIHRyYWZmaWMgZXhjZXB0XG4gICAqIHRoZSBleHBsaWMgaXQgQUxMT1cgZW50cmllcyB0aGF0IHlvdSBhZGQgYnkgY3JlYXRpbmcgYSBgTmV0d29ya0FjbGBcbiAgICogb2JqZWN0IGFuZCBjYWxsaW5nIGBhc3NvY2lhdGVOZXR3b3JrQWNsKClgLlxuICAgKi9cbiAgcHVibGljIGdldCBuZXR3b3JrQWNsKCk6IElOZXR3b3JrQWNsIHtcbiAgICByZXR1cm4gdGhpcy5fbmV0d29ya0FjbDtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGFuIGVudHJ5IHRvIHRoaXMgc3VibmV0cyByb3V0ZSB0YWJsZSB0aGF0IHBvaW50cyB0byB0aGUgcGFzc2VkIE5BVEdhdHdheUlkXG4gICAqIEBwYXJhbSBuYXRHYXRld2F5SWQgVGhlIElEIG9mIHRoZSBOQVQgZ2F0ZXdheVxuICAgKi9cbiAgcHVibGljIGFkZERlZmF1bHROYXRSb3V0ZShuYXRHYXRld2F5SWQ6IHN0cmluZykge1xuICAgIHRoaXMuYWRkUm91dGUoJ0RlZmF1bHRSb3V0ZScsIHtcbiAgICAgIHJvdXRlclR5cGU6IFJvdXRlclR5cGUuTkFUX0dBVEVXQVksXG4gICAgICByb3V0ZXJJZDogbmF0R2F0ZXdheUlkLFxuICAgICAgZW5hYmxlc0ludGVybmV0Q29ubmVjdGl2aXR5OiB0cnVlLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gZW50cnkgdG8gdGhpcyBzdWJuZXRzIHJvdXRlIHRhYmxlXG4gICAqL1xuICBwdWJsaWMgYWRkUm91dGUoaWQ6IHN0cmluZywgb3B0aW9uczogQWRkUm91dGVPcHRpb25zKSB7XG4gICAgaWYgKG9wdGlvbnMuZGVzdGluYXRpb25DaWRyQmxvY2sgJiYgb3B0aW9ucy5kZXN0aW5hdGlvbklwdjZDaWRyQmxvY2spIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHNwZWNpZnkgYm90aCBcXCdkZXN0aW5hdGlvbkNpZHJCbG9ja1xcJyBhbmQgXFwnZGVzdGluYXRpb25JcHY2Q2lkckJsb2NrXFwnJyk7XG4gICAgfVxuXG4gICAgY29uc3Qgcm91dGUgPSBuZXcgQ2ZuUm91dGUodGhpcywgaWQsIHtcbiAgICAgIHJvdXRlVGFibGVJZDogdGhpcy5yb3V0ZVRhYmxlLnJvdXRlVGFibGVJZCxcbiAgICAgIGRlc3RpbmF0aW9uQ2lkckJsb2NrOiBvcHRpb25zLmRlc3RpbmF0aW9uQ2lkckJsb2NrIHx8IChvcHRpb25zLmRlc3RpbmF0aW9uSXB2NkNpZHJCbG9jayA9PT0gdW5kZWZpbmVkID8gJzAuMC4wLjAvMCcgOiB1bmRlZmluZWQpLFxuICAgICAgZGVzdGluYXRpb25JcHY2Q2lkckJsb2NrOiBvcHRpb25zLmRlc3RpbmF0aW9uSXB2NkNpZHJCbG9jayxcbiAgICAgIFtyb3V0ZXJUeXBlVG9Qcm9wTmFtZShvcHRpb25zLnJvdXRlclR5cGUpXTogb3B0aW9ucy5yb3V0ZXJJZCxcbiAgICB9KTtcblxuICAgIGlmIChvcHRpb25zLmVuYWJsZXNJbnRlcm5ldENvbm5lY3Rpdml0eSkge1xuICAgICAgdGhpcy5faW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZC5hZGQocm91dGUpO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBhc3NvY2lhdGVOZXR3b3JrQWNsKGlkOiBzdHJpbmcsIG5ldHdvcmtBY2w6IElOZXR3b3JrQWNsKSB7XG4gICAgdGhpcy5fbmV0d29ya0FjbCA9IG5ldHdvcmtBY2w7XG5cbiAgICBjb25zdCBzY29wZSA9IENvbnN0cnVjdC5pc0NvbnN0cnVjdChuZXR3b3JrQWNsKSA/IG5ldHdvcmtBY2wgOiB0aGlzO1xuICAgIGNvbnN0IG90aGVyID0gQ29uc3RydWN0LmlzQ29uc3RydWN0KG5ldHdvcmtBY2wpID8gdGhpcyA6IG5ldHdvcmtBY2w7XG4gICAgbmV3IFN1Ym5ldE5ldHdvcmtBY2xBc3NvY2lhdGlvbihzY29wZSwgaWQgKyBvdGhlci5ub2RlLnVuaXF1ZUlkLCB7XG4gICAgICBuZXR3b3JrQWNsLFxuICAgICAgc3VibmV0OiB0aGlzLFxuICAgIH0pO1xuICB9XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYWRkaW5nIGEgbmV3IHJvdXRlIHRvIGEgc3VibmV0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWRkUm91dGVPcHRpb25zIHtcbiAgLyoqXG4gICAqIElQdjQgcmFuZ2UgdGhpcyByb3V0ZSBhcHBsaWVzIHRvXG4gICAqXG4gICAqIEBkZWZhdWx0ICcwLjAuMC4wLzAnXG4gICAqL1xuICByZWFkb25seSBkZXN0aW5hdGlvbkNpZHJCbG9jaz86IHN0cmluZztcblxuICAvKipcbiAgICogSVB2NiByYW5nZSB0aGlzIHJvdXRlIGFwcGxpZXMgdG9cbiAgICpcbiAgICogQGRlZmF1bHQgLSBVc2VzIElQdjZcbiAgICovXG4gIHJlYWRvbmx5IGRlc3RpbmF0aW9uSXB2NkNpZHJCbG9jaz86IHN0cmluZztcblxuICAvKipcbiAgICogV2hhdCB0eXBlIG9mIHJvdXRlciB0byByb3V0ZSB0aGlzIHRyYWZmaWMgdG9cbiAgICovXG4gIHJlYWRvbmx5IHJvdXRlclR5cGU6IFJvdXRlclR5cGU7XG5cbiAgLyoqXG4gICAqIFRoZSBJRCBvZiB0aGUgcm91dGVyXG4gICAqXG4gICAqIENhbiBiZSBhbiBpbnN0YW5jZSBJRCwgZ2F0ZXdheSBJRCwgZXRjLCBkZXBlbmRpbmcgb24gdGhlIHJvdXRlciB0eXBlLlxuICAgKi9cbiAgcmVhZG9ubHkgcm91dGVySWQ6IHN0cmluZztcblxuICAvKipcbiAgICogV2hldGhlciB0aGlzIHJvdXRlIHdpbGwgZW5hYmxlIGludGVybmV0IGNvbm5lY3Rpdml0eVxuICAgKlxuICAgKiBJZiB0cnVlLCB0aGlzIHJvdXRlIHdpbGwgYmUgYWRkZWQgYmVmb3JlIGFueSBBV1MgcmVzb3VyY2VzIHRoYXQgZGVwZW5kXG4gICAqIG9uIGludGVybmV0IGNvbm5lY3Rpdml0eSBpbiB0aGUgVlBDIHdpbGwgYmUgY3JlYXRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZXNJbnRlcm5ldENvbm5lY3Rpdml0eT86IGJvb2xlYW47XG59XG5cbi8qKlxuICogVHlwZSBvZiByb3V0ZXIgdXNlZCBpbiByb3V0ZVxuICovXG5leHBvcnQgZW51bSBSb3V0ZXJUeXBlIHtcbiAgLyoqXG4gICAqIEVncmVzcy1vbmx5IEludGVybmV0IEdhdGV3YXlcbiAgICovXG4gIEVHUkVTU19PTkxZX0lOVEVSTkVUX0dBVEVXQVkgPSAnRWdyZXNzT25seUludGVybmV0R2F0ZXdheScsXG5cbiAgLyoqXG4gICAqIEludGVybmV0IEdhdGV3YXlcbiAgICovXG4gIEdBVEVXQVkgPSAnR2F0ZXdheScsXG5cbiAgLyoqXG4gICAqIEluc3RhbmNlXG4gICAqL1xuICBJTlNUQU5DRSA9ICdJbnN0YW5jZScsXG5cbiAgLyoqXG4gICAqIE5BVCBHYXRld2F5XG4gICAqL1xuICBOQVRfR0FURVdBWSA9ICdOYXRHYXRld2F5JyxcblxuICAvKipcbiAgICogTmV0d29yayBJbnRlcmZhY2VcbiAgICovXG4gIE5FVFdPUktfSU5URVJGQUNFID0gJ05ldHdvcmtJbnRlcmZhY2UnLFxuXG4gIC8qKlxuICAgKiBWUEMgcGVlcmluZyBjb25uZWN0aW9uXG4gICAqL1xuICBWUENfUEVFUklOR19DT05ORUNUSU9OID0gJ1ZwY1BlZXJpbmdDb25uZWN0aW9uJyxcbn1cblxuZnVuY3Rpb24gcm91dGVyVHlwZVRvUHJvcE5hbWUocm91dGVyVHlwZTogUm91dGVyVHlwZSkge1xuICByZXR1cm4gKHtcbiAgICBbUm91dGVyVHlwZS5FR1JFU1NfT05MWV9JTlRFUk5FVF9HQVRFV0FZXTogJ2VncmVzc09ubHlJbnRlcm5ldEdhdGV3YXlJZCcsXG4gICAgW1JvdXRlclR5cGUuR0FURVdBWV06ICdnYXRld2F5SWQnLFxuICAgIFtSb3V0ZXJUeXBlLklOU1RBTkNFXTogJ2luc3RhbmNlSWQnLFxuICAgIFtSb3V0ZXJUeXBlLk5BVF9HQVRFV0FZXTogJ25hdEdhdGV3YXlJZCcsXG4gICAgW1JvdXRlclR5cGUuTkVUV09SS19JTlRFUkZBQ0VdOiAnbmV0d29ya0ludGVyZmFjZUlkJyxcbiAgICBbUm91dGVyVHlwZS5WUENfUEVFUklOR19DT05ORUNUSU9OXTogJ3ZwY1BlZXJpbmdDb25uZWN0aW9uSWQnLFxuICB9KVtyb3V0ZXJUeXBlXTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBQdWJsaWNTdWJuZXRQcm9wcyBleHRlbmRzIFN1Ym5ldFByb3BzIHtcblxufVxuXG5leHBvcnQgaW50ZXJmYWNlIElQdWJsaWNTdWJuZXQgZXh0ZW5kcyBJU3VibmV0IHsgfVxuXG5leHBvcnQgaW50ZXJmYWNlIFB1YmxpY1N1Ym5ldEF0dHJpYnV0ZXMgZXh0ZW5kcyBTdWJuZXRBdHRyaWJ1dGVzIHsgfVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBwdWJsaWMgVlBDIHN1Ym5ldCByZXNvdXJjZVxuICovXG5leHBvcnQgY2xhc3MgUHVibGljU3VibmV0IGV4dGVuZHMgU3VibmV0IGltcGxlbWVudHMgSVB1YmxpY1N1Ym5ldCB7XG5cbiAgcHVibGljIHN0YXRpYyBmcm9tUHVibGljU3VibmV0QXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogUHVibGljU3VibmV0QXR0cmlidXRlcyk6IElQdWJsaWNTdWJuZXQge1xuICAgIHJldHVybiBuZXcgSW1wb3J0ZWRTdWJuZXQoc2NvcGUsIGlkLCBhdHRycyk7XG4gIH1cblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUHVibGljU3VibmV0UHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgbmV3IG1hbmFnZWQgTkFUIGdhdGV3YXkgYXR0YWNoZWQgdG8gdGhpcyBwdWJsaWMgc3VibmV0LlxuICAgKiBBbHNvIGFkZHMgdGhlIEVJUCBmb3IgdGhlIG1hbmFnZWQgTkFULlxuICAgKiBAcmV0dXJucyBBIHJlZiB0byB0aGUgdGhlIE5BVCBHYXRld2F5IElEXG4gICAqL1xuICBwdWJsaWMgYWRkTmF0R2F0ZXdheSgpIHtcbiAgICAvLyBDcmVhdGUgYSBOQVQgR2F0ZXdheSBpbiB0aGlzIHB1YmxpYyBzdWJuZXRcbiAgICBjb25zdCBuZ3cgPSBuZXcgQ2ZuTmF0R2F0ZXdheSh0aGlzLCAnTkFUR2F0ZXdheScsIHtcbiAgICAgIHN1Ym5ldElkOiB0aGlzLnN1Ym5ldElkLFxuICAgICAgYWxsb2NhdGlvbklkOiBuZXcgQ2ZuRUlQKHRoaXMsICdFSVAnLCB7XG4gICAgICAgIGRvbWFpbjogJ3ZwYycsXG4gICAgICB9KS5hdHRyQWxsb2NhdGlvbklkLFxuICAgIH0pO1xuICAgIHJldHVybiBuZ3c7XG4gIH1cbn1cblxuZXhwb3J0IGludGVyZmFjZSBQcml2YXRlU3VibmV0UHJvcHMgZXh0ZW5kcyBTdWJuZXRQcm9wcyB7XG5cbn1cblxuZXhwb3J0IGludGVyZmFjZSBJUHJpdmF0ZVN1Ym5ldCBleHRlbmRzIElTdWJuZXQgeyB9XG5cbmV4cG9ydCBpbnRlcmZhY2UgUHJpdmF0ZVN1Ym5ldEF0dHJpYnV0ZXMgZXh0ZW5kcyBTdWJuZXRBdHRyaWJ1dGVzIHsgfVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBwcml2YXRlIFZQQyBzdWJuZXQgcmVzb3VyY2VcbiAqL1xuZXhwb3J0IGNsYXNzIFByaXZhdGVTdWJuZXQgZXh0ZW5kcyBTdWJuZXQgaW1wbGVtZW50cyBJUHJpdmF0ZVN1Ym5ldCB7XG5cbiAgcHVibGljIHN0YXRpYyBmcm9tUHJpdmF0ZVN1Ym5ldEF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IFByaXZhdGVTdWJuZXRBdHRyaWJ1dGVzKTogSVByaXZhdGVTdWJuZXQge1xuICAgIHJldHVybiBuZXcgSW1wb3J0ZWRTdWJuZXQoc2NvcGUsIGlkLCBhdHRycyk7XG4gIH1cblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUHJpdmF0ZVN1Ym5ldFByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG4gIH1cbn1cblxuZnVuY3Rpb24gaWZVbmRlZmluZWQ8VD4odmFsdWU6IFQgfCB1bmRlZmluZWQsIGRlZmF1bHRWYWx1ZTogVCk6IFQge1xuICByZXR1cm4gdmFsdWUgIT09IHVuZGVmaW5lZCA/IHZhbHVlIDogZGVmYXVsdFZhbHVlO1xufVxuXG5jbGFzcyBJbXBvcnRlZFZwYyBleHRlbmRzIFZwY0Jhc2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgdnBjSWQ6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IHB1YmxpY1N1Ym5ldHM6IElTdWJuZXRbXTtcbiAgcHVibGljIHJlYWRvbmx5IHByaXZhdGVTdWJuZXRzOiBJU3VibmV0W107XG4gIHB1YmxpYyByZWFkb25seSBpc29sYXRlZFN1Ym5ldHM6IElTdWJuZXRbXTtcbiAgcHVibGljIHJlYWRvbmx5IGF2YWlsYWJpbGl0eVpvbmVzOiBzdHJpbmdbXTtcbiAgcHVibGljIHJlYWRvbmx5IGludGVybmV0Q29ubmVjdGl2aXR5RXN0YWJsaXNoZWQ6IElEZXBlbmRhYmxlID0gbmV3IENvbmNyZXRlRGVwZW5kYWJsZSgpO1xuICBwcml2YXRlIHJlYWRvbmx5IGNpZHI/OiBzdHJpbmcgfCB1bmRlZmluZWQ7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFZwY0F0dHJpYnV0ZXMsIGlzSW5jb21wbGV0ZTogYm9vbGVhbikge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLnZwY0lkID0gcHJvcHMudnBjSWQ7XG4gICAgdGhpcy5jaWRyID0gcHJvcHMudnBjQ2lkckJsb2NrO1xuICAgIHRoaXMuYXZhaWxhYmlsaXR5Wm9uZXMgPSBwcm9wcy5hdmFpbGFiaWxpdHlab25lcztcbiAgICB0aGlzLl92cG5HYXRld2F5SWQgPSBwcm9wcy52cG5HYXRld2F5SWQ7XG4gICAgdGhpcy5pbmNvbXBsZXRlU3VibmV0RGVmaW5pdGlvbiA9IGlzSW5jb21wbGV0ZTtcblxuICAgIC8qIGVzbGludC1kaXNhYmxlIG1heC1sZW4gKi9cbiAgICBjb25zdCBwdWIgPSBuZXcgSW1wb3J0U3VibmV0R3JvdXAocHJvcHMucHVibGljU3VibmV0SWRzLCBwcm9wcy5wdWJsaWNTdWJuZXROYW1lcywgcHJvcHMucHVibGljU3VibmV0Um91dGVUYWJsZUlkcywgU3VibmV0VHlwZS5QVUJMSUMsIHRoaXMuYXZhaWxhYmlsaXR5Wm9uZXMsICdwdWJsaWNTdWJuZXRJZHMnLCAncHVibGljU3VibmV0TmFtZXMnLCAncHVibGljU3VibmV0Um91dGVUYWJsZUlkcycpO1xuICAgIGNvbnN0IHByaXYgPSBuZXcgSW1wb3J0U3VibmV0R3JvdXAocHJvcHMucHJpdmF0ZVN1Ym5ldElkcywgcHJvcHMucHJpdmF0ZVN1Ym5ldE5hbWVzLCBwcm9wcy5wcml2YXRlU3VibmV0Um91dGVUYWJsZUlkcywgU3VibmV0VHlwZS5QUklWQVRFLCB0aGlzLmF2YWlsYWJpbGl0eVpvbmVzLCAncHJpdmF0ZVN1Ym5ldElkcycsICdwcml2YXRlU3VibmV0TmFtZXMnLCAncHJpdmF0ZVN1Ym5ldFJvdXRlVGFibGVJZHMnKTtcbiAgICBjb25zdCBpc28gPSBuZXcgSW1wb3J0U3VibmV0R3JvdXAocHJvcHMuaXNvbGF0ZWRTdWJuZXRJZHMsIHByb3BzLmlzb2xhdGVkU3VibmV0TmFtZXMsIHByb3BzLmlzb2xhdGVkU3VibmV0Um91dGVUYWJsZUlkcywgU3VibmV0VHlwZS5JU09MQVRFRCwgdGhpcy5hdmFpbGFiaWxpdHlab25lcywgJ2lzb2xhdGVkU3VibmV0SWRzJywgJ2lzb2xhdGVkU3VibmV0TmFtZXMnLCAnaXNvbGF0ZWRTdWJuZXRSb3V0ZVRhYmxlSWRzJyk7XG4gICAgLyogZXNsaW50LWVuYWJsZSBtYXgtbGVuICovXG5cbiAgICB0aGlzLnB1YmxpY1N1Ym5ldHMgPSBwdWIuaW1wb3J0KHRoaXMpO1xuICAgIHRoaXMucHJpdmF0ZVN1Ym5ldHMgPSBwcml2LmltcG9ydCh0aGlzKTtcbiAgICB0aGlzLmlzb2xhdGVkU3VibmV0cyA9IGlzby5pbXBvcnQodGhpcyk7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHZwY0NpZHJCbG9jaygpOiBzdHJpbmcge1xuICAgIGlmICh0aGlzLmNpZHIgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdDYW5ub3QgcGVyZm9ybSB0aGlzIG9wZXJhdGlvbjogXFwndnBjQ2lkckJsb2NrXFwnIHdhcyBub3Qgc3VwcGxpZWQgd2hlbiBjcmVhdGluZyB0aGlzIFZQQycpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5jaWRyO1xuICB9XG59XG5cbmNsYXNzIExvb2tlZFVwVnBjIGV4dGVuZHMgVnBjQmFzZSB7XG4gIHB1YmxpYyByZWFkb25seSB2cGNJZDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZDogSURlcGVuZGFibGUgPSBuZXcgQ29uY3JldGVEZXBlbmRhYmxlKCk7XG4gIHB1YmxpYyByZWFkb25seSBhdmFpbGFiaWxpdHlab25lczogc3RyaW5nW107XG4gIHB1YmxpYyByZWFkb25seSBwdWJsaWNTdWJuZXRzOiBJU3VibmV0W107XG4gIHB1YmxpYyByZWFkb25seSBwcml2YXRlU3VibmV0czogSVN1Ym5ldFtdO1xuICBwdWJsaWMgcmVhZG9ubHkgaXNvbGF0ZWRTdWJuZXRzOiBJU3VibmV0W107XG4gIHByaXZhdGUgcmVhZG9ubHkgY2lkcj86IHN0cmluZyB8IHVuZGVmaW5lZDtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogY3hhcGkuVnBjQ29udGV4dFJlc3BvbnNlLCBpc0luY29tcGxldGU6IGJvb2xlYW4pIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy52cGNJZCA9IHByb3BzLnZwY0lkO1xuICAgIHRoaXMuY2lkciA9IHByb3BzLnZwY0NpZHJCbG9jaztcbiAgICB0aGlzLl92cG5HYXRld2F5SWQgPSBwcm9wcy52cG5HYXRld2F5SWQ7XG4gICAgdGhpcy5pbmNvbXBsZXRlU3VibmV0RGVmaW5pdGlvbiA9IGlzSW5jb21wbGV0ZTtcblxuICAgIGNvbnN0IHN1Ym5ldEdyb3VwcyA9IHByb3BzLnN1Ym5ldEdyb3VwcyB8fCBbXTtcbiAgICBjb25zdCBhdmFpbGFiaWxpdHlab25lcyA9IEFycmF5LmZyb20obmV3IFNldDxzdHJpbmc+KGZsYXRNYXAoc3VibmV0R3JvdXBzLCBzdWJuZXRHcm91cCA9PiB7XG4gICAgICByZXR1cm4gc3VibmV0R3JvdXAuc3VibmV0cy5tYXAoc3VibmV0ID0+IHN1Ym5ldC5hdmFpbGFiaWxpdHlab25lKTtcbiAgICB9KSkpO1xuICAgIGF2YWlsYWJpbGl0eVpvbmVzLnNvcnQoKGF6MSwgYXoyKSA9PiBhejEubG9jYWxlQ29tcGFyZShhejIpKTtcbiAgICB0aGlzLmF2YWlsYWJpbGl0eVpvbmVzID0gYXZhaWxhYmlsaXR5Wm9uZXM7XG5cbiAgICB0aGlzLnB1YmxpY1N1Ym5ldHMgPSB0aGlzLmV4dHJhY3RTdWJuZXRzT2ZUeXBlKHN1Ym5ldEdyb3VwcywgY3hhcGkuVnBjU3VibmV0R3JvdXBUeXBlLlBVQkxJQyk7XG4gICAgdGhpcy5wcml2YXRlU3VibmV0cyA9IHRoaXMuZXh0cmFjdFN1Ym5ldHNPZlR5cGUoc3VibmV0R3JvdXBzLCBjeGFwaS5WcGNTdWJuZXRHcm91cFR5cGUuUFJJVkFURSk7XG4gICAgdGhpcy5pc29sYXRlZFN1Ym5ldHMgPSB0aGlzLmV4dHJhY3RTdWJuZXRzT2ZUeXBlKHN1Ym5ldEdyb3VwcywgY3hhcGkuVnBjU3VibmV0R3JvdXBUeXBlLklTT0xBVEVEKTtcbiAgfVxuXG4gIHB1YmxpYyBnZXQgdnBjQ2lkckJsb2NrKCk6IHN0cmluZyB7XG4gICAgaWYgKHRoaXMuY2lkciA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAvLyBWYWx1ZSBtaWdodCBiZSBjYWNoZWQgZnJvbSBhbiBvbGQgQ0xJIHZlcnNpb24sIHNvIGJ1bXBpbmcgdGhlIENYIEFQSSBwcm90b2NvbCB0b1xuICAgICAgLy8gZm9yY2UgdGhlIHZhbHVlIHRvIGV4aXN0IHdvdWxkIG5vdCBoYXZlIGhlbHBlZC5cbiAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHBlcmZvcm0gdGhpcyBvcGVyYXRpb246IFxcJ3ZwY0NpZHJCbG9ja1xcJyB3YXMgbm90IGZvdW5kIHdoZW4gbG9va2luZyB1cCB0aGlzIFZQQy4gVXNlIGEgbmV3ZXIgdmVyc2lvbiBvZiB0aGUgQ0RLIENMSSBhbmQgY2xlYXIgdGhlIG9sZCBjb250ZXh0IHZhbHVlLicpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5jaWRyO1xuICB9XG5cbiAgcHJpdmF0ZSBleHRyYWN0U3VibmV0c09mVHlwZShzdWJuZXRHcm91cHM6IGN4YXBpLlZwY1N1Ym5ldEdyb3VwW10sIHN1Ym5ldEdyb3VwVHlwZTogY3hhcGkuVnBjU3VibmV0R3JvdXBUeXBlKTogSVN1Ym5ldFtdIHtcbiAgICByZXR1cm4gZmxhdE1hcChzdWJuZXRHcm91cHMuZmlsdGVyKHN1Ym5ldEdyb3VwID0+IHN1Ym5ldEdyb3VwLnR5cGUgPT09IHN1Ym5ldEdyb3VwVHlwZSksXG4gICAgICBzdWJuZXRHcm91cCA9PiB0aGlzLnN1Ym5ldEdyb3VwVG9TdWJuZXRzKHN1Ym5ldEdyb3VwKSk7XG4gIH1cblxuICBwcml2YXRlIHN1Ym5ldEdyb3VwVG9TdWJuZXRzKHN1Ym5ldEdyb3VwOiBjeGFwaS5WcGNTdWJuZXRHcm91cCk6IElTdWJuZXRbXSB7XG4gICAgY29uc3QgcmV0ID0gbmV3IEFycmF5PElTdWJuZXQ+KCk7XG4gICAgZm9yIChsZXQgaSA9IDA7IGkgPCBzdWJuZXRHcm91cC5zdWJuZXRzLmxlbmd0aDsgaSsrKSB7XG4gICAgICBjb25zdCB2cGNTdWJuZXQgPSBzdWJuZXRHcm91cC5zdWJuZXRzW2ldO1xuICAgICAgcmV0LnB1c2goU3VibmV0LmZyb21TdWJuZXRBdHRyaWJ1dGVzKHRoaXMsIGAke3N1Ym5ldEdyb3VwLm5hbWV9U3VibmV0JHtpICsgMX1gLCB7XG4gICAgICAgIGF2YWlsYWJpbGl0eVpvbmU6IHZwY1N1Ym5ldC5hdmFpbGFiaWxpdHlab25lLFxuICAgICAgICBzdWJuZXRJZDogdnBjU3VibmV0LnN1Ym5ldElkLFxuICAgICAgICByb3V0ZVRhYmxlSWQ6IHZwY1N1Ym5ldC5yb3V0ZVRhYmxlSWQsXG4gICAgICB9KSk7XG4gICAgfVxuICAgIHJldHVybiByZXQ7XG4gIH1cbn1cblxuZnVuY3Rpb24gZmxhdE1hcDxULCBVPih4czogVFtdLCBmbjogKHg6IFQpID0+IFVbXSk6IFVbXSB7XG4gIGNvbnN0IHJldCA9IG5ldyBBcnJheTxVPigpO1xuICBmb3IgKGNvbnN0IHggb2YgeHMpIHtcbiAgICByZXQucHVzaCguLi5mbih4KSk7XG4gIH1cbiAgcmV0dXJuIHJldDtcbn1cblxuY2xhc3MgQ29tcG9zaXRlRGVwZW5kYWJsZSBpbXBsZW1lbnRzIElEZXBlbmRhYmxlIHtcbiAgcHJpdmF0ZSByZWFkb25seSBkZXBlbmRhYmxlcyA9IG5ldyBBcnJheTxJRGVwZW5kYWJsZT4oKTtcblxuICBjb25zdHJ1Y3RvcigpIHtcbiAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICBEZXBlbmRhYmxlVHJhaXQuaW1wbGVtZW50KHRoaXMsIHtcbiAgICAgIGdldCBkZXBlbmRlbmN5Um9vdHMoKSB7XG4gICAgICAgIGNvbnN0IHJldCA9IG5ldyBBcnJheTxJQ29uc3RydWN0PigpO1xuICAgICAgICBmb3IgKGNvbnN0IGRlcCBvZiBzZWxmLmRlcGVuZGFibGVzKSB7XG4gICAgICAgICAgcmV0LnB1c2goLi4uRGVwZW5kYWJsZVRyYWl0LmdldChkZXApLmRlcGVuZGVuY3lSb290cyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJldDtcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgY29uc3RydWN0IHRvIHRoZSBkZXBlbmRlbmN5IHJvb3RzXG4gICAqL1xuICBwdWJsaWMgYWRkKGRlcDogSURlcGVuZGFibGUpIHtcbiAgICB0aGlzLmRlcGVuZGFibGVzLnB1c2goZGVwKTtcbiAgfVxufVxuXG4vKipcbiAqIEludm9rZSBhIGZ1bmN0aW9uIG9uIGEgdmFsdWUgKGZvciBpdHMgc2lkZSBlZmZlY3QpIGFuZCByZXR1cm4gdGhlIHZhbHVlXG4gKi9cbmZ1bmN0aW9uIHRhcDxUPih4OiBULCBmbjogKHg6IFQpID0+IHZvaWQpOiBUIHtcbiAgZm4oeCk7XG4gIHJldHVybiB4O1xufVxuXG5jbGFzcyBJbXBvcnRlZFN1Ym5ldCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVN1Ym5ldCwgSVB1YmxpY1N1Ym5ldCwgSVByaXZhdGVTdWJuZXQge1xuICBwdWJsaWMgcmVhZG9ubHkgaW50ZXJuZXRDb25uZWN0aXZpdHlFc3RhYmxpc2hlZDogSURlcGVuZGFibGUgPSBuZXcgQ29uY3JldGVEZXBlbmRhYmxlKCk7XG4gIHB1YmxpYyByZWFkb25seSBzdWJuZXRJZDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgcm91dGVUYWJsZTogSVJvdXRlVGFibGU7XG4gIHByaXZhdGUgcmVhZG9ubHkgX2F2YWlsYWJpbGl0eVpvbmU/OiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgX2lwdjRDaWRyQmxvY2s/OiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IFN1Ym5ldEF0dHJpYnV0ZXMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgaWYgKCFhdHRycy5yb3V0ZVRhYmxlSWQpIHtcbiAgICAgIGNvbnN0IHJlZiA9IFRva2VuLmlzVW5yZXNvbHZlZChhdHRycy5zdWJuZXRJZClcbiAgICAgICAgPyBgYXQgJyR7c2NvcGUubm9kZS5wYXRofS8ke2lkfSdgXG4gICAgICAgIDogYCcke2F0dHJzLnN1Ym5ldElkfSdgO1xuICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbiAgICAgIEFubm90YXRpb25zLm9mKHRoaXMpLmFkZFdhcm5pbmcoYE5vIHJvdXRlVGFibGVJZCB3YXMgcHJvdmlkZWQgdG8gdGhlIHN1Ym5ldCAke3JlZn0uIEF0dGVtcHRpbmcgdG8gcmVhZCBpdHMgLnJvdXRlVGFibGUucm91dGVUYWJsZUlkIHdpbGwgcmV0dXJuIG51bGwvdW5kZWZpbmVkLiAoTW9yZSBpbmZvOiBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1jZGsvcHVsbC8zMTcxKWApO1xuICAgIH1cblxuICAgIHRoaXMuX2lwdjRDaWRyQmxvY2sgPSBhdHRycy5pcHY0Q2lkckJsb2NrO1xuICAgIHRoaXMuX2F2YWlsYWJpbGl0eVpvbmUgPSBhdHRycy5hdmFpbGFiaWxpdHlab25lO1xuICAgIHRoaXMuc3VibmV0SWQgPSBhdHRycy5zdWJuZXRJZDtcbiAgICB0aGlzLnJvdXRlVGFibGUgPSB7XG4gICAgICAvLyBGb3JjaW5nIHJvdXRlVGFibGVJZCB0byBwcmV0ZW5kIG5vbi1udWxsIHRvIG1haW50YWluIGJhY2t3YXJkcy1jb21wYXRpYmlsaXR5LiBTZWUgaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL3B1bGwvMzE3MVxuICAgICAgcm91dGVUYWJsZUlkOiBhdHRycy5yb3V0ZVRhYmxlSWQhLFxuICAgIH07XG4gIH1cblxuICBwdWJsaWMgZ2V0IGF2YWlsYWJpbGl0eVpvbmUoKTogc3RyaW5nIHtcbiAgICBpZiAoIXRoaXMuX2F2YWlsYWJpbGl0eVpvbmUpIHtcbiAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBtYXgtbGVuXG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBjYW5ub3QgcmVmZXJlbmNlIGEgU3VibmV0XFwncyBhdmFpbGFiaWxpdHkgem9uZSBpZiBpdCB3YXMgbm90IHN1cHBsaWVkLiBBZGQgdGhlIGF2YWlsYWJpbGl0eVpvbmUgd2hlbiBpbXBvcnRpbmcgdXNpbmcgU3VibmV0LmZyb21TdWJuZXRBdHRyaWJ1dGVzKCknKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuX2F2YWlsYWJpbGl0eVpvbmU7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGlwdjRDaWRyQmxvY2soKTogc3RyaW5nIHtcbiAgICBpZiAoIXRoaXMuX2lwdjRDaWRyQmxvY2spIHtcbiAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTogbWF4LWxpbmUtbGVuZ3RoXG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1lvdSBjYW5ub3QgcmVmZXJlbmNlIGFuIGltcG9ydGVkIFN1Ym5ldFxcJ3MgSVB2NCBDSURSIGlmIGl0IHdhcyBub3Qgc3VwcGxpZWQuIEFkZCB0aGUgaXB2NENpZHJCbG9jayB3aGVuIGltcG9ydGluZyB1c2luZyBTdWJuZXQuZnJvbVN1Ym5ldEF0dHJpYnV0ZXMoKScpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5faXB2NENpZHJCbG9jaztcbiAgfVxuXG4gIHB1YmxpYyBhc3NvY2lhdGVOZXR3b3JrQWNsKGlkOiBzdHJpbmcsIG5ldHdvcmtBY2w6IElOZXR3b3JrQWNsKTogdm9pZCB7XG4gICAgY29uc3Qgc2NvcGUgPSBDb25zdHJ1Y3QuaXNDb25zdHJ1Y3QobmV0d29ya0FjbCkgPyBuZXR3b3JrQWNsIDogdGhpcztcbiAgICBjb25zdCBvdGhlciA9IENvbnN0cnVjdC5pc0NvbnN0cnVjdChuZXR3b3JrQWNsKSA/IHRoaXMgOiBuZXR3b3JrQWNsO1xuICAgIG5ldyBTdWJuZXROZXR3b3JrQWNsQXNzb2NpYXRpb24oc2NvcGUsIGlkICsgb3RoZXIubm9kZS51bmlxdWVJZCwge1xuICAgICAgbmV0d29ya0FjbCxcbiAgICAgIHN1Ym5ldDogdGhpcyxcbiAgICB9KTtcbiAgfVxufVxuXG4vKipcbiAqIERldGVybWluZSAoYW5kIHZhbGlkYXRlKSB0aGUgTkFUIGdhdGV3YXkgY291bnQgdy5yLnQuIHRoZSByZXN0IG9mIHRoZSBzdWJuZXQgY29uZmlndXJhdGlvblxuICpcbiAqIFdlIGhhdmUgdGhlIGZvbGxvd2luZyByZXF1aXJlbWVudHM6XG4gKlxuICogLSBOYXRHYXRld2F5Q291bnQgPSAwID09PiB0aGVyZSBhcmUgbm8gcHJpdmF0ZSBzdWJuZXRzXG4gKiAtIE5hdEdhdGV3YXlDb3VudCA+IDAgPT0+IHRoZXJlIG11c3QgYmUgcHVibGljIHN1Ym5ldHNcbiAqXG4gKiBEbyB3ZSB3YW50IHRvIHJlcXVpcmUgdGhhdCB0aGVyZSBhcmUgcHJpdmF0ZSBzdWJuZXRzIGlmIHRoZXJlIGFyZSBOYXRHYXRld2F5cz9cbiAqIFRoZXkgc2VlbSBwb2ludGxlc3MgYnV0IEkgc2VlIG5vIHJlYXNvbiB0byBwcmV2ZW50IGl0LlxuICovXG5mdW5jdGlvbiBkZXRlcm1pbmVOYXRHYXRld2F5Q291bnQocmVxdWVzdGVkQ291bnQ6IG51bWJlciB8IHVuZGVmaW5lZCwgc3VibmV0Q29uZmlnOiBTdWJuZXRDb25maWd1cmF0aW9uW10sIGF6Q291bnQ6IG51bWJlcikge1xuICBjb25zdCBoYXNQcml2YXRlU3VibmV0cyA9IHN1Ym5ldENvbmZpZy5zb21lKGMgPT4gYy5zdWJuZXRUeXBlID09PSBTdWJuZXRUeXBlLlBSSVZBVEUgJiYgIWMucmVzZXJ2ZWQpO1xuICBjb25zdCBoYXNQdWJsaWNTdWJuZXRzID0gc3VibmV0Q29uZmlnLnNvbWUoYyA9PiBjLnN1Ym5ldFR5cGUgPT09IFN1Ym5ldFR5cGUuUFVCTElDKTtcblxuICBjb25zdCBjb3VudCA9IHJlcXVlc3RlZENvdW50ICE9PSB1bmRlZmluZWQgPyBNYXRoLm1pbihyZXF1ZXN0ZWRDb3VudCwgYXpDb3VudCkgOiAoaGFzUHJpdmF0ZVN1Ym5ldHMgPyBhekNvdW50IDogMCk7XG5cbiAgaWYgKGNvdW50ID09PSAwICYmIGhhc1ByaXZhdGVTdWJuZXRzKSB7XG4gICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbiAgICB0aHJvdyBuZXcgRXJyb3IoJ0lmIHlvdSBkbyBub3Qgd2FudCBOQVQgZ2F0ZXdheXMgKG5hdEdhdGV3YXlzPTApLCBtYWtlIHN1cmUgeW91IGRvblxcJ3QgY29uZmlndXJlIGFueSBQUklWQVRFIHN1Ym5ldHMgaW4gXFwnc3VibmV0Q29uZmlndXJhdGlvblxcJyAobWFrZSB0aGVtIFBVQkxJQyBvciBJU09MQVRFRCBpbnN0ZWFkKScpO1xuICB9XG5cbiAgaWYgKGNvdW50ID4gMCAmJiAhaGFzUHVibGljU3VibmV0cykge1xuICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBtYXgtbGVuXG4gICAgdGhyb3cgbmV3IEVycm9yKGBJZiB5b3UgY29uZmlndXJlIFBSSVZBVEUgc3VibmV0cyBpbiAnc3VibmV0Q29uZmlndXJhdGlvbicsIHlvdSBtdXN0IGFsc28gY29uZmlndXJlIFBVQkxJQyBzdWJuZXRzIHRvIHB1dCB0aGUgTkFUIGdhdGV3YXlzIGludG8gKGdvdCAke0pTT04uc3RyaW5naWZ5KHN1Ym5ldENvbmZpZyl9LmApO1xuICB9XG5cbiAgcmV0dXJuIGNvdW50O1xufVxuXG4vKipcbiAqIFRoZXJlIGFyZSByZXR1cm5lZCB3aGVuIHRoZSBwcm92aWRlciBoYXMgbm90IHN1cHBsaWVkIHByb3BzIHlldFxuICpcbiAqIEl0J3Mgb25seSB1c2VkIGZvciB0ZXN0aW5nIGFuZCBvbiB0aGUgZmlyc3QgcnVuLXRocm91Z2guXG4gKi9cbmNvbnN0IERVTU1ZX1ZQQ19QUk9QUzogY3hhcGkuVnBjQ29udGV4dFJlc3BvbnNlID0ge1xuICBhdmFpbGFiaWxpdHlab25lczogW10sXG4gIHZwY0NpZHJCbG9jazogJzEuMi4zLjQvNScsXG4gIGlzb2xhdGVkU3VibmV0SWRzOiB1bmRlZmluZWQsXG4gIGlzb2xhdGVkU3VibmV0TmFtZXM6IHVuZGVmaW5lZCxcbiAgaXNvbGF0ZWRTdWJuZXRSb3V0ZVRhYmxlSWRzOiB1bmRlZmluZWQsXG4gIHByaXZhdGVTdWJuZXRJZHM6IHVuZGVmaW5lZCxcbiAgcHJpdmF0ZVN1Ym5ldE5hbWVzOiB1bmRlZmluZWQsXG4gIHByaXZhdGVTdWJuZXRSb3V0ZVRhYmxlSWRzOiB1bmRlZmluZWQsXG4gIHB1YmxpY1N1Ym5ldElkczogdW5kZWZpbmVkLFxuICBwdWJsaWNTdWJuZXROYW1lczogdW5kZWZpbmVkLFxuICBwdWJsaWNTdWJuZXRSb3V0ZVRhYmxlSWRzOiB1bmRlZmluZWQsXG4gIHN1Ym5ldEdyb3VwczogW1xuICAgIHtcbiAgICAgIG5hbWU6ICdQdWJsaWMnLFxuICAgICAgdHlwZTogY3hhcGkuVnBjU3VibmV0R3JvdXBUeXBlLlBVQkxJQyxcbiAgICAgIHN1Ym5ldHM6IFtcbiAgICAgICAge1xuICAgICAgICAgIGF2YWlsYWJpbGl0eVpvbmU6ICdkdW1teTFhJyxcbiAgICAgICAgICBzdWJuZXRJZDogJ3MtMTIzNDUnLFxuICAgICAgICAgIHJvdXRlVGFibGVJZDogJ3J0Yi0xMjM0NXMnLFxuICAgICAgICAgIGNpZHI6ICcxLjIuMy40LzUnLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgYXZhaWxhYmlsaXR5Wm9uZTogJ2R1bW15MWInLFxuICAgICAgICAgIHN1Ym5ldElkOiAncy02Nzg5MCcsXG4gICAgICAgICAgcm91dGVUYWJsZUlkOiAncnRiLTY3ODkwcycsXG4gICAgICAgICAgY2lkcjogJzEuMi4zLjQvNScsXG4gICAgICAgIH0sXG4gICAgICBdLFxuICAgIH0sXG4gICAge1xuICAgICAgbmFtZTogJ1ByaXZhdGUnLFxuICAgICAgdHlwZTogY3hhcGkuVnBjU3VibmV0R3JvdXBUeXBlLlBSSVZBVEUsXG4gICAgICBzdWJuZXRzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBhdmFpbGFiaWxpdHlab25lOiAnZHVtbXkxYScsXG4gICAgICAgICAgc3VibmV0SWQ6ICdwLTEyMzQ1JyxcbiAgICAgICAgICByb3V0ZVRhYmxlSWQ6ICdydGItMTIzNDVwJyxcbiAgICAgICAgICBjaWRyOiAnMS4yLjMuNC81JyxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgIGF2YWlsYWJpbGl0eVpvbmU6ICdkdW1teTFiJyxcbiAgICAgICAgICBzdWJuZXRJZDogJ3AtNjc4OTAnLFxuICAgICAgICAgIHJvdXRlVGFibGVJZDogJ3J0Yi01Nzg5MHAnLFxuICAgICAgICAgIGNpZHI6ICcxLjIuMy40LzUnLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9LFxuICBdLFxuICB2cGNJZDogJ3ZwYy0xMjM0NScsXG59O1xuIl19