"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Volume = exports.EbsDeviceVolumeType = exports.synthesizeBlockDeviceMappings = exports.BlockDeviceVolume = void 0;
const crypto = require("crypto");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * Describes a block device mapping for an EC2 instance or Auto Scaling group.
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info
     * @param virtualName Virtual device name
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * Creates a new Elastic Block Storage device
     *
     * @param volumeSize The volume size, in Gibibytes (GiB)
     * @param options additional device options
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * Creates a new Elastic Block Storage device from an existing snapshot
     *
     * @param snapshotId The snapshot ID of the volume to use
     * @param options additional device options
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * Creates a virtual, ephemeral device.
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index. Must be equal or greater than 0
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== EbsDeviceVolumeType.IO1) {
                core_1.Annotations.of(construct).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName,
            ebs,
            virtualName,
            noDevice: mappingEnabled === false ? {} : undefined,
        };
    });
}
exports.synthesizeBlockDeviceMappings = synthesizeBlockDeviceMappings;
/**
 * Supported EBS volume types for blockDevices
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    /**
     * Magnetic
     */
    EbsDeviceVolumeType["STANDARD"] = "standard";
    /**
     *  Provisioned IOPS SSD
     */
    EbsDeviceVolumeType["IO1"] = "io1";
    /**
     * General Purpose SSD
     */
    EbsDeviceVolumeType["GP2"] = "gp2";
    /**
     * Throughput Optimized HDD
     */
    EbsDeviceVolumeType["ST1"] = "st1";
    /**
     * Cold HDD
     */
    EbsDeviceVolumeType["SC1"] = "sc1";
    /**
     * General purpose SSD volume that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD"] = "gp2";
    /**
     * Highest-performance SSD volume for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD"] = "io1";
    /**
     * Low-cost HDD volume designed for frequently accessed, throughput-intensive workloads.
     */
    EbsDeviceVolumeType["THROUGHPUT_OPTIMIZED_HDD"] = "st1";
    /**
     * Lowest cost HDD volume designed for less frequently accessed workloads.
     */
    EbsDeviceVolumeType["COLD_HDD"] = "sc1";
    /**
     * Magnetic volumes are backed by magnetic drives and are suited for workloads where data is accessed infrequently, and scenarios where low-cost
     * storage for small volume sizes is important.
     */
    EbsDeviceVolumeType["MAGNETIC"] = "standard";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
/**
 * Common behavior of Volumes. Users should not use this class directly, and instead use ``Volume``.
 */
class VolumeBase extends core_1.Resource {
    grantAttachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:AttachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        if (this.encryptionKey) {
            // When attaching a volume, the EC2 Service will need to grant to itself permission
            // to be able to decrypt the encryption key. We restrict the CreateGrant for principle
            // of least privilege, in accordance with best practices.
            // See: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-permissions
            const kmsGrant = this.encryptionKey.grant(grantee, 'kms:CreateGrant');
            kmsGrant.principalStatement.addConditions({
                Bool: { 'kms:GrantIsForAWSResource': true },
                StringEquals: {
                    'kms:ViaService': `ec2.${core_1.Stack.of(this).region}.amazonaws.com`,
                    'kms:GrantConstraintType': 'EncryptionContextSubset',
                },
            });
        }
        return result;
    }
    grantAttachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantAttach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantAttachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    grantDetachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:DetachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        // Note: No encryption key permissions are required to detach an encrypted volume.
        return result;
    }
    grantDetachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantDetach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantDetachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    collectGrantResourceArns(instances) {
        const stack = core_1.Stack.of(this);
        const resourceArns = [
            `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:volume/${this.volumeId}`,
        ];
        const instanceArnPrefix = `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:instance`;
        if (instances) {
            instances.forEach(instance => resourceArns.push(`${instanceArnPrefix}/${instance === null || instance === void 0 ? void 0 : instance.instanceId}`));
        }
        else {
            resourceArns.push(`${instanceArnPrefix}/*`);
        }
        return resourceArns;
    }
    calculateResourceTagValue(constructs) {
        const md5 = crypto.createHash('md5');
        constructs.forEach(construct => md5.update(construct.node.uniqueId));
        return md5.digest('hex');
    }
}
/**
 * Creates a new EBS Volume in AWS EC2.
 */
class Volume extends VolumeBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.volumeName,
        });
        this.validateProps(props);
        const resource = new ec2_generated_1.CfnVolume(this, 'Resource', {
            availabilityZone: props.availabilityZone,
            autoEnableIo: props.autoEnableIo,
            encrypted: props.encrypted,
            kmsKeyId: (_a = props.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
            iops: props.iops,
            multiAttachEnabled: (_b = props.enableMultiAttach) !== null && _b !== void 0 ? _b : false,
            size: (_c = props.size) === null || _c === void 0 ? void 0 : _c.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }),
            snapshotId: props.snapshotId,
            volumeType: (_d = props.volumeType) !== null && _d !== void 0 ? _d : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
        });
        this.volumeId = resource.ref;
        this.availabilityZone = props.availabilityZone;
        this.encryptionKey = props.encryptionKey;
        if (this.encryptionKey) {
            // Per: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-requirements
            const principal = new aws_kms_1.ViaServicePrincipal(`ec2.${core_1.Stack.of(this).region}.amazonaws.com`, new aws_iam_1.AccountRootPrincipal()).withConditions({
                StringEquals: {
                    'kms:CallerAccount': core_1.Stack.of(this).account,
                },
            });
            const grant = this.encryptionKey.grant(principal, 
            // Describe & Generate are required to be able to create the CMK-encrypted Volume.
            'kms:DescribeKey', 'kms:GenerateDataKeyWithoutPlainText');
            if (props.snapshotId) {
                // ReEncrypt is required for when re-encrypting from an encrypted snapshot.
                (_e = grant.principalStatement) === null || _e === void 0 ? void 0 : _e.addActions('kms:ReEncrypt*');
            }
        }
    }
    /**
     * Import an existing EBS Volume into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Volume in the construct tree.
     * @param attrs the attributes of the imported Volume
     */
    static fromVolumeAttributes(scope, id, attrs) {
        class Import extends VolumeBase {
            constructor() {
                super(...arguments);
                this.volumeId = attrs.volumeId;
                this.availabilityZone = attrs.availabilityZone;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        // Check that the provided volumeId looks like it could be valid.
        if (!core_1.Token.isUnresolved(attrs.volumeId) && !/^vol-[0-9a-fA-F]+$/.test(attrs.volumeId)) {
            throw new Error('`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        }
        return new Import(scope, id);
    }
    validateProps(props) {
        var _a;
        if (!(props.size || props.snapshotId)) {
            throw new Error('Must provide at least one of `size` or `snapshotId`');
        }
        if (props.snapshotId && !core_1.Token.isUnresolved(props.snapshotId) && !/^snap-[0-9a-fA-F]+$/.test(props.snapshotId)) {
            throw new Error('`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        }
        if (props.encryptionKey && !props.encrypted) {
            throw new Error('`encrypted` must be true when providing an `encryptionKey`.');
        }
        if (props.iops) {
            if (props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
                throw new Error('`iops` may only be specified if the `volumeType` is `PROVISIONED_IOPS_SSD`/`IO1`');
            }
            if (props.iops < 100 || props.iops > 64000) {
                throw new Error('`iops` must be in the range 100 to 64,000, inclusive.');
            }
            if (props.size && (props.iops > 50 * props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }))) {
                throw new Error('`iops` has a maximum ratio of 50 IOPS/GiB.');
            }
        }
        if (props.enableMultiAttach && props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
            throw new Error('multi-attach is supported exclusively on `PROVISIONED_IOPS_SSD` volumes.');
        }
        if (props.size) {
            const size = props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
            // Enforce maximum volume size:
            // https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html#ebs-volume-characteristics
            const sizeRanges = {};
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD] = { Min: 1, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 4, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.COLD_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.MAGNETIC] = { Min: 1, Max: 1000 };
            const volumeType = (_a = props.volumeType) !== null && _a !== void 0 ? _a : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            const { Min, Max } = sizeRanges[volumeType];
            if (size < Min || size > Max) {
                throw new Error(`\`${volumeType}\` volumes must be between ${Min} GiB and ${Max} GiB in size.`);
            }
        }
    }
}
exports.Volume = Volume;
//# sourceMappingURL=data:application/json;base64,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