"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LookupMachineImage = exports.OperatingSystemType = exports.GenericWindowsImage = exports.GenericLinuxImage = exports.AmazonLinuxStorage = exports.AmazonLinuxVirt = exports.AmazonLinuxEdition = exports.AmazonLinuxGeneration = exports.AmazonLinuxImage = exports.AmazonLinuxCpuType = exports.WindowsImage = exports.GenericSSMParameterImage = exports.MachineImage = void 0;
const ssm = require("@aws-cdk/aws-ssm");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const user_data_1 = require("./user-data");
/**
 * Factory functions for standard Amazon Machine Image objects.
 */
class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestWindows(version, props) {
        return new WindowsImage(version, props);
    }
    /**
     * An Amazon Linux image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestAmazonLinux(props) {
        return new AmazonLinuxImage(props);
    }
    /**
     * A Linux image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericLinux(amiMap, props) {
        return new GenericLinuxImage(amiMap, props);
    }
    /**
     * A Windows image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericWindows(amiMap, props) {
        return new GenericWindowsImage(amiMap, props);
    }
    /**
     * An image specified in SSM parameter store that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     *
     * @param parameterName The name of SSM parameter containing the AMi id
     * @param os The operating system type of the AMI
     * @param userData optional user data for the given image
     */
    static fromSSMParameter(parameterName, os, userData) {
        return new GenericSSMParameterImage(parameterName, os, userData);
    }
    /**
     * Look up a shared Machine Image using DescribeImages
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static lookup(props) {
        return new LookupMachineImage(props);
    }
}
exports.MachineImage = MachineImage;
/**
 * Select the image based on a given SSM parameter
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class GenericSSMParameterImage {
    constructor(parameterName, os, userData) {
        this.parameterName = parameterName;
        this.os = os;
        this.userData = userData;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        var _a;
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            osType: this.os,
            userData: (_a = this.userData) !== null && _a !== void 0 ? _a : (this.os === OperatingSystemType.WINDOWS ? user_data_1.UserData.forWindows() : user_data_1.UserData.forLinux()),
        };
    }
}
exports.GenericSSMParameterImage = GenericSSMParameterImage;
/**
 * Select the latest version of the indicated Windows version
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
class WindowsImage extends GenericSSMParameterImage {
    constructor(version, props = {}) {
        super('/aws/service/ami-windows-latest/' + version, OperatingSystemType.WINDOWS, props.userData);
    }
}
exports.WindowsImage = WindowsImage;
/**
 * CPU type
 */
var AmazonLinuxCpuType;
(function (AmazonLinuxCpuType) {
    /**
     * arm64 CPU type
     */
    AmazonLinuxCpuType["ARM_64"] = "arm64";
    /**
     * x86_64 CPU type
     */
    AmazonLinuxCpuType["X86_64"] = "x86_64";
})(AmazonLinuxCpuType = exports.AmazonLinuxCpuType || (exports.AmazonLinuxCpuType = {}));
/**
 * Selects the latest version of Amazon Linux
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class AmazonLinuxImage extends GenericSSMParameterImage {
    constructor(props = {}) {
        const generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        const edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        const virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
        const storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
        const cpu = (props && props.cpuType) || AmazonLinuxCpuType.X86_64;
        const parts = [
            generation,
            'ami',
            edition !== AmazonLinuxEdition.STANDARD ? edition : undefined,
            virtualization,
            cpu,
            storage,
        ].filter(x => x !== undefined); // Get rid of undefineds
        const parameterName = '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
        super(parameterName, OperatingSystemType.LINUX, props.userData);
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
/**
 * What generation of Amazon Linux to use
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    /**
     * Amazon Linux
     */
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    /**
     * Amazon Linux 2
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux edition
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    /**
     * Standard edition
     */
    AmazonLinuxEdition["STANDARD"] = "standard";
    /**
     * Minimal edition
     */
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    /**
     * HVM virtualization (recommended)
     */
    AmazonLinuxVirt["HVM"] = "hvm";
    /**
     * PV virtualization
     */
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    /**
     * EBS-backed storage
     */
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "ebs";
    /**
     * General Purpose-based storage (recommended)
     */
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
class GenericLinuxImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forLinux(),
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
/**
 * Construct a Windows machine image from an AMI map
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
class GenericWindowsImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forWindows(),
            osType: OperatingSystemType.WINDOWS,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
/**
 * The OS type of a particular image
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
    /**
     * Used when the type of the operating system is not known
     * (for example, for imported Auto-Scaling Groups).
     */
    OperatingSystemType[OperatingSystemType["UNKNOWN"] = 2] = "UNKNOWN";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 */
class LookupMachineImage {
    constructor(props) {
        this.props = props;
    }
    getImage(scope) {
        var _a;
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        const osType = this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX;
        return {
            imageId: value,
            osType,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forOperatingSystem(osType),
        };
    }
}
exports.LookupMachineImage = LookupMachineImage;
//# sourceMappingURL=data:application/json;base64,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