"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitSource = exports.InitService = exports.InitPackage = exports.InitUser = exports.InitGroup = exports.InitFile = exports.InitCommand = exports.InitCommandWaitDuration = exports.InitElement = exports.InitServiceRestartHandle = void 0;
const fs = require("fs");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * An object that represents reasons to restart an InitService
 *
 * Pass an instance of this object to the `InitFile`, `InitCommand`,
 * `InitSource` and `InitPackage` objects, and finally to an `InitService`
 * itself to cause the actions (files, commands, sources, and packages)
 * to trigger a restart of the service.
 *
 * For example, the following will run a custom command to install Nginx,
 * and trigger the nginx service to be restarted after the command has run.
 *
 * ```ts
 * const handle = new ec2.InitServiceRestartHandle();
 * ec2.CloudFormationInit.fromElements(
 *   ec2.InitCommand.shellCommand('/usr/bin/custom-nginx-install.sh', { serviceRestartHandles: [handle] }),
 *   ec2.InitService.enable('nginx', { serviceRestartHandle: handle }),
 * );
 * ```
 */
class InitServiceRestartHandle {
    constructor() {
        this.commands = new Array();
        this.files = new Array();
        this.sources = new Array();
        this.packages = {};
    }
    /**
     * Add a command key to the restart set
     * @internal
     */
    _addCommand(key) {
        return this.commands.push(key);
    }
    /**
     * Add a file key to the restart set
     * @internal
     */
    _addFile(key) {
        return this.files.push(key);
    }
    /**
     * Add a source key to the restart set
     * @internal
     */
    _addSource(key) {
        return this.sources.push(key);
    }
    /**
     * Add a package key to the restart set
     * @internal
     */
    _addPackage(packageType, key) {
        if (!this.packages[packageType]) {
            this.packages[packageType] = [];
        }
        this.packages[packageType].push(key);
    }
    /**
     * Render the restart handles for use in an InitService declaration
     * @internal
     */
    _renderRestartHandles() {
        const nonEmpty = (x) => x.length > 0 ? x : undefined;
        return {
            commands: nonEmpty(this.commands),
            files: nonEmpty(this.files),
            packages: Object.keys(this.packages).length > 0 ? this.packages : undefined,
            sources: nonEmpty(this.sources),
        };
    }
}
exports.InitServiceRestartHandle = InitServiceRestartHandle;
/**
 * Base class for all CloudFormation Init elements
 */
class InitElement {
}
exports.InitElement = InitElement;
/**
 * Represents a duration to wait after a command has finished, in case of a reboot (Windows only).
 */
class InitCommandWaitDuration {
    /** Wait for a specified duration after a command. */
    static of(duration) {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return duration.toSeconds(); }
        }();
    }
    /** Do not wait for this command. */
    static none() {
        return InitCommandWaitDuration.of(core_1.Duration.seconds(0));
    }
    /** cfn-init will exit and resume only after a reboot. */
    static forever() {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return 'forever'; }
        }();
    }
}
exports.InitCommandWaitDuration = InitCommandWaitDuration;
/**
 * Command to execute on the instance
 */
class InitCommand extends InitElement {
    constructor(command, options) {
        super();
        this.command = command;
        this.options = options;
        this.elementType = cfn_init_internal_1.InitElementType.COMMAND.toString();
    }
    /**
     * Run a shell command
     *
     * Remember that some characters like `&`, `|`, `;`, `>` etc. have special meaning in a shell and
     * need to be preceded by a `\` if you want to treat them as part of a filename.
     */
    static shellCommand(shellCommand, options = {}) {
        return new InitCommand([shellCommand], options);
    }
    /**
     * Run a command from an argv array
     *
     * You do not need to escape space characters or enclose command parameters in quotes.
     */
    static argvCommand(argv, options = {}) {
        if (argv.length === 0) {
            throw new Error('Cannot define argvCommand with an empty arguments');
        }
        return new InitCommand(argv, options);
    }
    /** @internal */
    _bind(options) {
        var _a, _b;
        const commandKey = this.options.key || `${options.index}`.padStart(3, '0'); // 001, 005, etc.
        if (options.platform !== cfn_init_internal_1.InitPlatform.WINDOWS && this.options.waitAfterCompletion !== undefined) {
            throw new Error(`Command '${this.command}': 'waitAfterCompletion' is only valid for Windows systems.`);
        }
        for (const handle of (_a = this.options.serviceRestartHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addCommand(commandKey);
        }
        return {
            config: {
                [commandKey]: {
                    command: this.command,
                    env: this.options.env,
                    cwd: this.options.cwd,
                    test: this.options.testCmd,
                    ignoreErrors: this.options.ignoreErrors,
                    waitAfterCompletion: (_b = this.options.waitAfterCompletion) === null || _b === void 0 ? void 0 : _b._render(),
                },
            },
        };
    }
}
exports.InitCommand = InitCommand;
/**
 * Create files on the EC2 instance.
 */
class InitFile extends InitElement {
    constructor(fileName, options) {
        super();
        this.fileName = fileName;
        this.options = options;
        this.elementType = cfn_init_internal_1.InitElementType.FILE.toString();
    }
    /**
     * Use a literal string as the file content
     */
    static fromString(fileName, content, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content,
                        encoding: this.options.base64Encoded ? 'base64' : 'plain',
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Write a symlink with the given symlink target
     */
    static symlink(fileName, target, options = {}) {
        const { mode, ...otherOptions } = options;
        if (mode && mode.slice(0, 3) !== '120') {
            throw new Error('File mode for symlinks must begin with 120XXX');
        }
        return InitFile.fromString(fileName, target, { mode: (mode || '120644'), ...otherOptions });
    }
    /**
     * Use a JSON-compatible object as the file content, write it to a JSON file.
     *
     * May contain tokens.
     */
    static fromObject(fileName, obj, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content: obj,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Read a file from disk and use its contents
     *
     * The file will be embedded in the template, so care should be taken to not
     * exceed the template size.
     *
     * If options.base64encoded is set to true, this will base64-encode the file's contents.
     */
    static fromFileInline(targetFileName, sourceFileName, options = {}) {
        const encoding = options.base64Encoded ? 'base64' : 'utf8';
        const fileContents = fs.readFileSync(sourceFileName).toString(encoding);
        return InitFile.fromString(targetFileName, fileContents, options);
    }
    /**
     * Download from a URL at instance startup time
     */
    static fromUrl(fileName, url, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: url,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * Download a file from an S3 bucket at instance startup time
     */
    static fromS3Object(fileName, bucket, key, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: bucket.urlForObject(key),
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(fileName, options);
    }
    /**
     * Create an asset from the given file
     *
     * This is appropriate for files that are too large to embed into the template.
     */
    static fromAsset(targetFileName, path, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetFileName}Asset`, {
                    path,
                    ...options,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /**
     * Use a file from an asset at instance startup time
     */
    static fromExistingAsset(targetFileName, asset, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /** @internal */
    _bind(bindOptions) {
        var _a;
        for (const handle of (_a = this.options.serviceRestartHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addFile(this.fileName);
        }
        return this._doBind(bindOptions);
    }
    /**
     * Render the standard config block, given content vars
     * @internal
     */
    _standardConfig(fileOptions, platform, contentVars) {
        if (platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            if (fileOptions.group || fileOptions.owner || fileOptions.mode) {
                throw new Error('Owner, group, and mode options not supported for Windows.');
            }
            return {
                [this.fileName]: { ...contentVars },
            };
        }
        return {
            [this.fileName]: {
                ...contentVars,
                mode: fileOptions.mode || '000644',
                owner: fileOptions.owner || 'root',
                group: fileOptions.group || 'root',
            },
        };
    }
}
exports.InitFile = InitFile;
/**
 * Create Linux/UNIX groups and assign group IDs.
 *
 * Not supported for Windows systems.
 */
class InitGroup extends InitElement {
    constructor(groupName, groupId) {
        super();
        this.groupName = groupName;
        this.groupId = groupId;
        this.elementType = cfn_init_internal_1.InitElementType.GROUP.toString();
    }
    /**
     * Create a group from its name, and optionally, group id
     */
    static fromName(groupName, groupId) {
        return new InitGroup(groupName, groupId);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init groups are not supported on Windows');
        }
        return {
            config: {
                [this.groupName]: this.groupId !== undefined ? { gid: this.groupId } : {},
            },
        };
    }
}
exports.InitGroup = InitGroup;
/**
 * Create Linux/UNIX users and to assign user IDs.
 *
 * Users are created as non-interactive system users with a shell of
 * /sbin/nologin. This is by design and cannot be modified.
 *
 * Not supported for Windows systems.
 */
class InitUser extends InitElement {
    constructor(userName, userOptions) {
        super();
        this.userName = userName;
        this.userOptions = userOptions;
        this.elementType = cfn_init_internal_1.InitElementType.USER.toString();
    }
    /**
     * Create a user from user name.
     */
    static fromName(userName, options = {}) {
        return new InitUser(userName, options);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init users are not supported on Windows');
        }
        return {
            config: {
                [this.userName]: {
                    uid: this.userOptions.userId,
                    groups: this.userOptions.groups,
                    homeDir: this.userOptions.homeDir,
                },
            },
        };
    }
}
exports.InitUser = InitUser;
/**
 * A package to be installed during cfn-init time
 */
class InitPackage extends InitElement {
    constructor(type, versions, packageName, serviceHandles) {
        super();
        this.type = type;
        this.versions = versions;
        this.packageName = packageName;
        this.serviceHandles = serviceHandles;
        this.elementType = cfn_init_internal_1.InitElementType.PACKAGE.toString();
    }
    /**
     * Install an RPM from an HTTP URL or a location on disk
     */
    static rpm(location, options = {}) {
        return new InitPackage('rpm', [location], options.key, options.serviceRestartHandles);
    }
    /**
     * Install a package using Yum
     */
    static yum(packageName, options = {}) {
        var _a;
        return new InitPackage('yum', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install a package from RubyGems
     */
    static rubyGem(gemName, options = {}) {
        var _a;
        return new InitPackage('rubygems', (_a = options.version) !== null && _a !== void 0 ? _a : [], gemName, options.serviceRestartHandles);
    }
    /**
     * Install a package from PyPI
     */
    static python(packageName, options = {}) {
        var _a;
        return new InitPackage('python', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install a package using APT
     */
    static apt(packageName, options = {}) {
        var _a;
        return new InitPackage('apt', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * Install an MSI package from an HTTP URL or a location on disk
     */
    static msi(location, options = {}) {
        // The MSI package version must be a string, not an array.
        return new class extends InitPackage {
            renderPackageVersions() { return location; }
        }('msi', [location], options.key, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _a;
        if ((this.type === 'msi') !== (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS)) {
            if (this.type === 'msi') {
                throw new Error('MSI installers are only supported on Windows systems.');
            }
            else {
                throw new Error('Windows only supports the MSI package type');
            }
        }
        if (!this.packageName && !['rpm', 'msi'].includes(this.type)) {
            throw new Error('Package name must be specified for all package types besides RPM and MSI.');
        }
        const packageName = this.packageName || `${options.index}`.padStart(3, '0');
        for (const handle of (_a = this.serviceHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addPackage(this.type, packageName);
        }
        return {
            config: {
                [this.type]: {
                    [packageName]: this.renderPackageVersions(),
                },
            },
        };
    }
    renderPackageVersions() {
        return this.versions;
    }
}
exports.InitPackage = InitPackage;
/**
 * A services that be enabled, disabled or restarted when the instance is launched.
 */
class InitService extends InitElement {
    constructor(serviceName, serviceOptions) {
        super();
        this.serviceName = serviceName;
        this.serviceOptions = serviceOptions;
        this.elementType = cfn_init_internal_1.InitElementType.SERVICE.toString();
    }
    /**
     * Enable and start the given service, optionally restarting it
     */
    static enable(serviceName, options = {}) {
        var _a;
        const { enabled, ensureRunning, ...otherOptions } = options;
        return new InitService(serviceName, {
            enabled: enabled !== null && enabled !== void 0 ? enabled : true,
            ensureRunning: (_a = ensureRunning !== null && ensureRunning !== void 0 ? ensureRunning : enabled) !== null && _a !== void 0 ? _a : true,
            ...otherOptions,
        });
    }
    /**
     * Disable and stop the given service
     */
    static disable(serviceName) {
        return new InitService(serviceName, { enabled: false, ensureRunning: false });
    }
    /** @internal */
    _bind(options) {
        var _a;
        const serviceManager = options.platform === cfn_init_internal_1.InitPlatform.LINUX ? 'sysvinit' : 'windows';
        return {
            config: {
                [serviceManager]: {
                    [this.serviceName]: {
                        enabled: this.serviceOptions.enabled,
                        ensureRunning: this.serviceOptions.ensureRunning,
                        ...(_a = this.serviceOptions.serviceRestartHandle) === null || _a === void 0 ? void 0 : _a._renderRestartHandles(),
                    },
                },
            },
        };
    }
}
exports.InitService = InitService;
/**
 * Extract an archive into a directory
 */
class InitSource extends InitElement {
    constructor(targetDirectory, serviceHandles) {
        super();
        this.targetDirectory = targetDirectory;
        this.serviceHandles = serviceHandles;
        this.elementType = cfn_init_internal_1.InitElementType.SOURCE.toString();
    }
    /**
     * Retrieve a URL and extract it into the given directory
     */
    static fromUrl(targetDirectory, url, options = {}) {
        return new class extends InitSource {
            _doBind() {
                return {
                    config: { [this.targetDirectory]: url },
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Extract a GitHub branch into a given directory
     */
    static fromGitHub(targetDirectory, owner, repo, refSpec, options = {}) {
        return InitSource.fromUrl(targetDirectory, `https://github.com/${owner}/${repo}/tarball/${refSpec !== null && refSpec !== void 0 ? refSpec : 'master'}`, options);
    }
    /**
     * Extract an archive stored in an S3 bucket into the given directory
     */
    static fromS3Object(targetDirectory, bucket, key, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: { [this.targetDirectory]: bucket.urlForObject(key) },
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Create an InitSource from an asset created from the given path.
     */
    static fromAsset(targetDirectory, path, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetDirectory}Asset`, {
                    path,
                    ...bindOptions,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * Extract a directory from an existing directory asset.
     */
    static fromExistingAsset(targetDirectory, asset, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _a;
        for (const handle of (_a = this.serviceHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addSource(this.targetDirectory);
        }
        // Delegate actual bind to subclasses
        return this._doBind(options);
    }
}
exports.InitSource = InitSource;
/**
 * Render a standard S3 auth block for use in AWS::CloudFormation::Authentication
 *
 * This block is the same every time (modulo bucket name), so it has the same
 * key every time so the blocks are merged into one in the final render.
 */
function standardS3Auth(role, bucketName) {
    return {
        S3AccessCreds: {
            type: 'S3',
            roleName: role.roleName,
            buckets: [bucketName],
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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