# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['menta']

package_data = \
{'': ['*']}

install_requires = \
['PyNaCl>=1.5.0,<2.0.0']

setup_kwargs = {
    'name': 'menta',
    'version': '0.0.1a1',
    'description': 'Secure, simple and easy to use API token format',
    'long_description': '# menta [![build](https://github.com/alexbakker/menta/actions/workflows/build.yaml/badge.svg)](https://github.com/alexbakker/menta/actions/workflows/build.yaml?query=branch%3Amaster) ![coverage](https://alexbakker.me/gh/targets/menta/artifacts/coverage/coverage.svg) [![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black) ![PyPi version](https://img.shields.io/pypi/v/menta)\n\n__menta__ is a secure, simple and easy to use API token format. It uses\n__XChaCha20-Poly1305__ symmetric authenticated encryption to create encrypted\ntokens and to ensure their integrity. There is no asymmetric option, and there\nis zero cryptographic agility.\n\n__Menta is currently experimental__. The first stable version will be __v1__.\nWhen it\'s released, it\'ll be frozen and it will never be changed. A completely\nnew version will be released if we ever need to make changes or want to add new\nfeatures. If you start using this library today, you can be certain that it\'ll\nstill only accept v1 tokens tomorrow. Support for new versions will never be\nsilently introduced into your existing authentication code path.\n\nMenta was inspired by [Branca](https://www.branca.io/) and is very similar to\nit. There are a couple of differences:\n* The timestamp is included in the ciphertext and thus is not readable to anyone\n  without the key. This makes it less likely that users will trust the timestamp\n  without verifying the token first.\n* The timestamp is a 64-bit unsigned integer. So instead of overflowing in 2106,\n  we\'ll be good for the next couple of hundred billion years.\n* Base64 instead of base62 for better library support across programming\n  languages.\n\n## Usage\n\nThis repository serves as the reference implementation of Menta (in Python). It\ncan be used as follows:\n\n```python\nfrom menta import Menta, TokenData\n\n# create a new menta instance with an existing key\nkey = bytes.fromhex("1df408259cdbba9492c2d01ad4dd942de4047f03ff32515fc6f333627f0e22b8")\nmenta = Menta(key)\n\n# encode the text "hi!" into a new token\ntoken = menta.encode(TokenData("hi!"))\nprint("encoded:", token)\n\n# decode the token we just generated\ndata = menta.decode(token)\nprint("decoded:", data)\n\n# encoded: v1:uhViDSxQNyaSd0BjXPqgmT53N6t2uSwC3KzxhMEsGis00pSgcqmfaLlhkAFJIun8mZCH\n# decoded: TokenData(payload=b\'hi!\', timestamp=1653137637)\n```\n\nThere\'s also a utility for generating a new key:\n\n```python\nfrom menta import Menta\n\n# create a new menta instance with a freshly generated key\nmenta = Menta.generate()\nprint(menta.key.hex())\n```\n\n## Format\n\nMenta tokens start with a version indicator, followed by a __base64url__\nencoding of a concatenation of the nonce, the ciphertext and authentication tag:\n\n```\n"v1:" || nonce (24 bytes) || body ciphertext (? bytes) || tag (16 bytes)\n```\n\nThe contents of the body consists of a concatenation of a timestamp and the\npayload\n\n```\ntimestamp (8 bytes) || payload (? bytes)\n```\n\n### Key\n\nMenta requires a 256-bit key for use with __XChaCha20-Poly1305__. These 32 bytes\nmust be randomly generated using the operating system\'s CSPRNG.\n\n### Fields\n\n#### Version indicator\n\nEvery Menta token starts with a version indicator: ``v1:``\n\n#### Body\n\n##### Timestamp\n\nEvery Menta token contains the time at which it was generated: A Unix timestamp\n(seconds elapsed since 00:00:00 UTC on 1 January 1970).\n\n##### Payload\n\nThe payload is a binary blob of arbitrary length. We recommend using a\nserialization format like [MessagePack](https://pydantic-docs.helpmanual.io/) or\n[Protocol Buffers](https://developers.google.com/protocol-buffers) to encode the\npayload. If you prefer JSON, use a strict library like\n[Pydantic](https://pydantic-docs.helpmanual.io/) to validate the payload.\n\n#### Nonce\n\nThe 196-bit nonce used in encryption and decryption of the ciphertext. It must\nbe randomly generated using the operating system\'s CSPRNG.\n\n#### Authentication tag\n\nThe 128-bit tag used to authenticate the ciphertext.\n\n### Generating\n\nTo generate a new token, given a key and payload:\n\n1. Take note of the current time for the timestamp field.\n2. Construct the token body.\n\n    ```\n    timestamp (64-bit big-endian unsigned integer) || payload (? bytes)\n    ```\n\n3. Generate a random nonce.\n4. Construct the AAD by concatenating the version indicator and the nonce:\n\n    ```\n    "v1:" (ASCII) || nonce (24 bytes)\n    ```\n\n5. Calculate the ciphertext and authentication tag by encrypting the token body\n   using __XChaCha20-Poly1305__. Pass the result of the previous step as AAD.\n6. Concatenate the nonce, ciphertext and authentication tag. Encode the result\n   using __base64url__ as defined in [RFC\n   4648](https://datatracker.ietf.org/doc/html/rfc4648#section-5). Strip any\n   padding from the result that may have been added by the base64 encoding.\n\n    ```\n    nonce (24 bytes) || ciphertext (? bytes) || tag (16 bytes)\n    ```\n\n7. Construct the token by concatenating the version indicator and the result of\n   the previous step:\n\n    ```\n    "v1:" || base64url(nonce || ciphertext || tag)\n    ```\n\n### Decoding\n\nTo decode a token, given a key:\n\n1. Split the token on the ``:`` character into two parts called ``version`` and\n   ``body``. Verify that the result of the split is exactly 2 parts.\n2. Verify that the ``version`` is equal to exactly ``"v1"``.\n3. Decode the ``body`` using __base64url__ as defined in [RFC\n   4648](https://datatracker.ietf.org/doc/html/rfc4648#section-5). If the base64\n   library you\'re using expects padding, artificially add it beforehand.\n4. Split up the decoded body as follows:\n\n    ```\n    nonce (24 bytes) | ciphertext (? bytes) | tag (16 bytes)\n    ```\n\n5. Construct the AAD by concatenating the version indicator and the nonce:\n\n    ```\n    "v1:" (ASCII) || nonce (24 bytes)\n    ```\n\n5. Decrypt the ciphertext using __XChaCha20-Poly1305__ with the given key, and\n   the nonce and tag obtained in the previous steps.\n6. Deconstruct the resulting plaintext as follows:\n\n    ```\n    timestamp (64-bit big-endian unsigned integer) | payload (? bytes)\n    ```\n\n7. Optionally, if the user has configured a TTL, verify that the token hasn\'t\n   expired by adding the TTL to the timestamp and compare the result with the\n   current time.\n\n## Security\n\nThe format and implementation have not undergone a third-party security audit.\nThe goal is to keep Menta so simple that you can confidently say: "I trust\nmenta, because I trust XChaCha20-Poly1305".\n\nMenta depends on [libsodium](https://doc.libsodium.org/) through\n[PyNaCl](https://pynacl.readthedocs.io/en/latest/). It does not implement any\ncryptographic primitives itself.\n',
    'author': 'Alexander Bakker',
    'author_email': 'ab@alexbakker.me',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/alexbakker/menta',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
