"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretTargetAttachment = exports.AttachmentTargetType = exports.Secret = void 0;
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
const rotation_schedule_1 = require("./rotation-schedule");
const secretsmanager = require("./secretsmanager.generated");
/**
 * The common behavior of Secrets. Users should not use this class directly, and instead use ``Secret``.
 */
class SecretBase extends core_1.Resource {
    grantRead(grantee, versionStages) {
        // @see https://docs.aws.amazon.com/fr_fr/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:GetSecretValue', 'secretsmanager:DescribeSecret'],
            resourceArns: [this.secretArn],
            scope: this,
        });
        if (versionStages != null && result.principalStatement) {
            result.principalStatement.addCondition('ForAnyValue:StringEquals', {
                'secretsmanager:VersionStage': versionStages,
            });
        }
        if (this.encryptionKey) {
            // @see https://docs.aws.amazon.com/fr_fr/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantDecrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    grantWrite(grantee) {
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/auth-and-access_identity-based-policies.html
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: ['secretsmanager:PutSecretValue', 'secretsmanager:UpdateSecret'],
            resourceArns: [this.secretArn],
            scope: this,
        });
        if (this.encryptionKey) {
            // See https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html
            this.encryptionKey.grantEncrypt(new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, grantee.grantPrincipal));
        }
        return result;
    }
    get secretValue() {
        return this.secretValueFromJson('');
    }
    secretValueFromJson(jsonField) {
        return core_1.SecretValue.secretsManager(this.secretArn, { jsonField });
    }
    addRotationSchedule(id, options) {
        return new rotation_schedule_1.RotationSchedule(this, id, {
            secret: this,
            ...options,
        });
    }
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.ResourcePolicy(this, 'Policy', { secret: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    denyAccountRootDelete() {
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: ['secretsmanager:DeleteSecret'],
            effect: iam.Effect.DENY,
            resources: ['*'],
            principals: [new iam.AccountRootPrincipal()],
        }));
    }
}
/**
 * Creates a new secret in AWS SecretsManager.
 */
class Secret extends SecretBase {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.secretName,
        });
        this.autoCreatePolicy = true;
        if (props.generateSecretString &&
            (props.generateSecretString.secretStringTemplate || props.generateSecretString.generateStringKey) &&
            !(props.generateSecretString.secretStringTemplate && props.generateSecretString.generateStringKey)) {
            throw new Error('`secretStringTemplate` and `generateStringKey` must be specified together.');
        }
        const resource = new secretsmanager.CfnSecret(this, 'Resource', {
            description: props.description,
            kmsKeyId: props.encryptionKey && props.encryptionKey.keyArn,
            generateSecretString: props.generateSecretString || {},
            name: this.physicalName,
        });
        if (props.removalPolicy) {
            resource.applyRemovalPolicy(props.removalPolicy);
        }
        this.secretArn = this.getResourceArnAttribute(resource.ref, {
            service: 'secretsmanager',
            resource: 'secret',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.encryptionKey = props.encryptionKey;
        // @see https://docs.aws.amazon.com/kms/latest/developerguide/services-secrets-manager.html#asm-authz
        const principle = new kms.ViaServicePrincipal(`secretsmanager.${core_1.Stack.of(this).region}.amazonaws.com`, new iam.AccountPrincipal(core_1.Stack.of(this).account));
        (_a = this.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncryptDecrypt(principle);
        (_b = this.encryptionKey) === null || _b === void 0 ? void 0 : _b.grant(principle, 'kms:CreateGrant', 'kms:DescribeKey');
    }
    static fromSecretArn(scope, id, secretArn) {
        return Secret.fromSecretAttributes(scope, id, { secretArn });
    }
    /**
     * Import an existing secret into the Stack.
     *
     * @param scope the scope of the import.
     * @param id    the ID of the imported Secret in the construct tree.
     * @param attrs the attributes of the imported secret.
     */
    static fromSecretAttributes(scope, id, attrs) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.encryptionKey = attrs.encryptionKey;
                this.secretArn = attrs.secretArn;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target attachment to the secret.
     *
     * @returns an AttachedSecret
     *
     * @deprecated use `attach()` instead
     */
    addTargetAttachment(id, options) {
        return new SecretTargetAttachment(this, id, {
            secret: this,
            ...options,
        });
    }
    /**
     * Attach a target to this secret
     *
     * @param target The target to attach
     * @returns An attached secret
     */
    attach(target) {
        const id = 'Attachment';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('Secret is already attached to a target.');
        }
        return new SecretTargetAttachment(this, id, {
            secret: this,
            target,
        });
    }
}
exports.Secret = Secret;
/**
 * The type of service or database that's being associated with the secret.
 */
var AttachmentTargetType;
(function (AttachmentTargetType) {
    /**
     * A database instance
     *
     * @deprecated use RDS_DB_INSTANCE instead
     */
    AttachmentTargetType["INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * A database cluster
     *
     * @deprecated use RDS_DB_CLUSTER instead
     */
    AttachmentTargetType["CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::RDS::DBInstance
     */
    AttachmentTargetType["RDS_DB_INSTANCE"] = "AWS::RDS::DBInstance";
    /**
     * AWS::RDS::DBCluster
     */
    AttachmentTargetType["RDS_DB_CLUSTER"] = "AWS::RDS::DBCluster";
    /**
     * AWS::Redshift::Cluster
     */
    AttachmentTargetType["REDSHIFT_CLUSTER"] = "AWS::Redshift::Cluster";
    /**
     * AWS::DocDB::DBInstance
     */
    AttachmentTargetType["DOCDB_DB_INSTANCE"] = "AWS::DocDB::DBInstance";
    /**
     * AWS::DocDB::DBCluster
     */
    AttachmentTargetType["DOCDB_DB_CLUSTER"] = "AWS::DocDB::DBCluster";
})(AttachmentTargetType = exports.AttachmentTargetType || (exports.AttachmentTargetType = {}));
/**
 * An attached secret.
 */
class SecretTargetAttachment extends SecretBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.autoCreatePolicy = true;
        const attachment = new secretsmanager.CfnSecretTargetAttachment(this, 'Resource', {
            secretId: props.secret.secretArn,
            targetId: props.target.asSecretAttachmentTarget().targetId,
            targetType: props.target.asSecretAttachmentTarget().targetType,
        });
        this.encryptionKey = props.secret.encryptionKey;
        // This allows to reference the secret after attachment (dependency).
        this.secretArn = attachment.ref;
        this.secretTargetAttachmentSecretArn = attachment.ref;
    }
    static fromSecretTargetAttachmentSecretArn(scope, id, secretTargetAttachmentSecretArn) {
        class Import extends SecretBase {
            constructor() {
                super(...arguments);
                this.secretArn = secretTargetAttachmentSecretArn;
                this.secretTargetAttachmentSecretArn = secretTargetAttachmentSecretArn;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
}
exports.SecretTargetAttachment = SecretTargetAttachment;
//# sourceMappingURL=data:application/json;base64,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