/*******************************************************************************
 *
 * Copyright (c) 2020 Oskar Enoksson. All rights reserved.
 * Licensed under the MIT license. See LICENSE file in the project root for details.
 *
 * Description:
 * Base (generic) implementation of arithmetic on infinite polynomials over GF(2)
 *
 *******************************************************************************/

#define PY_SSIZE_T_CLEAN
#include <Python.h>

#include <string.h>
#include <stdio.h>

#ifdef DEBUG_PYGF2X
#define DBG_PRINTF(...) printf(__VA_ARGS__)
#else
#define DBG_PRINTF(...)
#endif

static const unsigned int BITS_PER_DIGIT = sizeof(digit)*8*15/16;
static const unsigned int N5_PER_DIGIT = sizeof(digit)*8*3/16;

/*
static const uint16_t squares_8[256] = {
    0x0000,0x0001,0x0004,0x0005,0x0010,0x0011,0x0014,0x0015,0x0040,0x0041,0x0044,0x0045,0x0050,0x0051,0x0054,0x0055,
    0x0100,0x0101,0x0104,0x0105,0x0110,0x0111,0x0114,0x0115,0x0140,0x0141,0x0144,0x0145,0x0150,0x0151,0x0154,0x0155,
    0x0400,0x0401,0x0404,0x0405,0x0410,0x0411,0x0414,0x0415,0x0440,0x0441,0x0444,0x0445,0x0450,0x0451,0x0454,0x0455,
    0x0500,0x0501,0x0504,0x0505,0x0510,0x0511,0x0514,0x0515,0x0540,0x0541,0x0544,0x0545,0x0550,0x0551,0x0554,0x0555,
    0x1000,0x1001,0x1004,0x1005,0x1010,0x1011,0x1014,0x1015,0x1040,0x1041,0x1044,0x1045,0x1050,0x1051,0x1054,0x1055,
    0x1100,0x1101,0x1104,0x1105,0x1110,0x1111,0x1114,0x1115,0x1140,0x1141,0x1144,0x1145,0x1150,0x1151,0x1154,0x1155,
    0x1400,0x1401,0x1404,0x1405,0x1410,0x1411,0x1414,0x1415,0x1440,0x1441,0x1444,0x1445,0x1450,0x1451,0x1454,0x1455,
    0x1500,0x1501,0x1504,0x1505,0x1510,0x1511,0x1514,0x1515,0x1540,0x1541,0x1544,0x1545,0x1550,0x1551,0x1554,0x1555,
    0x4000,0x4001,0x4004,0x4005,0x4010,0x4011,0x4014,0x4015,0x4040,0x4041,0x4044,0x4045,0x4050,0x4051,0x4054,0x4055,
    0x4100,0x4101,0x4104,0x4105,0x4110,0x4111,0x4114,0x4115,0x4140,0x4141,0x4144,0x4145,0x4150,0x4151,0x4154,0x4155,
    0x4400,0x4401,0x4404,0x4405,0x4410,0x4411,0x4414,0x4415,0x4440,0x4441,0x4444,0x4445,0x4450,0x4451,0x4454,0x4455,
    0x4500,0x4501,0x4504,0x4505,0x4510,0x4511,0x4514,0x4515,0x4540,0x4541,0x4544,0x4545,0x4550,0x4551,0x4554,0x4555,
    0x5000,0x5001,0x5004,0x5005,0x5010,0x5011,0x5014,0x5015,0x5040,0x5041,0x5044,0x5045,0x5050,0x5051,0x5054,0x5055,
    0x5100,0x5101,0x5104,0x5105,0x5110,0x5111,0x5114,0x5115,0x5140,0x5141,0x5144,0x5145,0x5150,0x5151,0x5154,0x5155,
    0x5400,0x5401,0x5404,0x5405,0x5410,0x5411,0x5414,0x5415,0x5440,0x5441,0x5444,0x5445,0x5450,0x5451,0x5454,0x5455,
    0x5500,0x5501,0x5504,0x5505,0x5510,0x5511,0x5514,0x5515,0x5540,0x5541,0x5544,0x5545,0x5550,0x5551,0x5554,0x5555,
};
*/

// Multiplication table up to 31x31, that is, 5-bit chunks.
static const uint16_t mul_5_5[32][32] = {
    {0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,
     0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000,0x000},
    {0x000,0x001,0x002,0x003,0x004,0x005,0x006,0x007,0x008,0x009,0x00a,0x00b,0x00c,0x00d,0x00e,0x00f,
     0x010,0x011,0x012,0x013,0x014,0x015,0x016,0x017,0x018,0x019,0x01a,0x01b,0x01c,0x01d,0x01e,0x01f},
    {0x000,0x002,0x004,0x006,0x008,0x00a,0x00c,0x00e,0x010,0x012,0x014,0x016,0x018,0x01a,0x01c,0x01e,
     0x020,0x022,0x024,0x026,0x028,0x02a,0x02c,0x02e,0x030,0x032,0x034,0x036,0x038,0x03a,0x03c,0x03e},
    {0x000,0x003,0x006,0x005,0x00c,0x00f,0x00a,0x009,0x018,0x01b,0x01e,0x01d,0x014,0x017,0x012,0x011,
     0x030,0x033,0x036,0x035,0x03c,0x03f,0x03a,0x039,0x028,0x02b,0x02e,0x02d,0x024,0x027,0x022,0x021},
    {0x000,0x004,0x008,0x00c,0x010,0x014,0x018,0x01c,0x020,0x024,0x028,0x02c,0x030,0x034,0x038,0x03c,
     0x040,0x044,0x048,0x04c,0x050,0x054,0x058,0x05c,0x060,0x064,0x068,0x06c,0x070,0x074,0x078,0x07c},
    {0x000,0x005,0x00a,0x00f,0x014,0x011,0x01e,0x01b,0x028,0x02d,0x022,0x027,0x03c,0x039,0x036,0x033,
     0x050,0x055,0x05a,0x05f,0x044,0x041,0x04e,0x04b,0x078,0x07d,0x072,0x077,0x06c,0x069,0x066,0x063},
    {0x000,0x006,0x00c,0x00a,0x018,0x01e,0x014,0x012,0x030,0x036,0x03c,0x03a,0x028,0x02e,0x024,0x022,
     0x060,0x066,0x06c,0x06a,0x078,0x07e,0x074,0x072,0x050,0x056,0x05c,0x05a,0x048,0x04e,0x044,0x042},
    {0x000,0x007,0x00e,0x009,0x01c,0x01b,0x012,0x015,0x038,0x03f,0x036,0x031,0x024,0x023,0x02a,0x02d,
     0x070,0x077,0x07e,0x079,0x06c,0x06b,0x062,0x065,0x048,0x04f,0x046,0x041,0x054,0x053,0x05a,0x05d},
    {0x000,0x008,0x010,0x018,0x020,0x028,0x030,0x038,0x040,0x048,0x050,0x058,0x060,0x068,0x070,0x078,
     0x080,0x088,0x090,0x098,0x0a0,0x0a8,0x0b0,0x0b8,0x0c0,0x0c8,0x0d0,0x0d8,0x0e0,0x0e8,0x0f0,0x0f8},
    {0x000,0x009,0x012,0x01b,0x024,0x02d,0x036,0x03f,0x048,0x041,0x05a,0x053,0x06c,0x065,0x07e,0x077,
     0x090,0x099,0x082,0x08b,0x0b4,0x0bd,0x0a6,0x0af,0x0d8,0x0d1,0x0ca,0x0c3,0x0fc,0x0f5,0x0ee,0x0e7},
    {0x000,0x00a,0x014,0x01e,0x028,0x022,0x03c,0x036,0x050,0x05a,0x044,0x04e,0x078,0x072,0x06c,0x066,
     0x0a0,0x0aa,0x0b4,0x0be,0x088,0x082,0x09c,0x096,0x0f0,0x0fa,0x0e4,0x0ee,0x0d8,0x0d2,0x0cc,0x0c6},
    {0x000,0x00b,0x016,0x01d,0x02c,0x027,0x03a,0x031,0x058,0x053,0x04e,0x045,0x074,0x07f,0x062,0x069,
     0x0b0,0x0bb,0x0a6,0x0ad,0x09c,0x097,0x08a,0x081,0x0e8,0x0e3,0x0fe,0x0f5,0x0c4,0x0cf,0x0d2,0x0d9},
    {0x000,0x00c,0x018,0x014,0x030,0x03c,0x028,0x024,0x060,0x06c,0x078,0x074,0x050,0x05c,0x048,0x044,
     0x0c0,0x0cc,0x0d8,0x0d4,0x0f0,0x0fc,0x0e8,0x0e4,0x0a0,0x0ac,0x0b8,0x0b4,0x090,0x09c,0x088,0x084},
    {0x000,0x00d,0x01a,0x017,0x034,0x039,0x02e,0x023,0x068,0x065,0x072,0x07f,0x05c,0x051,0x046,0x04b,
     0x0d0,0x0dd,0x0ca,0x0c7,0x0e4,0x0e9,0x0fe,0x0f3,0x0b8,0x0b5,0x0a2,0x0af,0x08c,0x081,0x096,0x09b},
    {0x000,0x00e,0x01c,0x012,0x038,0x036,0x024,0x02a,0x070,0x07e,0x06c,0x062,0x048,0x046,0x054,0x05a,
     0x0e0,0x0ee,0x0fc,0x0f2,0x0d8,0x0d6,0x0c4,0x0ca,0x090,0x09e,0x08c,0x082,0x0a8,0x0a6,0x0b4,0x0ba},
    {0x000,0x00f,0x01e,0x011,0x03c,0x033,0x022,0x02d,0x078,0x077,0x066,0x069,0x044,0x04b,0x05a,0x055,
     0x0f0,0x0ff,0x0ee,0x0e1,0x0cc,0x0c3,0x0d2,0x0dd,0x088,0x087,0x096,0x099,0x0b4,0x0bb,0x0aa,0x0a5},
    {0x000,0x010,0x020,0x030,0x040,0x050,0x060,0x070,0x080,0x090,0x0a0,0x0b0,0x0c0,0x0d0,0x0e0,0x0f0,
     0x100,0x110,0x120,0x130,0x140,0x150,0x160,0x170,0x180,0x190,0x1a0,0x1b0,0x1c0,0x1d0,0x1e0,0x1f0},
    {0x000,0x011,0x022,0x033,0x044,0x055,0x066,0x077,0x088,0x099,0x0aa,0x0bb,0x0cc,0x0dd,0x0ee,0x0ff,
     0x110,0x101,0x132,0x123,0x154,0x145,0x176,0x167,0x198,0x189,0x1ba,0x1ab,0x1dc,0x1cd,0x1fe,0x1ef},
    {0x000,0x012,0x024,0x036,0x048,0x05a,0x06c,0x07e,0x090,0x082,0x0b4,0x0a6,0x0d8,0x0ca,0x0fc,0x0ee,
     0x120,0x132,0x104,0x116,0x168,0x17a,0x14c,0x15e,0x1b0,0x1a2,0x194,0x186,0x1f8,0x1ea,0x1dc,0x1ce},
    {0x000,0x013,0x026,0x035,0x04c,0x05f,0x06a,0x079,0x098,0x08b,0x0be,0x0ad,0x0d4,0x0c7,0x0f2,0x0e1,
     0x130,0x123,0x116,0x105,0x17c,0x16f,0x15a,0x149,0x1a8,0x1bb,0x18e,0x19d,0x1e4,0x1f7,0x1c2,0x1d1},
    {0x000,0x014,0x028,0x03c,0x050,0x044,0x078,0x06c,0x0a0,0x0b4,0x088,0x09c,0x0f0,0x0e4,0x0d8,0x0cc,
     0x140,0x154,0x168,0x17c,0x110,0x104,0x138,0x12c,0x1e0,0x1f4,0x1c8,0x1dc,0x1b0,0x1a4,0x198,0x18c},
    {0x000,0x015,0x02a,0x03f,0x054,0x041,0x07e,0x06b,0x0a8,0x0bd,0x082,0x097,0x0fc,0x0e9,0x0d6,0x0c3,
     0x150,0x145,0x17a,0x16f,0x104,0x111,0x12e,0x13b,0x1f8,0x1ed,0x1d2,0x1c7,0x1ac,0x1b9,0x186,0x193},
    {0x000,0x016,0x02c,0x03a,0x058,0x04e,0x074,0x062,0x0b0,0x0a6,0x09c,0x08a,0x0e8,0x0fe,0x0c4,0x0d2,
     0x160,0x176,0x14c,0x15a,0x138,0x12e,0x114,0x102,0x1d0,0x1c6,0x1fc,0x1ea,0x188,0x19e,0x1a4,0x1b2},
    {0x000,0x017,0x02e,0x039,0x05c,0x04b,0x072,0x065,0x0b8,0x0af,0x096,0x081,0x0e4,0x0f3,0x0ca,0x0dd,
     0x170,0x167,0x15e,0x149,0x12c,0x13b,0x102,0x115,0x1c8,0x1df,0x1e6,0x1f1,0x194,0x183,0x1ba,0x1ad},
    {0x000,0x018,0x030,0x028,0x060,0x078,0x050,0x048,0x0c0,0x0d8,0x0f0,0x0e8,0x0a0,0x0b8,0x090,0x088,
     0x180,0x198,0x1b0,0x1a8,0x1e0,0x1f8,0x1d0,0x1c8,0x140,0x158,0x170,0x168,0x120,0x138,0x110,0x108},
    {0x000,0x019,0x032,0x02b,0x064,0x07d,0x056,0x04f,0x0c8,0x0d1,0x0fa,0x0e3,0x0ac,0x0b5,0x09e,0x087,
     0x190,0x189,0x1a2,0x1bb,0x1f4,0x1ed,0x1c6,0x1df,0x158,0x141,0x16a,0x173,0x13c,0x125,0x10e,0x117},
    {0x000,0x01a,0x034,0x02e,0x068,0x072,0x05c,0x046,0x0d0,0x0ca,0x0e4,0x0fe,0x0b8,0x0a2,0x08c,0x096,
     0x1a0,0x1ba,0x194,0x18e,0x1c8,0x1d2,0x1fc,0x1e6,0x170,0x16a,0x144,0x15e,0x118,0x102,0x12c,0x136},
    {0x000,0x01b,0x036,0x02d,0x06c,0x077,0x05a,0x041,0x0d8,0x0c3,0x0ee,0x0f5,0x0b4,0x0af,0x082,0x099,
     0x1b0,0x1ab,0x186,0x19d,0x1dc,0x1c7,0x1ea,0x1f1,0x168,0x173,0x15e,0x145,0x104,0x11f,0x132,0x129},
    {0x000,0x01c,0x038,0x024,0x070,0x06c,0x048,0x054,0x0e0,0x0fc,0x0d8,0x0c4,0x090,0x08c,0x0a8,0x0b4,
     0x1c0,0x1dc,0x1f8,0x1e4,0x1b0,0x1ac,0x188,0x194,0x120,0x13c,0x118,0x104,0x150,0x14c,0x168,0x174},
    {0x000,0x01d,0x03a,0x027,0x074,0x069,0x04e,0x053,0x0e8,0x0f5,0x0d2,0x0cf,0x09c,0x081,0x0a6,0x0bb,
     0x1d0,0x1cd,0x1ea,0x1f7,0x1a4,0x1b9,0x19e,0x183,0x138,0x125,0x102,0x11f,0x14c,0x151,0x176,0x16b},
    {0x000,0x01e,0x03c,0x022,0x078,0x066,0x044,0x05a,0x0f0,0x0ee,0x0cc,0x0d2,0x088,0x096,0x0b4,0x0aa,
     0x1e0,0x1fe,0x1dc,0x1c2,0x198,0x186,0x1a4,0x1ba,0x110,0x10e,0x12c,0x132,0x168,0x176,0x154,0x14a},
    {0x000,0x01f,0x03e,0x021,0x07c,0x063,0x042,0x05d,0x0f8,0x0e7,0x0c6,0x0d9,0x084,0x09b,0x0ba,0x0a5,
     0x1f0,0x1ef,0x1ce,0x1d1,0x18c,0x193,0x1b2,0x1ad,0x108,0x117,0x136,0x129,0x174,0x16b,0x14a,0x155},
};


static inline int nbits(PyLongObject *integer) {
    // return 1-based index of the most significant non-zero bit, or 0 if all bits are zero
    return _PyLong_NumBits((PyObject *)integer);
}

static uint16_t
mul_15_15(uint16_t l, uint16_t r) {
    // Multiply two unsigned 15-bit integers (stored in uint16_t)
    uint8_t l0 = l&0x1f;
    l>>=5;
    uint8_t l1 = l&0x1f;
    l>>=5;
    uint8_t l2 = l&0x1f;
    uint8_t r0 = r&0x1f;
    r>>=5;
    uint8_t r1 = r&0x1f;
    r>>=5;
    uint8_t r2 = r&0x1f;
    
    uint32_t p = mul_5_5[l2][r2];
    p <<= 5;
    p ^= mul_5_5[l1][r2] ^ mul_5_5[l2][r1];
    p <<= 5;
    p ^= mul_5_5[l0][r2] ^ mul_5_5[l1][r1] ^ mul_5_5[l2][r0];
    p <<= 5;
    p ^= mul_5_5[l0][r1] ^ mul_5_5[l1][r0];
    p <<= 5;
    p ^= mul_5_5[l0][r0];
        
    return p;
}

static uint64_t
mul_30_30(uint32_t l, uint32_t r) {
    // Multiply two 30-bit unsigned integers (stored in uint32_t)
    uint16_t l0 = l&0x3ff;
    l>>=10;
    uint16_t l1 = l&0x3ff;
    uint16_t r0 = r&0x3ff;
    r>>=10;
    uint16_t r1 = r&0x3ff;

    uint64_t p = mul_15_15(l1, r1);
    p<<=10;
    p ^= mul_15_15(l0, r1) ^ mul_15_15(l1, r0);
    p<<=10;
    p ^= mul_15_15(l0, r0);

    return p;
}

static PyObject *
pygf2x_mul(PyObject *self, PyObject *args) {
    // Multiply two Python integers, interpreted as polynomials over GF(2)
    (void)self;

    PyLongObject *fl, *fr;
    if (!PyArg_ParseTuple(args, "OO", &fl, &fr)) {
        PyErr_SetString(PyExc_TypeError, "Failed to parse arguments");
        return NULL;
    }

    if( ! PyLong_Check(fl) ||
	! PyLong_Check(fr) ) {
        PyErr_SetString(PyExc_TypeError, "Both arguments must be integers");
        return NULL;
    }
    if(((PyVarObject *)fl)->ob_size < 0 ||
       ((PyVarObject *)fr)->ob_size < 0) {
        PyErr_SetString(PyExc_ValueError, "Both arguments must be non-negative");
        return NULL;
    }

    int nbits_l = nbits(fl);
    int nbits_r = nbits(fr);

    int n5_l = (nbits_l+4)/5;
    int n5_r = (nbits_r+4)/5;
    int n5_p = n5_l+n5_r;

    int ndigs_p = (n5_l+n5_r+N5_PER_DIGIT-1)/N5_PER_DIGIT;
    digit result[ndigs_p];
    memset(result,0,ndigs_p*sizeof(digit));
    
    DBG_PRINTF("Bits per digit   : %-4d\n",BITS_PER_DIGIT);
    DBG_PRINTF("Left factor bits : %-4d\n",nbits_l);
    DBG_PRINTF("Right factor bits: %-4d\n",nbits_r);
    DBG_PRINTF("Product digits ? : %-4d\n",ndigs_p);

    for(int i=0; i<n5_p; i++) {
        uint16_t p5 = 0;
	int je = i;
	if(je > n5_l)
           je = n5_l;
	int jb = 0;
	if(i-jb > n5_r)
            jb = i-n5_r;
        for(int jl=jb; jl<=je; jl++) {
            int jr = i-jl;
	    int jdl = jl/N5_PER_DIGIT;
	    int jdr = jr/N5_PER_DIGIT;
	    uint8_t d5_l = ((fl->ob_digit[jdl])>>(5*(jl%N5_PER_DIGIT)))&0x1f;
	    uint8_t d5_r = ((fr->ob_digit[jdr])>>(5*(jr%N5_PER_DIGIT)))&0x1f;
	    uint16_t pi = mul_5_5[d5_l][d5_r];
	    p5 ^= pi;
	}
	uint8_t p5l = p5&0x1f;
	uint8_t p5h = p5>>5;

	result[i/N5_PER_DIGIT] ^= p5l<<(5*(i%N5_PER_DIGIT));
	result[(i+1)/N5_PER_DIGIT] ^= p5h<<(5*((i+1)%N5_PER_DIGIT));
    }

    // Remove zero digits
    while(result[ndigs_p-1]==0)
        ndigs_p--;
    DBG_PRINTF("Product digits ! : %-4d\n",ndigs_p);

    //for(int i=0; i<ndigs_p; i++)
    //  DBG_PRINTF("digit %d = %08x\n", i, result[i]);

    PyLongObject *p = _PyLong_New(ndigs_p);
    for(int id=0; id<ndigs_p; id++)
        p->ob_digit[id] = result[id];
      
    return Py_BuildValue("O",p);
}

static PyObject *
pygf2x_div(PyObject *self, PyObject *args) {
    // Divide two Python integers, interpreted as polynomials over GF(2)
    // Return quotient and remainder
    (void)self;

    PyLongObject *numerator, *denominator;
    if (!PyArg_ParseTuple(args, "OO", &numerator, &denominator)) {
        PyErr_SetString(PyExc_TypeError, "Failed to parse arguments");
        return NULL;
    }

    if( ! PyLong_Check(numerator) ||
        ! PyLong_Check(denominator) ) {
        PyErr_SetString(PyExc_TypeError, "Both arguments must be integers");
        return NULL;
    }
    if(((PyVarObject *)numerator)->ob_size < 0 ||
       ((PyVarObject *)denominator)->ob_size < 0) {
        PyErr_SetString(PyExc_ValueError, "Both arguments must be non-negative");
        return NULL;
    }

    int nbits_d = nbits(denominator);
    int nbits_n = nbits(numerator);
    int ndigs_n = (nbits_n+BITS_PER_DIGIT-1)/BITS_PER_DIGIT;

    if(nbits_d == 0) {
        PyErr_SetString(PyExc_ZeroDivisionError, "Denominator is zero");
        return NULL;
    }
    int nbits_q = nbits_n-nbits_d+1 > 0 ? nbits_n-nbits_d+1 : 0;
    int nbits_r = nbits_d-1;
    int ndigs_q = (nbits_q+BITS_PER_DIGIT-1)/BITS_PER_DIGIT;
    int ndigs_r = (nbits_r+BITS_PER_DIGIT-1)/BITS_PER_DIGIT;

    digit q_digits[ndigs_q];
    memset(q_digits,0,ndigs_q*sizeof(digit));
    digit r_digits[ndigs_n]; // Initialize to numerator
    memcpy(r_digits, numerator->ob_digit, ndigs_n*sizeof(digit));
    
    DBG_PRINTF("Bits per digit  : %-4d\n",BITS_PER_DIGIT);
    DBG_PRINTF("Numerator bits  : %-4d\n",nbits_n);
    DBG_PRINTF("Denominator bits: %-4d\n",nbits_d);
    DBG_PRINTF("Quotient bits  ?: %-4d\n",nbits_q);
    DBG_PRINTF("Remainder bits ?: %-4d\n",nbits_r);

    for(int nbi = nbits_n-1; nbi >= nbits_d-1; nbi--) {
        int ndi = (nbi/BITS_PER_DIGIT);    // Digit position
        int ndbi = nbi-ndi*BITS_PER_DIGIT; // Bit position in digit
        if(r_digits[ndi] & (1<<ndbi)) {
            // Numerator bit is set. Set quotient bit and subtract denominator
	    int qbi  = nbi - nbits_d +1;
	    int qdi  =  (qbi/BITS_PER_DIGIT);   // Digit position
	    int qdbi = qbi-qdi*BITS_PER_DIGIT;  // Bit position in digit
	    q_digits[qdi] |= (1<<qdbi);
            for(int dbi  = nbits_d-1; dbi >= 0 ; dbi--) {
                int ddi  = (dbi/BITS_PER_DIGIT);   // Digit position
                int ddbi = dbi-ddi*BITS_PER_DIGIT; // Bit position in digit
		int rbi  = nbi - (nbits_d-1 - dbi);
		int rdi  = (rbi/BITS_PER_DIGIT);   // Digit position
                int rdbi = rbi-rdi*BITS_PER_DIGIT; // Bit position in digit
		r_digits[rdi] ^= ((denominator->ob_digit[ddi]>>ddbi)&1)<<rdbi;
	    }
        }
    }

    
    // Remove leading zero digits
    while(q_digits[ndigs_q-1] == 0)
      ndigs_q -= 1;
    PyLongObject *q = _PyLong_New(ndigs_q);
    for(int i=0; i<ndigs_q; i++)
        q->ob_digit[i] = q_digits[i];

    // Remove leading zero digits
    while(r_digits[ndigs_r-1] == 0)
      ndigs_r -= 1;
    PyLongObject *r = _PyLong_New(ndigs_r);
    for(int i=0; i<ndigs_r; i++)
        r->ob_digit[i] = r_digits[i];

    return Py_BuildValue("OO", q, r);
}

PyMethodDef pygf2x_generic_functions[] =
{
    {
        "div",
        pygf2x_div,
        METH_VARARGS,
        "Divide two integers as polynomials over GF(2) (returns quotient and remainder)"
    },
    {
        "mul",
        pygf2x_mul,
        METH_VARARGS,
        "Multiply two integers as polynomials over GF(2)"
    },
    {
        NULL,                   // const char  *ml_name;  /* The name of the built-in function/method   */
        NULL,                   // PyCFunction ml_meth;   /* The C function that implements it          */
        0,                      // int         ml_flags;  /* Combination of METH_xxx flags, which mostly*/
                                //                        /* describe the args expected by the C func   */
        NULL                    // const char  *ml_doc;   /* The __doc__ attribute, or NULL             */
    }
};


struct PyModuleDef pygf2x_generic_module =
{
  // Python module definition
    .m_base = PyModuleDef_HEAD_INIT,
    .m_name = "pygf2x_generic",       // Name of the module.
    .m_doc  = NULL,                   // Docstring for the module - in this case empty.
    .m_size = -1,                     // Used by sub-interpreters, if you do not know what
                                      // it is then you do not need it, keep -1 .
    .m_methods = pygf2x_generic_functions,  // Structures of type `PyMethodDef` with functions
                                      // (or "methods") provided by the module.
    .m_slots = NULL,
    .m_traverse = NULL,
    .m_clear = NULL,
    .m_free = NULL
};


PyMODINIT_FUNC PyInit_pygf2x_generic(void)
{
  // Python module initialization
    PyObject *pygf2x = PyModule_Create(&pygf2x_generic_module);

    return pygf2x;
}
