"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const api = require("@aws-cdk/aws-apigateway");
const defaults = require("@aws-solutions-konstruk/core");
const core_1 = require("@aws-cdk/core");
class CognitoToApiGatewayToLambda extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the CognitoToApiGatewayToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {CognitoToApiGatewayToLambdaProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.fn = defaults.buildLambdaFunction(this, {
            deployLambda: props.deployLambda,
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        this.api = defaults.GlobalLambdaRestApi(this, this.fn, props.apiGatewayProps);
        this.userpool = defaults.buildUserPool(this, props.cognitoUserPoolProps);
        this.userpoolclient = defaults.buildUserPoolClient(this, this.userpool, props.cognitoUserPoolClientProps);
        const cfnAuthorizer = new api.CfnAuthorizer(this, 'CognitoAuthorizer', {
            restApiId: this.api.restApiId,
            type: 'COGNITO_USER_POOLS',
            providerArns: [this.userpool.userPoolArn],
            identitySource: "method.request.header.Authorization",
            name: "authorizer"
        });
        this.api.methods.forEach((apiMethod) => {
            // Leave the authorizer NONE for HTTP OPTIONS method, for the rest set it to COGNITO
            const child = apiMethod.node.findChild('Resource');
            if (apiMethod.httpMethod === 'OPTIONS') {
                child.addPropertyOverride('AuthorizationType', 'NONE');
            }
            else {
                child.addPropertyOverride('AuthorizationType', 'COGNITO_USER_POOLS');
                child.addPropertyOverride('AuthorizerId', { Ref: cfnAuthorizer.logicalId });
            }
        });
    }
    /**
     * @summary Returns an instance of api.RestApi created by the construct.
     * @returns {api.RestApi} Instance of RestApi created by the construct
     * @since 0.8.0
     * @access public
     */
    restApi() {
        return this.api;
    }
    /**
     * @summary Returns an instance of lambda.Function created by the construct.
     * @returns {lambda.Function} Instance of Function created by the construct
     * @since 0.8.0
     * @access public
     */
    lambdaFunction() {
        return this.fn;
    }
    /**
     * @summary Returns an instance of cognito.UserPool created by the construct.
     * @returns {cognito.UserPool} Instance of UserPool created by the construct
     * @since 0.8.0
     * @access public
     */
    userPool() {
        return this.userpool;
    }
    /**
     * @summary Returns an instance of cognito.UserPoolClient created by the construct.
     * @returns {cognito.UserPoolClient} Instance of UserPoolClient created by the construct
     * @since 0.8.0
     * @access public
     */
    userPoolClient() {
        return this.userpoolclient;
    }
}
exports.CognitoToApiGatewayToLambda = CognitoToApiGatewayToLambda;
//# sourceMappingURL=data:application/json;base64,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