"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ComputeEnvironment = exports.AllocationStrategy = exports.ComputeResourceType = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const batch_generated_1 = require("./batch.generated");
/**
 * (experimental) Property to specify if the compute environment uses On-Demand or SpotFleet compute resources.
 *
 * @experimental
 */
var ComputeResourceType;
(function (ComputeResourceType) {
    ComputeResourceType["ON_DEMAND"] = "EC2";
    ComputeResourceType["SPOT"] = "SPOT";
})(ComputeResourceType = exports.ComputeResourceType || (exports.ComputeResourceType = {}));
/**
 * (experimental) Properties for how to prepare compute resources that are provisioned for a compute environment.
 *
 * @experimental
 */
var AllocationStrategy;
(function (AllocationStrategy) {
    AllocationStrategy["BEST_FIT"] = "BEST_FIT";
    AllocationStrategy["BEST_FIT_PROGRESSIVE"] = "BEST_FIT_PROGRESSIVE";
    AllocationStrategy["SPOT_CAPACITY_OPTIMIZED"] = "SPOT_CAPACITY_OPTIMIZED";
})(AllocationStrategy = exports.AllocationStrategy || (exports.AllocationStrategy = {}));
/**
 * (experimental) Batch Compute Environment.
 *
 * Defines a batch compute environment to run batch jobs on.
 *
 * @experimental
 */
class ComputeEnvironment extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = { enabled: true, managed: true }) {
        super(scope, id, {
            physicalName: props.computeEnvironmentName,
        });
        this.validateProps(props);
        const spotFleetRole = this.getSpotFleetRole(props);
        let computeResources;
        // Only allow compute resources to be set when using MANAGED type
        if (props.computeResources && this.isManaged(props)) {
            computeResources = {
                allocationStrategy: props.computeResources.allocationStrategy
                    || (props.computeResources.type === ComputeResourceType.SPOT
                        ? AllocationStrategy.SPOT_CAPACITY_OPTIMIZED
                        : AllocationStrategy.BEST_FIT),
                bidPercentage: props.computeResources.bidPercentage,
                desiredvCpus: props.computeResources.desiredvCpus,
                ec2KeyPair: props.computeResources.ec2KeyPair,
                imageId: props.computeResources.image && props.computeResources.image.getImage(this).imageId,
                instanceRole: props.computeResources.instanceRole
                    ? props.computeResources.instanceRole
                    : new iam.CfnInstanceProfile(this, 'Instance-Profile', {
                        roles: [new iam.Role(this, 'Ecs-Instance-Role', {
                                assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
                                managedPolicies: [
                                    iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'),
                                ],
                            }).roleName],
                    }).attrArn,
                instanceTypes: this.buildInstanceTypes(props.computeResources.instanceTypes),
                launchTemplate: props.computeResources.launchTemplate,
                maxvCpus: props.computeResources.maxvCpus || 256,
                minvCpus: props.computeResources.minvCpus || 0,
                securityGroupIds: this.buildSecurityGroupIds(props.computeResources.vpc, props.computeResources.securityGroups),
                spotIamFleetRole: spotFleetRole ? spotFleetRole.roleArn : undefined,
                subnets: props.computeResources.vpc.selectSubnets(props.computeResources.vpcSubnets).subnetIds,
                tags: props.computeResources.computeResourcesTags,
                type: props.computeResources.type || ComputeResourceType.ON_DEMAND,
            };
        }
        const computeEnvironment = new batch_generated_1.CfnComputeEnvironment(this, 'Resource', {
            computeEnvironmentName: this.physicalName,
            computeResources,
            serviceRole: props.serviceRole
                ? props.serviceRole.roleArn
                : new iam.Role(this, 'Resource-Service-Instance-Role', {
                    managedPolicies: [
                        iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSBatchServiceRole'),
                    ],
                    assumedBy: new iam.ServicePrincipal('batch.amazonaws.com'),
                }).roleArn,
            state: props.enabled === undefined ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            type: this.isManaged(props) ? 'MANAGED' : 'UNMANAGED',
        });
        if (props.computeResources && props.computeResources.vpc) {
            this.node.addDependency(props.computeResources.vpc);
        }
        this.computeEnvironmentArn = this.getResourceArnAttribute(computeEnvironment.ref, {
            service: 'batch',
            resource: 'compute-environment',
            resourceName: this.physicalName,
        });
        this.computeEnvironmentName = this.getResourceNameAttribute(computeEnvironment.ref);
    }
    /**
     * (experimental) Fetches an existing batch compute environment by its amazon resource name.
     *
     * @experimental
     */
    static fromComputeEnvironmentArn(scope, id, computeEnvironmentArn) {
        const stack = core_1.Stack.of(scope);
        const computeEnvironmentName = stack.parseArn(computeEnvironmentArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.computeEnvironmentArn = computeEnvironmentArn;
                this.computeEnvironmentName = computeEnvironmentName;
            }
        }
        return new Import(scope, id);
    }
    isManaged(props) {
        return props.managed === undefined ? true : props.managed;
    }
    /**
     * Validates the properties provided for a new batch compute environment.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (!this.isManaged(props) && props.computeResources !== undefined) {
            throw new Error('It is not allowed to set computeResources on an AWS unmanaged compute environment');
        }
        if (this.isManaged(props) && props.computeResources === undefined) {
            throw new Error('computeResources is missing but required on a managed compute environment');
        }
        // Setting a bid percentage is only allowed on SPOT resources +
        // Cannot use SPOT_CAPACITY_OPTIMIZED when using ON_DEMAND
        if (props.computeResources) {
            if (props.computeResources.type === ComputeResourceType.ON_DEMAND) {
                // VALIDATE FOR ON_DEMAND
                // Bid percentage is not allowed
                if (props.computeResources.bidPercentage !== undefined) {
                    throw new Error('Setting the bid percentage is only allowed for SPOT type resources on a batch compute environment');
                }
                // SPOT_CAPACITY_OPTIMIZED allocation is not allowed
                if (props.computeResources.allocationStrategy && props.computeResources.allocationStrategy === AllocationStrategy.SPOT_CAPACITY_OPTIMIZED) {
                    throw new Error('The SPOT_CAPACITY_OPTIMIZED allocation strategy is only allowed if the environment is a SPOT type compute environment');
                }
            }
            else {
                // VALIDATE FOR SPOT
                // Bid percentage must be from 0 - 100
                if (props.computeResources.bidPercentage !== undefined &&
                    (props.computeResources.bidPercentage < 0 || props.computeResources.bidPercentage > 100)) {
                    throw new Error('Bid percentage can only be a value between 0 and 100');
                }
            }
            if (props.computeResources.minvCpus) {
                // minvCpus cannot be less than 0
                if (props.computeResources.minvCpus < 0) {
                    throw new Error('Minimum vCpus for a batch compute environment cannot be less than 0');
                }
                // minvCpus cannot exceed max vCpus
                if (props.computeResources.maxvCpus &&
                    props.computeResources.minvCpus > props.computeResources.maxvCpus) {
                    throw new Error('Minimum vCpus cannot be greater than the maximum vCpus');
                }
            }
        }
    }
    buildInstanceTypes(instanceTypes) {
        if (instanceTypes === undefined) {
            return [
                'optimal',
            ];
        }
        return instanceTypes.map((type) => type.toString());
    }
    buildSecurityGroupIds(vpc, securityGroups) {
        if (securityGroups === undefined) {
            return [
                new ec2.SecurityGroup(this, 'Resource-Security-Group', { vpc }).securityGroupId,
            ];
        }
        return securityGroups.map((group) => group.securityGroupId);
    }
    /**
     * Generates an AWS IAM role for provisioning spotfleet resources
     * if the allocation strategy is set to BEST_FIT or not defined.
     *
     * @param props - the compute environment construct properties
     */
    getSpotFleetRole(props) {
        var _a;
        if (((_a = props.computeResources) === null || _a === void 0 ? void 0 : _a.allocationStrategy) && props.computeResources.allocationStrategy !== AllocationStrategy.BEST_FIT) {
            return undefined;
        }
        if (props.computeResources) {
            if (props.computeResources.spotFleetRole) {
                return props.computeResources.spotFleetRole;
            }
            else if (props.computeResources.type === ComputeResourceType.SPOT) {
                return iam.Role.fromRoleArn(this, 'Resource-SpotFleet-Role', `arn:${this.stack.partition}:iam::${this.stack.account}:role/aws-service-role/spotfleet.amazonaws.com/AWSServiceRoleForEC2SpotFleet`);
            }
        }
        return undefined;
    }
}
exports.ComputeEnvironment = ComputeEnvironment;
//# sourceMappingURL=data:application/json;base64,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