"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-konstruk/core");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
class S3ToLambda extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the S3ToLambda class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {S3ToLambdaProps} props - user provided props for the construct
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.fn = defaults.buildLambdaFunction(this, {
            deployLambda: props.deployLambda,
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        this.bucket = defaults.buildS3Bucket(this, {
            deployBucket: props.deployBucket,
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        // Create S3 trigger to invoke lambda function
        this.fn.addEventSource(new aws_lambda_event_sources_1.S3EventSource(this.bucket, defaults.S3EventSourceProps(props.s3EventSourceProps)));
        this.addCfnNagSuppress();
    }
    addCfnNagSuppress() {
        // Extract the CfnBucket from the s3Bucket
        const s3BucketResource = this.bucket.node.findChild('Resource');
        s3BucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W51',
                        reason: `This S3 bucket Bucket does not need a bucket policy`
                    }]
            }
        };
        const root = core_1.Stack.of(this);
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        const notificationsResourceHandler = root.node.tryFindChild(logicalId);
        const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
        const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
        // Extract the CfnFunction from the Function
        const fnResource = notificationsResourceHandler.node.findChild('Resource');
        fnResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W58',
                        reason: `Lambda function has permission to write CloudWatch Logs via AWSLambdaBasicExecutionRole policy attached to the lambda role`
                    }]
            }
        };
        // Extract the CfnPolicy from the iam.Policy
        const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
        policyResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W12',
                        reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
                    }]
            }
        };
    }
    /**
     * @summary Returns an instance of the lambda.Function created by the construct.
     * @returns {lambda.Function} Instance of the Function created by the construct.
     * @since 0.8.0
     * @access public
     */
    lambdaFunction() {
        return this.fn;
    }
    /**
     * @summary Returns an instance of the s3.Bucket created by the construct.
     * @returns {s3.Bucket} Instance of the Bucket created by the construct.
     * @since 0.8.0
     * @access public
     */
    s3Bucket() {
        return this.bucket;
    }
}
exports.S3ToLambda = S3ToLambda;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBSUgsd0NBQWlEO0FBRWpELHlEQUF5RDtBQUN6RCxnRkFBc0Y7QUF3RHRGLE1BQWEsVUFBVyxTQUFRLGdCQUFTO0lBSXZDOzs7Ozs7O09BT0c7SUFDSCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCO1FBQzlELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLEVBQUUsR0FBRyxRQUFRLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFO1lBQzNDLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtZQUNoQyxpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCO1lBQzFDLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxtQkFBbUI7U0FDL0MsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLE1BQU0sR0FBRyxRQUFRLENBQUMsYUFBYSxDQUFDLElBQUksRUFBRTtZQUN6QyxZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVk7WUFDaEMsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtZQUMxQyxXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7U0FDL0IsQ0FBQyxDQUFDO1FBRUgsOENBQThDO1FBQzlDLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLElBQUksd0NBQWEsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUNsRCxRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRTFELElBQUksQ0FBQyxpQkFBaUIsRUFBRSxDQUFDO0lBQzNCLENBQUM7SUFFTyxpQkFBaUI7UUFDdkIsMENBQTBDO1FBQzFDLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBaUIsQ0FBQztRQUVoRixnQkFBZ0IsQ0FBQyxVQUFVLENBQUMsUUFBUSxHQUFHO1lBQ25DLE9BQU8sRUFBRTtnQkFDTCxpQkFBaUIsRUFBRSxDQUFDO3dCQUNoQixFQUFFLEVBQUUsS0FBSzt3QkFDVCxNQUFNLEVBQUUscURBQXFEO3FCQUNoRSxDQUFDO2FBQ0w7U0FDSixDQUFDO1FBRUYsTUFBTSxJQUFJLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUM1QixNQUFNLFNBQVMsR0FBRyw0REFBNEQsQ0FBQztRQUMvRSxNQUFNLDRCQUE0QixHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBb0IsQ0FBQztRQUMxRixNQUFNLG9DQUFvQyxHQUFHLDRCQUE0QixDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFhLENBQUM7UUFDN0csTUFBTSxzQ0FBc0MsR0FBRyxvQ0FBb0MsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLGVBQWUsQ0FBZSxDQUFDO1FBRWxJLDRDQUE0QztRQUM1QyxNQUFNLFVBQVUsR0FBRyw0QkFBNEIsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBdUIsQ0FBQztRQUVqRyxVQUFVLENBQUMsVUFBVSxDQUFDLFFBQVEsR0FBRztZQUMvQixPQUFPLEVBQUU7Z0JBQ0wsaUJBQWlCLEVBQUUsQ0FBQzt3QkFDaEIsRUFBRSxFQUFFLEtBQUs7d0JBQ1QsTUFBTSxFQUFFLDRIQUE0SDtxQkFDdkksQ0FBQzthQUNMO1NBQ0YsQ0FBQztRQUVGLDRDQUE0QztRQUM1QyxNQUFNLGNBQWMsR0FBRyxzQ0FBc0MsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBa0IsQ0FBQztRQUUxRyxjQUFjLENBQUMsVUFBVSxDQUFDLFFBQVEsR0FBRztZQUNuQyxPQUFPLEVBQUU7Z0JBQ0wsaUJBQWlCLEVBQUUsQ0FBQzt3QkFDaEIsRUFBRSxFQUFFLEtBQUs7d0JBQ1QsTUFBTSxFQUFFLGtHQUFrRztxQkFDN0csQ0FBQzthQUNMO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDO0lBQ2pCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLFFBQVE7UUFDWCxPQUFPLElBQUksQ0FBQyxNQUFNLENBQUM7SUFDdkIsQ0FBQztDQUNGO0FBakdELGdDQWlHQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDE5IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IENvbnN0cnVjdCwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGRlZmF1bHRzIGZyb20gJ0Bhd3Mtc29sdXRpb25zLWtvbnN0cnVrL2NvcmUnO1xuaW1wb3J0IHsgUzNFdmVudFNvdXJjZVByb3BzLCBTM0V2ZW50U291cmNlIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYS1ldmVudC1zb3VyY2VzJztcblxuLyoqXG4gKiBAc3VtbWFyeSBUaGUgcHJvcGVydGllcyBmb3IgdGhlIFMzVG9MYW1iZGEgY2xhc3MuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUzNUb0xhbWJkYVByb3BzIHtcbiAgLyoqXG4gICAqIFdoZXRoZXIgdG8gY3JlYXRlIGEgbmV3IGxhbWJkYSBmdW5jdGlvbiBvciB1c2UgYW4gZXhpc3RpbmcgbGFtYmRhIGZ1bmN0aW9uLlxuICAgKiBJZiBzZXQgdG8gZmFsc2UsIHlvdSBtdXN0IHByb3ZpZGUgYSBsYW1iZGEgZnVuY3Rpb24gb2JqZWN0IGFzIGBleGlzdGluZ09iamBcbiAgICpcbiAgICogQGRlZmF1bHQgLSB0cnVlXG4gICAqL1xuICByZWFkb25seSBkZXBsb3lMYW1iZGE6IGJvb2xlYW4sXG4gIC8qKlxuICAgKiBFeGlzdGluZyBpbnN0YW5jZSBvZiBMYW1iZGEgRnVuY3Rpb24gb2JqZWN0LlxuICAgKiBJZiBgZGVwbG95YCBpcyBzZXQgdG8gZmFsc2Ugb25seSB0aGVuIHRoaXMgcHJvcGVydHkgaXMgcmVxdWlyZWRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBleGlzdGluZ0xhbWJkYU9iaj86IGxhbWJkYS5GdW5jdGlvbixcbiAgLyoqXG4gICAqIE9wdGlvbmFsIHVzZXIgcHJvdmlkZWQgcHJvcHMgdG8gb3ZlcnJpZGUgdGhlIGRlZmF1bHQgcHJvcHMuXG4gICAqIElmIGBkZXBsb3lgIGlzIHNldCB0byB0cnVlIG9ubHkgdGhlbiB0aGlzIHByb3BlcnR5IGlzIHJlcXVpcmVkXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRGVmYXVsdCBwcm9wcyBhcmUgdXNlZFxuICAgKi9cbiAgcmVhZG9ubHkgbGFtYmRhRnVuY3Rpb25Qcm9wcz86IGxhbWJkYS5GdW5jdGlvblByb3BzLFxuICAvKipcbiAgICogV2hldGhlciB0byBjcmVhdGUgYSBTMyBCdWNrZXQgb3IgdXNlIGFuIGV4aXN0aW5nIFMzIEJ1Y2tldC5cbiAgICogSWYgc2V0IHRvIGZhbHNlLCB5b3UgbXVzdCBwcm92aWRlIFMzIEJ1Y2tldCBhcyBgZXhpc3RpbmdCdWNrZXRPYmpgXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgZGVwbG95QnVja2V0PzogYm9vbGVhbixcbiAgLyoqXG4gICAqIEV4aXN0aW5nIGluc3RhbmNlIG9mIFMzIEJ1Y2tldCBvYmplY3QuXG4gICAqIElmIGBkZXBsb3lCdWNrZXRgIGlzIHNldCB0byBmYWxzZSBvbmx5IHRoZW4gdGhpcyBwcm9wZXJ0eSBpcyByZXF1aXJlZFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGV4aXN0aW5nQnVja2V0T2JqPzogczMuQnVja2V0LFxuICAvKipcbiAgICogT3B0aW9uYWwgdXNlciBwcm92aWRlZCBwcm9wcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wcy5cbiAgICogSWYgYGRlcGxveWAgaXMgc2V0IHRvIHRydWUgb25seSB0aGVuIHRoaXMgcHJvcGVydHkgaXMgcmVxdWlyZWRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBEZWZhdWx0IHByb3BzIGFyZSB1c2VkXG4gICAqL1xuICByZWFkb25seSBidWNrZXRQcm9wcz86IHMzLkJ1Y2tldFByb3BzLFxuICAvKipcbiAgICogT3B0aW9uYWwgdXNlciBwcm92aWRlZCBwcm9wcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wc1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIERlZmF1bHQgcHJvcHMgYXJlIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IHMzRXZlbnRTb3VyY2VQcm9wcz86IFMzRXZlbnRTb3VyY2VQcm9wc1xufVxuXG5leHBvcnQgY2xhc3MgUzNUb0xhbWJkYSBleHRlbmRzIENvbnN0cnVjdCB7XG4gIC8vIFByaXZhdGUgdmFyaWFibGVzXG4gIHByaXZhdGUgZm46IGxhbWJkYS5GdW5jdGlvbjtcbiAgcHJpdmF0ZSBidWNrZXQ6IHMzLkJ1Y2tldDtcbiAgLyoqXG4gICAqIEBzdW1tYXJ5IENvbnN0cnVjdHMgYSBuZXcgaW5zdGFuY2Ugb2YgdGhlIFMzVG9MYW1iZGEgY2xhc3MuXG4gICAqIEBwYXJhbSB7Y2RrLkFwcH0gc2NvcGUgLSByZXByZXNlbnRzIHRoZSBzY29wZSBmb3IgYWxsIHRoZSByZXNvdXJjZXMuXG4gICAqIEBwYXJhbSB7c3RyaW5nfSBpZCAtIHRoaXMgaXMgYSBhIHNjb3BlLXVuaXF1ZSBpZC5cbiAgICogQHBhcmFtIHtTM1RvTGFtYmRhUHJvcHN9IHByb3BzIC0gdXNlciBwcm92aWRlZCBwcm9wcyBmb3IgdGhlIGNvbnN0cnVjdFxuICAgKiBAc2luY2UgMC44LjBcbiAgICogQGFjY2VzcyBwdWJsaWNcbiAgICovXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBTM1RvTGFtYmRhUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5mbiA9IGRlZmF1bHRzLmJ1aWxkTGFtYmRhRnVuY3Rpb24odGhpcywge1xuICAgICAgZGVwbG95TGFtYmRhOiBwcm9wcy5kZXBsb3lMYW1iZGEsXG4gICAgICBleGlzdGluZ0xhbWJkYU9iajogcHJvcHMuZXhpc3RpbmdMYW1iZGFPYmosXG4gICAgICBsYW1iZGFGdW5jdGlvblByb3BzOiBwcm9wcy5sYW1iZGFGdW5jdGlvblByb3BzXG4gICAgfSk7XG5cbiAgICB0aGlzLmJ1Y2tldCA9IGRlZmF1bHRzLmJ1aWxkUzNCdWNrZXQodGhpcywge1xuICAgICAgZGVwbG95QnVja2V0OiBwcm9wcy5kZXBsb3lCdWNrZXQsXG4gICAgICBleGlzdGluZ0J1Y2tldE9iajogcHJvcHMuZXhpc3RpbmdCdWNrZXRPYmosXG4gICAgICBidWNrZXRQcm9wczogcHJvcHMuYnVja2V0UHJvcHNcbiAgICB9KTtcblxuICAgIC8vIENyZWF0ZSBTMyB0cmlnZ2VyIHRvIGludm9rZSBsYW1iZGEgZnVuY3Rpb25cbiAgICB0aGlzLmZuLmFkZEV2ZW50U291cmNlKG5ldyBTM0V2ZW50U291cmNlKHRoaXMuYnVja2V0LFxuICAgICAgZGVmYXVsdHMuUzNFdmVudFNvdXJjZVByb3BzKHByb3BzLnMzRXZlbnRTb3VyY2VQcm9wcykpKTtcblxuICAgIHRoaXMuYWRkQ2ZuTmFnU3VwcHJlc3MoKTtcbiAgfVxuXG4gIHByaXZhdGUgYWRkQ2ZuTmFnU3VwcHJlc3MoKSB7XG4gICAgLy8gRXh0cmFjdCB0aGUgQ2ZuQnVja2V0IGZyb20gdGhlIHMzQnVja2V0XG4gICAgY29uc3QgczNCdWNrZXRSZXNvdXJjZSA9IHRoaXMuYnVja2V0Lm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIHMzLkNmbkJ1Y2tldDtcblxuICAgIHMzQnVja2V0UmVzb3VyY2UuY2ZuT3B0aW9ucy5tZXRhZGF0YSA9IHtcbiAgICAgICAgY2ZuX25hZzoge1xuICAgICAgICAgICAgcnVsZXNfdG9fc3VwcHJlc3M6IFt7XG4gICAgICAgICAgICAgICAgaWQ6ICdXNTEnLFxuICAgICAgICAgICAgICAgIHJlYXNvbjogYFRoaXMgUzMgYnVja2V0IEJ1Y2tldCBkb2VzIG5vdCBuZWVkIGEgYnVja2V0IHBvbGljeWBcbiAgICAgICAgICAgIH1dXG4gICAgICAgIH1cbiAgICB9O1xuXG4gICAgY29uc3Qgcm9vdCA9IFN0YWNrLm9mKHRoaXMpO1xuICAgIGNvbnN0IGxvZ2ljYWxJZCA9ICdCdWNrZXROb3RpZmljYXRpb25zSGFuZGxlcjA1MGEwNTg3Yjc1NDQ1NDdiZjMyNWYwOTRhM2RiODM0JztcbiAgICBjb25zdCBub3RpZmljYXRpb25zUmVzb3VyY2VIYW5kbGVyID0gcm9vdC5ub2RlLnRyeUZpbmRDaGlsZChsb2dpY2FsSWQpIGFzIGxhbWJkYS5GdW5jdGlvbjtcbiAgICBjb25zdCBub3RpZmljYXRpb25zUmVzb3VyY2VIYW5kbGVyUm9sZVJvbGUgPSBub3RpZmljYXRpb25zUmVzb3VyY2VIYW5kbGVyLm5vZGUuZmluZENoaWxkKCdSb2xlJykgYXMgaWFtLlJvbGU7XG4gICAgY29uc3Qgbm90aWZpY2F0aW9uc1Jlc291cmNlSGFuZGxlclJvbGVQb2xpY3kgPSBub3RpZmljYXRpb25zUmVzb3VyY2VIYW5kbGVyUm9sZVJvbGUubm9kZS5maW5kQ2hpbGQoJ0RlZmF1bHRQb2xpY3knKSBhcyBpYW0uUG9saWN5O1xuXG4gICAgLy8gRXh0cmFjdCB0aGUgQ2ZuRnVuY3Rpb24gZnJvbSB0aGUgRnVuY3Rpb25cbiAgICBjb25zdCBmblJlc291cmNlID0gbm90aWZpY2F0aW9uc1Jlc291cmNlSGFuZGxlci5ub2RlLmZpbmRDaGlsZCgnUmVzb3VyY2UnKSBhcyBsYW1iZGEuQ2ZuRnVuY3Rpb247XG5cbiAgICBmblJlc291cmNlLmNmbk9wdGlvbnMubWV0YWRhdGEgPSB7XG4gICAgICBjZm5fbmFnOiB7XG4gICAgICAgICAgcnVsZXNfdG9fc3VwcHJlc3M6IFt7XG4gICAgICAgICAgICAgIGlkOiAnVzU4JyxcbiAgICAgICAgICAgICAgcmVhc29uOiBgTGFtYmRhIGZ1bmN0aW9uIGhhcyBwZXJtaXNzaW9uIHRvIHdyaXRlIENsb3VkV2F0Y2ggTG9ncyB2aWEgQVdTTGFtYmRhQmFzaWNFeGVjdXRpb25Sb2xlIHBvbGljeSBhdHRhY2hlZCB0byB0aGUgbGFtYmRhIHJvbGVgXG4gICAgICAgICAgfV1cbiAgICAgIH1cbiAgICB9O1xuXG4gICAgLy8gRXh0cmFjdCB0aGUgQ2ZuUG9saWN5IGZyb20gdGhlIGlhbS5Qb2xpY3lcbiAgICBjb25zdCBwb2xpY3lSZXNvdXJjZSA9IG5vdGlmaWNhdGlvbnNSZXNvdXJjZUhhbmRsZXJSb2xlUG9saWN5Lm5vZGUuZmluZENoaWxkKCdSZXNvdXJjZScpIGFzIGlhbS5DZm5Qb2xpY3k7XG5cbiAgICBwb2xpY3lSZXNvdXJjZS5jZm5PcHRpb25zLm1ldGFkYXRhID0ge1xuICAgICAgY2ZuX25hZzoge1xuICAgICAgICAgIHJ1bGVzX3RvX3N1cHByZXNzOiBbe1xuICAgICAgICAgICAgICBpZDogJ1cxMicsXG4gICAgICAgICAgICAgIHJlYXNvbjogYEJ1Y2tldCByZXNvdXJjZSBpcyAnKicgZHVlIHRvIGNpcmN1bGFyIGRlcGVuZGVuY3kgd2l0aCBidWNrZXQgYW5kIHJvbGUgY3JlYXRpb24gYXQgdGhlIHNhbWUgdGltZWBcbiAgICAgICAgICB9XVxuICAgICAgfVxuICAgIH07XG4gIH1cblxuICAvKipcbiAgICogQHN1bW1hcnkgUmV0dXJucyBhbiBpbnN0YW5jZSBvZiB0aGUgbGFtYmRhLkZ1bmN0aW9uIGNyZWF0ZWQgYnkgdGhlIGNvbnN0cnVjdC5cbiAgICogQHJldHVybnMge2xhbWJkYS5GdW5jdGlvbn0gSW5zdGFuY2Ugb2YgdGhlIEZ1bmN0aW9uIGNyZWF0ZWQgYnkgdGhlIGNvbnN0cnVjdC5cbiAgICogQHNpbmNlIDAuOC4wXG4gICAqIEBhY2Nlc3MgcHVibGljXG4gICAqL1xuICBwdWJsaWMgbGFtYmRhRnVuY3Rpb24oKTogbGFtYmRhLkZ1bmN0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5mbjtcbiAgfVxuXG4gIC8qKlxuICAgKiBAc3VtbWFyeSBSZXR1cm5zIGFuIGluc3RhbmNlIG9mIHRoZSBzMy5CdWNrZXQgY3JlYXRlZCBieSB0aGUgY29uc3RydWN0LlxuICAgKiBAcmV0dXJucyB7czMuQnVja2V0fSBJbnN0YW5jZSBvZiB0aGUgQnVja2V0IGNyZWF0ZWQgYnkgdGhlIGNvbnN0cnVjdC5cbiAgICogQHNpbmNlIDAuOC4wXG4gICAqIEBhY2Nlc3MgcHVibGljXG4gICAqL1xuICBwdWJsaWMgczNCdWNrZXQoKTogczMuQnVja2V0IHtcbiAgICAgIHJldHVybiB0aGlzLmJ1Y2tldDtcbiAgfVxufSJdfQ==