import logging
from os.path import isdir, join, exists
from os import rename, remove
from glob import glob
import numpy as np
import subprocess
import secrets
from l3wrapper import l3wrapper_data_path
from l3wrapper.dictionary import build_class_dict, \
                                 build_item_dictionaries, \
                                 parse_raw_rules, \
                                 write_human_readable, \
                                 build_columns_dictionary, \
                                 Transaction
from l3wrapper.validation import check_column_names
from joblib import Parallel, delayed
import time
from collections import Counter
from operator import itemgetter


BIN_DIR = "bin"
TRAIN_BIN = "L3CFiltriItemTrain"
CLASSIFY_BIN = "L3CFiltriItemClassifica"
CLASSIFICATION_RESULTS = "classificati.txt"
LEVEL1_FILE = "livelloI.txt"
LEVEL2_FILE = "livelloII.txt"
LEVEL1_FILE_READABLE = "lvl1_R.txt"
LEVEL2_FILE_READABLE = "lvl2_R.txt"
FILTER_LEVEL1 = ''
FILTER_LEVEL2 = ''
FILTER_BOTH = ''
UNKNOWN_LABEL = 'Unlabeled'


def _create_column_names(X):
    # Keep L3 1-based indexing naming
    return [f"{i}" for i in range(1, X.shape[1] + 1)]


def _dump_array_to_file(X, filestem, ext):
    with open(f"{filestem}.{ext}", "w") as fp:
        [fp.write(f"{','.join(row)}\n") for row in X]


def _remove_fit_files(filestem):
    """Remove the files generated by the fit method.
    
    Retain the .cls and .diz files. These are required by the classification module of L3.
    """
    [remove(f"{filestem}{f}") for f in ["_stdout.txt", ".bin", ".data"]]


def _get_matching_rules(transaction, rules, max_matching):
    if max_matching < 1:
        raise ValueError("'max_matching' must be at least 1")

    matching_rules = list()
    rule_iter = (r for r in rules if r.match(transaction)) 
    count = 0
    while count < max_matching:
        try:
            matching_rules.append(next(rule_iter))
            count += 1
        except StopIteration:
            break

    return matching_rules


class L3Classifier:
    
    def __init__(self, min_sup: float, min_conf: float, l3_root: str = None):
        self.min_sup = min_sup
        self.min_conf = min_conf
        if l3_root:
            self.l3_root = l3_root
        else:
            self.l3_root = l3wrapper_data_path

        bin_dir = join(self.l3_root, BIN_DIR)
        if not exists(bin_dir) or not isdir(bin_dir):
            raise ValueError("No directory named 'bin' is present in the specified L3 root")

        self.train_bin_path = join(self.l3_root, BIN_DIR, TRAIN_BIN)
        self.classify_bin_path = join(self.l3_root, BIN_DIR, CLASSIFY_BIN)
        self.logger = logging.getLogger(__name__)
        
        self._class_dict = None
        self._item_id_to_item = None
        self._item_to_item_id = None
        self._column_id_to_name = None
        self._lvl1_rules = None
        self.n_lvl1_rules = None
        self._lvl2_rules = None
        self.n_lvl2_rules = None
        self._n_fit = 0
        self._current_token = None      # keep track of the latest token generated by the fit method
        self._n_items_used = None
        self._n_classes = None
        self.labeled_transactions = list()

    def __str__(self):
        return f"L3Classifier(min_sup={self.min_sup},min_conf={self.min_conf})"

    def fit(self, X, y, column_names=None, save_human_readable=True, remove_files=True):
        """Fit the L3 model according to the given training data.
        
        1. DONE
        2. DONE
        3. (optional) post-process the mined rules
        """
        token = secrets.token_hex(4)
        filestem = f"{token}"

        # TODO the character ':' is not allowed in any column name, enforce this.
        # Create column names if not provided
        if not column_names:
            column_names = _create_column_names(X)
        check_column_names(X, column_names)
        self._column_id_to_name = build_columns_dictionary(column_names)

        # Reshape required if y is a vector array
        if y.ndim == 1:
            y = y.reshape(-1, 1)

        # Dump X and y in a single .data (csv) file. "y" target labels are inserted as the last column
        X_todump = np.hstack([X, y])
        _dump_array_to_file(X_todump, filestem, "data")

        # Invoke the training module of L3.
        with open(f"{filestem}_stdout.txt", "w") as stdout:
            subprocess.run(
                [
                    self.train_bin_path,
                    filestem,                       # training file filestem
                    f"{self.min_sup * 100:.2f}",    # min sup
                    f"{self.min_conf * 100:.2f}",   # min conf
                    "nofiltro",                     # filtering measure for items (DEPRECATED)
                    "0",                            # filtering threshold (DEPRECATED)
                    "0",                            # specialistic/general rules (TO VERIFY)
                    "0",                            # max length allowed for rules (DEPRECATED)
                    self.l3_root                    # L3 root containing the 'bin' directory with binaries
                ],
                stdout=stdout
            )
        
        # rename useful (lvl1) and sparse (lvl2) rule files
        rename(LEVEL1_FILE, f"{token}_{LEVEL1_FILE}")
        rename(LEVEL2_FILE, f"{token}_{LEVEL2_FILE}")

        # read the mappings of classification labels
        self._class_dict = build_class_dict(filestem)
        self._n_classes = len(self._class_dict)

        # read the mappings item->"column_name","value"
        self._item_id_to_item, self._item_to_item_id = build_item_dictionaries(filestem)
        self._n_items_used = len(self._item_id_to_item)

        # parse the two rule sets and store them
        self._lvl1_rules = parse_raw_rules(f"{token}_{LEVEL1_FILE}")
        self._lvl2_rules = parse_raw_rules(f"{token}_{LEVEL2_FILE}")
        self.n_lvl1_rules = len(self._lvl1_rules)
        self.n_lvl2_rules = len(self._lvl2_rules)

        # translate the model to human readable format
        if save_human_readable:
            write_human_readable(f"{token}_{LEVEL1_FILE_READABLE}", self._lvl1_rules,
                                 self._item_id_to_item, self._column_id_to_name, self._class_dict)
            write_human_readable(f"{token}_{LEVEL2_FILE_READABLE}", self._lvl2_rules,
                                 self._item_id_to_item, self._column_id_to_name, self._class_dict)

        if remove_files:
            _remove_fit_files(token)

        self._current_token = token
        self._n_fit += 1
        
        return self

    def _get_class_label(self, matching_rules: list):
        """TODO Important method to weight majority voting"""
        class_ids = list()
        class_priority = {class_id : 0 for (class_id, _) in self._class_dict.items()}
        for rule in matching_rules:
            class_ids.append(rule.class_id)
            class_priority[rule.class_id] += rule.rank

        most_common = Counter(class_ids).most_common()
        most_common = sorted(most_common, key=lambda x: class_priority[x[0]])   # ascending by class priority
        most_common = sorted(most_common, key=itemgetter(1), reverse=True)      # descending by matched count

        return self._class_dict[most_common[0][0]]

    # TODO change max_matching and strategy to constructor parameters
    def predict(self, X, n_jobs=1, max_matching=1, strategy='majority'):
        if not self._current_token:
            raise RuntimeError("You must fit the model to data first.")

        # stime = time.time()
        # labels = Parallel(n_jobs=n_jobs, prefer="threads")(delayed(self._get_label)(X_row) for X_row in X)
         
        y_pred = list()
        for X_row in X:
            tr = Transaction(X_row, self._item_to_item_id)
            class_priority = {k: 0 for k, v in self._class_dict.items()} # used the majority voting is even
            used_level = None

            # match against level 1
            matching_rules = _get_matching_rules(tr, self._lvl1_rules, max_matching)

            # if level 1 was not used, match against level 2
            if not matching_rules:
                matching_rules = _get_matching_rules(tr, self._lvl2_rules, max_matching)
                if matching_rules:
                    used_level = 2
            else:
                used_level = 1
            
            if not matching_rules:
                y_pred.append(UNKNOWN_LABEL)
            else:
                y_pred.append(self._get_class_label(matching_rules))

            tr.used_level = used_level
            self.labeled_transactions.append(tr)        # keep track of labeled transaction

        # print(time.time() - stime)
        return np.array(y_pred)

    def score(self, X, y):
        """
        Return the mean accuracy on the given test data and labels.
        In multi-label classification, this is the subset accuracy
        which is a harsh metric since you require for each sample that
        each label set be correctly predicted.
        Parameters
        ----------
        X : array-like of shape (n_samples, n_features)
            Test samples.
        y : array-like of shape (n_samples,) or (n_samples, n_outputs)
            True labels for X.
        Returns
        -------
        score : float
            Mean accuracy of self.predict(X) wrt. y.
        """
        from sklearn.metrics import accuracy_score
        return accuracy_score(y, self.predict(X))
