# Guide to working with class-based checks

## Background - Why are we here?
In order to support cloud migrations including apps that contain configurations and other files within the `<app>/local` directory some new built-in functionality was added to gracefully handle configuration settings for local, default, and a merged view of both. These merged configuration and file views attempt to mimic how Splunk treats configurations and files within `<app>/default` and `<app>/local`, for more details see: [Config File Precedence](https://docs.splunk.com/Documentation/Splunk/latest/Admin/Wheretofindtheconfigurationfiles) and [App File Precedence](https://dev.splunk.com/enterprise/docs/developapps/manageknowledge/fileprecedence/).

Access is needed within the checks themselves to handle default, local, and merged configurations and file views depending on the situation. The way access is provided is through new methods that checks can choose to override (or not). This is one motivation for moving to class-based checks from the existing functional checks.

There has also been some friction with both authoring and maintaining functional checks in the past due to some "magic" that the current check framework provides in an opaque way. Examples: the name of the check function itself and docstring provided will be used to assign the check name and descriptions, respectively but the name must always begin with `check_`. Another example: the _name_ of the arguments provided to the check (e.g. `check_example(app, reporter, clamav)` include some elements that are provided out-of-the-box (`app`, and `reporter`) but can also refer to certain resources that may or may not have been provided within `resource_manager_context` when running the check (e.g. `clamav` used in dynamic_checks). Class-based checks attempt to remove some of that magic in favor of more explicit and transparent logic.

# Introduction

### What is a class-based check?
Any check that defines a class that inherits from the `splunk_appinspect.Check` class.

Example of `check_for_addon_builder_version` as a class-based check:
```python3
class CheckForAddonBuilderVersion(Check):
    def __init__(self):
        super().__init__(config=CheckConfig(
            name="check_for_addon_builder_version",
            description="Check that the `addon_builder.conf` contains an builder version number in the [base] stanza. "
                        "Ensure that apps built with Add-on Builder are maintained with an up-to-date version of Add-on Builder.",
            depends_on_config=("addon_builder",),
            cert_min_version="2.18.0",
            tags=(
                "cloud",
                "private_app",
                "private_classic",
                "private_victoria",
            )
        )
    def check_config(self, app, config):
        ...
        yield FailMessage(
            "No base section found in addon_builder.conf.",
            file_name=config["addon_builder"].get_relative_path(),
            remediation="Ensure addon_builder.conf was properly generated by Add-on Builder. Remove the file if no longer needed.",
        )
```

vs

`check_for_addon_builder_version` as a functional check:
```python3
@splunk_appinspect.tags("cloud", "private_app", "private_victoria", "private_classic")
@splunk_appinspect.cert_version(min="2.18.0")
def check_for_addon_builder_version(app, reporter):
    """Check that the `addon_builder.conf` indicates the app was built using an up-to-date version of Splunk Add-on Builder."""
    ...
    reporter_output = (
        "No base section found in addon_builder.conf."
    )
    reporter.fail(reporter_output, filename)
```

Note some key differences that will be detailed in the sections below:
1. `CheckConfig` is used in class-based checks to define check metadata
2. New `Check` methods are provided to help with default/local/merged configurations and file views
3. `CheckMessage`s are yielded within `Check` generator functions and no longer require a `reporter` object to report results

Note that functional checks are have historically been converted to the `Check` class at run-time, but we have introduced a new convention to begin defining checks as classes directly within `splunk_appinspect/checks/`.

# Key Concepts

## New `CheckConfig` for defining check metadata
Metadata that was being gathered from decorators, function names, and docstrings in functional checks are now explicitly defined within the `Check._config`.

### `CheckConfig` Fields
|Field (type)|Description from `checks.py`|Support in functional checks|
|------------|----------------------------|----------------------------|
|`name (str)`|Name of the check as reported to consumers|The function name (`check_*`) for functional checks is used to define this implicitly|
|`description (str)`|A description of what the check does|The docstring for functional checks is used to define this implicitly|
|`cert_min_version (str)`|Minimum certification version where this check applies parsed as a semantic version number|The `@splunk_appinspect.cert_version( min=... )` decorator is how this is defined for functional checks|
|`cert_max_version (str)`|Maximum certification version where this check applies, parsed as a semantic version number|The `@splunk_appinspect.cert_version( max=... )` decorator is how this is defined for functional checks|
|`tags (:obj:list of :obj:str)`|List of tags where this check should apply|The `@splunk_appinspect.tags(...)` decorator is how this is defined for functional checks|
|`report_display_order (int)`|Allows specifying an order for checks to appear within a group|The `@splunk_appinspect.display( report_display_order=... )` decorator is how this is defined for functional checks|
|`depends_on_config (:obj:list of :obj:str)`|A list of configuration file names (without `.conf`) that are required for certain check methods to apply. If none of the file names exist, the `Check.check*_config()` methods are not run and a `not_applicable` result is returned|N/A - this is **new** functionality not present in functional checks|
|`depends_on_data (:obj:list of :obj:str)`|A list of paths -- file names or directories -- that are required to exist in one of the data directories for certain check methods to apply. If none of the paths exist, the `Check.check*_data` methods are not run and a `not_applicable` result is returned|N/A - this is **new** functionality not present in functional checks|

## New `Check` methods
The following `Check` methods are automatically called when applicable to help with checks that depend on local configurations or other files within the `<app>/local/` directory, so define these for your new class-based checks as needed:

|Method signature|Description|When applicable|
|----------------|-----------|---------------|
|`check_config(self, app, config)`|Use this method to check configs across default and the merged view using the same logic for each.|This is called at most twice: 1) With the `config` argument equal to a `ConfigurationProxy` representing the default configuration if `depends_on_config` is specified AND at least one of the configs in `depends_on_config` exists within the `<app>/default/<config>.conf` directory 2) With the `config` argument equal to a `MergedConfigurationProxy` representing the merged configuration of local and default if `depends_on_config` is specified AND at least one of the configs in `depends_on_config` exists within `<app>/[default|local]/<config>.conf`|
|`check_data(self, app, file_view)`|Use this method to check files across default and the merged view using the same logic for each.|This is called at most twice: 1) With the `file_view` argument equal to a `FileView` representing the files within `<app>/default/data` if `depends_on_data` is specified AND at least one of the files or directory paths specified in `depends_on_config` exists within the `<app>/default/data` 2) With the `file_view` argument equal to a `MergedFileView` representing the merged view of files within `<app>/default/data` and `<app>/local/data` if `depends_on_data` is specified AND at least one of the files or directory paths specified in `depends_on_config` exists within `<app>/[default|local]/data`|
|`check_default_config(self, app, config)`|Use this method to provide logic to check configs specific to the `<app>/default` directory.|This is called at most once with the `config` argument equal to a `ConfigurationProxy` representing the default configuration if `depends_on_config` is specified AND at least one of the configs in `depends_on_config` exists within `<app>/default/<config>.conf`|
|`check_default_data(self, app, file_view)`|Use this method to provide logic to check file paths specific to the `<app>/default/data` directory.|This is called at most once with the `file_view` argument equal to a `FileView` representing the files within `<app>/default/data` if `depends_on_data` is specified AND at least one of the files or directory paths specified in `depends_on_config` exists within `<app>/default/data`|
|`check_merged_config(self, app, config)`|Use this method to provide logic to check configurations of the `<app>/default/<config>.conf` and `<app>/local/<config>.conf`.|This is called at most once with the `config` argument equal to a `MergedConfigurationProxy` representing the merged configuration of local and default if `depends_on_config` is specified AND at least one of the configs in `depends_on_config` exists within `<app>/[default|local]/<config>.conf`|
|`check_merged_data(self, app, file_view)`|Use this method to provide logic to check file views of the `<app>/default/data` and `<app>/local/data` directories.|This is called at most once with the `file_view` argument equal to a `MergedFileView` representing the merged view of files within `<app>/default/data` and `<app>/local/data` if `depends_on_data` is specified AND at least one of the files or directory paths specified in `depends_on_config` exists within `<app>/[default|local]/data`|
|`check_local_config(self, app, config)`|Use this method to provide logic to check configs specific to the `<app>/local` directory.|This is called at most once with the `config` argument equal to a `ConfigurationProxy` representing the local configuration if `depends_on_config` is specified AND at least one of the configs in `depends_on_config` exists within `<app>/local/<config>.conf`|
|`check_local_data(self, app, file_view)`|Use this method to provide logic to check file paths specific to the `<app>/local/data` directory.|This is called at most once with the `file_view` argument equal to a `FileView` representing the files within `<app>/local/data` if `depends_on_data` is specified AND at least one of the files or directory paths specified in `depends_on_config` exists within `<app>/local/data`|
|`check_lookups(self, app, file_view)`|This method will be called once if ANY files exist in the lookups/ directory.|If either `check_lookups` or `check_lookup_file` is defined but no `lookups/` directory is present within the app a NotApplicableMessage will automatically be `yield`ed|
|`check_lookup_file(self, app, path_in_app)`|This method will be called once for each lookup in the lookups/ directory.|If either `check_lookups` or `check_lookup_file` is defined but no `lookups/` directory is present within the app a NotApplicableMessage will automatically be `yield`ed|
|`check_metadata_files(self, app, file_view)`|This method will be called once if ANY files exist in the metadata/ directory.|If either `check_metadata_files` or `check_metadata_file` is defined but no `metadata/` directory is present within the app a NotApplicableMessage will automatically be `yield`ed|
|`check_metadata_file(self, app, path_in_app)`|This method will be called once for each file in the metadata/ directory.|If either `check_metadata_files` or `check_metadata_file` is defined but no `metadata/` directory is present within the app a NotApplicableMessage will automatically be `yield`ed|
|`check_static_files(self, app, file_view)`|This method will be called once if ANY files exist in the static/ directory.|If either `check_static_files` or `check_static_file` is defined but no `static/` directory is present within the app a NotApplicableMessage will automatically be `yield`ed|
|`check_static_file(self, app, path_in_app)`|This method will be called once for each file in the static/ directory.|If either `check_static_files` or `check_static_file` is defined but no `static/` directory is present within the app a NotApplicableMessage will automatically be `yield`ed|

## Yielding `CheckMessage`s and generator behavior
Class-based checks introduce support for a new way of gathering results from a check, with subclasses of `CheckMessage` which are `yield`ed from within class-based check generator methods including all of the "New `Check` Methods" listed in the section above. This is a departure from the way functional checks work which rely on a `reporter` instance being passed to the check and calling `reporter.failure(...)`, `reporter.manual_check(...)`, etc to gather results.

New field: `remediation`
While we're introducing new reporting behavior it seemed like a good time to add way to specifically describe to developers how to fix the issue that AppInspect is reporting. For example, if a check fails for detecting compiled `.pyc` files then the `remediation` might simply be `Remove any .pyc files`. Specific handling/formatting of this field can be added in the future.

If you aren't familiar with generators and `yield`s here are some good resources to get you started:
- https://realpython.com/introduction-to-python-generators/
- https://towardsdatascience.com/how-to-work-with-python-generators-d30d8d0af7fa

# General approach to writing class-based checks
(This also applies to converting functional checks to class-based checks)

**Some questions you should ask yourself is before getting started**
1. Does this check logic depend on configuration files in `<app>/default/<config>.conf` or `<app>/local/<config>.conf`?

If yes then you will want to define `depends_on_config` for each conf file your check will involve. Then you'll want to determine if all configuration values should be reported the same (probably the most common case) or whether default/local/merged configurations need to report different results for each case. This will help determine which of the following methods you will need to implement: `check_config` for handling all configs the same, `check_default_config` for special handling around default configs, `check_local_config` for special handling around local configs, or `check_merged_config` for special handling around merged configs.

For example, if we want to enforce that `web.conf` should only define `[endpoint:*]` and `[expose:*]` stanzas and that logic should apply whether they are defined in default or local then a `depends_on_config=('web')` should be defined but only the `check_config()` method needs to be defined and we can place logic to `yield FailMessage(...)` within that method which will run for both default and merged configurations - merged containing any local stanzas and properties.

2. Does this check logic depend on files within `<app>/default/data` or `<app>/local/data`?

If yes then you will want to define `depends_on_data` for each path or file your check will involve (examples: `ui/nav/default.xml` or `ui/views`). Then you'll want to determine if all files should be reported the same (probably the most common case) or whether default/local/merged files need to report different results for each case. This will help determine which of the following methods you will need to implement: `check_data` for handling all files the same regardless of whether they are default/local/merged, `check_default_data` for special handling around default data files, `check_local_data` for special handling around local data files, or `check_merged_data` for special handling around merged data files where if a file exists in local it will take precedence over default.

For example, if we want to enforce that any Simple XML files within `*/data/ui/views` should have `<version=1.1>` then `depends_on_data=(os.path.join('ui', 'views))` should be defined and only the `check_data()` method needs to be defined and we can place logic to `yield FailMessage(...)` within that method which will run for both default and merged views - merged containing any local view files.

3. If you have logic that depends on files within the `lookups`, `metadata`, or `static` directories then you will want to define `check_lookups`/`check_lookup_file`, `check_metadata_files`/`check_metadata_file`, or `check_static_files`/`check_static_file` respectively and depending on whether you need logic across all files within the directory or to treat each file individually.

4. If you answered **No** to all of the above then you most likely only need to overload the `check` method to place your check logic within there and `yield` the appropriate `CheckMessage`s within there. Do note that none of the other `check_*` methods will be automatically invoked if you overload this method, however.

For example, if we simply want to check that a Python file within the `bin/` directory is well formed then we can define that logic within `check(self, app)` and `yield` the appropriate `CheckMessage`s within that method. 