# Copyright 2019 Splunk Inc. All rights reserved.

"""
### REST endpoints and handler standards

REST endpoints are defined in a **restmap.conf** file in the **/default** and **/local** directory of the app. For more, see [restmap.conf](http://docs.splunk.com/Documentation/Splunk/latest/Admin/Restmapconf).
"""
import logging
import os

import splunk_appinspect
from splunk_appinspect.check_messages import FailMessage, NotApplicableMessage
from splunk_appinspect.checks import Check, CheckConfig

logger = logging.getLogger(__name__)

report_display_order = 23


class CheckRestmapConfExists(Check):
    def __init__(self):
        super().__init__(
            config=CheckConfig(
                name="check_restmap_conf_exists",
                description="Check that `restmap.conf` file exists at `default/restmap.conf` and  `local/restmap.conf` when using REST endpoints.",
                depends_on_config=("restmap",),
                cert_min_version="1.1.0",
                report_display_order=1,
                tags=(
                    "splunk_appinspect",
                    "restmap_config",
                    "cloud",
                    "private_app",
                    "private_victoria",
                    "migration_victoria",
                    "private_classic",
                ),
            )
        )

    def check_config(self, app, config):
        if not "restmap" in config:
            yield NotApplicableMessage(
                "No restmap.conf file exists.",
            )


class CheckRestHandlerScriptsExist(Check):
    def __init__(self):
        super().__init__(
            config=CheckConfig(
                name="check_rest_handler_scripts_exist",
                description="Check that each stanza in restmap.conf has a matching handler script. if not, fail this app.",
                depends_on_config=("restmap",),
                cert_min_version="1.1.0",
                tags=(
                    "splunk_appinspect",
                    "restmap_config",
                ),
            )
        )

    def check_config(self, app, config):
        yield from do_rest_handler_scripts_check(app, FailMessage, config)


def do_rest_handler_scripts_check(app, result_message, config):
    """Check that each stanza in restmap.conf has a matching handler script."""
    rest_map = app.get_rest_map(config)
    # From ACD-300, ACD-271,ACD-367
    # A rest config can have both, handler and handler_file. Or use the global handler
    # See
    # http://docs.splunk.com/Documentation/Splunk/latest/Admin/restmapconf
    global_handler = rest_map.global_handler_file()

    if global_handler.exists():
        message = f"A global rest handler was found at {global_handler.file_path}"
        logger.info(message)

    else:
        logger.info(
            "A global rest handler was not found at %s", global_handler.file_path
        )

        handler_list = rest_map.handlers()
        for handler in handler_list:
            if (
                handler.handler_file().exists()
                or handler.handler().exists()
                or handler.executable_script_file().exists()
            ):
                pass
            else:
                yield result_message(
                    f"Neither the handler or handlerfile specified in the stanza {handler.name} was found"
                    f" in app/bin for {handler.handler_file().file_path}, {handler.handler().file_path} or"
                    f" {handler.executable_script_file().file_path}.",
                    file_name=config["restmap"].get_relative_path(),
                    line_number=config["restmap"][handler.name].get_line_number(),
                )


class CheckRestHandlerScriptsExistForCloud(Check):
    def __init__(self):
        super().__init__(
            config=CheckConfig(
                name="check_rest_handler_scripts_exist_for_cloud",
                description="Check that each stanza in restmap.conf has a matching handler script.if not, throw a warning.",
                depends_on_config=("restmap",),
                cert_min_version="1.6.1",
                tags=(
                    "restmap_config",
                    "cloud",
                    "private_app",
                    "private_victoria",
                    "migration_victoria",
                    "private_classic",
                ),
            )
        )

    def check_config(self, app, config):
        yield from do_rest_handler_scripts_check(app, FailMessage, config)


class CheckRestHandlerPythonExecutableExists(Check):
    def __init__(self):
        super().__init__(
            config=CheckConfig(
                name="check_rest_handler_python_executable_exists",
                description="Check that python version is python3 for executables in restmap.conf.",
                depends_on_config=("restmap",),
                cert_min_version="2.1.0",
                tags=(
                    "splunk_appinspect",
                    "restmap_config",
                    "cloud",
                    "python3_version",
                    "private_app",
                    "private_victoria",
                    "migration_victoria",
                    "private_classic",
                ),
            )
        )

    def check_config(self, app, config):
        rest_map = app.get_rest_map(config)
        handler_list = rest_map.handlers()
        for handler in handler_list:
            # Skip non-python rest handler. It is reasonable to assume that Python files will have .py extension.
            if (
                handler.scripttype == "persist"
                and not handler.handler_module_file_name.endswith(".py")
            ):
                continue

            # Verify python rest handler.
            if not handler.python_version or handler.python_version != "python3":
                yield FailMessage(
                    f"The handler of stanza [{handler.name}] should be `python3` executable. ",
                    file_name=config["restmap"].get_relative_path(),
                    line_number=config["restmap"][handler.name].get_line_number(),
                )
