# Copyright 2019 Splunk Inc. All rights reserved.

"""
### addon_builder.conf standards

The **addon_builder.conf** file located at **default/addon_builder.conf** provides the information about the add on builder associated with the Splunk App
"""
import logging

from packaging import version

from splunk_appinspect.app_util import AppVersionNumberMatcher
from splunk_appinspect.checks import Check, CheckConfig
from splunk_appinspect.configuration_file import ConfigurationSetting
from splunk_appinspect.check_messages import FailMessage

report_display_order = 5
logger = logging.getLogger(__name__)


def parse_version(version_str):
    if isinstance(version_str, ConfigurationSetting):
        version_str = version_str.value

    if not AppVersionNumberMatcher().match(version_str):
        return None
    return version.parse(version_str)


class CheckForAddonBuilderVersion(Check):
    def __init__(self):
        super().__init__(config=CheckConfig(
            name="check_for_addon_builder_version",
            description="Check that the `addon_builder.conf` contains an builder version number in the [base] stanza. "
                        "Ensure that apps built with Add-on Builder are maintained with an up-to-date version of Add-on Builder.",
            depends_on_config=("addon_builder",),
            cert_min_version="2.18.0",
            tags=(
                "cloud",
                "private_app",
                "private_classic",
                "private_victoria",
                "migration_victoria",
            )
        ))

    MINIMUM_BUILDER_VERSION = parse_version("4.1.0")

    def check_config(self, app, config):
        if not config["addon_builder"].has_section("base"):
            yield FailMessage(
                "No base section found in addon_builder.conf.",
                file_name=config["addon_builder"].get_relative_path(),
                remediation="Ensure addon_builder.conf was properly generated by Add-on Builder. Remove the file if no longer needed.",
            )
            return

        if not config["addon_builder"].has_option("base", "builder_version"):
            yield FailMessage(
                "No builder_version specified in base section of addon_builder.conf.",
                file_name=config["addon_builder"].get_relative_path(),
                line_number=config["addon_builder"]["base"].get_line_number(),
                remediation="Ensure addon_builder.conf was properly generated by Add-on Builder. Remove the file if no longer needed.",
            )
            return

        addon_builder_version = parse_version(config["addon_builder"]["base"]["builder_version"])
        if addon_builder_version is None:
            yield FailMessage(
                "An invalid version number specified in addon_builder.conf for builder_version.",
                file_name=config["addon_builder"]["base"]["builder_version"].get_relative_path(),
                line_number=config["addon_builder"]["base"]["builder_version"].get_line_number(),
                remediation="Ensure addon_builder.conf was properly generated by Add-on Builder. Remove the file if no longer needed.",
            )
        elif addon_builder_version < self.MINIMUM_BUILDER_VERSION:
            yield FailMessage(
                f"The Add-on Builder version used to create this app ({addon_builder_version}) is below the minimum required version of {self.MINIMUM_BUILDER_VERSION}.",
                file_name=config["addon_builder"]["base"]["builder_version"].get_relative_path(),
                line_number=config["addon_builder"]["base"]["builder_version"].get_line_number(),
                remediation=f"Re-generate your add-on using at least Add-on Builder {self.MINIMUM_BUILDER_VERSION}.",
            )
