# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['tensor_theorem_prover',
 'tensor_theorem_prover.normalize',
 'tensor_theorem_prover.prover',
 'tensor_theorem_prover.prover.operations',
 'tensor_theorem_prover.types',
 'tensor_theorem_prover.util']

package_data = \
{'': ['*']}

install_requires = \
['numpy>=1.20.0,<2.0.0', 'typing-extensions>=3.7.4']

setup_kwargs = {
    'name': 'tensor-theorem-prover',
    'version': '0.10.0',
    'description': 'Customizable first-order logic theorem prover supporting approximate vector similarity in unification',
    'long_description': '# Tensor Theorem Prover\n\n[![ci](https://img.shields.io/github/workflow/status/chanind/tensor-theorem-prover/CI/main)](https://github.com/chanind/tensor-theorem-prover)\n[![Codecov](https://img.shields.io/codecov/c/github/chanind/tensor-theorem-prover/main)](https://codecov.io/gh/chanind/tensor-theorem-prover)\n[![PyPI](https://img.shields.io/pypi/v/tensor-theorem-prover?color=blue)](https://pypi.org/project/tensor-theorem-prover/)\n\nFirst-order logic theorem prover supporting unification with approximate vector similarity\n\n## Installation\n\n```\npip install tensor-theorem-prover\n```\n\n## Usage\n\ntensor-theorem-prover can be used either as a standard symbolic first-order theorem prover, or it can be used with vector embeddings and fuzzy unification.\n\nThe basic setup requires listing out first-order formulae, and using the `ResolutionProver` class to generate proofs.\n\n```python\nimport numpy as np\nfrom vector_theorem_prover import ResolutionProver, Constant, Predicate, Variable, Implies\n\nX = Variable("X")\nY = Variable("Y")\nZ = Variable("Z")\n# predicates and constants can be given an embedding array for fuzzy unification\ngrandpa_of = Predicate("grandpa_of", np.array([1.0, 1.0, 0.0, 0.3, ...]))\ngrandfather_of = Predicate("grandfather_of", np.array([1.01, 0.95, 0.05, 0.33, ...]))\nparent_of = Predicate("parent_of", np.array([ ... ]))\nfather_of = Predicate("father_of", np.array([ ... ]))\nbart = Constant("bart", np.array([ ... ]))\nhomer = Constant("homer", np.array([ ... ]))\nabe = Constant("abe", np.array([ ... ]))\n\nknowledge = [\n    parent_of(homer, bart),\n    father_of(abe, homer),\n    # father_of(X, Z) ^ parent_of(Z, Y) -> grandpa_of(X, Y)\n    Implies(And(father_of(X, Z), parent_of(Z, Y)), grandpa_of(X, Y))\n]\n\nprover = ResolutionProver(knowledge=knowledge)\n\n# query the prover to find who is bart\'s grandfather\nproof = prover.prove(grandfather_of(X, bart))\n\n# even though `grandpa_of` and `grandfather_of` are not identical symbols,\n# their embedding is close enough that the prover can still find the answer\nprint(proof.substitutions[X]) # abe\n\n# the prover will return `None` if a proof could not be found\nfailed_proof = prover.prove(grandfather_of(bart, homer))\nprint(failed_proof) # None\n\n```\n\n### Working with proof results\n\nThe `prover.prove()` method will return a `Proof` object if a successful proof is found. This object contains a list of all the resolutions, substitutions, and similarity calculations that went into proving the goal.\n\n```python\nproof = prover.prove(goal)\n\nproof.substitutions # a map of all variables in the goal to their bound values\nproof.similarity # the min similarity of all `unify` operations in this proof\nproof.depth # the number of steps in this proof\nproof.proof_steps # all the proof steps involved, including all resolutions and unifications along the way\n```\n\nThe `Proof` object can be printed as a string to get a visual overview of the steps involved in the proof.\n\n```python\nX = Variable("X")\nY = Variable("Y")\nfather_of = Predicate("father_of")\nparent_of = Predicate("parent_of")\nis_male = Predicate("is_male")\nbart = Constant("bart")\nhomer = Constant("homer")\n\nknowledge = [\n    parent_of(homer, bart),\n    is_male(homer),\n    Implies(And(parent_of(X, Y), is_male(X)), father_of(X, Y)),\n]\n\nprover = ResolutionProver(knowledge=knowledge)\n\ngoal = father_of(homer, X)\nproof = prover.prove(goal)\n\nprint(proof)\n# Goal: [¬father_of(homer,X)]\n# Subsitutions: {X -> bart}\n# Similarity: 1.0\n# Depth: 3\n# Steps:\n#   Similarity: 1.0\n#   Source: [¬father_of(homer,X)]\n#   Target: [father_of(X,Y) ∨ ¬is_male(X) ∨ ¬parent_of(X,Y)]\n#   Unify: father_of(homer,X) = father_of(X,Y)\n#   Subsitutions: {}, {X -> homer, Y -> X}\n#   Resolvent: [¬is_male(homer) ∨ ¬parent_of(homer,X)]\n#   ---\n#   Similarity: 1.0\n#   Source: [¬is_male(homer) ∨ ¬parent_of(homer,X)]\n#   Target: [parent_of(homer,bart)]\n#   Unify: parent_of(homer,X) = parent_of(homer,bart)\n#   Subsitutions: {X -> bart}, {}\n#   Resolvent: [¬is_male(homer)]\n#   ---\n#   Similarity: 1.0\n#   Source: [¬is_male(homer)]\n#   Target: [is_male(homer)]\n#   Unify: is_male(homer) = is_male(homer)\n#   Subsitutions: {}, {}\n#   Resolvent: []\n```\n\n### Finding all possible proofs\n\nThe `prover.prove()` method will return the proof with the highest similarity score among all possible proofs, if one exists. If you want to get a list of all the possible proofs in descending order of similarity score, you can call `prover.prove_all()` to return a list of all proofs.\n\n### Custom matching functions and similarity thresholds\n\nBy default, the prover will use cosine similarity for unification. If you\'d like to use a different similarity function, you can pass in a function to the prover to perform the similarity calculation however you wish.\n\n```python\ndef fancy_similarity(item1, item2):\n    norm = np.linalg.norm(item1.embedding) + np.linalg.norm(item2.embedding)\n    return np.linalg.norm(item1.embedding - item2.embedding) / norm\n\nprover = ResolutionProver(knowledge=knowledge, similarity_func=fancy_similarity)\n```\n\nBy default, there is a minimum similarity threshold of `0.5` for a unification to success. You can customize this as well when creating a `ResolutionProver` instance\n\n```python\nprover = ResolutionProver(knowledge=knowledge, min_similarity_threshold=0.9)\n```\n\n### Working with Tensors (Pytorch, Tensorflow, etc...)\n\nBy default, the similarity calculation assumes that the embeddings supplied for constants and predicates are numpy arrays. If you want to use tensors instead, this will work as long as you provide a `similarity_func` which can work with the tensor types you\'re using and return a float.\n\nFor example, if you\'re using Pytorch, it might look like the following:\n\n```python\nimport torch\n\ndef torch_cosine_similarity(item1, item2):\n    similarity = torch.nn.functional.cosine_similarity(\n        item1.embedding,\n        item2.embedding,\n        0\n    )\n    return similarity.item()\n\nprover = ResolutionProver(knowledge=knowledge, similarity_func=torch_cosine_similarity)\n\n# for pytorch you may want to wrap the proving in torch.no_grad()\nwith torch.no_grad():\n    proof = prover.prove(goal)\n```\n\n### Max proof depth\n\nBy default, the ResolutionProver will abort proofs after a depth of 10. You can customize this behavior by passing `max_proof_depth` when creating the prover\n\n```python\nprover = ResolutionProver(knowledge=knowledge, max_proof_depth=10)\n```\n\n### Max resolvent width\n\nBy default, the ResolutionProver has no limit on how wide resolvents can get during the proving process. If the proofs are running too slowly, you can try to set `max_resolvent_width` to limit how many literals intermediate resolvents are allowed to contain. This should narrow the search tree, but has the trade-off of not finding proofs if the proof requires unifying together a lot of very wide clauses.\n\n```python\nprover = ResolutionProver(knowledge=knowledge, max_resolvent_width=10)\n```\n\n## Acknowledgements\n\nThis library borrows code and ideas from the earier library [fuzzy-reasoner](https://github.com/fuzzy-reasoner/fuzzy-reasoner). The main difference between these libraries is that tensor-theorem-prover supports full first-order logic using Resolution, whereas fuzzy-reasoner is restricting to Horn clauses and uses backwards chaining.\n\nLike fuzzy-reasoner, this library also takes inspiration from the following papers for the idea of using vector similarity in theorem proving:\n\n- [End-to-End Differentiable Proving](https://arxiv.org/abs/1705.11040) by Rocktäschel et al.\n- [Braid - Weaving Symbolic and Neural Knowledge into Coherent Logical Explanations](https://arxiv.org/abs/2011.13354) by Kalyanpur et al.\n\n## Contributing\n\nContributions are welcome! Please leave an issue in the Github repo if you find any bugs, and open a pull request with and fixes or improvements that you\'d like to contribute.\n',
    'author': 'David Chanin',
    'author_email': 'chanindav@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/chanind/tensor-theorem-prover',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
