#!/usr/bin/python3
#
# restore waypoints for CLOCKWISE interior lap 1 in Spring Canyon Park in Fort Collins Colorado
#
# usage:
#    ./route_foco_park_interior_waypoint_set1_reduced.py
#
#
# Marc Compere, comperem@gmail.com
# created : 14 Jul 2019
# modified: 29 Jul 2020
#
# --------------------------------------------------------------
# Copyright 2018 - 2020 Marc Compere
#
# This file is part of the Mobility Virtual Environment (MoVE).
# MoVE is open source software licensed under the GNU GPLv3.
#
# MoVE is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 3
# as published by the Free Software Foundation.
#
# MoVE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License version 3 for more details.
#
# A copy of the GNU General Public License is included with MoVE
# in the COPYING file, or see <https://www.gnu.org/licenses/>.
# --------------------------------------------------------------

import numpy as np
import logging


# ------------------------------------------------------------------------------
# the restore_waypoint() function gets called from MoVE vehicle model
# or as a stand-alone program with __main__ below
def restore_waypoints( cfg ):
    import numpy as np
    from routes_functions import computeRouteStatistics
    import utm
    
    # create an empty route object 
    class Route:
        pass
    route = Route()
    
    # all waypoints must eventually be in an orthogonal, terrain-fixed XYZ Cartesian coordinate system
    # however, route, or path coordinates may be specified in lat/lon pairs or (X,Y) coordinates directly
    #
    # - if original route coordinates are specified in meters, then use 'meters' for route.originalUnits
    # - if original route coordinates are specified in lat/lon pairs in decimal degrees, use 'decimal_degrees'
    route.originalUnits = 'decimal_degrees' #'meters'
    route.desc = "route foco, CW interior lap 2 in Spring Canyon Park in Ft. Collins Colorado from lat/lon recorded in SensorLog iPhone app"
    
    # -------------------------------
    # specify the route coordinates:
    # -------------------------------
    # the lines below were copy-pasted directly from this .csv: fort_collins_spring_canyon_park_reduced_1_lap_13_Jul_2019.csv
    # the .csv file was written by fort_collins_lap_preprocessor.m which works in Matlab or Octave
    #
    # for a series of points without commas, python has something called
    # multi-line strings that are indicates by three quotes in a row (single or double quotes)
    #
    # format: lat,lon,speed (m/s)
    my_multi_line_string_of_lat_lon_points = """
40.5436398446,-105.1274738576,2.3628680000
40.5438139634,-105.1274902093,2.9084530000
40.5438465477,-105.1274804696,2.9084530000
40.5438718978,-105.1274629495,2.9084530000
40.5439253107,-105.1273752207,2.9041430000
40.5439352742,-105.1273333540,2.9041430000
40.5439368098,-105.1272874056,2.8965330000
40.5439624077,-105.1272109234,3.0778830000
40.5439973121,-105.1270088992,3.3141810000
40.5440207078,-105.1269457858,2.9672830000
40.5440273938,-105.1268769480,3.0781100000
40.5440404536,-105.1268106600,2.8254210000
40.5440452634,-105.1266997202,2.9891400000
40.5440399675,-105.1266464338,2.5779340000
40.5440457531,-105.1265849303,2.8416390000
40.5440440695,-105.1265482561,3.2217040000
40.5440301254,-105.1264711239,3.1939800000
40.5440273459,-105.1264048011,3.1939800000
40.5440320221,-105.1263165301,2.6756260000
40.5440108583,-105.1259536179,2.6275080000
40.5440041576,-105.1259259558,2.6275080000
40.5440196677,-105.1258332095,2.6394860000
40.5440103029,-105.1257707133,2.6071100000
40.5440082069,-105.1257174670,2.2526940000
40.5439914900,-105.1256647558,2.7552130000
40.5439938606,-105.1256323274,2.7552130000
40.5439919507,-105.1256061742,2.3007740000
40.5439511478,-105.1255712319,2.8345690000
40.5439267047,-105.1255676081,2.8345690000
40.5439051357,-105.1255740087,2.2879770000
40.5438858629,-105.1255746093,2.2659430000
40.5438617804,-105.1255912781,2.5171280000
40.5438424702,-105.1256212632,2.5171280000
40.5438233545,-105.1256619287,2.2658960000
40.5438105574,-105.1256790831,2.1319770000
40.5437688553,-105.1257085602,2.4144280000
40.5437516875,-105.1257154143,2.4316020000
40.5437122020,-105.1257167764,1.9777720000
40.5436369130,-105.1256833630,2.4889460000
40.5435858915,-105.1256834401,2.3464730000
40.5435218952,-105.1256122474,2.2253450000
40.5434937268,-105.1255637218,2.2148100000
40.5434701555,-105.1255333510,2.4731590000
40.5433785311,-105.1253491401,2.1827630000
40.5433583803,-105.1252757736,2.2932460000
40.5433359316,-105.1252268688,2.2135380000
40.5433035899,-105.1252085708,2.2838320000
40.5432519118,-105.1252075647,2.2838320000
40.5432077515,-105.1252151243,1.4151240000
40.5431458441,-105.1252082180,1.4396280000
40.5431160887,-105.1252146662,1.3792360000
40.5431009912,-105.1252124270,1.6469900000
40.5430866743,-105.1252156793,1.4946480000
40.5430712916,-105.1252130569,1.5541710000
40.5430428395,-105.1252251232,1.5541710000
40.5430302453,-105.1252201973,1.3266410000
40.5430211511,-105.1252217308,1.4577690000
40.5429908052,-105.1252145244,1.7097420000
40.5429770234,-105.1252228575,1.7097420000
40.5429665092,-105.1252353091,1.4807580000
40.5429552235,-105.1252414796,1.4807580000
40.5429509482,-105.1252481696,1.3229520000
40.5428932013,-105.1252931787,1.4373810000
40.5428812426,-105.1252964344,1.4800790000
40.5428633676,-105.1253187326,1.1817900000
40.5428460299,-105.1253268524,1.1817900000
40.5428325495,-105.1253117765,1.1029080000
40.5428153892,-105.1253019507,1.3076660000
40.5427749893,-105.1252654214,1.4572270000
40.5427333393,-105.1252494054,1.3660040000
40.5426892726,-105.1252578443,1.4073880000
40.5426692252,-105.1252540028,1.3853450000
40.5426603505,-105.1252496438,1.3853450000
40.5426515616,-105.1252491579,1.3074600000
40.5426012978,-105.1252667023,1.5057650000
40.5425907413,-105.1252658990,1.5057650000
40.5425785012,-105.1252701085,1.5431580000
40.5425710725,-105.1252766542,1.6115290000
40.5425397700,-105.1252819801,1.6007070000
40.5425213759,-105.1252790412,1.5035220000
40.5424820555,-105.1252826808,1.3816050000
40.5424679224,-105.1252790781,1.3973620000
40.5424544713,-105.1252803062,1.6360660000
40.5424376309,-105.1252880066,1.6360660000
40.5424259102,-105.1252968082,1.6476270000
40.5424186418,-105.1253088423,1.6476270000
40.5423751351,-105.1253283369,1.3472740000
40.5423353770,-105.1253814305,1.4063340000
40.5423213620,-105.1254244078,1.3714060000
40.5423171273,-105.1254412251,1.4975670000
40.5423047869,-105.1254579416,1.3716180000
40.5423000002,-105.1254762946,1.3873000000
40.5422865767,-105.1255017535,1.4294680000
40.5422795037,-105.1255092538,1.5648910000
40.5422509684,-105.1255548806,1.4450420000
40.5422272652,-105.1255699231,1.4479860000
40.5422295036,-105.1256134212,1.6264580000
40.5422213122,-105.1256480872,1.5210710000
40.5422234614,-105.1256682852,1.5210710000
40.5422068652,-105.1257818204,2.7675170000
40.5421988975,-105.1258030015,2.7698120000
40.5421833125,-105.1259532851,2.8903260000
40.5421880120,-105.1259815529,2.8131070000
40.5421792274,-105.1260363737,2.8131070000
40.5421682382,-105.1260712658,2.3338990000
40.5421517258,-105.1260954340,2.7097820000
40.5421360923,-105.1261106791,2.6109120000
40.5421228397,-105.1261389309,2.7097820000
40.5420886852,-105.1261829664,2.5183360000
40.5420726378,-105.1262148451,2.6135640000
40.5420591203,-105.1262253911,2.6135640000
40.5420445231,-105.1262573898,2.6135640000
40.5420390760,-105.1262767615,2.5005710000
40.5420209114,-105.1263062511,2.0428170000
40.5420000979,-105.1263258248,2.4626580000
40.5419797001,-105.1263348983,2.4626580000
40.5419585073,-105.1263512427,2.5495330000
40.5418945273,-105.1263624090,2.6741390000
40.5418809386,-105.1264116714,2.4322160000
40.5418950769,-105.1264478039,2.4302310000
40.5419611020,-105.1265061019,2.4616300000
40.5419927678,-105.1265544855,2.4616300000
40.5419994666,-105.1265822996,2.3045390000
40.5420131775,-105.1265988479,2.7095850000
40.5420361870,-105.1266139477,2.7095850000
40.5421328309,-105.1266117034,2.4556560000
40.5422314152,-105.1265515084,2.4282750000
40.5422763601,-105.1265453931,2.4282750000
40.5422931500,-105.1265515932,2.3993310000
40.5423123423,-105.1265503458,2.5016520000
40.5423278985,-105.1265391604,2.5016520000
40.5423490524,-105.1265508172,2.4727330000
40.5423724307,-105.1265543355,2.2691160000
40.5424493826,-105.1265939077,2.2126410000
40.5424946630,-105.1265890045,2.1619460000
40.5425179656,-105.1265788649,2.1292100000
40.5425924666,-105.1265259581,1.5932750000
40.5426329398,-105.1265177736,1.4996410000
40.5426660002,-105.1265030842,2.4030840000
40.5427085346,-105.1265043419,2.4030840000
40.5427304072,-105.1265121711,2.4138010000
40.5427689881,-105.1265096866,2.3763720000
40.5427866722,-105.1265004721,2.3833660000
40.5428089481,-105.1265000155,2.3833660000
40.5428325003,-105.1265074389,2.3833660000
40.5428494230,-105.1265082839,2.0415190000
40.5428943156,-105.1265005203,2.4874910000
40.5429618842,-105.1265071681,2.5878640000
40.5430052629,-105.1264981866,2.5219020000
40.5430356096,-105.1265011459,2.4940590000
40.5430586272,-105.1265106479,2.3609720000
40.5430777865,-105.1265137819,2.3609720000
40.5431302355,-105.1265336023,2.3188730000
40.5431455493,-105.1265488537,2.2918880000
40.5431692182,-105.1265615895,2.2918880000
40.5432038469,-105.1265995781,2.6225060000
40.5432267352,-105.1266141642,2.4257560000
40.5432808422,-105.1266762284,2.7981860000
40.5432898607,-105.1267313182,2.6410630000
40.5432988564,-105.1267586037,2.5289350000
40.5433187922,-105.1268021937,2.2840410000
40.5433236220,-105.1268228843,2.2840410000
40.5433358307,-105.1268325146,2.0852660000
40.5433414111,-105.1268522121,1.7405070000
40.5433617813,-105.1268678292,1.6684160000
40.5433639846,-105.1268759292,1.1947720000
40.5433661747,-105.1269007849,1.3001150000
40.5433782719,-105.1269281789,1.3763560000
40.5433806016,-105.1269487467,1.3583670000
40.5433904124,-105.1269855960,1.5750660000
40.5434012090,-105.1270052379,1.6456890000
40.5434153899,-105.1270217040,1.6558480000
40.5434276875,-105.1270310796,1.5120670000
40.5434314426,-105.1270442537,1.5120670000
40.5434396464,-105.1270500261,1.3798830000
40.5434478728,-105.1270607780,1.3641280000
40.5434688926,-105.1271089853,1.5667150000
40.5434704283,-105.1271256452,1.5667150000
40.5434774647,-105.1271619006,1.5458090000
40.5434858483,-105.1271782259,1.4073370000
40.5434872613,-105.1271970282,1.4566110000
40.5434958222,-105.1272233390,1.4407000000
40.5434982054,-105.1272420311,1.4407000000
40.5435153876,-105.1272740546,1.4924700000
40.5435195570,-105.1272896334,1.4924700000
40.5435300020,-105.1273139198,1.4853200000
40.5435426543,-105.1273757279,1.4559890000
40.5435441093,-105.1273979625,1.4150810000
40.5435488347,-105.1274165217,1.4150810000
40.5435568813,-105.1274289730,1.4239810000
40.5435660908,-105.1274375954,1.2158470000
        """
    # all math and route following must be done in orthogonal coordinates, so
    # convert these lat/lon pairs to UTM coordinates which assumes a flat Earth
    # within this UTM zone (there are 60 zones around Earth)
    
    list_of_rows = my_multi_line_string_of_lat_lon_points.split('\n')
    
    # initialize native python lists
    lat=[]
    lon=[]
    X=[]
    Y=[]
    spd_mps=[]
    cnt=0
    #print('---')
    for row in list_of_rows:
        #print(row)
        if row.strip() != '': # strip() removes leading and trailing whitespace
            #print( repr(row) ) # repr() shows invisible characters to distinguish spaces from tabs and newlines
            #str=row.strip().split("\t")
            str=row.strip().split(",")
            #print('processing row {0}: [{1}]'.format(cnt,str))
            
            lat_single_pt   = float( str[0] )
            lon_single_pt   = float( str[1] )
            speed_single_pt = float( str[2] )
            
            # utm library converts from lat/lon in decimal degrees to orthogonal
            # XYZ coordinates in meters within a single UTM zone
            (X_single_pt,Y_single_pt,zone,latBand) = utm.from_latlon(lat_single_pt,lon_single_pt) # these are typically giant numbers, like (X,Y)=(489174.484, 4488110.896)
            
            # list.append() is super fast for growing native python lists dynamically
            lat.append( lat_single_pt )
            lon.append( lon_single_pt )
            X.append( X_single_pt ) # X_origin and Y_origin are also typically giant numbers, like 
            Y.append( Y_single_pt )
            spd_mps.append( speed_single_pt )
            cnt=cnt+1
    #print('---')
    
    lat     = np.array(   lat  ) # convert native python lists to numpy array objects
    lon     = np.array(   lon  )
    X       = np.array(     X  )
    Y       = np.array(     Y  )
    Z       = np.zeros( len(X) ) # (m) initialize Z with numpy array of all zeros
    spd_mps = np.array( spd_mps  ) # (m/s) commanded speed for all waypoints is from SensorLog's original GPS-based recording
    
    
    # --------------------------------------------------------------------------
    # now stuff these arrays (python lists) into the route object create at the top
    # MoVE will use this function's return 'route' object from here on out
    route.lat = lat
    route.lon = lon
    route.X = X # (m) X array in cartesian XYZ coordinates
    route.Y = Y # (m) Y array in cartesian XYZ coordinates
    route.Z = Z # (m) Z array in cartesian XYZ coordinates
    route.spd_mps = spd_mps # (m/s)
    if '__file__' in locals():
        route.source = __file__ # this file's name
    else:
        route.source = __name__ # __main__ when debugging in a terminal console
    
    route = computeRouteStatistics(route)
    
    return(route)
# ------------------------------------------------------------------------------


logging.basicConfig(
    level=logging.DEBUG,
    format='(%(threadName)-14s) %(message)s',
)



if __name__ == "__main__":
    
    import matplotlib.pyplot as plt
    import multiprocessing
    import os
    from routes_functions import computeRouteStatistics, plot_graph
    
    print("starting __main__")
    
    class Cfg:
        pass
    
    cfg = Cfg()
    #cfg.vid=101
    #cfg.L_char = 1.0 # (m) example vehicle's characteristic length
    #cfg.v_max = 20.0 # (m/s) example vehicle's maximum speed
    
    # this function is called from the vehicle model, main_veh_model.py
    route = restore_waypoints( cfg )
    
    # print full route data structure
    #for key,val in route.__dict__.items():
    #    print("{0}={1}".format(key,val))
    
    # optionally bring the plot window up in a separate process
    makePlots=True #False
    if (makePlots==True):
        multiprocessing.Process(target=plot_graph, args=(route,)).start()
        print("exiting main")
        os._exit(0) # this exits immediately with no cleanup or buffer flushing

    
    



























