"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3SourceAction = exports.S3Trigger = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the S3 Action detect changes.
 *
 * This is the type of the {@link S3SourceAction.trigger} property.
 *
 * @stability stable
 */
var S3Trigger;
(function (S3Trigger) {
    S3Trigger["NONE"] = "None";
    S3Trigger["POLL"] = "Poll";
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 *
 * @stability stable
 */
class S3SourceAction extends action_1.Action {
    /**
     * @stability stable
     */
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /**
     * The variables emitted by this action.
     *
     * @stability stable
     */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    /**
     * This is a renamed version of the {@link IAction.bind} method.
     *
     * @stability stable
     */
    bound(_scope, stage, options) {
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = this.generateEventId(stage);
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey],
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role, this.props.bucketKey);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        let ret;
        const baseId = core_1.Names.nodeUniqueId(stage.pipeline.node) + 'SourceEventRule';
        if (core_1.Token.isUnresolved(this.props.bucketKey)) {
            // If bucketKey is a Token, don't include it in the ID.
            // Instead, use numbers to differentiate if multiple actions observe the same bucket
            let candidate = baseId;
            let counter = 0;
            while (this.props.bucket.node.tryFindChild(candidate) !== undefined) {
                counter += 1;
                candidate = baseId + counter;
            }
            ret = candidate;
        }
        else {
            // we can't use Tokens in construct IDs,
            // however, if bucketKey is not a Token,
            // we want it to differentiate between multiple actions
            // observing the same Bucket with different keys
            ret = baseId + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(ret)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
        }
        return ret;
    }
}
exports.S3SourceAction = S3SourceAction;
_a = JSII_RTTI_SYMBOL_1;
S3SourceAction[_a] = { fqn: "@aws-cdk/aws-codepipeline-actions.S3SourceAction", version: "1.111.0" };
//# sourceMappingURL=data:application/json;base64,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