"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToKinesisStreams = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToKinesisStreams class.
 */
class ApiGatewayToKinesisStreams extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayToKinesisStreams class.
     * @since 1.62.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Kinesis stream
        this.kinesisStream = defaults.buildKinesisStream(scope, {
            existingStreamObj: props.existingStreamObj,
            kinesisStreamProps: props.kinesisStreamProps
        });
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole, this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, props.apiGatewayProps, props.logGroupProps);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        this.kinesisStream.grantWrite(this.apiGatewayRole);
        // Setup API Gateway methods
        const requestValidator = this.apiGateway.addRequestValidator('request-validator', {
            requestValidatorName: 'request-body-validator',
            validateRequestBody: true
        });
        // PutRecord
        const putRecordResource = this.apiGateway.root.addResource('record');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecord',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordResource,
            requestTemplate: this.getPutRecordTemplate(props.putRecordRequestTemplate),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.getPutRecordModel(props.putRecordRequestModel) }
        });
        // PutRecords
        const putRecordsResource = this.apiGateway.root.addResource('records');
        defaults.addProxyMethodToApiResource({
            service: 'kinesis',
            action: 'PutRecords',
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: putRecordsResource,
            requestTemplate: this.getPutRecordsTemplate(props.putRecordsRequestTemplate),
            contentType: "'x-amz-json-1.1'",
            requestValidator,
            requestModel: { 'application/json': this.getPutRecordsModel(props.putRecordsRequestModel) }
        });
    }
    getPutRecordTemplate(putRecordTemplate) {
        if (putRecordTemplate !== undefined) {
            return putRecordTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Data": "$util.base64Encode($input.json('$.data'))", "PartitionKey": "$input.path('$.partitionKey')" }`;
    }
    getPutRecordModel(putRecordModel) {
        let modelProps;
        if (putRecordModel !== undefined) {
            modelProps = putRecordModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordModel',
                description: 'PutRecord proxy single-record payload',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecord proxy single-record payload',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['data', 'partitionKey'],
                    properties: {
                        data: { type: api.JsonSchemaType.STRING },
                        partitionKey: { type: api.JsonSchemaType.STRING }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordModel', modelProps);
    }
    getPutRecordsTemplate(putRecordsTemplate) {
        if (putRecordsTemplate !== undefined) {
            return putRecordsTemplate.replace("${StreamName}", this.kinesisStream.streamName);
        }
        return `{ "StreamName": "${this.kinesisStream.streamName}", "Records": [ #foreach($elem in $input.path('$.records')) { "Data": "$util.base64Encode($elem.data)", "PartitionKey": "$elem.partitionKey"}#if($foreach.hasNext),#end #end ] }`;
    }
    getPutRecordsModel(putRecordsModel) {
        let modelProps;
        if (putRecordsModel !== undefined) {
            modelProps = putRecordsModel;
        }
        else {
            modelProps = {
                contentType: 'application/json',
                modelName: 'PutRecordsModel',
                description: 'PutRecords proxy payload data',
                schema: {
                    schema: api.JsonSchemaVersion.DRAFT4,
                    title: 'PutRecords proxy payload data',
                    type: api.JsonSchemaType.OBJECT,
                    required: ['records'],
                    properties: {
                        records: {
                            type: api.JsonSchemaType.ARRAY,
                            items: {
                                type: api.JsonSchemaType.OBJECT,
                                required: ['data', 'partitionKey'],
                                properties: {
                                    data: { type: api.JsonSchemaType.STRING },
                                    partitionKey: { type: api.JsonSchemaType.STRING }
                                }
                            }
                        }
                    }
                }
            };
        }
        return this.apiGateway.addModel('PutRecordsModel', modelProps);
    }
}
exports.ApiGatewayToKinesisStreams = ApiGatewayToKinesisStreams;
_a = JSII_RTTI_SYMBOL_1;
ApiGatewayToKinesisStreams[_a] = { fqn: "@aws-solutions-constructs/aws-apigateway-kinesisstreams.ApiGatewayToKinesisStreams", version: "1.95.2" };
//# sourceMappingURL=data:application/json;base64,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