"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.GitHubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cx_api_1 = require("aws-cdk-lib/cx-api");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const helpers_internal_1 = require("aws-cdk-lib/pipelines/lib/helpers-internal");
const decamelize = require("decamelize");
const aws_credentials_1 = require("./aws-credentials");
const stage_1 = require("./stage");
const github_action_step_1 = require("./steps/github-action-step");
const wave_1 = require("./wave");
const github = require("./workflows-model");
const yaml_file_1 = require("./yaml-file");
const CDKOUT_ARTIFACT = 'cdk.out';
const ASSET_HASH_NAME = 'asset-hash';
/**
 * CDK Pipelines for GitHub workflows.
 */
class GitHubWorkflow extends pipelines_1.PipelineBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.jobOutputs = {};
        this.assetHashMap = {};
        this.stackProperties = {};
        // in order to keep track of if this pipeline has been built so we can
        // catch later calls to addWave() or addStage()
        this.builtGH = false;
        this.cdkCliVersion = props.cdkCliVersion;
        this.preSynthed = props.preSynthed ?? false;
        this.buildContainer = props.buildContainer;
        this.preBuildSteps = props.preBuildSteps ?? [];
        this.postBuildSteps = props.postBuildSteps ?? [];
        this.jobSettings = props.jobSettings;
        this.awsCredentials = this.getAwsCredentials(props);
        this.dockerCredentials = props.dockerCredentials ?? [];
        this.workflowPath = props.workflowPath ?? '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowFile = new yaml_file_1.YamlFile(this.workflowPath);
        this.workflowName = props.workflowName ?? 'deploy';
        this.workflowTriggers = props.workflowTriggers ?? {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
        this.runner = props.runner ?? github.Runner.UBUNTU_LATEST;
        this.publishAssetsAuthRegion = props.publishAssetsAuthRegion ?? 'us-west-2';
    }
    /**
     * Parse AWS credential configuration from deprecated properties For backwards compatibility.
     */
    getAwsCredentials(props) {
        if (props.gitHubActionRoleArn) {
            if (props.awsCreds) {
                throw new Error('Please provide only one method of authentication (remove githubActionRoleArn)');
            }
            return aws_credentials_1.AwsCredentials.fromOpenIdConnect({
                gitHubActionRoleArn: props.gitHubActionRoleArn,
            });
        }
        if (props.awsCredentials) {
            if (props.awsCreds) {
                throw new Error('Please provide only one method of authentication (remove awsCredentials)');
            }
            return aws_credentials_1.AwsCredentials.fromGitHubSecrets({
                accessKeyId: 'AWS_ACCESS_KEY_ID',
                secretAccessKey: 'AWS_SECRET_ACCESS_KEY',
                ...props.awsCredentials,
            });
        }
        return props.awsCreds ?? aws_credentials_1.AwsCredentials.fromGitHubSecrets();
    }
    /**
     * Deploy a single Stage by itself with options for further GitHub configuration.
     *
     * Add a Stage to the pipeline, to be deployed in sequence with other Stages added to the pipeline.
     * All Stacks in the stage will be deployed in an order automatically determined by their relative dependencies.
     */
    addStageWithGitHubOptions(stage, options) {
        const stageDeployment = this.addStage(stage, options);
        // keep track of GitHub specific options
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, 'environment', options?.gitHubEnvironment);
        this.addStackProps(stacks, 'capabilities', options?.stackCapabilities);
        this.addStackProps(stacks, 'settings', options?.jobSettings);
        return stageDeployment;
    }
    /**
     * Add a Wave to the pipeline, for deploying multiple Stages in parallel
     *
     * Use the return object of this method to deploy multiple stages in parallel.
     *
     * Example:
     *
     * ```ts
     * declare const pipeline: GitHubWorkflow; // assign pipeline a value
     *
     * const wave = pipeline.addWave('MyWave');
     * wave.addStage(new MyStage(this, 'Stage1'));
     * wave.addStage(new MyStage(this, 'Stage2'));
     * ```
     */
    addWave(id, options) {
        return this.addGitHubWave(id, options);
    }
    addGitHubWave(id, options) {
        if (this.builtGH) {
            throw new Error("addWave: can't add Waves anymore after buildPipeline() has been called");
        }
        const wave = new wave_1.GitHubWave(id, this, options);
        this.waves.push(wave);
        return wave;
    }
    /**
     * Support adding stages with GitHub options to waves - should ONLY be called internally.
     *
     * Use `pipeline.addWave()` and it'll call this when `wave.addStage()` is called.
     *
     * `pipeline.addStage()` will also call this, since it calls `pipeline.addWave().addStage()`.
     *
     *  @internal
     */
    _addStageFromWave(stage, stageDeployment, options) {
        if (!(stage instanceof stage_1.GitHubStage) && options === undefined) {
            return;
        }
        const ghStage = stage instanceof stage_1.GitHubStage ? stage : undefined;
        // keep track of GitHub specific options
        const stacks = stageDeployment.stacks;
        this.addStackProps(stacks, 'environment', ghStage?.props?.gitHubEnvironment ?? options?.gitHubEnvironment);
        this.addStackProps(stacks, 'capabilities', ghStage?.props?.stackCapabilities ?? options?.stackCapabilities);
        this.addStackProps(stacks, 'settings', ghStage?.props?.jobSettings ?? options?.jobSettings);
    }
    addStackProps(stacks, key, value) {
        if (value === undefined) {
            return;
        }
        for (const stack of stacks) {
            this.stackProperties[stack.stackArtifactId] = {
                ...this.stackProperties[stack.stackArtifactId],
                [key]: value,
            };
        }
    }
    doBuildPipeline() {
        this.builtGH = true;
        const app = aws_cdk_lib_1.Stage.of(this);
        if (!app) {
            throw new Error('The GitHub Workflow must be defined in the scope of an App');
        }
        const cdkoutDir = app.outdir;
        const jobs = new Array();
        const structure = new helpers_internal_1.PipelineGraph(this, {
            selfMutation: false,
            publishTemplate: true,
            prepareStep: false,
        });
        for (const stageNode of flatten(structure.graph.sortedChildren())) {
            if (!helpers_internal_1.isGraph(stageNode)) {
                throw new Error(`Top-level children must be graphs, got '${stageNode}'`);
            }
            const tranches = stageNode.sortedLeaves();
            for (const tranche of tranches) {
                for (const node of tranche) {
                    const job = this.jobForNode(node, {
                        assemblyDir: cdkoutDir,
                        structure,
                    });
                    if (job) {
                        jobs.push(job);
                    }
                }
            }
        }
        // convert jobs to a map and make sure there are no duplicates
        const jobmap = {};
        for (const job of jobs) {
            if (job.id in jobmap) {
                throw new Error(`duplicate job id ${job.id}`);
            }
            jobmap[job.id] = snakeCaseKeys(job.definition);
        }
        // Update jobs with late-bound output requests
        this.insertJobOutputs(jobmap);
        const workflow = {
            name: this.workflowName,
            on: snakeCaseKeys(this.workflowTriggers, '_'),
            jobs: jobmap,
        };
        // write as a yaml file
        this.workflowFile.update(workflow);
        // create directory if it does not exist
        fs_1.mkdirSync(path.dirname(this.workflowPath), { recursive: true });
        // GITHUB_WORKFLOW is set when GitHub Actions is running the workflow.
        // see: https://docs.github.com/en/actions/learn-github-actions/environment-variables#default-environment-variables
        const contextValue = this.node.tryGetContext('cdk-pipelines-github:diffProtection');
        const diffProtection = contextValue === 'false' ? false : contextValue ?? true;
        if (diffProtection && process.env.GITHUB_WORKFLOW === this.workflowName) {
            // check if workflow file has changed
            if (!fs_1.existsSync(this.workflowPath) || this.workflowFile.toYaml() !== fs_1.readFileSync(this.workflowPath, 'utf8')) {
                throw new Error(`Please commit the updated workflow file ${path.relative(__dirname, this.workflowPath)} when you change your pipeline definition.`);
            }
        }
        this.workflowFile.writeFile();
    }
    insertJobOutputs(jobmap) {
        for (const [jobId, jobOutputs] of Object.entries(this.jobOutputs)) {
            jobmap[jobId] = {
                ...jobmap[jobId],
                outputs: {
                    ...jobmap[jobId].outputs,
                    ...this.renderJobOutputs(jobOutputs),
                },
            };
        }
    }
    renderJobOutputs(outputs) {
        const renderedOutputs = {};
        for (const output of outputs) {
            renderedOutputs[output.outputName] = `\${{ steps.${output.stepId}.outputs.${output.outputName} }}`;
        }
        return renderedOutputs;
    }
    /**
     * Make an action from the given node and/or step
     */
    jobForNode(node, options) {
        switch (node.data?.type) {
            // Nothing for these, they are groupings (shouldn't even have popped up here)
            case 'group':
            case 'stack-group':
            case undefined:
                throw new Error(`jobForNode: did not expect to get group nodes: ${node.data?.type}`);
            case 'self-update':
                throw new Error('GitHub Workflows does not support self mutation');
            case 'publish-assets':
                return this.jobForAssetPublish(node, node.data.assets, options);
            case 'prepare':
                throw new Error('"prepare" is not supported by GitHub Workflows');
            case 'execute':
                return this.jobForDeploy(node, node.data.stack, node.data.captureOutputs);
            case 'step':
                if (node.data.isBuildStep) {
                    return this.jobForBuildStep(node, node.data.step);
                }
                else if (node.data.step instanceof pipelines_1.ShellStep) {
                    return this.jobForScriptStep(node, node.data.step);
                }
                else if (node.data.step instanceof github_action_step_1.GitHubActionStep) {
                    return this.jobForGitHubActionStep(node, node.data.step);
                }
                else {
                    throw new Error(`unsupported step type: ${node.data.step.constructor.name}`);
                }
            default:
                // The 'as any' is temporary, until the change upstream rolls out
                throw new Error(`GitHubWorfklow does not support graph nodes of type '${node.data?.type}'. You are probably using a feature this CDK Pipelines implementation does not support.`);
        }
    }
    jobForAssetPublish(node, assets, options) {
        if (assets.length === 0) {
            throw new Error('Asset Publish step must have at least 1 asset');
        }
        const installSuffix = this.cdkCliVersion ? `@${this.cdkCliVersion}` : '';
        const cdkoutDir = options.assemblyDir;
        const jobId = node.uniqueId;
        const assetId = assets[0].assetId;
        // check if asset is docker asset and if we have docker credentials
        const dockerLoginSteps = [];
        if (node.uniqueId.includes('DockerAsset') && this.dockerCredentials.length > 0) {
            for (const creds of this.dockerCredentials) {
                dockerLoginSteps.push(...this.stepsToConfigureDocker(creds));
            }
        }
        // create one file and make one step
        const relativeToAssembly = (p) => path.posix.join(cdkoutDir, path.relative(path.resolve(cdkoutDir), p));
        const fileContents = ['set -ex'].concat(assets.map((asset) => {
            return `npx cdk-assets --path "${relativeToAssembly(asset.assetManifestPath)}" --verbose publish "${asset.assetSelector}"`;
        }));
        // we need the jobId to reference the outputs later
        this.assetHashMap[assetId] = jobId;
        fileContents.push(`echo '${ASSET_HASH_NAME}=${assetId}' >> $GITHUB_OUTPUT`);
        const publishStepFile = path.join(cdkoutDir, `publish-${jobId}-step.sh`);
        fs_1.mkdirSync(path.dirname(publishStepFile), { recursive: true });
        fs_1.writeFileSync(publishStepFile, fileContents.join('\n'), { encoding: 'utf-8' });
        const publishStep = {
            id: 'Publish',
            name: `Publish ${jobId}`,
            run: `/bin/bash ./cdk.out/${path.relative(cdkoutDir, publishStepFile)}`,
        };
        return {
            id: jobId,
            definition: {
                name: `Publish Assets ${jobId}`,
                ...this.renderJobSettingParameters(),
                needs: this.renderDependencies(node),
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.awsCredentials.jobPermission(),
                },
                runsOn: this.runner.runsOn,
                outputs: {
                    [ASSET_HASH_NAME]: `\${{ steps.Publish.outputs.${ASSET_HASH_NAME} }}`,
                },
                steps: [
                    ...this.stepsToDownloadAssembly(cdkoutDir),
                    {
                        name: 'Install',
                        run: `npm install --no-save cdk-assets${installSuffix}`,
                    },
                    ...this.stepsToConfigureAws(this.publishAssetsAuthRegion),
                    ...dockerLoginSteps,
                    publishStep,
                ],
            },
        };
    }
    jobForDeploy(node, stack, _captureOutputs) {
        const region = stack.region;
        const account = stack.account;
        if (!region || !account) {
            throw new Error('"account" and "region" are required');
        }
        if (!stack.templateUrl) {
            throw new Error(`unable to determine template URL for stack ${stack.stackArtifactId}`);
        }
        const resolve = (s) => {
            return cx_api_1.EnvironmentPlaceholders.replace(s, {
                accountId: account,
                region: region,
                partition: 'aws',
            });
        };
        const replaceAssetHash = (template) => {
            const hash = path.parse(template.split('/').pop() ?? '').name;
            if (this.assetHashMap[hash] === undefined) {
                throw new Error(`Template asset hash ${hash} not found.`);
            }
            return template.replace(hash, `\${{ needs.${this.assetHashMap[hash]}.outputs.${ASSET_HASH_NAME} }}`);
        };
        const params = {
            'name': stack.stackName,
            'template': replaceAssetHash(resolve(stack.templateUrl)),
            'no-fail-on-empty-changeset': '1',
        };
        const capabilities = this.stackProperties[stack.stackArtifactId]?.capabilities;
        if (capabilities) {
            params.capabilities = Array(capabilities).join(',');
        }
        if (stack.executionRoleArn) {
            params['role-arn'] = resolve(stack.executionRoleArn);
        }
        const assumeRoleArn = stack.assumeRoleArn ? resolve(stack.assumeRoleArn) : undefined;
        return {
            id: node.uniqueId,
            definition: {
                name: `Deploy ${stack.stackArtifactId}`,
                ...this.renderJobSettingParameters(),
                ...this.stackProperties[stack.stackArtifactId]?.settings,
                permissions: {
                    contents: github.JobPermission.READ,
                    idToken: this.awsCredentials.jobPermission(),
                },
                ...(this.stackProperties[stack.stackArtifactId]?.environment ? {
                    environment: this.stackProperties[stack.stackArtifactId].environment,
                } : {}),
                needs: this.renderDependencies(node),
                runsOn: this.runner.runsOn,
                steps: [
                    ...this.stepsToConfigureAws(region, assumeRoleArn),
                    {
                        id: 'Deploy',
                        uses: 'aws-actions/aws-cloudformation-github-deploy@v1.1.0',
                        with: params,
                    },
                ],
            },
        };
    }
    jobForBuildStep(node, step) {
        if (!(step instanceof pipelines_1.ShellStep)) {
            throw new Error('synthStep must be a ScriptStep');
        }
        if (step.inputs.length > 0) {
            throw new Error('synthStep cannot have inputs');
        }
        if (step.outputs.length > 1) {
            throw new Error('synthStep must have a single output');
        }
        if (!step.primaryOutput) {
            throw new Error('synthStep requires a primaryOutput which contains cdk.out');
        }
        const cdkOut = step.outputs[0];
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: 'Synthesize',
                ...this.renderJobSettingParameters(),
                permissions: {
                    contents: github.JobPermission.READ,
                    // The Synthesize job does not use the GitHub Action Role on its own, but it's possible
                    // that it is being used in the preBuildSteps.
                    idToken: this.awsCredentials.jobPermission(),
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                container: this.buildContainer,
                steps: [
                    ...this.stepsToCheckout(),
                    ...this.preBuildSteps,
                    ...installSteps,
                    {
                        name: 'Build',
                        run: step.commands.join('\n'),
                    },
                    ...this.postBuildSteps,
                    ...this.stepsToUploadAssembly(cdkOut.directory),
                ],
            },
        };
    }
    /**
     * Searches for the stack that produced the output via the current
     * job's dependencies.
     *
     * This function should always find a stack, since it is guaranteed
     * that a CfnOutput comes from a referenced stack.
     */
    findStackOfOutput(ref, node) {
        for (const dep of node.allDeps) {
            if (dep.data?.type === 'execute' && ref.isProducedBy(dep.data.stack)) {
                return dep.uniqueId;
            }
        }
        // Should never happen
        throw new Error(`The output ${ref.outputName} is not referenced by any of the dependent stacks!`);
    }
    addJobOutput(jobId, output) {
        if (this.jobOutputs[jobId] === undefined) {
            this.jobOutputs[jobId] = [output];
        }
        else {
            this.jobOutputs[jobId].push(output);
        }
    }
    jobForScriptStep(node, step) {
        const envVariables = {};
        for (const [envName, ref] of Object.entries(step.envFromCfnOutputs)) {
            const jobId = this.findStackOfOutput(ref, node);
            this.addJobOutput(jobId, {
                outputName: ref.outputName,
                stepId: 'Deploy',
            });
            envVariables[envName] = `\${{ needs.${jobId}.outputs.${ref.outputName} }}`;
        }
        const downloadInputs = new Array();
        const uploadOutputs = new Array();
        for (const input of step.inputs) {
            downloadInputs.push({
                uses: 'actions/download-artifact@v3',
                with: {
                    name: input.fileSet.id,
                    path: input.directory,
                },
            });
        }
        for (const output of step.outputs) {
            uploadOutputs.push({
                uses: 'actions/upload-artifact@v3',
                with: {
                    name: output.fileSet.id,
                    path: output.directory,
                },
            });
        }
        const installSteps = step.installCommands.length > 0 ? [{
                name: 'Install',
                run: step.installCommands.join('\n'),
            }] : [];
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.renderJobSettingParameters(),
                permissions: {
                    contents: github.JobPermission.READ,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: {
                    ...step.env,
                    ...envVariables,
                },
                steps: [
                    ...downloadInputs,
                    ...installSteps,
                    { run: step.commands.join('\n') },
                    ...uploadOutputs,
                ],
            },
        };
    }
    jobForGitHubActionStep(node, step) {
        return {
            id: node.uniqueId,
            definition: {
                name: step.id,
                ...this.renderJobSettingParameters(),
                permissions: {
                    contents: github.JobPermission.WRITE,
                },
                runsOn: this.runner.runsOn,
                needs: this.renderDependencies(node),
                env: step.env,
                steps: step.jobSteps,
            },
        };
    }
    stepsToConfigureAws(region, assumeRoleArn) {
        return this.awsCredentials.credentialSteps(region, assumeRoleArn);
    }
    stepsToConfigureDocker(dockerCredential) {
        let params;
        if (dockerCredential.name === 'docker') {
            params = {
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        else if (dockerCredential.name === 'ecr') {
            params = {
                registry: dockerCredential.registry,
            };
        }
        else {
            params = {
                registry: dockerCredential.registry,
                username: `\${{ secrets.${dockerCredential.usernameKey} }}`,
                password: `\${{ secrets.${dockerCredential.passwordKey} }}`,
            };
        }
        return [
            {
                uses: 'docker/login-action@v2',
                with: params,
            },
        ];
    }
    stepsToDownloadAssembly(targetDir) {
        if (this.preSynthed) {
            return this.stepsToCheckout();
        }
        return [{
                name: `Download ${CDKOUT_ARTIFACT}`,
                uses: 'actions/download-artifact@v3',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: targetDir,
                },
            }];
    }
    stepsToCheckout() {
        return [
            {
                name: 'Checkout',
                uses: 'actions/checkout@v3',
            },
        ];
    }
    stepsToUploadAssembly(dir) {
        if (this.preSynthed) {
            return [];
        }
        return [{
                name: `Upload ${CDKOUT_ARTIFACT}`,
                uses: 'actions/upload-artifact@v3',
                with: {
                    name: CDKOUT_ARTIFACT,
                    path: dir,
                },
            }];
    }
    renderDependencies(node) {
        const deps = new Array();
        for (const d of node.allDeps) {
            if (d instanceof helpers_internal_1.Graph) {
                deps.push(...d.allLeaves().nodes);
            }
            else {
                deps.push(d);
            }
        }
        return deps.map(x => x.uniqueId);
    }
    renderJobSettingParameters() {
        return this.jobSettings;
    }
}
exports.GitHubWorkflow = GitHubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GitHubWorkflow[_a] = { fqn: "cdk-pipelines-github.GitHubWorkflow", version: "0.3.151" };
function snakeCaseKeys(obj, sep = '-') {
    if (typeof obj !== 'object' || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(o => snakeCaseKeys(o, sep));
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        // we don't want to snake case environment variables
        if (k !== 'env' && typeof v === 'object' && v != null) {
            v = snakeCaseKeys(v);
        }
        result[decamelize(k, { separator: sep })] = v;
    }
    return result;
}
function* flatten(xs) {
    for (const x of xs) {
        for (const y of x) {
            yield y;
        }
    }
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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