#!/usr/bin/env python
from setuptools import setup
setup(
  name = 'cs.fileutils',
  author = 'Cameron Simpson',
  author_email = 'cs@cskk.id.au',
  version = '20210717',
  url = 'https://bitbucket.org/cameron_simpson/css/commits/all',
  description =
    'My grab bag of convenience functions for files and filenames/pathnames.',
  long_description =
    ('My grab bag of convenience functions for files and filenames/pathnames.\n'    
 '\n'    
 '*Latest release 20210717*:\n'    
 'Updates for recent cs.mappings-20210717 release.\n'    
 '\n'    
 '## Function `abspath_from_file(path, from_file)`\n'    
 '\n'    
 'Return the absolute path of `path` with respect to `from_file`,\n'    
 'as one might do for an include file.\n'    
 '\n'    
 '## Function `append_ndjson(arg, *a, **kw)`\n'    
 '\n'    
 'Append an iterable of objects to a file as newline delimited JSON.\n'    
 '\n'    
 '## Class `BackedFile(ReadMixin)`\n'    
 '\n'    
 'A RawIOBase duck type\n'    
 'which uses a backing file for initial data\n'    
 'and writes new data to a front scratch file.\n'    
 '\n'    
 '### Method `BackedFile.__init__(self, back_file, dirpath=None)`\n'    
 '\n'    
 'Initialise the BackedFile using `back_file` for the backing data.\n'    
 '\n'    
 '### Method `BackedFile.__enter__(self)`\n'    
 '\n'    
 'BackedFile instances offer a context manager that take the lock,\n'    
 'allowing synchronous use of the file\n'    
 'without implementing a suite of special methods like pread/pwrite.\n'    
 '\n'    
 '### Method `BackedFile.close(self)`\n'    
 '\n'    
 'Close the BackedFile.\n'    
 'Flush contents. Close the front_file if necessary.\n'    
 '\n'    
 '### Method `BackedFile.datafrom(self, offset)`\n'    
 '\n'    
 'Generator yielding natural chunks from the file commencing at offset.\n'    
 '\n'    
 '### Method `BackedFile.seek(self, *a, **kw)`\n'    
 '\n'    
 'Adjust the current file pointer offset.\n'    
 '\n'    
 '### Method `BackedFile.switch_back_file(self, *a, **kw)`\n'    
 '\n'    
 'Switch out one back file for another. Return the old back file.\n'    
 '\n'    
 '### Method `BackedFile.tell(self)`\n'    
 '\n'    
 'Report the current file pointer offset.\n'    
 '\n'    
 '### Method `BackedFile.write(self, *a, **kw)`\n'    
 '\n'    
 'Write data to the front_file.\n'    
 '\n'    
 '## Class `BackedFile_TestMethods`\n'    
 '\n'    
 'Mixin for testing subclasses of BackedFile.\n'    
 'Tests self.backed_fp.\n'    
 '\n'    
 '### Method `BackedFile_TestMethods.test_BackedFile(self)`\n'    
 '\n'    
 'Test function for a BackedFile to use in unit test suites.\n'    
 '\n'    
 '## Function `byteses_as_fd(bss, **kw)`\n'    
 '\n'    
 'Deliver the iterable of bytes `bss` as a readable file descriptor.\n'    
 '      Return the file descriptor.\n'    
 '      Any keyword arguments are passed to `CornuCopyBuffer.as_fd`.\n'    
 '\n'    
 '      Example:\n'    
 '\n'    
 '           # present a passphrase for use as in input file descrptor\n'    
 '           # for a subprocess\n'    
 "           rfd = byteses_as_fd([(passphrase + '\n"    
 "').encode()])\n"    
 '\n'    
 '## Function `common_path_prefix(*paths)`\n'    
 '\n'    
 'Return the common path prefix of the `paths`.\n'    
 '\n'    
 "Note that the common prefix of `'/a/b/c1'` and `'/a/b/c2'`\n"    
 "is `'/a/b/'`, _not_ `'/a/b/c'`.\n"    
 '\n'    
 'Callers may find it useful to preadjust the supplied paths\n'    
 'with `normpath`, `abspath` or `realpath` from `os.path`;\n'    
 'see the `os.path` documentation for the various caveats\n'    
 'which go with those functions.\n'    
 '\n'    
 'Examples:\n'    
 '\n'    
 '    >>> # the obvious\n'    
 "    >>> common_path_prefix('', '')\n"    
 "    ''\n"    
 "    >>> common_path_prefix('/', '/')\n"    
 "    '/'\n"    
 "    >>> common_path_prefix('a', 'a')\n"    
 "    'a'\n"    
 "    >>> common_path_prefix('a', 'b')\n"    
 "    ''\n"    
 '    >>> # nonempty directory path prefixes end in os.sep\n'    
 "    >>> common_path_prefix('/', '/a')\n"    
 "    '/'\n"    
 '    >>> # identical paths include the final basename\n'    
 "    >>> common_path_prefix('p/a', 'p/a')\n"    
 "    'p/a'\n"    
 '    >>> # the comparison does not normalise paths\n'    
 "    >>> common_path_prefix('p//a', 'p//a')\n"    
 "    'p//a'\n"    
 "    >>> common_path_prefix('p//a', 'p//b')\n"    
 "    'p//'\n"    
 "    >>> common_path_prefix('p//a', 'p/a')\n"    
 "    'p/'\n"    
 "    >>> common_path_prefix('p/a', 'p/b')\n"    
 "    'p/'\n"    
 '    >>> # the comparison strips complete unequal path components\n'    
 "    >>> common_path_prefix('p/a1', 'p/a2')\n"    
 "    'p/'\n"    
 "    >>> common_path_prefix('p/a/b1', 'p/a/b2')\n"    
 "    'p/a/'\n"    
 '    >>> # contrast with cs.lex.common_prefix\n'    
 "    >>> common_prefix('abc/def', 'abc/def1')\n"    
 "    'abc/def'\n"    
 "    >>> common_path_prefix('abc/def', 'abc/def1')\n"    
 "    'abc/'\n"    
 "    >>> common_prefix('abc/def', 'abc/def1', 'abc/def2')\n"    
 "    'abc/def'\n"    
 "    >>> common_path_prefix('abc/def', 'abc/def1', 'abc/def2')\n"    
 "    'abc/'\n"    
 '\n'    
 "## Function `compare(f1, f2, mode='rb')`\n"    
 '\n'    
 'Compare the contents of two file-like objects `f1` and `f2` for equality.\n'    
 '\n'    
 'If `f1` or `f2` is a string, open the named file using `mode`\n'    
 '(default: `"rb"`).\n'    
 '\n'    
 '## Function `copy_data(fpin, fpout, nbytes, rsize=None)`\n'    
 '\n'    
 'Copy `nbytes` of data from `fpin` to `fpout`,\n'    
 'return the number of bytes copied.\n'    
 '\n'    
 'Parameters:\n'    
 '* `nbytes`: number of bytes to copy.\n'    
 '  If `None`, copy until EOF.\n'    
 '* `rsize`: read size, default `DEFAULT_READSIZE`.\n'    
 '\n'    
 '## Function `crop_name(name, ext=None, name_max=255)`\n'    
 '\n'    
 'Crop a file basename so as not to exceed `name_max` in length.\n'    
 'Return the original `name` if it already short enough.\n'    
 'Otherwise crop `name` before the file extension\n'    
 'to make it short enough.\n'    
 '\n'    
 'Parameters:\n'    
 '* `name`: the file basename to crop\n'    
 '* `ext`: optional file extension;\n'    
 '  the default is to infer the extension with `os.path.splitext`.\n'    
 '* `name_max`: optional maximum length, default: `255`\n'    
 '\n'    
 '## Function `datafrom(arg, *a, **kw)`\n'    
 '\n'    
 'General purpose reader for files yielding data from `offset`.\n'    
 '\n'    
 '*WARNING*: this function might move the file pointer.\n'    
 '\n'    
 'Parameters:\n'    
 '* `f`: the file from which to read data;\n'    
 '  if a string, the file is opened with mode="rb";\n'    
 '  if an int, treated as an OS file descriptor;\n'    
 '  otherwise presumed to be a file-like object.\n'    
 '  If that object has a `.fileno()` method, treat that as an\n'    
 '  OS file descriptor and use it.\n'    
 '* `offset`: starting offset for the data\n'    
 '* `maxlength`: optional maximum amount of data to yield\n'    
 '* `readsize`: read size, default DEFAULT_READSIZE.\n'    
 '\n'    
 'For file-like objects, the read1 method is used in preference\n'    
 'to read if available. The file pointer is briefly moved during\n'    
 'fetches.\n'    
 '\n'    
 '## Function `datafrom_fd(fd, offset=None, readsize=None, aligned=True, '    
 'maxlength=None)`\n'    
 '\n'    
 'General purpose reader for file descriptors yielding data from `offset`.\n'    
 '**Note**: This does not move the file descriptor position\n'    
 '**if** the file is seekable.\n'    
 '\n'    
 'Parameters:\n'    
 '* `fd`: the file descriptor from which to read.\n'    
 '* `offset`: the offset from which to read.\n'    
 '  If omitted, use the current file descriptor position.\n'    
 '* `readsize`: the read size, default: `DEFAULT_READSIZE`\n'    
 '* `aligned`: if true (the default), the first read is sized\n'    
 '  to align the new offset with a multiple of `readsize`.\n'    
 '* `maxlength`: if specified yield no more than this many bytes of data.\n'    
 '\n'    
 '## Function `file_based(*da, **dkw)`\n'    
 '\n'    
 'A decorator which caches a value obtained from a file.\n'    
 '\n'    
 'In addition to all the keyword arguments for `@cs.deco.cachedmethod`,\n'    
 'this decorator also accepts the following arguments:\n'    
 '* `attr_name`: the name for the associated attribute, used as\n'    
 '  the basis for the internal cache value attribute\n'    
 '* `filename`: the filename to monitor.\n'    
 '  Default from the `._{attr_name}__filename` attribute.\n'    
 '  This value will be passed to the method as the `filename` keyword\n'    
 '  parameter.\n'    
 '* `poll_delay`: delay between file polls, default `DEFAULT_POLL_INTERVAL`.\n'    
 '* `sig_func`: signature function used to encapsulate the relevant\n'    
 '  information about the file; default\n'    
 '  cs.filestate.FileState({filename}).\n'    
 '\n'    
 'If the decorated function raises OSError with errno == ENOENT,\n'    
 'this returns None. Other exceptions are reraised.\n'    
 '\n'    
 '## Function `file_data(fp, nbytes=None, rsize=None)`\n'    
 '\n'    
 'Read `nbytes` of data from `fp` and yield the chunks as read.\n'    
 '\n'    
 'Parameters:\n'    
 '* `nbytes`: number of bytes to read; if None read until EOF.\n'    
 '* `rsize`: read size, default DEFAULT_READSIZE.\n'    
 '\n'    
 '## Function `file_property(*da, **dkw)`\n'    
 '\n'    
 'A property whose value reloads if a file changes.\n'    
 '\n'    
 '## Function `files_property(func)`\n'    
 '\n'    
 'A property whose value reloads if any of a list of files changes.\n'    
 '\n'    
 'Note: this is just the default mode for `make_files_property`.\n'    
 '\n'    
 '`func` accepts the file path and returns the new value.\n'    
 "The underlying attribute name is `'_'+func.__name__`,\n"    
 'the default from `make_files_property()`.\n'    
 'The attribute *{attr_name}*`_lock` is a mutex controlling access to the '    
 'property.\n'    
 'The attributes *{attr_name}*`_filestates` and *{attr_name}*`_paths` track '    
 'the\n'    
 'associated file states.\n'    
 'The attribute *{attr_name}*`_lastpoll` tracks the last poll time.\n'    
 '\n'    
 'The decorated function is passed the current list of files\n'    
 'and returns the new list of files and the associated value.\n'    
 '\n'    
 'One example use would be a configuration file with recurive\n'    
 'include operations; the inner function would parse the first\n'    
 'file in the list, and the parse would accumulate this filename\n'    
 'and those of any included files so that they can be monitored,\n'    
 'triggering a fresh parse if one changes.\n'    
 '\n'    
 'Example:\n'    
 '\n'    
 '    class C(object):\n'    
 '      def __init__(self):\n'    
 "        self._foo_path = '.foorc'\n"    
 '      @files_property\n'    
 '      def foo(self,paths):\n'    
 '        new_paths, result = parse(paths[0])\n'    
 '        return new_paths, result\n'    
 '\n'    
 'The load function is called on the first access and on every\n'    
 "access thereafter where an associated file's `FileState` has\n"    
 'changed and the time since the last successful load exceeds\n'    
 'the poll_rate (1s). An attempt at avoiding races is made by\n'    
 'ignoring reloads that raise exceptions and ignoring reloads\n'    
 'where files that were stat()ed during the change check have\n'    
 'changed state after the load.\n'    
 '\n'    
 '## Function `find(path, select=None, sort_names=True)`\n'    
 '\n'    
 'Walk a directory tree `path`\n'    
 'yielding selected paths.\n'    
 '\n'    
 'Note: not selecting a directory prunes all its descendants.\n'    
 '\n'    
 '## Function `findup(path, test, first=False)`\n'    
 '\n'    
 'Test the pathname `abspath(path)` and each of its ancestors\n'    
 'against the callable `test`,\n'    
 'yielding paths satisfying the test.\n'    
 '\n'    
 'If `first` is true (default `False`)\n'    
 'this function always yields exactly one value,\n'    
 'either the first path satisfying the test or `None`.\n'    
 'This mode supports a use such as:\n'    
 '\n'    
 '    matched_path = next(findup(path, test, first=True))\n'    
 '    # post condition: matched_path will be `None` on no match\n'    
 '    # otherwise the first matching path\n'    
 '\n'    
 '## Function `iter_fd(fd, **kw)`\n'    
 '\n'    
 'Iterate over data from the file descriptor `fd`.\n'    
 '\n'    
 '## Function `iter_file(f, **kw)`\n'    
 '\n'    
 'Iterate over data from the file `f`.\n'    
 '\n'    
 '## Function `lines_of(fp, partials=None)`\n'    
 '\n'    
 'Generator yielding lines from a file until EOF.\n'    
 'Intended for file-like objects that lack a line iteration API.\n'    
 '\n'    
 '## Function `lockfile(path, ext=None, poll_interval=None, timeout=None, '    
 'runstate=None)`\n'    
 '\n'    
 'A context manager which takes and holds a lock file.\n'    
 '\n'    
 'Parameters:\n'    
 '* `path`: the base associated with the lock file.\n'    
 '* `ext`: the extension to the base used to construct the lock file name.\n'    
 "  Default: `'.lock'`\n"    
 '* `timeout`: maximum time to wait before failing.\n'    
 '  Default: `None` (wait forever).\n'    
 '* `poll_interval`: polling frequency when timeout is not `0`.\n'    
 '* `runstate`: optional `RunState` duck instance supporting cancellation.\n'    
 '\n'    
 '## Function `longpath(path, environ=None, prefixes=None)`\n'    
 '\n'    
 'Return `path` with prefixes and environment variables substituted.\n'    
 'The converse of `shortpath()`.\n'    
 '\n'    
 '## Function `make_files_property(attr_name=None, unset_object=None, '    
 'poll_rate=1.0)`\n'    
 '\n'    
 'Construct a decorator that watches multiple associated files.\n'    
 '\n'    
 'Parameters:\n'    
 "* `attr_name`: the underlying attribute, default: `'_'+func.__name__`\n"    
 '* `unset_object`: the sentinel value for "uninitialised", default: `None`\n'    
 '* `poll_rate`: how often in seconds to poll the file for changes,\n'    
 '  default from `DEFAULT_POLL_INTERVAL`: `1.0`\n'    
 '\n'    
 'The attribute *attr_name*`_lock` controls access to the property.\n'    
 'The attributes *attr_name*`_filestates` and *attr_name*`_paths` track the\n'    
 "associated files' state.\n"    
 'The attribute *attr_name*`_lastpoll` tracks the last poll time.\n'    
 '\n'    
 'The decorated function is passed the current list of files\n'    
 'and returns the new list of files and the associated value.\n'    
 '\n'    
 'One example use would be a configuration file with recursive\n'    
 'include operations; the inner function would parse the first\n'    
 'file in the list, and the parse would accumulate this filename\n'    
 'and those of any included files so that they can be monitored,\n'    
 'triggering a fresh parse if one changes.\n'    
 '\n'    
 'Example:\n'    
 '\n'    
 '    class C(object):\n'    
 '      def __init__(self):\n'    
 "        self._foo_path = '.foorc'\n"    
 '      @files_property\n'    
 '      def foo(self,paths):\n'    
 '        new_paths, result = parse(paths[0])\n'    
 '        return new_paths, result\n'    
 '\n'    
 'The load function is called on the first access and on every\n'    
 "access thereafter where an associated file's `FileState` has\n"    
 'changed and the time since the last successful load exceeds\n'    
 'the `poll_rate`.\n'    
 '\n'    
 'An attempt at avoiding races is made by\n'    
 'ignoring reloads that raise exceptions and ignoring reloads\n'    
 'where files that were `os.stat()`ed during the change check have\n'    
 'changed state after the load.\n'    
 '\n'    
 '## Function `makelockfile(path, ext=None, poll_interval=None, timeout=None, '    
 'runstate=None)`\n'    
 '\n'    
 'Create a lockfile and return its path.\n'    
 '\n'    
 'The lockfile can be removed with `os.remove`.\n'    
 'This is the core functionality supporting the `lockfile()`\n'    
 'context manager.\n'    
 '\n'    
 'Parameters:\n'    
 '* `path`: the base associated with the lock file,\n'    
 '  often the filesystem object whose access is being managed.\n'    
 '* `ext`: the extension to the base used to construct the lockfile name.\n'    
 '  Default: ".lock"\n'    
 '* `timeout`: maximum time to wait before failing.\n'    
 '  Default: `None` (wait forever).\n'    
 '  Note that zero is an accepted value\n'    
 '  and requires the lock to succeed on the first attempt.\n'    
 '* `poll_interval`: polling frequency when timeout is not 0.\n'    
 '* `runstate`: optional `RunState` duck instance supporting cancellation.\n'    
 '  Note that if a cancelled `RunState` is provided\n'    
 '  no attempt will be made to make the lockfile.\n'    
 '\n'    
 '## Function `max_suffix(dirpath, pfx)`\n'    
 '\n'    
 'Compute the highest existing numeric suffix\n'    
 'for names starting with the prefix `pfx`.\n'    
 '\n'    
 'This is generally used as a starting point for picking\n'    
 'a new numeric suffix.\n'    
 '\n'    
 "## Function `mkdirn(path, sep='')`\n"    
 '\n'    
 'Create a new directory named `path+sep+n`,\n'    
 'where `n` exceeds any name already present.\n'    
 '\n'    
 'Parameters:\n'    
 '* `path`: the basic directory path.\n'    
 '* `sep`: a separator between `path` and `n`.\n'    
 "  Default: `''`\n"    
 '\n'    
 '## Function `NamedTemporaryCopy(f, progress=False, progress_label=None, '    
 '**kw)`\n'    
 '\n'    
 'A context manager yielding a temporary copy of `filename`\n'    
 'as returned by `NamedTemporaryFile(**kw)`.\n'    
 '\n'    
 'Parameters:\n'    
 '* `f`: the name of the file to copy, or an open binary file,\n'    
 '  or a `CornuCopyBuffer`\n'    
 '* `progress`: an optional progress indicator, default `False`;\n'    
 '  if a `bool`, show a progress bar for the copy phase if true;\n'    
 '  if an `int`, show a progress bar for the copy phase\n'    
 '  if the file size equals or exceeds the value;\n'    
 '  otherwise it should be a `cs.progress.Progress` instance\n'    
 '* `progress_label`: option progress bar label,\n'    
 '  only used if a progress bar is made\n'    
 'Other keyword parameters are passed to `tempfile.NamedTemporaryFile`.\n'    
 '\n'    
 '## Class `NullFile`\n'    
 '\n'    
 'Writable file that discards its input.\n'    
 '\n'    
 'Note that this is _not_ an open of `os.devnull`;\n'    
 'it just discards writes and is not the underlying file descriptor.\n'    
 '\n'    
 '### Method `NullFile.__init__(self)`\n'    
 '\n'    
 'Initialise the file offset to 0.\n'    
 '\n'    
 '### Method `NullFile.flush(self)`\n'    
 '\n'    
 'Flush buffered data to the subsystem.\n'    
 '\n'    
 '### Method `NullFile.write(self, data)`\n'    
 '\n'    
 'Discard data, advance file offset by length of data.\n'    
 '\n'    
 '## Class `Pathname(builtins.str)`\n'    
 '\n'    
 'Subclass of str presenting convenience properties useful for\n'    
 'format strings related to file paths.\n'    
 '\n'    
 '### Method `Pathname.__format__(self, fmt_spec)`\n'    
 '\n'    
 'Calling format(<Pathname>, fmt_spec) treat `fmt_spec` as a new style\n'    
 'formatting string with a single positional parameter of `self`.\n'    
 '\n'    
 '### Property `Pathname.abs`\n'    
 '\n'    
 'The absolute form of this Pathname.\n'    
 '\n'    
 '### Property `Pathname.basename`\n'    
 '\n'    
 'The basename of this Pathname.\n'    
 '\n'    
 '### Property `Pathname.dirname`\n'    
 '\n'    
 'The dirname of the Pathname.\n'    
 '\n'    
 '### Property `Pathname.isabs`\n'    
 '\n'    
 'Whether this Pathname is an absolute Pathname.\n'    
 '\n'    
 '### Property `Pathname.short`\n'    
 '\n'    
 'The shortened form of this Pathname.\n'    
 '\n'    
 '### Method `Pathname.shorten(self, environ=None, prefixes=None)`\n'    
 '\n'    
 'Shorten a Pathname using ~ and ~user.\n'    
 '\n'    
 '## Function `poll_file(path, old_state, reload_file, missing_ok=False)`\n'    
 '\n'    
 'Watch a file for modification by polling its state as obtained\n'    
 'by `FileState()`.\n'    
 'Call `reload_file(path)` if the state changes.\n'    
 'Return `(new_state,reload_file(path))` if the file was modified\n'    
 'and was unchanged (stable state) before and after the reload_file().\n'    
 'Otherwise return `(None,None)`.\n'    
 '\n'    
 'This may raise an `OSError` if the `path` cannot be `os.stat()`ed\n'    
 'and of course for any exceptions that occur calling `reload_file`.\n'    
 '\n'    
 'If `missing_ok` is true then a failure to `os.stat()` which\n'    
 'raises `OSError` with `ENOENT` will just return `(None,None)`.\n'    
 '\n'    
 '## Function `read_data(fp, nbytes, rsize=None)`\n'    
 '\n'    
 'Read `nbytes` of data from `fp`, return the data.\n'    
 '\n'    
 'Parameters:\n'    
 '* `nbytes`: number of bytes to copy.\n'    
 '  If `None`, copy until EOF.\n'    
 '* `rsize`: read size, default `DEFAULT_READSIZE`.\n'    
 '\n'    
 '## Function `read_from(fp, rsize=None, tail_mode=False, tail_delay=None)`\n'    
 '\n'    
 'Generator to present text or data from an open file until EOF.\n'    
 '\n'    
 'Parameters:\n'    
 '* `rsize`: read size, default: DEFAULT_READSIZE\n'    
 '* `tail_mode`: if true, yield an empty chunk at EOF, allowing resumption\n'    
 '  if the file grows.\n'    
 '\n'    
 '## Class `ReadMixin`\n'    
 '\n'    
 'Useful read methods to accomodate modes not necessarily available in a '    
 'class.\n'    
 '\n'    
 'Note that this mixin presumes that the attribute `self._lock`\n'    
 'is a threading.RLock like context manager.\n'    
 '\n'    
 'Classes using this mixin should consider overriding the default\n'    
 '.datafrom method with something more efficient or direct.\n'    
 '\n'    
 '### Method `ReadMixin.bufferfrom(self, offset)`\n'    
 '\n'    
 'Return a CornuCopyBuffer from the specified `offset`.\n'    
 '\n'    
 '### Method `ReadMixin.datafrom(self, offset, readsize=None)`\n'    
 '\n'    
 'Yield data from the specified `offset` onward in some\n'    
 'approximation of the "natural" chunk size.\n'    
 '\n'    
 '*NOTE*: UNLIKE the global datafrom() function, this method\n'    
 'MUST NOT move the logical file position. Implementors may need\n'    
 'to save and restore the file pointer within a lock around\n'    
 'the I/O if they do not use a direct access method like\n'    
 'os.pread.\n'    
 '\n'    
 'The aspiration here is to read data with only a single call\n'    
 'to the underlying storage, and to return the chunks in\n'    
 'natural sizes instead of some default read size.\n'    
 '\n'    
 'Classes using this mixin must implement this method.\n'    
 '\n'    
 '### Method `ReadMixin.read(self, size=-1, offset=None, longread=False)`\n'    
 '\n'    
 'Read up to `size` bytes, honouring the "single system call"\n'    
 'spirit unless `longread` is true.\n'    
 '\n'    
 'Parameters:\n'    
 '* `size`: the number of bytes requested. A size of -1 requests\n'    
 '  all bytes to the end of the file.\n'    
 '* `offset`: the starting point of the read; if None, use the\n'    
 '  current file position; if not None, seek to this position\n'    
 '  before reading, even if `size` == 0.\n'    
 '* `longread`: switch from "single system call" to "as many\n'    
 '  as required to obtain `size` bytes"; short data will still\n'    
 '  be returned if the file is too short.\n'    
 '\n'    
 '### Method `ReadMixin.read_n(self, n)`\n'    
 '\n'    
 'Read `n` bytes of data and return them.\n'    
 '\n'    
 'Unlike traditional file.read(), RawIOBase.read() may return short\n'    
 'data, thus this workalike, which may only return short data if it\n'    
 'hits EOF.\n'    
 '\n'    
 '### Method `ReadMixin.readinto(self, *a, **kw)`\n'    
 '\n'    
 'Read data into a bytearray.\n'    
 '\n'    
 "## Function `rewrite(filepath, srcf, mode='w', backup_ext=None, "    
 'do_rename=False, do_diff=None, empty_ok=False, overwrite_anyway=False)`\n'    
 '\n'    
 'Rewrite the file `filepath` with data from the file object `srcf`.\n'    
 '\n'    
 'Parameters:\n'    
 '* `filepath`: the name of the file to rewrite.\n'    
 '* `srcf`: the source file containing the new content.\n'    
 "* `mode`: the write-mode for the file, default `'w'` (for text);\n"    
 "  use `'wb'` for binary data.\n"    
 '* `empty_ok`: if true (default `False`),\n'    
 '  do not raise `ValueError` if the new data are empty.\n'    
 '* `overwrite_anyway`: if true (default `False`),\n'    
 '  skip the content check and overwrite unconditionally.\n'    
 '* `backup_ext`: if a nonempty string,\n'    
 '  take a backup of the original at `filepath + backup_ext`.\n'    
 '* `do_diff`: if not `None`, call `do_diff(filepath,tempfile)`.\n'    
 '* `do_rename`: if true (default `False`),\n'    
 '  rename the temp file to `filepath`\n'    
 '  after copying the permission bits.\n'    
 '  Otherwise (default), copy the tempfile to `filepath`;\n'    
 "  this preserves the file's inode and permissions etc.\n"    
 '\n'    
 "## Function `rewrite_cmgr(filepath, mode='w', **kw)`\n"    
 '\n'    
 'Rewrite a file, presented as a context manager.\n'    
 '\n'    
 'Parameters:\n'    
 '* `mode`: file write mode, defaulting to "w" for text.\n'    
 '\n'    
 'Other keyword parameters are passed to `rewrite()`.\n'    
 '\n'    
 'Example:\n'    
 '\n'    
 '    with rewrite_cmgr(pathname, do_rename=True) as f:\n'    
 '        ... write new content to f ...\n'    
 '\n'    
 '## Class `RWFileBlockCache`\n'    
 '\n'    
 'A scratch file for storing data.\n'    
 '\n'    
 '### Method `RWFileBlockCache.__init__(self, pathname=None, dirpath=None, '    
 'suffix=None, lock=None)`\n'    
 '\n'    
 'Initialise the file.\n'    
 '\n'    
 'Parameters:\n'    
 '* `pathname`: path of file. If None, create a new file with\n'    
 '  tempfile.mkstemp using dir=`dirpath` and unlink that file once\n'    
 '  opened.\n'    
 '* `dirpath`: location for the file if made by mkstemp as above.\n'    
 '* `lock`: an object to use as a mutex, allowing sharing with\n'    
 '  some outer system. A Lock will be allocated if omitted.\n'    
 '\n'    
 '### Method `RWFileBlockCache.close(self)`\n'    
 '\n'    
 'Close the file descriptors.\n'    
 '\n'    
 '### Property `RWFileBlockCache.closed`\n'    
 '\n'    
 'Test whether the file descriptor has been closed.\n'    
 '\n'    
 '### Method `RWFileBlockCache.get(self, offset, length)`\n'    
 '\n'    
 'Get data from `offset` of length `length`.\n'    
 '\n'    
 '### Method `RWFileBlockCache.put(self, data)`\n'    
 '\n'    
 'Store `data`, return offset.\n'    
 '\n'    
 '## Function `saferename(oldpath, newpath)`\n'    
 '\n'    
 'Rename a path using `os.rename()`,\n'    
 'but raise an exception if the target path already exists.\n'    
 'Note: slightly racey.\n'    
 '\n'    
 '## Function `scan_ndjson(arg, *a, **kw)`\n'    
 '\n'    
 'Read a newline delimited JSON file, yield instances of `dictclass`\n'    
 '(default `dict`, otherwise a class which can be instantiated\n'    
 'by `dictclass(a_dict)`).\n'    
 '\n'    
 '`error_list` is an optional list to accrue `(lineno,exception)` tuples\n'    
 'for errors encountered during the scan.\n'    
 '\n'    
 '## Function `seekable(fp)`\n'    
 '\n'    
 'Try to test whether a filelike object is seekable.\n'    
 '\n'    
 'First try the `IOBase.seekable` method, otherwise try getting a file\n'    
 'descriptor from `fp.fileno` and `os.stat()`ing that,\n'    
 'otherwise return `False`.\n'    
 '\n'    
 '## Function `shortpath(path, environ=None, prefixes=None)`\n'    
 '\n'    
 'Return `path` with the first matching leading prefix replaced.\n'    
 '\n'    
 'Parameters:\n'    
 '* `environ`: environment mapping if not os.environ\n'    
 '* `prefixes`: iterable of `(prefix,subst)` to consider for replacement;\n'    
 '  each `prefix` is subject to environment variable\n'    
 '  substitution before consideration\n'    
 '  The default considers "$HOME/" for replacement by "~/".\n'    
 '\n'    
 '## Class `Tee`\n'    
 '\n'    
 'An object with .write, .flush and .close methods\n'    
 'which copies data to multiple output files.\n'    
 '\n'    
 '### Method `Tee.__init__(self, *fps)`\n'    
 '\n'    
 'Initialise the Tee; any arguments are taken to be output file objects.\n'    
 '\n'    
 '### Method `Tee.add(self, output)`\n'    
 '\n'    
 'Add a new output.\n'    
 '\n'    
 '### Method `Tee.close(self)`\n'    
 '\n'    
 'Close all the outputs and close the Tee.\n'    
 '\n'    
 '### Method `Tee.flush(self)`\n'    
 '\n'    
 'Flush all the outputs.\n'    
 '\n'    
 '### Method `Tee.write(self, data)`\n'    
 '\n'    
 'Write the data to all the outputs.\n'    
 'Note: does not detect or accodmodate short writes.\n'    
 '\n'    
 '## Function `tee(fp, fp2)`\n'    
 '\n'    
 'Context manager duplicating .write and .flush from fp to fp2.\n'    
 '\n'    
 '## Function `tmpdir()`\n'    
 '\n'    
 'Return the pathname of the default temporary directory for scratch data,\n'    
 "the environment variable `$TMPDIR` or `'/tmp'`.\n"    
 '\n'    
 '## Function `tmpdirn(tmp=None)`\n'    
 '\n'    
 'Make a new temporary directory with a numeric suffix.\n'    
 '\n'    
 '## Function `trysaferename(oldpath, newpath)`\n'    
 '\n'    
 'A `saferename()` that returns `True` on success,\n'    
 '`False` on failure.\n'    
 '\n'    
 '## Class '    
 '`UUIDNDJSONMapping(cs.obj.SingletonMixin,cs.mappings.IndexedSetMixin)`\n'    
 '\n'    
 'A subclass of `IndexedSetMixin` which maintains records\n'    
 'from a newline delimited JSON file.\n'    
 '\n'    
 '### Method `UUIDNDJSONMapping.__init__(self, filename, dictclass=<class '    
 "'cs.mappings.UUIDedDict'>, create=False)`\n"    
 '\n'    
 'Initialise the mapping.\n'    
 '\n'    
 'Parameters:\n'    
 '* `filename`: the file containing the newline delimited JSON data;\n'    
 '  this need not yet exist\n'    
 '* `dictclass`: a optional `dict` subclass to hold each record,\n'    
 '  default `UUIDedDict`\n'    
 '* `create`: if true, ensure the file exists\n'    
 '  by transiently opening it for append if it is missing;\n'    
 '  default `False`\n'    
 '\n'    
 '### Method `UUIDNDJSONMapping.add_backend(self, record)`\n'    
 '\n'    
 'Append `record` to the backing file.\n'    
 '\n'    
 '### Method `UUIDNDJSONMapping.rewrite_backend(self)`\n'    
 '\n'    
 'Rewrite the backing file.\n'    
 '\n'    
 'Because the record updates are normally written in append mode,\n'    
 'a rewrite will be required every so often.\n'    
 '\n'    
 '### Method `UUIDNDJSONMapping.scan(self)`\n'    
 '\n'    
 'Scan the backing file, yield records.\n'    
 '\n'    
 '## Function `write_ndjson(arg, *a, **kw)`\n'    
 '\n'    
 'Transcribe an iterable of objects to a file as newline delimited JSON.\n'    
 '\n'    
 '# Release Log\n'    
 '\n'    
 '\n'    
 '\n'    
 '*Release 20210717*:\n'    
 'Updates for recent cs.mappings-20210717 release.\n'    
 '\n'    
 '*Release 20210420*:\n'    
 '* Forensic prefix for NamedTemporaryCopy.\n'    
 '* UUIDNDJSONMapping: provide an empty .scan_errors on instantiation, avoids '    
 'AttributeError if a scan never occurs.\n'    
 '\n'    
 '*Release 20210306*:\n'    
 '* datafrom_fd: fix use-before-set of is_seekable.\n'    
 '* RWFileBlockCache.put: remove assert(len(data)>0), adjust logic.\n'    
 '\n'    
 '*Release 20210131*:\n'    
 'crop_name: put ext before name_max, more likely to be specified, I think.\n'    
 '\n'    
 '*Release 20201227.1*:\n'    
 'Docstring tweak.\n'    
 '\n'    
 '*Release 20201227*:\n'    
 'scan_ndjson: optional errors_list to accrue errors during the scan.\n'    
 '\n'    
 '*Release 20201108*:\n'    
 'Bugfix rewrite_cmgr, failed to flush a file before copying its contents.\n'    
 '\n'    
 '*Release 20201102*:\n'    
 '* Newline delimited JSON (ndjson) support.\n'    
 '* New UUIDNDJSONMapping implementing a singleton '    
 'cs.mappings.LoadableMappingMixin of cs.mappings.UUIDedDict subclass '    
 'instances backed by an NDJSON file.\n'    
 '* New scan_ndjson() function to yield newline delimited JSON records.\n'    
 '* New write_ndjson() function to write newline delimited JSON records.\n'    
 '* New append_ndjson() function to append a single newline delimited JSON '    
 'record to a file.\n'    
 '* New NamedTemporaryCopy for creating a temporary copy of a file with an '    
 'optional progress bar.\n'    
 '* rewrite_cmgr: turn into a simple wrapper for rewrite.\n'    
 '* datafrom: make the offset parameter optional, tweak the @strable open '    
 'function.\n'    
 '* datafrom_fd: support nonseekable file descriptors, document that for these '    
 'the file position is moved (no pread support).\n'    
 "* New iter_fd and iter_file to return iterators of a file's data by "    
 'utilising a CornuCopyBuffer.\n'    
 '* New byteses_as_fd to return a readable file descriptor receiving an '    
 'iterable of bytes via a CornuCopyBuffer.\n'    
 '\n'    
 '*Release 20200914*:\n'    
 'New common_path_prefix to compare pathnames.\n'    
 '\n'    
 '*Release 20200517*:\n'    
 '* New crop_name() function to crop a file basename to fit within a specific '    
 'length.\n'    
 '* New find() function complimenting findup (UNTESTED).\n'    
 '\n'    
 '*Release 20200318*:\n'    
 'New findup(path,test) generator to walk up a file tree.\n'    
 '\n'    
 '*Release 20191006*:\n'    
 'Adjust import of cs.deco.cachedmethod.\n'    
 '\n'    
 '*Release 20190729*:\n'    
 'datafrom_fd: make `offset` optional, defaulting to fd position at call.\n'    
 '\n'    
 '*Release 20190617*:\n'    
 '@file_based: adjust use of @cached from cached(wrap0, **dkw) to '    
 'cached(**dkw)(wrap0).\n'    
 '\n'    
 '*Release 20190101*:\n'    
 'datafrom: add maxlength keyword arg, bugfix fd and f.fileno cases.\n'    
 '\n'    
 '*Release 20181109*:\n'    
 '* Various bugfixes for BackedFile.\n'    
 "* Use a file's .read1 method if available in some scenarios.\n"    
 '* makelockfile: accept am optional RunState control parameter, improve some '    
 'behaviour.\n'    
 '* datafrom_fd: new optional maxlength parameter limiting the amount of data '    
 'returned.\n'    
 '* datafrom_fd: by default, perform an initial read to align all subsequent '    
 'reads with the readsize.\n'    
 '* drop fdreader, add datafrom(f, offset, readsize) accepting a file or a '    
 'file descriptor, expose datafrom_fd.\n'    
 '* ReadMixin.datafrom now mandatory. Add ReadMixin.bufferfrom.\n'    
 '* Assorted other improvements, minor bugfixes, documentation improvements.\n'    
 '\n'    
 '*Release 20171231.1*:\n'    
 'Trite DISTINFO fix, no semantic changes.\n'    
 '\n'    
 '*Release 20171231*:\n'    
 'Update imports, bump DEFAULT_READSIZE from 8KiB to 128KiB.\n'    
 '\n'    
 '*Release 20170608*:\n'    
 '* Move lockfile and the SharedAppend* classes to cs.sharedfile.\n'    
 '* BackedFile internal changes.\n'    
 '\n'    
 '*Release 20160918*:\n'    
 '* BackedFile: redo implementation of .front_file to fix resource leak; add '    
 '.__len__; add methods .spans, .front_spans and .back_spans to return '    
 'information about front vs back data.\n'    
 '* seek: bugfix: seek should return the new file offset.\n'    
 '* BackedFile does not subclass RawIOBase, it just works like one.\n'    
 '\n'    
 '*Release 20160828*:\n'    
 '* Use "install_requires" instead of "requires" in DISTINFO.\n'    
 '* Rename maxFilenameSuffix to max_suffix.\n'    
 '* Pull in OpenSocket file-like socket wrapper from cs.venti.tcp.\n'    
 '* Update for cs.asynchron changes.\n'    
 '* ... then move cs.fileutils.OpenSocket into new module cs.socketutils.\n'    
 '* New Tee class, for copying output to multiple files.\n'    
 '* NullFile class which discards writes (==> no-op for Tee).\n'    
 '* New class SavingFile to accrue output and move to specified pathname when '    
 'complete.\n'    
 '* Memory usage improvements.\n'    
 '* Polyfill non-threadsafe implementation of pread if os.pread does not '    
 'exist.\n'    
 '* New function seekable() to probe a file for seekability.\n'    
 '* SharedAppendFile: provide new .open(filemode) context manager for allowing '    
 'direct file output for external users.\n'    
 '* New function makelockfile() presenting the logic to create a lock file '    
 'separately from the lockfile context manager.\n'    
 '* Assorted bugfixes and improvements.\n'    
 '\n'    
 '*Release 20150116*:\n'    
 'Initial PyPI release.'),
  install_requires = ['cs.buffer', 'cs.deco', 'cs.env', 'cs.filestate', 'cs.lex>=20200914', 'cs.logutils', 'cs.mappings>=20210717', 'cs.obj', 'cs.pfx', 'cs.progress', 'cs.py3', 'cs.range', 'cs.result', 'cs.threads', 'cs.timeutils', 'cs.units'],
  classifiers = ['Programming Language :: Python', 'Programming Language :: Python :: 2', 'Programming Language :: Python :: 3', 'Development Status :: 4 - Beta', 'Intended Audience :: Developers', 'Operating System :: OS Independent', 'Topic :: Software Development :: Libraries :: Python Modules', 'License :: OSI Approved :: GNU General Public License v3 or later (GPLv3+)'],
  keywords = ['python2', 'python3'],
  license = 'GNU General Public License v3 or later (GPLv3+)',
  long_description_content_type = 'text/markdown',
  package_dir = {'': 'lib/python'},
  py_modules = ['cs.fileutils'],
)
