import React from 'react';
import * as NGL from '@osscar/ngl';
import * as _ from 'underscore';
import Grid from '@material-ui/core/Grid';
import Radio from '@material-ui/core/Radio';
import RadioGroup from '@material-ui/core/RadioGroup';
import FormControlLabel from '@material-ui/core/FormControlLabel';
import Slider from '@material-ui/core/Slider';
// import Button from '@material-ui/core/Button';
// import Switch from '@material-ui/core/Switch';

export interface IProps {
  data: string;
}

export interface IState {
  filter?: string;
  value: string;
}

const marks1 = [
  {
    value: 0,
    label: '0%',
  },
  {
    value: 20,
    label: '20%',
  },
  {
    value: 40,
    label: '40%',
  },
  {
    value: 60,
    label: '60%',
  },
  {
    value: 80,
    label: '80%',
  },
  {
    value: 100,
    label: '100%',
  },
];

const marks2 = [
  {
    value: -0.02,
    label: '-0.02',
  },
  {
    value: -0.01,
    label: '-0.01',
  },
  {
    value: 0,
    label: '0',
  },
  {
    value: 0.01,
    label: '0.01',
  },
  {
    value: 0.02,
    label: '0.02',
  },
];

export class Visualizer extends React.Component<IProps, IState> {
  private _stage: any;
  private uuid: string;
  private dark: boolean;

  constructor(props: IProps, context: any) {
    super(props, context);

    this.dark = true;
    this.uuid = _.uniqueId('ngl_');
    this.state = { value: 'black' };

    window.requestAnimationFrame(() => {
      NGL.DatasourceRegistry.add(
        'data',
        new NGL.StaticDatasource(
          '//cdn.rawgit.com/arose/ngl/v2.0.0-dev.32/data/'
        )
      );

      // Create NGL Stage object
      this._stage = new NGL.Stage(this.uuid, { quality: 'high' });

      const data = this.props.data;
      const stringBlob = new Blob([data], { type: 'text/plain' });

      this._stage.loadFile(stringBlob, { ext: 'cube' }).then((o: any) => {
        o.addRepresentation('surface', {
          name: 'positive_surface',
          visible: true,
          isolevelType: 'value',
          isolevel: 0.01,
          color: 'red',
          opacity: 0.7,
          opaqueBack: false,
        });
        o.addRepresentation('surface', {
          name: 'negative_surface',
          visible: true,
          isolevelType: 'value',
          isolevel: -0.01,
          color: 'blue',
          opacity: 0.7,
          opaqueBack: false,
        });
        o.autoView();
      }),
        function (e: any) {
          console.log('information:' + e);
        };
    });
  }

  toggle_backgroundColor = (): void => {
    if (this.dark) {
      this._stage.setParameters({ backgroundColor: 'white' });
    } else {
      this._stage.setParameters({ backgroundColor: 'black' });
    }
    this.dark = !this.dark;
  };

  handleRadioChange = (event: React.ChangeEvent<HTMLInputElement>): void => {
    this._stage.setParameters({ backgroundColor: event.target.value });
    this.setState({ value: event.target.value });
  };

  valuetext(value: number): string {
    return String(value) + '%';
  }

  handleOpacityChange = (
    event: React.ChangeEvent<unknown>,
    value: number | number[]
  ): void => {
    const transparency = (value as number) / 100.0;
    this._stage
      .getRepresentationsByName('positive_surface')
      .setParameters({ opacity: transparency });
    this._stage
      .getRepresentationsByName('negative_surface')
      .setParameters({ opacity: transparency });
  };

  handleIsovalueChange = (
    event: React.ChangeEvent<unknown>,
    value: number | number[]
  ): void => {
    const val = value as number[];

    this._stage
      .getRepresentationsByName('positive_surface')
      .setParameters({ isolevel: val[1] });
    this._stage
      .getRepresentationsByName('negative_surface')
      .setParameters({ isolevel: val[0] });
  };

  render(): JSX.Element {
    return (
      <div className="container">
        <Grid
          container
          spacing={3}
          justify="center"
          style={{ marginTop: '20px' }}
        >
          <Grid item sm={8}>
            <div
              id={this.uuid}
              style={{
                width: '100%',
                height: '400px',
                backgroundColor: 'black',
              }}
            ></div>
          </Grid>
          <Grid item sm={1}>
            <Slider
              orientation="vertical"
              getAriaValueText={this.valuetext}
              valueLabelDisplay="auto"
              defaultValue={70}
              aria-labelledby="vertical-slider"
              min={0}
              max={100}
              marks={marks1}
              onChange={this.handleOpacityChange}
              color={'primary'}
            />
          </Grid>
          <Grid item sm={1}>
            <Slider
              orientation="vertical"
              defaultValue={[0.01, -0.01]}
              aria-labelledby="vertical-slider"
              getAriaValueText={this.valuetext}
              valueLabelDisplay="on"
              marks={marks2}
              min={-0.02}
              max={0.02}
              step={0.001}
              onChange={this.handleIsovalueChange}
              color={'secondary'}
            />
          </Grid>
        </Grid>
        <Grid container direction="row" justify="center" alignItems="center">
          <RadioGroup
            aria-label="backgroundcolor"
            name="backgroundcolor1"
            value={this.state.value}
            onChange={this.handleRadioChange}
            row
          >
            <FormControlLabel value="black" control={<Radio />} label="Black" />
            <FormControlLabel value="white" control={<Radio />} label="White" />
            <FormControlLabel value="gray" control={<Radio />} label="Gray" />
            <FormControlLabel
              value="yellow"
              control={<Radio />}
              label="Yellow"
            />
          </RadioGroup>
        </Grid>
      </div>
    );
  }
}
