# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['eodhdc',
 'eodhdc.clients',
 'eodhdc.groups',
 'eodhdc.groups.coro',
 'eodhdc.groups.sync']

package_data = \
{'': ['*']}

install_requires = \
['pandas>=1.3,<2.0', 'requests>=2.28.2,<3.0.0', 'websockets>=10.4,<11.0']

extras_require = \
{'aiohttp': ['aiohttp>=3.8.3,<4.0.0'], 'httpx': ['httpx>=0.23.3,<0.24.0']}

setup_kwargs = {
    'name': 'eodhdc',
    'version': '0.9.0',
    'description': 'EOD Historical Data API Python Client',
    'long_description': '# EODHDC\n\n![license](https://badgen.net/pypi/license/eodhdc)\n![version](https://badgen.net/pypi/v/eodhdc)\n![versions](https://badgen.net/pypi/python/eodhdc)\n![coverage](https://raw.githubusercontent.com/wargx/eodhdc/main/reports/coverage.svg)\n![pylint](https://raw.githubusercontent.com/wargx/eodhdc/main/reports/pylint.svg)  \n![pytest:37](https://raw.githubusercontent.com/wargx/eodhdc/main/reports/pytest-py37.svg)\n![pytest:38](https://raw.githubusercontent.com/wargx/eodhdc/main/reports/pytest-py38.svg)\n![pytest:39](https://raw.githubusercontent.com/wargx/eodhdc/main/reports/pytest-py39.svg)\n![pytest:310](https://raw.githubusercontent.com/wargx/eodhdc/main/reports/pytest-py310.svg)\n![pytest:311](https://raw.githubusercontent.com/wargx/eodhdc/main/reports/pytest-py311.svg)\n\nPython client for the EOD Historical Data service REST / WebSockets API and provides various financial data \nincluding stock market, splits and dividends, fundamental and economic, exchanges and alternative data feeds.\nProvides synchronous and asynchronous interfaces for HTTP API, asynchronous interface for WebSockets.   \n\n## Installation\n\nFor normal usage you will need API key which you can get from [here](https://eodhistoricaldata.com/).  \nSupported Python version >= 3.7\n\nPackage can be installed using pip or poetry:\n```\npip install eodhdc\n```\n```\npoetry add eodhdc\n```\nTo support additional HTTP clients install with extras:\n```\npip install eodhdc[httpx,aiohttp]\n```\n\n## Quickstart\n\nAsynchronous usage\n```python\nimport asyncio\nfrom eodhdc import EODHDClient\n\neodhdc = EODHDClient("httpxa", key="demo")\nasync with eodhdc.session:\n    results = await asyncio.gather(\n        eodhdc.market.historical("MCD.US", start="2023-01-01", finish="2023-01-10", fmt="csv"),\n        eodhdc.market.historical("MCD.US", start="2023-01-01", finish="2023-01-10", fmt="json")\n    )\nfor result in results:\n    print(result)\n```\n\nSynchronous usage\n```python\nfrom eodhdc import EODHDClient\n\neodhdc = EODHDClient("requests", key="demo")\nresult = eodhdc.market.historical(\n    "AAPL.US", start="2023-01-01", finish="2023-01-10",\n    fmt="json", output="pandas:./response.csv", writer={"header": False}\n)\nprint(result, "\\n")\n```\n\nWebSockets usage\n```python\nfrom eodhdc import EODHDWebSockets\n\neodhdws = EODHDWebSockets(buffer=100)\nasync with eodhdws.connect("us") as websocket:\n    await eodhdws.subscribe(websocket, ["TSLA", "EURUSD"])\n    async for message in eodhdws.receive(websocket):\n        print(message)\n        if len(eodhdws.buffer) > 5:\n            await eodhdws.unsubscribe(websocket, ["EURUSD"])\n        if len(eodhdws.buffer) > 10:\n            eodhdws.deactivate()\n\nprint(eodhdws.buffer)\n```\n\nAlso check `playground.py` for quickstart examples. \n\n## Description\n\n### Main client modules\n\n- EODHDClient: HTTP API client, parameters are:\n  - client: http client module to use.\n  - key: api token.\n  - args: http client `get` args to use across requests.\n\n- EODHDWebSockets: WebSockets API client, parameters are:\n  - key: api token.\n  - buffer: enable and set buffer size.\n  - args: websocket client args.\n\nEODHDClient will automatically determine sync or async http client and provide corresponding interface \nwith same signature, so for example usage can easily be changed:\n\n```python\neodhdc = EODHDClient("httpxs")\nresult = eodhdc.market.historical(...)\n```\n\n```python\neodhdc = EODHDClient("httpsxa")\nresult = await eodhdc.market.historical(...)\n```\n\nAsynchronous version of EODHDClient can be used without context manager, do not forget to call `destroy` \nmethod to close session in that case. \n\nEODHDWebSockets client provides following methods:\n- connect: connect to web-socket endpoint, returns context manager.\n- authorize: check authorization status, do not use directly as it will consume messages. \n- subscribe: subscribe to the tickers.\n- unsubscribe: unsubscribe from the tickers.\n- receive: receive messages, async generator.\n- activate: activate message loop.\n- deactivate: deactivate message loop.\n\n### HTTP client modules\n\n- requests: default, well known, synchronous module.\n- httpxs: httpx library, synchronous mode, \'httpx\' extra.\n- httpxa: httpx library, asynchronous mode, \'httpx\' extra.\n- aiohttp: aiohttp library, asynchronous mode, \'aiohttp\' extra.\n\n### HTTP API groups\n\nMain HTTP API module contains groups that corresponds to EODHD API groups, and can be accessed like:   \n```eodhdc.market.<method>``` or  ```eodhdc.exchange.<method>```  \nSee below mapping for client groups and methods.    \nVisit official API [documentation](https://eodhistoricaldata.com/financial-apis/) for detailed description. \n\n### HTTP API group methods\n\nIn addition to original API parameters each method have:\n- args: override or add http client `get` args.\n- output: output format and optionally file location, format is `<type>[:path]`\n  - types: response type\n    - "response": raw binary response body\n    - "content": decoded as response content type\n    - "pandas": pandas dataframe\n  - path: additionally save response to file\n    - for "response" and "content" will save as is\n    - for "pandas" will save in format specified by extension: \n      parquet, pickle, csv, hdf, xlsx, json, html, feather, tex, dta, md  \n- writer: pandas writer parameters, see [original](https://pandas.pydata.org/docs/reference/api/pandas.DataFrame.to_csv.html) `to_<format>` methods for more details.  \n  note that some formats may require 3rd-party libraries.  \n  additionally writer can be provided with:\n  - change:columns - dict to rename DataFrame columns. \n  - change:reorder - bool to use columns dict for DataFrame columns order.\n  - change:reindex - str or list to set DataFrame columns as index.\n\n### API support status\n\nAPI support status and mapping for client groups and methods.   \n\n- [x] HTTP\n  - [x] Stock Market Prices, Splits and Dividends Data API\n    - [x] End-Of-Day Historical Stock Market Data API\n      - market.historical \n    - [x] Live (Delayed) Stock Prices API\n      - market.delayed\n    - [x] Historical Splits and Dividends API\n      - market.dividends\n      - market.splits\n    - [x] Technical Indicator API\n      - market.indicators\n    - [x] Intraday Historical Data API\n      - market.intraday\n    - [x] Options Data API\n      - market.options\n  - [x] Fundamental and Economic Financial Data API\n    - [x] Fundamental Data for Cryptocurrencies\n      - fundamental.crypto\n    - [x] Historical Market Capitalization API\n      - fundamental.capitalization\n    - [x] Insider Transactions API\n      - fundamental.insider\n    - [x] Fundamental Data: Stocks, ETFs, Mutual Funds, Indices\n      - fundamental.fundamentals\n      - fundamental.bulk\n    - [x] Calendar. Upcoming Earnings, Trends, IPOs and Splits\n      - fundamental.calendar\n    - [x] Bonds Fundamentals and Historical API\n      - fundamental.bonds\n  - [x] Exchanges (Stock Market) Financial APIs\n    - [x] Bulk API for EOD, Splits and Dividends\n      - exchange.bulk\n    - [x] Exchanges API. Get List of Tickers\n      - exchange.exchanges\n      - exchange.tickers\n    - [x] Exchanges API. Trading Hours, Stock Market Holidays, Symbols Change History\n      - exchange.details\n      - exchange.history\n    - [x] Stock Market Screener API\n      - exchange.screener\n    - [x] Search API for Stocks, ETFs, Mutual Funds and Indices\n      - exchange.search\n  - [x] Alternative Data Financial API\n    - [x] Sentiment Data Financial API for News and Tweets\n      - alternative.sentiment\n    - [x] Economic Events Data API\n      - alternative.events\n    - [x] Stock Market and Financial News API\n      - alternative.news\n    - [x] Macro Indicators API\n      - alternative.macroindicators\n    - [x] Macroeconomic Data API\n      - alternative.macroeconomic\n- [x] WebSockets\n  - [x] Stock Market Prices, Splits and Dividends Data API\n    - [x] Real-Time Data API\n\n### Exceptions\n\nExceptions hierarchy:\n\n- **ClientException**: Base HTTP client exception.\n  - **ClientConnectionTimeout**: Client connection timeout exception. \n  - **ClientConnectionError**: Client connection error exception. \n  - **ClientHTTPError**: Client HTTP error exception.\n- **ModuleException**: Base module exception. \n  - **FileIOError**: File IO exception. \n  - **UnsupportedContentType**: Unsupported response content exception. \n  - **UnsupportedExtension**: Unsupported pandas extension exception. \n  - **JSONDecodeError**: JSON decoding exception. \n  - **BytesDecodeError**: Bytes decoding exception. \n  - **PandasRuntimeError**: Pandas runtime exception. \n  - **UnknownClient**: Unknown client exception. \n  - **ImproperClient**: Improper client exception.\n- **WebsocketException**: Base websocket exception. \n  - **WebsocketUnknownEndpoint**: Websocket unknown endpoint exception. \n  - **WebsocketAuthError**: Websocket authentication exception. \n  - **WebsocketResponseError**: Websocket response error exception.\n\n## Custom HTTP clients\n\nAdditionally, you can provide your own HTTP client by passing its module instead of name string.  \nModule should implement `get` method and `create` and `destroy` can be provided for asynchronous session management.  \nCheck modules under `eodhd.clients` for details about required parameters, return data type and exceptions handling.\n\n## Disclaimer\n\nThe information in this document is for informational and educational purposes only. Nothing in this document \ncan be construed as financial, legal, or tax advice. The content of this document is solely the opinion of the \nauthor, who is not a licensed financial advisor or registered investment advisor. The author is not affiliated \nas a promoter of EOD Historical Data services. \n\nThis document is not an offer to buy or sell financial instruments. Never invest more than you can afford to \nlose. You should consult a registered professional advisor before making any investment.\n',
    'author': 'Warg',
    'author_email': 'warg.silencer@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/wargx/eodhdc',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7.16,<4.0.0',
}


setup(**setup_kwargs)
