"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const cdk = require("@aws-cdk/core");
const validation_1 = require("./private/validation");
const servicecatalogappregistry_generated_1 = require("./servicecatalogappregistry.generated");
class ApplicationBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedAttributeGroups = new Set();
        this.associatedResources = new Set();
    }
    /**
     * Associate an attribute group with application
     * If the attribute group is already associated, it will ignore duplicate request.
     */
    associateAttributeGroup(attributeGroup) {
        if (!this.associatedAttributeGroups.has(attributeGroup.node.addr)) {
            const hashId = this.generateUniqueHash(attributeGroup.node.addr);
            new servicecatalogappregistry_generated_1.CfnAttributeGroupAssociation(this, `AttributeGroupAssociation${hashId}`, {
                application: this.applicationId,
                attributeGroup: attributeGroup.attributeGroupId,
            });
            this.associatedAttributeGroups.add(attributeGroup.node.addr);
        }
    }
    /**
     * Associate a stack with the application
     * If the resource is already associated, it will ignore duplicate request.
     * A stack can only be associated with one application.
     */
    associateStack(stack) {
        if (!this.associatedResources.has(stack.node.addr)) {
            const hashId = this.generateUniqueHash(stack.node.addr);
            new servicecatalogappregistry_generated_1.CfnResourceAssociation(this, `ResourceAssociation${hashId}`, {
                application: this.applicationId,
                resource: stack.stackId,
                resourceType: 'CFN_STACK',
            });
            this.associatedResources.add(stack.node.addr);
        }
    }
}
/**
 * A Service Catalog AppRegistry Application.
 */
class Application extends ApplicationBase {
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_servicecatalogappregistry_ApplicationProps(props);
        this.validateApplicationProps(props);
        const application = new servicecatalogappregistry_generated_1.CfnApplication(this, 'Resource', {
            name: props.applicationName,
            description: props.description,
        });
        this.applicationArn = application.attrArn;
        this.applicationId = application.attrId;
        this.nodeAddress = cdk.Names.nodeUniqueId(application.node);
    }
    /**
     * Imports an Application construct that represents an external application.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param applicationArn the Amazon Resource Name of the existing AppRegistry Application
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const arn = cdk.Stack.of(scope).splitArn(applicationArn, cdk.ArnFormat.SLASH_RESOURCE_SLASH_RESOURCE_NAME);
        const applicationId = arn.resourceName;
        if (!applicationId) {
            throw new Error('Missing required Application ID from Application ARN: ' + applicationArn);
        }
        class Import extends ApplicationBase {
            constructor() {
                super(...arguments);
                this.applicationArn = applicationArn;
                this.applicationId = applicationId;
            }
            generateUniqueHash(resourceAddress) {
                return hashValues(this.applicationArn, resourceAddress);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: applicationArn,
        });
    }
    generateUniqueHash(resourceAddress) {
        return hashValues(this.nodeAddress, resourceAddress);
    }
    validateApplicationProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'application name', 1, 256, props.applicationName);
        validation_1.InputValidator.validateRegex(this.node.path, 'application name', /^[a-zA-Z0-9-_]+$/, props.applicationName);
        validation_1.InputValidator.validateLength(this.node.path, 'application description', 0, 1024, props.description);
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-servicecatalogappregistry.Application", version: "1.148.0" };
/**
 * Generates a unique hash identfifer using SHA256 encryption algorithm
 */
function hashValues(...values) {
    const sha256 = crypto.createHash('sha256');
    values.forEach(val => sha256.update(val));
    return sha256.digest('hex').slice(0, 12);
}
//# sourceMappingURL=data:application/json;base64,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