"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const events = require("@aws-cdk/aws-events");
const kinesis = require("@aws-cdk/aws-kinesis");
require("@aws-cdk/assert/jest");
const lib_1 = require("../lib");
// --------------------------------------------------------------
// Test snapshot match with default parameters
// --------------------------------------------------------------
function deployNewStack(stack) {
    const props = {
        eventRuleProps: {
            description: 'event rule props',
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    };
    return new lib_1.EventsRuleToKinesisStreams(stack, 'test-events-rule-kinesis-streams-default-parameters', props);
}
// --------------------------------------------------------------
// Test properties
// --------------------------------------------------------------
test('Test properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStack(stack);
    // Assertions
    expect(construct.eventsRule !== null);
    expect(construct.kinesisStream !== null);
    expect(construct.eventsRole !== null);
});
// --------------------------------------------------------------
// Test default AWS managed encryption key property
// --------------------------------------------------------------
test('Test default AWS managed encryption key property', () => {
    const stack = new cdk.Stack();
    deployNewStack(stack);
    // Assertions
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: "KMS",
            KeyId: "alias/aws/kinesis"
        }
    });
});
// --------------------------------------------------------------
// Test existing resources
// --------------------------------------------------------------
test('Test existing resources', () => {
    const stack = new cdk.Stack();
    // create resource
    const existingStream = new kinesis.Stream(stack, 'test-existing-stream', {
        streamName: 'existing-stream',
        shardCount: 5,
        retentionPeriod: cdk.Duration.hours(48),
        encryption: kinesis.StreamEncryption.UNENCRYPTED
    });
    new lib_1.EventsRuleToKinesisStreams(stack, 'test-events-rule-kinesis-stream-existing-resource', {
        existingStreamObj: existingStream,
        eventRuleProps: {
            description: 'event rule props',
            schedule: events.Schedule.rate(cdk.Duration.minutes(5))
        }
    });
    expect(stack).toHaveResource('AWS::Kinesis::Stream', {
        Name: 'existing-stream',
        ShardCount: 5,
        RetentionPeriodHours: 48,
    });
});
test('check eventbus property, snapshot & eventbus exists', () => {
    const stack = new cdk.Stack();
    const props = {
        eventRuleProps: {
            description: 'event rule props',
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: {}
    };
    const construct = new lib_1.EventsRuleToKinesisStreams(stack, 'test-events-rule-kinesis-streams-default-parameters', props);
    expect(construct.eventsRule !== null);
    expect(construct.kinesisStream !== null);
    expect(construct.eventsRole !== null);
    expect(construct.eventBus !== null);
    // Check whether eventbus exists
    expect(stack).toHaveResource('AWS::Events::EventBus');
});
test('check exception while passing existingEventBus & eventBusProps', () => {
    const stack = new cdk.Stack();
    const props = {
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: {},
        existingEventBusInterface: new events.EventBus(stack, `test-existing-eventbus`, {})
    };
    const app = () => {
        new lib_1.EventsRuleToKinesisStreams(stack, 'test-eventsrule-kinesisstreams', props);
    };
    expect(app).toThrowError();
});
test('check custom event bus resource with props when deploy:true', () => {
    const stack = new cdk.Stack();
    const props = {
        eventBusProps: {
            eventBusName: `testeventbus`
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        }
    };
    new lib_1.EventsRuleToKinesisStreams(stack, 'test-new-eventsrule-with-props-kinsesisstreams', props);
    expect(stack).toHaveResource('AWS::Events::EventBus', {
        Name: `testeventbus`
    });
});
//# sourceMappingURL=data:application/json;base64,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