"""
Enterprise Middleware Layer for Kailash SDK
===========================================

Consolidated enterprise-grade middleware components that provide a comprehensive
foundation for building production-ready applications with the Kailash SDK.

This middleware layer consolidates and enhances existing api/ and mcp/ implementations
into a unified, enterprise-grade middleware stack built entirely with Kailash SDK
components, following strict SDK patterns and best practices.

Core Design Principles
---------------------
1. **100% SDK Components**: Every middleware component uses authentic Kailash SDK nodes
2. **No Custom Orchestration**: Delegates all execution to SDK runtime engines
3. **Event-Driven Architecture**: Real-time communication via comprehensive event streams
4. **Enterprise Security**: Multi-tenant isolation with RBAC/ABAC access control
5. **Performance Optimized**: Sub-200ms latency with intelligent caching
6. **Comprehensive Testing**: 17/17 integration tests passing for production reliability

Architecture Overview
--------------------
The middleware consists of these interconnected layers:

**Agent-UI Communication Layer**:
- Session management with automatic cleanup
- Dynamic workflow creation using WorkflowBuilder.from_dict()
- Real-time execution monitoring and progress tracking
- Event-driven state synchronization

**Real-time Communication Layer**:
- WebSocket connections for bi-directional communication
- Server-Sent Events (SSE) for unidirectional streaming
- Webhook management for external integrations
- Event filtering and subscription management

**API Gateway Layer**:
- RESTful API endpoints with OpenAPI documentation
- JWT-based authentication using JWTAuthManager
- Request/response middleware with comprehensive logging
- Dynamic schema generation for node discovery

**Database Integration Layer**:
- Repository pattern using AsyncSQLDatabaseNode
- Audit logging with AuditLogNode for compliance
- Security event tracking with SecurityEventNode
- Workflow and execution persistence

**AI Chat Integration Layer**:
- Semantic search using AsyncPostgreSQLVectorNode
- LLM integration for natural language workflow generation
- Context-aware conversation management
- Intent recognition and response generation

Key Components
-------------

**Core Middleware**:
- `AgentUIMiddleware`: Central orchestration hub for frontend communication
- `RealtimeMiddleware`: Real-time communication management
- `APIGateway`: RESTful API layer with authentication
- `EventStream`: Comprehensive event management system

**Authentication & Security**:
- `KailashJWTAuthManager`: JWT token management using SDK security nodes
- `MiddlewareAccessControlManager`: RBAC/ABAC access control
- `MiddlewareAuthenticationMiddleware`: Request authentication middleware

**AI & Chat Integration**:
- `AIChatMiddleware`: AI-powered conversation management
- `WorkflowGenerator`: Natural language to workflow conversion
- `ChatMessage`: Structured chat message handling

**MCP Integration**:
- `MiddlewareMCPServer`: Enhanced MCP server with caching and metrics
- `MiddlewareMCPClient`: Robust MCP client with connection management
- `MCPToolNode`: MCP tool integration as SDK nodes
- `MCPResourceNode`: MCP resource access as SDK nodes

**Database & Persistence**:
- `MiddlewareWorkflowRepository`: Workflow persistence using SDK nodes
- `MiddlewareExecutionRepository`: Execution tracking and history
- `MiddlewareDatabaseManager`: Connection and transaction management

**Schema & Discovery**:
- `NodeSchemaGenerator`: Dynamic schema generation for all SDK nodes
- `DynamicSchemaRegistry`: Caching and optimization for schema queries

Enterprise Features
------------------
- **Multi-tenant Isolation**: Complete session and data isolation
- **Comprehensive Monitoring**: Built-in metrics, logging, and health checks
- **Automatic Scaling**: Connection pooling and resource optimization
- **Security Compliance**: Audit trails, access control, and security events
- **Developer Experience**: Rich error messages, debugging tools, and documentation

Usage Examples
--------------

**Basic Middleware Stack**:
    >>> from kailash.middleware import AgentUIMiddleware, APIGateway
    >>>
    >>> # Create agent-UI middleware with session management
    >>> agent_ui = AgentUIMiddleware(
    ...     max_sessions=1000,
    ...     session_timeout_minutes=60,
    ...     enable_persistence=True
    ... )
    >>>
    >>> # Create API gateway with authentication
    >>> gateway = create_gateway(
    ...     title="My Kailash API",
    ...     cors_origins=["https://myapp.com"],
    ...     enable_docs=True
    ... )
    >>> gateway.agent_ui = agent_ui

**Real-time Communication**:
    >>> from kailash.middleware import RealtimeMiddleware, EventStream
    >>>
    >>> # Create real-time middleware with WebSocket support
    >>> realtime = RealtimeMiddleware(agent_ui)
    >>>
    >>> # Subscribe to workflow events
    >>> async def handle_workflow_events(event):
    ...     print(f"Workflow {event.workflow_id}: {event.type}")
    >>>
    >>> await realtime.event_stream.subscribe(
    ...     "workflow_monitor",
    ...     handle_workflow_events
    ... )

**AI Chat Integration**:
    >>> from kailash.middleware import AIChatMiddleware
    >>>
    >>> # Create AI chat with semantic search
    >>> ai_chat = AIChatMiddleware(
    ...     agent_ui,
    ...     enable_vector_search=True,
    ...     vector_database_url="postgresql://..."
    ... )
    >>>
    >>> # Start chat session and get workflow suggestions
    >>> session_id = await ai_chat.start_chat_session("user123")
    >>> response = await ai_chat.send_message(
    ...     session_id,
    ...     "Create a workflow to process CSV data and generate charts"
    ... )

**Dynamic Workflow Creation**:
    >>> # Create workflows dynamically from configuration
    >>> workflow_config = {
    ...     "name": "data_processing_pipeline",
    ...     "nodes": [
    ...         {
    ...             "id": "csv_reader",
    ...             "type": "CSVReaderNode",
    ...             "config": {"file_path": "/data/input.csv"}
    ...         },
    ...         {
    ...             "id": "data_processor",
    ...             "type": "PythonCodeNode",
    ...             "config": {
    ...                 "name": "data_processor",
    ...                 "code": "result = {'processed': len(data)}"
    ...             }
    ...         }
    ...     ],
    ...     "connections": [
    ...         {
    ...             "from_node": "csv_reader",
    ...             "from_output": "data",
    ...             "to_node": "data_processor",
    ...             "to_input": "data"
    ...         }
    ...     ]
    ... }
    >>>
    >>> # Create and execute workflow
    >>> session_id = await agent_ui.create_session("user123")
    >>> workflow_id = await agent_ui.create_dynamic_workflow(
    ...     session_id, workflow_config
    ... )
    >>> # Use execute() for consistency with runtime API (preferred)
    >>> execution_id = await agent_ui.execute(
    ...     session_id, workflow_id, inputs={}
    ... )
    >>> # Note: execute_workflow() is deprecated and will be removed in v1.0.0

Production Deployment
--------------------
The middleware is designed for production deployment with:

- **Health Checks**: Built-in endpoints for monitoring service health
- **Metrics Export**: Prometheus-compatible metrics for observability
- **Graceful Shutdown**: Clean resource cleanup and connection management
- **Configuration Management**: Environment-based configuration
- **Logging Integration**: Structured logging with correlation IDs
- **Error Handling**: Comprehensive error recovery and reporting

Version History
--------------
- **v1.0.0**: Initial enterprise middleware release
  - Core agent-UI communication
  - Real-time event streaming
  - Dynamic workflow creation
  - JWT authentication integration
  - Comprehensive test coverage (17/17 tests passing)

Author: Kailash SDK Team
License: See LICENSE file
Documentation: https://docs.kailash.ai/middleware/
"""

from .auth.access_control import (
    MiddlewareAccessControlManager,
    MiddlewareAuthenticationMiddleware,
)
from .auth.auth_manager import AuthLevel, MiddlewareAuthManager

# Authentication & Access Control
from .auth.jwt_auth import JWTAuthManager
from .communication.ai_chat import AIChatMiddleware, ChatMessage, WorkflowGenerator
from .communication.api_gateway import APIGateway, create_gateway

# Communication Layer
from .communication.events import (
    EventFilter,
    EventPriority,
    EventStream,
    EventType,
    NodeEvent,
    UIEvent,
    WorkflowEvent,
)
from .communication.realtime import RealtimeMiddleware

# Core Middleware Components
from .core.agent_ui import AgentUIMiddleware
from .core.schema import DynamicSchemaRegistry, NodeSchemaGenerator
from .core.workflows import MiddlewareWorkflows, WorkflowBasedMiddleware

# Database Layer
from .database import (
    CustomNodeModel,
    MiddlewareDatabaseManager,
    MiddlewareWorkflowRepository,
    WorkflowExecutionModel,
    WorkflowModel,
)
from .mcp.client_integration import (
    MCPClientConfig,
    MCPServerConnection,
    MiddlewareMCPClient,
)

# MCP Integration
from .mcp.enhanced_server import (
    MCPResourceNode,
    MCPServerConfig,
    MCPToolNode,
    MiddlewareMCPServer,
)

__all__ = [
    # Core Components
    "AgentUIMiddleware",
    "AIChatMiddleware",
    "ChatMessage",
    "WorkflowGenerator",
    "APIGateway",
    "create_gateway",
    "EventStream",
    "EventType",
    "EventPriority",
    "EventFilter",
    "WorkflowEvent",
    "NodeEvent",
    "UIEvent",
    "RealtimeMiddleware",
    "NodeSchemaGenerator",
    "DynamicSchemaRegistry",
    # Authentication & Access Control
    "JWTAuthManager",
    "MiddlewareAuthManager",
    "AuthLevel",
    "MiddlewareAccessControlManager",
    "MiddlewareAuthenticationMiddleware",
    # MCP Integration
    "MiddlewareMCPServer",
    "MCPServerConfig",
    "MCPToolNode",
    "MCPResourceNode",
    "MiddlewareMCPClient",
    "MCPClientConfig",
    "MCPServerConnection",
    # Database Layer
    "WorkflowModel",
    "WorkflowExecutionModel",
    "CustomNodeModel",
    "MiddlewareWorkflowRepository",
    "MiddlewareDatabaseManager",
    # Workflow-based Optimizations
    "MiddlewareWorkflows",
    "WorkflowBasedMiddleware",
]

__version__ = "1.0.0"
