from typing import Dict, List, Optional, Union

import terrascript.core as core


@core.schema
class CognitoIdentityProviders(core.Schema):

    client_id: Optional[Union[str, core.StringOut]] = core.attr(str, default=None)

    provider_name: Optional[Union[str, core.StringOut]] = core.attr(str, default=None)

    server_side_token_check: Optional[Union[bool, core.BoolOut]] = core.attr(bool, default=None)

    def __init__(
        self,
        *,
        client_id: Optional[Union[str, core.StringOut]] = None,
        provider_name: Optional[Union[str, core.StringOut]] = None,
        server_side_token_check: Optional[Union[bool, core.BoolOut]] = None,
    ):
        super().__init__(
            args=CognitoIdentityProviders.Args(
                client_id=client_id,
                provider_name=provider_name,
                server_side_token_check=server_side_token_check,
            ),
        )

    @core.schema_args
    class Args(core.SchemaArgs):
        client_id: Optional[Union[str, core.StringOut]] = core.arg(default=None)

        provider_name: Optional[Union[str, core.StringOut]] = core.arg(default=None)

        server_side_token_check: Optional[Union[bool, core.BoolOut]] = core.arg(default=None)


@core.resource(type="aws_cognito_identity_pool", namespace="aws_cognito_identity")
class Pool(core.Resource):

    allow_classic_flow: Optional[Union[bool, core.BoolOut]] = core.attr(bool, default=None)

    allow_unauthenticated_identities: Optional[Union[bool, core.BoolOut]] = core.attr(
        bool, default=None
    )

    """
    The ARN of the identity pool.
    """
    arn: Union[str, core.StringOut] = core.attr(str, computed=True)

    cognito_identity_providers: Optional[
        Union[List[CognitoIdentityProviders], core.ArrayOut[CognitoIdentityProviders]]
    ] = core.attr(CognitoIdentityProviders, default=None, kind=core.Kind.array)

    developer_provider_name: Optional[Union[str, core.StringOut]] = core.attr(str, default=None)

    """
    An identity pool ID, e.g. `us-west-2_abc123`.
    """
    id: Union[str, core.StringOut] = core.attr(str, computed=True)

    identity_pool_name: Union[str, core.StringOut] = core.attr(str)

    openid_connect_provider_arns: Optional[
        Union[List[str], core.ArrayOut[core.StringOut]]
    ] = core.attr(str, default=None, kind=core.Kind.array)

    saml_provider_arns: Optional[Union[List[str], core.ArrayOut[core.StringOut]]] = core.attr(
        str, default=None, kind=core.Kind.array
    )

    supported_login_providers: Optional[
        Union[Dict[str, str], core.MapOut[core.StringOut]]
    ] = core.attr(str, default=None, kind=core.Kind.map)

    """
    (Optional) A map of tags to assign to the Identity Pool. If configured with a provider [`default_tag
    s` configuration block](https://registry.terraform.io/providers/hashicorp/aws/latest/docs#default_ta
    gs-configuration-block) present, tags with matching keys will overwrite those defined at the provide
    r-level.
    """
    tags: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = core.attr(
        str, default=None, kind=core.Kind.map
    )

    """
    A map of tags assigned to the resource, including those inherited from the provider [`default_tags`
    configuration block](https://registry.terraform.io/providers/hashicorp/aws/latest/docs#default_tags-
    configuration-block).
    """
    tags_all: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = core.attr(
        str, default=None, computed=True, kind=core.Kind.map
    )

    def __init__(
        self,
        resource_name: str,
        *,
        identity_pool_name: Union[str, core.StringOut],
        allow_classic_flow: Optional[Union[bool, core.BoolOut]] = None,
        allow_unauthenticated_identities: Optional[Union[bool, core.BoolOut]] = None,
        cognito_identity_providers: Optional[
            Union[List[CognitoIdentityProviders], core.ArrayOut[CognitoIdentityProviders]]
        ] = None,
        developer_provider_name: Optional[Union[str, core.StringOut]] = None,
        openid_connect_provider_arns: Optional[
            Union[List[str], core.ArrayOut[core.StringOut]]
        ] = None,
        saml_provider_arns: Optional[Union[List[str], core.ArrayOut[core.StringOut]]] = None,
        supported_login_providers: Optional[
            Union[Dict[str, str], core.MapOut[core.StringOut]]
        ] = None,
        tags: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = None,
        tags_all: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = None,
        depends_on: Optional[Union[List[str], core.ArrayOut[core.StringOut]]] = None,
        provider: Optional[Union[str, core.StringOut]] = None,
        lifecycle: Optional[core.Lifecycle] = None,
    ):
        super().__init__(
            name=resource_name,
            args=Pool.Args(
                identity_pool_name=identity_pool_name,
                allow_classic_flow=allow_classic_flow,
                allow_unauthenticated_identities=allow_unauthenticated_identities,
                cognito_identity_providers=cognito_identity_providers,
                developer_provider_name=developer_provider_name,
                openid_connect_provider_arns=openid_connect_provider_arns,
                saml_provider_arns=saml_provider_arns,
                supported_login_providers=supported_login_providers,
                tags=tags,
                tags_all=tags_all,
                depends_on=depends_on,
                provider=provider,
                lifecycle=lifecycle,
            ),
        )

    @core.schema_args
    class Args(core.Resource.Args):
        allow_classic_flow: Optional[Union[bool, core.BoolOut]] = core.arg(default=None)

        allow_unauthenticated_identities: Optional[Union[bool, core.BoolOut]] = core.arg(
            default=None
        )

        cognito_identity_providers: Optional[
            Union[List[CognitoIdentityProviders], core.ArrayOut[CognitoIdentityProviders]]
        ] = core.arg(default=None)

        developer_provider_name: Optional[Union[str, core.StringOut]] = core.arg(default=None)

        identity_pool_name: Union[str, core.StringOut] = core.arg()

        openid_connect_provider_arns: Optional[
            Union[List[str], core.ArrayOut[core.StringOut]]
        ] = core.arg(default=None)

        saml_provider_arns: Optional[Union[List[str], core.ArrayOut[core.StringOut]]] = core.arg(
            default=None
        )

        supported_login_providers: Optional[
            Union[Dict[str, str], core.MapOut[core.StringOut]]
        ] = core.arg(default=None)

        tags: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = core.arg(default=None)

        tags_all: Optional[Union[Dict[str, str], core.MapOut[core.StringOut]]] = core.arg(
            default=None
        )
