# AUTOGENERATED! DO NOT EDIT! File to edit: 02_MLPipeline.ipynb (unless otherwise specified).

__all__ = ['MLPipeline', 'DIRECTORY_PATH', 'TRAIN_FILE', 'TEST_FILE', 'SAMPLE_SUB_FILE', 'scikit_model', 'tmlt']

# Cell
from .DataFrameLoader import *
from .PreProcessor import *
from .Logger import *

# Cell
# hide
import pandas as pd
import numpy as np
from sklearn.compose import ColumnTransformer
from sklearn.pipeline import Pipeline
from sklearn.impute import SimpleImputer
from sklearn.preprocessing import OneHotEncoder, OrdinalEncoder, StandardScaler, MinMaxScaler
from sklearn.ensemble import RandomForestRegressor
from sklearn.metrics import mean_absolute_error, roc_auc_score,accuracy_score
from sklearn.model_selection import cross_val_score, GridSearchCV, StratifiedKFold
from tune_sklearn import TuneGridSearchCV


# for displaying diagram of pipelines
from sklearn import set_config
set_config(display="diagram")

# for finding n_jobs in all sklearn estimators
from sklearn.utils import all_estimators
import inspect

# Just to compare fit times
import time

# for os specific settings
import os

# Cell

class MLPipeline:
    """
    Represent MLPipeline class

    Attributes:\n
    pipeline: An MLPipeline instance \n
    dfl: A DataFrameLoader instance \n
    pp: A PreProcessor Instance \n
    model: The given Model
    """

    def __init__(self):
        self.dfl = None
        self.pp = None
        self.model = None
        self.spl = None
        self.transformer_type = None
        self.has_n_jobs = self.create_has_n_jobs()
        self.IDEAL_CPU_CORES = self.find_ideal_cpu_cores()


    def __str__(self):
        """Returns human readable string reprsentation"""
        attr_str = ("spl, dfl, pp, model")
        return ("Training Pipeline object with attributes:"+attr_str)

    def __repr__(self):
        return self.__str__()

    #helper method to find ideal cpu cores
    def find_ideal_cpu_cores(self):
        if os.cpu_count() > 2:
            ideal_cpu_cores = os.cpu_count()-1
            logger.info(f"{os.cpu_count()} cores found, parallel processing is enabled!")
        else:
            ideal_cpu_cores = None
            logger.info(f"{os.cpu_count()} cores found, parallel processing NOT enabled!")
        return ideal_cpu_cores

    #Helper method to find all sklearn estimators with support for parallelism aka n_jobs
    def create_has_n_jobs(self):
        self.has_n_jobs = ['XGBRegressor', 'XGBClassifier']
        for est in all_estimators():
            s = inspect.signature(est[1])
            if 'n_jobs' in s.parameters:
                self.has_n_jobs.append(est[0])
        return self.has_n_jobs

    # core methods

    # Bundle preprocessing and modeling code in a training pipeline
    def create_final_sklearn_pipeline(self, transformer_type, model):
        self.spl = Pipeline(
            steps=[('preprocessor', transformer_type),
                   ('model', model)])
        return self.spl

    # Core methods for Simple Training
    def prepare_data_for_training(self, train_file_path:str,
                                  test_file_path:str,
                                  idx_col:str, target:str,
                                  random_state:int,
                                  model:object):

        # check if given model supports n_jobs aka cpu core based Parallelism
        estimator_name = model.__class__.__name__
        # print(estimator_name)
        # print(self.has_n_jobs)
        if estimator_name in self.has_n_jobs :
            # In order to OS not to kill the job, leave one processor out
            model.n_jobs = self.IDEAL_CPU_CORES
            self.model = model
        else:
            print(f"{estimator_name} doesn't support parallelism yet! Training will continue on a single thread.")
            self.model = model

        # call DataFrameLoader module
        self.dfl = DataFrameLoader().from_csv(
            train_file_path=train_file_path,
            test_file_path=test_file_path,
            idx_col=idx_col,
            target=target,
            random_state=random_state)

        # call PreProcessor module
        self.pp = PreProcessor().preprocess_all_cols(dataframeloader=self.dfl)

        # call create final sklearn pipelien method
        self.spl = self.create_final_sklearn_pipeline(transformer_type=self.pp.transformer_type,
                                     model = model)
        # return MLPipeline
        return self

    # Force to update the preprocessor in pipeline
    def update_preprocessor(self,
                            num_cols__imputer=SimpleImputer(strategy='median'),
                            num_cols__scaler=StandardScaler(),
                            cat_cols__imputer=SimpleImputer(strategy='constant'),
                            cat_cols__encoder=OneHotEncoder(handle_unknown='ignore')):
        # change preprocessor
        self.pp = PreProcessor().preprocess_all_cols(self.dfl,
                                                     num_cols__imputer=num_cols__imputer,
                                                     num_cols__scaler=num_cols__scaler,
                                                     cat_cols__imputer=cat_cols__imputer,
                                                     cat_cols__encoder=cat_cols__encoder)
        # recall create final sklearn pipelien method
        self.spl = self.create_final_sklearn_pipeline(transformer_type=self.pp.transformer_type,
                                     model = self.model)


    # Force to update the model in pipeline
    def update_model(self, model:object):
        #change model
        self.model = model
        # recall create final sklearn pipelien method
        self.spl = self.create_final_sklearn_pipeline(transformer_type=self.pp.transformer_type,
                                     model = self.model)

    # HELPER METHODS
    # cross validation
    def do_cross_validation(self, cv:int, scoring:str):
        scores = cross_val_score(
            estimator=self.spl,
            X=self.dfl.X,
            y=self.dfl.y,
            scoring=scoring,
            cv=cv)
        # Multiply by -1 since sklearn calculates *negative* scoring for some of the metrics
        if "neg_" in scoring:
            scores = -1 * scores
        return scores

    # Core methods for GridSearch
    def do_grid_search(self, param_grid:object, cv:int,
                       scoring:str, n_jobs=None):

        if n_jobs is None:
            n_jobs = self.IDEAL_CPU_CORES

        # create GridSeachCV instance
        grid_search = GridSearchCV(estimator=self.spl,
                                   param_grid=param_grid,
                                   cv=cv,
                                   scoring=scoring,
                                   n_jobs=n_jobs)
        # now call fit
        grid_search.fit(self.dfl.X, self.dfl.y)
        return grid_search

    # Core methods for Tune SK-Learn GridSearch
    def do_tune_grid_search(self,
                            param_grid:object,
                            scoring:str=None,
                            mode:str='max',
                            cv:int=5,
                            early_stopping=True,
                            time_budget_s:int=None,
                            name:str=None,
                            use_gpu:bool=False,
                            stopper:object=None,
                            max_iters:int=10,
                            n_jobs=None):

        if n_jobs is None:
            n_jobs = self.IDEAL_CPU_CORES

        # create GridSeachCV instance
        tune_search = TuneGridSearchCV(
            estimator=self.spl,
            param_grid=param_grid,
            scoring=scoring,
            mode=mode,
            cv=cv,
            time_budget_s=time_budget_s,
            name=name,
            use_gpu=use_gpu,
            early_stopping=early_stopping,
            stopper=stopper,
            max_iters=max_iters,
            n_jobs=n_jobs,
            pipeline_auto_early_stop=True)

        # now call fit
        tune_search.fit(self.dfl.X, self.dfl.y)
        return tune_search


    # do k-fold training
    def do_k_fold_training(self, n_splits:int, metrics:object, random_state=42):

        #create stratified K Folds instance
        k_fold = StratifiedKFold(n_splits=n_splits,
                             random_state=random_state,
                             shuffle=True)

        # list contains metrics score for each fold
        metrics_score = []
        n=0
        for train_idx, valid_idx in k_fold.split(self.dfl.X, self.dfl.y):
            # create X_train
            self.dfl.X_train = self.dfl.X.iloc[train_idx]
            # create X_valid
            self.dfl.X_valid = self.dfl.X.iloc[valid_idx]
            # create y_train
            self.dfl.y_train = self.dfl.y.iloc[train_idx]
            # create y_valid
            self.dfl.y_valid = self.dfl.y.iloc[valid_idx]

            # fit
            self.spl.fit(self.dfl.X_train, self.dfl.y_train)

            #evaluate metrics based upon input
            if "proba" in metrics.__globals__:
                metrics_score.append(metrics(self.dfl.y_valid,
                                               self.spl.predict_proba(self.dfl.X_valid)[:,1]))
            else:
                metrics_score.append(metrics(self.dfl.y_valid,
                                               self.spl.predict(self.dfl.X_valid)))

            print(f"fold: {n+1} , {str(metrics.__name__)}: {metrics_score[n]}")
            # increment fold counter label
            n += 1
        return k_fold, metrics_score

    def do_k_fold_prediction(self, k_fold:object):
        # create preds dataframe
        preds = np.zeros(self.dfl.X_test.shape[0])
        for _ in range(k_fold.n_splits):
            # predict
            preds += self.spl.predict(self.dfl.X_test) / k_fold.n_splits
        return preds


    # helper method for update_preprocessor
    # to create params value dict from grid_search object
    def get_preprocessor_best_params(self, grid_search_object:object):
        pp_best_params = {}
        for k in grid_search_object.best_params_:
            #print(k)
            if 'preprocessor' in k:
                key = k.split('__')[1] + "__" + k.split('__')[2]
                pp_best_params[key] = grid_search_object.best_params_[k]
        return pp_best_params

    # helper method for update_model
    # to create params value dict from grid_search object
    def get_model_best_params(self, grid_search_object:object):
        model_best_params = {}
        for k in grid_search_object.best_params_:
            #print(k)
            if 'model' in k:
                key = k.split('__')[1]
                model_best_params[key] = grid_search_object.best_params_[k]
        return model_best_params

# Cell
# Dataset file names and Paths
DIRECTORY_PATH = "https://raw.githubusercontent.com/psmathur/tabular_ml_toolkit/master/input/home_data/"
TRAIN_FILE = "train.csv"
TEST_FILE = "test.csv"
SAMPLE_SUB_FILE = "sample_submission.csv"

# Cell
# create scikit-learn ml model
scikit_model = RandomForestRegressor(n_estimators=200, random_state=42)

# createm ml pipeline for scikit-learn model
tmlt = MLPipeline().prepare_data_for_training(
    train_file_path= DIRECTORY_PATH+TRAIN_FILE,
    test_file_path= DIRECTORY_PATH+TEST_FILE,
    idx_col="Id", target="SalePrice",
    model=scikit_model,
    random_state=42)