# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2013-2020 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

from openquake.hazardlib.gsim.base import CoeffsTable

COEFFS_ALPINE_60Bars = CoeffsTable(sa_damping=5, table="""\
IMT        a1                     a2                      a3                      a4                       a5                       a6                        a7                         a8                      a9                       a10                     a11                       a12                     a13                      a14                      a15                        a16                    a17                       a18                       a19                        a20                      a21                       a22                        a23                        tau     mean_phi_ss  sigma_tot phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv        -8.65875145149632      7.85817700835325        -4.37604485905558       1.52739461974064         -0.284706616386616       0.0261357301466931        -0.000933130429778511      3.11537783603186        -1.94375128235722        0.352135879004547       -0.0201605473274912       -4.56711247403471       1.58575418689428         -0.233405179755817       0.0134531673741502         1.99257330049254       -0.585055071136962        0.0635009483831737        -0.00290923000116716       -0.323156152322977       0.0803365659780564        -0.00510932500205881       3.79546159737313e-05       0.5010  0.4600       0.68015   0.00000 0.00000 0.00000 5   7   16   36
pga        -5.31469000163385      6.29572485962852        -3.74491587095984       1.40159376065882         -0.27368447520637        0.0258739991721139        -0.000942851323042687      1.97930317147459        -1.33416251074601        0.246123619582706       -0.0142523214424259       -2.98558884032479       0.6862655312843          -0.0702598559814007      0.0040639140462277         1.1394868684306        -0.10046152533857         -0.0268063021812697       0.00241807455266556        -0.182065337620896       -0.00308557773857536      0.0107164392112091         -0.000914491633952009      0.3532  0.4600       0.57998   0.58000 0.47000 0.35000 5   7   16   36
0.01       -6.32689193743566      7.63743711931681        -4.48405421921681       1.61633476913778         -0.307833559459082       0.0286747563813065        -0.00103538386298397       2.95420548609406        -1.89117028420609        0.347606166168672       -0.020186652596393        -4.36402498704838       1.49730213342102         -0.222616105514736       0.0132213673551242         1.86816065843033       -0.539780444788655        0.0570697530453794        -0.00268877482392405       -0.311736761637935       0.0763925982915731        -0.0046066686709857        2.55527707898788e-05       0.3529  0.4600       0.57975   0.58000 0.47000 0.35000 5   7   16   36
0.02       -6.06355212894335      7.50345153418458        -4.42356958836002       1.60242212814027         -0.306211877146339       0.0285861827885923        -0.00103390105031202       3.21627576766787        -1.98615047654115        0.358828593356167       -0.0206436802246187       -4.59193691276484       1.57446385829661         -0.231144197578945       0.0135344460332887         1.87074143348111       -0.550295403509245        0.0572233419921096        -0.00258419633159396       -0.297241570852122       0.0747651226667024        -0.00411477881443113       -2.74093828241347e-05      0.3674  0.4570       0.58639   0.56796 0.46097 0.37408 5   7   16   36
0.03       -5.3796420657365       7.1387886048008         -4.30055896891798       1.57516433534759         -0.302564375897607       0.0283358047584564        -0.00102743853246151       1.63114432623011        -1.37520794062202        0.280997726325841       -0.0174177073431298       -2.43964308746832       0.835814508519198        -0.144293365562295       0.01016765822114           0.762261120225724      -0.20436733166419         0.0186083943547794        -0.00116361153816653       -0.110848242508166       0.0203235312015262        0.00170590986605956        -0.000228442644135246      0.3749  0.4552       0.58975   0.56092 0.45569 0.38817 5   7   16   36
0.04       -4.94401570045191      6.80605703753267        -4.14171712349972       1.53089901578086         -0.295700952774104       0.0277928114863034        -0.00101029924896663       0.853938204267404       -1.11184090057681        0.251783622991282       -0.0164114611987266       -1.52592346397361       0.562418686553495        -0.115278900915407       0.00919789090389513        0.405196138708034      -0.10502764149638         0.00823030612851996       -0.000856624780581297      -0.0703349606966451      0.00896325880866239       0.00293412908851099        -0.000258170590231865      0.3801  0.4540       0.59211   0.55592 0.45194 0.39816 5   7   16   36
0.05       -4.87788114221256      6.62136876687483        -4.01243995384785       1.49174157306924         -0.289596346142106       0.0273126779936286        -0.000995162099986964      1.31290224676162        -1.28709015905025        0.272438837274782       -0.017152033117731        -2.228676618227         0.807190280492041        -0.141256581643575       0.0100297473269013         0.857166706500377      -0.239882784038062        0.0208426506847125        -0.00121381916995922       -0.163446552398879       0.032661826211388         0.00106483806173643        -0.00021969308509938       0.3855  0.4530       0.59483   0.55204 0.44903 0.40592 5   7   16   36
0.1        -6.34431086270862      7.5780246548641         -4.34003142230199       1.56586031425303         -0.299946901468051       0.0280933180494762        -0.00101913138241269       4.56918845590931        -2.59180671361611        0.44965575710947        -0.0251496252490116       -6.31587019639304       2.37763980616828         -0.359531932511243       0.0203654918038924         2.91370745871451       -0.966236697048657        0.122926819026844         -0.00622031875815376       -0.50198090682801        0.143278573563769         -0.0148438008574466        0.000589893983812977       0.3864  0.4500       0.59312   0.54000 0.44000 0.43000 5   7   16   36
0.15       -6.84763242895318      7.87000388618251        -4.44998766512561       1.59154562270891         -0.303265788491283       0.0283077134037778        -0.00102444680777035       4.48739846882487        -2.53852631088059        0.440331431821927       -0.0245996160481055       -6.1133594786451        2.3390866160675          -0.363536726913453       0.0209362397321114         2.77485033369533       -0.951367631343794        0.129604571629895         -0.00689976391735384       -0.466097347690948       0.141727823006568         -0.0168707302811377        0.000777794204656026       0.3841  0.4675       0.60507   0.58095 0.47510 0.40075 5   7   16   36
0.2        -7.29227372671405      8.13942309560137        -4.56863133592809       1.62358617209995         -0.307951547712962       0.0286485840391632        -0.00103409385399879       4.36036305532997        -2.4984228046885         0.436575886111958       -0.0244959652476353       -5.9079024007467        2.32648933634059         -0.371464117327509       0.0217141113754647         2.65267532188177       -0.949645367349937        0.135734777112964         -0.00745289116944309       -0.43872960011329        0.141960264669707         -0.0182012444866795        0.000893389110404023       0.3690  0.4800       0.60546   0.61000 0.50000 0.38000 5   7   16   36
0.25       -7.73054138387465      8.45872077327395        -4.7358526093091        1.67317154083448         -0.315667132507769       0.0292422039135893        -0.00105195411086086       4.24667659538858        -2.46253297310816        0.433075034524838       -0.02439272157624         -5.7352678693926        2.31165354213656         -0.376514509780389       0.0222491030635771         2.55244861926371       -0.943309084656694        0.139073473502926         -0.00778634991064552       -0.417203926282832       0.140686458071138         -0.0187919995716921        0.000952819130014054       0.3445  0.4800       0.59082   0.62651 0.50000 0.37450 5   7   16   36
0.3        -8.17988467334188      8.85183972882936        -4.96025049488185       1.74113063101943         -0.326306724080913       0.0300644839154054        -0.00107684380059227       4.12869362576131        -2.42057920440422        0.428102339262794       -0.0241933050870513       -5.5689954537947        2.28526761331705         -0.37804332462455        0.0225355837414222         2.46010572831301       -0.930344969759           0.140236385505043         -0.00796510292359454       -0.39843961649765        0.138140301333051         -0.0189466197989054        0.000981330512346222       0.3377  0.4800       0.58688   0.64000 0.50000 0.37000 5   7   16   36
0.35       -8.66188171184458      9.33287639349336        -5.24308809583791       1.82618003648722         -0.339474154406273       0.0310723176165364        -0.00110711369256522       4.01626093451147        -2.38099478845544        0.423568770322557       -0.0240254202995875       -5.42604328064403       2.26470420396123         -0.380202568223964       0.0228599240706022         2.38488787543394       -0.921523377893384        0.141856129674051         -0.00817159299275658       -0.383755256105111       0.136439878931545         -0.0191923563633181        0.001015439531321          0.3482  0.4740       0.58812   0.62793 0.49396 0.37000 5   7   16   36
0.4        -9.18604365192386      9.90612885086109        -5.58290069497208       1.92699422568528         -0.354861236247393       0.032235514051287         -0.00114166926819002       3.88835715730968        -2.33209769602149        0.417292589917895       -0.0237534158055873       -5.27142196053445       2.23008606428153         -0.379007044560026       0.0229542562049625         2.30582573913596       -0.904946863620042        0.141488693849224         -0.00823741836706706       -0.368925269782073       0.133310502595041         -0.0190583079085078        0.00102231239241716        0.3552  0.4687       0.58811   0.61747 0.48874 0.37000 5   7   16   36
0.45       -9.68077374412945      10.4735632604716        -5.92160677883614       2.02630729904249         -0.369736238468528       0.0333344773603099        -0.00117346268144044       3.73895938957622        -2.27243033943263        0.409326594688858       -0.0233978543207019       -5.1133064176886        2.19200421616623         -0.377352210967507       0.0230423593562648         2.23138795527085       -0.889484813443787        0.141395291226649         -0.00833098088118912       -0.355759551922876       0.130669020402349         -0.0190242829246802        0.00103719788584282        0.3495  0.4641       0.58097   0.60825 0.48413 0.37000 5   7   16   36
0.5        -10.1833802463038      11.0843893031093        -6.28533312657581       2.13146776945271         -0.385250933098357       0.0344636078716458        -0.0012056266427278        3.56291879133144        -2.20011305470088        0.399416121864487       -0.0229446091920297       -4.93917375962532       2.14344805926688         -0.373757875043809       0.0230180061801003         2.15366623582567       -0.870485053485944        0.140557022432524         -0.00837840009741128       -0.342683208488467       0.127620838921729         -0.018893517661208         0.00104585918653073        0.3548  0.4600       0.58095   0.60000 0.48000 0.37000 5   7   16   36
0.55       -10.6243469588972      11.6354370494869        -6.61402957004652       2.22479478179912         -0.39861148015093        0.0353969438638686        -0.00123081588451206       3.34940514788907        -2.10881317010488        0.386374091511869       -0.0223224150191596       -4.73849833024474       2.07852180160289         -0.36712380668784        0.0228145649610222         2.06790979234295       -0.845286667710233        0.138450984025654         -0.0083458319819895        -0.328859135246228       0.123708762400072         -0.0185706818456272        0.00104171945661489        0.3639  0.4586       0.58544   0.59175 0.47587 0.37413 5   7   16   36
0.6        -10.9988109325128      12.114583940365         -6.89999935928425       2.30419382308659         -0.409545556914131       0.0361184500552806        -0.00124871950334325       3.10697307531469        -2.00460796423662        0.371482365294788       -0.021614809927102        -4.52206712692312       2.00557626535477         -0.359323786728137       0.0225612974639261         1.97987669690339       -0.818630936023193        0.136185452238801         -0.00831256951746349       -0.31530443963875        0.119828537964791         -0.0182705828372625        0.00104048590578779        0.3757  0.4574       0.59187   0.58422 0.47211 0.37789 5   7   16   36
0.65       -11.3512026319281      12.5802587316429        -7.17607478387407       2.3793592186049          -0.419612507799693       0.0367569277596558        -0.00126360560986596       2.84768606404911        -1.89272475083308        0.355459429683846       -0.0208529963237487       -4.30165700100619       1.92890273624425         -0.350751550995951       0.0222611532259719         1.89436792882545       -0.791888801025945        0.133771087580844         -0.00826603313897666       -0.302711467475983       0.116133085722302         -0.017974414756867         0.00103838275561395        0.3831  0.4562       0.59573   0.57729 0.46864 0.38136 5   7   16   36
0.7        -11.6004279044629      12.913284260413         -7.37355303832086       2.43021071819417         -0.4256543914868         0.0370576097227716        -0.0012671979534059        2.55279835167444        -1.76192132405262        0.336180860103017       -0.0199091334189132       -4.04513639069658       1.82903111143369         -0.337696313455597       0.0216896861746544         1.79286814420037       -0.753794343802052        0.129052958792721         -0.00807568636216131       -0.287605529011828       0.110509773123532         -0.0172769074937917        0.00101082180315902        0.3886  0.4551       0.59846   0.57087 0.46544 0.38456 5   7   16   36
0.75       -11.8603004280464      13.2759261400801        -7.58678451620096       2.48498820374787         -0.432245372046351       0.0374010987288753        -0.00127224233064803       2.24014759905471        -1.62185371163361        0.315369477146071       -0.0188835663944239       -3.77511420123633       1.71910058043383         -0.32265999639252        0.0210008136476775         1.68695728828935       -0.711399288623593        0.123424930298402         -0.00782973158316038       -0.272017619249973       0.104241711271665         -0.0164377316474162        0.000974507753172413       0.3852  0.4542       0.59554   0.56490 0.46245 0.38755 5   7   16   36
0.8        -12.0176717161099      13.495943977926         -7.71432236349306       2.51353769699166         -0.434564911675156       0.0373916446268329        -0.00126563708181116       1.91686378247181        -1.47528499314529        0.293321302104035       -0.0177833553037679       -3.49274569993584       1.59758266646052         -0.304989221415728       0.0201369130254802         1.57496427086918       -0.662451859773172        0.116245936745555         -0.00747852923869257       -0.255399941491243       0.0967706921043071        -0.0153098376111316        0.000918437336017169       0.3850  0.4532       0.59468   0.55932 0.45966 0.39034 5   7   16   36
0.85       -12.1290310458393      13.6536269447805        -7.80306747988614       2.52997347640759         -0.434909698551476       0.0372210285121824        -0.00125382189728355       1.58904699086541        -1.32517700001645        0.270535192105264       -0.0166370723039191       -3.20436908010954       1.46805732509913         -0.285390174094795       0.0191442432577689         1.45998137321526       -0.60892075375381         0.107939037803778         -0.00705133358240655       -0.23829596755932        0.0884739092519852        -0.0139777406848569        0.000848582383765583       0.3811  0.4523       0.59150   0.55407 0.45703 0.39297 5   7   16   36
0.9        -12.1754104827713      13.7176044158744        -7.83449535245302       2.52889428648464         -0.432436751706857       0.0368216342946972        -0.00123460493431406       1.26413210409957        -1.17483839351762        0.24748175962826        -0.0154660837517016       -2.91587620967333       1.33296579915129         -0.264167984825845       0.0180338299193111         1.34391524365954       -0.551486616311079        0.0985630597166428        -0.00654832349983898       -0.220907567582692       0.0794030996635413        -0.0124399758203431        0.0007643074152778         0.3799  0.4515       0.59006   0.54912 0.45456 0.39544 5   7   16   36
0.95       -12.2044961384555      13.7565538011934        -7.84905402481722       2.52252936975727         -0.429139408619667       0.0363597423720752        -0.00121355002152439       0.947168041497943       -1.02717529346776        0.224706386878877       -0.0143034469305691       -2.63162069491017       1.19524664087918         -0.241935121623825       0.0168453558863804         1.2285317483733        -0.491518350431163        0.0884233000800714        -0.00598997581182795       -0.203503170777188       0.0697921902089322        -0.0107508525745266        0.000669317514430019       0.3733  0.4507       0.58525   0.54444 0.45222 0.39778 5   7   16   36
1          -12.1795370113731      13.7132246193085        -7.81301985012853       2.50088959619489         -0.423426426857811       0.0357052354678675        -0.00118637242180901       0.643176289790913       -0.883485644330466       0.202238987223403       -0.0131419710429831       -2.35517000797328       1.0550945572454          -0.218481367360749       0.0155560332420378         1.11480305384198       -0.428665718757932        0.0773349624099544        -0.0053604505043136        -0.186153682777865       0.0595274064990588        -0.00886889912729578       0.000560351060712774       0.3716  0.4500       0.58362   0.54000 0.45000 0.40000 5   7   16   36
1.05       -12.1165366626058      13.6102437733086        -7.73981311566209       2.4680719162289          -0.415969908841846       0.0349143220899144        -0.00115495167595242       0.355385651138043       -0.745880822507659       0.180497006241999       -0.0120074864600627       -2.09004332793089       0.915341812954372        -0.194418542264127       0.0142053121773741         1.00439682327536       -0.36442010720273         0.0656296053398608        -0.00468121102978142       -0.169139370782075       0.0488760908489275        -0.00685404637812166       0.000441259573154984       0.3691  0.4482       0.58063   0.53956 0.44778 0.40000 5   7   16   36
1.1        -12.0210379099074      13.4544805298802        -7.63355830280403       2.42539052421664         -0.406993641269423       0.0340062192089138        -0.00111994190744572       0.0886482198009641      -0.616814670387012       0.159890838847101       -0.0109224935062633       -1.84238136618742       0.779452246829752        -0.170384870526812       0.0128313353497625         0.900516786126541      -0.300695422876658        0.0536777189563475        -0.00397540118709886       -0.153034969201214       0.0381944754063248        -0.00477738095597673       0.000316611553218219       0.3650  0.4465       0.57674   0.53913 0.44566 0.40000 5   7   16   36
1.15       -11.8960579970421      13.2495843430913        -7.49655276510345       2.3736029786462          -0.396628833416528       0.0329922833032517        -0.00108173144033541       -0.155684875105127      -0.496783086125259       0.14047713907855        -0.00988893693610611      -1.61281017582771       0.647760626484731        -0.146430279961017       0.0114361148725243         0.803126579983783      -0.237530865785825        0.0414853165618036        -0.00324284980444319       -0.137789630283001       0.0274706863210051        -0.00263658616029839       0.000186142177407789       0.3627  0.4449       0.57402   0.53873 0.44364 0.40000 5   7   16   36
1.2        -11.7434636304199      12.9986246140762        -7.3309167201457        2.31339921190078         -0.384988963435134       0.0318816845174657        -0.00104061008462844       -0.381110358483954      -0.383593680051076       0.121828627620502       -0.00888069557955401      -1.39568437325934       0.516932780456081        -0.121924732532862       0.00998205719174358        0.708950932060111      -0.173003742954199        0.0286890561833519        -0.00246194272192172       -0.122779989311309       0.0163376293042449        -0.000361849332159592      4.5690602386664e-05        0.3625  0.4434       0.57267   0.53834 0.44170 0.40000 5   7   16   36
1.25       -11.5926707423578      12.7409722754421        -7.15896528221466       2.25139005519995         -0.37313827963861        0.0307623897313601        -0.000999497841528905      -0.568290563474007      -0.288125629601404       0.105893065191151       -0.00800945132071919      -1.2171311189896        0.402067445151419        -0.0996120505557034      0.00862715353831925        0.631621499106222      -0.11509125086476         0.0167469677876177        -0.00171652345635349       -0.110410524148226       0.00621189038919793       0.00178752682909116        -8.98957055501344e-05      0.3631  0.4419       0.57195   0.53797 0.43984 0.40000 5   7   16   36
1.3        -11.3909878070866      12.4058256906154        -6.94146957452442       2.17626273320635         -0.359300477769032       0.0294908174095764        -0.00095372521874746       -0.74170660295306       -0.197731662683375       0.0905722497482138      -0.00716300235049178      -1.04842127784766       0.289231198416434        -0.0773517312815431      0.00726770899525228        0.5578518677467        -0.0577468643688325       0.0048152441098027        -0.000971711341919513      -0.0985826846959561      -0.00379356950347789      0.0039193428745855         -0.000223813132134637      0.3610  0.4404       0.56949   0.53761 0.43806 0.40000 5   7   16   36
1.35       -11.1975940756142      12.07575980752          -6.72538866435725       2.10172555169545         -0.345634617432115       0.0282411695725836        -0.000908935299805077      -0.889986426553788      -0.117215364088385       0.0764916761547775      -0.00636571668043435      -0.899531283756684      0.181322945200827        -0.0552203100439617      0.005884532546579          0.490672372316677      -0.00096661749564703      -0.00740116357853381      -0.000194296431050373      -0.0875166745128394      -0.0139051419761733       0.00613678255237413        -0.000365528486983678      0.3605  0.4391       0.56808   0.53727 0.43634 0.40000 5   7   16   36
1.4        -10.9804894381271      11.7024449872893        -6.48231906384576       2.01934361291732         -0.330794639069269       0.0269033653971636        -0.000861507411569368      -1.00288766608429       -0.0534783323712504      0.0650430808783444      -0.00570485190152688      -0.788256178607722      0.0905959160693988       -0.0357295241537077      0.00463697206206119        0.440741559437085      0.0478689108954358        -0.0183791374645523       0.00051855196955486        -0.0792655466274016      -0.0226979646311961       0.00814454070730636        -0.000496126881058344      0.3616  0.4377       0.56779   0.53694 0.43469 0.40000 5   7   16   36
1.45       -10.7387735003184      11.2884131529433        -6.21493084284263       1.93007515367773         -0.314944739350323       0.0254911678399596        -0.000811896880847182      -1.09873783932735       0.00406998929551105      0.0542910901599343      -0.00506723603237834      -0.688537401163256      0.00186634614931829      -0.0160831923149584      0.00336061684208483        0.394021840998408      0.0969778974697264        -0.029637380524525        0.00125624567181798        -0.0713072186504414      -0.0316495368700467       0.0102156830587489         -0.000631660237957282      0.3604  0.4365       0.56604   0.53662 0.43309 0.40000 5   7   16   36
1.5        -10.4919633853597      10.8631047686342        -5.94083247185099       1.83925349440107         -0.298952126686346       0.0240767967483139        -0.000762523546890327      -1.17580041023771       0.0540894365505764       0.044517479444981       -0.0044709092933846       -0.606056858788306      -0.0807220150847499      0.00286064923274443      0.00211000594856003        0.354696085384425      0.143377644723029         -0.0405625917581805       0.00197992754706714        -0.0645355002895721      -0.0401310950357525       0.0122215704493774         -0.000763998389558801      0.3559  0.4352       0.56223   0.53631 0.43155 0.40000 5   7   16   36
1.55       -10.2974495697444      10.5065026083775        -5.70533815720322       1.76008722415343         -0.284900317944731       0.0228290572846448        -0.000718889912856058      -1.21741396712159       0.0871171007822791       0.0374533669503973      -0.00401761358604044      -0.560545642292263      -0.146004386284132       0.0190659468494059       0.00100408062351468        0.331672212397723      0.182132027637879         -0.0502717440828037       0.00263912509607074        -0.0603131659964723      -0.047410088699334        0.0140351011841823         -0.000886132434558427      0.3507  0.4340       0.55804   0.53601 0.43005 0.40000 5   7   16   36
1.6        -10.0608543443571      10.0844822341777        -5.43162030511555       1.66998455156688         -0.269194877811737       0.0214540623894021        -0.0006713218685679        -1.24155131544127       0.112807384580884        0.0313917879542119      -0.0036089477389863       -0.531583916152389      -0.204617811448836       0.0343826199589012       -6.26814479241891e-05      0.316006730869159      0.217703535753815         -0.0595157986892322       0.00327541051161075        -0.0573122885481959      -0.0541121998266441       0.0157562088913107         -0.00100329657488032       0.3479  0.4329       0.55537   0.53572 0.42861 0.40000 5   7   16   36
1.65       -9.82766520745761      9.66290753000509        -5.15782207245742       1.58020727294576         -0.253627638828993       0.020098037398916         -0.000624620429396877      -1.24483408815055       0.12958171219166         0.0265906068741378      -0.00325939062204343      -0.522398060915442      -0.255314855152455       0.0486420954026141       -0.00108210350585456       0.308581845925214      0.249982746020979         -0.0683178203733977       0.00389165458225578        -0.0555550092917421      -0.0603076889217176       0.0174072366089256         -0.00111717118094394       0.3486  0.4318       0.55495   0.53544 0.42721 0.40000 5   7   16   36
1.7        -9.59244643489873      9.23470544155279        -4.879999276633         1.48957189972873         -0.238001490033522       0.0187439389104228        -0.000578191313522384      -1.22991634587749       0.138353932395569        0.0229532802383305      -0.00296588490684986      -0.530850414172782      -0.298236120827394       0.0617429875177761       -0.00204357224627463       0.308760585815846      0.278708557158293         -0.0765573834682871       0.00447816684801963        -0.0549950277956724      -0.0659070360676986       0.0189585990041752         -0.00122555512102799       0.3485  0.4307       0.55401   0.53517 0.42585 0.40000 5   7   16   36
1.75       -9.35967704052859      8.80737321972316        -4.60281240795096       1.39951018927609         -0.222550990612055       0.0174114482904832        -0.00053270027851953       -1.20027406640391       0.140887976238121        0.0201892351942106      -0.00271278191362115      -0.553182228039351      -0.33524800447137        0.073986518334255        -0.00296292252322432       0.314982676272411      0.304622731767059         -0.0843471062300593       0.00504037355069389        -0.0554106846704366      -0.0710087775905702       0.0204236695826417         -0.00132897432768611       0.3470  0.4296       0.55226   0.53491 0.42453 0.40000 5   7   16   36
1.8        -9.11407840055304      8.35586408800226        -4.31090531899992       1.30533795192728         -0.206512663008318       0.0160368716138508        -0.000486008729571035      -1.14431004256055       0.130371959874325        0.0195578811673562      -0.00257449860103428      -0.605258269534203      -0.356966712141302       0.0836318196848094       -0.00373697349812041       0.335454250465783      0.322877224521207         -0.0907898454208943       0.00552522796629837        -0.0582341615237234      -0.0747688547420187       0.0216467354254372         -0.00141824726942853       0.3469  0.4286       0.55137   0.53465 0.42325 0.40000 5   7   16   36
1.85       -8.90068844313518      7.9516141816724         -4.04760700652393       1.22018972364116         -0.192014519334466       0.0147965446580548        -0.000443992652929604      -1.08209328854446       0.118583364084734        0.0188741153500208      -0.00242123646803106      -0.660094089008369      -0.379375920527606       0.0936787466242656       -0.00454633333416504       0.355961236617556      0.341964657697977         -0.0974937285226093       0.00603043944602702        -0.0609221433723983      -0.0787177892574688       0.0229196810890016         -0.00151123498953631       0.3448  0.4276       0.54931   0.53440 0.42200 0.40000 5   7   16   36
1.9        -8.67895681542497      7.53212136160157        -3.7752830999781        1.13262661517981         -0.177190557912272       0.0135345802714025        -0.000401415408911227      -1.00403377659027       0.0991575582448954       0.0194061434763803      -0.00233192321627377      -0.732186744612621      -0.392812373969263       0.102187868098509        -0.00526918408778778       0.384747154464736      0.356399736702574         -0.103354980064583        0.00648635308902235        -0.0650245681195155      -0.0818267848403212       0.0240351759519849         -0.00159480901828573       0.3435  0.4266       0.54775   0.53416 0.42079 0.40000 5   7   16   36
1.95       -8.4642244037567       7.12071862275467        -3.50796119918045       1.04691088397            -0.162735469879781       0.0123089297626789        -0.000360220332514162      -0.911154286586885      0.0729051229901854       0.0209841226696251      -0.00229596328015043      -0.820454835718664      -0.398162065037551       0.109345979608812        -0.00591704892953151       0.421241102405562      0.366628859309871         -0.108465795594491        0.00689855644318238        -0.0704395249981801      -0.0841701176114172       0.0250080543637616         -0.00166984565445949       0.3422  0.4257       0.54621   0.53392 0.41961 0.40000 5   7   16   36
2          -7.83770145489928      6.07188862862987        -2.86436695876577       0.850334289427231        -0.130819716492896       0.00967475552567557       -0.000273226794952762      -0.62917562743282       -0.0628481959419428      0.0424915960577013      -0.00341840055268141      -1.14202411461916       -0.269710842621687       0.0926479275892739       -0.00521681407606963       0.573065803225286      0.311010995009485         -0.101993105173376        0.0066696514260539         -0.095649447259471       -0.0752125115928811       0.0240051081822503         -0.00163680587424627       0.3412  0.4248       0.54482   0.53369 0.41845 0.40000 5   7   16   36
2.5        -6.51722151721519      3.16964064522617        -0.936262225911006      0.234219342888514        -0.0283396161843896      0.00114440451673939       7.63296518015248e-06       0.642814386929648       -0.46637507609868        0.0773849165930253      -0.00391450073459671      -2.36186725580886       -0.14099556405315        0.131572822353246        -0.00977970825434678       1.09500200051713       0.308164481162873         -0.131876374065076        0.00945747231422048        -0.176574449497538       -0.0783218153005598       0.029418897653178          -0.00210967701968961       0.3321  0.4166       0.53278   0.53166 0.40830 0.40000 5   7   16   36
3          -5.54577669335058      0.922306148406204       0.527346070176398       -0.216506332966857       0.04343661991642         -0.00455802647772572      0.000186523680739068       1.87373240442575        -0.84755472764872        0.108438465173355       -0.00420828163023053      -3.42053347594367       -0.0395815899404806      0.168810710191356        -0.0140207799190575        1.51642246381928       0.31216792678282          -0.158377822036962        0.0119219976065103         -0.239049005249037       -0.0814486393779304       0.0339403081375849         -0.00251022814473902       0.3244  0.4100       0.52282   0.53000 0.40000 0.40000 5   7   16   36
3.5        -5.04330552555988      -0.649315611712872      1.57962358179259        -0.534557981094911       0.0922155301020585       -0.00824939648673273      0.000295868718893165       3.89371479460061        -1.73185963565765        0.236837944061824       -0.0103891537728855       -5.75787290266484       0.940842545989138        0.0287589596018022       -0.00723916066519796       2.67502128982006       -0.201938445133363        -0.0793556715803847       0.00778577343847562        -0.440131292491045       0.0136024331102916        0.0183988132949693         -0.00165275027729622       0.3293  0.4100       0.52588   0.53000 0.40000 0.40000 5   7   16   36
4          -4.2225304967382       -2.19753302412409       2.48762626636418        -0.791360423681221       0.129842397023272        -0.0109777536021221       0.000372801076244668       3.93636090320446        -1.5408498804173         0.17689093129539        -0.00591328097414771      -5.3904634149122        0.448148182440251        0.149644286485233        -0.015452881305532         2.40637887738939       0.0692913315077956        -0.141527322032787        0.0118949428661375         -0.386611076289938       -0.0329680646966726       0.0286454750409389         -0.00232048151512642       0.3244  0.4100       0.52282   0.53000 0.40000 0.40000 5   7   16   36
5          -5.63126325508582      -1.10699895984668       1.94198594622954        -0.609466168079661       0.0938114094185452       -0.00730702130966831      0.000226646391993371       6.84127589845963        -2.97963840549242        0.412510082560923       -0.0186219921532312       -9.01282847850288       2.34718362358587         -0.179633526270394       0.00324629705464769        4.29327244839703       -1.00030242988993         0.0540307917870391        0.000391947723575771       -0.724288710961063       0.169594426570596         -0.00957312608929916       -3.15814831478276e-05      0.3293  0.4100       0.52588   0.53000 0.40000 0.40000 5   7   16   36
6          -6.76907053920891      0.202138074104118       1.13249676238449        -0.335955311412779       0.0431281335038359       -0.00252330805068563      4.81278784375445e-05       7.65594965989276        -3.39508840881319        0.482861913050879       -0.0225463337832774       -9.97746756729124       2.94929773271389         -0.296333811904845       0.0103662399355933         4.81485017899813       -1.38514405955285         0.133380998773158         -0.00455625439659101       -0.821123188466859       0.247775121042361         -0.0260523701271966        0.000999149874580045       0.3244  0.4100       0.52282   0.53000 0.40000 0.40000 5   7   16   36
7          -8.01442369974692      1.88786823491218        0.0744192603367332      0.00931277061574391      -0.0178435579540363      0.0029681495873398        -0.000148609112869026      7.92625054316848        -3.61028736170505        0.529908645097125       -0.0256288880732491       -10.3471103141499       3.35192497152072         -0.391576701905342       0.0167565233076556         5.04380483489874       -1.66147515546696         0.198455754896224         -0.00883743849124209       -0.866762064547418       0.304626761999033         -0.0392841518248452        0.00185536287317698        0.3293  0.4100       0.52588   0.53000 0.40000 0.40000 5   7   16   36
8          -9.1796161266094       3.5781245586623         -0.993043607513015      0.352513068748286        -0.0771607041254957      0.00819243236517093       -0.000331887710819227      7.86250187925788        -3.69570811421508        0.5604321701027         -0.0280208937560745       -10.3391322055039       3.60529393602782         -0.466264632914397       0.0221633160505526         5.07912054871935       -1.84895208211491         0.249060057017656         -0.012312515226379         -0.877499127618483       0.343434856293276         -0.0492974084609317        0.00252139350759871        0.3244  0.4100       0.52282   0.53000 0.40000 0.40000 5   7   16   36
10         -11.030416932437       6.4073906881075         -2.78872801692642       0.9220857725619          -0.173662685554259       0.0165097358257458        -0.000617444372892825      7.24803069617542        -3.65629472617652        0.590630841586317       -0.0312711518118421       -9.7325653015372        3.81629835282476         -0.564462507535905       0.030013935165111          4.8242446831363        -2.03005414194837         0.312885387538251         -0.0169624719152322        -0.837632534522922       0.380527293209967         -0.0611530356764195        0.00334168724343673        0.3293  0.4100       0.52588   0.53000 0.40000 0.40000 5   7   16   36
""")

COEFFS_ALPINE_10Bars = CoeffsTable(sa_damping=5, table="""\
IMT        a1                     a2                     a3                     a4                      a5                       a6                         a7                         a8                       a9                      a10                    a11                       a12                    a13                   a14                     a15                      a16                    a17                     a18                     a19                        a20                     a21                     a22                       a23                       tau    mean_phi_ss  sigma_tot  phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv        -6.73278985832203      5.19133282365296       -2.59678564556577      0.880394919068679       -0.163204261514581       0.0149731051892756         -0.000534896073786955      2.34609878459778         -1.49390180525907       0.265902409393499      -0.0148184219802127       -4.06272471084695      1.33873053770146      -0.183541997287608      0.00992977794838915      1.79248681870456       -0.504507954892307      0.0483559156296854      -0.00177227886957151       -0.29782904101622       0.0734342574227862      -0.00400231608228209      -6.42256411144852e-05     0.5010 0.4600       0.68015    0.00000 0.00000 0.00000 5   7   16   36
pga        -4.95222817214066      5.57658327256026       -2.93739649357216      0.996074067935278       -0.184045318828244       0.0168307795663732         -0.00060001704332761       3.28169773334305         -2.11667759580151       0.389914683878208      -0.0225599657233531       -5.49982105320247      2.27668684986964      -0.369709685277719      0.0215945550199431       2.5992355053784        -1.04575619118427       0.155524901937975       -0.00847366439312516       -0.463325210650184      0.181843648447585       -0.0255060742167307       0.00127781726470167       0.3532 0.4600       0.57998    0.58000 0.47000 0.35000 5   7   16   36
0.01       -4.94340564263821      5.78393494087865       -3.11447105767403      1.05657797414732        -0.194332271209223       0.0177020390839611         -0.000629466901279084      2.38949071486829         -1.57499726488321       0.287626072403424      -0.0164519316781909       -4.14616629421843      1.43706414493117      -0.208912674535153      0.011877699983892        1.8331386651959        -0.568289665255383      0.0632407033004863      -0.00284445673005812       -0.320745687623285      0.092718298645911       -0.00816272617946801      0.000212406695378281      0.3529 0.4600       0.57975    0.58000 0.47000 0.35000 5   7   16   36
0.02       -4.57082410507914      5.51985232802787       -2.98250238830287      1.01952874403102        -0.188430677167646       0.0172055128583565         -0.000612571836443491      2.72881196730669         -1.72494631814995       0.310029501256239      -0.0176053819415808       -4.39786743201624      1.53345943755556      -0.221602522454486      0.0124607424699436       1.80027441594302       -0.556229554665553      0.0591743628750943      -0.00249455021671123       -0.294499241157592      0.0833655716043716      -0.00619723668703394      7.23869306782044e-05      0.3674 0.4570       0.58639    0.56796 0.46097 0.37408 5   7   16   36
0.03       -3.82459362054579      5.00700459454185       -2.76386312006616      0.963147139659348       -0.180053836524008       0.0165586485645485         -0.000592620598255524      1.52904195043912         -1.31823673271313       0.267571965169328      -0.0163750813694059       -2.72713764261347      1.04807746772381      -0.179224273823143      0.0116581344168136       0.960875554332413      -0.35587845273702       0.0467991045647839      -0.00262082086863915       -0.162301769578943      0.0593669082598373      -0.00599483607234699      0.000207635579879689      0.3749 0.4552       0.58975    0.56092 0.45569 0.38817 5   7   16   36
0.04       -3.8120968904055       5.15815895250165       -2.84526884678431      0.983147475364237       -0.182709301524906       0.0167464351821095         -0.000598119048453512      1.47399903062782         -1.4839415376309        0.318616595642638      -0.0201446231995355       -3.027143800769        1.52898379097062      -0.29651088342609       0.0196664777457471       1.36353880128916       -0.740831080631579      0.131793916072025       -0.00823119687832395       -0.274996865472656      0.14711223203177        -0.0244262048292354       0.00140331597566806       0.3801 0.4540       0.59211    0.55592 0.45194 0.39816 5   7   16   36
0.05       -3.95552441394056      5.26517689962899       -2.86897467281632      0.984326102724191       -0.18230835205613        0.0166777830377943         -0.000594819944496831      1.81914753277093         -1.61136390981386       0.332289554357177      -0.020537864194039        -3.61760327110464      1.73454977848668      -0.317091863898606      0.0202061171832118       1.79891205478414       -0.883007695967814      0.146617726620382       -0.00870853230687144       -0.370813399899288      0.175863628483687       -0.0273620486680476       0.00149913042747762       0.3855 0.4530       0.59483    0.55204 0.44903 0.40592 5   7   16   36
0.1        -5.85229400523193      6.6030379684983        -3.31625064033802      1.0754290797222         -0.193593414357971       0.0174431297107644         -0.000616207929043142      5.11214265964732         -2.95039346123074       0.516699425696372      -0.0289700318793403       -7.55464055056388      3.21110309510992      -0.516584580693888      0.029351455501348        3.68577233857576       -1.49565004596841       0.225377423941959       -0.0122264098339665        -0.655284329360909      0.249531382586354       -0.0357102398497449       0.00182995145814776       0.3864 0.4500       0.59312    0.54000 0.44000 0.43000 5   7   16   36
0.15       -6.57461241714824      6.95353608086672       -3.35925006052761      1.06444670932433        -0.189678185748279       0.0170131666210095         -0.000599687282981999      5.01995670788962         -2.8790002814629        0.503091704738226      -0.0281477949586574       -7.16228238258447      3.02805447857801      -0.491892759196714      0.0282192641793496       3.38012693578566       -1.35854651497304       0.207965013118943       -0.0114751782104665        -0.577941329703092      0.218102555282251       -0.031879501876696        0.00167048834570013       0.3841 0.4675       0.60507    0.58095 0.47510 0.40075 5   7   16   36
0.2        -6.79072043690986      6.68834661324982       -3.09739038017926      0.975391585149994       -0.174871642430532       0.0157949766708873         -0.000559952782487129      4.90935742743472         -2.84714490871819       0.500923465890195      -0.0281486479934738       -6.89069169084763      2.96658132926816      -0.490611183296693      0.0284839754564247       3.19815963867319       -1.31368015388883       0.205793718992675       -0.0115491826355822        -0.536485416669622      0.208256387368126       -0.031259378854171        0.00167233162131797       0.3690 0.4800       0.60546    0.61000 0.50000 0.38000 5   7   16   36
0.25       -6.70097293059521      6.09599089840666       -2.69627389371743      0.856287703544691       -0.15659116195702        0.0143727143051738         -0.000515488825373093      4.81951774657993         -2.82535004964033       0.500221380101         -0.0282248119649121       -6.69285414119525      2.93209546431812      -0.492213585252545      0.0288503809532075       3.07105143889861       -1.28740081483836       0.205361169562607       -0.0116723342663118        -0.508689082829116      0.202372688296399       -0.0309584792821746       0.00168036111936607       0.3445 0.4800       0.59082    0.62651 0.50000 0.37450 5   7   16   36
0.3        -5.94561011303405      4.90816786672558       -2.06188216980998      0.684937856392754       -0.131342434859873       0.012442400869633          -0.000455680146758953      3.32480259831466         -1.97177397882023       0.344785647971105      -0.0191443449956011       -4.50314889902324      1.68286489170276      -0.262735860152548      0.0152907775126379       1.87436095420539       -0.597433965680859      0.0770864908164495      -0.00401243584858059       -0.289165753857585      0.0755957582459038      -0.00716692052879628      0.000248286369854768      0.3377 0.4800       0.58688    0.64000 0.50000 0.37000 5   7   16   36
0.35       -6.24911257839285      4.83527733942736       -1.96610544932997      0.661137051294115       -0.129101595527354       0.0123860581448454         -0.000457287224356212      4.57778449405185         -2.73691523733108       0.489768062995806      -0.0278279009970345       -6.29327818002346      2.82520857397399      -0.485202816344541      0.0288888576322702       2.83947703735591       -1.2220753543365        0.199948365329002       -0.0115902060219476        -0.462368571895056      0.189247166205328       -0.0296406462533634       0.0016419540564382        0.3482 0.4740       0.58812    0.62793 0.49396 0.37000 5   7   16   36
0.4        -5.63766078867142      3.95534396885482       -1.54129760017866      0.555728655432419       -0.114509883078247       0.011315584563428          -0.00042493719296405       3.301827528637           -2.01227978670943       0.358383591323122      -0.0201810254793487       -4.46713830947527      1.78247373295307      -0.293730232809254      0.017583718797051        1.85206204436123       -0.649707463290218      0.0933461906222039      -0.00522059040543528       -0.282953497608553      0.0844524718821182      -0.0099049095560594       0.00045208871597796       0.3552 0.4687       0.58811    0.61747 0.48874 0.37000 5   7   16   36
0.45       -6.01322078661928      4.16892406064532       -1.66617458210565      0.605224923686399       -0.124577357190736       0.012290358967416          -0.000460885228238679      4.37047147757931         -2.6691727015374        0.483480676894787      -0.0277016311374837       -6.06172334108689      2.79973321327779      -0.492277520465417      0.0297499758847221       2.72681541711853       -1.21328389740911       0.204305842744495       -0.0120788101568115        -0.441783835177094      0.18765594092771        -0.0303163966797262       0.00172033162821437       0.3495 0.4641       0.58097    0.60825 0.48413 0.37000 5   7   16   36
0.5        -5.54703204583303      3.62776335833839       -1.46843192912975      0.570642649881715       -0.121463101046155       0.0121589861317415         -0.000459251673284492      3.14236956844305         -1.98539213919441       0.361505180223164      -0.020696789594243        -4.4094761589569       1.87492008577257      -0.32545550538878       0.0200459496828099       1.86108300401723       -0.721192678711538      0.114307400651482       -0.00678198241472554       -0.287676413938331      0.0992049372660278      -0.0139554787447069       0.000748465932960341      0.3548 0.4600       0.58095    0.60000 0.48000 0.37000 5   7   16   36
0.55       -6.06297391997808      4.160266878749         -1.79243238425597      0.676889135181971       -0.139841478239593       0.0137444422244049         -0.000513081091919269      3.95717895534362         -2.49059460456163       0.458080567828257      -0.0265110148845786       -5.67406700769499      2.67957293180692      -0.481978738862601      0.0296032106594153       2.56134788655627       -1.16822952793179       0.201716386331658       -0.0121538304563782        -0.415214911883725      0.180918129173038       -0.0299716598112439       0.00173677549486058       0.3639 0.4586       0.58544    0.59175 0.47587 0.37413 5   7   16   36
0.6        -6.18478598875312      4.34787728269479       -1.97027589332668      0.74496582376613        -0.152205429407265       0.014823376556702          -0.000549676207949072      3.75587026353962         -2.40848066333342       0.447248858375136      -0.0260503157792383       -5.53647783603621      2.65350543028408      -0.483410476544477      0.0299351192818486       2.51950144254915       -1.16945284789964       0.204912793811777       -0.0124635933736659        -0.410387743886105      0.182336694460134       -0.0307010729933113       0.0017995448916711        0.3757 0.4574       0.59187    0.58422 0.47211 0.37789 5   7   16   36
0.65       -5.90485368982075      4.17474614927986       -1.99203464072608      0.77197394440651        -0.158120134161773       0.0153633986939397         -0.000568091052762387      2.50320536553828         -1.71279399802747       0.323317037002003      -0.018936324574651        -3.92109619949952      1.74182299182066      -0.317936054680062      0.0202580852065794       1.68657851528508       -0.68957552160742       0.116311275196392       -0.00720799403482084       -0.263342737896303      0.0963966936311044      -0.01462536970795         0.000835944536378728      0.3831 0.4562       0.59573    0.57729 0.46864 0.38136 5   7   16   36
0.7        -6.10036083465591      4.52311610668377       -2.26326225803764      0.864680637574711       -0.173806697364539       0.0166658720736759         -0.000610631533580735      2.20155398126771         -1.56828708390131       0.300353795143007      -0.0177283559861386       -3.66807609369106      1.63310263612858      -0.301908016176159      0.0194536306001369       1.58668751986902       -0.646581619297509      0.109863447420577       -0.00687690944577935       -0.248363778998082      0.0897696874362339      -0.0135748602211045       0.000779324702904831      0.3886 0.4551       0.59846    0.57087 0.46544 0.38456 5   7   16   36
0.75       -6.23111941194256      4.78058937826422       -2.4743287517683       0.936914346718785       -0.185820274057095       0.0176378622789439         -0.000641418393178178      1.94557221001802         -1.45807705241623       0.284795721983869      -0.0170071484721875       -3.51363865660367      1.59301516707154      -0.300311534206056      0.0195892239637662       1.54900128787676       -0.646483207289082      0.112428949313383       -0.00713390056693586       -0.245595760753681      0.0915347774557756      -0.0142916498569251       0.00083827214667508       0.3852 0.4542       0.59554    0.56490 0.46245 0.38755 5   7   16   36
0.8        -7.13702869989993      6.01314880728875       -3.20520687943345      1.15667468989297        -0.220939864224904       0.0204860479554101         -0.000733725416688691      2.63400594382884         -1.90702764285242       0.37347832448344       -0.0224734009748261       -4.74698408378535      2.39237289224797      -0.458280702697524      0.029362713792563        2.2701645892846        -1.11145037925085       0.204415515065537       -0.012846775819591         -0.380976613590171      0.178692522489454       -0.0315528318707114       0.00191345497718209       0.3850 0.4532       0.59468    0.55932 0.45966 0.39034 5   7   16   36
0.85       -7.44200515764853      6.54977771976624       -3.57403824972817      1.27299181163815        -0.239568140578552       0.0219703126372611         -0.000780622106748292      2.33370108278065         -1.76899920971921       0.352559539478834      -0.0214272667283639       -4.54475492558777      2.31569321796063      -0.448870013982316      0.0289925763404901       2.20948821138035       -1.09227153787793       0.202753936496775       -0.0128233564274814        -0.374283157210801      0.177041581636076       -0.031500814761593        0.0019207716417359        0.3811 0.4523       0.59150    0.55407 0.45703 0.39297 5   7   16   36
0.9        -7.7439035952972       7.08053949732918       -3.93365384232043      1.38480279474258        -0.257233752644751       0.0233595596144882         -0.000823960297166735      2.04280601661303         -1.63501561958394       0.332205442201762      -0.0204069264666367       -4.35768992226324      2.2440951833129       -0.439991594167071      0.0286401761562121       2.15779179440842       -1.07636186281153       0.201514529117409       -0.0128205433147243        -0.36936950000068       0.17610887468198        -0.0315594717979801       0.00193454251758839       0.3799 0.4515       0.59006    0.54912 0.45456 0.39544 5   7   16   36
0.95       -8.04891888102555      7.61209841215741       -4.28791052890251      1.49339629343731        -0.274173861736115       0.0246758801017241         -0.000864551095490243      1.76113960468288         -1.5040353611962        0.312092308985943      -0.0193867782810629       -4.17722478626316      2.17062664695211      -0.430019325856996      0.0281892713233033       2.10692218624099       -1.05743643282192       0.199244883756869       -0.0127344270414855        -0.364203459401222      0.174344965150859       -0.0313701912063937       0.00192899634289215       0.3733 0.4507       0.58525    0.54444 0.45222 0.39778 5   7   16   36
1          -8.36604625803202      8.15558265728435       -4.64325304992274      1.60078276923341        -0.290729129143335       0.0259486499157244         -0.00090340255094146       1.49250570989151         -1.37776270012625       0.292492732369721      -0.0183823859790109       -4.00820637938167      2.09822589823509      -0.419590826186291      0.0276860223423893       2.0603659825817        -1.03799943471887       0.196539232983493       -0.0126097290686529        -0.359648836081843      0.172410891580887       -0.0310930443844892       0.00191622560280307       0.3716 0.4500       0.58362    0.54000 0.45000 0.40000 5   7   16   36
1.05       -8.64004770382651      8.62928257222459       -4.95229226099623      1.69300784326539        -0.304685191053783       0.0269974540743613         -0.000934594321353496      1.23018253519249         -1.25178443198509       0.272509171963461      -0.017336419524401        -3.83806632916473      2.01885909527759      -0.407087337680945      0.0270272163145875       2.01049254233146       -1.01317395123152       0.192410337612248       -0.0123831304072641        -0.354212390267101      0.169348316102347       -0.0305326087598764       0.0018836064737227        0.3691 0.4482       0.58063    0.53956 0.44778 0.40000 5   7   16   36
1.1        -8.98249409329411      9.1899088115485        -5.30538357576962      1.79703718458428        -0.320408297157372       0.0281856354896864         -0.000970287366709895      1.01842450368902         -1.15255267791962       0.25713380562502       -0.0165490003953797       -3.72856170137406      1.97048753735136      -0.399835105367406      0.0266609643499768       1.99011747438066       -1.00341689173438       0.190848827957735       -0.0123006078523918        -0.353750415405057      0.168839183409124       -0.0304115768709193       0.00187587230002156       0.3650 0.4465       0.57674    0.53913 0.44566 0.40000 5   7   16   36
1.15       -9.24762828256563      9.63389720872763       -5.58649524234659      1.8786364177732         -0.332410881819215       0.0290604634611394         -0.000995440909904803      0.786290991616621        -1.0373462037359        0.238285846061375      -0.015534348164595        -3.57658975793022      1.88986647790778      -0.385672555418686      0.0258462431463983       1.94348185442612       -0.974163574704019      0.185148492373499       -0.0119507461899547        -0.348216482575042      0.164614968563136       -0.0295069450677278       0.00181768742004964       0.3627 0.4449       0.57402    0.53873 0.44364 0.40000 5   7   16   36
1.2        -9.50653772333204      10.0534807773951       -5.84676121473736      1.95283599159771        -0.343116146085693       0.0298234109492259         -0.00101679262477446       0.599575589719684        -0.945280248278197      0.223299222155991      -0.0147306330410192       -3.47422192269275      1.83314108262021      -0.37535330336132       0.0252366686449207       1.91957155633632       -0.955700979951568      0.181175337724714       -0.0116931424720605        -0.346356369714756      0.162130015061414       -0.0288821990386464       0.00177456407456576       0.3625 0.4434       0.57267    0.53834 0.44170 0.40000 5   7   16   36
1.25       -9.75360643953075      10.447660095092        -6.08801854864642      2.02059140484129        -0.352717978834936       0.0304928681569519         -0.00103502128951496       0.42554213113925         -0.85726997393624       0.208652272952043      -0.0139301867109276       -3.37702791690119      1.77429191338847      -0.364043178712573      0.0245452648742628       1.89539078335774       -0.934523003596274      0.176411150124935       -0.0113778799111334        -0.344163137817712      0.159029283972906       -0.0280977919955088       0.00172032801668485       0.3631 0.4419       0.57195    0.53797 0.43984 0.40000 5   7   16   36
1.3        -9.99090637878099      10.815493496161        -6.30900901740079      2.08151676766717        -0.3611635875153         0.0310653655261301         -0.00105003470561441       0.281580137904347        -0.78295133501405       0.196057772930378      -0.0132311416189576       -3.30656897046017      1.72533957619042      -0.353884864133991      0.0238946603530229       1.88136802135698       -0.916410460970817      0.171876875728064       -0.0110626336762616        -0.343351562033223      0.156252358076985       -0.0273169370329975       0.00166400504744978       0.3610 0.4404       0.56949    0.53761 0.43806 0.40000 5   7   16   36
1.35       -10.1732258347161      11.0932709503838       -6.47327443441961      2.12484430795851        -0.366727441603741       0.0313986497478359         -0.001057110925981         0.150355022589769        -0.7125543017913        0.183746527894407      -0.0125302770074956       -3.23837849206709      1.67233467546656      -0.342323984917297      0.0231344080014947       1.86451476710503       -0.893901154602647      0.166199558709841       -0.0106663800753101        -0.341614720166507      0.152468472644512       -0.0262945703570642       0.00159116218854305       0.3605 0.4391       0.56808    0.53727 0.43634 0.40000 5   7   16   36
1.4        -10.3982496146427      11.4221416163443       -6.66315128796271      2.17519243481226        -0.373384314816261       0.0318219244912458         -0.00106721152382925       0.0545485943782803       -0.659329853671787      0.174186523129257      -0.0119746583270905       -3.20768530189717      1.63674725587407      -0.333492431639839      0.0225173427855966       1.86487312793412       -0.87945574086619       0.161811109824644       -0.0103395315658473        -0.342706196488601      0.150038386555801       -0.0254947705246085       0.00153064745941975       0.3616 0.4377       0.56779    0.53694 0.43469 0.40000 5   7   16   36
1.45       -10.5788865726781      11.6769178094042       -6.80633180553599      2.21114878358646        -0.377705788671299       0.0320532200885894         -0.00107100289738497       -0.0273002885767304      -0.610795289596277      0.165066692118875      -0.0114275335670715       -3.18127421741347      1.59870689694998      -0.323616999557168      0.0218150215763575       1.86356065280971       -0.861591131717806      0.156502353127328       -0.00994686513042261       -0.343083717624501      0.146785779665896       -0.0244957262664151       0.00145649320432776       0.3604 0.4365       0.56604    0.53662 0.43309 0.40000 5   7   16   36
1.5        -10.7253177574857      11.8715518140139       -6.91087090681177      2.23514388351234        -0.380089943632234       0.0321259278094772         -0.00106960663973207       -0.0846338042400997      -0.573806087666027      0.157746137199576      -0.0109730820343427       -3.17660000802614      1.56973438560236      -0.315012316702924      0.0211724755762935       1.87080422805397       -0.847077112780071      0.151636560133445       -0.0095739613441794        -0.344703867165261      0.144006012608065       -0.0235582230381599       0.00138506210098862       0.3559 0.4352       0.56223    0.53631 0.43155 0.40000 5   7   16   36
1.55       -10.8756461700121      12.0652372718373       -7.01249798347479      2.25800548809555        -0.382291642442912       0.0321854038650211         -0.00106785920711947       -0.129842174957687       -0.540724965186488      0.15076539534134       -0.0105231494260511       -3.17700549065034      1.5397026446205       -0.305755746147869      0.0204738676587          1.8780205208605        -0.830742687506774      0.146236968014799       -0.00916214879290558       -0.3460643535327        0.140799233673241       -0.0225126979964717       0.00130617901784245       0.3507 0.4340       0.55804    0.53601 0.43005 0.40000 5   7   16   36
1.6        -10.9833303061195      12.1857083380499       -7.06793313906808      2.26671918052959        -0.382214916463758       0.0320588722308826         -0.00106003229442053       -0.147300497836489       -0.521186733777122      0.145960928112605      -0.0101883856639216       -3.20328889914106      1.5214107369498       -0.298281457313746      0.0198656691638992       1.89584237245577       -0.819095354037188      0.141536681199212       -0.0087854950949253        -0.349008126263547      0.138288651974306       -0.0215716281995423       0.00123261704997079       0.3479 0.4329       0.55537    0.53572 0.42861 0.40000 5   7   16   36
1.65       -11.0899665126792      12.3006239355091       -7.11868427638342      2.2739084214567         -0.381902198006421       0.0319149243967214         -0.00105171534835182       -0.167359509570766       -0.496756998734472      0.139847311480168      -0.00975975766240516      -3.21249849722016      1.48849511595247      -0.287558249080705      0.0190435125735342       1.90101830233647       -0.797740342945102      0.134767848995223       -0.00827537554815845       -0.349304499970012      0.13383784990281        -0.0202250973386755       0.00113313473304186       0.3486 0.4318       0.55495    0.53544 0.42721 0.40000 5   7   16   36
1.7        -11.1831263709283      12.3852051576195       -7.14873996212983      2.27470609574521        -0.380581969725576       0.0316915361119575         -0.00104090377535202       -0.147267519129297       -0.494017407425819      0.137531539253905      -0.00954708418223327      -3.2705829515606       1.48282322977806      -0.281762727757784      0.0185103336233735       1.93111568857079       -0.790813960614231      0.130690422952715       -0.00792707627195025       -0.35402293783389       0.132023304340622       -0.0193812276163175       0.00106437657254575       0.3485 0.4307       0.55401    0.53517 0.42585 0.40000 5   7   16   36
1.75       -11.2556835266852      12.4334459536616       -7.15582338267866      2.26856802247532        -0.378170521228041       0.0313815703168549         -0.00102735095815433       -0.114639677899369       -0.496112318440612      0.13580994576068       -0.00935736774865301      -3.3349607213661       1.47777071334648      -0.275733281391485      0.0179497702923096       1.96180946388431       -0.782862665446572      0.126269152203232       -0.0075524188501977        -0.358576745636079      0.129900560447983       -0.018456817998954        0.000989919823768119      0.3470 0.4296       0.55226    0.53491 0.42453 0.40000 5   7   16   36
1.8        -11.315531077027       12.4582799109555       -7.14769940187915      2.25776686972357        -0.37501775941836        0.0310122761516156         -0.00101190338277755       -0.0764805073923097      -0.499199367727124      0.13400661336363       -0.00915222374159775      -3.39768497806275      1.46916288522359      -0.268763657411095      0.0173232440595533       1.98943944960988       -0.772025551743955      0.121200043175719       -0.00713548544554314       -0.362371548342261      0.127174367331994       -0.0174049419059874       0.000907401018253535      0.3469 0.4286       0.55137    0.53465 0.42325 0.40000 5   7   16   36
1.85       -11.3565804841774      12.4498250771054       -7.11887146800487      2.24087104272805        -0.370934584333346       0.0305715297738476         -0.000994267517406146      -0.0186450370867856      -0.512060703810163      0.133808849305079      -0.00903402325431378      -3.47949349501701      1.46999900018757      -0.263360574164293      0.016782840828388        2.02542001700028       -0.765502628760823      0.116875707537619       -0.00676086099032189       -0.367501838953298      0.125171274610374       -0.0164826252224252       0.000832437727328154      0.3448 0.4276       0.54931    0.53440 0.42200 0.40000 5   7   16   36
1.9        -11.3866551392288      12.420511812446        -7.07647688657865      2.21985431328561        -0.366205375975999       0.0300799159977509         -0.000975035491432875      0.048437447787294        -0.528490543403606      0.13405373581059       -0.00893332291167883      -3.56652219761784      1.47199589653691      -0.257974017228695      0.0162365237981807       2.06256609652599       -0.758965074652872      0.112483756157234       -0.00638033619550008       -0.372704603132533      0.123125365496202       -0.0155458963408271       0.000756498816422018      0.3435 0.4266       0.54775    0.53416 0.42079 0.40000 5   7   16   36
1.95       -11.4094681646761      12.3764486014874       -7.02449833957993      2.19601888170733        -0.361056269881429       0.0295573328626014         -0.000954904523741714      0.124255053032286        -0.548220600602145      0.134694091030571      -0.00884735158865774      -3.65739506426897      1.47427157596644      -0.2524207338958        0.0156721213939992       2.09978124958472       -0.751673140680373      0.107866882179624       -0.00598337216392213       -0.377725364883556      0.120862094405582       -0.0145574054943464       0.000677077032917168      0.3422 0.4257       0.54621    0.53392 0.41961 0.40000 5   7   16   36
2          -11.4104083726402      12.2959340041708       -6.95015081226028      2.16553448721745        -0.354869121668454       0.0289525841567947         -0.000932167134727324      0.213692745065788        -0.574867010603279      0.136497500486239      -0.00882584102534638      -3.76235482569529      1.48424218900983      -0.248265326043622      0.0151908215158811       2.14380991921642       -0.748468780450556      0.104044499755378       -0.00563588586138726       -0.383947748622917      0.119369736068187       -0.013724939774648        0.000607607446643715      0.3412 0.4248       0.54482    0.53369 0.41845 0.40000 5   7   16   36
2.5        -10.9748525869435      10.7200918460498       -5.73823357669001      1.72788915506767        -0.273714102312352       0.0215126354343333         -0.000665171874042226      1.43274271347114         -0.981886014242371      0.174915452551358      -0.00959949697481199      -5.01086773847599      1.66956163580764      -0.21998524639139       0.0110972574413553       2.64313490363654       -0.74906642193535       0.0722304477816063      -0.00257433851906472       -0.453345423541743      0.10999693945859        -0.0066924192189494       2.9929806734529e-06       0.3321 0.4166       0.53278    0.53166 0.40830 0.40000 5   7   16   36
3          -10.0426500597687      8.38473513473738       -4.10122715143259      1.18070917665489        -0.178565863121727       0.0132319103160701         -0.000380241725688191      2.8378309031869          -1.48233084245619       0.229326854165008      -0.0112874183253203       -6.33134422902414      1.93779138539259      -0.213335054210633      0.00858410282278933      3.1659168305865        -0.80523754049732       0.0563184519617396      -0.00069866531412869       -0.527913248520498      0.113303779717436       -0.00324722696850752      -0.00033732669351653      0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
3.5        -8.98782221978901      5.9775989188725        -2.47257166588478      0.653242245423418       -0.0894382402799798      0.00567145742688805        -0.00012597016946599       4.10651953296154         -1.94359349832809       0.281713770032626      -0.0130800319757917       -7.48485317242467      2.2117488248719       -0.220528294244496      0.00741647302027329      3.62995124773412       -0.893692564244345      0.0542976032769123      4.1225850974368e-05        -0.596715593695166      0.126183185498724       -0.00307451774374699      -0.000426058915575056     0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
4          -7.99082880220222      3.80320234843554       -1.03341608960612      0.196374890306585       -0.0136931069926708      -0.000638156709208547      8.25810719170418e-05       5.13736108674829         -2.32457390508651       0.32642140819448       -0.0147129360155303       -8.40811712966002      2.46472186592868      -0.237138039738006      0.00732797404209495      4.01190806921891       -1.00005738642651       0.0629969648176079      -0.000131965244391872      -0.65582669542103       0.145447700205707       -0.00539994028386364      -0.000319005761990151     0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
5          -6.41548938025774      0.496897777719119      1.09539620538868       -0.461871541761616      0.0924639531548771       -0.00922594336376245       0.000357879687198602       6.44889680883604         -2.82196834458942       0.387743212441718      -0.0171591286208272       -9.54734844498387      2.84574426514523      -0.280051443402318      0.00897879343139589      4.51171515955593       -1.21380651872628       0.0973595612887599      -0.00208894128940711       -0.739677040942849      0.190900770699287       -0.0141327668912969       0.000233202628044862      0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
6          -5.46005368562988      -1.46338733862657      2.30722105921999       -0.820407032307081      0.147303323872222        -0.0133870978682755        0.000481503128776463       7.0089812074946          -3.06040710057          0.422345729820926      -0.0188645307836445       -9.99795766248519      3.09512648069959      -0.328210151679156      0.0120275689532275       4.74568630562725       -1.40783343214948       0.141847719324491       -0.00511047259024386       -0.786338899339847      0.236832319629399       -0.0250674841196594       0.000981159619176709      0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
7          -5.04710840980126      -2.34299076084136      2.81080290777983       -0.954435812070674      0.164737834448179        -0.0144014435039904        0.000499759527300183       7.09844760433372         -3.13956847443155       0.4408311798391        -0.0201355180233623       -10.0220586512196      3.25233973330443      -0.375615694221094      0.0155978951155117       4.80477949575464       -1.5706181456209        0.187068174427546       -0.00836823976877248       -0.806164683284062      0.277554423827534       -0.0358742951436459       0.00174078455298868       0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
8          -5.05887349849838      -2.42229367141502      2.80460475213018       -0.930608396463293      0.15645126762248         -0.0132988833334388        0.000448656496196594       6.90760243551623         -3.1228943554662        0.449261867030979      -0.0211036254614461       -9.78978280026697      3.34131060456186      -0.418237184260972      0.0191329398555369       4.74990023448993       -1.69780055875101       0.227994214706586       -0.0114159984838179        -0.806804607932295      0.310669747637311       -0.0454193985878918       0.00242210479676379       0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
10         -5.8007586747662       -1.2658468980392       1.97581558699183       -0.63653441383809       0.100857640438812        -0.00798460886972265       0.000248186696977229       6.18996962135453         -2.98256016581481       0.456061705068717      -0.0227999670958581       -9.03305398501815      3.42758128117858      -0.492457093216468      0.0256686238132356       4.49192180764796       -1.87719579553943       0.29499856370269        -0.016517434147936         -0.779372881787741      0.357842010752413       -0.0602692635867201       0.00349158214771011       0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
    """)

COEFFS_ALPINE_20Bars = CoeffsTable(sa_damping=5, table="""\
IMT     a1                  a2                  a3                  a4                  a5                      a6                      a7                      a8                  a9                  a10                 a11                     a12                 a13                 a14                 a15                     a16                 a17                 a18                 a19                     a20                 a21                     a22                     a23                     tau    mean_phi_ss  sigma_tot  phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv     -7.66350339621451   6.41539628193894    -3.38616039536562   1.16182309697538    -0.215624169489156      0.019774355974069       -0.000706006936127713   3.24530765616474    -2.01122873806628   0.362014116033285   -0.020569882347644      -5.18409334741816   1.98208234347632    -0.305309968337773  0.0174121777586387      2.39251147399043    -0.850925188550457  0.113959321163703   -0.0058130325625725     -0.405040599567112  0.135620488327894       -0.0158029615017966     0.000664906503149209    0.5010 0.4600       0.68015    0.00000 0.00000 0.00000 5   7   16   36
pga     -4.96953455014154   5.82067856086996    -3.2768350343017    1.16845091842966    -0.221909256870022      0.0206313522210434      -0.000743744634241191   2.183074165704      -1.4544201038088    0.266864563389985   -0.0153455311503787     -3.7215537295329    1.16670178082041    -0.15909964640612   0.00906634157168612     1.61640823053804    -0.420972296823298  0.0347156149165072  -0.00116965482880633    -0.281401417342575  0.0647397586933695      -0.0025715860353975     -0.000124219357349644   0.3532 0.4600       0.57998    0.58000 0.47000 0.35000 5   7   16   36
0.01    -5.38472107478477   6.27865816836232    -3.49758393788734   1.22836513861202    -0.231132357571168      0.0213731577149176      -0.000767842788439632   3.1470210411851     -2.00033378315102   0.365514938840523   -0.0210731684824731     -5.054871729838     1.94031417087879    -0.302728655662501  0.0176181353030162      2.311310309608      -0.833237374724108  0.112398281704809   -0.00585137032022773    -0.404084487063766  0.138684018845361       -0.0166351193300232     0.000729657881851781    0.3529 0.4600       0.57975    0.58000 0.47000 0.35000 5   7   16   36
0.02    -4.34922626047625   5.24186991630727    -2.97300471572712   1.08349961532049    -0.208788683825398      0.0195753419617709      -0.000709419908493187   2.37834311987479    -1.49047745071823   0.264594391742451   -0.014897951603392      -3.73899952430656   1.08410225031794    -0.134003626512102  0.00723698927730051     1.44566544364068    -0.309368743983153  0.00976907807581136 0.000517322646502146    -0.228853810058319  0.037053651748871       0.00323865983490129     -0.000511828489996419   0.3674 0.4570       0.58639    0.56796 0.46097 0.37408 5   7   16   36
0.03    -3.8340571804484    5.07494875910598    -2.95341667252163   1.08483384448463    -0.209423601416374      0.0196526818211164      -0.000712994316226537   0.997401068932584   -0.969864951562606  0.199778379413602   -0.0122812805813303     -1.78749735561161   0.417553139269728   -0.0550415754487681 0.00410889740163263     0.435718614846251   0.00276467006123326 -0.0254430768897155 0.00185530394116008     -0.0619615773784126 -0.00991328495213778    0.00815451992065866     -0.000679481131013462   0.3749 0.4552       0.58975    0.56092 0.45569 0.38817 5   7   16   36
0.04    -3.69669963091019   5.07053683192196    -2.95404144649123   1.08329690032042    -0.209001817648113      0.0196164181755743      -0.000711947229889393   0.882328854526999   -1.11528169526599   0.249051661264175   -0.0160367796057983     -1.96819884863107   0.843248345483709   -0.163938032556538  0.011695708785071       0.747208641298988   -0.332976577051552  0.0509837848823974  -0.00326742883286379    -0.153799360261242  0.0657964589019142      -0.00808356495510902    0.000387088577090506    0.3801 0.4540       0.59211    0.55592 0.45194 0.39816 5   7   16   36
0.05    -4.08175535084334   5.52309388070958    -3.18262366497234   1.1471148153447     -0.218944916465924      0.0204234211530826      -0.000738438007462351   1.37359309412403    -1.31943907833756   0.276020213724502   -0.0171811562493494     -2.75901015024493   1.1577906306763     -0.204438299749298  0.0134312611625328      1.27724015575035    -0.526311556738133  0.075414906438198   -0.00434532773170283    -0.264854591297971  0.102685443095803       -0.0125830346926084     0.000584194905859436    0.3855 0.4530       0.59483    0.55204 0.44903 0.40592 5   7   16   36
0.1     -5.90538350206182   6.80810846903265    -3.62371822171338   1.24094992522167    -0.231094184022679      0.02127936884919        -0.000763160121503506   4.82326726465243    -2.7532176019984    0.47866377905324    -0.026732319927236      -6.99572745231078   2.82584978695034    -0.441891715386816  0.0249454298441913      3.3579773418044     -1.26591114765081   0.17974893628121    -0.009476451357461      -0.592635205002267  0.205175255244359       -0.0267726064060531     0.00128426897582265     0.3864 0.4500       0.59312    0.54000 0.44000 0.43000 5   7   16   36
0.15    -6.47348527478311   7.03482847911494    -3.63625777477542   1.22864707398199    -0.22772057947311       0.0209317299876861      -0.000750137509158155   4.7412140170544     -2.69266446303801   0.467494317658716   -0.0260657401000663     -6.67995985574002   2.70076926832881    -0.428864060497327  0.0245177065748849      3.11612233687552    -1.17567689818766   0.171697333270957   -0.00929002567216396    -0.530801466127175  0.18498036036209        -0.0251738643014698     0.00125919794399208     0.3841 0.4675       0.60507    0.58095 0.47510 0.40075 5   7   16   36
0.2     -6.71455446718371   6.88753878990417    -3.47403725081481   1.17397059096678    -0.218739949777196      0.020197934608534       -0.000726246851345664   4.62512021816788    -2.65834685836788   0.464890271189291   -0.0260390245992036     -6.43432300711359   2.65896249898655    -0.431438862258635  0.0250074527251344      2.9570590412129     -1.14770495006212   0.172883345261417   -0.00956468742553114    -0.494691238627515  0.179038974949694       -0.0253325393756172     0.00130793196288049     0.3690 0.4800       0.60546    0.61000 0.50000 0.38000 5   7   16   36
0.25    -6.81701477137238   6.637979138563      -3.29301969112801   1.12225790700625    -0.211190945473602      0.0196365668687324      -0.000709333954042109   4.52365619732153    -2.62991272575566   0.462950127124806   -0.0260404582616264     -6.24355100538754   2.63114805684931    -0.434389324349527  0.0254510032170292      2.83896269621126    -1.12874667322162   0.173962006107357   -0.00977984249221937    -0.469075775053833  0.174880615353046       -0.0253918379379419     0.00133832790039329     0.3445 0.4800       0.59082    0.62651 0.50000 0.37450 5   7   16   36
0.3     -6.82540937372167   6.34185152031615    -3.11617242193922   1.07736250099439    -0.205252197537676      0.0192309212637603      -0.000697958756618962   4.42466740437931    -2.59927273104707   0.460230083163119   -0.0259858940031683     -6.08015368133781   2.60529950225259    -0.436397748966591  0.025800852629997       2.74303534805463    -1.11251903905397   0.174659680395205   -0.00994797203579315    -0.449186010980504  0.171401362129672       -0.0253873481288177     0.00135996471122388     0.3377 0.4800       0.58688    0.64000 0.50000 0.37000 5   7   16   36
0.35    -6.88547035679688   6.19594636401097    -3.04953912147416   1.06829487079996    -0.205231422638953      0.0193127179375422      -0.000702514260035554   4.30616706113635    -2.55537525921762   0.454921907395547   -0.0257768728789413     -5.91035659231715   2.56634633061626    -0.4351943350962    0.0259396269064089      2.64954010904035    -1.0909216857923    0.173829471520216   -0.0100117246704394     -0.430893688389723  0.167141098530208       -0.0251275324065614     0.00136370129398459     0.3482 0.4740       0.58812    0.62793 0.49396 0.37000 5   7   16   36
0.4     -7.00658226396669   6.19336698099255    -3.08061536104192   1.08942143911056    -0.20999456442314       0.0197743578861152      -0.000719076461407369   4.18331268640283    -2.5090903653125    0.449253417297744   -0.0255522296281197     -5.75768092338404   2.53259729168812    -0.434615514877077  0.0261075345538023      2.57062204577227    -1.07412514443477   0.173636375938745   -0.0101068748402759     -0.416081055419704  0.163947156130547       -0.025012479199376      0.00137469805303937     0.3552 0.4687       0.58811    0.61747 0.48874 0.37000 5   7   16   36
0.45    -7.17928336242308   6.32142461064966    -3.19709435681083   1.13591284014694    -0.218631961946864      0.0205334629717109      -0.000744757905675428   4.0306967028837     -2.44701559877348   0.440921750745562   -0.0251833577924021     -5.59233925850771   2.48938962020257    -0.432260357973057  0.0261766391451555      2.49197515724404    -1.05550130931218   0.17304074221705    -0.0101805656653806     -0.402224290854925  0.160756037577441       -0.0248835423358487     0.00138532153235269     0.3495 0.4641       0.58097    0.60825 0.48413 0.37000 5   7   16   36
0.5     -7.41818700045223   6.59273151116301    -3.40221741722761   1.2079199592301     -0.231090159460805      0.0215822797016816      -0.00077925615273419    3.84848329615285    -2.37031881600166   0.430256497128173   -0.0246930775553318     -5.41528961560006   2.43874108846747    -0.42857519331842   0.0261702105433431      2.4139130153327     -1.03598619301357   0.172217126898999   -0.0102382389340917     -0.389264204034589  0.157700089069777       -0.0247610688283378     0.0013955033050224      0.3548 0.4600       0.58095    0.60000 0.48000 0.37000 5   7   16   36
0.55    -7.68952110504839   6.94980825391982    -3.65972255940834   1.29421022055043    -0.245533871558342      0.0227688464718762      -0.000817542291856766   3.63871038181249    -2.28054956273414   0.417574524362176   -0.024101137906611      -5.23217786755565   2.38502344128157    -0.424499133744833  0.0261512724628862      2.34029517182157    -1.01863208709074   0.171832854206489   -0.0103257042137082     -0.377942874804484  0.155407090288163       -0.0247838410345176     0.00141487968893727     0.3639 0.4586       0.58544    0.59175 0.47587 0.37413 5   7   16   36
0.6     -7.98003130011875   7.36659350693142    -3.9525209677978    1.38946746643059    -0.261104322833172      0.0240234064169944      -0.000857360349011289   3.39858564514933    -2.17585603257697   0.402512883243905   -0.023385750264146      -5.03635675970613   2.32414599977694    -0.419245574135594  0.0260727180668385      2.26688661864794    -1.00090236884621   0.171401456727442   -0.0104138002246373     -0.367402413176737  0.153387624904925       -0.0248583346327942     0.00143785735954104     0.3757 0.4574       0.59187    0.58422 0.47211 0.37789 5   7   16   36
0.65    -8.29323725653567   7.84104259725775    -4.27761571667352   1.49264064237185    -0.277640009916134      0.0253345673532349      -0.000898413673551182   3.13054232152722    -2.05704276872512   0.38511360329938    -0.0225433442116912     -4.82305067245075   2.25176499201552    -0.411703357818733  0.0258500233855415      2.18862784203734    -0.978744048666987  0.169946346052738   -0.0104306381247376     -0.35642018978575   0.150705331081957       -0.024762209063622      0.00144819292520297     0.3831 0.4562       0.59573    0.57729 0.46864 0.38136 5   7   16   36
0.7     -8.62718219476745   8.36239409047746    -4.6272546087356    1.60140449949276    -0.294793406298244      0.0266765747559694      -0.000939943922695082   2.85210317197429    -1.93364314921436   0.367058379051458   -0.0216704287653065     -4.61527512952506   2.18150502708048    -0.404486524958413  0.0256461313661247      2.11817747887447    -0.960116046453437  0.169074258881423   -0.0104808666657732     -0.347367000835608  0.148919024695982       -0.024819795485414      0.00146752801565604     0.3886 0.4551       0.59846    0.57087 0.46544 0.38456 5   7   16   36
0.75    -8.95398776635675   8.88706477630876    -4.97563791547066   1.70811751936165    -0.311353934497105      0.0279513213486612      -0.000978753863181421   2.55240744351254    -1.79865253539596   0.346974655847546   -0.0206829050212431     -4.39226263368924   2.10018429843747    -0.39502069920543   0.0253031249020843      2.04295911908784    -0.937056350954969  0.167227902335664   -0.0104687241347498     -0.337818849778296  0.146501490841149       -0.0247321305169768     0.00147745847768206     0.3852 0.4542       0.59554    0.56490 0.46245 0.38755 5   7   16   36
0.8     -9.25400471123396   9.37956127038294    -5.30104122071031   1.80639442103434    -0.326335169296402      0.0290814116995232      -0.00101240663272197    2.24212271476514    -1.6570076638796    0.325596149300869   -0.0196158446809296     -4.15987463869995   2.00928839943675    -0.38327257150213   0.0248028363053113      1.96338566773087    -0.908633522765398  0.164029196739834   -0.0103601767434112     -0.327487971716444  0.143046039750428       -0.0243784809518495     0.00146814040223484     0.3850 0.4532       0.59468    0.55932 0.45966 0.39034 5   7   16   36
0.85    -9.55106404681347   9.87051705862662    -5.62078396030977   1.90144471564439    -0.340593097591595      0.0301390143509753      -0.00104333947766568    1.93316266968767    -1.51516250090995   0.304060650079765   -0.0185345528430993     -3.9336145242964    1.91800063873719    -0.371009620757707  0.0242554847473573      1.88806991469736    -0.880295257491208  0.160589300800766   -0.0102281094569091     -0.318044216447442  0.13966359470831        -0.023994474062025      0.0014555360714993      0.3811 0.4523       0.59150    0.55407 0.45703 0.39297 5   7   16   36
0.9     -9.82942414845484   10.3296448068551    -5.91617230657417   1.98782743566116    -0.353301597525865      0.0310606165985151      -0.00106959455204054    1.6502653201644     -1.38697077765441   0.284856912888897   -0.0175832364059919     -3.74143122542975   1.84218504269326    -0.361105890281684  0.0238285614858491      1.8300024345554     -0.859445550463661  0.158247226755009   -0.0101501726670353     -0.311588822435097  0.137551168746552       -0.0237946339637017     0.00145192396640513     0.3799 0.4515       0.59006    0.54912 0.45456 0.39544 5   7   16   36
0.95    -10.0867351750072   10.7631198285962    -6.19335420502026   2.0675910672484     -0.36478099545104       0.031870325774435       -0.00109187821495785    1.32920859967593    -1.23414844273082   0.260808858889834   -0.016333247790163      -3.49126469391291   1.72649974176724    -0.343106917372244  0.0228929946850209      1.73979961655521    -0.816244733077451  0.15134207600923    -0.00978446820711025    -0.299250176358428  0.131332145892039       -0.0227563912453754     0.00139541719681515     0.3733 0.4507       0.58525    0.54444 0.45222 0.39778 5   7   16   36
1       -10.3311828274539   11.1678565370315    -6.44772935516779   2.13937421985604    -0.374870494797845      0.032561135006254       -0.00111016603547461    1.0461624192626     -1.10070530464561   0.240001178877321   -0.0152609178619739     -3.28579207146319   1.63149643844598    -0.32832275032246   0.0221246633626351      1.67109209275376    -0.782686646850532  0.145902381053986   -0.00949384856409544    -0.290521486451813  0.126713341742667       -0.0219607697669311     0.00135135344636023     0.3716 0.4500       0.58362    0.54000 0.45000 0.40000 5   7   16   36
1.05    -10.5325442048575   11.5024585147616    -6.65636766286583   2.19658283784266    -0.382534214262881      0.0330488437784112      -0.00112168963054236    0.778132973779715   -0.972554632529517  0.219736348474947   -0.0142024069263657     -3.09003863641705   1.53565552297735    -0.312602391799698  0.0212693589464059      1.60454221932056    -0.746816781271633  0.139610315286601   -0.00913568892990458    -0.281861074153548  0.121497998182711       -0.0209802574035439     0.00129334316334353     0.3691 0.4482       0.58063    0.53956 0.44778 0.40000 5   7   16   36
1.1     -10.7183051390115   11.8065662160393    -6.84257258392744   2.24624259626934    -0.388914401504099      0.0334285964412896      -0.00112963273609356    0.52758840852512    -0.851499829849947  0.200405666054441   -0.0131835792924102     -2.9073210976938    1.44192625279919    -0.296645553023069  0.0203756973387242      1.54210220000488    -0.710463213677427  0.13290864824479    -0.00874110151700551    -0.273655191770953  0.116053297658104       -0.019904215942751      0.00122768330537985     0.3650 0.4465       0.57674    0.53913 0.44566 0.40000 5   7   16   36
1.15    -10.8801306016248   12.066092599937     -6.99818916192476   2.28604997992713    -0.393663672886217      0.0336734211760969      -0.00113314571580363    0.300247597356298   -0.739975398234543  0.182337836480895   -0.0122186688807021     -2.74295368398453   1.35232037238042    -0.280673759526319  0.0194503751353788      1.48573908305161    -0.674191303621265  0.125818206906079   -0.00830767991929362    -0.266148711460701  0.110413952211545       -0.0187228739854491     0.00115307932273487     0.3627 0.4449       0.57402    0.53873 0.44364 0.40000 5   7   16   36
1.2     -11.0152972103246   12.2773375092821    -7.12127395583252   2.31546096284617    -0.396695486681693      0.0337759658149422      -0.00113197122771141    0.0939860389488029  -0.637368809690047  0.16551978574597    -0.0113118501932509     -2.59542037828504   1.26731270564538    -0.264988244823596  0.0185217050867476      1.43539077298796    -0.63886204122994   0.118641015656525   -0.00786039542006982    -0.259446708062091  0.104826438578368       -0.0175115955778421     0.00107548420101026     0.3625 0.4434       0.57267    0.53834 0.44170 0.40000 5   7   16   36
1.25    -11.1059630263915   12.4117973824515    -7.19535452446611   2.32981519026186    -0.39731193142014       0.0336830248087097      -0.00112448103809476    -0.0779100888228613 -0.551094573485921  0.151248832629178   -0.0105355740648399     -2.48026972217091   1.19545933315553    -0.251031483168566  0.017666076330601       1.39812687281272    -0.608238038498304  0.111973799679939   -0.00742815272677894    -0.254639530372839  0.0998449846239381      -0.0163508015744439     0.000998240695007833    0.3631 0.4419       0.57195    0.53797 0.43984 0.40000 5   7   16   36
1.3     -11.2112565710828   12.5619418179197    -7.27577944359384   2.3456237472637     -0.398137365307471      0.0336074405608199      -0.00111760764118228    -0.245393684145968  -0.463236371453104  0.13619538631638    -0.00969362289849217    -2.357598073508     1.11311941221724    -0.234470294615162  0.0166321295703377      1.353652694939      -0.570386120362008  0.103655602891498   -0.00688743362829945    -0.248244392587307  0.093437665080709       -0.0148769980892622     0.000900882405705021    0.3610 0.4404       0.56949    0.53761 0.43806 0.40000 5   7   16   36
1.35    -11.2573196291457   12.6153337484491    -7.29649332796818   2.34339587758626    -0.39609204285332       0.0333001804579531      -0.0011032519610969     -0.384581891546293  -0.388416347067025  0.12311932418824    -0.00895039516431955    -2.26062782767335   1.04093843493866    -0.219218740633455  0.0156529695285103      1.31946851073829    -0.536245748813507  0.0957596448374469  -0.00636130338474328    -0.243354321627041  0.0875422874916109      -0.013456432604508      0.000805040789967484    0.3605 0.4391       0.56808    0.53727 0.43634 0.40000 5   7   16   36
1.4     -11.322413399174    12.6855071255058    -7.32329727780871   2.3425633351025     -0.394265353943352      0.0330128673303335      -0.00108964921810781    -0.491910674438289  -0.328306748213914  0.112297980991977   -0.00832179064033565    -2.19282633631991   0.980440701315094   -0.205525852499629  0.0147431098598575      1.29690891336875    -0.506309095810241  0.0883578290286366  -0.00585388729374094    -0.240145531908765  0.082206554561282       -0.0120944150588404     0.000711031671441893    0.3616 0.4377       0.56779    0.53694 0.43469 0.40000 5   7   16   36
1.45    -11.3481513971929   12.6906586668563    -7.30994524853657   2.3296796043878     -0.390539727921711      0.0325741064220209      -0.00107121685942365    -0.581660479161115  -0.275056500225838  0.102329120274305   -0.00772651410007367    -2.13691523500563   0.922439470397428   -0.191789196703085  0.0138113366871882      1.27769518871982    -0.476298238072984  0.0807206729910838  -0.00532442234443603    -0.237249589788844  0.0767398256823465      -0.0106752306793727     0.000612469464711475    0.3604 0.4365       0.56604    0.53662 0.43309 0.40000 5   7   16   36
1.5     -11.3554073940229   12.661122988472     -7.27411325157252   2.30998986163049    -0.385752157766402      0.032052245699172       -0.0010501954677833     -0.653057580813345  -0.229064473795677  0.0932895362061374  -0.00716961784396311    -2.09317292981489   0.86695282276517    -0.177999003610256  0.0128565299181008      1.26167715520994    -0.44599306796696   0.0727896121978386  -0.00476847590337283    -0.234601956831561  0.0710732319890787      -0.00918152697625973    0.000508070566782258    0.3559 0.4352       0.56223    0.53631 0.43155 0.40000 5   7   16   36
1.55    -11.3472800806612   12.6001301476985    -7.21744867316603   2.28397599954667    -0.379981518514921      0.0314538849349613      -0.00102680334429224    -0.693973405224857  -0.198122491035763  0.086707161188797   -0.00674503437241412    -2.08151108898749   0.827124371240383   -0.166783316855078  0.0120429430976345      1.26035759303665    -0.423069988761021  0.066112364633012   -0.00428352455090233    -0.234365817722369  0.0666581659120124      -0.00790712193686303    0.000416413434626705    0.3507 0.4340       0.55804    0.53601 0.43005 0.40000 5   7   16   36
1.6     -11.3217352944814   12.5074482774199    -7.14069735923681   2.25198376973435    -0.373291263010065      0.0307844557171106      -0.00100122222412749    -0.721100556258798  -0.171998674148207  0.0806202356689261  -0.00633339191430672    -2.07702864918078   0.787445498659355   -0.155131934977106  0.011185413042941       1.26005875224185    -0.398917536573258  0.0590054713372423  -0.0037654239795315     -0.234048694189808  0.061914092169932       -0.00654173819958977    0.000318246170105581    0.3479 0.4329       0.55537    0.53572 0.42861 0.40000 5   7   16   36
1.65    -11.2780703658821   12.3820049391407    -7.04341074948109   2.21395175522434    -0.365684004371138      0.0300451858483512      -0.000973524256159587   -0.730293624630151  -0.153533270030559  0.0756213889986456  -0.00597328824714134    -2.08656046073285   0.752586030441434   -0.144007142002873  0.0103458941352173      1.26460736738613    -0.376139640565216  0.0520008367160787  -0.00324814345802234    -0.234351514079361  0.0573171907436992      -0.00518211568630196    0.000219712134743968    0.3486 0.4318       0.55495    0.53544 0.42721 0.40000 5   7   16   36
1.7     -11.2303317225837   12.2463022602122    -6.93917994374095   2.17394134919288    -0.357804671249283      0.0292883194802539      -0.000945410713607765   -0.730174552177727  -0.13769179530133   0.0707730240204124  -0.00560879602780385    -2.09916995922545   0.716217207164027   -0.132234572712645  0.00945426525194131     1.26846437163074    -0.351525156424386  0.0445005993483922  -0.00269580102188586    -0.234299120434789  0.0522976413211714      -0.00372130854193048    0.000114342157680069    0.3485 0.4307       0.55401    0.53517 0.42585 0.40000 5   7   16   36
1.75    -11.1708220066922   12.0838203624169    -6.81699556937433   2.1285254232695     -0.349096308567589      0.0284678164493413      -0.000915347981221257   -0.691582295615506  -0.14257682700679   0.0695458359589712  -0.00544952626083388    -2.15715134573969   0.705060375812655   -0.125006065851667  0.00882752081001752     1.29471717013016    -0.339803075306072  0.0393881853989919  -0.00228554963724727    -0.238112253819236  0.0495654196111606      -0.00269237654476671    3.50120588419197e-05    0.3470 0.4296       0.55226    0.53491 0.42453 0.40000 5   7   16   36
1.8     -11.0694007341655   11.8570698767738    -6.65672615712302   2.07187478573104    -0.338621878033761      0.0275054594814258      -0.000880700673535871   -0.653888934196751  -0.144295622428505  0.067415943584912   -0.00522418258444617    -2.20400191019513   0.684108998751664   -0.115583592076898  0.00805632069022355     1.3127096764013     -0.321738051872631  0.0329266892333052  -0.0017888496250944     -0.240225408661454  0.0456005735946044      -0.00140798216913804    -6.04430432752079e-05   0.3469 0.4286       0.55137    0.53465 0.42325 0.40000 5   7   16   36
1.85    -11.000606177756    11.6731417747062    -6.52088209394779   2.02278391849404    -0.329442583258059      0.026657594015256       -0.000850112691804232   -0.589846785691927  -0.159347978317199  0.0675117790429162  -0.00512097944094412    -2.28160080096007   0.679495583772706   -0.10903433436761   0.0074504433266383      1.34597184173704    -0.312228892557386  0.0280337634985067  -0.00138514414839025    -0.244993511789908  0.0431825877809075      -0.000414804231662214   -0.000138350498112247   0.3448 0.4276       0.54931    0.53440 0.42200 0.40000 5   7   16   36
1.9     -10.9219391423002   11.4708233192548    -6.373357576645     1.97024687240808    -0.319739373227607      0.0257696604254752      -0.000818298471008324   -0.516956180431708  -0.177936092468906  0.068071160509383   -0.00503765558846705    -2.36367114583276   0.675837574706839   -0.102480468592576  0.00683750196867879     1.37983399679527    -0.30245426379959   0.023024308870584   -0.000971997074260515   -0.249698495388614  0.0406481635216345      0.000608817484356686    -0.000218410316681783   0.3435 0.4266       0.54775    0.53416 0.42079 0.40000 5   7   16   36
1.95    -10.8081988752134   11.2160495593496    -6.19575498021981   1.90918446260998    -0.308756304023205      0.0247840752040148      -0.000783500107767623   -0.440613362601142  -0.196908377976003  0.0685104809680007  -0.00493975047561805    -2.44421879697093   0.669775421140434   -0.0953241654733106 0.00618352096954312     1.41163364207682    -0.29100919836861   0.0176648626546182  -0.000537099316285367   -0.253948585404107  0.0378115588525523      0.00168946679175163     -0.000301757716863323   0.3422 0.4257       0.54621    0.53392 0.41961 0.40000 5   7   16   36
2       -10.7076029772115   10.9774006877941    -6.02677763614829   1.85068476549844    -0.298199786889239      0.0238352009900597      -0.000749966346286751   -0.357037880192297  -0.218305701192991  0.0691862207380322  -0.00484743381686633    -2.52870613125684   0.664297318846124   -0.0880921129444614 0.00551846773094128     1.44417665556358    -0.279378887619697  0.0122074603504111  -9.42009115575505e-05   -0.25819956527626   0.034895510972193       0.00279304696058511     -0.000386761426164905   0.3412 0.4248       0.54482    0.53369 0.41845 0.40000 5   7   16   36
2.5     -9.45026085786815   8.1345093033155     -4.07350439813141   1.19993475228528    -0.184923990931828      0.0139595442701294      -0.000409649083080124   0.925159209539385   -0.648008817944398  0.110676548723718   -0.0057787961322325     -3.79691243597903   0.840751326528431   -0.0575740695784329 0.00133363697573875     1.96194467885712    -0.283369659445823  -0.0187089512082201 0.00286456714446472     -0.333218365706428  0.0280497301985865      0.00923243828294749     -0.000944213233998793   0.3321 0.4166       0.53278    0.53166 0.40830 0.40000 5   7   16   36
3       -8.15995277314289   5.26977608818576    -2.15288670562021   0.58161581174392    -0.0810093257617915     0.00519082151994112     -0.0001162194030313     2.42201482819622    -1.19128656173633   0.172079474590876   -0.0078562628887102     -5.236441613596     1.17027154409154    -0.0624161125680909 -0.000454534437633485   2.56587973625794    -0.386711942147059  -0.0253049039948338 0.00414642429184283     -0.425238675601732  0.0420644855420403      0.0105348686896874      -0.00114847992383259    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
3.5     -6.2997801501677    1.8976291582323     -0.0466720680239387 -0.0674645814985814 0.0246665509458405      -0.00350338922455262    0.000168502352875881    2.78275034787416    -1.12132383005355   0.126687420369295   -0.00391825291851743    -5.10186501569665   0.679484245752524   0.0730912999920142  -0.0100963633359311     2.35040899983319    -0.0685328532546098 -0.10401715519702   0.00949414363223699     -0.373366416547249  -0.0177146651513414     0.0245315525785753      -0.00207582324196428    0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
4       -6.17165457636165   1.08741285839474    0.544835964034375   -0.256645123198395  0.0550024661152043      -0.00588124281035571    0.000240878703550891    3.73406557981256    -1.45765008765613   0.163650794160915   -0.00512724183919252    -5.96590917315677   0.908864338394906   0.0590561968411846  -0.0102473739246526     2.7239152650245     -0.178304685972757  -0.0936287392221809 0.00917758126893109     -0.43365601574044   0.00395630821823917     0.021578805891076       -0.00192563458292277    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
5       -5.32246843230992   -1.29873899423697   2.13115259747594    -0.738952723697037  0.129849794705474       -0.0116355064209732     0.000414684916332705    6.37928898944185    -2.72932673146888   0.366612569085875   -0.0158388355857783     -9.08692960178832   2.47495111371966    -0.205158298870909  0.00451860621143634     4.3172039572291     -1.05485939234316   0.0653612138274385  -0.000194990436402515   -0.717433212938562  0.171536514616327       -0.010199894212881      7.49543792660113e-07    0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
6       -5.15536187455355   -2.01803991295076   2.56549509457207    -0.850402498685731  0.142711614497608       -0.0121750247770742     0.000413842277162327    7.05573094286909    -3.04500366421922   0.41628985236607    -0.0184562285519872     -9.76505337153882   2.87676874088953    -0.282523433627954  0.00928592245874677     4.68379630112241    -1.3359823543158    0.125831681540768   -0.0041092379010143     -0.788237198636398  0.233017404958603       -0.0238790275335089     0.000894544358875789    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
7       -5.47455381483621   -1.75600529118084   2.36569367872872    -0.76525054331266   0.12365943890998        -0.0101034737557302     0.000328039496113714    7.21886226840016    -3.17789402066149   0.445717327171495   -0.0204042812839462     -9.93329211700924   3.13453543446889    -0.349149677651825  0.0139687274976092      4.81939480398828    -1.54949972197028   0.179911786157477   -0.00782478997836478    -0.820709476891398  0.281723718449701       -0.0359437621586015     0.00170940391205555     0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
8       -6.06455608889937   -0.943449354266199  1.81697115801755    -0.574746723263349  0.0880486519563544      -0.00672724409555567    0.000201557019825063    7.10073788375401    -3.21420359770589   0.464998404759224   -0.0220474770403571     -9.82441373957094   3.30887571473712    -0.407987495095502  0.0184369281973525      4.82232322873564    -1.71407237602152   0.227108798710474   -0.0111791747718704     -0.829797827545474  0.319872077635461       -0.0461840792343584     0.00241375476287162     0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
10      -7.5717793835432    1.38092657540823    0.291567219456898   -0.0720604290901808 -0.000734361388540064   0.00125720155755716     -8.42157134229904e-05   6.44856220479746    -3.12903826152125   0.483802792451363   -0.0245138277273963     -9.16279362497034   3.46655002881849    -0.495557733628144  0.0257022159744324      4.58765272836436    -1.90789920147259   0.295255507819644   -0.0162298613905411     -0.801763658743867  0.365746440081433       -0.0603144359090216     0.00340897628335253     0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
""")

COEFFS_ALPINE_30Bars = CoeffsTable(sa_damping=5, table="""\
IMT     a1                      a2                  a3                  a4                  a5                      a6                      a7                      a8                  a9                  a10                     a11                     a12                     a13                 a14                 a15                     a16                 a17                 a18                     a19                     a20                 a21                     a22                     a23                     tau    mean_phi_ss  sigma_tot  phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv     -8.02427539247774       6.95012805618901    -3.75859485606588   1.30025931271459    -0.2418855811281        0.0221996823747765      -0.000792809072266703   3.19868519942345    -1.98618086750029   0.358204112044895       -0.0204022896870365     -4.9632962388136        1.83887618493617    -0.279039680378663  0.0159478775375502      2.25051217716853    -0.75577290372324   0.0957144384823565      -0.00475110759772212    -0.376207530646805  0.116015155908963       -0.0119721012821358     0.000437798568856948    0.5010 0.4600       0.68015    0.00000 0.00000 0.00000 5   7   16   36
pga     -5.71741545930115       6.75676066771624    -3.85347395057457   1.36954925069729    -0.259153910709526      0.0240481612157102      -0.000866069258518799   2.94261750607218    -1.89961191267445   0.349792401898739       -0.0202844412353876     -4.66037274926156       1.71693424236026    -0.26374617728095   0.0154741450924714      2.09894555891519    -0.70419642752357   0.0887654627948645      -0.00450324590835412    -0.364540943661492  0.113450550254033       -0.011874554671447      0.000452085680896803    0.3532 0.4600       0.57998    0.58000 0.47000 0.35000 5   7   16   36
0.010   -5.71083559205643       6.75731979416221    -3.85156504662242   1.36907334519732    -0.259111031635484      0.0240460504835565      -0.000866019627488765   3.07725203586482    -1.96020831928967   0.35882189434991        -0.0207350875161727     -4.80522379182051       1.77889554654535    -0.27321035932519   0.0159772064877911      2.15243966019926    -0.727234303295697  0.0921894061323387      -0.00468164406575918    -0.371334599757223  0.116441735367175       -0.0122934108769744     0.000472437200612621    0.3529 0.4600       0.57975    0.58000 0.47000 0.35000 5   7   16   36
0.020   -4.63969611270189       5.64166693853918    -3.27331790966751   1.20745740271403    -0.234076307779456      0.0220307012249414      -0.000800566515229674   2.35367213635134    -1.47371031069622   0.261887649091995       -0.0147828370853365     -3.55682157063845       0.959016170412724   -0.110787977952692  0.00595004690791463     1.32784998707658    -0.2263816559761    -0.00638575933810555    0.00145854351902604     -0.203901053920176  0.019254237207597       0.00679593885804257     -0.000724988812571048   0.3674 0.4570       0.58639    0.56796 0.46097 0.37408 5   7   16   36
0.030   -4.27729545067323       5.68939273877461    -3.37695419327849   1.24570987609375    -0.240750853130992      0.0226172463697655      -0.000821412917887904   1.06754614720689    -1.01854193144986   0.210406918335269       -0.0130051153389086     -1.78006635683252       0.414254012929584   -0.0568113860367364 0.0044027235628418      0.418937923789581   0.0148626527319325  -0.0269559260287373     0.00186461740629993     -0.0551892502961882 -0.0148844070050398     0.00904380909896112     -0.000721772121757168   0.3749 0.4552       0.58975    0.56092 0.45569 0.38817 5   7   16   36
0.040   -4.68368037582253       6.51124347813623    -3.86381309319572   1.38806565438339    -0.26316514516496       0.0244384998745605      -0.000881221764724184   0.712598366366742   -1.01159461079009   0.229982628341454       -0.0149414255092475     -1.56051470206493       0.579475048766373   -0.11399144359522   0.00877492556666643     0.485100413947618   -0.159060239866417  0.0170894695774525      -0.00123477144671196    -0.0983591376874046 0.0284637813577328      -0.000693208073489346   -6.23059459757346e-05   0.3801 0.4540       0.59211    0.55592 0.45194 0.39816 5   7   16   36
0.050   -4.59343486690738       6.21084353514885    -3.63863267660181   1.31552156095610    -0.251222880762898      0.0234487292205822      -0.000848349141394645   1.56601486910759    -1.44477897577728   0.301684488144338       -0.0188374510797589     -2.9050023136542        1.25222879312805    -0.225914570908024  0.0149816340368881      1.32194426782048    -0.553296441128785  0.0820838191958187      -0.00488002720674916    -0.265937309501838  0.102652515708705       -0.0127871580325846     0.000618998596423376    0.3855 0.4530       0.59483    0.55204 0.44903 0.40592 5   7   16   36
0.100   -6.04257779364882       7.05664658260253    -3.86708567277423   1.35443673764491    -0.255464674204503      0.023709105920379       -0.000854860416594679   4.74705945305708    -2.70399112643649   0.469819083770924       -0.0262535853594942     -6.76833102714501       2.67391509540006    -0.413744051633379  0.0233703534991447      3.20752471285721    -1.16311047669565   0.160042135794573       -0.00833580551413354    -0.561852018642519  0.183894945481408       -0.0226219901651123     0.00103989263524328     0.3864 0.4500       0.59312    0.54000 0.44000 0.43000 5   7   16   36
0.150   -6.54144308762078       7.24125967517427    -3.87811949514386   1.34557331685764    -0.252966989829385      0.0234461622747066      -0.000844803938377178   4.66378980805982    -2.6451126182277    0.459122224337057       -0.0256170648187831     -6.4907049268571        2.57830632571541    -0.406567081415845  0.0232880631843014      3.00083202391487    -1.09882714948664   0.157108757065037       -0.00845287327636019    -0.508915274328193  0.170142222427949       -0.0222878420219704     0.00108979684859759     0.3841 0.4675       0.60507    0.58095 0.47510 0.40075 5   7   16   36
0.200   -6.85197424096635       7.24136204962472    -3.81778101340486   1.32313758207459    -0.249198107456556      0.0231330402010278      -0.000834394820326212   4.54376381292065    -2.60885013469424   0.456169748784712       -0.0255690609325818     -6.26072231327573       2.54819151196145    -0.411417334969108  0.0239098986960496      2.85600812425503    -1.08120409000046   0.160333343328086       -0.00884856018152705    -0.476157646727338  0.166612917216759       -0.0229226857947077     0.00116695792011801     0.3690 0.4800       0.60546    0.61000 0.50000 0.38000 5   7   16   36
0.250   -7.05767252206135       7.17240089995735    -3.75014097876256   1.30513634703987    -0.246835498374365      0.0229721716929377      -0.000829787484376383   4.44496074843536    -2.58173481693266   0.454390173662867       -0.0255748357087646     -6.08603513060514       2.53054043740796    -0.416126820252469  0.0244451838606881      2.74915342245547    -1.06954035312757   0.162724336111845       -0.00913529711643253    -0.452832349825982  0.163957479092972       -0.0232566915416931     0.0012126037727712      0.3445 0.4800       0.59082    0.62651 0.50000 0.37450 5   7   16   36
0.300   -7.20174233233851       7.09679426383756    -3.70512637920222   1.29811894648122    -0.246571078750729      0.0229950908333047      -0.000831329125465501   4.32886706675921    -2.54110502961676   0.449796059430038       -0.0254074738898967     -5.90882853570938       2.49730956260131    -0.416718677293394  0.0247048906557284      2.64812247192598    -1.05092789109815   0.16298701930819        -0.00927584728946110    -0.432244663351754  0.160203767735632       -0.0232076851141992     0.00123155571701768     0.3377 0.4800       0.58688    0.64000 0.50000 0.37000 5   7   16   36
0.350   -7.4363292282377        7.21331950277027    -3.78838611780774   1.33093648682696    -0.252707557856716      0.0235353099381938      -0.000849535459457834   4.21531694463280    -2.50003545680412   0.444963744257668       -0.0252231326905225     -5.75190889805845       2.46641657275790    -0.416938342970002  0.0249197968411213      2.56254130753428    -1.03446473707015   0.163105469338432       -0.00939281299058610    -0.415430783488043  0.156922693283012       -0.0231339971204195     0.00124603249450495     0.3482 0.4740       0.58812    0.62793 0.49396 0.37000 5   7   16   36
0.400   -7.71499346502983       7.44294713659890    -3.94851567607577   1.38693901828943    -0.262401872438871      0.0243481628060661      -0.000876045047733955   4.08155527203655    -2.44736369952088   0.438078293787826       -0.0249241619892783     -5.58921228238611       2.42717593483229    -0.41528767446709   0.0250195409156385      2.47956594676561    -1.01566502003018   0.162548156995506       -0.00946596900295946    -0.400000825823982  0.153452301366941       -0.0229743009443869     0.00125464693331053     0.3552 0.4687       0.58811    0.61747 0.48874 0.37000 5   7   16   36
0.450   -8.03333034248985       7.77656660630253    -4.17483097404328   1.46191078195454    -0.274881395754832      0.0253660731341555      -0.000908571424657285   3.92792271697929    -2.38466855614476   0.429573678298645       -0.024540290277664      -5.42246456243612       2.38294431821594    -0.412550722583803  0.025050533957135       2.39998357917176    -0.996389179151563  0.161727055838631       -0.00951720017727872    -0.38592765818900   0.150115726966284       -0.0228005713466817     0.00126096524922356     0.3495 0.4641       0.58097    0.60825 0.48413 0.37000 5   7   16   36
0.500   -8.39473388252808       8.21203764973555    -4.46262330681554   1.55377908090103    -0.289751226907948      0.0265537969116069      -0.000945899895458743   3.75363565839936    -2.31341759131976   0.419999373365328       -0.0241181615781144     -5.25266185397248       2.33716741927757    -0.409746402447437  0.0250929251045552      2.32467652280641    -0.978773450323259  0.16124198333900        -0.00959305710609669    -0.373312081010485  0.147308508635377       -0.022722963521841      0.00127355401052293     0.3548 0.4600       0.58095    0.60000 0.48000 0.37000 5   7   16   36
0.550   -8.75045542616889       8.67766848577156    -4.77065189765352   1.65052664625226    -0.305133371752358      0.0277616447971022      -0.00098326450304783    3.54051500013547    -2.22117450178391   0.40672404725114        -0.0234824929751554     -5.06597813020464       2.28094032487379    -0.405055598352458  0.0250281854583311      2.24975399458058    -0.960624626464086  0.160677984489713       -0.00966775429793986    -0.361816862467849  0.144931498854217       -0.0227310292410707     0.00129209553915484     0.3639 0.4586       0.58544    0.59175 0.47587 0.37413 5   7   16   36
0.600   -9.11899507890131       9.18873955605806    -5.10384418488345   1.75301066646268    -0.321122925782402      0.028996089822312       -0.00102085141429032    3.30351523833755    -2.1186716778242    0.392053218489949       -0.0227872310437684     -4.8686849158065        2.21880546936868    -0.399377664695816  0.0249106785137534      2.17414033902496    -0.941224220662437  0.15981650743447        -0.00972194295157465    -0.350714389448445  0.14249679334346        -0.0227058773731834     0.00130759712978891     0.3757 0.4574       0.59187    0.58422 0.47211 0.37789 5   7   16   36
0.650   -9.47688012550723       9.70821112832477    -5.44065713007961   1.85500972273515    -0.33676193806499       0.0301818554116453      -0.00105628921557600    3.03211793512761    -1.99797310911535   0.374254746628619       -0.0219167262828727     -4.6446430314859        2.13935846002737    -0.390315732704789  0.0245849201498543      2.08891492060811    -0.914448240830092  0.157388640420577       -0.00967419205187993    -0.338359360184437  0.138905901423238       -0.0224215462050239     0.00130563743494541     0.3831 0.4562       0.59573    0.57729 0.46864 0.38136 5   7   16   36
0.700   -9.78069491282943       10.1670932816676    -5.73997629703105   1.94435982410571    -0.350134654732821      0.0311649978232317      -0.00108460947775804    2.73036795482804    -1.86151560886694   0.353821867398047       -0.0209040781848768     -4.39740679889645       2.04543242232215    -0.378588804883284  0.024108330017862       1.99620079599362    -0.882256859111524  0.15392701821613        -0.00956852393136056    -0.325196183564614  0.13459816090547        -0.0219988526262609     0.00129623308274609     0.3886 0.4551       0.59846    0.57087 0.46544 0.38456 5   7   16   36
0.750   -10.1203705768869       10.6865857481750    -6.0703355599030    2.04133375866764    -0.364508052448878      0.0322163292512665      -0.00111483687992661    2.43786948011982    -1.73162169935194   0.334743215858317       -0.0199765362233111     -4.17585254694978       1.96492830459380    -0.369187064979889  0.0237617876987759      1.92003559669474    -0.858416600374694  0.151873030168685       -0.00953941064118147    -0.315300221261952  0.131916363356259       -0.0218496567142038     0.00130160213437891     0.3852 0.4542       0.59554    0.56490 0.46245 0.38755 5   7   16   36
0.800   -10.4080548435039       11.1388337459855    -6.35809106639097   2.12429384472843    -0.376450264020433      0.0330562889188509      -0.00113780123252082    2.12490103782926    -1.58936755594474   0.313338208461504       -0.0189103418685183     -3.93119717903137       1.86674884055503    -0.3560456508041    0.0231765752721418      1.83253971858591    -0.825114896467477  0.147747385413034       -0.00937559306898765    -0.303460279555002  0.127527995743215       -0.0213222283564916     0.00128221463980395     0.3850 0.4532       0.59468    0.55932 0.45966 0.39034 5   7   16   36
0.850   -10.6558674075639       11.5376905455436    -6.61115981453909   2.19580402577375    -0.386421678229184      0.0337271421389132      -0.00115504456389217    1.80174666840012    -1.4402270269588    0.290545329121773       -0.0177572974670732     -3.67431964527964       1.75700496053134    -0.34026018174012   0.0224151938949176      1.73887923409652    -0.785392749944269  0.142113342393767       -0.00910985803301300    -0.29055072641816   0.121968732073076       -0.0205177146642133     0.00124400771213266     0.3811 0.4523       0.59150    0.55407 0.45703 0.39297 5   7   16   36
0.900   -10.8822850710592       11.9027851633770    -6.83948693697689   2.25870100410257    -0.394873307464185      0.0342656548130336      -0.00116775609572560    1.49383332328580    -1.29816072272918   0.268847200823248       -0.0166605884473005     -3.43639867612401       1.65341539008129    -0.325080091976394  0.021669987952284       1.65465225803905    -0.74834019458269   0.136675011868707       -0.00884497106330575    -0.279276180571393  0.116848019656195       -0.0197430655683739     0.00120576977991479     0.3799 0.4515       0.59006    0.54912 0.45456 0.39544 5   7   16   36
0.950   -11.0694708120648       12.2088468406194    -7.02939497484454   2.30926917656557    -0.401266533272376      0.0346321599529782      -0.00117477334731837    1.18851334992649    -1.15527556825162   0.246707135980997       -0.0155255353415557     -3.19501523225847       1.54199590854932    -0.307763559421698  0.0207706306648876      1.56658509026110    -0.705626885740806  0.129785648268186       -0.00847785133364734    -0.267095679245845  0.110563286105994       -0.0186791203096275     0.00114738284709277     0.3733 0.4507       0.58525    0.54444 0.45222 0.39778 5   7   16   36
1.000   -11.2434451184206       12.4923597105489    -7.20218238835887   2.35390495338666    -0.406634538423138      0.034911827673615       -0.00117890679713679    0.890006932552654   -1.01353761202877   0.224445815249125       -0.0143701151416791     -2.95501080681294       1.42560355719265    -0.288894923243518  0.0197563675066041      1.47719857199340    -0.658959156736457  0.121826557894659       -0.00803550350643066    -0.254478123102377  0.103458464600572       -0.0174057628558404     0.00107462417417709     0.3716 0.4500       0.58362    0.54000 0.45000 0.40000 5   7   16   36
1.050   -11.3584048948865       12.6802905369744    -7.31432063686905   2.37971002981244    -0.408936109983866      0.0349393494981317      -0.00117476763129077    0.621407770567323   -0.886182836699041  0.204456074519657       -0.0133325666836064     -2.74611389171986       1.32169371944727    -0.271668137599096  0.0188124181831902      1.40173213853728    -0.617398941227737  0.11446395897851        -0.00761447855154983    -0.244095540491207  0.0971356457959069      -0.0162181843334121     0.00100456268773369     0.3691 0.4482       0.58063    0.53956 0.44778 0.40000 5   7   16   36
1.100   -11.4255940393005       12.7946198508571    -7.38055091958408   2.39136329833585    -0.408944107231591      0.0347791858514135      -0.00116450583583393    0.35028699220041    -0.753248466039177  0.182940560507703       -0.012184887352421      -2.52012566671275       1.20046280757053    -0.250389219810771  0.0175959833457513      1.31366156915475    -0.564464453046516  0.104537332117341       -0.00702415613494461    -0.231089623127557  0.0885645920790664      -0.014533655494195      0.000902011196890295    0.3650 0.4465       0.57674    0.53913 0.44566 0.40000 5   7   16   36
1.150   -11.5206024247038       12.9369066417702    -7.45816383372052   2.40580343588156    -0.409389726533091      0.0346579307928644      -0.00115567801171905    0.131894703816798   -0.648019857394751  0.16618426279498        -0.0113040710112039     -2.35607883880774       1.11084239062101    -0.23449013122044   0.0166817453003094      1.25585340469503    -0.527088659000366  0.0972950511960303      -0.00658687078172489    -0.223257099775855  0.0826902827572182      -0.0133222561804548     0.000826826519333490    0.3627 0.4449       0.57402    0.53873 0.44364 0.40000 5   7   16   36
1.200   -11.5548941065035       12.9844695860317    -7.47770905897883   2.40263265808618    -0.407012991004476      0.0343072486478047      -0.00113938967474170    -0.0817474583225913 -0.541325561327077  0.148640336395405       -0.0103554924667595     -2.18394683846976       1.00960699046957    -0.215635362481788  0.0155606809560919      1.19005584416599    -0.481414430035186  0.088094838682862       -0.00601704060358624    -0.213617135139389  0.0751278375793067      -0.0117255356561531     0.000725951306095893    0.3625 0.4434       0.57267    0.53834 0.44170 0.40000 5   7   16   36
1.250   -11.592469967071        13.0290372368033    -7.49267363015852   2.39789307796439    -0.404415501796335      0.0339436385413351      -0.00112288293407109    -0.272560675420583  -0.444115518187943  0.132396769556349       -0.00946573423990161    -2.02880222721752       0.912488807174514   -0.196900297836342  0.0144225902190489      1.12952931916187    -0.435908527137329  0.0786114593840091      -0.00541886176190463    -0.204534050029145  0.0674030070140149      -0.0100473358875785     0.000618371652404864    0.3631 0.4419       0.57195    0.53797 0.43984 0.40000 5   7   16   36
1.300   -11.5725019514513       12.9791366316436    -7.44913986192573   2.37544563118754    -0.398992746833766      0.0333512372170827      -0.00109895211005593    -0.433800051310443  -0.360131757209832  0.118095284423152       -0.00866994033514574    -1.90206648170764       0.826380531343479   -0.179556175928009  0.0133419694433029      1.08114961612586    -0.394858420720693  0.0696582555412802      -0.00484120908605961    -0.19735081722999   0.0603678588015943      -0.008451663423608      0.000514067853812624    0.3610 0.4404       0.56949    0.53761 0.43806 0.40000 5   7   16   36
1.350   -11.5302334788276       12.8894722015963    -7.38031417075469   2.34546052295269    -0.392415593133094      0.0326705375162723      -0.00107234409119425    -0.576428263655858  -0.28323630641321   0.104656350161021       -0.00790717341045193    -1.78594704523456       0.741054741843817   -0.161750533054265  0.0122110408152638      1.03458196069059    -0.35220955663172   0.0601183743416613      -0.00421801605435710    -0.190108280367659  0.0528609393351267      -0.0067220015306317     0.000400119389741656    0.3605 0.4391       0.56808    0.53727 0.43634 0.40000 5   7   16   36
1.400   -11.4972186274795       12.8007675380219    -7.30841225559352   2.31429342040238    -0.385674101455111      0.0319807550122525      -0.00104560288840011    -0.671524561608451  -0.230452825907509  0.095229878893255       -0.00736306576414993    -1.72355053636253       0.682659132495731   -0.148430771802456  0.0113262693502961      1.01354329490141    -0.322555938533057  0.0527813191607016      -0.00371764376579648    -0.18712688829623   0.0475620829817103      -0.00537312668004972    0.000307693671326449    0.3616 0.4377       0.56779    0.53694 0.43469 0.40000 5   7   16   36
1.450   -11.4195319395337       12.6453213236976    -7.19719813767098   2.27159859026990    -0.377138107010456      0.0311488352026328      -0.00101434830363186    -0.778141421456844  -0.167113235736716  0.0834090125840525      -0.00666048799955851    -1.63189898772543       0.601693400644884   -0.130330804587016  0.0101369822686183      0.973900391614257   -0.279306117028419  0.0426321900774437      -0.00304053315491054    -0.180536004643084  0.0396984164272891      -0.00350090945042907    0.000182634877938307    0.3604 0.4365       0.56604    0.53662 0.43309 0.40000 5   7   16   36
1.500   -11.3367884739166       12.4726781617873    -7.07357167155028   2.22521151455471    -0.368067498226132      0.0302795652640148      -0.000982090166585892   -0.847677910565779  -0.122128227701625  0.074580077142653       -0.00611824371473508    -1.58009850631027       0.540176363984941   -0.115401794410508  0.00911892071123188     0.952624994871338   -0.24519577971659   0.0340067693489926      -0.00244743644104231    -0.176955477031128  0.0333686966530831      -0.00189046229001913    7.2286381798523e-05     0.3559 0.4352       0.56223    0.53631 0.43155 0.40000 5   7   16   36
1.550   -11.2439985544765       12.2807832994331    -6.93787396483302   2.17543148327433    -0.358521360974625      0.0293779897825444      -0.000948993881249096   -0.906123338744701  -0.0800556665202803 0.0658666779284114      -0.00556598169880350    -1.53069615651207       0.475405009768873   -0.0993476305735935 0.00801454723890612     0.929371727706813   -0.207735238333013  0.0245189193371044      -0.00179428533985330    -0.172699127518076  0.0262924425782017      -0.000103638333456848   -4.99128425823447e-05   0.3507 0.4340       0.55804    0.53601 0.43005 0.40000 5   7   16   36
1.600   -11.1318014866681       12.0498146098567    -6.77671451978077   2.11798585008856    -0.347785267892374      0.0283833826200044      -0.00091298582587529    -0.922465733833354  -0.0600736961130719 0.0609333718155644      -0.00522439110947861    -1.53161620120724       0.437748004139518   -0.0880720360166748 0.00718396649641217     0.931237371439902   -0.184285385142414  0.0175702009588207      -0.00128979638001952    -0.172800612181591  0.0217120592535567      0.00122203556875875     -0.000144744388282831   0.3479 0.4329       0.55537    0.53572 0.42861 0.40000 5   7   16   36
1.650   -11.0214980107168       11.8200593065913    -6.61640117439333   2.06112261733413    -0.337218861198859      0.0274096837196516      -0.000877900120689197   -0.940891832092705  -0.035347244051605  0.0547072514786436      -0.00479019937856585    -1.51736567680645       0.386591154281609   -0.0737700500194005 0.00615459995276473     0.92195137395405    -0.152087303148969  0.00874818684839996     -0.00066503996804581    -0.170600636788958  0.0154220598439115      0.00290550438398581     -0.000262270855760089   0.3486 0.4318       0.55495    0.53544 0.42721 0.40000 5   7   16   36
1.700   -10.907872494417        11.5773852448944    -6.44686622293748   2.00160915428188    -0.326286813145398      0.026411990384887       -0.000842220294846059   -0.921757818270309  -0.0308090513691903 0.0520180050342187      -0.00455764449217310    -1.54921605707996       0.361339969263877   -0.0642055241238451 0.00540544694689631     0.936058384531321   -0.133590884031093  0.00251014689424271     -0.00019646010191906    -0.172501945896687  0.0116052762116218      0.00411342730446928     -0.000350709298562498   0.3485 0.4307       0.55401    0.53517 0.42585 0.40000 5   7   16   36
1.750   -10.7319803376794       11.2399510569831    -6.22154466701633   1.92568674000841    -0.312776944887367      0.0252071938534082      -0.000799846359528678   -0.898555706472186  -0.0256317760072361 0.0488338533890521      -0.00428017066933552    -1.57665842175951       0.32979192705051    -0.0530159854467884 0.00454133178058091     0.94550124970628    -0.110672825949459  -0.00475919466264019    0.000341889302772934    -0.173366163537556  0.00691911782233516     0.00551563875727782     -0.000452013689841498   0.3470 0.4296       0.55226    0.53491 0.42453 0.40000 5   7   16   36
1.800   -10.6249943895562       10.9969465257879    -6.05033049900368   1.86610031410167    -0.301965180199354      0.0242319180373565      -0.000765319858475906   -0.840556782058621  -0.0385396165334348 0.0487359058498329      -0.00417503436674963    -1.64504502161967       0.320302587615936   -0.0457402157450039 0.00390386327294929     0.975015904218212   -0.098986928600342  -0.00997293082162629    0.000758331488802763    -0.177654832872234  0.00420434605730123     0.00654941510017446     -0.000531148378246152   0.3469 0.4286       0.55137    0.53465 0.42325 0.40000 5   7   16   36
1.850   -10.4592326493643       10.6673069125058    -5.82886359170623   1.79183451656495    -0.288854832605435      0.0230722484666768      -0.00072483657895736    -0.789988775656595  -0.0449445005285835 0.0471733836222543      -0.00397258042493485    -1.69468327646344       0.297250651413615   -0.0356317137412083 0.00308511169827735     0.992523835706808   -0.0790903135689955 -0.016865686724208      0.00128137270109186     -0.179590827396885  -7.69319572458671e-05   0.00790064420866006     -0.000630400001685647   0.3448 0.4276       0.54931    0.53440 0.42200 0.40000 5   7   16   36
1.900   -10.3067238962655       10.3530635581907    -5.61594171882498   1.72035019893869    -0.276254423078198      0.021960314309983       -0.00068611744563021    -0.719933215533563  -0.060896723354468  0.0471578044226169      -0.00385280495640246    -1.76553086957942       0.285144396590143   -0.0274060594954246 0.00237308841888393     1.0201978369114     -0.0648025476822199 -0.0227387586674418     0.00174405614696672     -0.183270107592257  -0.00334434482949477    0.00906030854619592     -0.000718001210461077   0.3435 0.4266       0.54775    0.53416 0.42079 0.40000 5   7   16   36
1.950   -10.1645494570412       10.0491947757745    -5.40833951486553   1.65061741382296    -0.263989026056281      0.0208810059524684      -0.000648640633344389   -0.627227546633916  -0.0886321423117641 0.049161540993338       -0.00384663886172773    -1.86284191661435       0.287666568688784   -0.0218317708433683 0.00181758147270664     1.06096436149658    -0.0581628766963732 -0.0271707083119142     0.00211933130965729     -0.189216221438487  -0.00523400494234292    0.00995379649451998     -0.000789202496526005   0.3422 0.4257       0.54621    0.53392 0.41961 0.40000 5   7   16   36
2.000   -9.97611421060513       9.68040955595283    -5.16434476425085   1.57054677060604    -0.250146061921281      0.0196783299259515      -0.000607280944531105   -0.551099562638134  -0.105031571044585  0.0488627740496125      -0.00369563907128913    -1.93096691224323       0.271242276192679   -0.0125079877317705 0.00102949520980057     1.08488596352439    -0.0408486186770101 -0.0336950212537157     0.00262400348455074     -0.19203234915282   -0.00908184774047224    0.0112294563009559      -0.000884033007291836   0.3412 0.4248       0.54482    0.53369 0.41845 0.40000 5   7   16   36
2.500   -8.42768976685938       6.39698699875634    -2.96619474281131   0.855004361390159   -0.128198731489864      0.00924567163545499     -0.000253744628553489   0.783860840733186   -0.557708514262289  0.0939144194394868      -0.00481942500132716    -3.24371198820028       0.464193762318793   0.0151650132795478  -0.00296276945830301    1.6332845864263     -0.0599170538364606 -0.0615933736159097     0.00537661686197551     -0.274081463777265  -0.011984640152255      0.0168565883415998      -0.00138694458950623    0.3321 0.4166       0.53278    0.53166 0.40830 0.40000 5   7   16   36
3.000   -6.67657129125055       2.98577953079914    -0.782750141168787  0.173045646049555   -0.0162764437521773     -9.73715423873754e-06   5.04417244373589e-05    1.74939531911913    -0.785733737877618  0.0967027863635196      -0.00343607391013163    -3.89254018133886       0.314222092526139   0.101194609821606   -0.0102130359006909     1.80976708701355    0.0988585327818178  -0.118522967315255      0.00972594838228447     -0.289031769009057  -0.0456006038021656     0.0273970599033041      -0.00216017907343921    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
3.500   -5.41032646051120       0.376974962003832   0.889308233462355   -0.342524277625624  0.0667705897091167      -0.00673185317099637    0.000266416787652285    2.88217951368969    -1.164183115354170  0.1337128136872640      -0.00433460241113247    -4.91492257538025       0.518921362171178   0.105152511984897   -0.0119428499170258     2.24165514403481    0.0205743180816499  -0.121224946924987      0.0104534370756347      -0.357037908846174  -0.0311192625605435     0.0270428076737813      -0.00220975940506026    0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
4.000   -5.39845182223528       -0.206787932742771  1.31881556423355    -0.475183829729772  0.0867926337080162      -0.00816731346663926    0.00030482794059450     3.71380820086058    -1.424715108270970  0.1558717579108710      -0.00463454653477691    -5.6192064869185        0.647352539909196   0.111103194060013   -0.0133441350626458     2.53294404312058    -0.0366863189322946 -0.121526539108522      0.0108258374463756      -0.402733881671848  -0.0188867090574179     0.0260572778979113      -0.00218924460825411    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
5.000   -5.15914894615794       -1.647141645627890  2.31591427793213    -0.77038570381648   0.1298068646518260      -0.01116856598851060    0.00038375777539973     6.51288711652675    -2.794785315365800  0.3779985556987640      -0.0165168450276606     -9.02311789387031       2.40056877023837    -0.190009957321495  0.00368295249483568     4.29583772023421    -1.02563175791926   0.0593792647536806      0.000129648970019664    -0.718691652994647  0.169968165864347       -0.00982833244488065    -1.77421496093564e-05   0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
6.000   -5.47022491862098       -1.618934264392480  2.28776734434393    -0.737118740593868  0.1184911380792100      -0.00965718777940326    0.000313589939786376    7.24529136220998    -3.149330064658320  0.4355023967591250      -0.0196188177161499     -9.81433688537642       2.87982976107876    -0.282358975261132  0.00933638538725177     4.72567300937053    -1.3485754118194    0.127439508336503       -0.00420067174607053    -0.800518703306941  0.238502173336001       -0.0247084127152711     0.000936463850853216    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
7.000   -6.18302725649498       -0.751052389640527  1.72001869327789    -0.539137206565445  0.0809820165405954      -0.00605334372092063    0.000177104586201079    7.45888246619470    -3.318389658047950  0.4723576803070340      -0.0220323045332307     -10.0726855247561       3.19982507196935    -0.361007144047592  0.0147262703793659      4.90578979233542    -1.59122887004735   0.186633702650372       -0.00818396396515737    -0.839906287541049  0.291472476567902       -0.037433815366904      0.001779749521907       0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
8.000   -7.05597524602239       0.496738333754760   0.910073727655276   -0.269988854931764  0.0327933896581766      -0.00165933331679218    1.78904644661442e-05    7.35582128133733    -3.369416613927210  0.4950505738558620      -0.0239028043257329     -9.99270122597201       3.3975372751163     -0.424415855416397  0.01945422182746        4.91666150051552    -1.76143170713736   0.234445225778897       -0.0115379419987368     -0.848952713110261  0.329398657473949       -0.0474807968507647     0.0024620736195773      0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
10.00   -8.84235151039667       3.239490569684570   -0.856827685808916  0.301690757093318   -0.0663008500872744     0.00709035320719849     -0.000289932346434402   6.73530041669423    -3.312402736522330  0.5202150682418290      -0.0267878285244298     -9.3758856483688        3.59045134492446    -0.518849767515115  0.0271096620447494      4.68836659602167    -1.95937224856324   0.302649673470594       -0.01653727522587       -0.819120990468147  0.373666949305236       -0.061075839709767      0.00341029422857993     0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
    """)

COEFFS_ALPINE_50Bars = CoeffsTable(sa_damping=5, table="""\
IMT       a1                a2                  a3                  a4                  a5                      a6                      a7                    a8                  a9                  a10                     a11                  a12                 a13                     a14                     a15                   a16                 a17                     a18                     a19                   a20                     a21                     a22                     a23                   tau    mean_phi_ss  sigma_tot  phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv       -8.46277436251113 7.58065050629491    -4.19018557556284   1.45977608100059    -0.272048410578197      0.02497772151020        -0.000891986208739702 3.14664446436245    -1.9592975957007    0.354372787141493       -0.020253273784605   -4.69519418160372   1.66598857272442        -0.247713965969552      0.0142293556871612    2.0754117766367     -0.638958447184736      0.0735633610789567      -0.00347979535417182  -0.340173689760659      0.0916106317444659      -0.0072555969649928     0.000161968985669108  0.5010 0.4600       0.68015    0.00000 0.00000 0.00000 5   7   16   36
pga       -5.59030023944254 6.69983150227806    -3.94497081548402   1.44388126555530    -0.277862320582453      0.0260418181553515      -0.000943877343705686 2.09974376987949    -1.40237002135934   0.258142389293052       -0.0149233782779983  -3.23552059172217   0.839265093059718       -0.0981504278630207     0.0056447922850299    1.2876465574871     -0.193852385075366      -0.00931719617821042    0.00140558083537031   -0.211220721395361      0.0156464571283417      0.00714865679387383     -0.000705100827621744 0.3532 0.4600       0.57998    0.58000 0.47000 0.35000 5   7   16   36
0.010     -6.15176197785336 7.38980262765841    -4.30548582242746   1.54594367698574    -0.29389281790599       0.0273467383664598      -0.000986721550662611 2.99722623939086    -1.91456081062864   0.351345451750572       -0.0203678083374813  -4.50479289664292   1.58562456535910        -0.238330472117644      0.0140709922308788    1.95836306788451    -0.598440196536173      0.067964058942332       -0.00330270375726144  -0.330645744956306      0.0889527882849623      -0.00699827767935554    0.000163691585164169  0.3529 0.4600       0.57975    0.58000 0.47000 0.35000 5   7   16   36
0.020     -5.87142195844516 7.23612718050162    -4.23467535078321   1.52884885645351    -0.291705824453942      0.0272059746681052      -0.000983317751549238 3.26515667822224    -2.01388240902454   0.363463042975721       -0.0208808120481714  -4.72959962635308   1.66092251198292        -0.246505685826029      0.0143629228958137    1.95272409242359    -0.603415827557399      0.0670310465435614      -0.00313233180102044  -0.313922830509235      0.0858117565920863      -0.00620958266075941    9.27999768354344e-05  0.3674 0.4570       0.58639    0.56796 0.46097 0.37408 5   7   16   36
0.030     -5.19793919377044 6.88772403381954    -4.12086135245999   1.50396543366221    -0.288357229730445      0.0269728296056303      -0.000977187538520265 1.67031393955005    -1.39493089955527   0.283910970087161       -0.0175449240838276  -2.55588835165861   0.905834241956532       -0.156291104901251      0.0107893526118366    0.834679223307674   -0.250030742369549      0.0269035601475856      -0.0016200777935277   -0.126654611499073      0.0306138234127982      -0.000234376417541379   -0.000117446729620898 0.3749 0.4552       0.58975    0.56092 0.45569 0.38817 5   7   16   36
0.040     -4.85977382942357 6.72493765969129    -4.06748064455718   1.49132182927852    -0.286485084210035      0.0268308000500942      -0.000973012229589182 0.735316126906787   -1.03253543875081   0.235552129979204       -0.0153731302414726  -1.43953233488912   0.504730623252513       -0.102466475459001      0.00830155117857926   0.383120329089947   -0.0914044122539554     0.00498395347687        -0.000603994916419358 -0.0714431656388617     0.0101091198564966      0.00277567334669136     -0.000257080941580559 0.3801 0.4540       0.59211    0.55592 0.45194 0.39816 5   7   16   36
0.050     -4.32202933816735 5.79825733782605    -3.4901840992789    1.31695460057417    -0.258580166953316      0.0245533158061608      -0.000898279161840359 1.41910921963051    -1.35016734129851   0.283754060070643       -0.0177874549960828  -2.46298316341662   0.954885422827494       -0.168430634159713      0.0115736181006772    1.01012557818812    -0.339150277695986      0.0396254964736593      -0.00230644499363986  -0.196323591658797      0.0543846315223598      -0.00311175022366477    2.66621467283475e-05  0.3855 0.4530       0.59483    0.55204 0.44903 0.40592 5   7   16   36
0.100     -6.23362598162032 7.39450970316458    -4.18356406089381   1.49840240551216    -0.286000472166993      0.0267318616362644      -0.000968419661353698 4.61790733460305    -2.62188049248342   0.454908418967471       -0.0254276081688672  -6.44438364394086   2.46093968271205        -0.374523388668401      0.0211798211657673    2.99836294288371    -1.02253907233748       0.133404370913635       -0.00680814630383444  -0.519409028173631      0.155027084906124       -0.0170677876139543     0.000716762686645216  0.3864 0.4500       0.59312    0.54000 0.44000 0.43000 5   7   16   36
0.150     -6.72569722944736 7.63857005842899    -4.25492766279285   1.51132126168427    -0.287240590063566      0.0267798844982809      -0.000968486750698322 4.54787933353597    -2.57478550672328   0.446739329024974       -0.0249492762783906  -6.23597255230411   2.41614571298472        -0.377380011121355      0.0216945533415611    2.84767769378978    -0.998463697098774      0.138352110565933       -0.00739287121444065  -0.479873272453606      0.150787184361565       -0.0185880922902852     0.000876347481272189  0.3841 0.4675       0.60507    0.58095 0.47510 0.40075 5   7   16   36
0.200     -7.13083310672755 7.82785870267093    -4.32077981581731   1.52757099341918    -0.289530653841873      0.0269403226518324      -0.000972764495805845 4.42940250260721    -2.5395910228326    0.443917987082447       -0.0249032942217864  -6.0313697114831    2.40308752384893        -0.385276100301228      0.0224787718051157    2.72285654260603    -0.994404894641303      0.144067159286935       -0.00792620970081994  -0.451608582638798      0.150316261465363       -0.0197904616161868     0.000985300125299954  0.3690 0.4800       0.60546    0.61000 0.50000 0.38000 5   7   16   36
0.250     -7.4886604649952  8.01973939979102    -4.41319669917903   1.55608277747474    -0.294161819860974      0.0273072956136670      -0.000984009719689623 4.30504277264229    -2.49737574650623   0.439274208607036       -0.0247347555785191  -5.83786977853706   2.37536008202791        -0.387989731033899      0.0228818107391243    2.61013223433494    -0.980196765870759      0.145957800230184       -0.00817698252461771  -0.427694313467837      0.147522027182972       -0.0200991441794944     0.00102851986073656   0.3445 0.4800       0.59082    0.62651 0.50000 0.37450 5   7   16   36
0.300     -7.86586021161939 8.29897937113879    -4.57270005596752   1.60658005937741    -0.302392857922564      0.0279651679907639      -0.00100452116641346  4.21016934278356    -2.46952376675122   0.437038071555299       -0.0247048624234476  -5.7031966644560    2.36891845675182        -0.393585839172741      0.0234326261847164    2.53487097780736    -0.978182687559968      0.149387066005702       -0.00850489811302877  -0.412053260671306      0.146997210052861       -0.0206749184683137     0.00108488276464702   0.3377 0.4800       0.58688    0.64000 0.50000 0.37000 5   7   16   36
0.350     -8.27297316174424 8.67443162456561    -4.79979952582619   1.67761963123773    -0.313770065912138      0.0288623693737385      -0.00103224249159867  4.07898847941629    -2.41809498969578   0.430214788533876       -0.0243974661596443  -5.52822801592046   2.32752120114216        -0.391648800056119      0.0235041558154843    2.44095842615187    -0.956967323209964      0.14852597044547        -0.00855600120273193  -0.393830250454582      0.14292189646683        -0.0204391635229771     0.00108851600878917   0.3482 0.4740       0.58812    0.62793 0.49396 0.37000 5   7   16   36
0.400     -8.72327434197351 9.14591394820791    -5.08762373387873   1.76589929480222    -0.327643874703594      0.0299399667186165      -0.00106513174158758  3.95014907201539    -2.36826807978444   0.423769589459965       -0.0241177997651913  -5.37299796992257   2.29244740457191        -0.39050663437148       0.0236137514845189    2.3616342670337     -0.940223356637424      0.148209922244303       -0.00863203620510582  -0.378959358170559      0.139766454666123       -0.0203144617516294     0.00109719734812053   0.3552 0.4687       0.58811    0.61747 0.48874 0.37000 5   7   16   36
0.450     -9.16305389842239 9.64104692091518    -5.39179715488176   1.85777202003866    -0.341787381243722      0.031013629124216       -0.00109709325706749  3.79896688603358    -2.30750334903698   0.415621898835496       -0.0237532312819533  -5.21216956972142   2.25269198878663        -0.388598077809407      0.0236911969471573    2.28606498984187    -0.924077949158377      0.148024802690097       -0.00872306717902196  -0.365672804734342      0.137059515548865       -0.0202758532833256     0.0011124486196868    0.3495 0.4641       0.58097    0.60825 0.48413 0.37000 5   7   16   36
0.500     -9.63518565634555 10.2188503884941    -5.74526036231034   1.96292724887029    -0.357778686668624      0.0322173975510343      -0.00113274433837339  3.62143181742110    -2.23394617516556   0.405463976156989       -0.0232855280015122  -5.03708336353475   2.20310352967619        -0.384796564126534      0.0236546490920046    2.2078864214682     -0.904463511532115      0.147035955619311       -0.00875967937914473  -0.352523328274095      0.133892683025506       -0.0201120806837896     0.00111851448999066   0.3548 0.4600       0.58095    0.60000 0.48000 0.37000 5   7   16   36
0.550     -10.0701992189654 10.7732940203905    -6.0849875892882    2.06250584156047    -0.372597750244529      0.0333040606381559      -0.00116395437739441  3.41166391096836    -2.14469149257672   0.392818791157252       -0.0226889711281925  -4.84477852934071   2.14344568240333        -0.379271777962645      0.0235258359028153    2.12726012605108    -0.882575598016428      0.145628723741482       -0.0087741169200272   -0.339681196251456      0.130619920726706       -0.0199235885239283     0.00112337371201697   0.3639 0.4586       0.58544    0.59175 0.47587 0.37413 5   7   16   36
0.600     -10.4711048635462 11.3047882206140    -6.41047433559598   2.15637614298729    -0.386246166810353      0.0342762677547576      -0.00119089774604460  3.16648956155966    -2.03797660908307   0.377376300783553       -0.0219461542715611  -4.62815828915394   2.06961691908570        -0.371249724815253      0.0232579050722626    2.03981411995601    -0.855896356465791      0.143320779381236       -0.00873676042806751  -0.326315299471127      0.126789266809972       -0.0196246920338622     0.00112181981085422   0.3757 0.4574       0.59187    0.58422 0.47211 0.37789 5   7   16   36
0.650     -10.829831632483  11.7945111364228    -6.70979868872991   2.24114890486840    -0.398236810198776      0.0350994879918857      -0.00121261808117907  2.90212893672910    -1.92246424430908   0.360620765004314       -0.021139150109676   -4.40534463384989   1.99125557390961        -0.362365646848085      0.0229402919211523    1.95377071887889    -0.828742310100715      0.140826034617139       -0.00868542770644702  -0.313670753290963      0.123042159268965       -0.0193155463759215     0.00111874232050964   0.3831 0.4562       0.59573    0.57729 0.46864 0.38136 5   7   16   36
0.700     -11.1348594489821 12.2236186667930    -6.97196645004074   2.31368622642435    -0.408095263555775      0.0357370789852424      -0.00122797254330539  2.60856829000896    -1.79231690563847   0.341484393528386       -0.0202058905585525  -4.15796118373308   1.89768479841330        -0.350655927023334      0.0224579682331633    1.85868511132035    -0.795130258726651      0.137055820991892       -0.00855711880872674  -0.299869402448964      0.118334278145969       -0.018810521223052      0.00110370372054163   0.3886 0.4551       0.59846    0.57087 0.46544 0.38456 5   7   16   36
0.750     -11.3895513217964 12.5906671126769    -7.19525394953105   2.37353253802887    -0.415774317955381      0.0361874660952909      -0.00123697691086656  2.29867640395590    -1.65309970035696   0.320737568158873       -0.0191803839668333  -3.89780294308279   1.79331212358396        -0.336600864675309      0.02182423104056      1.75903049047452    -0.756386414559869      0.132074911579398       -0.00834720022507797  -0.285477003154089      0.112768358599261       -0.0180943004770235     0.00107410313857812   0.3852 0.4542       0.59554    0.56490 0.46245 0.38755 5   7   16   36
0.800     -11.614562310713  12.9237076371553    -7.39646377107161   2.42599996084563    -0.422185443792998      0.0365305813166494      -0.00124243229691343  1.97239611297685    -1.50392527850537   0.298119659026602       -0.0180436636443251  -3.61826143037673   1.67332485374323        -0.319206181596071      0.0209766834219343    1.64971507764622    -0.70910353958387       0.125208499074039       -0.00801447232891177  -0.269433361503054      0.105663251993741       -0.0170349121662025     0.00102203625940779   0.3850 0.4532       0.59468    0.55932 0.45966 0.39034 5   7   16   36
0.850     -11.7922959666527 13.1866107204577    -7.55224318671615   2.46395769269073    -0.426184312839939      0.0366726206945376      -0.00124123117940975  1.67121311550783    -1.3694325144637    0.278232180278403       -0.0170693092801637  -3.37674352682709   1.57300532853335        -0.305231474815727      0.0203256231218908    1.56159169697326    -0.672603727733456      0.120207189920758       -0.00778893811774259  -0.257306883523437      0.100548205858251       -0.0163221790108408     0.000990021676730472  0.3811 0.4523       0.59150    0.55407 0.45703 0.39297 5   7   16   36
0.900     -11.8862730821799 13.3351265812086    -7.64008060871288   2.48126575399997    -0.426840481737922      0.036540422088714       -0.00123106118515134  1.33154135681071    -1.20974362435554   0.253365127715384       -0.0157879944592504  -3.07603828104693   1.43077318383403        -0.282658591541305      0.0191331983014934    1.44053516629933    -0.612375895474342      0.110298286148559       -0.00725279236434809  -0.239150948301079      0.0910568591434022      -0.0147025753491856     0.000900516311529247  0.3799 0.4515       0.59006    0.54912 0.45456 0.39544 5   7   16   36
0.950     -11.9597327910817 13.4493006992699    -7.7039760235660    2.49083864275230    -0.426235369025113      0.0363066614739945      -0.00121766685261002  1.01776595293051    -1.06310416700405   0.230669816833013       -0.0146254869698819  -2.80508237843028   1.30098753531700        -0.261869214646342      0.0180276133495103    1.3337491761981     -0.557731635323306      0.101142722022822       -0.00675142420176733  -0.223401831408132      0.0824821960710003      -0.0132052163640666     0.000816612314871904  0.3733 0.4507       0.58525    0.54444 0.45222 0.39778 5   7   16   36
1.000     -11.9581451025538 13.4527366403493    -7.70076778964392   2.48007878019018    -0.422370463145758      0.0358080665966385      -0.00119567589718212  0.702287504344789   -0.912578941304935  0.206919832151055       -0.0133876518111379  -2.52312767272613   1.15861418609039        -0.23807477057429       0.0167195771984643    1.21905913203036    -0.494990465308685      0.0901117682422546      -0.00612543977375978  -0.206038603519443      0.0723346060605477      -0.0113497737665615     0.000709090913816604  0.3716 0.4500       0.58362    0.54000 0.45000 0.40000 5   7   16   36
1.050     -11.9949636379039 13.5021900876592    -7.72009670283548   2.47558402480633    -0.41952793156918       0.0353987372323792      -0.00117685759469461  0.43429884481106    -0.786171896529186  0.187190817933752       -0.012369130028602   -2.29433332750883   1.04110237508363        -0.21819738836183       0.0156169996251846    1.12930799955708    -0.443623460258523      0.0808488551139718      -0.00559160034422158  -0.192808487561045      0.0640621637646565      -0.00978748568079536    0.000616990401418091  0.3691 0.4482       0.58063    0.53956 0.44778 0.40000 5   7   16   36
1.100     -11.9609990119199 13.4447347707227    -7.67500021305939   2.45181310969904    -0.413636284322363      0.0347447994155984      -0.00115019023398465  0.171792261074961   -0.659256023980965  0.166930652056485       -0.0113021222927558  -2.06162683745088   0.914806567211054       -0.195976605966118      0.0143494030169801    1.03478610528668    -0.386006429590891      0.0700532636997054      -0.00495342767852711  -0.17847122459899       0.054538131370436       -0.00792909709468392    0.000505030038209451  0.3650 0.4465       0.57674    0.53913 0.44566 0.40000 5   7   16   36
1.150     -11.8781511923272 13.3095724866418    -7.58199417613571   2.41363708214683    -0.40547307916894       0.033909679697311       -0.00111774161610426  -0.0733247927573355 -0.538712142114445  0.147395090438185       -0.0102596077348849  -1.83988685994651   0.788730366234975       -0.173099759848067      0.0130164159559217    0.942965076515293   -0.326787194264092      0.0586100667931316      -0.00426375160220777  -0.164319904872689      0.0445836256301872      -0.00593283769332104    0.000382740428734748  0.3627 0.4449       0.57402    0.53873 0.44364 0.40000 5   7   16   36
1.200     -11.8021141038793 13.1769231244197    -7.48808615094628   2.37519485530325    -0.39733553895642       0.0330850012674325      -0.00108594666990340  -0.289194480109041  -0.431042165643027  0.129741610880791       -0.0093084536439792  -1.64529605774800   0.672234579734328       -0.151315509418612      0.011723337141148     0.862167600053185   -0.270869068207315      0.0474511842263674      -0.00357956188099987  -0.151785549417831      0.0350472994261978      -0.00396115532152323    0.000260132651743685  0.3625 0.4434       0.57267    0.53834 0.44170 0.40000 5   7   16   36
1.250     -11.6584039096298 12.9374268364130    -7.32936973698429   2.31738855854158    -0.386137634972548      0.032014316954755       -0.00104621529253344  -0.484442992175088  -0.331210002660005  0.113028337453127       -0.00839215390814297 -1.46521876798465   0.558006786488582       -0.129235898353331      0.0103848220654675    0.785620092324558   -0.214342618440993      0.0358276712544366      -0.0028540993155426   -0.139670591365576      0.0252410044114054      -0.00188148964069532    0.000128835962517662  0.3631 0.4419       0.57195    0.53797 0.43984 0.40000 5   7   16   36
1.300     -11.5801081175716 12.7888089378889    -7.2227001491234    2.27538243347493    -0.377579348785784      0.0311714247167985      -0.00101439010918112  -0.654809753810345  -0.242081769907578  0.0978617611306403      -0.00755106846268455 -1.30650263830353   0.451090558851226       -0.107994077135646      0.00907956402309892   0.71734043016381    -0.16025983799785       0.0244391389119698      -0.0021367770284234   -0.12874882127164       0.0157609143815441      0.000168275503391164    -1.31360554039819e-06 0.3610 0.4404       0.56949    0.53761 0.43806 0.40000 5   7   16   36
1.350     -11.3739989476099 12.4442022104095    -6.99930718507673   2.19861997375889    -0.363496724717061      0.0298798273070846      -0.000967911091774405 -0.801584220764737  -0.162754518681337  0.0840000219231284      -0.00676513764906378 -1.16864547810354   0.351026265520659       -0.0873710029986066     0.00778332890506876   0.657520929012951   -0.108643079742382      0.0132093396486519      -0.00141598336394423  -0.11911515430796       0.00664678032069636     0.00219599566223394     -0.000132191163867678 0.3605 0.4391       0.56808    0.53727 0.43634 0.40000 5   7   16   36
1.400     -11.2588789711267 12.2246692090657    -6.84712120586695   2.14320057728601    -0.352918159887293      0.0288849544633734      -0.000931542325047501 -0.908173990056552  -0.10292156075597   0.0732940678768098      -0.00614880058325977 -1.07461601067962   0.271768388924063       -0.0701244179436839     0.00667202590026648   0.618021311634715   -0.0667972862699399     0.00360115743887897     -0.0007864736924067   -0.112812595641757      -0.000837264340225372   0.00394701398796561     -0.000247126288650395 0.3616 0.4377       0.56779    0.53694 0.43469 0.40000 5   7   16   36
1.450     -11.0549318997446 11.8714838117141    -6.61668744279631   2.06490497964829    -0.338773739703979      0.0276061965370316      -0.000886085447208172 -1.01108050134973   -0.0413276116401352 0.0617791107281291      -0.00546462758852666 -0.970609328374637  0.181066786128193       -0.0500707896406135     0.00536597693188345   0.569319702148477   -0.0167438038735176     -0.00788369766993034    -3.04888033185672e-05 -0.104457292410139      -0.0100017891801455     0.00607153115226833     -0.000386967207877444 0.3604 0.4365       0.56604    0.53662 0.43309 0.40000 5   7   16   36
1.500     -10.8630667145058 11.5280356590331    -6.39019479977922   1.98787265292976    -0.324892277702879      0.026355317391037       -0.000841759721577968 -1.07997764167244   0.00375964481436417 0.052922227824976       -0.00492213060311999 -0.905053656832546  0.109247940512159       -0.0331481883504235     0.00423316443087172   0.53924674408585    0.0237813323819396      -0.0177213491180649     0.00063088031913207   -0.0992902100557333     -0.0174731498119008     0.00789328062270415     -0.000508942463020817 0.3559 0.4352       0.56223    0.53631 0.43155 0.40000 5   7   16   36
1.550     -10.6993381369264 11.2175386214239    -6.1815302385103    1.91639761360733    -0.311991303067309      0.0251936243874911      -0.000800660708399015 -1.10847226641869   0.0285537276160759  0.047421264944073       -0.00456170785346130 -0.885047288918211  0.0605708120387341      -0.0201433956895336     0.00331878503649131   0.530788254366364   0.0529827024733062      -0.0255820354250639     0.00117903893286413   -0.0977335211062396     -0.0229974937625837     0.00936609072316375     -0.000610519772528141 0.3507 0.4340       0.55804    0.53601 0.43005 0.40000 5   7   16   36
1.600     -10.4499583213478 10.7832693370065    -5.90173233756918   1.82423076327618    -0.295858673302721      0.0237734104175807      -0.00075124235611301  -1.15727528468144   0.0686090228539974  0.0386970047327156      -0.00399531136750116 -0.827110262806658  -0.0148156729893711     -0.00165915619273248    0.00205990660668236   0.500162976496153   0.0973508935621542      -0.0365213746769877     0.00191987472861017   -0.0920469651148876     -0.0313036031759141     0.0114007923251905      -0.000747219368571935 0.3479 0.4329       0.55537    0.53572 0.42861 0.40000 5   7   16   36
1.650     -10.2695403007128 10.4394531946091    -5.67253860772811   1.74719618581430    -0.282216442903041      0.0225645305579132      -0.000709025222306853 -1.15378266504943   0.0809955416312286  0.0347360770176827      -0.00369557713359646 -0.831919811392387  -0.0562210807292327     0.0108245158405986      0.00114467630052706   0.500233142888849   0.124670481395406       -0.0443861422125266     0.00248184754015317   -0.0915601760778861     -0.0366640506808228     0.0128959197955379      -0.000852191807641178 0.3486 0.4318       0.55495    0.53544 0.42721 0.40000 5   7   16   36
1.700     -10.0851251193758 10.0850795185225    -5.4367957550031    1.66855202623676    -0.26840156267927       0.0213491216387068      -0.000666836987420628 -1.1285869877384    0.0835979020758775  0.0322348257067852      -0.00346788483284947 -0.858479335332959  -0.0879214142874693     0.0218572813578826      0.000301742417897314  0.510021837022006   0.147455051880588       -0.0515421508672373     0.00300704877775135   -0.0926560156909515     -0.0412459487021052     0.0142636017736156      -0.000950301146158572 0.3485 0.4307       0.55401    0.53517 0.42585 0.40000 5   7   16   36
1.750     -9.8529844806179  9.66010215380819    -5.16038098885146   1.57820077910105    -0.252786638266623      0.0199920585578778      -0.000620158101578571 -1.10140855636606   0.0869059120180973  0.0294018773794279      -0.00321368401932204 -0.882286300917828  -0.122838133846721      0.0336121269440657      -0.000585784286443485 0.517430175334183   0.17206054299983        -0.0590460616862518     0.00355121212370857   -0.0932800762274621     -0.0461260802301354     0.0156811798570958      -0.00105073850487069  0.3470 0.4296       0.55226    0.53491 0.42453 0.40000 5   7   16   36
1.800     -9.6308853451126  9.24915492325434    -4.89292555184776   1.49100002795485    -0.237770907265093      0.0186921717914302      -0.000575619287410375 -1.06947066218158   0.0900227908366208  0.0262871601067123      -0.00292957681842313 -0.905098510680778  -0.161150720564872      0.046342934868225       -0.00154601820217044  0.522528337457338   0.199192598981104       -0.0671726436165666     0.00413941012581353   -0.0933205469352875     -0.0515197224622008     0.0172182986114409      -0.00115947709574478  0.3469 0.4286       0.55137    0.53465 0.42325 0.40000 5   7   16   36
1.850     -9.42661648627444 8.85672280087400    -4.63487739693911   1.40670438854844    -0.223272226764221      0.0174397009052671      -0.000532801920969797 -1.00231866806362   0.0750863817007537  0.0262014280632096      -0.00281189935955960 -0.969397747744924  -0.177491836020124      0.0552662488346678      -0.00229081529397661  0.54776575202603    0.215305490966781       -0.0733415978997851     0.00461478964247506   -0.096753761815194      -0.0550122625257704     0.0184120767575879      -0.00124821681135411  0.3448 0.4276       0.54931    0.53440 0.42200 0.40000 5   7   16   36
1.900     -9.23308206595711 8.47977262465146    -4.38654955517159   1.32570008314989    -0.209376295907254      0.0162428953012206      -0.000492010911136646 -0.930387845997108  0.0589303013517743  0.0261632929997208      -0.00269024023886451 -1.03553584024441   -0.193944640588766      0.0643217355935606      -0.00304683923006318  0.573158076283225   0.231583035702311       -0.0795558794635384     0.00509287714074223   -0.10017916393276       -0.0585080585674063     0.019603467339573       -0.00133659149227709  0.3435 0.4266       0.54775    0.53416 0.42079 0.40000 5   7   16   36
1.950     -9.03267162471189 8.08791964317238    -4.12905696255836   1.24224166373830    -0.195156384948958      0.015025547379368       -0.000450729152975109 -0.831569335498681  0.028619377653287   0.0285480805269086      -0.00270355955158651 -1.1349114082803    -0.191814686175251      0.0700352226864892      -0.00360911275477449  0.615473582261431   0.237939843264766       -0.0839288469798334     0.00546188155533656   -0.106579832908764      -0.0602028021918002     0.0204539528823783      -0.0014045167273573   0.3422 0.4257       0.54621    0.53392 0.41961 0.40000 5   7   16   36
2.000     -8.82863068465115 7.69216010487980    -3.87038977912957   1.15885881611569    -0.181020476070451      0.013820793236543       -0.00041003771522083  -0.738467268676936  0.00334654060025816 0.0298021140609932      -0.00264218352851411 -1.2208045910456    -0.199200650471393      0.0776973935729019      -0.0042940178281064   0.649324323161614   0.249855248521256       -0.0894054322711175     0.00589932960311168   -0.111338384266706      -0.0629385528778528     0.0215080803267449      -0.00148499985456737  0.3412 0.4248       0.54482    0.53369 0.41845 0.40000 5   7   16   36
2.500     -7.08348696410041 4.12550351563706    -1.53444032765611   0.415579970891054   -0.0571918356432457     0.0034554667843732      -6.59057491481806e-05 0.674442343781608   -0.486217790845281  0.0808194077480625      -0.00409174061015541 -2.61781093809145   0.0341265726561682      0.098046956698467       -0.00782687951205312  1.2528871127489     0.20046712203176        -0.111396120486723      0.00827380655815574   -0.205263538309866      -0.0588458285859726     0.0257447147771371      -0.00189920562310224  0.3321 0.4166       0.53278    0.53166 0.40830 0.40000 5   7   16   36
3.000     -6.11919558701915 1.86038668856475    -0.040541095360978  -0.0506343287471614 0.0181156975904317      -0.00261911824775958    0.000127761322208581  1.92266855504086    -0.880348855921501  0.114614136305619       -0.00456044137564636 -3.68718151259291   0.142719765061869       0.133935761692992       -0.0119915829699352   1.6713043987926     0.206879143983291       -0.1384214002428        0.0107725463556572    -0.265951493915323      -0.0632790022520034     0.0305304079881707      -0.00231595741921212  0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
3.500     -5.30258271142734 -0.180753703609002  1.29755805651629    -0.458181008185273  0.0819909548634181      -0.00760938630880056    0.000281739271755821  3.84831513124347    -1.70855430371685   0.232576629993746       -0.0101239686098605  -5.88081467594137   1.0343467682154         0.0108000120801096      -0.00622525683078032  2.75013700141112    -0.257428038145727      -0.0688559033198623     0.00719896987500246   -0.452576413788002      0.0227273525517516      0.0166937991452037      -0.0015588094588817   0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
4.000     -5.05582608504808 -1.13903521173706   1.93463024598385    -0.645305153209225  0.109388793937667       -0.00954507768221735    0.000333664832705417  5.08217744609836    -2.20648870861536   0.298607224543568       -0.0130033841837082  -7.19375536529031   1.53289686558961        -0.0529504066481639     -0.00347610241073145  3.38121300293738    -0.523885636850481      -0.0296002823259113     0.00521451661873273   -0.560920963147212      0.0737950201212629      0.00835321176449438     -0.00110021239285952  0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
5.000     -5.37211463924181 -1.44544126593344   2.15971732149651    -0.68995658066025   0.109891396630229       -0.00890687134325725    0.000288515723909978  6.72671858240978    -2.91217181789525   0.399639453199624       -0.0178305428081085  -8.99708127619316   2.34891406055507        -0.179710894633751      0.0031820173420075    4.286884257912      -1.00254472982187       0.0545297998006564      0.000382400792339706  -0.721768040097473      0.169093890130861       -0.00953065550495821    -3.44203177137766e-05 0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
6.000     -6.28530255075682 -0.484603196073713  1.56187874455924    -0.481066795972676  0.0696987010705011      -0.00498555901548499    0.000138201029006892  7.52133411504863    -3.31178694983245   0.466521754590376       -0.0215273856503539  -9.91603435143021   2.91765760803818        -0.28967842986563       0.00989188667906388   4.78496554415023    -1.37118904145178       0.130911826665787       -0.00440388775475607  -0.814927945681046      0.244886941095207       -0.0256227997671617     0.000978707733650216  0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
7.000     -7.39599823347359 0.995737320552461   0.625562983689146   -0.171800101578979  0.0143456529666372      6.52344981533968e-05    -4.48969893944614e-05 7.77408892617778    -3.51272094473493   0.510395207868884       -0.0244002669518105  -10.2522907004007   3.29538962009899        -0.37986932401322       0.0159743312227645    4.99993960756539    -1.63781637458877       0.194272197731113       -0.00859681922707332  -0.858902471619047      0.300706947521901       -0.0387233846698785     0.0018319243386288    0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
8.000     -8.55468243393022 2.67121565431662    -0.434084981395819  0.170234005116985   -0.0450873210624716     0.00533137760482041     -0.000230798241570966 7.71549776889735    -3.59772736056867   0.540384047216496       -0.0267418521226501  -10.2432854817137   3.54398550876769        -0.453333151069785      0.0213022049692242    5.03805677721267    -1.82595562628027       0.245086308432867       -0.0120953065016613   -0.870844941347957      0.340232302628311       -0.0489207907527363     0.00251262954172925   0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
10.00     -10.3837541428057 5.47650331910390    -2.22509123259836   0.742386538803105   -0.14279264490099       0.0138216515404767      -0.000524696977114909 7.09314297954152    -3.55008000264125   0.568593200359083       -0.0298568843701883  -9.62634853349033   3.74596953603524        -0.549794113299229      0.0290600891183059    4.78781287693964    -2.010452117059         0.309932564149234       -0.0168326636777266   -0.833452405096288      0.37920812089815        -0.0612486220754326     0.003368216240734     0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
    """)

COEFFS_ALPINE_75Bars = CoeffsTable(sa_damping=5, table="""\
IMT     a1                  a2                  a3                  a4                  a5                           a6                         a7                      a8                  a9                  a10                 a11                     a12                 a13                 a14                     a15                     a16                 a17                 a18                     a19                     a20                 a21                     a22                     a23                     tau    mean_phi_ss  sigma_tot  phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv     -8.79258935089329   8.04377956891795    -4.50204761616682   1.57436118358543    -0.2936332094222             0.026959304370498600       -0.000962529126454320   3.08399490951917    -1.92698076800152   0.349492969702486   -0.0200346401551332     -4.45386122473234   1.51433804842312    -0.220565025436411      0.0127494318842956      1.91981818727018    -0.537626549622912  0.0546275202974513      -0.00240425876590603    -0.308252939659737  0.0704663001991279      -0.00323045801587478    -7.07356205172433e-05   0.5010 0.4600       0.68015    0.00000 0.00000 0.00000 5   7   16   36
pga     -6.49637351492575   7.87173863446885    -4.65111097711909   1.67981549519530    -0.320089460753416           0.029824505922715400       -0.001077105246834080   2.77236858349711    -1.8042369264732    0.334137388764005   -0.0195035664924781     -4.07542642354293   1.34457419030398    -0.19665906964692       0.0118004580801271      1.72090431673415    -0.455566171656945  0.0421318664758975      -0.00184906771151628    -0.28536772260574   0.0603942095441127      -0.00169205542927187    -0.00014071293672638    0.3532 0.4600       0.57998    0.58000 0.47000 0.35000 5   7   16   36
0.010   -6.4875923841274    7.86443033264761    -4.64334859043159   1.67746303300781    -0.319745355341145           0.029798577751661200       -0.001076317174457000   2.91185221880752    -1.86639393692714   0.343345326727985   -0.0199618194615103     -4.23851617014292   1.41719767752921    -0.208147695564034      0.0124267861089279      1.78836476199477    -0.487389680877431  0.0472820379625075      -0.00213430686632027    -0.295065243313866  0.0652556734933412      -0.00248261582576433    -9.72092241938994e-05   0.3529 0.4600       0.57975    0.58000 0.47000 0.35000 5   7   16   36
0.02    -6.21675204497831   7.71583851108833    -4.57362883657938   1.66094573362704    -0.317757166249784           0.029684857843770000       -0.001074167703662870   3.16559287521649    -1.95587346311972   0.353491253775633   -0.0203534113762135     -4.46350464603347   1.49262893345419    -0.216344340599647      0.012719288095581       1.7945692182438     -0.500517850329642  0.0479468151453272      -0.00205983419739415    -0.281814859726649  0.0645027341085831      -0.0021610892974663     -0.000140097607248218   0.3674 0.4570       0.58639    0.56796 0.46097 0.37408 5   7   16   36
0.03    -5.51568214733412   7.32375190918703    -4.4345405517502    1.62919497536306    -0.313460685696694           0.029387551321096100       -0.001066357549344390   1.59941156678265    -1.35820630687646   0.278313138631201   -0.0172897428319913     -2.34687530199703   0.778700105214277   -0.134348022519734      0.00964436366465601     0.704669849095948   -0.167452742209292  0.0118414853387539      -0.000788933289956785   -0.0982114940685386 0.0119946618212363      0.0032842211029814      -0.000318905363253994   0.3749 0.4552       0.58975    0.56092 0.45569 0.38817 5   7   16   36
0.04    -5.18454605892775   7.12547837316401    -4.34973682093686   1.60659221188391    -0.310078776133208           0.029133694926292300       -0.001058906763193380   0.989031006557902   -1.19908286208595   0.269340861738234   -0.0175212582777871     -1.71269998091545   0.686955899335995   -0.14196646642178       0.010995923925137       0.506443240204535   -0.173074695643051  0.0230308733048936      -0.00187359281860763    -0.0868962475831448 0.020111089677488       0.000462439497279187    -8.42112552558363e-05   0.3801 0.4540       0.59211    0.55592 0.45194 0.39816 5   7   16   36
0.05    -5.00913882695838   6.79711925708578    -4.14238699427697   1.54520007411128    -0.300507187543808           0.028373889395882000       -0.001034642509005080   1.32224617455405    -1.29170548089409   0.273535646921393   -0.0172411238385529     -2.18429208680281   0.774988394521494   -0.135756502247142      0.00977113772831404     0.815318682451546   -0.209717975754419  0.0152725588842638      -0.000916309569950278   -0.152105444230074  0.0245451624534802      0.00260693039636085     -0.000305952972737499   0.3855 0.4530       0.59483    0.55204 0.44903 0.40592 5   7   16   36
0.10    -6.42155541591505   7.71159782825328    -4.45849835544046   1.61794122109022    -0.310796686287489           0.029155916581072900       -0.001058771941998110   4.50424677629564    -2.55085449710802   0.442142471401388   -0.024725299717385      -6.17824631053111   2.2883127264841     -0.343104524743932      0.0194445526200129      2.82777331755952    -0.909336060692024  0.112236994714465       -0.00561060612525734    -0.48479661803998   0.131771769922707       -0.0126562374433012     0.000463859441630166    0.3864 0.4500       0.59312    0.54000 0.44000 0.43000 5   7   16   36
0.15    -6.94898533602135   8.05833841777793    -4.60707631550902   1.65582572580578    -0.316064527902763           0.029524997526117700       -0.001068947749254150   4.44311122054194    -2.5109860948137    0.435392085558855   -0.0243285654452588     -6.02299768442596   2.28117047352211    -0.353125366219836      0.0203703121870069      2.72113887825418    -0.916092830402202  0.123067678139271       -0.00653519608569529    -0.455926469627983  0.134949734226089       -0.0155904514491441     0.000705192953422354    0.3841 0.4675       0.60507    0.58095 0.47510 0.40075 5   7   16   36
0.20    -7.43140361907233   8.40058720545741    -4.77223774169086   1.70143893267297    -0.322759933590155           0.030013254675328100       -0.001082847973987720   4.3036443111639     -2.46380223119858   0.430318432085384   -0.0241455723769771     -5.80833221330292   2.26387936262335    -0.360118736664939      0.0210850089041317      2.59651055784003    -0.913484901262882  0.129002289252245       -0.00707184541844335    -0.428469693635409  0.135255481339345       -0.0169293117107198     0.000820303097943819    0.3690 0.4800       0.60546    0.61000 0.50000 0.38000 5   7   16   36
0.25    -7.92297708653674   8.80960100566352    -4.99267002799744   1.76576741728758    -0.332562514847774           0.030753221325322800       -0.001104731025377050   4.17348323098503    -2.41816351142948   0.424980704182593   -0.0239318266466204     -5.6183610080512    2.23893975900774    -0.363200650560755      0.0214959830559508      2.48850150030212    -0.90270448706172   0.131469009831429       -0.00734935670963006    -0.405725250516246  0.133301743991057       -0.0173866375498347     0.000871073021788007    0.3445 0.4800       0.59082    0.62651 0.50000 0.37450 5   7   16   36
0.30    -8.46457972333153   9.33715361513875    -5.29201033589102   1.85418315051172    -0.346136358427324           0.031788269250494900       -0.001135778271575390   4.07836006886033    -2.39014496680376   0.422637788954225   -0.0238897014961851     -5.48929560631845   2.23600381633291    -0.369245484694974      0.0220574722352406      2.4172960075479     -0.903307832529323  0.135287328424272       -0.00769221482871347    -0.390881085607202  0.13329916531164        -0.0180441262088827     0.000930946771812734    0.3377 0.4800       0.58688    0.64000 0.50000 0.37000 5   7   16   36
0.35    -8.97636263838892   9.86003458730357    -5.59439799458926   1.94279736787599    -0.359473347153334           0.032777848433416700       -0.001164468609278880   3.95706077975352    -2.34504086279749   0.416971920387281   -0.0236472661706837     -5.33148087899945   2.20543221310924    -0.369181207714854      0.0222257779722777      2.3330617716249     -0.888254240292492  0.135501995734129       -0.00779911594625877    -0.374422925655851  0.130357031399184       -0.0180088174832459     0.000945102853018643    0.3482 0.4740       0.58812    0.62793 0.49396 0.37000 5   7   16   36
0.40    -9.58054891165313   10.5437333966375    -5.99242856914871   2.05857652687095    -0.376870273165868           0.034076547394656000       -0.001202639930281190   3.84395138404084    -2.30505620179771   0.412372770991125   -0.0234759951103998     -5.20222885493194   2.18698846426214    -0.371181781531861      0.0225211971719162      2.26887076521376    -0.88142836890057   0.137115944523566       -0.0079926882389357     -0.362395116933416  0.129094103444283       -0.018260322706965      0.000977182048578723    0.3552 0.4687       0.58811    0.61747 0.48874 0.37000 5   7   16   36
0.45    -10.1003750700039   11.1391195747564    -6.33978456138263   2.15775781388739    -0.391290842997426           0.035105380358314700       -0.001231153693199430   3.69086963794407    -2.24365055957528   0.404092129406456   -0.0231001440734321     -5.03462142441391   2.14297788644415    -0.368228886280178      0.0225172300587793      2.18824504784214    -0.861935552029083  0.136130969362486       -0.00802301202322586    -0.348000624659845  0.125641135487219       -0.0180473055174597     0.000979446313380743    0.3495 0.4641       0.58097    0.60825 0.48413 0.37000 5   7   16   36
0.50    -10.6067955220404   11.7433998390405    -6.69170972327515   2.25690012125541    -0.405462646777183           0.036096712317583500       -0.001257987390848380   3.52943492130265    -2.18055832577414   0.395943386628905   -0.0227521867556441     -4.87982852390125   2.10660162170701    -0.36693316541978       0.0226282134963942      2.12059835895043    -0.849304096294292  0.136498249171002       -0.00814137843288167    -0.336688903215909  0.123709934048471       -0.0181284219867314     0.00100056412099785     0.3548 0.4600       0.58095    0.60000 0.48000 0.37000 5   7   16   36
0.55    -11.0453945615526   12.2807134864570    -7.0047888424676    2.34312545324540    -0.417312984517734           0.036879045624672500       -0.001277429988144200   3.31167019400931    -2.08721095236787   0.382557792280688   -0.0221104965748969     -4.67053814671376   2.0368630328549     -0.359403286148736      0.0223709886779782      2.02988484540894    -0.821331769387894  0.133889675284303       -0.00807985966475487    -0.321958357160842  0.119298377568437       -0.0177186493598624     0.000991687601730219    0.3639 0.4586       0.58544    0.59175 0.47587 0.37413 5   7   16   36
0.60    -11.4295253909742   12.7622299284149    -7.2845115476028    2.41836489930937    -0.427245676636249           0.037494750610766000       -0.001291190921880710   3.07146250471327    -1.98425396767145   0.367863305819008   -0.0214121040397935     -4.45259225320051   1.96228380502706    -0.351142798463493      0.0220812531958627      1.93986412284516    -0.792952826103998  0.13119755024788        -0.00801508604441731    -0.307903518665507  0.115004906395082       -0.0173199155105115     0.000983318899731608    0.3757 0.4574       0.59187    0.58422 0.47211 0.37789 5   7   16   36
0.65    -11.7446189255009   13.1638191840161    -7.51658058507351   2.47844388186404    -0.434610121443849           0.037891744314673700       -0.001297582469163700   2.80233261634057    -1.86719533089757   0.35093315317644    -0.0205978177359329     -4.21375186419492   1.87479769268693    -0.34052765838784       0.0216571208257068      1.84363939994438    -0.759765613513414  0.127554071927988       -0.00789372059540583    -0.293307923826774  0.11009170202611        -0.0167911197203163     0.000967068060294179    0.3831 0.4562       0.59573    0.57729 0.46864 0.38136 5   7   16   36
0.70    -11.9934831501094   13.4853221538910    -7.70063929497842   2.52343365665145    -0.439459893513683           0.038078001451806400       -0.001296975796061610   2.50972764071302    -1.73832739435117   0.332083427189067   -0.0196820273093013     -3.95624996683001   1.774604478045      -0.32745411193713       0.02108662707576        1.74090894010868    -0.721016403797642  0.122739824783603       -0.00769923700190516    -0.27792184734549   0.104312718098624       -0.0160705506450928     0.00093851566584195     0.3886 0.4551       0.59846    0.57087 0.46544 0.38456 5   7   16   36
0.75    -12.1771813771535   13.7257607162188    -7.83632450091753   2.55346036301019    -0.441861205831544           0.038062803400647600       -0.001289798716336400   2.19050669136061    -1.5947225953102    0.310631042389663   -0.0186180988677611     -3.66915565639999   1.65387008277176    -0.310272082976119      0.0202628977375258      1.62422575184716    -0.671628775779829  0.115709842150423       -0.00736481555299123    -0.26024158041407   0.0966723551626035      -0.0149554035919014     0.000884767089906794    0.3852 0.4542       0.59554    0.56490 0.46245 0.38755 5   7   16   36
0.80    -12.3282984224435   13.9294706876446    -7.94955225034381   2.57645293501993    -0.443129301124479           0.037957835862386800       -0.001279850030665500   1.85184934689906    -1.43942451489063   0.28699796501978    -0.0174251342329117     -3.35716677213452   1.51409772996236    -0.289114106086396      0.0191882478358915      1.49477526741132    -0.611711205292084  0.106416996848441       -0.00688532975951791    -0.240337167757098  0.0871175244616604      -0.0134246444586864     0.000804217325301106    0.3850 0.4532       0.59468    0.55932 0.45966 0.39034 5   7   16   36
0.85    -12.3619179615435   13.9607263606749    -7.9583623351726    2.56765612694125    -0.439255586596872           0.037430150224854900       -0.001255985372935820   1.53462704675689    -1.2962522418547    0.265579408647079   -0.0163630413269425     -3.07814349885617   1.39034732573393    -0.270648662258181      0.0182666424049099      1.38410563388802    -0.560809997602814  0.0986415029927488      -0.00649270125033115    -0.223982630812115  0.0792809644835306      -0.0121883448270131     0.000740773625529971    0.3811 0.4523       0.59150    0.55407 0.45703 0.39297 5   7   16   36
0.90    -12.3915221446487   13.9931508335694    -7.96723611691972   2.55885423088625    -0.435431895704698           0.036914153825131900       -0.001232822596446600   1.19951478952810    -1.14034398175199   0.241551847624106   -0.0151369840409879     -2.76755739836427   1.24165566304348    -0.246862542443972      0.0170038458040194      1.25485487163326    -0.495084405289948  0.08768913482064        -0.00589560463549598    -0.204113664024344  0.0686368501955645      -0.0103507178388109     0.000638580012077928    0.3799 0.4515       0.59006    0.54912 0.45456 0.39544 5   7   16   36
0.95    -12.3271622179915   13.8868104405280    -7.89278585923823   2.52507325231301    -0.427642654035884           0.036078607026934000       -0.001199352416905230   0.877442756258114   -0.990110084127535  0.21836174161911    -0.0139527661013294     -2.46930045467272   1.09524745065619    -0.223009297573371      0.0157210171935273      1.13104639137239    -0.429771018442517  0.0765496332417734      -0.00527922834488017    -0.185155480095236  0.0580360805414685      -0.00847779261567531    0.000533036844696612    0.3733 0.4507       0.58525    0.54444 0.45222 0.39778 5   7   16   36
1.00    -12.2750048475035   13.7925235265100    -7.82265723922337   2.49255885755551    -0.420140802890858           0.035277374610433900       -0.001167418365112900   0.590651791944919   -0.856865611943822  0.197878588363516   -0.012910645232768      -2.20712362332126   0.962973909771783   -0.201003726487474      0.0145184028989784      1.02290735511275    -0.369786631712462  0.0659820496628516      -0.0046813115189442     -0.168615635833199  0.0481614858135606      -0.00666623490682732    0.000428422802326004    0.3716 0.4500       0.58362    0.54000 0.45000 0.40000 5   7   16   36
1.05    -12.167571863905    13.6206309540103    -7.7070525791719    2.44670606765750    -0.410570440279165           0.034313591881814500       -0.001130376186644600   0.28743982268111    -0.710453366618039  0.174543340680894   -0.0116837028416522     -1.91337418192747   0.805600398198241   -0.173624232743604      0.0129710177357504      0.896344362991052   -0.295456868777715  0.0523735868846414      -0.00388934542454021    -0.148679489556916  0.0356806256635043      -0.00430698969698832    0.00028901013111988     0.3691 0.4482       0.58063    0.53956 0.44778 0.40000 5   7   16   36
1.10    -11.9971862156797   13.3477524362452    -7.5291768324343    2.38210548753494    -0.398032446395399           0.033111684270352900       -0.001085678329440510   0.018586591599727   -0.580313671928354  0.153773644033567   -0.0105911110527611     -1.65546729002535   0.662944728463375   -0.148306627558068      0.0115222956397357      0.785852212111655   -0.227301500363363  0.0395776866506911      -0.0031343172235108     -0.131299006498217  0.0241394401849806      -0.00206639272162178    0.000154794880522637    0.3650 0.4465       0.57674    0.53913 0.44566 0.40000 5   7   16   36
1.15    -11.8236240677078   13.0658141309163    -7.34482144624203   2.31588267035931    -0.385347316105196           0.031909344836342500       -0.001041372862622760   -0.229132493235452  -0.458258996889664  0.133991218126228   -0.00953634664252515    -1.4133796803072    0.522869224677497   -0.122732160096558      0.0100306731269898      0.680708652721936   -0.158922335162446  0.0263854890167507      -0.00234287766262392    -0.114615450533914  0.0124511259437704      0.000257379799696194    1.36814779055479e-05    0.3627 0.4449       0.57402    0.53873 0.44364 0.40000 5   7   16   36
1.20    -11.6351586426677   12.7575581713128    -7.14429144344421   2.24526976259725    -0.372085678592926           0.030672008347692600       -0.000996326735215678   -0.452868135804902  -0.346242081291756  0.115597295402499   -0.00854510566465387    -1.1916141736176    0.388724063559057   -0.0976068847090878     0.00854185306101436     0.583107809986285   -0.09197067438086   0.0131374739366446      -0.00153654753242955    -0.0989412049844522 0.000849341011577503    0.00261887498909623     -0.000131633935579949   0.3625 0.4434       0.57267    0.53834 0.44170 0.40000 5   7   16   36
1.25    -11.4190010713896   12.4010414638067    -6.91343397525373   2.16566687745718    -0.357444777640615           0.029328142400102500       -0.000947991275067985   -0.648510979393486  -0.246480934123943  0.0989792868535597  -0.00763942407674244    -0.995898245431203  0.264022403195128   -0.0736024686343505     0.00709646389676662     0.496267882899783   -0.0285771448961346 0.000265178026783147    -0.000742581599550195   -0.0849007008332981 -0.010229686889619      0.00492604246177454     -0.000275150299294962   0.3631 0.4419       0.57195    0.53797 0.43984 0.40000 5   7   16   36
1.30    -11.1917928942284   12.0236350254525    -6.66979450799955   2.08269771787100    -0.342384489159176           0.027961150066680500       -0.000899266893454426   -0.822046612789351  -0.155365352193723  0.0834510367480428  -0.00677769343213297    -0.818565985836958  0.144241902294024   -0.0498686176874192     0.005643245687711       0.416385645465046   0.0335137327744355  -0.012658693946336      6.49461403248228e-05    -0.0718492858384647 -0.0211735035469962     0.00725336001575224     -0.000421437982684768   0.3610 0.4404       0.56949    0.53761 0.43806 0.40000 5   7   16   36
1.35    -10.9512903049813   11.6195918490589    -6.40928799618411   1.99498745169470    -0.326659999507413           0.026548791951194200       -0.000849343836107831   -0.964732194059332  -0.0780387223988469 0.0699605580989519  -0.00601558316485411    -0.671269824193527  0.0365237962545589  -0.0277450228309271     0.00426154895378459     0.349287586148845   0.0906773921200366  -0.0249526131823957     0.000845725210770521    -0.060765018041325  -0.0313702493955294     0.00948624292285991     -0.000563759237243513   0.3605 0.4391       0.56808    0.53727 0.43634 0.40000 5   7   16   36
1.40    -10.703216989219    11.1990194449864    -6.13842014334681   1.90456390076679    -0.310603860917352           0.025118787819028600       -0.000799150230537859   -1.08362064898969   -0.0106063653817886 0.0578264300149716  -0.0053149392844634     -0.545489480516626  -0.0636611058491074 -0.00646452685654953    0.00290951702055459     0.290934510891228   0.145014167135732   -0.036964802196986      0.00161842117307177     -0.0509916955334771 -0.0411538174797226     0.0116787665850736      -0.000704951593942213   0.3616 0.4377       0.56779    0.53694 0.43469 0.40000 5   7   16   36
1.45    -10.4474647009905   10.7621280816868    -5.8574858569814    1.81152976188110    -0.294230489586345           0.023671918620183200       -0.000748693919825705   -1.17958666488779   0.0471907178896637  0.0470231122180482  -0.00467466315268848    -0.440238943538024  -0.156440768267878  0.0139630535122428      0.00158802267779335     0.240763279982581   0.196596643082396   -0.0486905716916759     0.0023826466697799      -0.0424166873988386 -0.0505422893167929     0.0138310555549937      -0.000845006499011742   0.3604 0.4365       0.56604    0.53662 0.43309 0.40000 5   7   16   36
1.50    -10.1927466651198   10.3210176734090    -5.5734567915804    1.71794697307903    -0.277868056552553           0.022234900477998600       -0.000698848041573846   -1.24725034324395   0.0920403845721005  0.0381864402769335  -0.00413362787453791    -0.365024589232778  -0.235634737083817  0.0323066187900131      0.000374331962057972    0.204486619048212   0.241662714331343   -0.0593735841546201     0.0030906323467113      -0.0361427500786255 -0.0588070190920401     0.0157962194479624      -0.000974631343959612   0.3559 0.4352       0.56223    0.53631 0.43155 0.40000 5   7   16   36
1.55    -9.91089314773954   9.83484838554996    -5.26242089919018   1.61660991048881    -0.260340183283867           0.020709017381287300       -0.000646279281184051   -1.2933500821383    0.127551131279256   0.0306882884295208  -0.00365619745518721    -0.309419529501239  -0.307082164509023  0.0496277261207317      -0.000794182287156644   0.17609415775604    0.283682520127114   -0.0696608890190989     0.00378129206017332     -0.03102872574906   -0.0666191518844858     0.0177011933150348      -0.00110153292495162    0.3507 0.4340       0.55804    0.53601 0.43005 0.40000 5   7   16   36
1.60    -9.67611106976348   9.41478479744091    -4.99056411940023   1.52756150706121    -0.244914850513575           0.019367447981791100       -0.000600171253118401   -1.31772400610962   0.153981399324518   0.0244310078177366  -0.00323402288648523    -0.275336013358968  -0.369891293417233  0.0658026476514209      -0.0019128609664095     0.157193075384505   0.321776221818662   -0.0794059212878138     0.00444723861281994     -0.0274321391572616 -0.0737765273287003     0.0195113958157602      -0.00122391269401927    0.3479 0.4329       0.55537    0.53572 0.42861 0.40000 5   7   16   36
1.65    -9.41957120454348   8.95764269206837    -4.69640732134827   1.43213552008491    -0.228536698519053           0.017953646071954800       -0.000551860065534762   -1.31949246914358   0.170140760063703   0.0197228140165952  -0.00288950041280721    -0.264285588054399  -0.422322110718742  0.08039919687102        -0.00295080339225864    0.148534995396104   0.355074310790673   -0.0883905682371217     0.00507268517316986     -0.0254774616771476 -0.0801287576852375     0.0211881323603307      -0.0013389121411996     0.3486 0.4318       0.55495    0.53544 0.42721 0.40000 5   7   16   36
1.70    -9.16944425143818   8.50696230553620    -4.40617225818608   1.33829126221051    -0.212501696557179           0.016575669417590000       -0.000504969071279594   -1.30356262274586   0.178474630915611   0.0161587854400454  -0.00260038123991945    -0.271018039869124  -0.466686796712962  0.0937627077822072      -0.00392559268532087    0.14783305640027    0.384497634395895   -0.0967425397361036     0.00566384756189154     -0.0248143798741175 -0.0858095440088794     0.0227495590673411      -0.00144743576908101    0.3485 0.4307       0.55401    0.53517 0.42585 0.40000 5   7   16   36
1.75    -8.91685002016907   8.04886113010203    -4.11150894253508   1.24348826668085    -0.196393625874156           0.015198522490502100       -0.00045831400937522    -1.26740239174906   0.177614769371154   0.0139814484430567  -0.00238068362275816    -0.298438589831309  -0.501528645605781  0.105654378153395       -0.00482447949489323    0.15634409385189    0.409475286318697   -0.104377479484584      0.00621670596004934     -0.0256288073681167 -0.0907426367713743     0.0241856764401681      -0.00154907427128651    0.3470 0.4296       0.55226    0.53491 0.42453 0.40000 5   7   16   36
1.80    -8.66630490244506   7.59163901039394    -3.81768385040903   1.14933354327319    -0.180470502217827           0.013843271826392200       -0.000412585934980808   -1.21745904539166   0.170928841698316   0.0126093265242006  -0.00219746679275606    -0.339868361131246  -0.5300971928096    0.116603788739319       -0.00567596652269104    0.171350370295106   0.431220280886432   -0.111476903398558      0.00674017639316287     -0.0275424551799676 -0.0950786608921123     0.0255160437746538      -0.00164461434050406    0.3469 0.4286       0.55137    0.53465 0.42325 0.40000 5   7   16   36
1.85    -8.43360888116405   7.15787227539177    -3.53777125791673   1.05970568145384    -0.165353291814076           0.012560926072876900       -0.000369470649978402   -1.14862182265863   0.155420500568578   0.0126047060413427  -0.00208446799592463    -0.401880298983253  -0.548492908552371  0.125874137326782       -0.00643577488369044    0.195914949394201   0.447923212750976   -0.117700600195492      0.00721397126317062     -0.0310278804783865 -0.098541553428957      0.026689358139396       -0.00173104232827908    0.3448 0.4276       0.54931    0.53440 0.42200 0.40000 5   7   16   36
1.90    -8.20956021076711   6.73453656104192    -3.26413366911687   0.97227902489825    -0.150658053565193           0.011318963504201800       -0.000327861779664812   -1.06603269177831   0.133763903903131   0.0135089719088836  -0.00201597434778855    -0.477641837413791  -0.560406757129292  0.134128476387575       -0.00714280155687515    0.226622568819903   0.461488567870591   -0.123401531605625      0.00765934937373286     -0.0355113506032878 -0.101459548992485      0.0277678266668184      -0.0018121686811484     0.3435 0.4266       0.54775    0.53416 0.42079 0.40000 5   7   16   36
1.95    -7.98261304001309   6.30648567237763    -2.98842835333257   0.884625133960452   -0.135996517044339           0.010085297046666900       -0.000286690881851727   -0.977456000968407  0.110335677064951   0.0145322656198114  -0.00194607113160887    -0.557310983846841  -0.571375237117683  0.142365226873269       -0.00785511247545549    0.258420898029988   0.474761695979947   -0.129093531572624      0.00810624629132578     -0.0400974944263273 -0.10433752764898       0.0288428091709537      -0.00189332890370654    0.3422 0.4257       0.54621    0.53392 0.41961 0.40000 5   7   16   36
2.00    -7.77169800782679   5.89872391260318    -2.72456203219806   0.800816091382436   -0.122020943429663           0.008913778399441020       -0.000247746776494106   -0.864011028499006  0.073568400749506   0.0178769435519462  -0.00200746696451346    -0.669384169247054  -0.563620135258215  0.147147527497258       -0.00836283655153392    0.307235844779459   0.477708644987322   -0.13281947105264       0.00843417483597114     -0.0477420651709543 -0.105301327203805      0.0295464930548874      -0.001951747383035      0.3412 0.4248       0.54482    0.53369 0.41845 0.40000 5   7   16   36
2.50    -5.72873257559985   1.93344314382164    -0.193623655679253  0.0140160275433936  0.00622390289374895          -0.001604085231969020      9.48692193094478e-05    0.535267417848339   -0.404796329309001  0.0664413262810956  -0.0033013693773513     -1.98932544318621   -0.38418473734329   0.17852416418513        -0.012591721565665      0.862443846066804   0.461873770352399   -0.161825804578753      0.0112672283211493      -0.132610744334754  -0.107489768264409      0.0351247713259229      -0.00245615879975327    0.3321 0.4166       0.53278    0.53166 0.40830 0.40000 5   7   16   36
3.00    -5.30585005462791   0.522587074125903   0.762346990332983   -0.280193280551713  0.0520831244861663           -0.005112957952256490      0.000199302446917342    1.757729701397010   -0.769052102660023  0.0928103624597346  -0.0032476267880528     -3.08189255204271   -0.271893352227604  0.214592443672515       -0.0167909521146791     1.31835934953498    0.448058897771113   -0.185127426317397      0.0135389707509811      -0.203150248662976  -0.106061774606106      0.03878541903608        -0.00280332149630756    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
3.50    -4.40478592015394   -1.47554273558007   2.017488133065120   -0.651365807837228  0.108690374435257            -0.009409307127026000      0.000327671949439051    3.029965281826240   -1.23002513744009   0.145066242687905   -0.0050436738350427     -4.35351237146446   0.0914614793455659  0.187944048410314       -0.0166727879439113     1.90370511944271    0.270711139195862   -0.16890850338881       0.013145669622138       -0.300819508505976  -0.0723902596139231     0.0348125785451249      -0.00264244723916128    0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
4.00    -4.98869904094196   -1.36004850667911   2.054132978290820   -0.659600637379771  0.106878996231751            -0.008893547228760000      0.000297145781597626    5.221359703071210   -2.2837497393445    0.312971997772707   -0.0138817410381709     -7.07713044928687   1.43234118006328    -0.0337531148953788     -0.00450884250620358    3.30749171656366    -0.463663156271938  -0.0410225195231432     0.00583500497503193     -0.549934734795496  0.0646609302570601      0.0100842295771914      -0.00119389706761422    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
5.00    -5.89999404166068   -0.743766368116282  1.712058574981400   -0.527931223159225  0.0781215633405329           -0.005790259378395180      0.000169236482801189    6.940688293222950   -3.03904499248227   0.423931667704077   -0.0193268404553538     -9.03279867461785   2.35108257649525    -0.180714763084378      0.00337348045257934     4.30024792448512    -0.999763869439221  0.0538904449228874      0.000383108546492251    -0.726336207782242  0.169987897302955       -0.00960074619171955    -2.9990737727456e-05    0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
6.00    -7.18201358525468   0.788728427602718   0.767305205000969   -0.213396333331872  0.0208460939225532           -0.000472032120580259      -2.64657758759701e-05   7.773020179813920   -3.46803383535405   0.497218765306305   -0.0234425300340224     -10.0344894233555   2.97997069292593    -0.302796775173874      0.0108205265398019      4.83993709781673    -1.39695669123678   0.13548175692753        -0.0046860909373807     -0.825862389101745  0.249885024790641       -0.0263487931541931     0.00101228060104498     0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
7.00    -8.53091022049537   2.63132777967413    -0.382733692027545  0.158646393412051   -0.0442243394649519          0.005333446698672040       -0.000232651627740126   8.057752204772220   -3.69493312089833   0.546891588316646   -0.0267002546760455     -10.4340311151749   3.4047446953532     -0.402588372006057      0.0174924386937258      5.08246861792324    -1.68272590811052   0.20230293319752        -0.00906364836137158    -0.873342129875663  0.307919681101597       -0.0397652744424241     0.00187634648545059     0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
8.00    -9.73422704320600   4.37761753298518    -1.48050061271611   0.50950766291239    -0.104467590147923           0.010603585455453600       -0.000416324566514146   7.998346968816910   -3.7851877207237    0.578620200755153   -0.0291759571908356     -10.434497723113    3.66549789754518    -0.478796341123511      0.0229884793398445      5.11788655410578    -1.87042671668226   0.252792599442768       -0.0125188375226768     -0.883307171064198  0.346135273473453       -0.0496088370935914     0.00252835893448223     0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
10.0    -11.4997540658424   7.07730436398346    -3.19303215572863   1.05093724206555    -0.19577311955085            0.018429815094604000       -0.000683404151488038   7.371283108860070   -3.74078098699777   0.608202528331009   -0.0324013627293985     -9.81432180047924   3.87043814380671    -0.575885549856136      0.0307649867858398      4.84912628663873    -2.04273433997934   0.314682756031614       -0.0170342130323825     -0.839835255862144  0.380767393020544       -0.0609159651428684     0.00331087699129787     0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
    """)

COEFFS_ALPINE_90Bars = CoeffsTable(sa_damping=5, table="""\
IMT     a1                  a2                  a3                  a4                  a5                      a6                      a7                      a8                  a9                  a10                 a11                     a12                     a13                     a14                 a15                     a16                     a17                     a18                     a19                     a20                     a21                     a22                     a23                     tau    mean_phi_ss  sigma_tot  phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv     -8.94803317546735   8.25761287353259    -4.64271839903686   1.62519074052252    -0.30311061004023       0.0278236677633493      -0.000993163191843175   3.06096938243978    -1.91503909166005   0.34773746534614    -0.0199610986102721     -4.35486830052005       1.45164645881236        -0.209406589440039  0.0121487722939859      1.8553783993169         -0.495396283831093      0.0467792516950912      -0.00196346152121308    -0.294903676984147      0.0615784398427024      -0.00154833932859103    -0.000166925476761199   0.5010 0.4600       0.68015    0.00000 0.00000 0.00000 5   7   16   36
pga     -6.65272663705094   8.08878595298962    -4.80269233584456   1.7380991941156     -0.331463862842251      0.0308981644867294      -0.00111620508947924    2.73827402216534    -1.78423836048001   0.330722973033323   -0.0193261109448114     -3.96048167707253       1.2705635256631         -0.183268885985897  0.0110657292971236      1.64621499199193        -0.406107496395491      0.0328687637250015      -0.00132402248384967    -0.269554654384873      0.0497514158698079      0.000341503795207526    -0.000258282962208745   0.3532 0.4600       0.57998    0.58000 0.47000 0.35000 5   7   16   36
0.01    -6.63893938369328   8.07659935354836    -4.79283097513347   1.73525197403788    -0.331055369612356      0.030868042304764       -0.00111531544283505    2.86606773340668    -1.83981866372912   0.338765143270263   -0.0197180693441092     -4.11149216842617       1.33687105448213        -0.193699052826744  0.0116343238410594      1.70876224117082        -0.435573705390246      0.0376493696609416      -0.00159009738442181    -0.278541436191357      0.0542996883810904      -0.000403055998735466   -0.000217053419605113   0.3529 0.4600       0.57975    0.58000 0.47000 0.35000 5   7   16   36
0.02    -6.36890293592876   7.92305336856559    -4.71899089571496   1.71764126453399    -0.328951055586892      0.0307506629232678      -0.00111323869487889    3.1259282362498     -1.93219768328067   0.349388783746324   -0.0201365523452208     -4.35365649110155       1.42244208739851        -0.203786749603578  0.0120398884181481      1.7291805710775         -0.457617962015722      0.0400112273891914      -0.00161748940455009    -0.268387875861406      0.0555250768855422      -0.000459655978159537   -0.000237253491330577   0.3674 0.4570       0.58639    0.56796 0.46097 0.37408 5   7   16   36
0.03    -5.32761883890001   7.11392535227562    -4.3627285898483    1.62570495396143    -0.315454205804598      0.0297243596403812      -0.00108210027159882    1.12710867966411    -1.07002799391328   0.22367420481137    -0.014016326784672      -1.58890105401339       0.300530983364257       -0.0416253493077389 0.00399038342639909     0.276661749372171       0.107506483463734       -0.0423686492222069     0.00256283727320317     -0.0173283385961771     -0.0406039428704381     0.0137592435503095      -0.00097183368523694    0.3749 0.4552       0.58975    0.56092 0.45569 0.38817 5   7   16   36
0.04    -5.3147660214145    7.30324159928111    -4.48004791360314   1.65963305120687    -0.320831767749357      0.0301751027304058      -0.00109753876470597    0.95128581940459    -1.17810886635864   0.265881136132843   -0.0173464028667221     -1.60178456484762       0.617690745752307       -0.12962820451389   0.0103267074585481      0.430324307120204       -0.123637424405877      0.0138126452330245      -0.00135095613804197    -0.0693983454142194     0.00848405659840599     0.00268376953089322     -0.0002132469908206     0.3801 0.4540       0.59211    0.55592 0.45194 0.39816 5   7   16   36
0.05    -5.44852161353544   7.34123453650929    -4.46066916205285   1.65027521868093    -0.319262247877148      0.0300548708239151      -0.0010939297337486     1.74586696638944    -1.54784764252322   0.322375933373927   -0.0202012604783158     -2.75050413148732       1.12375808272528        -0.204426625437094  0.0140779958393308      1.10627015090463        -0.390190878539361      0.0512718283256026      -0.00321030505003971    -0.20095167015803       0.0548537739454678      -0.00349931480869811    8.85307165976623e-05    0.3855 0.4530       0.59483    0.55204 0.44903 0.40592 5   7   16   36
0.1     -6.52412611443117   7.88030001761649    -4.59619757418892   1.67533631599973    -0.322436383996386      0.0302789264324569      -0.0011002810235287     4.44197212201997    -2.51134368136621   0.434879814872928   -0.0243148728888436     -6.04533567471067       2.20186237792801        -0.327224287199392  0.0185559667941668      2.74434842261766        -0.854026448815246      0.101865244192642       -0.00502059645990858    -0.468037476201747      0.120539732354275       -0.0105257012590167     0.000341484051711837    0.3864 0.4500       0.59312    0.54000 0.44000 0.43000 5   7   16   36
0.15    -7.05622002004159   8.257857447919      -4.77023445563363   1.72145067141602    -0.32898815301739       0.0307452094335939      -0.00111332700780617    4.37511791196785    -2.46929345029332   0.427799512051733   -0.0238992541605638     -5.90186743805531       2.20478678664531        -0.339213957598066  0.0195924614528765      2.65143410861053        -0.871149252054027      0.114686940253224       -0.00605822378961742    -0.442911681113393      0.126440191260781       -0.013978302378315      0.000612262063066027    0.3841 0.4675       0.60507    0.58095 0.47510 0.40075 5   7   16   36
0.2     -7.5996060317478    8.70114862117717    -4.99642662176284   1.78460081566883    -0.338292633379899      0.0314281348226792      -0.0011329964927681     4.24693959198696    -2.42889790442688   0.42396564691317    -0.0237876114947005     -5.7094437168142        2.20129215145471        -0.348691921769615  0.0204457887449959      2.54050909641514        -0.877182842404611      0.122188596672269       -0.00668254379642361    -0.418183069613773      0.128483394773764       -0.01563343660052       0.000745094101886859    0.3690 0.4800       0.60546    0.61000 0.50000 0.38000 5   7   16   36
0.25    -8.16515889359014   9.22196762242501    -5.2804106910865    1.86635804179714    -0.350580649626921      0.0323457873383452      -0.00115991040105017    4.14003700805627    -2.39747301960981   0.421299003528677   -0.0237322739736342     -5.55825268443305       2.20071774363609        -0.356401314895201  0.0211327279663628      2.45490692318939        -0.880718335672572      0.127410918423448       -0.00712619288197632    -0.399672815116889      0.129264959120613       -0.0166222655540094     0.000828157230752506    0.3445 0.4800       0.59082    0.62651 0.50000 0.37450 5   7   16   36
0.3     -8.7474293531944    9.81369091452027    -5.61423803423783   1.96297622028117    -0.36506466621779       0.0334218452653572      -0.00119126599198386    4.02179833187207    -2.35571441019312   0.41634141711421    -0.0235309981235541     -5.3981708910902        2.17888139835287        -0.3587072652945    0.0214578953917803      2.36721873104903        -0.871140544385342      0.12918698959393        -0.00733831884177811    -0.381854339538177      0.127407371654419       -0.0169046896094578     0.000863859442328111    0.3377 0.4800       0.58688    0.64000 0.50000 0.37000 5   7   16   36
0.35    -9.36352601714453   10.4841441795925    -5.99675559409738   2.07302325856421    -0.381429835786961      0.0346286147228768      -0.0012261927706556     3.9173517621193     -2.32163873654399   0.41287912306817    -0.0234269207048582     -5.26573392059597       2.16504548218842        -0.361968680885899  0.0218334175390872      2.29700753543452        -0.865434338190052      0.131277110030579       -0.00756298708268857    -0.367958245255449      0.126186931337784       -0.0172179120711643     0.00090007502865065     0.3482 0.4740       0.58812    0.62793 0.49396 0.37000 5   7   16   36
0.4     -9.9510467343533    11.1396821698705    -6.37220935201203   2.1795124481153     -0.396896214868145      0.0357341618677646      -0.001256956305176      3.78408312183875    -2.26961071365006   0.405960325331216   -0.0231120743590497     -5.1082234865418        2.12924277268091        -0.360516253497031  0.0219092394238217      2.21755570546748        -0.849115972498674      0.130986252922268       -0.00763468415619269    -0.353149588622868      0.123180881816778       -0.0171184974987632     0.000909673477475852    0.3552 0.4687       0.58811    0.61747 0.48874 0.37000 5   7   16   36
0.45    -10.5049929651735   11.7741238021454    -6.7344839466472    2.28050953869054    -0.411199643228366      0.0367232007307833      -0.00128329182119924    3.64243463012034    -2.21489156835578   0.398876506858552   -0.0228035391804558     -4.95573864528768       2.09411078511887        -0.359135603006229  0.0219923605500976      2.14491985557511        -0.834405413257492      0.130867713325486       -0.00771411769354598    -0.340176683746983      0.12059168625972        -0.0170651134017167     0.000921163703530932    0.3495 0.4641       0.58097    0.60825 0.48413 0.37000 5   7   16   36
0.5     -11.0079159807579   12.3672856738244    -7.0729194221941    2.37301711941519    -0.423883425210711      0.0375603011991471      -0.00130410508070119    3.44888639403781    -2.13236259767113   0.387012989998316   -0.0222295887730711     -4.7498123950679        2.02557539470689        -0.351478820125135  0.0217048468112582      2.0478960332513         -0.80296070872822       0.127461030180691       -0.00759317139530524    -0.323318274639238      0.115068295217205       -0.016419655343717      0.00089587471160287     0.3548 0.4600       0.58095    0.60000 0.48000 0.37000 5   7   16   36
0.55    -11.4513935986291   12.8957494956643    -7.37233292031199   2.45285733419295    -0.434391560052119      0.0382108353332891      -0.00131862108093279    3.26716678312708    -2.06142521811069   0.37792665850691    -0.0218482162161893     -4.59193755319491       1.98833631346757        -0.350281973141587  0.0218370457506449      1.98559139096896        -0.79322463991216       0.128464092179511       -0.00775712786147724    -0.313831978914929      0.114060357974105       -0.0166918934683435     0.000930038395081245    0.3639 0.4586       0.58544    0.59175 0.47587 0.37413 5   7   16   36
0.6     -11.8115140338191   13.3316685518971    -7.61810558255943   2.51573334700968    -0.44201191188041       0.0386133169635105      -0.00132469998905444    3.01979134283091    -1.95474881343212   0.362575952866341   -0.0211112365543991     -4.35965175795068       1.90526498710402        -0.340378466056744  0.0214448642863212      1.88694669904354        -0.759587698081503      0.124742265932089       -0.00762779809257124    -0.298121197865199      0.108743594986114       -0.0160920729527879     0.000909058381946342    0.3757 0.4574       0.59187    0.58422 0.47211 0.37789 5   7   16   36
0.65    -12.1005740726112   13.6826608503419    -7.81346067552053   2.56290572102359    -0.447038748535948      0.0388005289230028      -0.00132370399052105    2.7568520634726     -1.84159352183002   0.346390400819921   -0.0203413083769196     -4.12453416703918       1.81976729403999        -0.330097410322513  0.0210386778698326      1.79187421285957        -0.726915811830917      0.121179816405151       -0.00751073431635188    -0.283673017846181      0.103887753683876       -0.0155730477015491     0.000893420043099699    0.3831 0.4562       0.59573    0.57729 0.46864 0.38136 5   7   16   36
0.7     -12.3109567232122   13.9374585622562    -7.95224370087393   2.59266822233641    -0.449213758104868      0.0387520042658699      -0.00131496631330216    2.45915614157517    -1.71016801638964   0.327105736911732   -0.0194008221240958     -3.85374529143071       1.71142621264331        -0.315449396619633  0.0203715275572488      1.6808853987807         -0.682969027562948      0.115354166837439       -0.00725411616953646    -0.266700344471063      0.0971012670645442      -0.0146565571628196     0.000852866622525402    0.3886 0.4551       0.59846    0.57087 0.46544 0.38456 5   7   16   36
0.75    -12.4430899061033   14.0918595490324    -8.03168028206555   2.60436491256866    -0.44847113633096       0.0384655549227566      -0.00129850896270111    2.13772597875988    -1.56569558384919   0.30554038651569    -0.0183319996057231     -3.55773517509166       1.58516592528114        -0.297217084610243  0.0194851589344218      1.55848915262146        -0.629938097054961      0.107630241136342       -0.00687857668414035    -0.247924381067222      0.0887572552428164      -0.0134083380083834     0.000791322363833641    0.3852 0.4542       0.59554    0.56490 0.46245 0.38755 5   7   16   36
0.8     -12.5171126902419   14.1716286874052    -8.06652355371077   2.60250315233706    -0.445562739835801      0.0380054960167312      -0.00127653544479936    1.79781152210907    -1.41055096023825   0.282053000980934   -0.0171529814265426     -3.23803736633912       1.44123732473951        -0.275375157154408  0.0183755529679983      1.42411996970592        -0.567221411003441      0.0978656854051505      -0.00637459423495403    -0.22710094566025       0.0786749065456992      -0.0117896748019415     0.000706293692065911    0.3850 0.4532       0.59468    0.55932 0.45966 0.39034 5   7   16   36
0.85    -12.5361305905871   14.1726816657033    -8.05333633029004   2.5862070364439     -0.440394477039911      0.0373680070878932      -0.0012490304635134     1.48058273174033    -1.26733076646808   0.260611217091795   -0.0160882027576548     -2.94966953626325       1.31071219747759        -0.255497136042441  0.0173633850935993      1.30646464990962        -0.511300039006446      0.0890548067092703      -0.00591628688982375    -0.209293203012138      0.0697986632092572      -0.0103400228313192     0.000629377745841757    0.3811 0.4523       0.59150    0.55407 0.45703 0.39297 5   7   16   36
0.9     -12.5018888851015   14.1011127487731    -7.99676398481824   2.55695353635112    -0.433198048458534      0.0365709433887836      -0.0012165361272979     1.15290336681505    -1.11637966585331   0.237581723417742   -0.0149245983224433     -2.64193172366657       1.16358416768122        -0.232033843942633  0.0161226883076199      1.17753073727502        -0.445591210610168      0.0781246651684888      -0.00532266187649071    -0.189403086985565      0.0591087861955487      -0.0084989795533852     0.000527501739264048    0.3799 0.4515       0.59006    0.54912 0.45456 0.39544 5   7   16   36
0.95    -12.4050432991518   13.9397194799698    -7.88659200643668   2.51181341951269    -0.42352547661556       0.0355790386543333      -0.00117792755938229    0.824477618421439   -0.962113020251344  0.213610415375335   -0.0136927236337321     -2.32469759035956       1.00430975173967        -0.205618154013454  0.0146819862212142      1.04157231695976        -0.371985685347743      0.0653336485681453      -0.00460469529023078    -0.168076491171767      0.0468877087114078      -0.0063033146489028     0.000402109761311457    0.3733 0.4507       0.58525    0.54444 0.45222 0.39778 5   7   16   36
1       -12.2742324663155   13.7262825167811    -7.74428684955111   2.45720245764044    -0.412419761691176      0.0344792367956571      -0.00113610979954935    0.516744754789165   -0.817039184088801  0.191002133469808   -0.0125280945201615     -2.02754170522109       0.85099606894276        -0.179700797868634  0.0132493375969447      0.91437619634635        -0.300382270280627      0.052598663535233       -0.00387932068159755    -0.148153742167263      0.0349428202065387      -0.00410626832775152    0.000274893209557219    0.3716 0.4500       0.58362    0.54000 0.45000 0.40000 5   7   16   36
1.05    -12.0765528622753   13.4127249061026    -7.54219551058911   2.38495833544192    -0.39857225041734       0.0331633624543223      -0.00108747618084555    0.219331318917039   -0.674242855607305  0.168376134892499   -0.0113452743261151     -1.73311849352833       0.692466766949586       -0.152074554430091  0.0116883116344451      0.785855787856854       -0.224337219254882      0.0386481647314346      -0.00306773285810787    -0.127729968106646      0.0220625413667527      -0.00166864720857094    0.000130993550184928    0.3691 0.4482       0.58063    0.53956 0.44778 0.40000 5   7   16   36
1.1     -11.9148716272066   13.1509284513051    -7.37034543582023   2.32233635966212    -0.386432556520028      0.0320040479098914      -0.00104457943357097    -0.0606701586013031 -0.537435879864067  0.146372130563701   -0.010180093702489      -1.45134471659573       0.534533157929006       -0.12383889548552   0.0100656215969973      0.661481842804365       -0.147222714247         0.0241445463452034      -0.00221141619712539    -0.107805267295671      0.00887830278186186     0.000883633763360638    -2.15949654686717e-05   0.3650 0.4465       0.57674    0.53913 0.44566 0.40000 5   7   16   36
1.15    -11.6857306201776   12.7801288971462    -7.13140208498355   2.23951488157151    -0.371083178887649      0.0305851406125003      -0.000993239149195347   -0.301691222525331  -0.419822048211738  0.127498019869207   -0.00918336803484775    -1.21067055088919       0.395158976145363       -0.098462707545683  0.0085915483758236      0.555689298033574       -0.07828212943575       0.0108828970681648      -0.00141921456491637    -0.0908906954676145     -0.00298539068373565    0.00323374311454854     -0.00016365659466901    0.3627 0.4449       0.57402    0.53873 0.44364 0.40000 5   7   16   36
1.2     -11.4541092565961   12.4032533103694    -6.88894786312178   2.15625683891643    -0.355814960161282      0.0291870598154267      -0.000943061190897703   -0.525716869740914  -0.307575793733774  0.109071997340956   -0.00819127105989819    -0.980970672184559      0.25538968260991        -0.0722529381064995 0.00703947930449164     0.452846182078298       -0.00765445865618715    -0.00306360760667127    -0.00057271745551994    -0.0742349405669248     -0.0152688868297785     0.00572259540872141     -0.000316161116699763   0.3625 0.4434       0.57267    0.53834 0.44170 0.40000 5   7   16   36
1.25    -11.1941607969915   11.9794399982197    -6.61817339723534   2.06495450240813    -0.339372075560515      0.0277033984025592      -0.000890415998431553   -0.723951192823106  -0.206211498365503  0.092163159677894   -0.00726913832681148    -0.775045426114116      0.123796991629337       -0.0469367676693578 0.00551762321828858     0.359856538398857       0.0598899617441873      -0.0167167785087962     0.000266107131155585    -0.0590835886323192     -0.0270945767272141     0.00816835521535907     -0.000467511537662174   0.3631 0.4419       0.57195    0.53797 0.43984 0.40000 5   7   16   36
1.3     -10.9281674854766   11.5406221957995    -6.33747784956692   1.97099559361677    -0.322600970572445      0.0262021050076581      -0.000837491923580335   -0.895372511509658  -0.116218599698084  0.0768424154096393  -0.00642017949436338    -0.593384547400914      0.000536594906970235    -0.022506703419724  0.00402344059740673     0.276543881606182       0.124589539595521       -0.0301539738991734     0.00110374442644457     -0.0453384496694919     -0.0385575113362283     0.0105972688959971      -0.000619757049033733   0.3610 0.4404       0.56949    0.53761 0.43806 0.40000 5   7   16   36
1.35    -10.6682922691723   11.1073965797411    -6.06014246351386   1.87860910387472    -0.306215667196245      0.0247445516238038      -0.000786404185373799   -1.04824780519287   -0.0326199550749627 0.0621754148776888  -0.00558869288638254    -0.424677265401781      -0.121008098131977      0.00225886629584289 0.00248484945853664     0.196943299099645       0.189874471231656       -0.0439990642967797     0.00197647177449426     -0.0319447112763408     -0.0502550551249779     0.013116202878655       -0.000778991846892403   0.3605 0.4391       0.56808    0.53727 0.43634 0.40000 5   7   16   36
1.4     -10.3807643373144   10.6260328991388    -5.75340122655126   1.77775605097086    -0.288566129785058      0.0231917395961965      -0.000732439831582717   -1.16599065460799   0.0342265484563125  0.0501538632692863  -0.00489531978651416    -0.294650182669392      -0.224397640240418      0.0241517773429506  0.00109801447008383     0.135719504569212       0.246297977935709       -0.0563988404933046     0.00277068551250847     -0.021648705085713      -0.0604117319871057     0.0153761740025805      -0.000923824408726993   0.3616 0.4377       0.56779    0.53694 0.43469 0.40000 5   7   16   36
1.45    -10.0936831201909   10.1397508524691    -5.44329088127328   1.67632978279449    -0.270933563642684      0.0216500852845427      -0.000679155143980448   -1.25349264085351   0.0871467643005023  0.0402611988320187  -0.00430989364232765    -0.197390568473098      -0.312823153541416      0.0437268478392032  -0.000167886487557549   0.0897635336194282      0.295507439588151       -0.0676364719519399     0.00350180708252567     -0.0138828696169615     -0.0693274957342874     0.0174289149131502      -0.00105705603424553    0.3604 0.4365       0.56604    0.53662 0.43309 0.40000 5   7   16   36
1.5     -9.79907863171745   9.63810038764214    -5.12421695685638   1.57277273835917    -0.253081686558323      0.0201010842731723      -0.0006259638791232     -1.31656838023916   0.129316442959523   0.0319348248127351  -0.0038001896148408     -0.123907040341741      -0.391427474505999      0.0619435728944503  -0.0013709123011851     0.0540758360339362      0.340437706891422       -0.0782722350276823     0.0042045009295633      -0.00773354491942493    -0.0775470020415915     0.0193795691185345      -0.00118526092860621    0.3559 0.4352       0.56223    0.53631 0.43155 0.40000 5   7   16   36
1.55    -9.51902258607652   9.15393554401923    -4.81497736422202   1.47242679621646    -0.235815702379439      0.0186064236926197      -0.000574758325337779   -1.36601872661785   0.167493202457707   0.0238624234633978  -0.00328551164253108    -0.0581610078789893     -0.470249670911649      0.0807560769482691  -0.0026312890065295     0.0196471925298604      0.386768503021801       -0.0894185043959424     0.00494746207090532     -0.00151626512966664    -0.0861400223099207     0.0214390191972629      -0.00132152408022243    0.3507 0.4340       0.55804    0.53601 0.43005 0.40000 5   7   16   36
1.6     -9.25766355505242   8.6934076317561     -4.51975081386629   1.37679188581237    -0.219423309783575      0.0171938202412865      -0.000526589504993326   -1.38932705108318   0.193374492604112   0.0177102346623029  -0.00287030829964133    -0.0212436082506494     -0.535226836361921      0.0973360583131912  -0.00377185023360277    -0.00110241639504277    0.426214890593861       -0.0994080792120429     0.00562625845510098     0.002395365158518       -0.0935228495694928     0.0232883188290172      -0.00144584877243175    0.3479 0.4329       0.55537    0.53572 0.42861 0.40000 5   7   16   36
1.65    -8.98418859362607   8.21096704836861    -4.21164682533016   1.27773549869142    -0.202573926315765      0.015751362452288       -0.000477666908176797   -1.39056016158647   0.209352504812619   0.0130292498468628  -0.00252747864790209    -0.00727028934871961    -0.589806496746011      0.112329401290211   -0.00483144222749884    -0.0114732587707136     0.460720349485909       -0.10860596123418       0.00626284483280012     0.00462561347412669     -0.100065078919591      0.024997101424357       -0.00156243172491723    0.3486 0.4318       0.55495    0.53544 0.42721 0.40000 5   7   16   36
1.7     -8.72164567019667   7.74051890122459    -3.91044262454939   1.18111506741441    -0.18620107975438       0.014355360813705       -0.000430502021930003   -1.36791431524569   0.214098788961475   0.0100973626065154  -0.002274854509883      -0.0194275024118142     -0.63162119722956       0.125239035008665   -0.00577797992094778    -0.00960642215967662    0.488904061377424       -0.116721320481105      0.00683832453342026     0.00482073601682377     -0.105504910703578      0.0265102090353895      -0.00166767630068795    0.3485 0.4307       0.55401    0.53517 0.42585 0.40000 5   7   16   36
1.75    -8.4591650364867    7.26771940450535    -3.60814237298841   1.08457204290069    -0.169923214937892      0.0129739794533237      -0.000384026501752304   -1.33031044575269   0.212746150264374   0.00797196863905766 -0.00205673777038382    -0.0460115910083674     -0.667623125957788      0.137383258137235   -0.00669188117806307    -0.00153166432105993    0.514451560820034       -0.124471165199024      0.00739746345145762     0.00404164730915879     -0.110506306244208      0.0279594030782656      -0.00176994872583203    0.3470 0.4296       0.55226    0.53491 0.42453 0.40000 5   7   16   36
1.8     -8.18680160180047   6.77933303830351    -3.29739680274992   0.985960415766211   -0.15339689503341       0.0115788103930968      -0.000337292449871531   -1.28302807324426   0.207875366216806   0.00623717298315271 -0.00185122206942408    -0.0812071492304205     -0.700330699615899      0.149117231578729   -0.00758897866178782    0.0102566597484727      0.53828030482531        -0.131953821077504      0.00794256445311423     0.00264702100689937     -0.115169565684049      0.0293481305038763      -0.00186868137894298    0.3469 0.4286       0.55137    0.53465 0.42325 0.40000 5   7   16   36
1.85    -7.95929951740532   6.35231271844265    -3.02230459069294   0.898318936659162   -0.138712006501872      0.0103420762725451      -0.000296013168004773   -1.20581740268606   0.187646849996872   0.00708996709481565 -0.0017885234706443     -0.151995737564498      -0.71385827461499       0.157474372850563   -0.00829268424016534    0.0393216960586788      0.552422098211421       -0.137683688826036      0.00838540967065562     -0.00165663118201637    -0.118155524489811      0.0304281302176321      -0.00194922416639908    0.3448 0.4276       0.54931    0.53440 0.42200 0.40000 5   7   16   36
1.9     -7.72210224316808   5.91019734365851    -2.73897158359813   0.808598811054706   -0.123762141693261      0.00908888230741114     -0.000254344217578371   -1.12063290146282   0.164696641788704   0.0082061673608143  -0.00173147127366326    -0.230196148134714      -0.724620457439465      0.165514128047599   -0.00898578213689036    0.0717824601247428      0.565008095656521       -0.143185545117812      0.00881756282548822     -0.00654904111015766    -0.120826324687597      0.0314554311796031      -0.00202697310184471    0.3435 0.4266       0.54775    0.53416 0.42079 0.40000 5   7   16   36
1.95    -7.497922238966     5.48571492991086    -2.46636900808452   0.722435128483289   -0.109452012430718      0.00789383483969574     -0.000214762021453911   -1.0227044387956    0.135814596642727   0.0102404503698089  -0.00172163006768318    -0.321692883083363      -0.728522501013522      0.172381054537624   -0.0096129107276157     0.110253260980422       0.574165220496862       -0.148060544308446      0.00921285559513499     -0.0124126615685703     -0.122898701132766      0.0323685618389673      -0.00209784689313667    0.3422 0.4257       0.54621    0.53392 0.41961 0.40000 5   7   16   36
2       -7.28134109858198   5.07114642593926    -2.20006384737329   0.638509230187268   -0.095568172040893      0.00673913784937326     -0.000176669069460097   -0.911706629753733  0.100903384382481   0.0131940716580957  -0.00175817320822749    -0.428266390415649      -0.724679969696029      0.177943236004371   -0.0101680601740573     0.156109305406398       0.579195273801759       -0.152199894432282      0.00956597516201906     -0.0195482880573344     -0.124220201568343      0.0331435800802128      -0.00216065579105158    0.3412 0.4248       0.54482    0.53369 0.41845 0.40000 5   7   16   36
2.5     -5.66493605200896   1.72326725100075    -0.0351237163912251 -0.0359968365453758 0.0139987163789643      -0.00218519264327732    0.000111399694081619    0.616640989637283   -0.448674903162923  0.074407363184412   -0.00377521280631313    -1.97327716076029       -0.409669214502711      0.183094338795595   -0.0127758531470834     0.851342580669687       0.475697676453893       -0.16377070941898       0.0112981566929394      -0.132107366072133      -0.108732354686175      0.0351624117862608      -0.00243829468662888    0.3321 0.4166       0.53278    0.53166 0.40830 0.40000 5   7   16   36
3       -4.85391128584776   -0.247991697726551  1.23763207934956    -0.420239133115993  0.0735589797398048      -0.00676355003869186    0.000249574487344593    1.82374272989116    -0.808702937096147  0.100496493085643   -0.0037273970710154     -2.99383572377122       -0.339088599331491      0.227367652241774   -0.0174986252775561     1.25440440618792        0.494763969182138       -0.193908245387153      0.0140255477064407      -0.191562841128241      -0.114402115817218      0.0403335196779162      -0.00288801557418332    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
3.5     -4.72725789440957   -1.14259723287439   1.85137710597002    -0.598757736755993  0.0988170510255515      -0.00844349382752947    0.000290310043901488    3.49489480788702    -1.5245457237196    0.20288421123207    -0.00862244592198293    -4.90916770989974       0.446917781192768       0.115781860196384   -0.0120408917326846     2.21168491570713        0.0702692439769532      -0.127611294777349      0.010465295285617       -0.359337036753886      -0.0338378188276011     0.0268179394569034      -0.00212181526815231    0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
4       -4.69486154906571   -1.60442499439056   2.1261113361808     -0.662638085842482  0.104570064604275       -0.00847623624791257    0.000275959845677309    4.09200993200369    -1.62818593468563   0.193068215807852   -0.00689210777799705    -5.30982824812619       0.370486413181534       0.164831698509239   -0.0162754366461803     2.34264341078942        0.124009445330578       -0.152282074238383      0.0125022227773353      -0.375620245251544      -0.0424130825701055     0.0305346670543767      -0.00243002964224622    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
5       -6.21513368530134   -0.311600674770361  1.44163969243601    -0.434349329661122  0.0605163523672389      -0.00411897287698779    0.000106866888293571    7.0507209276198     -3.10634364640467   0.43699954960609    -0.0201356203462683     -9.06721089396862       2.36576024168723        -0.184062802520817  0.00364500801659908     4.31318014698661        -1.00363855305046       0.0546649057368454      0.00031676950580385     -0.729162521811405      0.170952235597322       -0.00974411236369846    -2.12686273377959e-05   0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
6       -7.67523057792267   1.49258012101564    0.334610916732413   -0.0711890912665074 -0.00449483589794548    0.0018215817046671      -0.000108725895971355   7.90328602156591    -3.55082887612656   0.513688061188794   -0.0244761970352626     -10.1114982198768       3.02552796497825        -0.312475958667375  0.0114859001311475      4.87432674610762        -1.41564558895876       0.139064317345564       -0.00491311337094721    -0.832023298733121      0.253038334452963       -0.0268695484568766     0.00104012041308098     0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
7       -9.0752402828588    3.41461178841863    -0.861321902706698  0.313577749650426   -0.0713504547657582     0.00774590118778419     -0.000317753786732739   8.19038036365461    -3.78208118895684   0.564579215659887   -0.0278232236051886     -10.5219415223574       3.46054282398411        -0.414489457707859  0.0182955246309285      5.11936750559427        -1.70379497754436       0.206238135218458       -0.00930033252880361    -0.879237572633405      0.310897907357215       -0.0402008121455911     0.00189519156606616     0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
8       -10.2727254211356   5.15152963562668    -1.95059622931702   0.660350633238254   -0.130601918255729      0.0129015641069298      -0.00049644234570486    8.13516999065011    -3.8770506347392    0.597501687353903   -0.0303829854729913     -10.5338281384062       3.73075834505631        -0.492687509335659  0.0239139854165329      5.15754244343945        -1.89361165627928       0.257040904874272       -0.0127656691916866     -0.889039827067333      0.348959790783433       -0.0499742881726843     0.00254019247606934     0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
10      -11.9364654874712   7.69701806383256    -3.56524547690055   1.16913255093572    -0.215972168569508      0.0201751879449513      -0.000743008256826006   7.4885926958246     -3.82276360580006   0.625447003754857   -0.0335178838563254     -9.88928440872158       3.92205669671066        -0.587080806386783  0.0315134724726338      4.86878122646019        -2.05283443148545       0.316097357476342       -0.0170879851466899     -0.840810234946994      0.380324378921919       -0.0605634517510679     0.00327384056736418     0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
    """)

COEFFS_ALPINE_120Bars = CoeffsTable(sa_damping=5, table="""\
IMT     a1                  a2                  a3                  a4                  a5                  a6                      a7                      a8                  a9                  a10                     a11                     a12                 a13                 a14                     a15                     a16                 a17                 a18                 a19                     a20                     a21                     a22                     a23                     tau    mean_phi_ss  sigma_tot  phi_11  phi_21  C2      Mc1 Mc2 Rc11 Rc21
pgv     -9.16614124176187   8.5536116023133     -4.83801215712285   1.69646845841656    -0.316475757452586  0.0290456295529755      -0.0010365105877135     3.00084848981711    -1.88063949619261   0.341881364594834       -0.0196537927672152     -4.16813220025304   1.33273681480934    -0.187823077361569      0.0109532985036027      1.7378616158844     -0.418377516891457  0.0323324748670333  -0.00113966828542721    -0.270987698668834      0.0456798358701235      0.00147741818510175     -0.000341725031210552   0.5010 0.4600       0.68015    0.00000 0.00000 0.00000 5   7   16   36
pga     -6.88231352240753   8.40625353693458    -5.02569957433775   1.8245241479402     -0.348401010169869  0.0325002227525965      -0.00117461090068465    2.66561159602939    -1.7411830960781    0.323174084423307       -0.0189178205940999     -3.75832717085326   1.14164029376429    -0.159931240859824      0.0097784611508645      1.51831429080794    -0.32233618123231   0.0172352952193103  -0.000438180686294045   -0.242818410393417      0.0319456924920846      0.0037277119371228      -0.000453612993375562   0.3532 0.4600       0.57998    0.58000 0.47000 0.35000 5   7   16   36
0.010   -6.87318757037523   8.40024456493794    -5.01940592721014   1.82277530109803    -0.348178008671882  0.0324862553980003      -0.00117428737340175    2.79416275382463    -1.79660537539846   0.331118284000246       -0.0193012830523281     -3.91588736586921   1.21168238426998    -0.171046276209854      0.010387198912652       1.58638298203658    -0.35525974595253   0.0226896916252815  -0.000745505565092869   -0.253002963925942      0.0372659555522727      0.00282958084788079     -0.000402832861161009   0.3529 0.4600       0.57975    0.58000 0.47000 0.35000 5   7   16   36
0.020   -6.58987139547207   8.22092930974043    -4.92962534898201   1.80081838330131    -0.345491127319363  0.032331616349931       -0.00117132258805061    3.04782126748864    -1.88461847943297   0.34087453043457        -0.0196672145078261     -4.16373476827779   1.30111316332671    -0.181900580442754      0.0108398961432164      1.61845071998755    -0.385304517287491  0.0266232133902474  -0.000867851797636056   -0.245913419967209      0.0405981658641112      0.00236004534759621     -0.000398167328554953   0.3674 0.4570       0.58639    0.56796 0.46097 0.37408 5   7   16   36
0.030   -5.22644968640237   6.9388152880075     -4.29846438021748   1.62785800136013    -0.319152483234729  0.0302616671698398      -0.00110603652284104    1.04824695844754    -1.02555453027557   0.216063537723369       -0.0136105080821688     -1.40376699940501   0.186423328653234   -0.0212256633217706     0.00287179660094089     0.163949047248236   0.179477706906054   -0.0557713974677912 0.00332498115823577     0.00692544413829601     -0.0564963884115238     0.0167971463693446      -0.00114882185820062    0.3749 0.4552       0.58975    0.56092 0.45569 0.38817 5   7   16   36
0.040   -5.23201953024704   7.23339398974315    -4.50404781879414   1.69277117920178    -0.329996091802119  0.0311932920593143      -0.00113829234724488    0.481045409036451   -0.885707680018511  0.209930043595803       -0.0139786291705883     -0.796968317851684  0.100157450787583   -0.0286779490741387     0.00416676954989564     -0.0443116655120648 0.187393764919147   -0.0478304561149145 0.00245726292112268     0.0240558501061482      -0.0535036992695517     0.0150732910776705      -0.000983264475647864   0.3801 0.4540       0.59211    0.55592 0.45194 0.39816 5   7   16   36
0.050   -5.21327557067981   7.0507115002086     -4.3572559319188    1.64602533032402    -0.322521917464684  0.030594343023134       -0.00111903113522958    1.33820987769818    -1.30624390627387   0.27793587653618        -0.0176160661507944     -2.05949471025801   0.696608396733644   -0.124052961960103      0.00933016756461845     0.697029362456066   -0.131086473012225  0.00160206040008883 -0.000233839656604314   -0.119777634503338      0.00261853528938404     0.00662195270576827     -0.000522698680183309   0.3855 0.4530       0.59483    0.55204 0.44903 0.40592 5   7   16   36
0.100   -6.67753989040742   8.133702846968      -4.80452894487227   1.76252928825617    -0.340137306724264  0.0319865779938396      -0.00116337388906605    4.3318878600331     -2.44062785825229   0.421716546714264       -0.0235615545333576     -5.82642849352793   2.05900066071827    -0.300891271487652      0.0170774162256147      2.60953952374302    -0.764634195841848  0.0851233526900471  -0.00407007766216204    -0.441182441699788      0.102566998826145       -0.00712879863977291    0.000147274229079961    0.3864 0.4500       0.59312    0.54000 0.44000 0.43000 5   7   16   36
0.150   -7.2625568429117    8.61812456162286    -5.04914082963358   1.82986094151356    -0.349930291433608  0.0326989237004075      -0.00118380323240411    4.28005143117857    -2.40946405540804   0.416774392917208       -0.0232723990870367     -5.72921983913444   2.09393667682088    -0.318910632564168      0.0184577266753677      2.55143330340099    -0.805669452159823  0.102432256229909   -0.00536295325826587    -0.424161053876695      0.114009814338464       -0.0116179469462549     0.000476856045685082    0.3841 0.4675       0.60507    0.58095 0.47510 0.40075 5   7   16   36
0.200   -7.90693476394091   9.23215311216998    -5.37829423077317   1.92230383980049    -0.363532233836641  0.0336973188024521      -0.00121265999114315    4.13684805783038    -2.36024169448298   0.411233518496458       -0.023054308744793      -5.52794295172324   2.08590024742388    -0.3273574551384        0.0192331512876494      2.43932277627867    -0.811609353845033  0.109811492550331   -0.00596956641853677    -0.399720428340695      0.116365575080385       -0.0133111314329841     0.000609828897100166    0.3690 0.4800       0.60546    0.61000 0.50000 0.38000 5   7   16   36
0.250   -8.59782970337955   9.94236601228695    -5.76893682427372   2.03291251450105    -0.379865478610532  0.034898376604015       -0.0012474044390243     4.04453149843479    -2.33762582395532   0.41015565205322        -0.0230879673044698     -5.40301680951655   2.10140706644166    -0.337886319123369      0.0200704812146336      2.36901723557562    -0.824621720254674  0.116704593475266   -0.00650247969649903    -0.384076668620015      0.118933349651199       -0.0146156156114991     0.000709768995520301    0.3445 0.4800       0.59082    0.62651 0.50000 0.37450 5   7   16   36
0.300   -9.24848396315529   10.6335968774866    -6.15371369545185   2.14107123955891    -0.395534057842911  0.0360177648050597      -0.00127854400129294    3.9295991760392     -2.29833281462809   0.405686846998309       -0.0229162691946264     -5.25268454602941   2.08629734157786    -0.34143236975777       0.0204649495695013      2.28786964227256    -0.819579909274144  0.119335417007369   -0.00676332854284225    -0.367595978632842      0.118011581525625       -0.0150780300438799     0.000755930775787992    0.3377 0.4800       0.58688    0.64000 0.50000 0.37000 5   7   16   36
0.350   -9.90794992413042   11.3604598123555    -6.55921220728459   2.25380010625706    -0.411612288086852  0.0371445739020757      -0.00130914228725859    3.82301112633331    -2.26309259492895   0.401932528734953       -0.0227880724622289     -5.12086447635212   2.07307241642758    -0.344663080488718      0.0208264826826657      2.21900792221203    -0.814918274184565  0.121564714694576   -0.00699099885267551    -0.354045796336756      0.117054809098933       -0.0154341739243421     0.000793975414356066    0.3482 0.4740       0.58812    0.62793 0.49396 0.37000 5   7   16   36
0.400   -10.544952957825    12.074209208925     -6.95642600171417   2.36241633538823    -0.426722819533795  0.0381684543184724      -0.00133568559665311    3.7002821407703     -2.21832057159496   0.39645531765856        -0.0225615249020652     -4.97500794600823   2.0449180699653     -0.344578121044841      0.0209754778069195      2.14456602758527    -0.801797327074708  0.121803471729975   -0.00708731362948392    -0.339959426689831      0.114497099224227       -0.0154028373476264     0.000806120761989308    0.3552 0.4687       0.58811    0.61747 0.48874 0.37000 5   7   16   36
0.450   -11.1360560742848   12.7469186203387    -7.32849856884132   2.46209155426334    -0.440155032758338  0.0390373704532479      -0.00135667006040652    3.56648449128622    -2.16892568040306   0.390412651001782       -0.0223159104897273     -4.83227633408681   2.01633284235429    -0.344433973896036      0.0211311289578695      2.07699289821331    -0.79054950672133   0.122357194716668   -0.00720748406463474    -0.327876069662287      0.112527518002608       -0.0154734238739806     0.000825376699375696    0.3495 0.4641       0.58097    0.60825 0.48413 0.37000 5   7   16   36
0.500   -11.5782532183455   13.2306297959114    -7.5898297506077    2.52758776011141    -0.447905174227307  0.0394259386309821      -0.00136146310333963    3.39398834769963    -2.10025543738246   0.381279764064141       -0.02190968713048       -4.65750350947963   1.96854394418435    -0.340910345593747      0.0211009527973719      1.99801643636259    -0.771347066937364  0.121447062768427   -0.00724559068178984    -0.31448271310126       0.109389334668168       -0.0153235724125615     0.000832231620826125    0.3548 0.4600       0.58095    0.60000 0.48000 0.37000 5   7   16   36
0.550   -12.0288635757164   13.7540541026542    -7.87476390627586   2.59943567443721    -0.456585433552316  0.0398876502129199      -0.00136867999362645    3.19559101736757    -2.01913005292344   0.370205689374764       -0.0214042049980799     -4.4658008638113    1.90923808632235    -0.335207153634653      0.0209427739869933      1.91450538325071    -0.747501775229503  0.119537314458273   -0.00722032414500195    -0.300776841240848      0.105541939176752       -0.0150059728926899     0.000827887438664785    0.3639 0.4586       0.58544    0.59175 0.47587 0.37413 5   7   16   36
0.600   -12.3478647409335   14.1104905917809    -8.06248113749968   2.64169912111541    -0.460438158200089  0.0399494530428721      -0.0013626886478371     2.95541118623595    -1.91721711772914   0.355783702060018       -0.020723293842429      -4.23734018755499   1.828390291037      -0.325681290364564      0.0205706380062457      1.81696052889521    -0.714443115874896  0.115911698734019   -0.00709618193349775    -0.285208384661681      0.100300598999046       -0.0144212942984968     0.000807918903072094    0.3757 0.4574       0.59187    0.58422 0.47211 0.37789 5   7   16   36
0.650   -12.572007118494    14.3497862817049    -8.18213851316735   2.66336049309439    -0.460968248269161  0.0397403932646978      -0.00134791297205506    2.68405081320806    -1.79985745926972   0.338887978003344       -0.0199132864185143     -3.9811842548623    1.73009759209177    -0.312928858811516      0.0200123479438653      1.70870154055075    -0.673505500200128  0.110734481001627   -0.0068792688926357     -0.26818647585247       0.0938109941912179      -0.0135821004560975     0.000772471788707295    0.3831 0.4562       0.59573    0.57729 0.46864 0.38136 5   7   16   36
0.700   -12.6719583306285   14.4242591013082    -8.20532401193021   2.6560036897422     -0.456839457293426  0.0391514153799975      -0.00132075557571953    2.38520517578459    -1.66815582628717   0.319580407509854       -0.0189717581690827     -3.6974027395739    1.61312018953104    -0.296528917188004      0.0192343694327479      1.58847100092288    -0.623311891482427  0.103644592078577   -0.00654314006022385    -0.249338231190447      0.0857510359174766      -0.0124087011256143     0.000715818095335365    0.3886 0.4551       0.59846    0.57087 0.46544 0.38456 5   7   16   36
0.750   -12.7784578390526   14.5262452498981    -8.24586784853504   2.65407235671588    -0.453674378266196  0.038653372516539       -0.00129707687758688    2.06134280833314    -1.52314076370208   0.298026697436629       -0.0179081786269961     -3.38693139907454   1.47811384640551    -0.27668574728628       0.018254638137844       1.45646607557875    -0.564280504567181  0.0948062840227606  -0.00610305399873436    -0.228691754909478      0.07622941504357        -0.0109426332835677     0.000641755399685129    0.3852 0.4542       0.59554    0.56490 0.46245 0.38755 5   7   16   36
0.800   -12.7243323286695   14.4021836273309    -8.15348241534848   2.61239887860485    -0.444158967144159  0.0376379899553784      -0.00125649395299139    1.72089058315549    -1.36828884808989   0.274667980976482       -0.0167396972365755     -3.05572671010391   1.326720520354      -0.253411675205454      0.0170594645758244      1.31422137822414    -0.496341002408718  0.0840400682229986  -0.00553954832397421    -0.206335633751607      0.0651280214016852      -0.00913035155346891    0.000545346771807197    0.3850 0.4532       0.59468    0.55932 0.45966 0.39034 5   7   16   36
0.850   -12.6864707141445   14.3080109585666    -8.07834554325701   2.57623844053653    -0.435664686351523  0.0367216430617685      -0.00121973881992266    1.39104248170768    -1.21825265373811   0.252057208892128       -0.0156103052492253     -2.73578904953856   1.17653956985951    -0.229826460244183      0.0158280081315446      1.17720575298056    -0.428114992721009  0.0728897028259892  -0.00494233605177319    -0.184845480615951      0.0538934512764694      -0.00723152918453222    0.000441755968298112    0.3811 0.4523       0.59150    0.55407 0.45703 0.39297 5   7   16   36
0.900   -12.5356260841217   14.0545763686803    -7.90983691982311   2.51241336335452    -0.422804260359985  0.0354545060034342      -0.00117168503948606    1.04864294588403    -1.05928350961702   0.227632856407207       -0.0143684591091277     -2.39699495078404   1.01042279395794    -0.202835342110489      0.0143812910175358      1.03052896084629    -0.351385844238455  0.0599115913263998  -0.00422963144933719    -0.161753160871153      0.0412085281243685      -0.00502179932384592    0.000318605155708569    0.3799 0.4515       0.59006    0.54912 0.45456 0.39544 5   7   16   36
0.950   -12.3814169656326   13.7971841034156    -7.73862633667527   2.44829785121776    -0.410055062257418  0.0342125323902299      -0.00112502019087709    0.72208336597255    -0.906440909614721  0.2039872632129         -0.0131589628302478     -2.06701328100821   0.842220135861636   -0.174678238766397      0.0128371008582957      0.884793746503138   -0.270951747060665  0.0457894214363457  -0.0034324674815453     -0.1384057752           0.0275808750712386      -0.0025528492536317     0.000176992855526843    0.3733 0.4507       0.58525    0.54444 0.45222 0.39778 5   7   16   36
1.000   -12.1676146798665   13.4526272998171    -7.51576155861625   2.36910518602362    -0.394995421770764  0.0327916742161048      -0.00107282242732455    0.406820017356657   -0.757122743064475  0.180640419500945       -0.0119535474584716     -1.74587735657122   0.673305385243094   -0.14578659785509       0.0112288848568811      0.742630239258612   -0.189491300280964  0.0311726396291028  -0.00259616888804187    -0.115647659883136      0.0137643318129016      4.5086718764472e-07     2.88597375967832e-05    0.3716 0.4500       0.58362    0.54000 0.45000 0.40000 5   7   16   36
1.050   -11.9245277086894   13.0646239179707    -7.26706824320951   2.28263175433863    -0.378890320350975  0.0312968733779208      -0.00101858558640135    0.100664022001482   -0.609164584558253  0.15708881018041        -0.0107183385738436     -1.4254554145016    0.497758829286232   -0.114919048113745      0.00947700615429062     0.598068840508308   -0.102855144259788  0.0152068191652948  -0.00166635271564916    -0.0922025893923401     -0.00111373643947663    0.00281691417146278     -0.000137163265654774   0.3691 0.4482       0.58063    0.53956 0.44778 0.40000 5   7   16   36
1.100   -11.6483493705586   12.6206521249509    -6.98394103201247   2.18628346937096    -0.361340179709505  0.0296971393503954      -0.00096134615778051    -0.157994148027099  -0.485408339127266  0.13759237963692        -0.00970595406912792    -1.16090940375929   0.34943724117127    -0.0885288881930855     0.00797035017580522     0.480929522664997   -0.0296371605876051 0.00147590838559497 -0.00086081492995395    -0.0734589589450654     -0.0136591939205188     0.00523758944835876     -0.000280804622257841   0.3650 0.4465       0.57674    0.53913 0.44566 0.40000 5   7   16   36
1.150   -11.3616171063633   12.1639437301144    -6.69475809040723   2.08893122292665    -0.343794547167699  0.0281123742850867      -0.000905074612173743   -0.421879704963737  -0.354189433024339  0.116199348849543       -0.00856115887526197    -0.876144876023262  0.182156874347576   -0.0578343794100757     0.00617916675162444     0.350044703980776   0.0553817190502535  -0.0148601349228659 0.000114820217638033    -0.0519704105666978     -0.0284686400684947     0.00815149995826046     -0.000456507937509765   0.3627 0.4449       0.57402    0.53873 0.44364 0.40000 5   7   16   36
1.200   -11.0566454239514   11.6729511665574    -6.38391424802944   1.98541818785332    -0.325369002967795  0.0264658297119943      -0.000847105542757712   -0.647244761843784  -0.241163792902081  0.0976544704147119      -0.00756396381178796    -0.632662406781527  0.0329491877340229  -0.0298268574437819     0.00452299702091522     0.238137083247113   0.132199652926703   -0.0299772208565398 0.00102859860940765     -0.0335918854147468     -0.0419234260427871     0.0108600189560659      -0.000621525932348628   0.3625 0.4434       0.57267    0.53834 0.44170 0.40000 5   7   16   36
1.250   -10.7321677953501   11.1480596619438    -6.05276230997602   1.8763941742622     -0.306197741189293  0.0247705693891114      -0.000787931186874093   -0.842097763284416  -0.141622557028377  0.081085061086166       -0.00666275730486558    -0.419785129537381  -0.104167620485206  -0.00342214319253199    0.00293810730046201     0.139782343043432   0.203782952918702   -0.0444086044431263 0.00191197246980686     -0.0173865554987793     -0.0545286708432123     0.0134539454399725      -0.00078126466596369    0.3631 0.4419       0.57195    0.53797 0.43984 0.40000 5   7   16   36
1.300   -10.4278889708149   10.6495969969037    -5.73707063130793   1.772520937498      -0.287981292108908  0.0231652471136226      -0.00073209371989631    -1.01674144262673   -0.0493686794243618 0.0653278721340639      -0.00578818037738292    -0.222950873973523  -0.238007303030042  0.0230516813537147      0.00132380760819421     0.0470091439351994  0.275025987788648   -0.0590942358205966 0.00282206741128537     -0.00188475414495378    -0.0671984553210433     0.0161106126771353      -0.000946575803973431   0.3610 0.4404       0.56949    0.53761 0.43806 0.40000 5   7   16   36
1.350   -10.111034836717    10.1272981957959    -5.40698521010364   1.66484700295067    -0.269277432347071  0.0215307803911315      -0.0006756387497751     -1.16105944413093   0.0290300649246445  0.051673049308285       -0.00501961212123395    -0.0578394501402944 -0.357905274805647  0.0474366973437478      -0.000184206041236076   -0.0312553820371068 0.339731612421415   -0.072762882511598  0.00367808927750534     0.0112271255423175      -0.0787478670311161     0.0185849142887571      -0.00110177306227943    0.3605 0.4391       0.56808    0.53727 0.43634 0.40000 5   7   16   36
1.400   -9.77416685921279   9.57161010490233    -5.0575064919266    1.55198377737767    -0.249872325184113  0.0198500442949403      -0.000618012425417171   -1.28235184162984   0.0982177170993213  0.0391981382033217      -0.00429900588408497    0.086442354169205   -0.470990925999608  0.0712010926382996      -0.00168189965520318    -0.101208354835734  0.402071803848566   -0.0862956927033671 0.00453897389002437     0.0231327270152153      -0.0899915485385934     0.0210519614906153      -0.00125874646408778    0.3616 0.4377       0.56779    0.53694 0.43469 0.40000 5   7   16   36
1.450   -9.45334123454595   9.03392053658778    -4.71803421749954   1.44251093528622    -0.231113833361836  0.0182314552200345      -0.000562716505724685   -1.37209272799703   0.152597853711788   0.0290378641431906      -0.00369875339407614    0.195882703053803   -0.567640854351956  0.0923278458434309      -0.00303696244914242    -0.154668208815057  0.456305483923134   -0.098469062503948  0.00532324961212576     0.0322667151341685      -0.0998151618205897     0.0232716066044842      -0.00140134398314583    0.3604 0.4365       0.56604    0.53662 0.43309 0.40000 5   7   16   36
1.500   -9.14172671470501   8.50710025063375    -4.38549379622975   1.33571607737975    -0.212910709326176  0.016669142195704       -0.000509611298113157   -1.44397935555193   0.200645791764493   0.0195472889780623      -0.00311735110785663    0.290621699285056   -0.660582277786018  0.113346838072497       -0.0044092204195848     -0.202806738645811  0.509614746599482   -0.110733566930294  0.00612385187327086     0.0406971953371182      -0.109566104532503      0.0255191081373913      -0.00154736676943011    0.3559 0.4352       0.56223    0.53631 0.43155 0.40000 5   7   16   36
1.550   -8.83090464981166   7.97543691959563    -4.04961335884487   1.22833707685146    -0.194714908777066  0.0151161941902662      -0.000457084717973791   -1.48156140324701   0.231949120011791   0.0127510053994563      -0.00267891648427218    0.346076446951212   -0.733895255337572  0.131107507914119       -0.00560256408639182    -0.232093164469257  0.553135183326935   -0.121318050838677  0.00682958001605857     0.0459481354703581      -0.11760958276336       0.027465621692818       -0.00167604090604266    0.3507 0.4340       0.55804    0.53601 0.43005 0.40000 5   7   16   36
1.600   -8.54191235554397   7.47225546373688    -3.73053283967895   1.12643414027683    -0.177498686030656  0.0136522793801596      -0.000407760255299841   -1.49455377599126   0.251878513605679   0.00770560076859913     -0.00232994482990853    0.373925341124824   -0.793915393888377  0.146709435388224       -0.00667915790096932    -0.248101022573522  0.58988236276727    -0.130745716546545  0.00747018577568497     0.048944779130943       -0.124449039402891      0.029199166439894       -0.00179243530071274    0.3479 0.4329       0.55537    0.53572 0.42861 0.40000 5   7   16   36
1.650   -8.23544313249218   6.94042692683253    -3.39498866209800   1.02008023269196    -0.159659865419206  0.0121445582789007      -0.000357205483094947   -1.4898798121505    0.26466701855909    0.00358714033619851     -0.00201960396974911    0.384580060794776   -0.847202771429721  0.161467725636213       -0.00772207069582351    -0.256724950911885  0.623622113832748   -0.139781216539949  0.00809441421493705     0.0507858776839012      -0.130789926193565      0.0308637365929767      -0.00190575759588717    0.3486 0.4318       0.55495    0.53544 0.42721 0.40000 5   7   16   36
1.700   -7.95810810547402   6.44964649942221    -3.08406554680961   0.921560577987324   -0.143175703737854  0.0107562889373607      -0.000310846980284432   -1.47161147019438   0.27250069759688    2.21121995440107e-05    -0.0017270890002026     0.38461937446905    -0.897373395050543  0.176014639824429       -0.00876728309076267    -0.261845602143194  0.656539428280484   -0.148813579383524  0.00872470688132845     0.0522211969734394      -0.137063152538399      0.0325372343522004      -0.00202055376431245    0.3485 0.4307       0.55401    0.53517 0.42585 0.40000 5   7   16   36
1.750   -7.68882371669755   5.96700109409082    -2.77789463902680   0.824830236493568   -0.127058399252682  0.00940471369713207     -0.000265894211278317   -1.4282797465278    0.268021500887899   -0.0015460227665364     -0.00154139644918077    0.354619160241305   -0.931818025066292  0.187850080740749       -0.00965933233969153    -0.252049311814616  0.68120711337353    -0.156365940934642  0.00926902485297111     0.0510766577864166      -0.141853167269781      0.0339378893162992      -0.00211925076998896    0.3470 0.4296       0.55226    0.53491 0.42453 0.40000 5   7   16   36
1.800   -7.18340555621623   5.06529336869104    -2.21439627151052   0.652224567316997   -0.0991699146526274 0.0071227050614115      -0.000191291378289661   -0.95836024066062   0.0155093391712551  0.0423332819391313      -0.00401479247031752    -0.245272385493223  -0.632124231565154  0.138289881963929       -0.00696358162476451    0.0539302851813073  0.531036354385977   -0.131700666079031  0.00793053584814442     -0.00248943757102476    -0.115505337117208      0.0295795078306018      -0.0018807846566593     0.3469 0.4286       0.55137    0.53465 0.42325 0.40000 5   7   16   36
1.850   -7.18459161719645   5.04591449810748    -2.19265569148373   0.640676239315895   -0.096555735774789  0.00686306181711705     -0.000181890434126344   -1.29036873308084   0.235775798413188   -0.00119339166838002    -0.00134266829482724    0.236843144807058   -0.97226112821599   0.206894752019655       -0.0111955191625635     -0.204589125859625  0.715688719115456   -0.168906924546651  0.0102139753573762      0.0440369313953044      -0.148748785828112      0.0362577471222827      -0.00228895923509394    0.3448 0.4276       0.54931    0.53440 0.42200 0.40000 5   7   16   36
1.900   -6.95044722982101   4.60898197750829    -1.91445059641722   0.553461551379208   -0.0821932807352035 0.00567393930714085     -0.000142841298371417   -1.19777218444334   0.208777352939445   0.000637750760735521    -0.00132660290057169    0.151477882484801   -0.979208499292527  0.2142225141394         -0.011844491092395      -0.168220734347903  0.72607661458803    -0.173980804838015  0.0106190043685399      0.0383854262217916      -0.150973617442378      0.0371967263559825      -0.00236110364250294    0.3435 0.4266       0.54775    0.53416 0.42079 0.40000 5   7   16   36
1.950   -6.99436017230978   4.51804572346961    -1.82387973165078   0.521097514587047   -0.0766954340931655 0.00522507432135057     -0.000128626229038115   -0.824490793695964  0.0266232307939637  0.0297995978966467      -0.00286011248334559    -0.262168807019469  -0.809441927996854  0.191486595996315       -0.0108573385770167     0.0225095862038305  0.655170824222137   -0.165650822759592  0.0103233891983045      0.00738252756440923     -0.140053277758876      0.0360192328547399      -0.00232622345666143    0.3422 0.4257       0.54621    0.53392 0.41961 0.40000 5   7   16   36
2.000   -6.83024626850747   4.17629249307823    -1.60064454590347   0.450740336581553   -0.0651528392813533 0.00427735447663008     -9.78325233339896e-05   -0.648200723476902  -0.0513216659326501 0.0414018050765209      -0.00344006736617799    -0.48149985782726   -0.729272171583645  0.181314084238832       -0.0104025778682697     0.137086624916004   0.613122167424212   -0.159991680147853  0.0100430620703446      -0.0131911978239637     -0.132109166782905      0.0348566696280396      -0.002263351857969      0.3412 0.4248       0.54482    0.53369 0.41845 0.40000 5   7   16   36
2.500   -5.0836349182634    0.724501981093627   0.584712715338312   -0.219110224677996  0.0420605734826033  -0.00433048735905618    0.000176026231143706    0.621419224995207   -0.450237111060974  0.0749075446068058      -0.00382949235336997    -1.69986499958692   -0.600679370691872  0.219753620849811       -0.0149019201370735     0.675666204249255   0.597400929763647   -0.186953331534424  0.0126334251921722      -0.0998142513115101     -0.130972380057538      0.0393665509264665      -0.00267852616434165    0.3321 0.4166       0.53278    0.53166 0.40830 0.40000 5   7   16   36
3.000   -4.33250281753284   -1.13618069218914   1.76807961784154    -0.567764496658062  0.0944412945207331  -0.00820590243473543    0.000287674134188114    2.00742613833316    -0.920874657362815  0.122143080515564       -0.00505897715891375    -3.02696515919793   -0.332736709989806  0.225086385957971       -0.0172150725489179     1.26242439347087    0.496542234426723   -0.193282176922218  0.0138837811212986      -0.194610400990698      -0.113419631595161      0.0399191247875949      -0.00284174825238909    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
3.500   -4.28596979461164   -1.8087225419961    2.21170052782595    -0.687463123737981  0.109315476758272   -0.00897194516922032    0.000296585082537636    3.19264124637151    -1.32429929966184   0.162898791142085       -0.00613890395734041    -4.1886576796904    -0.0408845083421402 0.212859589387536       -0.0180120409096456     1.79016183006952    0.35741915378327    -0.185130709742452  0.0140262737114305      -0.281665238535764      -0.0869512431096741     0.0375210865688497      -0.00278868921781588    0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
4.000   -5.02698618019238   -1.18423190684053   1.8650091376584     -0.567631348042036  0.0856415778741415  -0.0065873429161796     0.000202561595637352    4.22271266154357    -1.71055825404477   0.209457649276819       -0.00792796113237774    -5.29484745270224   0.354379111865771   0.166106069001157       -0.0161820140728211     2.33041874231081    0.134845537718508   -0.15332664115241   0.0124720590133478      -0.375372954871387      -0.0429799195840215     0.0304620577953309      -0.00241004583669512    0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
5.000   -6.80957002151689   0.515680788610421   0.929767149368741   -0.26156754300457   0.0287918560061865  -0.001167899166934      -1.47917507802548e-06   7.23460777434586    -3.22123165199228   0.459562046770686       -0.0215399343251032     -9.13634274732083   2.40142794974582    -0.192154644821443      0.00425974833870515     4.33826713821094    -1.01392716011891   0.0568256053785099  0.000150560623997055    -0.733906831508175      0.172797862091891       -0.0100444152814652     -2.78954508385977e-06   0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
6.000   -8.46302671759177   2.61684074990111    -0.353892217398549  0.153797017771454   -0.0443406314488406 0.00540616924800489     -0.000236538139967414   8.10998050397232    -3.68468767001664   0.540594292626257       -0.0261743231626202     -10.2353626601625   3.10233875351692    -0.329142995824045      0.0126381229629054      4.92609835044716    -1.44487084700428   0.144822259689876   -0.00528345263575864    -0.840650957307107      0.257472509049913       -0.0276017430896028     0.00107906307770713     0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
7.000   -9.92659340717547   4.63593286082826    -1.6038646640223    0.552589379845611   -0.112949015732382  0.0114232516758311      -0.000446701599650152   8.40379873317417    -3.9241597257035    0.593650081965538       -0.0296774716036952     -10.6694034976732   3.55710901627899    -0.435392930714737      0.0197134436148288      5.17910884270801    -1.73920608450842   0.213068666921564   -0.00972100094037355    -0.888242236478402      0.315560045830422       -0.0409083554378973     0.00192750830690766     0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
8.000   -11.087582067043    6.31918359084893    -2.65660011887733   0.885704573538717   -0.169417563021583  0.0162932068686083      -0.000613913890379303   8.33358568704997    -4.01281202628224   0.625767148646312       -0.032204892356232      -10.6685351978147   3.82254992555098    -0.512860458134614      0.0252863373098357      5.2051245323595     -1.92176581938289   0.262282599546393   -0.0130737617517631     -0.894833427629427      0.351481132785386       -0.0501978653411605     0.00253837202388271     0.3244 0.4100       0.52282    0.53000 0.40000 0.40000 5   7   16   36
10.00   -12.5414320889531   8.54495254333636    -4.07044990992193   1.32873848682302    -0.243081229451699  0.0224988176343856      -0.00082156196294421    7.66172131326708    -3.94485757705519   0.651334791113019       -0.0352040065490708     -9.99737472531347   3.99830693255826    -0.60405148499109       0.0326700670966756      4.89299289315057    -2.06506206619852   0.317795008898314   -0.0171519647919213     -0.840860287000719      0.378775815314078       -0.0598780446893658     0.00320996962208241     0.3293 0.4100       0.52588    0.53000 0.40000 0.40000 5   7   16   36
""")
