# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2014-2020 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

"""
Module exports 
:class:`DouglasEtAl2013StochasticSD001Q200K005`
:class:`DouglasEtAl2013StochasticSD001Q200K020`
:class:`DouglasEtAl2013StochasticSD001Q200K040`
:class:`DouglasEtAl2013StochasticSD001Q200K060`
:class:`DouglasEtAl2013StochasticSD001Q600K005`
:class:`DouglasEtAl2013StochasticSD001Q600K020`
:class:`DouglasEtAl2013StochasticSD001Q600K040`
:class:`DouglasEtAl2013StochasticSD001Q600K060`
:class:`DouglasEtAl2013StochasticSD001Q1800K005`
:class:`DouglasEtAl2013StochasticSD001Q1800K020`
:class:`DouglasEtAl2013StochasticSD001Q1800K040`
:class:`DouglasEtAl2013StochasticSD001Q1800K060`
:class:`DouglasEtAl2013StochasticSD010Q200K005`
:class:`DouglasEtAl2013StochasticSD010Q200K020`
:class:`DouglasEtAl2013StochasticSD010Q200K040`
:class:`DouglasEtAl2013StochasticSD010Q200K060`
:class:`DouglasEtAl2013StochasticSD010Q600K005`
:class:`DouglasEtAl2013StochasticSD010Q600K020`
:class:`DouglasEtAl2013StochasticSD010Q600K040`
:class:`DouglasEtAl2013StochasticSD010Q600K060`
:class:`DouglasEtAl2013StochasticSD010Q1800K005`
:class:`DouglasEtAl2013StochasticSD010Q1800K020`
:class:`DouglasEtAl2013StochasticSD010Q1800K040`
:class:`DouglasEtAl2013StochasticSD010Q1800K060`
:class:`DouglasEtAl2013StochasticSD100Q200K005`
:class:`DouglasEtAl2013StochasticSD100Q200K020`
:class:`DouglasEtAl2013StochasticSD100Q200K040`
:class:`DouglasEtAl2013StochasticSD100Q200K060`
:class:`DouglasEtAl2013StochasticSD100Q600K005`
:class:`DouglasEtAl2013StochasticSD100Q600K020`
:class:`DouglasEtAl2013StochasticSD100Q600K040`
:class:`DouglasEtAl2013StochasticSD100Q600K060`
:class:`DouglasEtAl2013StochasticSD100Q1800K005`
:class:`DouglasEtAl2013StochasticSD100Q1800K020`
:class:`DouglasEtAl2013StochasticSD100Q1800K040`
:class:`DouglasEtAl2013StochasticSD100Q1800K060`
"""
import numpy as np
from scipy.constants import g
from math import sqrt
from openquake.hazardlib.gsim.base import GMPE, CoeffsTable
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, PGV, SA


class DouglasEtAl2013StochasticSD001Q200K005(GMPE):
    """
    Implements the GMPE for induced seismicity in Geothermal Areas derived
    from stochastic simulations of ground motion constructed by:

    Douglas, J., Edwards, B., Convertito, V., Sharma, N., Tramelli, A.,
        Kraaijpoel, D., Cabrera, B. M., Maercklin, N., and Troise, C. (2013)
        "Predicting Ground Motion for Induced Earthquakes in Geothermal Areas"
        Bulleting of the Seismological Society of America, 103(3), 1875 - 1897

    The stochastic model by Douglas et al. (2013) provides coefficients for
    36 GMPEs, corresponding to different values of Stress Drop (1 bar, 10 bar,
    100 bar), Attentuation Quality Factor Q (200, 600, 1800) and high-frequency
    Kappa (0.005, 0.02, 0.04, 0.05 s).

    The present model is implemented for Stress Drop 1 bar, Q 200 and
    Kappa 0.005 s.

    The models for each combination of Stress Drop, Q and Kappa
    are implemented in subclasses, with only the median coefficients modified
    in each subclass


    Notes on implementation:

        1) Aleatory uncertainty terms are not supplied for the stochastic
           coefficients. Instead the adjusted aleatory uncertainty coefficients
           derived from empirical observations are applied to the stochastic
           model.

        2) In the initial coefficient set for the stochastic model coefficients
           for spectral accelerations up to 10 s are provided. However, the
           empirical aleatory uncertainties are provided only for periods up
           to 0.5012 s. Therefore, after consulation with J. Douglas, it is
           decided to limit longest applicable spectral period to Sa (0.5 s),
           rather than extrapolate the empricial aleatory coefficients to
           longer periods.

        3) For PGA and Sa (< 0.01 s) the aleatory uncertainty coefficients
           for Sa (0.01 s) are applied (J. Douglas, pers. comm.)

        4) For Sa (< 0.01 s) the coefficients are interpolated assuming
           PGA occurs at Sa (0.005 s) (J. Dougla, pers. comm.). We therefore
           limit the short period range to 0.005 s
    """
    #: The supported tectonic region type is Geothermal because
    #: the equations have been developed for geothermal regions
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.GEOTHERMAL

    #: The supported intensity measure types are PGA, PGV, and SA, see table
    #: 4.a, pages 22-23
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = set([
        PGA,
        PGV,
        SA
    ])

    #: The supported intensity measure component is 'average horizontal', see
    #: section entitiled "Empirical Analysis", paragraph 1
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.AVERAGE_HORIZONTAL

    #: The supported standard deviations are total, inter and intra event, see
    #: table 4.a, pages 22-23
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = set([
        const.StdDev.INTER_EVENT,
        const.StdDev.INTRA_EVENT,
        const.StdDev.TOTAL
    ])

    #: No additional site term is defined
    REQUIRES_SITES_PARAMETERS = set()

    #: The required rupture parameters are magnitude
    REQUIRES_RUPTURE_PARAMETERS = {'mag'}

    #: The required distance parameter is hypocentral distance
    REQUIRES_DISTANCES = {'rhypo'}

    def get_mean_and_stddevs(self, sites, rup, dists, imt, stddev_types):
        """
        See :meth:`superclass method
        <.base.GroundShakingIntensityModel.get_mean_and_stddevs>`
        for spec of input and result values.
        """
        C = self.COEFFS[imt]
        C_SIG = self.SIGMA_COEFFS[imt]

        mean = (self.get_magnitude_scaling_term(C, rup.mag) +
                self.get_distance_scaling_term(C, dists.rhypo))

        std_devs = self.get_stddevs(C_SIG, stddev_types, len(dists.rhypo))

        #: Mean ground motions initially returned in cm/s/s (for PGA, SA)
        #: and cm/s for PGV
        if not imt.name == "PGV":
            # Convert mean from log(cm/s/s) to g
            mean = np.log(np.exp(mean) / (100. * g))

        return mean, std_devs

    def get_magnitude_scaling_term(self, C, mag):
        """
        Returns the magnitude scaling term (equation 1)
        """
        mval = mag - 3.0
        return C['b1'] + C['b2'] * mval + C['b3'] * (mval ** 2.0) +\
            C['b4'] * (mval ** 3.0)

    def get_distance_scaling_term(self, C, rhyp):
        """
        Returns the distance scaling term (equation 1)
        """
        rval = rhyp + C['bh']
        return C['b5'] * np.log(rval) + C['b6'] * rval

    def get_stddevs(self, C_SIG, stddev_types, num_sites):
        """
        Returns the standard deviations

        N.B. In the paper, and with confirmation from the author, the
        aleatory variability terms from the empirical model are used in
        conjunction with the median coefficients from the stochastic model.
        In the empirical model, coefficients for a single-station intra-event
        sigma are derived. These are labeled as "phi". Inter-event coefficients
        corresponding to two observed geothermal sequences (Soultz-Sous-Forets
        and Basel) are also derived. The inter-event standard deviation is
        therefore taken as the ordinary mean of the two inter-event
        sigma terms
        """
        stddevs = []
        intra = C_SIG['phi']
        inter = (C_SIG['tau_s'] + C_SIG['tau_b']) / 2.0
        total = sqrt(intra ** 2.0 + inter ** 2.0)
        for stddev_type in stddev_types:
            assert stddev_type in self.DEFINED_FOR_STANDARD_DEVIATION_TYPES
            if stddev_type == const.StdDev.TOTAL:
                stddevs.append(total + np.zeros(num_sites))
            elif stddev_type == const.StdDev.INTER_EVENT:
                stddevs.append(inter + np.zeros(num_sites))
            elif stddev_type == const.StdDev.INTRA_EVENT:
                stddevs.append(intra + np.zeros(num_sites))
        return stddevs

    # IMT > 0.5 seconds removed from the present implementation
    # For median values, PGA is assumed equivalent to Sa (0.005 s)
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -0.911899   1.898031   -0.219505    0.021594   -1.329907   -0.016893   0.000000   0.272453
               PGA    3.691836   1.401122   -0.219469    0.032238   -1.650703   -0.023499   0.110000   0.302012
          0.005000    3.691836   1.401122   -0.219469    0.032238   -1.650703   -0.023499   0.110000   0.302012
          0.010000    4.088129   1.396525   -0.212368    0.029869   -1.837709   -0.016626   0.000000   0.320983
          0.020000    5.931361   1.372185   -0.197027    0.025748   -2.482646   -0.002140   0.950000   0.346455
          0.030000    7.088704   1.335125   -0.182551    0.023845   -2.730126   -0.005237   2.010000   0.332423
          0.040000    6.338391   1.291277   -0.174672    0.027038   -2.290429   -0.024085   1.810000   0.282412
          0.050000    5.263431   1.250568   -0.178681    0.035678   -1.800646   -0.039319   1.160000   0.220561
          0.075000    4.001717   1.220868   -0.222160    0.059438   -1.270830   -0.047952   0.350000   0.115105
          0.100000    3.534052   1.274291   -0.269713    0.070710   -1.129130   -0.043285   0.140000   0.072277
          0.150000    2.999563   1.453761   -0.336446    0.069763   -1.047842   -0.033311   0.010000   0.047346
          0.200000    2.643874   1.642154   -0.373914    0.055965   -1.040540   -0.026098   0.000000   0.051645
          0.300000    2.084980   1.965771   -0.401441    0.021288   -1.059220   -0.017376   0.000000   0.073924
          0.400000    1.625988   2.209726   -0.398218   -0.008625   -1.086292   -0.012415   0.000000   0.092529
          0.500000    1.238325   2.390165   -0.382327   -0.030766   -1.110481   -0.009430   0.000000   0.107816
         """)

    SIGMA_COEFFS = CoeffsTable(sa_damping=5, table="""
                               IMT         phi       tau_s       tau_b
                               pgv  0.53545879  0.65762034  0.55823845
                               pga  0.57602321  0.90206692  0.63679205
                            0.0050  0.57602321  0.90206692  0.63679205
                            0.0100  0.57602321  0.90206692  0.63679205
                            0.0104  0.57689520  0.90103091  0.63516156
                            0.0108  0.57783868  0.89116278  0.63196573
                            0.0112  0.57908780  0.87922269  0.62955373
                            0.0116  0.58103639  0.86723876  0.62733527
                            0.0121  0.58375210  0.85574873  0.62533168
                            0.0126  0.58662287  0.84854712  0.62395612
                            0.0130  0.58949877  0.84463184  0.62280841
                            0.0136  0.59226021  0.84360137  0.62191123
                            0.0141  0.59452817  0.84469266  0.62127164
                            0.0146  0.59719947  0.84835823  0.62133721
                            0.0152  0.60028971  0.85435847  0.62185373
                            0.0158  0.60362608  0.86262581  0.62244475
                            0.0164  0.60729815  0.87254971  0.62366185
                            0.0170  0.61179020  0.88519712  0.62549912
                            0.0177  0.61694496  0.90037312  0.62785447
                            0.0184  0.62248778  0.91757430  0.63166420
                            0.0191  0.62856213  0.93643339  0.63664928
                            0.0198  0.63519785  0.95313131  0.64157563
                            0.0206  0.64030966  0.97297199  0.64821970
                            0.0214  0.64621927  0.98848791  0.65515479
                            0.0222  0.65175226  0.99862058  0.66084582
                            0.0231  0.65698191  0.99593666  0.66425118
                            0.0240  0.66244820  0.98019441  0.66579263
                            0.0249  0.66842953  0.96775821  0.66790592
                            0.0259  0.67341753  0.96397265  0.67111507
                            0.0269  0.67807259  0.95589014  0.67574265
                            0.0279  0.68314489  0.94744882  0.68205857
                            0.0290  0.68760782  0.94059228  0.68860503
                            0.0301  0.69337407  0.93669825  0.69561266
                            0.0313  0.69831040  0.93197705  0.70264339
                            0.0325  0.70284223  0.92599937  0.71146997
                            0.0337  0.70700450  0.91851520  0.71952427
                            0.0350  0.71188544  0.90651575  0.72670161
                            0.0364  0.71608136  0.89308687  0.73385972
                            0.0378  0.71825235  0.88306961  0.74283159
                            0.0393  0.71758703  0.87559117  0.75090906
                            0.0408  0.71556058  0.86479011  0.75939834
                            0.0424  0.71317989  0.84854056  0.76688850
                            0.0440  0.71032003  0.83184489  0.77087696
                            0.0457  0.70305223  0.81781543  0.77179373
                            0.0475  0.69566273  0.80345026  0.76990948
                            0.0493  0.68796458  0.78437968  0.76712272
                            0.0513  0.68056087  0.75940460  0.76354482
                            0.0532  0.67268550  0.73333365  0.75718439
                            0.0553  0.66257190  0.70820063  0.75383256
                            0.0574  0.65076968  0.68528905  0.74765620
                            0.0597  0.63769895  0.66285926  0.73605336
                            0.0620  0.62804828  0.63826332  0.72269728
                            0.0644  0.61769784  0.61642661  0.71055894
                            0.0669  0.60678498  0.59579093  0.69844462
                            0.0695  0.59586816  0.58030542  0.68934140
                            0.0722  0.58258386  0.56640014  0.68200191
                            0.0750  0.57090572  0.55440356  0.67200412
                            0.0779  0.55972340  0.54483417  0.66070232
                            0.0809  0.54989996  0.53656315  0.65087230
                            0.0840  0.54218255  0.52645499  0.63999870
                            0.0873  0.53296740  0.51733857  0.62434404
                            0.0906  0.52298265  0.50745354  0.60940104
                            0.0941  0.51472294  0.49884621  0.59478133
                            0.0978  0.50819820  0.49137806  0.57445883
                            0.1016  0.50224816  0.48410339  0.55462523
                            0.1055  0.49877218  0.47243431  0.53473527
                            0.1096  0.49429293  0.46249198  0.51691791
                            0.1138  0.48943841  0.45880771  0.50033901
                            0.1183  0.48311183  0.45618579  0.48086407
                            0.1228  0.47694259  0.45337171  0.45851082
                            0.1276  0.47249189  0.45057534  0.43827336
                            0.1325  0.46703351  0.44524279  0.41509572
                            0.1377  0.46067205  0.44242663  0.39124108
                            0.1430  0.45466430  0.44466659  0.36692056
                            0.1485  0.45042283  0.43986274  0.34495155
                            0.1543  0.44817331  0.43876880  0.31754765
                            0.1603  0.44238566  0.41177095  0.29155521
                            0.1665  0.43701439  0.41028471  0.27984172
                            0.1729  0.43321154  0.38811549  0.27612026
                            0.1796  0.43021356  0.38221438  0.26793366
                            0.1866  0.42772176  0.37672571  0.26812502
                            0.1938  0.42501528  0.37275559  0.26418376
                            0.2013  0.42682516  0.37548950  0.26481937
                            0.2091  0.42487345  0.39198400  0.27158879
                            0.2172  0.42496052  0.38891404  0.27302182
                            0.2256  0.42359637  0.38854359  0.28490270
                            0.2344  0.42421801  0.38753753  0.29370418
                            0.2435  0.42432813  0.36517965  0.30141728
                            0.2529  0.42269786  0.36397903  0.31056065
                            0.2627  0.42148251  0.36956366  0.31864898
                            0.2729  0.41637955  0.37677653  0.32300185
                            0.2834  0.41748144  0.38346451  0.33456689
                            0.2944  0.41965081  0.38089013  0.34241338
                            0.3058  0.42104089  0.37814987  0.35063710
                            0.3177  0.42270009  0.34736191  0.36544202
                            0.3300  0.42334190  0.35085452  0.37534349
                            0.3428  0.42847146  0.34871270  0.36351831
                            0.3560  0.42568535  0.36172585  0.40147718
                            0.3698  0.43095089  0.36448882  0.38073027
                            0.3841  0.42794813  0.37338013  0.41080089
                            0.3990  0.43214173  0.38505632  0.42015150
                            0.4145  0.43634237  0.37807826  0.40857767
                            0.4305  0.43677233  0.38083883  0.42110004
                            0.4472  0.43246639  0.38896171  0.42343771
                            0.4825  0.41364167  0.39946754  0.43467297
                            0.5012  0.41676790  0.40295239  0.43749117
                            """)


class DouglasEtAl2013StochasticSD001Q200K020(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 001 - Q 200 - Kappa 0.02
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -1.386232   2.002569   -0.243693    0.021568   -1.235992   -0.017009   0.000000   0.221558
               PGA    2.514236   1.514284   -0.254728    0.034245   -1.368549   -0.026112   0.040000   0.251214
          0.005000    2.514236   1.514284   -0.254728    0.034245   -1.368549   -0.026112   0.040000   0.251214
          0.010000    2.526405   1.516447   -0.252939    0.033405   -1.373784   -0.026004   0.060000   0.248973
          0.020000    2.845356   1.504545   -0.240064    0.029654   -1.498791   -0.022700   0.090000   0.277776
          0.030000    3.648650   1.469251   -0.220269    0.025622   -1.754011   -0.018753   0.490000   0.310281
          0.040000    4.099641   1.419897   -0.204483    0.025538   -1.813257   -0.022844   0.800000   0.300761
          0.050000    4.017106   1.366809   -0.198300    0.030874   -1.672333   -0.031036   0.770000   0.259098
          0.075000    3.432272   1.286782   -0.225641    0.054608   -1.302754   -0.041710   0.370000   0.145882
          0.100000    3.095306   1.308094   -0.270743    0.068599   -1.145346   -0.040538   0.170000   0.085920
          0.150000    2.689339   1.467746   -0.338665    0.069757   -1.048710   -0.032642   0.040000   0.046990
          0.200000    2.370813   1.651499   -0.377802    0.056938   -1.023793   -0.026355   0.000000   0.045030
          0.300000    1.859320   1.974641   -0.408833    0.023401   -1.029673   -0.018122   0.000000   0.058825
          0.400000    1.414448   2.222739   -0.408549   -0.006291   -1.047952   -0.013325   0.000000   0.069394
          0.500000    1.031582   2.409070   -0.394672   -0.028883   -1.066816   -0.010357   0.000000   0.078167
    """)


class DouglasEtAl2013StochasticSD001Q200K040(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 001 - Q 200 - Kappa 0.04
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -1.788106   2.115736   -0.262747    0.018942   -1.183227   -0.016038   0.000000   0.182817
               PGA    1.730746   1.641503   -0.286487    0.033214   -1.253266   -0.024278   0.000000   0.210349
          0.005000    1.730746   1.641503   -0.286487    0.033214   -1.253266   -0.024278   0.000000   0.210349
          0.010000    1.690656   1.645102   -0.285898    0.032735   -1.238134   -0.024761   0.000000   0.205189
          0.020000    1.712960   1.642704   -0.281724    0.031506   -1.244281   -0.024784   0.010000   0.207743
          0.030000    1.911187   1.628613   -0.268113    0.027751   -1.313453   -0.023501   0.050000   0.233174
          0.040000    2.262531   1.592429   -0.248109    0.024380   -1.407567   -0.023393   0.190000   0.262727
          0.050000    2.505791   1.538042   -0.231744    0.025439   -1.427319   -0.026479   0.290000   0.263940
          0.075000    2.603442   1.407264   -0.233098    0.045395   -1.288275   -0.034832   0.270000   0.183375
          0.100000    2.508251   1.373744   -0.270432    0.063396   -1.160310   -0.036461   0.170000   0.110077
          0.150000    2.277599   1.491856   -0.338778    0.068431   -1.050338   -0.031594   0.050000   0.051968
          0.200000    2.039725   1.665845   -0.379240    0.056574   -1.016114   -0.026178   0.000000   0.044308
          0.300000    1.610954   1.985064   -0.412594    0.023940   -1.014323   -0.018377   0.000000   0.054123
          0.400000    1.197240   2.234736   -0.414919   -0.005173   -1.025421   -0.013737   0.000000   0.059470
          0.500000    0.826458   2.425072   -0.403468   -0.027633   -1.038408   -0.010818   0.000000   0.061644
         """)


class DouglasEtAl2013StochasticSD001Q200K060(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 001 - Q 200 - Kappa 0.06
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -2.094492   2.210213   -0.273918    0.015167   -1.151849   -0.015183   0.000000   0.157734
               PGA    1.221953   1.750749   -0.308085    0.030116   -1.207025   -0.022117   0.000000   0.183270
          0.005000    1.221953   1.750749   -0.308085    0.030116   -1.207025   -0.022117   0.000000   0.183270
          0.010000    1.175258   1.754990   -0.307949    0.029748   -1.189989   -0.022613   0.000000   0.177373
          0.020000    1.152647   1.754891   -0.306030    0.029228   -1.179664   -0.023028   0.000000   0.175078
          0.030000    1.179156   1.749522   -0.300167    0.027713   -1.186448   -0.023093   0.000000   0.181042
          0.040000    1.314236   1.732339   -0.286382    0.024460   -1.227136   -0.022806   0.020000   0.203709
          0.050000    1.515464   1.695458   -0.268091    0.022288   -1.269403   -0.023695   0.080000   0.226609
          0.075000    1.806574   1.552540   -0.246343    0.034953   -1.234426   -0.029987   0.130000   0.202406
          0.100000    1.910935   1.465001   -0.270167    0.055562   -1.155550   -0.032674   0.120000   0.134368
          0.150000    1.873662   1.523761   -0.336936    0.066218   -1.053210   -0.030297   0.050000   0.060015
          0.200000    1.722087   1.682213   -0.378975    0.055713   -1.013127   -0.025800   0.000000   0.045528
          0.300000    1.385170   1.995187   -0.413693    0.023498   -1.008733   -0.018330   0.000000   0.053691
          0.400000    1.013212   2.244722   -0.417310   -0.005358   -1.017300   -0.013748   0.000000   0.057483
          0.500000    0.659877   2.436691   -0.407553   -0.027523   -1.025821   -0.010902   0.000000   0.056268
         """)


class DouglasEtAl2013StochasticSD001Q600K005(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 001 - Q 600 - Kappa 0.005
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -1.018152   1.715318   -0.175091    0.017816   -1.216500   -0.011993   0.000000   0.181752
               PGA    3.589080   1.206609   -0.155223    0.024240   -1.401654   -0.019345   0.110000   0.190322
          0.005000    3.589080   1.206609   -0.155223    0.024240   -1.401654   -0.019345   0.110000   0.190322
          0.010000    4.911415   1.188752   -0.138552    0.018628   -1.869825   -0.008674   0.590000   0.220320
          0.020000    5.191928   1.135551   -0.109690    0.011910   -1.699357   -0.026272   0.840000   0.183679
          0.030000    4.424745   1.090555   -0.111595    0.020546   -1.299011   -0.036744   0.280000   0.117423
          0.040000    4.140453   1.078605   -0.132558    0.033221   -1.178108   -0.034707   0.130000   0.086542
          0.050000    3.979718   1.086974   -0.157835    0.044612   -1.132899   -0.030961   0.090000   0.072361
          0.075000    3.678728   1.148153   -0.218371    0.064142   -1.084879   -0.023978   0.060000   0.054940
          0.100000    3.429408   1.236027   -0.267416    0.072596   -1.062416   -0.019770   0.050000   0.043964
          0.150000    2.980985   1.434977   -0.333077    0.069592   -1.031349   -0.015240   0.000000   0.037687
          0.200000    2.636753   1.628285   -0.368781    0.054682   -1.032007   -0.012047   0.000000   0.048223
          0.300000    2.076595   1.952199   -0.392267    0.018512   -1.050101   -0.008015   0.000000   0.072718
          0.400000    1.611798   2.190614   -0.385264   -0.011874   -1.072593   -0.005658   0.000000   0.089592
          0.500000    1.217886   2.362928   -0.366572   -0.033591   -1.091887   -0.004292   0.000000   0.101936
         """)


class DouglasEtAl2013StochasticSD001Q600K020(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 001 - Q 600 - Kappa 0.020
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -1.474885   1.853039   -0.216446    0.022361   -1.159570   -0.010882   0.000000   0.143047
               PGA    2.342749   1.348400   -0.210078    0.032118   -1.205713   -0.017276   0.000000   0.154153
          0.005000    2.342749   1.348400   -0.210078    0.032118   -1.205713   -0.017276   0.000000   0.154153
          0.010000    2.338170   1.348362   -0.206283    0.030567   -1.198439   -0.017789   0.000000   0.151975
          0.020000    2.706148   1.321070   -0.178654    0.021194   -1.287942   -0.018291   0.090000   0.183344
          0.030000    3.016630   1.247722   -0.147054    0.017290   -1.278445   -0.023509   0.160000   0.168414
          0.040000    3.092182   1.180248   -0.146904    0.027996   -1.200855   -0.026782   0.120000   0.123757
          0.050000    3.113437   1.150349   -0.164788    0.040982   -1.147124   -0.026756   0.090000   0.093016
          0.075000    3.078805   1.174853   -0.221704    0.063057   -1.088814   -0.022760   0.070000   0.059778
          0.100000    2.966859   1.252330   -0.270653    0.072523   -1.061810   -0.019288   0.060000   0.044742
          0.150000    2.661828   1.444789   -0.337856    0.070850   -1.027668   -0.015126   0.020000   0.033863
          0.200000    2.363970   1.636500   -0.376007    0.057180   -1.015779   -0.012390   0.000000   0.040007
          0.300000    1.853490   1.962959   -0.404872    0.022695   -1.023575   -0.008562   0.000000   0.056907
          0.400000    1.405774   2.209079   -0.402015   -0.007610   -1.039558   -0.006239   0.000000   0.067075
          0.500000    1.018525   2.390841   -0.385720   -0.030343   -1.054876   -0.004861   0.000000   0.074311
         """)


class DouglasEtAl2013StochasticSD001Q600K040(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 001 - Q 600 - Kappa 0.040
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -1.849385   1.993331   -0.247386    0.022239   -1.131444   -0.009860   0.000000   0.118667
               PGA    1.632396   1.500688   -0.256860    0.035012   -1.163818   -0.014136   0.000000   0.130186
          0.005000    1.632396   1.500688   -0.256860    0.035012   -1.163818   -0.014136   0.000000   0.130186
          0.010000    1.590524   1.503304   -0.255601    0.034326   -1.146756   -0.014684   0.000000   0.124409
          0.020000    1.601125   1.497411   -0.247909    0.031791   -1.142438   -0.015191   0.000000   0.126477
          0.030000    1.734189   1.470071   -0.220930    0.023079   -1.158830   -0.016597   0.000000   0.147039
          0.040000    1.895940   1.400327   -0.190818    0.020210   -1.149369   -0.019357   0.000000   0.146678
          0.050000    2.058032   1.319473   -0.184034    0.030082   -1.132438   -0.020935   0.020000   0.123518
          0.075000    2.312639   1.243607   -0.224363    0.058026   -1.091850   -0.020564   0.060000   0.072872
          0.100000    2.369735   1.287114   -0.272167    0.070363   -1.062386   -0.018425   0.060000   0.049408
          0.150000    2.249018   1.461465   -0.340081    0.070421   -1.026095   -0.014902   0.030000   0.033795
          0.200000    2.033724   1.648897   -0.379300    0.057443   -1.007859   -0.012485   0.000000   0.038504
          0.300000    1.606988   1.974347   -0.411268    0.024159   -1.009716   -0.008804   0.000000   0.052462
          0.400000    1.192377   2.224049   -0.412193   -0.005299   -1.020254   -0.006499   0.000000   0.057828
          0.500000    0.819211   2.412271   -0.399162   -0.027965   -1.031211   -0.005096   0.000000   0.059291
         """)


class DouglasEtAl2013StochasticSD001Q600K060(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 001 - Q 600 - Kappa 0.060
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -2.138493   2.106366   -0.265296    0.019400   -1.112992   -0.009315   0.000000   0.103976
               PGA    1.148370   1.628251   -0.287747    0.033829   -1.144203   -0.012436   0.000000   0.115490
          0.005000    1.148370   1.628251   -0.287747    0.033829   -1.144203   -0.012436   0.000000   0.115490
          0.010000    1.100454   1.631890   -0.287275    0.033378   -1.126216   -0.012952   0.000000   0.109143
          0.020000    1.075112   1.629870   -0.284137    0.032524   -1.113361   -0.013411   0.000000   0.105730
          0.030000    1.094950   1.620201   -0.273619    0.029244   -1.112136   -0.013837   0.000000   0.110630
          0.040000    1.183773   1.589289   -0.248480    0.022363   -1.119182   -0.014903   0.000000   0.124711
          0.050000    1.298335   1.524158   -0.223210    0.021525   -1.113655   -0.016526   0.000000   0.125700
          0.075000    1.600712   1.363323   -0.228312    0.048127   -1.083727   -0.018230   0.020000   0.087996
          0.100000    1.797430   1.344676   -0.271421    0.065820   -1.062362   -0.017284   0.050000   0.057198
          0.150000    1.843526   1.483017   -0.340243    0.069135   -1.024584   -0.014640   0.030000   0.035033
          0.200000    1.716380   1.662307   -0.380380    0.056912   -1.003788   -0.012474   0.000000   0.038419
          0.300000    1.382126   1.984756   -0.413535    0.023986   -1.004561   -0.008831   0.000000   0.052283
          0.400000    1.009701   2.235505   -0.416269   -0.005047   -1.013269   -0.006494   0.000000   0.056224
          0.500000    0.655144   2.426646   -0.405547   -0.027322   -1.020755   -0.005098   0.000000   0.054644
         """)


class DouglasEtAl2013StochasticSD001Q1800K005(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 001 - Q 1800 - Kappa 0.005
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -1.089531   1.630072   -0.146313    0.012334   -1.168735   -0.008155   0.000000   0.124303
               PGA    3.388708   1.122579   -0.119078    0.016528   -1.225495   -0.015096   0.000000   0.126719
          0.005000    3.388708   1.122579   -0.119078    0.016528   -1.225495   -0.015096   0.000000   0.126719
          0.010000    4.343610   1.083149   -0.087678    0.006140   -1.384188   -0.019949   0.280000   0.136126
          0.020000    4.203717   1.040558   -0.081142    0.010225   -1.161755   -0.024475   0.030000   0.085515
          0.030000    4.121190   1.039103   -0.101964    0.022050   -1.124788   -0.019930   0.020000   0.070847
          0.040000    4.023197   1.050014   -0.127977    0.034235   -1.108946   -0.016813   0.030000   0.063947
          0.050000    3.919910   1.068474   -0.154822    0.045171   -1.097666   -0.014763   0.040000   0.059170
          0.075000    3.669707   1.138406   -0.216292    0.064143   -1.077873   -0.011797   0.060000   0.048840
          0.100000    3.422799   1.229220   -0.265269    0.072228   -1.058209   -0.010340   0.050000   0.039674
          0.150000    2.975243   1.430294   -0.330036    0.068526   -1.028421   -0.008782   0.000000   0.035051
          0.200000    2.629737   1.624149   -0.364343    0.052880   -1.028892   -0.007153   0.000000   0.046455
          0.300000    2.065365   1.946322   -0.384991    0.015780   -1.045247   -0.004754   0.000000   0.069751
          0.400000    1.596682   2.180485   -0.375923   -0.014639   -1.065822   -0.003215   0.000000   0.083940
          0.500000    1.200007   2.347832   -0.356093   -0.035852   -1.083534   -0.002317   0.000000   0.093695
         """)


class DouglasEtAl2013StochasticSD001Q1800K020(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 001 - Q 1800 - Kappa 0.020
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -1.518957   1.789716   -0.200505    0.020891   -1.136435   -0.007080   0.000000   0.101454
               PGA    2.278964   1.281239   -0.187155    0.029012   -1.160774   -0.010135   0.000000   0.106119
          0.005000    2.278964   1.281239   -0.187155    0.029012   -1.160774   -0.010135   0.000000   0.106119
          0.010000    2.271277   1.279336   -0.181408    0.026653   -1.148104   -0.010825   0.000000   0.102448
          0.020000    2.510287   1.231997   -0.139437    0.012738   -1.153744   -0.013661   0.000000   0.113884
          0.030000    2.773305   1.144600   -0.121580    0.018206   -1.126277   -0.015656   0.000000   0.090105
          0.040000    2.949617   1.107808   -0.137090    0.031775   -1.108997   -0.015022   0.020000   0.072360
          0.050000    3.040573   1.105361   -0.160743    0.043794   -1.098533   -0.013808   0.040000   0.063054
          0.075000    3.060273   1.157222   -0.220450    0.064070   -1.075989   -0.011541   0.060000   0.049386
          0.100000    2.960898   1.242219   -0.269693    0.072949   -1.057246   -0.010197   0.060000   0.039080
          0.150000    2.658229   1.439044   -0.336715    0.070825   -1.025484   -0.008759   0.020000   0.031182
          0.200000    2.360601   1.631946   -0.374442    0.056871   -1.014045   -0.007510   0.000000   0.038634
          0.300000    1.848651   1.958532   -0.402005    0.021836   -1.021364   -0.005251   0.000000   0.055764
          0.400000    1.398495   2.202972   -0.397681   -0.008785   -1.036128   -0.003747   0.000000   0.064827
          0.500000    1.008973   2.382012   -0.380224   -0.031489   -1.050182   -0.002863   0.000000   0.070363
         """)


class DouglasEtAl2013StochasticSD001Q1800K040(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 001 - Q 1800 - Kappa 0.040
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -1.876373   1.944053   -0.238715    0.022681   -1.117377   -0.006641   0.000000   0.088373
               PGA    1.591976   1.445796   -0.242326    0.034488   -1.141256   -0.008294   0.000000   0.093962
          0.005000    1.591976   1.445796   -0.242326    0.034488   -1.141256   -0.008294   0.000000   0.093962
          0.010000    1.548842   1.447824   -0.240619    0.033647   -1.123375   -0.008823   0.000000   0.087449
          0.020000    1.552707   1.439552   -0.229645    0.029698   -1.113939   -0.009382   0.000000   0.086773
          0.030000    1.660404   1.398755   -0.193116    0.018145   -1.113617   -0.010550   0.000000   0.093327
          0.040000    1.826075   1.309703   -0.168290    0.021147   -1.102613   -0.011673   0.000000   0.085376
          0.050000    2.002537   1.237327   -0.173306    0.034828   -1.091085   -0.012009   0.010000   0.073042
          0.075000    2.289213   1.207877   -0.223633    0.060750   -1.073209   -0.011037   0.050000   0.052352
          0.100000    2.362841   1.269949   -0.272128    0.071572   -1.055776   -0.009992   0.060000   0.039929
          0.150000    2.246155   1.454130   -0.339822    0.070809   -1.023775   -0.008719   0.030000   0.030636
          0.200000    2.031745   1.643962   -0.378796    0.057605   -1.006470   -0.007687   0.000000   0.037296
          0.300000    1.604800   1.970501   -0.410250    0.024070   -1.008472   -0.005511   0.000000   0.051876
          0.400000    1.189085   2.219741   -0.410435   -0.005587   -1.018448   -0.004004   0.000000   0.056958
          0.500000    0.814452   2.406676   -0.396598   -0.028355   -1.028546   -0.003097   0.000000   0.057671
         """)


class DouglasEtAl2013StochasticSD001Q1800K060(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 001 - Q 1800 - Kappa 0.060
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -2.156614   2.065720   -0.260325    0.020572   -1.102852   -0.006529   0.000000   0.080242
               PGA    1.119820   1.581392   -0.277976    0.034521   -1.129253   -0.007548   0.000000   0.086455
          0.005000    1.119820   1.581392   -0.277976    0.034521   -1.129253   -0.007548   0.000000   0.086455
          0.010000    1.071372   1.584734   -0.277309    0.034013   -1.110965   -0.008052   0.000000   0.079704
          0.020000    1.044234   1.581690   -0.273399    0.032916   -1.097208   -0.008458   0.000000   0.075238
          0.030000    1.055709   1.569219   -0.258863    0.027938   -1.091392   -0.008858   0.000000   0.076428
          0.040000    1.129287   1.526973   -0.227602    0.019847   -1.090705   -0.009487   0.000000   0.080059
          0.050000    1.245606   1.447151   -0.205176    0.022882   -1.084029   -0.010139   0.000000   0.075944
          0.075000    1.594773   1.306652   -0.226203    0.052707   -1.071221   -0.010234   0.040000   0.056913
          0.100000    1.796256   1.316835   -0.272206    0.068042   -1.056110   -0.009622   0.060000   0.041932
          0.150000    1.848211   1.473278   -0.340626    0.069790   -1.024771   -0.008579   0.040000   0.030797
          0.200000    1.714875   1.656796   -0.380397    0.057214   -1.002290   -0.007777   0.000000   0.037193
          0.300000    1.380683   1.981208   -0.413167    0.024102   -1.003529   -0.005575   0.000000   0.051883
          0.400000    1.007776   2.232006   -0.415518   -0.005016   -1.012018   -0.004014   0.000000   0.055722
          0.500000    0.652442   2.422519   -0.404333   -0.027354   -1.019019   -0.003103   0.000000   0.053813
         """)


class DouglasEtAl2013StochasticSD010Q200K005(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 200 - Kappa 0.005
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    0.605318   2.239348   -0.247962    0.007715   -1.495398   -0.019604   0.250000   0.324479
               PGA    5.875280   1.745388   -0.275130    0.025412   -1.934005   -0.023881   0.550000   0.381508
          0.005000    5.875280   1.745388   -0.275130    0.025412   -1.934005   -0.023881   0.550000   0.381508
          0.010000    6.217695   1.729050   -0.262333    0.022410   -2.101025   -0.018003   0.290000   0.423520
          0.020000    9.158260   1.673934   -0.245104    0.022102   -3.094216    0.002983   2.030000   0.445541
          0.030000    9.978339   1.602046   -0.239414    0.030472   -3.143382   -0.007983   3.080000   0.394974
          0.040000    8.264174   1.537990   -0.249128    0.044234   -2.363145   -0.033505   2.250000   0.306393
          0.050000    6.872820   1.507449   -0.272239    0.057068   -1.786378   -0.048164   1.330000   0.221661
          0.075000    5.482143   1.584770   -0.336688    0.064514   -1.270507   -0.051998   0.430000   0.105096
          0.100000    4.910161   1.746579   -0.377144    0.051911   -1.138473   -0.045404   0.190000   0.073122
          0.150000    4.182565   2.060525   -0.407661    0.017125   -1.061257   -0.034238   0.010000   0.071638
          0.200000    3.708136   2.306609   -0.407169   -0.012862   -1.069274   -0.026148   0.000000   0.079692
          0.300000    2.981990   2.633960   -0.375320   -0.050843   -1.108363   -0.016665   0.000000   0.097484
          0.400000    2.414119   2.821389   -0.333948   -0.067658   -1.145763   -0.011656   0.000000   0.118908
          0.500000    2.022012   2.929733   -0.294952   -0.072515   -1.203786   -0.007894   0.080000   0.139949
         """)


class DouglasEtAl2013StochasticSD010Q200K020(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 200 - Kappa 0.020
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -0.096779   2.383856   -0.266292    0.002110   -1.354216   -0.019738   0.230000   0.245101
               PGA    4.374535   1.917336   -0.307134    0.020805   -1.574418   -0.027204   0.450000   0.295194
          0.005000    4.374535   1.917336   -0.307134    0.020805   -1.574418   -0.027204   0.450000   0.295194
          0.010000    4.425233   1.915916   -0.303907    0.020040   -1.593433   -0.026748   0.510000   0.295897
          0.020000    5.022820   1.881192   -0.284724    0.016837   -1.817288   -0.021213   0.670000   0.352047
          0.030000    6.182574   1.804058   -0.267546    0.020760   -2.152706   -0.017533   1.390000   0.378696
          0.040000    6.307568   1.719413   -0.264235    0.031634   -2.049173   -0.027098   1.520000   0.339448
          0.050000    5.833904   1.654281   -0.275591    0.044755   -1.759576   -0.038042   1.190000   0.270470
          0.075000    4.919003   1.649480   -0.331893    0.059787   -1.307550   -0.046727   0.510000   0.133370
          0.100000    4.457979   1.778000   -0.375302    0.050641   -1.149814   -0.043367   0.250000   0.080117
          0.150000    3.852046   2.076912   -0.410452    0.017520   -1.053774   -0.033966   0.070000   0.063501
          0.200000    3.376321   2.322466   -0.412791   -0.011989   -1.028408   -0.027215   0.000000   0.062581
          0.300000    2.676051   2.657490   -0.384656   -0.050445   -1.048271   -0.018226   0.000000   0.061316
          0.400000    2.134605   2.855400   -0.345028   -0.068645   -1.085810   -0.012884   0.030000   0.069769
          0.500000    1.736610   2.973288   -0.306396   -0.074972   -1.137275   -0.009075   0.110000   0.085640
         """)


class DouglasEtAl2013StochasticSD010Q200K040(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 200 - Kappa 0.040
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -0.687488   2.527271   -0.274220   -0.006401   -1.270136   -0.018579   0.200000   0.190232
               PGA    3.279024   2.097278   -0.327817    0.011371   -1.389128   -0.026188   0.300000   0.232413
          0.005000    3.279024   2.097278   -0.327817    0.011371   -1.389128   -0.026188   0.300000   0.232413
          0.010000    3.284303   2.099098   -0.326753    0.011143   -1.391236   -0.026116   0.370000   0.226683
          0.020000    3.339767   2.088811   -0.320785    0.010447   -1.408222   -0.025946   0.390000   0.236114
          0.030000    3.750529   2.047995   -0.303726    0.009511   -1.545612   -0.023438   0.560000   0.280550
          0.040000    4.250578   1.969295   -0.288486    0.015107   -1.658691   -0.024347   0.830000   0.307671
          0.050000    4.362890   1.881148   -0.285246    0.026259   -1.597237   -0.030367   0.830000   0.289479
          0.075000    4.120221   1.770714   -0.322112    0.049365   -1.323147   -0.040091   0.510000   0.173247
          0.100000    3.864185   1.834616   -0.367246    0.047074   -1.167873   -0.040033   0.290000   0.099092
          0.150000    3.422233   2.098779   -0.408472    0.016665   -1.050839   -0.033293   0.090000   0.064081
          0.200000    3.018385   2.338399   -0.413324   -0.012508   -1.012372   -0.027349   0.000000   0.059582
          0.300000    2.386917   2.674846   -0.388335   -0.050871   -1.019597   -0.018788   0.000000   0.049715
          0.400000    1.844803   2.879976   -0.351108   -0.069598   -1.039009   -0.013838   0.010000   0.044455
          0.500000    1.443464   3.006837   -0.314023   -0.076915   -1.081092   -0.010058   0.090000   0.049779
         """)


class DouglasEtAl2013StochasticSD010Q200K060(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 200 - Kappa 0.060
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -1.097634   2.637309   -0.274444   -0.014043   -1.231086   -0.017202   0.210000   0.157031
               PGA    2.578048   2.240621   -0.336572    0.001667   -1.308454   -0.024230   0.240000   0.193963
          0.005000    2.578048   2.240621   -0.336572    0.001667   -1.308454   -0.024230   0.240000   0.193963
          0.010000    2.562845   2.243931   -0.336075    0.001470   -1.303756   -0.024302   0.300000   0.186931
          0.020000    2.569763   2.239660   -0.334000    0.001646   -1.303550   -0.024424   0.340000   0.185469
          0.030000    2.642638   2.222963   -0.326549    0.001279   -1.325062   -0.024235   0.350000   0.200132
          0.040000    2.921247   2.177306   -0.312293    0.002562   -1.405149   -0.023560   0.460000   0.234883
          0.050000    3.194821   2.101826   -0.300431    0.009233   -1.445578   -0.025567   0.570000   0.255596
          0.075000    3.305510   1.926738   -0.313200    0.035011   -1.298833   -0.034756   0.430000   0.199228
          0.100000    3.255483   1.913197   -0.355178    0.041352   -1.172368   -0.036714   0.280000   0.121239
          0.150000    3.004214   2.123610   -0.403630    0.015702   -1.051662   -0.032377   0.100000   0.067149
          0.200000    2.688200   2.353180   -0.411302   -0.013331   -1.007721   -0.027087   0.000000   0.059876
          0.300000    2.156829   2.686815   -0.387751   -0.052032   -1.017363   -0.018479   0.000000   0.050439
          0.400000    1.641840   2.894441   -0.351882   -0.070909   -1.029199   -0.013671   0.000000   0.040298
          0.500000    1.182235   3.026362   -0.316201   -0.078545   -1.036206   -0.010984   0.000000   0.036532
         """)


class DouglasEtAl2013StochasticSD010Q600K005(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 600 - Kappa 0.005
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    0.399226   1.976778   -0.208627    0.015319   -1.298302   -0.015378   0.130000   0.223024
               PGA    5.552878   1.438059   -0.208984    0.029624   -1.555031   -0.020992   0.400000   0.244204
          0.005000    5.552878   1.438059   -0.208984    0.029624   -1.555031   -0.020992   0.400000   0.244204
          0.010000    7.303571   1.395409   -0.177926    0.020530   -2.157182   -0.008337   1.160000   0.298737
          0.020000    7.023987   1.276633   -0.153224    0.027707   -1.745299   -0.032132   1.060000   0.206642
          0.030000    6.188354   1.223851   -0.189734    0.052538   -1.342493   -0.039370   0.450000   0.117224
          0.040000    5.858554   1.253123   -0.235363    0.067220   -1.221775   -0.036003   0.280000   0.080921
          0.050000    5.650634   1.316139   -0.274299    0.073340   -1.174988   -0.031743   0.230000   0.063701
          0.075000    5.218693   1.512609   -0.341146    0.069154   -1.119276   -0.024434   0.170000   0.049166
          0.100000    4.823678   1.708044   -0.377812    0.053403   -1.082143   -0.020453   0.100000   0.052826
          0.150000    4.184307   2.037115   -0.403919    0.016870   -1.050870   -0.015655   0.010000   0.067581
          0.200000    3.711971   2.283654   -0.400351   -0.013652   -1.060171   -0.012098   0.000000   0.078407
          0.300000    2.982740   2.601639   -0.364046   -0.051219   -1.093085   -0.007740   0.000000   0.096058
          0.400000    2.426178   2.775209   -0.320150   -0.066530   -1.129523   -0.005234   0.020000   0.114668
          0.500000    2.002393   2.870147   -0.280178   -0.069583   -1.169527   -0.003495   0.070000   0.132142
         """)


class DouglasEtAl2013StochasticSD010Q600K020(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 600 - Kappa 0.02
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -0.277203   2.185642   -0.250640    0.012533   -1.213880   -0.013574   0.100000   0.159635
               PGA    3.959118   1.671948   -0.271802    0.031204   -1.286048   -0.019660   0.160000   0.179971
          0.005000    3.959118   1.671948   -0.271802    0.031204   -1.286048   -0.019660   0.160000   0.179971
          0.010000    4.017162   1.665213   -0.264841    0.029351   -1.301096   -0.019577   0.230000   0.181727
          0.020000    4.563020   1.588633   -0.220228    0.020149   -1.427232   -0.020657   0.440000   0.231305
          0.030000    4.769998   1.441805   -0.205692    0.036515   -1.348874   -0.027225   0.410000   0.182017
          0.040000    4.794690   1.373421   -0.236283    0.057856   -1.252219   -0.029232   0.320000   0.120403
          0.050000    4.768092   1.386150   -0.273635    0.068505   -1.191185   -0.028265   0.260000   0.083776
          0.075000    4.597890   1.542450   -0.342634    0.068098   -1.119271   -0.023541   0.190000   0.051075
          0.100000    4.352686   1.728234   -0.381417    0.053625   -1.081155   -0.020010   0.140000   0.049248
          0.150000    3.833814   2.054402   -0.411632    0.018525   -1.036691   -0.015778   0.050000   0.057232
          0.200000    3.376930   2.304931   -0.411897   -0.011399   -1.021748   -0.012950   0.000000   0.059595
          0.300000    2.677023   2.638076   -0.380831   -0.049952   -1.040550   -0.008698   0.000000   0.059030
          0.400000    2.148759   2.828242   -0.338913   -0.067566   -1.079788   -0.005782   0.050000   0.066852
          0.500000    1.738307   2.936455   -0.298828   -0.072794   -1.121944   -0.003786   0.120000   0.081141
         """)


class DouglasEtAl2013StochasticSD010Q600K040(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 600 - Kappa 0.04
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -0.789537   2.379491   -0.271641    0.003383   -1.184514   -0.011619   0.120000   0.123741
               PGA    3.038850   1.905216   -0.312136    0.023319   -1.222269   -0.016180   0.120000   0.141692
          0.005000    3.038850   1.905216   -0.312136    0.023319   -1.222269   -0.016180   0.120000   0.141692
          0.010000    3.023028   1.904740   -0.310129    0.022996   -1.214832   -0.016406   0.170000   0.135111
          0.020000    3.087900   1.885074   -0.297838    0.020616   -1.226169   -0.016693   0.210000   0.144902
          0.030000    3.306016   1.803091   -0.261877    0.017577   -1.249219   -0.018916   0.250000   0.173652
          0.040000    3.492669   1.665705   -0.248320    0.033194   -1.220502   -0.022021   0.240000   0.155980
          0.050000    3.648336   1.575797   -0.267915    0.051968   -1.187511   -0.023153   0.230000   0.119504
          0.075000    3.797474   1.607881   -0.336743    0.063623   -1.120779   -0.021884   0.190000   0.062318
          0.100000    3.743565   1.761896   -0.379258    0.051789   -1.083615   -0.019261   0.160000   0.051151
          0.150000    3.397650   2.073851   -0.412679    0.017806   -1.029702   -0.015741   0.060000   0.056234
          0.200000    3.019391   2.322874   -0.414958   -0.011765   -1.006630   -0.013207   0.000000   0.056985
          0.300000    2.387884   2.661951   -0.387985   -0.050139   -1.015168   -0.009072   0.000000   0.047948
          0.400000    1.852708   2.864039   -0.349409   -0.068621   -1.035729   -0.006484   0.020000   0.042075
          0.500000    1.448716   2.985416   -0.311157   -0.075406   -1.073942   -0.004340   0.100000   0.046815
         """)


class DouglasEtAl2013StochasticSD010Q600K060(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 600 - Kappa 0.06
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -1.157299   2.521549   -0.277618   -0.005771   -1.173632   -0.010296   0.160000   0.103647
               PGA    2.426138   2.085137   -0.330841    0.012940   -1.198333   -0.014021   0.120000   0.120614
          0.005000    2.426138   2.085137   -0.330841    0.012940   -1.198333   -0.014021   0.120000   0.120614
          0.010000    2.400702   2.087152   -0.329965    0.012770   -1.189374   -0.014202   0.170000   0.112920
          0.020000    2.395202   2.078493   -0.326412    0.012920   -1.182158   -0.014496   0.200000   0.109869
          0.030000    2.445459   2.048721   -0.311341    0.011013   -1.184271   -0.015197   0.200000   0.122909
          0.040000    2.579776   1.963805   -0.284702    0.013228   -1.185542   -0.017070   0.200000   0.139326
          0.050000    2.728047   1.844045   -0.275243    0.027647   -1.167326   -0.018984   0.190000   0.130413
          0.075000    3.042038   1.719669   -0.324075    0.054696   -1.120435   -0.019803   0.180000   0.077569
          0.100000    3.137381   1.810541   -0.371739    0.048885   -1.080864   -0.018494   0.150000   0.055346
          0.150000    2.981524   2.093499   -0.410666    0.016837   -1.028456   -0.015529   0.070000   0.056141
          0.200000    2.690148   2.337571   -0.414405   -0.012744   -1.001874   -0.013184   0.000000   0.057222
          0.300000    2.157144   2.676639   -0.388658   -0.051411   -1.013559   -0.008795   0.000000   0.049033
          0.400000    1.640818   2.883541   -0.351944   -0.070088   -1.024134   -0.006368   0.000000   0.038112
          0.500000    1.188612   3.012661   -0.315520   -0.077446   -1.032666   -0.005083   0.010000   0.033828
         """)


class DouglasEtAl2013StochasticSD010Q1800K005(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 600 - Kappa 0.005
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    0.270388   1.840800   -0.173459    0.013618   -1.217478   -0.010519   0.070000   0.144994
               PGA    5.210924   1.292567   -0.160512    0.024235   -1.317580   -0.016126   0.180000   0.149861
          0.005000    5.210924   1.292567   -0.160512    0.024235   -1.317580   -0.016126   0.180000   0.149861
          0.010000    6.212053   1.203080   -0.105015    0.009911   -1.463246   -0.022325   0.490000   0.155971
          0.020000    6.012874   1.127925   -0.131939    0.035733   -1.226591   -0.024736   0.200000   0.083440
          0.030000    5.893671   1.156908   -0.185876    0.057161   -1.183315   -0.019867   0.180000   0.066813
          0.040000    5.757238   1.218068   -0.233651    0.069310   -1.164147   -0.016652   0.190000   0.056752
          0.050000    5.594412   1.293592   -0.272710    0.074360   -1.143826   -0.014781   0.180000   0.049030
          0.075000    5.194722   1.500166   -0.338908    0.069080   -1.106676   -0.012170   0.150000   0.042578
          0.100000    4.817394   1.698307   -0.374583    0.052713   -1.077829   -0.010853   0.100000   0.049220
          0.150000    4.177675   2.027663   -0.398342    0.015300   -1.046479   -0.009150   0.010000   0.064945
          0.200000    3.702838   2.271508   -0.392561   -0.015579   -1.054285   -0.007221   0.000000   0.074407
          0.300000    2.969130   2.581006   -0.353443   -0.052654   -1.084101   -0.004489   0.000000   0.087899
          0.400000    2.401649   2.745922   -0.308500   -0.066850   -1.114613   -0.002864   0.010000   0.103110
          0.500000    1.976276   2.834255   -0.268448   -0.068982   -1.152828   -0.001530   0.060000   0.118231
         """)


class DouglasEtAl2013StochasticSD010Q1800K020(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 1800 - Kappa 0.02
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -0.328874   2.093100   -0.236729    0.015454   -1.182377   -0.008473   0.080000   0.108501
               PGA    3.826758   1.563844   -0.247996    0.032719   -1.209190   -0.011652   0.080000   0.115103
          0.005000    3.826758   1.563844   -0.247996    0.032719   -1.209190   -0.011652   0.080000   0.115103
          0.010000    3.867316   1.552800   -0.236514    0.029180   -1.212293   -0.011958   0.140000   0.113953
          0.020000    4.192960   1.425188   -0.174651    0.019976   -1.221323   -0.015210   0.180000   0.124076
          0.030000    4.493093   1.291756   -0.192799    0.047693   -1.187651   -0.016284   0.180000   0.087425
          0.040000    4.651621   1.287639   -0.236539    0.065028   -1.164574   -0.015190   0.190000   0.065066
          0.050000    4.694163   1.337759   -0.275529    0.072278   -1.145164   -0.013964   0.190000   0.052351
          0.075000    4.584276   1.524234   -0.343354    0.069212   -1.109348   -0.011766   0.180000   0.041097
          0.100000    4.349370   1.717243   -0.381367    0.054140   -1.077856   -0.010627   0.140000   0.045250
          0.150000    3.831766   2.046878   -0.410685    0.018671   -1.034827   -0.009263   0.050000   0.055689
          0.200000    3.374756   2.297556   -0.410105   -0.011449   -1.019820   -0.007977   0.000000   0.058145
          0.300000    2.673029   2.627803   -0.377335   -0.050075   -1.037184   -0.005346   0.000000   0.056193
          0.400000    2.142555   2.812981   -0.334171   -0.067283   -1.074526   -0.003251   0.050000   0.061894
          0.500000    1.730373   2.916070   -0.293558   -0.071862   -1.115004   -0.001721   0.120000   0.074430
         """)


class DouglasEtAl2013StochasticSD010Q1800K040(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 1800 - Kappa 0.04
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -0.804310   2.314662   -0.267297    0.007093   -1.171683   -0.007235   0.120000   0.089725
               PGA    2.983275   1.824241   -0.301354    0.027162   -1.189682   -0.009263   0.090000   0.097007
          0.005000    2.983275   1.824241   -0.301354    0.027162   -1.189682   -0.009263   0.090000   0.097007
          0.010000    2.965410   1.822366   -0.298627    0.026695   -1.181314   -0.009444   0.140000   0.089042
          0.020000    2.999099   1.795352   -0.279256    0.021952   -1.176732   -0.010069   0.160000   0.092881
          0.030000    3.158499   1.677095   -0.234751    0.021129   -1.170522   -0.011700   0.160000   0.097854
          0.040000    3.390475   1.532751   -0.238059    0.044042   -1.157034   -0.012455   0.170000   0.082131
          0.050000    3.587686   1.479256   -0.269543    0.060799   -1.142415   -0.012413   0.180000   0.064258
          0.075000    3.785946   1.574919   -0.340568    0.066008   -1.107611   -0.011351   0.180000   0.043294
          0.100000    3.734359   1.746595   -0.381165    0.052706   -1.076366   -0.010463   0.150000   0.045210
          0.150000    3.397181   2.066709   -0.413158    0.018179   -1.028223   -0.009369   0.060000   0.055059
          0.200000    3.019110   2.317323   -0.414922   -0.011478   -1.005578   -0.008296   0.000000   0.056311
          0.300000    2.387054   2.656172   -0.387233   -0.049876   -1.013793   -0.005718   0.000000   0.046995
          0.400000    1.850819   2.856136   -0.347922   -0.068234   -1.033281   -0.003952   0.020000   0.040144
          0.500000    1.445371   2.974479   -0.309067   -0.074729   -1.070102   -0.002317   0.100000   0.043841
         """)


class DouglasEtAl2013StochasticSD010Q1800K060(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 010 - Q 1800 - Kappa 0.06
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -1.164129   2.472701   -0.277072   -0.002441   -1.165144   -0.006598   0.160000   0.078713
               PGA    2.401375   2.021365   -0.326031    0.017062   -1.181478   -0.008100   0.110000   0.087220
          0.005000    2.401375   2.021365   -0.326031    0.017062   -1.181478   -0.008100   0.110000   0.087220
          0.010000    2.383071   2.022707   -0.324905    0.016869   -1.175527   -0.008140   0.170000   0.078838
          0.020000    2.366778   2.011492   -0.320272    0.016913   -1.163682   -0.008467   0.190000   0.073975
          0.030000    2.394725   1.971296   -0.298124    0.013341   -1.155032   -0.009180   0.180000   0.079139
          0.040000    2.503078   1.860052   -0.268241    0.018555   -1.144001   -0.010278   0.160000   0.080541
          0.050000    2.675186   1.733416   -0.268437    0.037274   -1.132902   -0.010845   0.160000   0.071537
          0.075000    3.031875   1.665585   -0.329913    0.059259   -1.104804   -0.010761   0.170000   0.047240
          0.100000    3.137839   1.787652   -0.375751    0.050269   -1.075381   -0.010220   0.150000   0.045421
          0.150000    2.989434   2.085540   -0.412033    0.017203   -1.029966   -0.009215   0.080000   0.054873
          0.200000    2.690545   2.332560   -0.414978   -0.012490   -1.000968   -0.008353   0.000000   0.056782
          0.300000    2.156812   2.672552   -0.388704   -0.051138   -1.012544   -0.005478   0.000000   0.048499
          0.400000    1.639836   2.878667   -0.351628   -0.069741   -1.022513   -0.003854   0.000000   0.037069
          0.500000    1.187019   3.006206   -0.314832   -0.076964   -1.030241   -0.003068   0.010000   0.032154
         """)


class DouglasEtAl2013StochasticSD100Q200K005(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 200 - Kappa 0.005
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    2.324839   2.605780   -0.245000   -0.010095   -1.848593   -0.015707   0.770000   0.343939
               PGA    8.158879   2.173797   -0.303404    0.005287   -2.363083   -0.019970   1.090000   0.428530
          0.005000    8.158879   2.173797   -0.303404    0.005287   -2.363083   -0.019970   1.090000   0.428530
          0.010000    8.853782   2.134252   -0.287264    0.004035   -2.668435   -0.010186   0.910000   0.500552
          0.020000   12.942473   2.039290   -0.283197    0.016091   -3.956837    0.012329   3.430000   0.494293
          0.030000   12.518440   1.960681   -0.298372    0.031082   -3.493679   -0.013055   3.960000   0.404167
          0.040000    9.664712   1.931359   -0.325217    0.040121   -2.338914   -0.044850   2.370000   0.289277
          0.050000    8.104683   1.964937   -0.354654    0.038966   -1.740614   -0.057019   1.330000   0.197091
          0.075000    6.599562   2.186655   -0.397113    0.010796   -1.269169   -0.055330   0.440000   0.098251
          0.100000    5.885144   2.413337   -0.401910   -0.018813   -1.148164   -0.046934   0.180000   0.077783
          0.150000    4.973426   2.734479   -0.373557   -0.056450   -1.090615   -0.034214   0.000000   0.072902
          0.200000    4.392482   2.927525   -0.334102   -0.073516   -1.117532   -0.025239   0.000000   0.083125
          0.300000    3.555967   3.118749   -0.262818   -0.078628   -1.183826   -0.014927   0.020000   0.115761
          0.400000    3.050238   3.189181   -0.210182   -0.069147   -1.282536   -0.008164   0.150000   0.144658
          0.500000    2.686220   3.210507   -0.173196   -0.056596   -1.368027   -0.003961   0.280000   0.165988
         """)


class DouglasEtAl2013StochasticSD100Q200K020(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 200 - Kappa 0.02
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    1.192307   2.774275   -0.247878   -0.020415   -1.586657   -0.018039   0.650000   0.237832
               PGA    6.143483   2.393645   -0.318235   -0.007270   -1.865758   -0.025405   0.940000   0.303305
          0.005000    6.143483   2.393645   -0.318235   -0.007270   -1.865758   -0.025405   0.940000   0.303305
          0.010000    6.223932   2.385919   -0.314315   -0.007310   -1.893840   -0.024825   1.010000   0.309814
          0.020000    7.395436   2.312158   -0.297741   -0.003325   -2.311054   -0.015025   1.480000   0.387875
          0.030000    8.809595   2.203084   -0.297912    0.010720   -2.660964   -0.014302   2.450000   0.394692
          0.040000    8.239240   2.123928   -0.313728    0.023209   -2.270418   -0.031732   2.150000   0.329825
          0.050000    7.290576   2.099340   -0.337922    0.028303   -1.817399   -0.045198   1.470000   0.245893
          0.075000    6.039440   2.233795   -0.386761    0.009680   -1.307969   -0.051047   0.580000   0.115030
          0.100000    5.424826   2.438553   -0.398675   -0.018711   -1.155387   -0.045509   0.290000   0.075484
          0.150000    4.583166   2.755714   -0.376066   -0.056662   -1.058678   -0.034872   0.070000   0.051392
          0.200000    3.970968   2.954375   -0.339018   -0.074544   -1.041010   -0.027479   0.000000   0.042082
          0.300000    3.140590   3.159942   -0.269243   -0.081692   -1.084405   -0.017543   0.030000   0.054047
          0.400000    2.612667   3.242156   -0.215940   -0.073873   -1.167260   -0.010844   0.160000   0.077176
          0.500000    2.245892   3.271182   -0.177449   -0.062250   -1.250837   -0.006324   0.310000   0.097320
         """)


class DouglasEtAl2013StochasticSD100Q200K040(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 200 - Kappa 0.04
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    0.324034   2.920347   -0.238177   -0.030803   -1.439854   -0.017624   0.550000   0.172292
               PGA    4.650034   2.597451   -0.315493   -0.022452   -1.589438   -0.025503   0.680000   0.223062
          0.005000    4.650034   2.597451   -0.315493   -0.022452   -1.589438   -0.025503   0.680000   0.223062
          0.010000    4.671582   2.596766   -0.314468   -0.022152   -1.594957   -0.025432   0.790000   0.217563
          0.020000    4.798084   2.575099   -0.309272   -0.020982   -1.635139   -0.024755   0.830000   0.235535
          0.030000    5.557493   2.500164   -0.299179   -0.014196   -1.877718   -0.020408   1.220000   0.291112
          0.040000    6.148756   2.397408   -0.300585   -0.001277   -1.974914   -0.023559   1.570000   0.307954
          0.050000    5.995187   2.317777   -0.314726    0.009409   -1.785848   -0.033387   1.370000   0.273661
          0.075000    5.272165   2.319055   -0.364402    0.006274   -1.349497   -0.044766   0.670000   0.146677
          0.100000    4.816352   2.474277   -0.386946   -0.018584   -1.171118   -0.042984   0.350000   0.085370
          0.150000    4.127549   2.772932   -0.372761   -0.056995   -1.047844   -0.034619   0.090000   0.049354
          0.200000    3.582874   2.970861   -0.337984   -0.075490   -1.016088   -0.027886   0.000000   0.034151
          0.300000    2.773830   3.183724   -0.270486   -0.083629   -1.028815   -0.018856   0.000000   0.029088
          0.400000    2.194582   3.276930   -0.217971   -0.077148   -1.075743   -0.013005   0.080000   0.041300
          0.500000    1.802847   3.316343   -0.179194   -0.066798   -1.144702   -0.008539   0.230000   0.054250
         """)


class DouglasEtAl2013StochasticSD100Q200K060(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 200 - Kappa 0.06
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -0.243760   3.020036   -0.225340   -0.037543   -1.371197   -0.016464   0.530000   0.134978
               PGA    3.731088   2.742746   -0.304881   -0.033736   -1.469668   -0.023754   0.570000   0.177376
          0.005000    3.731088   2.742746   -0.304881   -0.033736   -1.469668   -0.023754   0.570000   0.177376
          0.010000    3.716108   2.744634   -0.304259   -0.033612   -1.463141   -0.023946   0.660000   0.170046
          0.020000    3.736639   2.736668   -0.303250   -0.032646   -1.464693   -0.024120   0.720000   0.169964
          0.030000    3.891647   2.705779   -0.298266   -0.030194   -1.511139   -0.023452   0.750000   0.193032
          0.040000    4.376431   2.633772   -0.293080   -0.022386   -1.643959   -0.022241   0.990000   0.233269
          0.050000    4.693648   2.543442   -0.296599   -0.011107   -1.664449   -0.025929   1.120000   0.247545
          0.075000    4.494445   2.434735   -0.337168   -0.000239   -1.370047   -0.038550   0.700000   0.172198
          0.100000    4.218543   2.520488   -0.369418   -0.018234   -1.188714   -0.039946   0.390000   0.100034
          0.150000    3.687990   2.788017   -0.366988   -0.056695   -1.042816   -0.034124   0.090000   0.050771
          0.200000    3.246029   2.981239   -0.334865   -0.075871   -1.012369   -0.027583   0.000000   0.035588
          0.300000    2.547385   3.193296   -0.268101   -0.084616   -1.034418   -0.018055   0.000000   0.035487
          0.400000    1.929091   3.291331   -0.216130   -0.078781   -1.047607   -0.013193   0.000000   0.038458
          0.500000    1.420032   3.338090   -0.177763   -0.069273   -1.058550   -0.010544   0.020000   0.041357
         """)


class DouglasEtAl2013StochasticSD100Q600K005(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 600 - Kappa 0.005
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    1.930467   2.282914   -0.232531    0.009811   -1.506516   -0.014738   0.480000   0.250677
               PGA    7.550856   1.757832   -0.264748    0.027942   -1.791055   -0.020854   0.770000   0.291210
          0.005000    7.550856   1.757832   -0.264748    0.027942   -1.791055   -0.020854   0.770000   0.291210
          0.010000    9.846235   1.656046   -0.221924    0.023280   -2.534813   -0.007769   1.850000   0.370776
          0.020000    8.661442   1.479972   -0.239039    0.055831   -1.765309   -0.038570   1.160000   0.204246
          0.030000    7.786541   1.519310   -0.305741    0.070126   -1.388340   -0.041419   0.580000   0.101092
          0.040000    7.378668   1.655445   -0.352406    0.063164   -1.275818   -0.036481   0.420000   0.070106
          0.050000    7.067322   1.804172   -0.380635    0.049506   -1.223677   -0.031860   0.350000   0.063643
          0.075000    6.404616   2.131057   -0.407507    0.012208   -1.152531   -0.024536   0.230000   0.066290
          0.100000    5.841743   2.379046   -0.404887   -0.018207   -1.109157   -0.020542   0.120000   0.068159
          0.150000    5.002262   2.704322   -0.370890   -0.055531   -1.085122   -0.015352   0.010000   0.072647
          0.200000    4.415789   2.891066   -0.328853   -0.071690   -1.105752   -0.011463   0.000000   0.085337
          0.300000    3.603809   3.064510   -0.255542   -0.074405   -1.170795   -0.006312   0.050000   0.116966
          0.400000    3.064236   3.118271   -0.203224   -0.062765   -1.244979   -0.002904   0.150000   0.141752
          0.500000    2.675408   3.127573   -0.167713   -0.048927   -1.312643   -0.000629   0.260000   0.158514
         """)


class DouglasEtAl2013StochasticSD100Q600K020(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 600 - Kappa 0.02
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    0.859027   2.560333   -0.256716   -0.004586   -1.347298   -0.013226   0.350000   0.158683
               PGA    5.483004   2.098021   -0.314032    0.014585   -1.434920   -0.019686   0.430000   0.188718
          0.005000    5.483004   2.098021   -0.314032    0.014585   -1.434920   -0.019686   0.430000   0.188718
          0.010000    5.590563   2.078050   -0.304781    0.013859   -1.462124   -0.019582   0.530000   0.199207
          0.020000    6.323929   1.907170   -0.264365    0.022448   -1.598684   -0.022629   0.830000   0.254143
          0.030000    6.360553   1.743506   -0.290349    0.051202   -1.428295   -0.030254   0.640000   0.170097
          0.040000    6.280100   1.761941   -0.338737    0.056355   -1.307509   -0.030999   0.490000   0.102912
          0.050000    6.156226   1.864363   -0.372860    0.046698   -1.237172   -0.029178   0.400000   0.074913
          0.075000    5.764401   2.160858   -0.407244    0.011582   -1.149344   -0.023889   0.280000   0.063359
          0.100000    5.332119   2.404752   -0.408221   -0.018371   -1.096195   -0.020483   0.180000   0.059761
          0.150000    4.572257   2.736641   -0.378958   -0.055996   -1.043519   -0.016192   0.050000   0.047032
          0.200000    3.979601   2.935795   -0.339802   -0.073476   -1.034428   -0.013040   0.000000   0.039410
          0.300000    3.168438   3.133724   -0.268065   -0.079516   -1.080795   -0.007915   0.050000   0.053262
          0.400000    2.638659   3.204443   -0.214189   -0.070215   -1.155198   -0.004100   0.180000   0.075824
          0.500000    2.259140   3.222666   -0.176200   -0.057265   -1.226319   -0.001508   0.320000   0.093694
         """)


class DouglasEtAl2013StochasticSD100Q600K040(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 600 - Kappa 0.04
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    0.119378   2.780484   -0.253594   -0.020673   -1.291456   -0.011053   0.340000   0.113228
               PGA    4.259541   2.392809   -0.326052   -0.006053   -1.333966   -0.016023   0.330000   0.136785
          0.005000    4.259541   2.392809   -0.326052   -0.006053   -1.333966   -0.016023   0.330000   0.136785
          0.010000    4.251123   2.387990   -0.324301   -0.005490   -1.326403   -0.016327   0.410000   0.130354
          0.020000    4.388952   2.343842   -0.311970   -0.004082   -1.352805   -0.016678   0.480000   0.150035
          0.030000    4.707088   2.187035   -0.290198    0.010393   -1.368542   -0.020254   0.540000   0.177981
          0.040000    4.887089   2.040138   -0.308165    0.032418   -1.305366   -0.023816   0.480000   0.144024
          0.050000    4.988855   2.016285   -0.343966    0.037208   -1.248685   -0.024623   0.420000   0.103860
          0.075000    4.943346   2.207905   -0.395758    0.010369   -1.152846   -0.022581   0.300000   0.066877
          0.100000    4.703471   2.431518   -0.403352   -0.019166   -1.096833   -0.019933   0.210000   0.059521
          0.150000    4.108059   2.757050   -0.378089   -0.057122   -1.029874   -0.016352   0.060000   0.044629
          0.200000    3.587053   2.959047   -0.340715   -0.075120   -1.010851   -0.013446   0.000000   0.031430
          0.300000    2.778855   3.170929   -0.271699   -0.082663   -1.023315   -0.009086   0.000000   0.026837
          0.400000    2.216010   3.258911   -0.218633   -0.075493   -1.073414   -0.005610   0.100000   0.039524
          0.500000    1.814270   3.291360   -0.179852   -0.064276   -1.133084   -0.003021   0.240000   0.052149
         """)


class DouglasEtAl2013StochasticSD100Q600K060(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 600 - Kappa 0.06
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -0.385966   2.921215   -0.241518   -0.031192   -1.266778   -0.009682   0.370000   0.089260
               PGA    3.476405   2.592604   -0.319988   -0.021982   -1.298680   -0.013552   0.320000   0.110342
          0.005000    3.476405   2.592604   -0.319988   -0.021982   -1.298680   -0.013552   0.320000   0.110342
          0.010000    3.452832   2.592519   -0.319177   -0.021680   -1.288799   -0.013792   0.400000   0.102343
          0.020000    3.444480   2.577292   -0.317317   -0.020104   -1.275470   -0.014372   0.430000   0.100487
          0.030000    3.559513   2.517009   -0.305692   -0.015947   -1.286041   -0.015339   0.440000   0.121158
          0.040000    3.765142   2.381394   -0.296502   -0.000644   -1.278677   -0.018072   0.440000   0.135740
          0.050000    3.934182   2.261618   -0.311935    0.015917   -1.240187   -0.020347   0.400000   0.118929
          0.075000    4.152084   2.280749   -0.373161    0.008879   -1.157861   -0.020808   0.310000   0.073522
          0.100000    4.087559   2.461745   -0.392945   -0.018970   -1.098281   -0.019254   0.220000   0.060250
          0.150000    3.670521   2.771939   -0.374531   -0.057610   -1.024198   -0.016317   0.060000   0.044506
          0.200000    3.250587   2.971728   -0.338438   -0.076056   -1.007970   -0.013277   0.000000   0.033263
          0.300000    2.549104   3.185494   -0.269864   -0.084272   -1.029927   -0.008270   0.000000   0.033185
          0.400000    1.929702   3.281582   -0.217537   -0.078047   -1.040710   -0.005898   0.000000   0.035687
          0.500000    1.436761   3.324546   -0.179123   -0.068022   -1.055423   -0.004614   0.040000   0.038816
         """)


class DouglasEtAl2013StochasticSD100Q1800K005(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 1800 - Kappa 0.005
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    1.659760   2.085934   -0.203331    0.016642   -1.342952   -0.010418   0.280000   0.159418
               PGA    6.996713   1.528199   -0.215985    0.031603   -1.447689   -0.016366   0.390000   0.171628
          0.005000    6.996713   1.528199   -0.215985    0.031603   -1.447689   -0.016366   0.390000   0.171628
          0.010000    7.996946   1.329101   -0.149067    0.030138   -1.539869   -0.024932   0.650000   0.162791
          0.020000    7.745417   1.303194   -0.239201    0.070601   -1.302264   -0.024200   0.360000   0.070283
          0.030000    7.533536   1.452888   -0.310254    0.074700   -1.253909   -0.018972   0.340000   0.051370
          0.040000    7.283422   1.621081   -0.354306    0.064965   -1.223874   -0.015936   0.330000   0.049267
          0.050000    7.014133   1.780964   -0.380886    0.050399   -1.194817   -0.014241   0.300000   0.053286
          0.075000    6.382674   2.115311   -0.405410    0.012315   -1.139575   -0.012050   0.210000   0.062320
          0.100000    5.829893   2.363514   -0.401048   -0.018405   -1.100022   -0.010993   0.110000   0.064916
          0.150000    4.998001   2.683528   -0.364277   -0.055598   -1.077174   -0.008931   0.010000   0.067943
          0.200000    4.409395   2.862732   -0.320536   -0.071028   -1.094833   -0.006715   0.000000   0.078367
          0.300000    3.577077   3.022605   -0.246269   -0.071985   -1.147776   -0.003428   0.030000   0.105969
          0.400000    3.035435   3.067520   -0.194531   -0.059294   -1.218545   -0.000714   0.130000   0.127315
          0.500000    2.627273   3.071980   -0.159897   -0.045152   -1.276219    0.000940   0.220000   0.141400
         """)


class DouglasEtAl2013StochasticSD100Q1800K020(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 1800 - Kappa 0.02
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    0.756564   2.445973   -0.253403    0.003281   -1.282695   -0.007661   0.260000   0.104487
               PGA    5.272572   1.949753   -0.301116    0.023568   -1.313398   -0.011023   0.260000   0.114391
          0.005000    5.272572   1.949753   -0.301116    0.023568   -1.313398   -0.011023   0.260000   0.114391
          0.010000    5.345043   1.917835   -0.283639    0.020702   -1.318254   -0.011704   0.340000   0.120689
          0.020000    5.771054   1.657865   -0.239029    0.040756   -1.307160   -0.015882   0.370000   0.119677
          0.030000    6.063955   1.584963   -0.300301    0.065591   -1.258102   -0.016140   0.350000   0.070969
          0.040000    6.139691   1.688081   -0.349963    0.061473   -1.223802   -0.014780   0.340000   0.054120
          0.050000    6.092356   1.825691   -0.380057    0.048709   -1.197646   -0.013483   0.330000   0.053389
          0.075000    5.746209   2.145732   -0.409657    0.012215   -1.137992   -0.011748   0.260000   0.059090
          0.100000    5.323348   2.394423   -0.409179   -0.017886   -1.090278   -0.010964   0.170000   0.057788
          0.150000    4.572524   2.727532   -0.378733   -0.055465   -1.041507   -0.009563   0.050000   0.045419
          0.200000    3.980240   2.924996   -0.338776   -0.072780   -1.031869   -0.008004   0.000000   0.037113
          0.300000    3.168217   3.116511   -0.265955   -0.078095   -1.075671   -0.004554   0.050000   0.049664
          0.400000    2.628906   3.180334   -0.211884   -0.067926   -1.143790   -0.001683   0.170000   0.070456
          0.500000    2.248402   3.193443   -0.174358   -0.054350   -1.212779    0.000481   0.310000   0.086297
         """)


class DouglasEtAl2013StochasticSD100Q1800K040(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 1800 - Kappa 0.04
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV    0.081975   2.712222   -0.257520   -0.015426   -1.262417   -0.006158   0.300000   0.080425
               PGA    4.181635   2.297079   -0.326131    0.001468   -1.283519   -0.008255   0.260000   0.090545
          0.005000    4.181635   2.297079   -0.326131    0.001468   -1.283519   -0.008255   0.260000   0.090545
          0.010000    4.154262   2.289368   -0.323698    0.002146   -1.268003   -0.008716   0.320000   0.082240
          0.020000    4.227258   2.223034   -0.302232    0.003032   -1.261366   -0.009833   0.340000   0.091578
          0.030000    4.484681   2.013020   -0.282213    0.025868   -1.246485   -0.011863   0.340000   0.090374
          0.040000    4.756849   1.902764   -0.318835    0.045593   -1.222495   -0.012411   0.340000   0.068900
          0.050000    4.916688   1.936909   -0.358234    0.042875   -1.196738   -0.012258   0.330000   0.058287
          0.075000    4.929939   2.185970   -0.402288    0.010777   -1.139932   -0.011315   0.280000   0.058738
          0.100000    4.696966   2.421114   -0.406325   -0.019033   -1.091044   -0.010749   0.200000   0.057348
          0.150000    4.109324   2.751228   -0.379171   -0.056885   -1.028808   -0.009828   0.060000   0.043924
          0.200000    3.588184   2.953742   -0.341266   -0.074778   -1.009789   -0.008448   0.000000   0.030578
          0.300000    2.779909   3.163968   -0.271766   -0.082047   -1.021491   -0.005690   0.000000   0.025361
          0.400000    2.216651   3.248583   -0.218440   -0.074468   -1.069983   -0.003062   0.100000   0.037555
          0.500000    1.814014   3.277280   -0.179677   -0.062794   -1.127646   -0.000993   0.240000   0.049294
         """)


class DouglasEtAl2013StochasticSD100Q1800K060(
        DouglasEtAl2013StochasticSD001Q200K005):
    """
    Stress Drop 100 - Q 1800 - Kappa 0.04
    """
    COEFFS = CoeffsTable(sa_damping=5, table="""\
               IMT          b1         b2          b3          b4          b5          b6         bh      total
               PGV   -0.408664   2.875644   -0.246923   -0.027865   -1.247456   -0.005569   0.340000   0.066551
               PGA    3.437980   2.525510   -0.323993   -0.016500   -1.270032   -0.006956   0.280000   0.078198
          0.005000    3.437980   2.525510   -0.323993   -0.016500   -1.270032   -0.006956   0.280000   0.078198
          0.010000    3.396878   2.524213   -0.323032   -0.016096   -1.253095   -0.007381   0.340000   0.069395
          0.020000    3.386952   2.504298   -0.320406   -0.014186   -1.238463   -0.007826   0.370000   0.065232
          0.030000    3.452606   2.418102   -0.302310   -0.008615   -1.224039   -0.009038   0.340000   0.073601
          0.040000    3.643587   2.252296   -0.296655    0.011741   -1.207910   -0.010340   0.320000   0.072091
          0.050000    3.858735   2.151563   -0.322469    0.026514   -1.188204   -0.010905   0.310000   0.062145
          0.075000    4.128572   2.245399   -0.384097    0.009787   -1.136556   -0.010915   0.270000   0.057688
          0.100000    4.083878   2.448406   -0.398178   -0.019212   -1.091818   -0.010515   0.210000   0.056591
          0.150000    3.664775   2.766866   -0.376248   -0.057638   -1.019992   -0.010036   0.050000   0.043918
          0.200000    3.252018   2.968006   -0.339355   -0.075944   -1.007291   -0.008344   0.000000   0.032812
          0.300000    2.549538   3.181794   -0.270357   -0.084010   -1.028688   -0.004909   0.000000   0.032229
          0.400000    1.929709   3.276521   -0.217924   -0.077582   -1.038443   -0.003369   0.000000   0.034334
          0.500000    1.436627   3.317359   -0.179491   -0.067294   -1.052000   -0.002598   0.040000   0.037175
         """)
