# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2013-2020 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

"""
Module exports :class:`AkkarEtAlRjb2014`
               :class:`AkkarEtAlRepi2014`
               :class:`AkkarEtAlRhypo2014`.
"""
import numpy as np
from openquake.hazardlib.gsim.base import GMPE, CoeffsTable
from openquake.hazardlib import const
from openquake.hazardlib.imt import PGA, PGV, SA


class AkkarEtAlRjb2014(GMPE):
    """
    Implements GMPE developed by S. Akkar, M. A. Sandikkaya, and J. J. Bommer
    as published in "Empirical Ground-Motion Models for Point- and Extended-
    Source Crustal Earthquake Scenarios in Europe and the Middle East",
    Bulletin of Earthquake Engineering (2014), 12(1): 359 - 387
    The class implements the equations for Joyner-Boore distance and based on
    manuscript provided by the original authors.
    """
    #: The supported tectonic region type is active shallow crust because
    #: the equations have been developed for "all seismically- active regions
    #: bordering the Mediterranean Sea and extending to the Middle East", see
    #: section 'A New Generation of European Ground-Motion Models', page 4.
    DEFINED_FOR_TECTONIC_REGION_TYPE = const.TRT.ACTIVE_SHALLOW_CRUST

    #: The supported intensity measure types are PGA, PGV, and SA, see table
    #: 4.a, pages 22-23
    DEFINED_FOR_INTENSITY_MEASURE_TYPES = set([
        PGA,
        PGV,
        SA
    ])

    #: The supported intensity measure component is 'average horizontal', see
    #: section 'A New Generation of European Ground-Motion Models', page 8
    DEFINED_FOR_INTENSITY_MEASURE_COMPONENT = const.IMC.AVERAGE_HORIZONTAL

    #: The supported standard deviations are total, inter and intra event, see
    #: table 4.a, pages 22-23
    DEFINED_FOR_STANDARD_DEVIATION_TYPES = set([
        const.StdDev.TOTAL,
        const.StdDev.INTER_EVENT,
        const.StdDev.INTRA_EVENT
    ])

    #: The required site parameter is vs30, see equation 1, page 20.
    REQUIRES_SITES_PARAMETERS = {'vs30'}

    #: The required rupture parameters are rake and magnitude, see equation 1,
    #: page 20.
    REQUIRES_RUPTURE_PARAMETERS = {'rake', 'mag'}

    #: The required distance parameter is 'Joyner-Boore' distance, because
    #: coefficients in table 4.a, pages 22-23, are used.
    REQUIRES_DISTANCES = {'rjb'}

    def __init__(self, adjustment_factor=1.0):
        super().__init__()
        self.adjustment_factor = np.log(adjustment_factor)

    def get_mean_and_stddevs(self, sites, rup, dists, imt, stddev_types):
        """
        See :meth:`superclass method
        <.base.GroundShakingIntensityModel.get_mean_and_stddevs>`
        for spec of input and result values.

        Implement equation 1, page 20.
        """
        # compute median PGA on rock, needed to compute non-linear site
        # amplification
        C_pga = self.COEFFS[PGA()]
        median_pga = np.exp(
            self._compute_mean(C_pga, rup.mag, dists, rup.rake)
        )

        # compute full mean value by adding nonlinear site amplification terms
        C = self.COEFFS[imt]
        mean = (self._compute_mean(C, rup.mag, dists, rup.rake) +
                self._compute_non_linear_term(C, median_pga, sites))

        stddevs = self._get_stddevs(C, stddev_types, num_sites=sites.vs30.size)

        return mean + self.adjustment_factor, stddevs

    def _get_stddevs(self, C, stddev_types, num_sites):
        """
        Return standard deviations as defined in table 4a, p. 22.
        """
        stddevs = []
        for stddev_type in stddev_types:
            assert stddev_type in self.DEFINED_FOR_STANDARD_DEVIATION_TYPES
            if stddev_type == const.StdDev.TOTAL:
                sigma_t = np.sqrt(C['sigma'] ** 2 + C['tau'] ** 2)
                stddevs.append(sigma_t + np.zeros(num_sites))
            elif stddev_type == const.StdDev.INTRA_EVENT:
                stddevs.append(C['sigma'] + np.zeros(num_sites))
            elif stddev_type == const.StdDev.INTER_EVENT:
                stddevs.append(C['tau'] + np.zeros(num_sites))
        return stddevs

    def _compute_linear_magnitude_term(self, C, mag):
        """
        Compute and return second term in equations (2a)
        and (2b), page 20.
        """
        if mag <= self.c1:
            # this is the second term in eq. (2a), p. 20
            return C['a2'] * (mag - self.c1)
        else:
            # this is the second term in eq. (2b), p. 20
            return C['a7'] * (mag - self.c1)

    def _compute_quadratic_magnitude_term(self, C, mag):
        """
        Compute and return third term in equations (2a)
        and (2b), page  20.
        """
        return C['a3'] * (8.5 - mag) ** 2

    def _compute_logarithmic_distance_term(self, C, mag, dists):
        """
        Compute and return fourth term in equations (2a)
        and (2b), page 20.
        """
        return (
            (C['a4'] + C['a5'] * (mag - self.c1)) *
            np.log(np.sqrt(dists.rjb ** 2 + C['a6'] ** 2))
        )

    def _compute_faulting_style_term(self, C, rake):
        """
        Compute and return fifth and sixth terms in equations (2a)
        and (2b), pages 20.
        """
        Fn = float(rake > -135.0 and rake < -45.0)
        Fr = float(rake > 45.0 and rake < 135.0)

        return C['a8'] * Fn + C['a9'] * Fr

    def _compute_non_linear_term(self, C, pga_only, sites):
        """
        Compute non-linear term, equation (3a) to (3c), page 20.
        """
        Vref = 750.0
        Vcon = 1000.0
        lnS = np.zeros_like(sites.vs30)

        # equation (3a)
        idx = sites.vs30 < Vref
        lnS[idx] = (
            C['b1'] * np.log(sites.vs30[idx] / Vref) +
            C['b2'] * np.log(
                (pga_only[idx] + C['c'] * (sites.vs30[idx] / Vref) ** C['n']) /
                ((pga_only[idx] + C['c']) * (sites.vs30[idx] / Vref) ** C['n'])
            )
        )

        # equation (3b)
        idx = (sites.vs30 >= Vref) & (sites.vs30 <= Vcon)
        lnS[idx] = C['b1'] * np.log(sites.vs30[idx]/Vref)

        # equation (3c)
        idx = sites.vs30 > Vcon
        lnS[idx] = C['b1'] * np.log(Vcon/Vref)

        return lnS

    def _compute_mean(self, C, mag, dists, rake):
        """
        Compute and return mean value without site conditions,
        that is equations (1a) and (1b), p.2981-2982.
        """
        mean = (
            C['a1'] +
            self._compute_linear_magnitude_term(C, mag) +
            self._compute_quadratic_magnitude_term(C, mag) +
            self._compute_logarithmic_distance_term(C, mag, dists) +
            self._compute_faulting_style_term(C, rake)
        )

        return mean

    #: c1 is the reference magnitude, fixed to 6.75Mw (which happens to be the
    #: same value used in Boore and Atkinson, 2008)
    #: see paragraph 'Functional Form of Predictive Equations and Regressions',
    #: page 21
    c1 = 6.75

    #: Coefficient table (from Table 3 and 4a, page 22)
    #: Table 4.a: Period-dependent regression coefficients of the RJB
    #: ground-motion model
    #: sigma is the 'intra-event' standard deviation, while tau is the
    #: 'inter-event' standard deviation
    COEFFS = CoeffsTable(sa_damping=5, table="""\
    IMT         a1           a2           a3            a4           a5          a6       a7           a8           a9          c1        Vcon      Vref     c        n        b1            b2            sigma       tau
    pga         1.85329      0.0029      -0.02807      -1.23452      0.2529      7.5      -0.5096      -0.1091      0.0937      6.75      1000      750      2.5      3.2      -0.41997      -0.28846      0.6201      0.3501
    0.010       1.87032      0.0029      -0.02740      -1.23698      0.2529      7.5      -0.5096      -0.1115      0.0953      6.75      1000      750      2.5      3.2      -0.41729      -0.28685      0.6215      0.3526
    0.020       1.95279      0.0029      -0.02715      -1.25363      0.2529      7.5      -0.5096      -0.1040      0.1029      6.75      1000      750      2.5      3.2      -0.39998      -0.28241      0.6266      0.3555
    0.030       2.07006      0.0029      -0.02403      -1.27525      0.2529      7.5      -0.5096      -0.0973      0.1148      6.75      1000      750      2.5      3.2      -0.34799      -0.26842      0.6410      0.3565
    0.040       2.20452      0.0029      -0.01797      -1.30123      0.2529      7.5      -0.5096      -0.0884      0.1073      6.75      1000      750      2.5      3.2      -0.27572      -0.24759      0.6534      0.3484
    0.050       2.35413      0.0029      -0.01248      -1.32632      0.2529      7.5      -0.5096      -0.0853      0.1052      6.75      1000      750      2.5      3.2      -0.21231      -0.22385      0.6622      0.3551
    0.075       2.63078      0.0029      -0.00532      -1.35722      0.2529      7.5      -0.5096      -0.0779      0.0837      6.75      1000      750      2.5      3.2      -0.14427      -0.17525      0.6626      0.3759
    0.100       2.85412      0.0029      -0.00925      -1.38182      0.2529      7.5      -0.5096      -0.0749      0.0761      6.75      1000      750      2.5      3.2      -0.27064      -0.29293      0.6670      0.4067
    0.110       2.89772      0.0029      -0.01062      -1.38345      0.2529      7.5      -0.5096      -0.0704      0.0707      6.75      1000      750      2.5      3.2      -0.31025      -0.31837      0.6712      0.4059
    0.120       2.92748      0.0029      -0.01291      -1.37997      0.2529      7.5      -0.5096      -0.0604      0.0653      6.75      1000      750      2.5      3.2      -0.34796      -0.33860      0.6768      0.4022
    0.130       2.95162      0.0029      -0.01592      -1.37627      0.2529      7.5      -0.5096      -0.0490      0.0617      6.75      1000      750      2.5      3.2      -0.39668      -0.36646      0.6789      0.4017
    0.140       2.96299      0.0029      -0.01866      -1.37155      0.2529      7.5      -0.5096      -0.0377      0.0581      6.75      1000      750      2.5      3.2      -0.43996      -0.38417      0.6822      0.3945
    0.150       2.96622      0.0029      -0.02193      -1.36460      0.2529      7.5      -0.5096      -0.0265      0.0545      6.75      1000      750      2.5      3.2      -0.48313      -0.39551      0.6796      0.3893
    0.160       2.93166      0.0029      -0.02429      -1.35074      0.2529      7.5      -0.5096      -0.0194      0.0509      6.75      1000      750      2.5      3.2      -0.52431      -0.40869      0.6762      0.3928
    0.170       2.88988      0.0029      -0.02712      -1.33454      0.2529      7.5      -0.5096      -0.0125      0.0507      6.75      1000      750      2.5      3.2      -0.55680      -0.41528      0.6723      0.396
    0.180       2.84627      0.0029      -0.03003      -1.31959      0.2529      7.5      -0.5096      -0.0056      0.0502      6.75      1000      750      2.5      3.2      -0.58922      -0.42717      0.6694      0.396
    0.190       2.79778      0.0029      -0.03300      -1.30450      0.2529      7.5      -0.5096      0.00000      0.0497      6.75      1000      750      2.5      3.2      -0.62635      -0.44130      0.6647      0.3932
    0.200       2.73872      0.0029      -0.03462      -1.28877      0.2529      7.5      -0.5096      0.00000      0.0493      6.75      1000      750      2.5      3.2      -0.65315      -0.44644      0.6645      0.3842
    0.220       2.63479      0.0029      -0.03789      -1.26125      0.2529      7.5      -0.5096      0.00000      0.0488      6.75      1000      750      2.5      3.2      -0.68711      -0.44872      0.6600      0.3887
    0.240       2.53886      0.0029      -0.04173      -1.23600      0.2529      7.5      -0.5096      0.00000      0.0483      6.75      1000      750      2.5      3.2      -0.72744      -0.46341      0.6651      0.3792
    0.260       2.48747      0.0029      -0.04768      -1.21882      0.2529      7.5      -0.5096      0.00000      0.0478      6.75      1000      750      2.5      3.2      -0.77335      -0.48705      0.6650      0.3754
    0.280       2.38739      0.0029      -0.05178      -1.19543      0.2529      7.5      -0.5096      0.00000      0.0474      6.75      1000      750      2.5      3.2      -0.80508      -0.47334      0.6590      0.3757
    0.300       2.30150      0.0029      -0.05672      -1.17072      0.2529      7.5      -0.5096      0.00000      0.0469      6.75      1000      750      2.5      3.2      -0.82609      -0.45730      0.6599      0.3816
    0.320       2.17298      0.0029      -0.06015      -1.13847      0.2529      7.5      -0.5096      0.00000      0.0464      6.75      1000      750      2.5      3.2      -0.84080      -0.44267      0.6654      0.3866
    0.340       2.07474      0.0029      -0.06508      -1.11131      0.2529      7.5      -0.5096      0.00000      0.0459      6.75      1000      750      2.5      3.2      -0.86251      -0.43888      0.6651      0.3881
    0.360       2.01953      0.0029      -0.06974      -1.09484      0.2529      7.5      -0.5096      0.00000      0.0459      6.75      1000      750      2.5      3.2      -0.87479      -0.43820      0.6662      0.3924
    0.380       1.95078      0.0029      -0.07346      -1.07812      0.2529      7.5      -0.5096      0.00000      0.0429      6.75      1000      750      2.5      3.2      -0.88522      -0.43678      0.6698      0.3945
    0.400       1.89372      0.0029      -0.07684      -1.06530      0.2529      7.5      -0.5096      0.00000      0.0400      6.75      1000      750      2.5      3.2      -0.89517      -0.43008      0.6697      0.3962
    0.420       1.83717      0.0029      -0.08010      -1.05451      0.2529      7.5      -0.5096      0.00000      0.0374      6.75      1000      750      2.5      3.2      -0.90875      -0.42190      0.6696      0.389
    0.440       1.77528      0.0029      -0.08296      -1.04332      0.2529      7.5      -0.5096      0.00000      0.0349      6.75      1000      750      2.5      3.2      -0.91922      -0.40903      0.6641      0.3929
    0.460       1.73155      0.0029      -0.08623      -1.03572      0.2529      7.5      -0.5096      0.00000      0.0323      6.75      1000      750      2.5      3.2      -0.92670      -0.39442      0.6575      0.4009
    0.480       1.70132      0.0029      -0.09070      -1.02724      0.2529      7.5      -0.5096      0.00000      0.0297      6.75      1000      750      2.5      3.2      -0.93720      -0.38462      0.6540      0.4022
    0.500       1.67127      0.0029      -0.09490      -1.01909      0.2529      7.5      -0.5096      0.00000      0.0271      6.75      1000      750      2.5      3.2      -0.94614      -0.37408      0.6512      0.4021
    0.550       1.53838      0.0029      -0.10275      -0.99351      0.2529      7.5      -0.5096      0.00000      0.0245      6.75      1000      750      2.5      3.2      -0.96564      -0.35582      0.6570      0.4057
    0.600       1.37505      0.0029      -0.10747      -0.96429      0.2529      7.5      -0.5096      0.00000      0.0219      6.75      1000      750      2.5      3.2      -0.98499      -0.34053      0.6630      0.406
    0.650       1.21156      0.0029      -0.11262      -0.93347      0.2529      7.5      -0.5096      0.00000      0.0193      6.75      1000      750      2.5      3.2      -0.99733      -0.30949      0.6652      0.4124
    0.700       1.09262      0.0029      -0.11835      -0.91162      0.2529      7.5      -0.5096      0.00000      0.0167      6.75      1000      750      2.5      3.2      -1.00469      -0.28772      0.6696      0.4135
    0.750       0.95211      0.0029      -0.12347      -0.88393      0.2529      7.5      -0.5096      0.00000      0.0141      6.75      1000      750      2.5      3.2      -1.00786      -0.28957      0.6744      0.4043
    0.800       0.85227      0.0029      -0.12678      -0.86884      0.2529      7.5      -0.5096      0.00000      0.0115      6.75      1000      750      2.5      3.2      -1.00606      -0.28555      0.6716      0.3974
    0.850       0.76564      0.0029      -0.13133      -0.85442      0.2529      7.5      -0.5096      0.00000      0.0089      6.75      1000      750      2.5      3.2      -1.01093      -0.28364      0.6713      0.3971
    0.900       0.66856      0.0029      -0.13551      -0.83929      0.2529      7.5      -0.5096      0.00000      0.0062      6.75      1000      750      2.5      3.2      -1.01576      -0.28037      0.6738      0.3986
    0.950       0.58739      0.0029      -0.13957      -0.82668      0.2529      7.5      -0.5096      0.00000      0.0016      6.75      1000      750      2.5      3.2      -1.01353      -0.28390      0.6767      0.3949
    1.000       0.52349      0.0029      -0.14345      -0.81838      0.2529      7.5      -0.5096      0.00000      0.0000      6.75      1000      750      2.5      3.2      -1.01331      -0.28702      0.6787      0.3943
    1.100       0.37680      0.0029      -0.15051      -0.79691      0.2529      7.5      -0.5096      0.00000      0.0000      6.75      1000      750      2.5      3.2      -1.01240      -0.27669      0.6912      0.3806
    1.200       0.23251      0.0029      -0.15527      -0.77813      0.2529      7.5      -0.5096      0.00000      0.0000      6.75      1000      750      2.5      3.2      -1.00489      -0.27538      0.7015      0.3802
    1.300       0.10481      0.0029      -0.16106      -0.75888      0.2529      7.5      -0.5096      0.00000      0.0000      6.75      1000      750      2.5      3.2      -0.98876      -0.25008      0.7017      0.3803
    1.400       0.00887      0.0029      -0.16654      -0.74871      0.2529      7.5      -0.5096      0.00000      0.0000      6.75      1000      750      2.5      3.2      -0.97760      -0.23508      0.7141      0.3766
    1.500      -0.01867      0.0029      -0.17187      -0.75751      0.2529      7.5      -0.5096      0.00000      0.0000      6.75      1000      750      2.5      3.2      -0.98071      -0.24695      0.7164      0.3799
    1.600      -0.09960      0.0029      -0.17728      -0.74823      0.2529      7.5      -0.5096      0.00000      0.0000      6.75      1000      750      2.5      3.2      -0.96369      -0.22870      0.7198      0.3817
    1.700      -0.21166      0.0029      -0.17908      -0.73766      0.2529      7.5      -0.5096      0.00000      0.0000      6.75      1000      750      2.5      3.2      -0.94634      -0.21655      0.7226      0.3724
    1.800      -0.27300      0.0029      -0.18438      -0.72996      0.2529      7.5      -0.5096      0.00000      -0.003      6.75      1000      750      2.5      3.2      -0.93606      -0.20302      0.7241      0.371
    1.900      -0.35366      0.0029      -0.18741      -0.72279      0.2529      7.5      -0.5096      0.00000      -0.006      6.75      1000      750      2.5      3.2      -0.91408      -0.18228      0.7266      0.3745
    2.000      -0.42891      0.0029      -0.19029      -0.72033      0.2529      7.5      -0.5096      0.00000      -0.009      6.75      1000      750      2.5      3.2      -0.91007      -0.17336      0.7254      0.3717
    2.200      -0.55307      0.0029      -0.19683      -0.71662      0.2529      7.5      -0.5096      0.00000      -0.0141     6.75      1000      750      2.5      3.2      -0.89376      -0.15463      0.7207      0.3758
    2.400      -0.67806      0.0029      -0.20339      -0.70452      0.2529      7.5      -0.5096      0.00000      -0.0284     6.75      1000      750      2.5      3.2      -0.87052      -0.13181      0.7144      0.3973
    2.600      -0.80494      0.0029      -0.20703      -0.69691      0.2529      7.5      -0.5096      0.00000      -0.0408     6.75      1000      750      2.5      3.2      -0.85889      -0.14066      0.7122      0.4001
    2.800      -0.91278      0.0029      -0.21074      -0.69560      0.2529      7.5      -0.5096      0.00000      -0.0534     6.75      1000      750      2.5      3.2      -0.86106      -0.13882      0.7129      0.4025
    3.000      -1.05642      0.0029      -0.21392      -0.69085      0.2529      7.5      -0.5096      0.00000      -0.0683     6.75      1000      750      2.5      3.2      -0.85793      -0.13336      0.6997      0.4046
    3.200      -1.17715      0.0029      -0.21361      -0.67711      0.2529      7.5      -0.5096      0.00000      -0.078      6.75      1000      750      2.5      3.2      -0.82094      -0.13770      0.6820      0.4194
    3.400      -1.22091      0.0029      -0.21951      -0.68177      0.2529      7.5      -0.5096      0.00000      -0.0943     6.75      1000      750      2.5      3.2      -0.84449      -0.15337      0.6682      0.3971
    3.600      -1.34547      0.0029      -0.22724      -0.65918      0.2529      7.5      -0.5096      0.00000      -0.1278     6.75      1000      750      2.5      3.2      -0.83216      -0.10884      0.6508      0.4211
    3.800      -1.39790      0.0029      -0.23180      -0.65298      0.2529      7.5      -0.5096      0.00000      -0.1744     6.75      1000      750      2.5      3.2      -0.79216      -0.08884      0.6389      0.415
    4.000      -1.37536      0.0029      -0.23848      -0.66482      0.2529      7.5      -0.5096      0.00000      -0.2231     6.75      1000      750      2.5      3.2      -0.75645      -0.07749      0.6196      0.3566
    pgv         5.61201      0.0029      -0.09980      -0.98388      0.2529      7.5      -0.5096      -0.0616      0.0630      6.75      1000      750      2.5      3.2      -0.72057      -0.19688      0.6014      0.3311
    """)


class AkkarEtAlRepi2014(AkkarEtAlRjb2014):
    """
    Implements GMPE developed by S. Akkar, M. A. Sandikkaya, and J. J. Bommer
    as published in "Empirical Ground-Motion Models for Point- and Extended-
    Source Crustal Earthquake Scenarios in Europe and the Middle East",
    Bullettin of Earthquake Engineering (2014).

    The class implements the equations for epicentral distance and based on
    manuscript provided by the original authors.
    """
    REQUIRES_DISTANCES = set(('repi', ))

    def _compute_logarithmic_distance_term(self, C, mag, dists):
        """
        Compute and return fourth term in equations (2a)
        and (2b), page 20.
        """
        return (
            (C['a4'] + C['a5'] * (mag - self.c1)) *
            np.log(np.sqrt(dists.repi ** 2 + C['a6'] ** 2))
        )

    COEFFS = CoeffsTable(sa_damping=5, table="""\
      IMT         a1       a2         a3         a4       a5    a6        a7        a8        a9     c1   Vcon  Vref     c     n         b1         b2    sigma      tau
      pga    2.52977   0.0029   -0.05496   -1.31001   0.2529   7.5   -0.5096   -0.1091    0.0937   6.75   1000   750   2.5   3.2   -0.41997   -0.28846   0.6375   0.3581
      pgv    6.13498   0.0029   -0.12091   -1.04013   0.2529   7.5   -0.5096   -0.0616    0.0630   6.75   1000   750   2.5   3.2   -0.72057   -0.19688   0.6143   0.3485
    0.010    2.54832   0.0029   -0.05434   -1.31268   0.2529   7.5   -0.5096   -0.1115    0.0953   6.75   1000   750   2.5   3.2   -0.41729   -0.28685   0.6389   0.3607
    0.020    2.64420   0.0029   -0.05452   -1.33135   0.2529   7.5   -0.5096   -0.1040    0.1029   6.75   1000   750   2.5   3.2   -0.39998   -0.28241   0.6434   0.3615
    0.030    2.77723   0.0029   -0.05196   -1.35509   0.2529   7.5   -0.5096   -0.0973    0.1148   6.75   1000   750   2.5   3.2   -0.34799   -0.26842   0.6569   0.3617
    0.040    2.92666   0.0029   -0.04657   -1.38259   0.2529   7.5   -0.5096   -0.0884    0.1073   6.75   1000   750   2.5   3.2   -0.27572   -0.24759   0.6693   0.3530
    0.050    3.09355   0.0029   -0.04168   -1.41008   0.2529   7.5   -0.5096   -0.0853    0.1052   6.75   1000   750   2.5   3.2   -0.21231   -0.22385   0.6773   0.3612
    0.075    3.38462   0.0029   -0.03506   -1.44268   0.2529   7.5   -0.5096   -0.0779    0.0837   6.75   1000   750   2.5   3.2   -0.14427   -0.17525   0.6791   0.3853
    0.100    3.61906   0.0029   -0.03936   -1.46870   0.2529   7.5   -0.5096   -0.0749    0.0761   6.75   1000   750   2.5   3.2   -0.27064   -0.29293   0.6851   0.4160
    0.110    3.66537   0.0029   -0.04081   -1.47079   0.2529   7.5   -0.5096   -0.0704    0.0707   6.75   1000   750   2.5   3.2   -0.31025   -0.31837   0.6884   0.4163
    0.120    3.68544   0.0029   -0.04295   -1.46520   0.2529   7.5   -0.5096   -0.0604    0.0653   6.75   1000   750   2.5   3.2   -0.34796   -0.33860   0.6960   0.4118
    0.130    3.70155   0.0029   -0.04581   -1.45986   0.2529   7.5   -0.5096   -0.0490    0.0617   6.75   1000   750   2.5   3.2   -0.39668   -0.36646   0.6997   0.4102
    0.140    3.70871   0.0029   -0.04848   -1.45433   0.2529   7.5   -0.5096   -0.0377    0.0581   6.75   1000   750   2.5   3.2   -0.43996   -0.38417   0.7032   0.4028
    0.150    3.70477   0.0029   -0.05156   -1.44613   0.2529   7.5   -0.5096   -0.0265    0.0545   6.75   1000   750   2.5   3.2   -0.48313   -0.39551   0.7011   0.3978
    0.160    3.65565   0.0029   -0.05350   -1.42989   0.2529   7.5   -0.5096   -0.0194    0.0509   6.75   1000   750   2.5   3.2   -0.52431   -0.40869   0.6997   0.3989
    0.170    3.59764   0.0029   -0.05583   -1.41110   0.2529   7.5   -0.5096   -0.0125    0.0507   6.75   1000   750   2.5   3.2   -0.55680   -0.41528   0.6970   0.4030
    0.180    3.53732   0.0029   -0.05830   -1.39329   0.2529   7.5   -0.5096   -0.0056    0.0502   6.75   1000   750   2.5   3.2   -0.58922   -0.42717   0.6956   0.4041
    0.190    3.47722   0.0029   -0.06090   -1.37648   0.2529   7.5   -0.5096    0.0000    0.0497   6.75   1000   750   2.5   3.2   -0.62635   -0.44130   0.6915   0.4017
    0.200    3.40112   0.0029   -0.06210   -1.35770   0.2529   7.5   -0.5096    0.0000    0.0493   6.75   1000   750   2.5   3.2   -0.65315   -0.44644   0.6922   0.3965
    0.220    3.27214   0.0029   -0.06461   -1.32624   0.2529   7.5   -0.5096    0.0000    0.0488   6.75   1000   750   2.5   3.2   -0.68711   -0.44872   0.6893   0.4005
    0.240    3.15842   0.0029   -0.06791   -1.29833   0.2529   7.5   -0.5096    0.0000    0.0483   6.75   1000   750   2.5   3.2   -0.72744   -0.46341   0.6942   0.3919
    0.260    3.09498   0.0029   -0.07344   -1.27945   0.2529   7.5   -0.5096    0.0000    0.0478   6.75   1000   750   2.5   3.2   -0.77335   -0.48705   0.6938   0.3898
    0.280    2.98090   0.0029   -0.07698   -1.25442   0.2529   7.5   -0.5096    0.0000    0.0474   6.75   1000   750   2.5   3.2   -0.80508   -0.47334   0.6877   0.3883
    0.300    2.87449   0.0029   -0.08126   -1.22665   0.2529   7.5   -0.5096    0.0000    0.0469   6.75   1000   750   2.5   3.2   -0.82609   -0.45730   0.6897   0.3894
    0.320    2.72364   0.0029   -0.08387   -1.19143   0.2529   7.5   -0.5096    0.0000    0.0464   6.75   1000   750   2.5   3.2   -0.84080   -0.44267   0.6947   0.3941
    0.340    2.60904   0.0029   -0.08816   -1.16231   0.2529   7.5   -0.5096    0.0000    0.0459   6.75   1000   750   2.5   3.2   -0.86251   -0.43888   0.6939   0.3937
    0.360    2.54266   0.0029   -0.09239   -1.14444   0.2529   7.5   -0.5096    0.0000    0.0459   6.75   1000   750   2.5   3.2   -0.87479   -0.43820   0.6945   0.3997
    0.380    2.46615   0.0029   -0.09576   -1.12700   0.2529   7.5   -0.5096    0.0000    0.0429   6.75   1000   750   2.5   3.2   -0.88522   -0.43678   0.6971   0.4012
    0.400    2.40119   0.0029   -0.09885   -1.11318   0.2529   7.5   -0.5096    0.0000    0.0400   6.75   1000   750   2.5   3.2   -0.89517   -0.43008   0.6971   0.4012
    0.420    2.34540   0.0029   -0.10198   -1.10318   0.2529   7.5   -0.5096    0.0000    0.0374   6.75   1000   750   2.5   3.2   -0.90875   -0.42190   0.6955   0.3946
    0.440    2.28213   0.0029   -0.10464   -1.09241   0.2529   7.5   -0.5096    0.0000    0.0349   6.75   1000   750   2.5   3.2   -0.91922   -0.40903   0.6891   0.3985
    0.460    2.23440   0.0029   -0.10771   -1.08445   0.2529   7.5   -0.5096    0.0000    0.0323   6.75   1000   750   2.5   3.2   -0.92670   -0.39442   0.6825   0.4056
    0.480    2.20123   0.0029   -0.11199   -1.07592   0.2529   7.5   -0.5096    0.0000    0.0297   6.75   1000   750   2.5   3.2   -0.93720   -0.38462   0.6785   0.4068
    0.500    2.16953   0.0029   -0.11604   -1.06795   0.2529   7.5   -0.5096    0.0000    0.0271   6.75   1000   750   2.5   3.2   -0.94614   -0.37408   0.6751   0.4065
    0.550    2.03012   0.0029   -0.12344   -1.04242   0.2529   7.5   -0.5096    0.0000    0.0245   6.75   1000   750   2.5   3.2   -0.96564   -0.35582   0.6788   0.4087
    0.600    1.84644   0.0029   -0.12745   -1.01046   0.2529   7.5   -0.5096    0.0000    0.0219   6.75   1000   750   2.5   3.2   -0.98499   -0.34053   0.6845   0.4085
    0.650    1.66760   0.0029   -0.13195   -0.97801   0.2529   7.5   -0.5096    0.0000    0.0193   6.75   1000   750   2.5   3.2   -0.99733   -0.30949   0.6857   0.4128
    0.700    1.53432   0.0029   -0.13715   -0.95428   0.2529   7.5   -0.5096    0.0000    0.0167   6.75   1000   750   2.5   3.2   -1.00469   -0.28772   0.6902   0.4101
    0.750    1.38296   0.0029   -0.14169   -0.92585   0.2529   7.5   -0.5096    0.0000    0.0141   6.75   1000   750   2.5   3.2   -1.00786   -0.28957   0.6937   0.4011
    0.800    1.28662   0.0029   -0.14485   -0.91241   0.2529   7.5   -0.5096    0.0000    0.0115   6.75   1000   750   2.5   3.2   -1.00606   -0.28555   0.6884   0.3962
    0.850    1.20114   0.0029   -0.14922   -0.89909   0.2529   7.5   -0.5096    0.0000    0.0089   6.75   1000   750   2.5   3.2   -1.01093   -0.28364   0.6866   0.3951
    0.900    1.09991   0.0029   -0.15320   -0.88377   0.2529   7.5   -0.5096    0.0000    0.0062   6.75   1000   750   2.5   3.2   -1.01576   -0.28037   0.6881   0.3979
    0.950    1.01256   0.0029   -0.15700   -0.87050   0.2529   7.5   -0.5096    0.0000    0.0016   6.75   1000   750   2.5   3.2   -1.01353   -0.28390   0.6904   0.3955
    1.000    0.94162   0.0029   -0.16069   -0.86109   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -1.01331   -0.28702   0.6922   0.3965
    1.100    0.78017   0.0029   -0.16727   -0.83743   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -1.01240   -0.27669   0.7047   0.3819
    1.200    0.63219   0.0029   -0.17174   -0.81877   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -1.00489   -0.27538   0.7138   0.3807
    1.300    0.48905   0.0029   -0.17712   -0.79698   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -0.98876   -0.25008   0.7137   0.3827
    1.400    0.38492   0.0029   -0.18237   -0.78548   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -0.97760   -0.23508   0.7263   0.3787
    1.500    0.36315   0.0029   -0.18790   -0.79498   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -0.98071   -0.24695   0.7287   0.3821
    1.600    0.28812   0.0029   -0.19363   -0.78665   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -0.96369   -0.22870   0.7307   0.3854
    1.700    0.18172   0.0029   -0.19545   -0.77778   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -0.94634   -0.21655   0.7322   0.3751
    1.800    0.13021   0.0029   -0.20069   -0.77282   0.2529   7.5   -0.5096    0.0000   -0.0030   6.75   1000   750   2.5   3.2   -0.93606   -0.20302   0.7316   0.3749
    1.900    0.05074   0.0029   -0.20386   -0.76574   0.2529   7.5   -0.5096    0.0000   -0.0060   6.75   1000   750   2.5   3.2   -0.91408   -0.18228   0.7341   0.3761
    2.000   -0.02806   0.0029   -0.20666   -0.76260   0.2529   7.5   -0.5096    0.0000   -0.0090   6.75   1000   750   2.5   3.2   -0.91007   -0.17336   0.7333   0.3734
    2.200   -0.15016   0.0029   -0.21319   -0.75952   0.2529   7.5   -0.5096    0.0000   -0.0141   6.75   1000   750   2.5   3.2   -0.89376   -0.15463   0.7277   0.3794
    2.400   -0.26608   0.0029   -0.21960   -0.75011   0.2529   7.5   -0.5096    0.0000   -0.0284   6.75   1000   750   2.5   3.2   -0.87052   -0.13181   0.7199   0.4025
    2.600   -0.39025   0.0029   -0.22330   -0.74326   0.2529   7.5   -0.5096    0.0000   -0.0408   6.75   1000   750   2.5   3.2   -0.85889   -0.14066   0.7171   0.4049
    2.800   -0.49742   0.0029   -0.22716   -0.74185   0.2529   7.5   -0.5096    0.0000   -0.0534   6.75   1000   750   2.5   3.2   -0.86106   -0.13882   0.7175   0.4090
    3.000   -0.64241   0.0029   -0.23038   -0.73634   0.2529   7.5   -0.5096    0.0000   -0.0683   6.75   1000   750   2.5   3.2   -0.85793   -0.13336   0.7051   0.4115
    3.200   -0.76670   0.0029   -0.23049   -0.72149   0.2529   7.5   -0.5096    0.0000   -0.0780   6.75   1000   750   2.5   3.2   -0.82094   -0.13770   0.6876   0.4280
    3.400   -0.80566   0.0029   -0.23726   -0.72539   0.2529   7.5   -0.5096    0.0000   -0.0943   6.75   1000   750   2.5   3.2   -0.84449   -0.15337   0.6750   0.4029
    3.600   -0.94500   0.0029   -0.24437   -0.70115   0.2529   7.5   -0.5096    0.0000   -0.1278   6.75   1000   750   2.5   3.2   -0.83216   -0.10884   0.6571   0.4252
    3.800   -0.98457   0.0029   -0.24930   -0.69696   0.2529   7.5   -0.5096    0.0000   -0.1744   6.75   1000   750   2.5   3.2   -0.79216   -0.08884   0.6438   0.4243
    4.000   -0.93329   0.0029   -0.25756   -0.71210   0.2529   7.5   -0.5096    0.0000   -0.2231   6.75   1000   750   2.5   3.2   -0.75645   -0.07749   0.6241   0.3659
    """)


class AkkarEtAlRhyp2014(AkkarEtAlRjb2014):
    """
    Implements GMPE developed by S. Akkar, M. A. Sandikkaya, and J. J. Bommer
    as published in "Empirical Ground-Motion Models for Point- and Extended-
    Source Crustal Earthquake Scenarios in Europe and the Middle East",
    Bullettin of Earthquake Engineering (2014).

    The class implements the equations for hypocentral distance and based on
    manuscript provided by the original authors.
    """
    REQUIRES_DISTANCES = set(('rhypo', ))

    def _compute_logarithmic_distance_term(self, C, mag, dists):
        """
        Compute and return fourth term in equations (2a)
        and (2b), page 20.
        """
        return (
            (C['a4'] + C['a5'] * (mag - self.c1)) *
            np.log(np.sqrt(dists.rhypo ** 2 + C['a6'] ** 2))
        )

    COEFFS = CoeffsTable(sa_damping=5, table="""\
      IMT         a1       a2         a3         a4       a5    a6        a7        a8        a9     c1   Vcon  Vref     c     n         b1         b2    sigma      tau
      pga    3.26685   0.0029   -0.04846   -1.47905   0.2529   7.5   -0.5096   -0.1091    0.0937   6.75   1000   750   2.5   3.2   -0.41997   -0.28846   0.6475   0.3472
      pgv    6.72743   0.0029   -0.11474   -1.17694   0.2529   7.5   -0.5096   -0.0616    0.0630   6.75   1000   750   2.5   3.2   -0.72057   -0.19688   0.6280   0.3312
    0.010    3.28656   0.0029   -0.04784   -1.48197   0.2529   7.5   -0.5096   -0.1115    0.0953   6.75   1000   750   2.5   3.2   -0.41729   -0.28685   0.6492   0.3481
    0.020    3.38936   0.0029   -0.04796   -1.50214   0.2529   7.5   -0.5096   -0.1040    0.1029   6.75   1000   750   2.5   3.2   -0.39998   -0.28241   0.6543   0.3508
    0.030    3.53155   0.0029   -0.04537   -1.52781   0.2529   7.5   -0.5096   -0.0973    0.1148   6.75   1000   750   2.5   3.2   -0.34799   -0.26842   0.6685   0.3526
    0.040    3.68895   0.0029   -0.03991   -1.55693   0.2529   7.5   -0.5096   -0.0884    0.1073   6.75   1000   750   2.5   3.2   -0.27572   -0.24759   0.6816   0.3513
    0.050    3.86581   0.0029   -0.03490   -1.58672   0.2529   7.5   -0.5096   -0.0853    0.1052   6.75   1000   750   2.5   3.2   -0.21231   -0.22385   0.6899   0.3659
    0.075    4.18224   0.0029   -0.02826   -1.62527   0.2529   7.5   -0.5096   -0.0779    0.0837   6.75   1000   750   2.5   3.2   -0.14427   -0.17525   0.6881   0.3942
    0.100    4.43750   0.0029   -0.03256   -1.65601   0.2529   7.5   -0.5096   -0.0749    0.0761   6.75   1000   750   2.5   3.2   -0.27064   -0.29293   0.6936   0.4122
    0.110    4.48828   0.0029   -0.03407   -1.65903   0.2529   7.5   -0.5096   -0.0704    0.0707   6.75   1000   750   2.5   3.2   -0.31025   -0.31837   0.6965   0.4065
    0.120    4.51414   0.0029   -0.03635   -1.65470   0.2529   7.5   -0.5096   -0.0604    0.0653   6.75   1000   750   2.5   3.2   -0.34796   -0.33860   0.7022   0.3964
    0.130    4.53290   0.0029   -0.03929   -1.64994   0.2529   7.5   -0.5096   -0.0490    0.0617   6.75   1000   750   2.5   3.2   -0.39668   -0.36646   0.7043   0.3937
    0.140    4.53834   0.0029   -0.04200   -1.64398   0.2529   7.5   -0.5096   -0.0377    0.0581   6.75   1000   750   2.5   3.2   -0.43996   -0.38417   0.7071   0.3853
    0.150    4.52949   0.0029   -0.04509   -1.63467   0.2529   7.5   -0.5096   -0.0265    0.0545   6.75   1000   750   2.5   3.2   -0.48313   -0.39551   0.7048   0.3779
    0.160    4.47016   0.0029   -0.04701   -1.61626   0.2529   7.5   -0.5096   -0.0194    0.0509   6.75   1000   750   2.5   3.2   -0.52431   -0.40869   0.7032   0.3851
    0.170    4.40011   0.0029   -0.04932   -1.59485   0.2529   7.5   -0.5096   -0.0125    0.0507   6.75   1000   750   2.5   3.2   -0.55680   -0.41528   0.7011   0.3900
    0.180    4.33238   0.0029   -0.05181   -1.57545   0.2529   7.5   -0.5096   -0.0056    0.0502   6.75   1000   750   2.5   3.2   -0.58922   -0.42717   0.6992   0.3889
    0.190    4.26395   0.0029   -0.05442   -1.55685   0.2529   7.5   -0.5096    0.0000    0.0497   6.75   1000   750   2.5   3.2   -0.62635   -0.44130   0.6947   0.3903
    0.200    4.17750   0.0029   -0.05565   -1.53574   0.2529   7.5   -0.5096    0.0000    0.0493   6.75   1000   750   2.5   3.2   -0.65315   -0.44644   0.6954   0.3848
    0.220    4.03111   0.0029   -0.05817   -1.50045   0.2529   7.5   -0.5096    0.0000    0.0488   6.75   1000   750   2.5   3.2   -0.68711   -0.44872   0.6925   0.3891
    0.240    3.90131   0.0029   -0.06152   -1.46889   0.2529   7.5   -0.5096    0.0000    0.0483   6.75   1000   750   2.5   3.2   -0.72744   -0.46341   0.6973   0.3839
    0.260    3.82611   0.0029   -0.06706   -1.44738   0.2529   7.5   -0.5096    0.0000    0.0478   6.75   1000   750   2.5   3.2   -0.77335   -0.48705   0.6973   0.3839
    0.280    3.69780   0.0029   -0.07060   -1.41925   0.2529   7.5   -0.5096    0.0000    0.0474   6.75   1000   750   2.5   3.2   -0.80508   -0.47334   0.6914   0.3865
    0.300    3.57698   0.0029   -0.07490   -1.38832   0.2529   7.5   -0.5096    0.0000    0.0469   6.75   1000   750   2.5   3.2   -0.82609   -0.45730   0.6934   0.3896
    0.320    3.40759   0.0029   -0.07756   -1.34898   0.2529   7.5   -0.5096    0.0000    0.0464   6.75   1000   750   2.5   3.2   -0.84080   -0.44267   0.6992   0.3908
    0.340    3.27580   0.0029   -0.08183   -1.31609   0.2529   7.5   -0.5096    0.0000    0.0459   6.75   1000   750   2.5   3.2   -0.86251   -0.43888   0.6990   0.3888
    0.360    3.19725   0.0029   -0.08602   -1.29558   0.2529   7.5   -0.5096    0.0000    0.0459   6.75   1000   750   2.5   3.2   -0.87479   -0.43820   0.7006   0.3916
    0.380    3.11035   0.0029   -0.08937   -1.27591   0.2529   7.5   -0.5096    0.0000    0.0429   6.75   1000   750   2.5   3.2   -0.88522   -0.43678   0.7036   0.3913
    0.400    3.03752   0.0029   -0.09243   -1.26045   0.2529   7.5   -0.5096    0.0000    0.0400   6.75   1000   750   2.5   3.2   -0.89517   -0.43008   0.7037   0.3894
    0.420    2.97485   0.0029   -0.09556   -1.24891   0.2529   7.5   -0.5096    0.0000    0.0374   6.75   1000   750   2.5   3.2   -0.90875   -0.42190   0.7023   0.3847
    0.440    2.90617   0.0029   -0.09822   -1.23700   0.2529   7.5   -0.5096    0.0000    0.0349   6.75   1000   750   2.5   3.2   -0.91922   -0.40903   0.6956   0.3908
    0.460    2.85484   0.0029   -0.10132   -1.22822   0.2529   7.5   -0.5096    0.0000    0.0323   6.75   1000   750   2.5   3.2   -0.92670   -0.39442   0.6893   0.3986
    0.480    2.81720   0.0029   -0.10560   -1.21874   0.2529   7.5   -0.5096    0.0000    0.0297   6.75   1000   750   2.5   3.2   -0.93720   -0.38462   0.6852   0.4017
    0.500    2.77997   0.0029   -0.10964   -1.20953   0.2529   7.5   -0.5096    0.0000    0.0271   6.75   1000   750   2.5   3.2   -0.94614   -0.37408   0.6821   0.4017
    0.550    2.62299   0.0029   -0.11701   -1.18010   0.2529   7.5   -0.5096    0.0000    0.0245   6.75   1000   750   2.5   3.2   -0.96564   -0.35582   0.6866   0.4044
    0.600    2.42234   0.0029   -0.12106   -1.14424   0.2529   7.5   -0.5096    0.0000    0.0219   6.75   1000   750   2.5   3.2   -0.98499   -0.34053   0.6926   0.4005
    0.650    2.22770   0.0029   -0.12555   -1.10853   0.2529   7.5   -0.5096    0.0000    0.0193   6.75   1000   750   2.5   3.2   -0.99733   -0.30949   0.6949   0.3981
    0.700    2.08102   0.0029   -0.13074   -1.08192   0.2529   7.5   -0.5096    0.0000    0.0167   6.75   1000   750   2.5   3.2   -1.00469   -0.28772   0.6993   0.3967
    0.750    1.91625   0.0029   -0.13547   -1.05027   0.2529   7.5   -0.5096    0.0000    0.0141   6.75   1000   750   2.5   3.2   -1.00786   -0.28957   0.7028   0.3890
    0.800    1.81167   0.0029   -0.13856   -1.03514   0.2529   7.5   -0.5096    0.0000    0.0115   6.75   1000   750   2.5   3.2   -1.00606   -0.28555   0.6981   0.3824
    0.850    1.71853   0.0029   -0.14294   -1.02010   0.2529   7.5   -0.5096    0.0000    0.0089   6.75   1000   750   2.5   3.2   -1.01093   -0.28364   0.6959   0.3831
    0.900    1.60822   0.0029   -0.14669   -1.00315   0.2529   7.5   -0.5096    0.0000    0.0062   6.75   1000   750   2.5   3.2   -1.01576   -0.28037   0.6983   0.3825
    0.950    1.51532   0.0029   -0.15056   -0.98859   0.2529   7.5   -0.5096    0.0000    0.0016   6.75   1000   750   2.5   3.2   -1.01353   -0.28390   0.7006   0.3797
    1.000    1.43982   0.0029   -0.15427   -0.97812   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -1.01331   -0.28702   0.7022   0.3826
    1.100    1.26728   0.0029   -0.16107   -0.95163   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -1.01240   -0.27669   0.7137   0.3721
    1.200    1.11475   0.0029   -0.16630   -0.93048   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -1.00489   -0.27538   0.7224   0.3723
    1.300    0.95965   0.0029   -0.17170   -0.90604   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -0.98876   -0.25008   0.7226   0.3746
    1.400    0.85203   0.0029   -0.17699   -0.89379   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -0.97760   -0.23508   0.7349   0.3697
    1.500    0.83007   0.0029   -0.18248   -0.90319   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -0.98071   -0.24695   0.7378   0.3758
    1.600    0.74487   0.0029   -0.18787   -0.89323   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -0.96369   -0.22870   0.7406   0.3794
    1.700    0.63568   0.0029   -0.18961   -0.88392   0.2529   7.5   -0.5096    0.0000    0.0000   6.75   1000   750   2.5   3.2   -0.94634   -0.21655   0.7418   0.3686
    1.800    0.56996   0.0029   -0.19551   -0.87459   0.2529   7.5   -0.5096    0.0000   -0.0030   6.75   1000   750   2.5   3.2   -0.93606   -0.20302   0.7431   0.3692
    1.900    0.48500   0.0029   -0.19853   -0.86659   0.2529   7.5   -0.5096    0.0000   -0.0060   6.75   1000   750   2.5   3.2   -0.91408   -0.18228   0.7457   0.3705
    2.000    0.40614   0.0029   -0.20136   -0.86343   0.2529   7.5   -0.5096    0.0000   -0.0090   6.75   1000   750   2.5   3.2   -0.91007   -0.17336   0.7446   0.3676
    2.200    0.28608   0.0029   -0.20791   -0.86086   0.2529   7.5   -0.5096    0.0000   -0.0141   6.75   1000   750   2.5   3.2   -0.89376   -0.15463   0.7391   0.3718
    2.400    0.15432   0.0029   -0.21480   -0.84778   0.2529   7.5   -0.5096    0.0000   -0.0284   6.75   1000   750   2.5   3.2   -0.87052   -0.13181   0.7311   0.3941
    2.600    0.02250   0.0029   -0.21843   -0.83937   0.2529   7.5   -0.5096    0.0000   -0.0408   6.75   1000   750   2.5   3.2   -0.85889   -0.14066   0.7281   0.3967
    2.800   -0.07822   0.0029   -0.22224   -0.83964   0.2529   7.5   -0.5096    0.0000   -0.0534   6.75   1000   750   2.5   3.2   -0.86106   -0.13882   0.7279   0.3987
    3.000   -0.22534   0.0029   -0.22564   -0.83314   0.2529   7.5   -0.5096    0.0000   -0.0683   6.75   1000   750   2.5   3.2   -0.85793   -0.13336   0.7154   0.4019
    3.200   -0.36165   0.0029   -0.22496   -0.81702   0.2529   7.5   -0.5096    0.0000   -0.0780   6.75   1000   750   2.5   3.2   -0.82094   -0.13770   0.6984   0.4113
    3.400   -0.39423   0.0029   -0.23237   -0.82109   0.2529   7.5   -0.5096    0.0000   -0.0943   6.75   1000   750   2.5   3.2   -0.84449   -0.15337   0.6867   0.3800
    3.600   -0.54126   0.0029   -0.24003   -0.79431   0.2529   7.5   -0.5096    0.0000   -0.1278   6.75   1000   750   2.5   3.2   -0.83216   -0.10884   0.6687   0.4009
    3.800   -0.59607   0.0029   -0.24448   -0.78785   0.2529   7.5   -0.5096    0.0000   -0.1744   6.75   1000   750   2.5   3.2   -0.79216   -0.08884   0.6565   0.3952
    4.000   -0.51893   0.0029   -0.25256   -0.80922   0.2529   7.5   -0.5096    0.0000   -0.2231   6.75   1000   750   2.5   3.2   -0.75645   -0.07749   0.6364   0.3318
    """)
