"""
validataclass
Copyright (c) 2021, binary butterfly GmbH and contributors
Use of this source code is governed by an MIT-style license that can be found in the LICENSE file.
"""

import dataclasses
from typing import Dict

from .dataclass_defaults import Default, NoDefault
from .unset_value import UnsetValue

__all__ = [
    'ValidataclassMixin',
]


# Helper function
def _get_dataclass_field_defaults(cls) -> Dict[str, Default]:
    """
    Returns a dictionary containing the defaults (as Default objects) for all fields.

    (Internal helper function.)
    """
    assert dataclasses.is_dataclass(cls)
    return {field.name: field.metadata.get('validator_default', NoDefault) for field in dataclasses.fields(cls)}


class ValidataclassMixin:
    """
    Mixin class for validataclasses that provides some commonly used methods like `to_dict()` and `create_with_defaults()`.

    Example:

    ```
    @validataclass
    class ExampleClass(ValidataclassMixin):
        pass
    ```
    """

    def to_dict(self, *, keep_unset_values: bool = False) -> dict:
        """
        Returns the data of the object as a dictionary (recursively resolving inner dataclasses as well).

        Filters out all fields with `UnsetValue`, unless the optional parameter `keep_unset_values` is True.

        Parameters:
            keep_unset_values: If true, fields with value `UnsetValue` are NOT removed from the dictionary (default: False)
        """
        data = dataclasses.asdict(self)  # noqa

        # Filter out all UnsetValues (unless said otherwise)
        if not keep_unset_values:
            data = {key: value for key, value in data.items() if value is not UnsetValue}

        return data

    @classmethod
    def create_with_defaults(cls, **kwargs):
        """
        Creates an object of the dataclass with default values (unless overwritten by keyword arguments).

        This class method is an alternative to the regular constructor generated by the dataclass decorator, which currently does
        not support the custom default system, thus requiring to specify all fields including the optional ones.
        """
        # Collect default values for all optional fields
        values = {
            field_name: field_default.get_value()
            for field_name, field_default in _get_dataclass_field_defaults(cls).items()
            if isinstance(field_default, Default) and field_default is not NoDefault
        }

        # Set fields according to arguments
        values.update(kwargs)

        # Create object
        return cls(**values)  # noqa
