# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['vial']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pyvial',
    'version': '0.15.0',
    'description': 'A micro web framework for AWS Lambda.',
    'long_description': '# Vial\n<p>\n    <a href="https://github.com/pre-commit/pre-commit">\n        <img src="https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white" alt="Pre-Commit Enabled" style="max-width:100%;">\n    </a>\n</p>\n\nVial is an unopinionated micro web framework for AWS Lambda. It\'s main purpose is to provide an easy to use\ninterface while also making the core functionality of the framework as modular as possible.\n\n## Installation\nTo add vial to your project, run the following command:\n```\npoetry add pyvial\n```\n\n## Usage\n### Entry Point\nThe main entry point of the application is always the `Vial#__call__` function. When deploying to AWS Lambda,\nthe Lambda handler should point to the `Vial` object in whichever file it\'s defined in. As an example:\n```\nfrom vial.app import Vial\n\napp = Vial(__name__)\n```\nIf this code snippet is defined in an `app.py` file, the handler would be `app.app`.\n\n### Basic API\n```\nfrom vial.app import Vial\n\napp = Vial(__name__)\n\n\n@app.get("/hello-world")\ndef hello_world() -> dict[str, str]:\n    return {"hello": "world"}\n```\nBasic `serverless.yml` file to deploy the project with the serverless framework:\n```\nservice: my-function\nprovider:\n  name: aws\n  runtime: python3.9\n  memorySize: 128\n  region: us-west-2\n\npackage:\n  patterns:\n    - app.py\n\nfunctions:\n  api:\n    handler: app.app\n    events:\n      - http: get /hello-world\n\ncustom:\n  pythonRequirements:\n    usePoetry: true\n\nplugins:\n  - serverless-python-requirements\n```\nYou can now deploy the project with `serverless deploy`.\n\n### Current Request\nThe current request is tracked within a contextual object that wraps the lambda request. It can be accessed like so:\n```\nfrom vial import request\nfrom vial.app import Vial\nfrom vial.types import Request\n\napp = Vial(__name__)\n\n\n@app.get("/hello-world")\ndef hello_world() -> dict[str, list[str]]:\n    request: Request = request.get()\n    query_params = request.query_parameters\n    if not query_params:\n        raise ValueError("Must provide at least one query parameter")\n    return dict(query_params)\n```\nThe `request.get()` function is only available during a lambda request and will raise an error if called outside of one.\n\n### Path Parameters\nYou can define path parameters like this:\n```\n@app.get("/users/{user_id}")\ndef get_user(user_id: str) -> User:\n    return user_service.get(user_id)\n```\n\nVial supports some path parameter parsing as part of the invocation process. For example when using a UUID\nas a path parameter, Vial can convert it from a string to a UUID automatically:\n```\nfrom uuid import UUID\n\n@app.get("/users/{user_id:uuid}")\ndef get_user(user_id: UUID) -> User:\n    return user_service.get(user_id)\n```\n\nThe following parsers are supported by default:\n\n| Parser        | Type              |\n| ------------- | ----------------- |\n| `str`         | `str`             |\n| `bool`        | `bool`            |\n| `int`         | `int`             |\n| `float`       | `float`           |\n| `decimal`     | `decimal.Decimal` |\n| `uuid`        | `uuid.UUID`       |\n\nYou can register your own parser like this:\n```\n@app.parser("list")\ndef list_parser(value: str) -> list[str]:\n    return [value]\n\n\n@app.get("/users/{user_id:list}")\ndef get_user(user_ids: list[str]) -> list[User]:\n    return [user_service.get(user_id) for user_id in user_ids]\n```\nAs parsers are bound directly to the registered route function, they have to be defined before the route\nfunction that uses one is registered.\n\n## Resources\nAs your application grows, you may want to split certain functionality amongst resources and files, similar to\nblueprints of other popular frameworks like Flask.\n\nYou can define a resource like this:\n```\nfrom dataclasses import dataclass\n\nfrom vial.app import Resource, Vial\n\nstores_app = Resource(__name__)\n\n\n@dataclass\nclass Store:\n    store_id: str\n\n\n@stores_app.get("/stores/{store_id}")\ndef get_store(store_id: str) -> Store:\n    return Store(store_id)\n\n\napp = Vial(__name__)\n\napp.register_resource(stores_app)\n```\nA test case with this example is available in `tests/samples/test_with_resources.py`.\n\n## Middleware\nYou can register middleware functions to be executed before / after route invocations. All middleware is scoped to\nwhere it\'s registered. A middleware function registered with the `Vial` instance is scoped to all routes within\nthe application, but a function registered with a `Resource` instance will only be invoked for routes defined in\nthat specific resource.\n\nBelow is an example of registering a middleware to log route invocation:\n```\nfrom __future__ import annotations\n\nfrom vial import request\nfrom vial.app import Vial\nfrom vial.middleware import CallChain\nfrom vial.types import Request, Response\n\napp = Vial(__name__)\n\n\n@app.middleware\ndef log_events(event: Request, chain: CallChain) -> Response:\n    app.logger.info("Began execution of %s", event.context)\n    event.headers["custom-injected-header"] = "hello there"\n    try:\n        return chain(event)\n    finally:\n        app.logger.info("Completed execution of %s", event.context)\n\n\n@app.get("/hello-world")\ndef hello_world() -> dict[str, str | list[str]]:\n    return {"hello": "world", **request.get().headers}\n```\nA test case with this example is available in `tests/samples/test_with_middleware.py`.\n\n\n## Json Encoding\nYou can customize how Vial serializes / deserializes JSON objects by passing a custom encoder. The below\nexample shows how to substitute the native JSON module with another library like `simplejson`:\n```\nfrom decimal import Decimal\nfrom typing import Any\n\nimport simplejson\n\nfrom vial.app import Vial\nfrom vial.json import Json\n\n\nclass SimpleJson(Json):\n    @staticmethod\n    def dumps(value: Any) -> str:\n        return simplejson.dumps(value)\n\n    @staticmethod\n    def loads(value: str) -> Any:\n        return simplejson.loads(value)\n\n\nclass SimpleJsonVial(Vial):\n    json_class = SimpleJson\n\n\napp = SimpleJsonVial(__name__)\n\n\n@app.get("/prices")\ndef get_prices() -> dict[str, Decimal]:\n    # Decimal is not supported natively by the json module, but is by simplejson.\n    return {"bread": Decimal("42.24"), "cheese": Decimal("129.34")}\n```\nA test case with this example is available in `tests/samples/test_with_json_encoding.py`.\n\n## Testing\nThe `vial.gateway.Gateway` class provides functionality to interact with the Vial application locally,\nwithout deploying to AWS Lambda. It can be constructed using the original `Vial` application instance,\nexposing the application endpoints with basic URL path matching.\n\nHere is an example test case using `pytest`:\n```\nfrom http import HTTPStatus\n\nimport pytest\n\nfrom vial import request\nfrom vial.app import Vial\nfrom vial.errors import BadRequestError\nfrom vial.gateway import Gateway\n\napp = Vial(__name__)\n\n\n@app.get("/health")\ndef health() -> dict[str, str]:\n    return {"status": "OK"}\n\n\n@app.post("/stores/{store_id}")\ndef create_store(store_id: str) -> dict[str, str]:\n    if not (body := request.get().body):\n        raise BadRequestError("Bad request")\n    return {"store_id": store_id, **app.json.loads(body)}\n\n\n@pytest.fixture(name="gateway")\ndef gateway_fixture() -> Gateway:\n    return Gateway(app)\n\n\ndef test_health(gateway: Gateway) -> None:\n    response = gateway.get("/health")\n    assert response.status == HTTPStatus.OK\n    assert response.body == {"status": "OK"}\n\n\ndef test_create_store(gateway: Gateway) -> None:\n    response = gateway.post("/stores/my-cool-store", app.json.dumps({"store_name": "My very cool store"}))\n    assert response.status == HTTPStatus.OK\n    assert response.body == {"store_id": "my-cool-store", "store_name": "My very cool store"}\n```\nThis code is also available in `tests/samples/test_with_gateway.py`.\n',
    'author': 'Michael Dimchuk',
    'author_email': 'michaeldimchuk@gmail.com',
    'maintainer': 'Michael Dimchuk',
    'maintainer_email': 'michaeldimchuk@gmail.com',
    'url': 'https://github.com/michaeldimchuk/pyvial',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
