"""Tests for core paxplot functions"""

import unittest
import core


class PaxplotLib(unittest.TestCase):
    def test_parallel_blank(self):
        """
        Test of blank figure
        """
        # Run
        paxfig = core.pax_parallel(n_axes=4)

        # Test
        self.assertEqual(paxfig.axes.__len__(), 4)
        self.assertEqual(
            paxfig.subplotpars.__getattribute__('wspace'),
            0.0
        )
        self.assertEqual(
            paxfig.subplotpars.__getattribute__('hspace'),
            0.0
        )
        for ax in paxfig.axes:
            self.assertEqual(
                ax.spines['top'].get_visible(),
                False
            ),
            self.assertEqual(
                ax.spines['bottom'].get_visible(),
                False
            )
            self.assertEqual(
                ax.spines['right'].get_visible(),
                False
            )
            self.assertEqual(ax.get_ylim(), (0.0, 1.0))
            self.assertEqual(ax.get_xlim(), (0.0, 1.0))
            self.assertEqual(ax.get_xticks(), [0])
            self.assertEqual(
                ax.xaxis.majorTicks[0].tick1line.get_markersize(),
                0.0
            )

        # Last axis
        position = paxfig.axes[-1].get_position()
        width = position.x1 - position.x0
        self.assertEqual(width, 0.0)

    def test_parallel_basic(self):
        """
        Basic test of parallel functionality
        """
        # Setup
        data = [
            [0.0, 0.0],
            [1.0, 1.0],
            [2.0, 2.0]
        ]

        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)

        # Test attribute data
        self.assertEqual(
            paxfig.axes[0].paxfig_lim, (0.0, 2.0)
        )
        self.assertEqual(
            paxfig.axes[1].paxfig_lim, (0.0, 2.0)
        )

        # Test plotted data
        self.assertTrue(
            (paxfig.axes[0].lines[0].get_ydata() == [0.0, 0.0]).all()
        )
        self.assertTrue(
            (paxfig.axes[0].lines[1].get_ydata() == [0.5, 0.5]).all()
        )
        self.assertTrue(
            (paxfig.axes[0].lines[2].get_ydata() == [1.0, 1.0]).all()
        )

        # Test ticks
        self.assertEqual(paxfig.axes[0].get_yticklabels()[0].get_text(), '0.0')
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[0].get_position()[1], 0.0
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[-1].get_text(), '2.0'
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[-1].get_position()[1], 1.0
        )

    def test_parallel_limits(self):
        """
        Testing plotting with custom limits
        """
        # Setup
        data = [
            [0.0, 0.0, 0.0],
            [1.0, 1.0, 1.0],
            [2.0, 2.0, 2.0]
        ]

        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)
        paxfig.set_lim(ax_idx=0, bottom=-1.0, top=3)
        paxfig.set_lim(ax_idx=1, bottom=-1.0, top=3)
        paxfig.set_lim(ax_idx=2, bottom=0.0, top=1.5)

        # Test plotted data
        self.assertEqual(paxfig.axes[0].lines[0].get_ydata()[0], 0.25)
        self.assertEqual(paxfig.axes[0].lines[1].get_ydata()[0], 0.50)
        self.assertEqual(paxfig.axes[0].lines[2].get_ydata()[0], 0.75)
        self.assertEqual(paxfig.axes[1].lines[0].get_ydata()[1], 0.0, 1)
        self.assertAlmostEqual(
            paxfig.axes[1].lines[1].get_ydata()[1], 0.6666, 2
        )
        self.assertAlmostEqual(
            paxfig.axes[1].lines[2].get_ydata()[1], 1.3333, 2
        )

        # Test ticks
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[0].get_text(), '-1.0'
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[0].get_position()[1], 0.0
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[-1].get_text(), '3.0'
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[-1].get_position()[1], 1.0
        )
        self.assertEqual(
            paxfig.axes[1].get_yticklabels()[0].get_text(), '-1.0'
        )
        self.assertEqual(
            paxfig.axes[1].get_yticklabels()[0].get_position()[1], 0.0
        )
        self.assertEqual(
            paxfig.axes[1].get_yticklabels()[-1].get_text(), '3.0'
        )
        self.assertEqual(
            paxfig.axes[1].get_yticklabels()[-1].get_position()[1], 1.0
        )
        self.assertEqual(
            paxfig.axes[2].get_yticklabels()[0].get_text(), '0.0'
        )
        self.assertEqual(
            paxfig.axes[2].get_yticklabels()[0].get_position()[1], 0.0
        )
        self.assertEqual(
            paxfig.axes[2].get_yticklabels()[-1].get_text(), '1.5'
        )
        self.assertEqual(
            paxfig.axes[2].get_yticklabels()[-1].get_position()[1], 1.0
        )

    def test_parallel_ticks(self):
        """
        Testing plotting with modified ticks
        """
        # Setup
        data = [
            [0.0, 0.0, 2.0, 0.0],
            [1.0, 1.0, 1.0, 1.0],
            [3.0, 2.0, 0.0, 3.0],
        ]

        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)
        tick_lists = [
            [0.0, 1.0, 3.0],
            [0.0, 1.0, 2.5],
            [-1.0, 1.0, 2.0],
            [0.0, 1.0, 3.5]
        ]
        for i, ticks in enumerate(tick_lists):
            paxfig.set_ticks(ax_idx=i, ticks=ticks)

        # Test tick labels
        for i, ax in enumerate(paxfig.axes):
            for j, label in enumerate(ax.get_yticklabels()):
                self.assertEqual(
                    label.get_text(),
                    str(tick_lists[i][j])
                )

        # Test tick positioning (random)
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[2].get_position()[1], 1.0
        )
        self.assertEqual(
            paxfig.axes[1].get_yticklabels()[1].get_position()[1], 0.4
        )
        self.assertAlmostEqual(
            paxfig.axes[2].get_yticklabels()[1].get_position()[1],
            0.6666,
            2
        )
        self.assertAlmostEqual(
            paxfig.axes[3].get_yticklabels()[1].get_position()[1],
            0.28571,
            2
        )

    def test_parallel_even_ticks(self):
        """
        Test even tick setting
        """
        # Setup
        data = [
            [0.0, 0.0, 2.0, 0.0],
            [1.0, 1.0, 1.0, 1.0],
            [3.0, 2.0, 0.0, 3.0],
        ]

        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)
        paxfig.set_even_ticks(
            ax_idx=0,
            n_ticks=30,
        )

        # Test tick position
        for i in range(31):
            self.assertAlmostEqual(
                paxfig.axes[0].get_yticklabels()[i].get_position()[1],
                i*0.03333,
                3
            )

        # Test tick labels
        for i in range(31):
            self.assertEqual(
                paxfig.axes[0].get_yticklabels()[i].get_text(),
                str(round(i*0.1, 2)),
            )

    def test_parallel_custom_ticks(self):
        """
        Testing plotting with custom ticks
        """
        # Setup
        data = [
            [0.0, 0.0],
            [1.0, 1.0],
            [2.0, 2.0]
        ]

        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)
        paxfig.set_ticks(
            ax_idx=0,
            ticks=[0.0, 1.0, 2.0],
            labels=['my heart', 'is the \ncode to', '1612']
        )

        # Test ticks
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[0].get_text(), 'my heart'
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[2].get_text(), '1612'
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[0].get_position()[1], 0.0
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[2].get_position()[1], 1.0
        )

    def test_parallel_label(self):
        """
        Testing setting labels
        """
        # Setup
        data = [
            [0.0, 0.0],
            [1.0, 1.0],
            [2.0, 2.0]
        ]

        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)
        paxfig.set_label(
            ax_idx=0,
            label='Em7-Dm7-G7'
        )
        paxfig.set_label(
            ax_idx=1,
            label='Cmaj7-B7'
        )

        # Test
        self.assertEqual(
            paxfig.axes[0].get_xticklabels()[0].get_text(), 'Em7-Dm7-G7'
        )
        self.assertEqual(
            paxfig.axes[1].get_xticklabels()[0].get_text(), 'Cmaj7-B7'
        )

    def test_parallel_invert(self):
        """
        Test inverting axis
        """
        # Setup
        data = [
            [0.0, 0.0, 0.0],
            [1.0, 1.0, 1.0],
            [2.0, 2.0, 2.0]
        ]

        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)
        paxfig.invert_axis(ax_idx=0)
        paxfig.invert_axis(ax_idx=2)

        # Test plotted data
        self.assertEqual(paxfig.axes[0].lines[0].get_ydata()[0], 1.0)
        self.assertEqual(paxfig.axes[0].lines[2].get_ydata()[0], 0.0)
        self.assertEqual(paxfig.axes[1].lines[0].get_ydata()[1], 1.0)
        self.assertEqual(paxfig.axes[1].lines[2].get_ydata()[1], 0.0)

        # Test ticks
        self.assertEqual(paxfig.axes[0].get_yticklabels()[0].get_text(), '0.0')
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[-1].get_text(), '2.0'
        )
        self.assertEqual(paxfig.axes[2].get_yticklabels()[0].get_text(), '0.0')
        self.assertEqual(
            paxfig.axes[2].get_yticklabels()[-1].get_text(), '2.0'
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[0].get_position()[1], 1.0
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[-1].get_position()[1], 0.0
        )
        self.assertEqual(
            paxfig.axes[2].get_yticklabels()[0].get_position()[1], 1.0
        )
        self.assertEqual(
            paxfig.axes[2].get_yticklabels()[-1].get_position()[1], 0.0
        )

    def test_parallel_invert_middle(self):
        """
        Test inverting middle axes
        """
        # Setup
        data = [
            [0.0, 0.0, 2.0],
            [1.0, 1.0, 1.0],
            [3.0, 2.0, 0.0],
        ]

        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)
        paxfig.invert_axis(1)

        # Test plotted data
        self.assertEqual(paxfig.axes[0].lines[0].get_ydata()[0], 0.0)
        self.assertEqual(paxfig.axes[0].lines[2].get_ydata()[0], 1.0)
        self.assertEqual(paxfig.axes[1].lines[0].get_ydata()[1], 1.0)
        self.assertEqual(paxfig.axes[1].lines[2].get_ydata()[1], 0.0)

        # Test ticks
        self.assertEqual(paxfig.axes[0].get_yticklabels()[0].get_text(), '0.0')
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[-1].get_text(), '3.0'
        )
        self.assertEqual(paxfig.axes[1].get_yticklabels()[0].get_text(), '0.0')
        self.assertEqual(
            paxfig.axes[1].get_yticklabels()[-1].get_text(), '2.0'
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[0].get_position()[1], 0.0
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[-1].get_position()[1], 1.0
        )
        self.assertEqual(
            paxfig.axes[1].get_yticklabels()[0].get_position()[0], 0.0
        )
        self.assertEqual(
            paxfig.axes[2].get_yticklabels()[-1].get_position()[1], 1.0
        )

    def test_parallel_legend(self):
        """
        Test creating legend
        """
        # Setup
        data = [
            [0.0, 0.0, 2.0],
            [1.0, 1.0, 1.0],
            [2.0, 2.0, 0.0],
        ]

        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)
        paxfig.add_legend(labels=['A', 'B', 'C'])

        # Legend tests
        legend_text = paxfig.axes[-1].get_legend().get_texts()
        self.assertEqual(legend_text[0].get_text(), 'A')
        self.assertEqual(legend_text[1].get_text(), 'B')
        self.assertEqual(legend_text[2].get_text(), 'C')

        # Gridspec tests
        self.assertEqual(
            paxfig.axes[0].get_gridspec().get_width_ratios(),
            [1.0, 1.0, 0.0, 1.0]
        )

    def test_parallel_colorbar(self):
        """
        Test creating colorbar
        """
        # Setup
        data = [
            [0.0, 0.0, 2.0],
            [1.0, 1.0, 1.0],
            [3.0, 2.0, 0.0],
        ]

        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)
        paxfig.add_colorbar(
            ax_idx=2,
            cmap='viridis',
            colorbar_kwargs={'label': 'test'}
        )

        # Line color tests
        self.assertEqual(
            paxfig.axes[0].lines[0].get_color(),
            '#fde725'
        )
        self.assertEqual(
            paxfig.axes[0].lines[1].get_color(),
            '#21918c'
        )

        # Colorbar tests
        self.assertEqual(
            paxfig.axes[-1].get_label(),
            '<colorbar>'
        )
        self.assertEqual(
            paxfig.axes[-1].get_ylabel(),
            'test'
        )
        self.assertEqual(
            paxfig.axes[-1].get_ylim(),
            (0.0, 2.0)
        )

        # Gridspec tests
        self.assertEqual(
            paxfig.axes[0].get_gridspec().get_width_ratios(),
            [1.0, 1.0, 0.0, 0.5]
        )

    def test_parallel_combo(self):
        """
        Test applying multiple functions. This is more of an
        integration test that can quickly check functionality
        the many methods working together.
        """
        # Setup
        data = [
            [0.0, 0.0, 2.0],
            [1.0, 1.0, 1.0],
            [3.0, 2.0, 0.0],
        ]
        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)
        paxfig.set_lim(0, bottom=0, top=4)
        paxfig.set_lim(1, bottom=0, top=4)
        paxfig.set_lim(2, bottom=0, top=5)
        paxfig.set_ticks(0, [0, 1, 2, 3])
        paxfig.set_ticks(1, [0, 0.5, 2.0], ['a 0', 'b 0.5', 'c 2.0'])
        paxfig.set_ticks(2, [-1.0, 0.0, 0.5, 1.0, 2.0, 5.0])
        paxfig.set_label(0, 'foo')
        paxfig.invert_axis(1)
        paxfig.add_colorbar(0)

        # Y data tests (random)
        self.assertEqual(paxfig.axes[0].lines[1].get_ydata()[0], 0.25)
        self.assertEqual(paxfig.axes[1].lines[2].get_ydata()[0], 0.50)

        # Tick label tests (random)
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[0].get_text(),
            '0'
        )
        self.assertEqual(
            paxfig.axes[1].get_yticklabels()[1].get_text(),
            'b 0.5'
        )

        # Tick positioning tests (random)
        self.assertEqual(
            paxfig.axes[1].get_yticklabels()[-1].get_position()[1],
            0.5
        )
        self.assertEqual(
            paxfig.axes[0].get_yticklabels()[-1].get_position()[1],
            0.75
        )

        # Label test
        self.assertEqual(
            paxfig.axes[0].get_xticklabels()[0].get_text(), 'foo'
        )

        # Colorbar tests
        self.assertEqual(
            paxfig.axes[-1].get_label(),
            '<colorbar>'
        )
        self.assertEqual(
            paxfig.axes[-1].get_ylim(),
            (0.0, 4.0)
        )


class PaxplotException(unittest.TestCase):
    def test_paxfig_creation(self):
        """
        Various ways to fail figure creation
        """
        # Nothing supplied
        with self.assertRaises(TypeError):
            core.pax_parallel()

        # Non-int n_axes
        with self.assertRaises(TypeError):
            core.pax_parallel(n_axes=0.1)

    def test_plot(self):
        """
        Various ways paxfig.plot can fail
        """
        # Too few axes
        with self.assertWarns(Warning):
            paxfig = core.pax_parallel(n_axes=4)
            paxfig.plot(
                [
                    [0.0, 0.0, 2.0],
                    [1.0, 1.0, 1.0],
                ]
            )

        # Too many axes
        with self.assertRaises(ValueError):
            paxfig = core.pax_parallel(n_axes=2)
            paxfig.plot(
                [
                    [0.0, 0.0, 2.0],
                    [1.0, 1.0, 1.0],
                ]
            )

        # Non-plottable data
        with self.assertRaises(TypeError):
            paxfig = core.pax_parallel(n_axes=3)
            paxfig.plot(
                [
                    ['foo', 0.0, 2.0],
                    ['bar', 1.0, 1.0],
                ]
            )

    def test_lim(self):
        """
        Various ways paxfig.set_lim can fail
        """
        # Setup
        data = [
            [0.0, 0.0, 0.0],
            [1.0, 1.0, 1.0],
            [2.0, 2.0, 2.0]
        ]
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)

        # Requesting axis that doesn't exist
        with self.assertRaises(IndexError):
            paxfig.set_lim(ax_idx=4, bottom=-1.0, top=3)

        # Bottom larger than top
        with self.assertRaises(ValueError):
            paxfig.set_lim(ax_idx=0, bottom=3, top=1)

        # Non integer value for ax_idx
        with self.assertRaises(TypeError):
            paxfig.set_lim(ax_idx='foo', bottom=0, top=1)

        # Non numeric value for bottom
        with self.assertRaises(TypeError):
            paxfig.set_lim(ax_idx=0, bottom='foo', top=1)

        # Non numeric value for top
        with self.assertRaises(TypeError):
            paxfig.set_lim(ax_idx=0, bottom=0, top='foo')

    def test_ticks(self):
        """
        Various ways paxfig.set_ticks can fail
        """
        # Setup
        data = [
            [0.0, 0.0, 2.0, 0.0],
            [1.0, 1.0, 1.0, 1.0],
            [3.0, 2.0, 0.0, 3.0],
        ]
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)

        # Requesting axis that doesn't exist
        with self.assertRaises(IndexError):
            paxfig.set_ticks(ax_idx=5, ticks=[0, 1, 2])

        # Non integer value for ax_idx
        with self.assertRaises(TypeError):
            paxfig.set_ticks(ax_idx='foo', ticks=[0, 1, 2])

        # Ticks non array-like
        with self.assertRaises(TypeError):
            paxfig.set_ticks(ax_idx=0, ticks='foo')

        # Non-numeric ticks
        with self.assertRaises(TypeError):
            paxfig.set_ticks(ax_idx=0, ticks=[1, 2, 'three'])

        # Labels non array-like or different lengths
        with self.assertRaises(ValueError):
            paxfig.set_ticks(ax_idx=0, ticks=[1, 2, 3], labels='A')

    def test_even_ticks(self):
        """
        Various ways paxfig.set_even_ticks can fail
        """
        # Setup
        data = [
            [0.0, 0.0, 2.0, 0.0],
            [1.0, 1.0, 1.0, 1.0],
            [3.0, 2.0, 0.0, 3.0],
        ]

        # Run
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)

        # Requesting axis that doesn't exist
        with self.assertRaises(IndexError):
            paxfig.set_even_ticks(
                ax_idx=5,
                n_ticks=30,
                minimum=0.0,
                maximum=3.0
            )

        # Non integer value for ax_idx
        with self.assertRaises(TypeError):
            paxfig.set_even_ticks(
                ax_idx='foo',
                n_ticks=30,
                minimum=0.0,
                maximum=3.0
            )

        # Non integer value for n_ticks
        with self.assertRaises(TypeError):
            paxfig.set_even_ticks(
                ax_idx=0,
                n_ticks='foo',
                minimum=0.0,
                maximum=3.0
            )

        # Maximum greater than minimum
        with self.assertRaises(ValueError):
            paxfig.set_even_ticks(
                ax_idx=0,
                n_ticks=10,
                minimum=3.0,
                maximum=0.0
            )

    def test_labels(self):
        """
        Various ways paxfig.set_label can fail
        """
        # Setup
        data = [
            [0.0, 0.0],
            [1.0, 1.0],
            [2.0, 2.0]
        ]
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)

        # Requesting axis that doesn't exist
        with self.assertRaises(IndexError):
            paxfig.set_label(ax_idx=3, label='foo')

        # Non integer value for ax_idx
        with self.assertRaises(TypeError):
            paxfig.set_label(ax_idx='foo', label='bar')

    def test_invert(self):
        """
        Various ways paxfig.invert_axis can fail
        """
        # Setup
        data = [
            [0.0, 0.0, 0.0],
            [1.0, 1.0, 1.0],
            [2.0, 2.0, 2.0]
        ]
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)

        # Requesting axis that doesn't exist
        with self.assertRaises(IndexError):
            paxfig.invert_axis(ax_idx=3)

        # Non integer value for ax_idx
        with self.assertRaises(TypeError):
            paxfig.invert_axis(ax_idx='foo')

    def test_legend(self):
        """
        Various ways paxfig.add_legend can fail
        """
        # Setup
        data = [
            [0.0, 0.0, 2.0],
            [1.0, 1.0, 1.0],
            [2.0, 2.0, 0.0],
        ]
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)

        # Not enough labels provided
        with self.assertRaises(IndexError):
            paxfig.add_legend(labels=['A', 'B'])

        # Too many labels provided
        with self.assertWarns(Warning):
            paxfig.add_legend(labels=['A', 'B', 'C', 'D'])

    def test_colorbar(self):
        """
        Various ways paxfig.add_colorbar can fail
        """
        # Setup
        data = [
            [0.0, 0.0, 2.0],
            [1.0, 1.0, 1.0],
            [3.0, 2.0, 0.0],
        ]
        paxfig = core.pax_parallel(n_axes=len(data[0]))
        paxfig.plot(data)

        # Requesting axis that doesn't exist
        with self.assertRaises(IndexError):
            paxfig.add_colorbar(
                ax_idx=4,
                cmap='viridis',
            )

        # Non integer value for ax_idx
        with self.assertRaises(TypeError):
            paxfig.add_colorbar(
                ax_idx='foo',
                cmap='viridis',
            )

        # Colorbar that doesn't exist (default message helpful enough)
        with self.assertRaises(ValueError):
            paxfig.add_colorbar(
                ax_idx=0,
                cmap='foo',
            )


if __name__ == '__main__':
    unittest.main()
