# AUTOGENERATED! DO NOT EDIT! File to edit: model.ipynb (unless otherwise specified).

__all__ = ['Encoder', 'Attention', 'Decoder', 'Translator']

# Cell

import numpy as np
import tensorflow as tf

from tqdm import tqdm

from tensorflow import GradientTape, Variable, matmul, tanh
from tensorflow import nn
from tensorflow.keras.layers import Bidirectional, LSTM, Dense, Embedding
from tensorflow.keras.optimizers import Adagrad
from tensorflow.keras.losses import CategoricalCrossentropy
from tensorflow.python.keras.activations import softmax
from tensorflow.python.keras.models import Model

from .dataset_ted_en_az import get_dataset

# Cell

class Encoder(Model):
    def __init__(self, vocab_size, seq_len, encoder_hidden_size, embedding_size):

        super().__init__()

        self.embedding = Embedding(vocab_size, embedding_size, input_length=seq_len)

        self.recurrent_layers = [
            Bidirectional(LSTM(encoder_hidden_size, return_sequences=True, return_state=True), merge_mode="ave")
            for _ in range(2)
        ]

    def call(self, inputs, **kwargs):

        inputs = self.embedding(inputs)

        for layer in self.recurrent_layers:
            inputs, forward_hidden, _, backward_hidden, _ = layer(inputs)

        return (forward_hidden + backward_hidden) / 2, inputs

# Cell

class Attention(Model):
    def __init__(self, sequence_length, attention_hidden_size, encoder_hidden_size, decoder_hidden_size):

        super().__init__()

        conv_kernel_size = 6

        # TODO: initialize these to some nicer values
        self.V = Variable(np.random.randn(encoder_hidden_size, attention_hidden_size).astype(np.float32))
        self.W = Variable(np.random.randn(decoder_hidden_size, attention_hidden_size).astype(np.float32))
        self.Q = Variable(np.random.randn(conv_kernel_size, 1, attention_hidden_size).astype(np.float32))
        self.U = Variable(np.random.randn(sequence_length, sequence_length).astype(np.float32))
        self.w = Variable(np.random.randn(attention_hidden_size, 1).astype(np.float32))
        self.b = Variable(np.random.randn(1).astype(np.float32))

    def call(self, encoder_outputs, previous_attention, previous_decoder_state):
        """
        :param encoder_outputs: (batch_size, max_seq_len, encoder_hidden_size)
        :param previous_attention: (1, max_seq_len)
        :param previous_decoder_state: (batch_size, decoder_hidden_size)
        :return: aw (1, max_seq_len)
                 context (batch_size, 1, attention_hidden_size)
        """

        Vh = matmul(encoder_outputs, self.V)            # (batch_size, max_seq_len, attention_hidden_size)

        Wst1 = matmul(previous_decoder_state, self.W)   # (batch_size, attention_hidden_size)
        Wst1 = tf.expand_dims(Wst1, axis=1)             # (batch_size, 1, attention_hidden_size)

        F = nn.conv1d(tf.expand_dims(previous_attention, -1),
                      self.Q,
                      stride=1,
                      padding="SAME")                   # (1, attention_hidden_size, conv_filters)
        F = tf.squeeze(F)                               # (conv_filters, attention_hidden_size)
        Uf = tf.expand_dims(matmul(self.U, F), 0)       # (1, max_seq_len, attention_hidden_size)

        etl = matmul(tanh(Wst1 + Vh + Uf + self.b), self.w) # (max_seq_len)

        aw = softmax(tf.squeeze(etl))                   # (max_seq_len)

        # return attention weights and context
        return aw, tf.expand_dims(tf.reduce_sum(matmul(tf.expand_dims(aw, 0), encoder_outputs), axis=1), 1)

# Cell

class Decoder(Model):
    def __init__(self, decoder_hidden_size, vocab_size):

        super().__init__()
        self.lstm = LSTM(decoder_hidden_size, return_sequences=False, return_state=True, stateful=True)
        self.dense = Dense(vocab_size)

    def call(self, previous_word, context):
        """
        :param previous_word: in encoded form - lstm output (batch_size, encoder_hidden_size)
        :param context: (batch_size, 1, encoder_hidden_size)
        :return: (batch_size, vocab_size)
        """
        _, _, lstm_output = self.lstm(tf.concat([context, tf.expand_dims(previous_word, 1)], 1))

        output = self.dense(lstm_output)

        return lstm_output, output

    def reset(self):
        self.lstm.reset_states()

# Cell

class Translator(Model):
    def __init__(self, in_vocab_size, out_vocabulary, seq_len, embedding_size):

        super().__init__()

        self.in_vocab_size = in_vocab_size
        self.seq_len = seq_len
        self.embedding_size = embedding_size
        self.out_vocabulary = out_vocabulary
        self.out_vocab_size = len(self.out_vocabulary)

        self.encoder_hidden_size = 10
        self.decoder_hidden_size = 10
        self.attention_hidden_size = 10

        # components
        self.__encoder = Encoder(in_vocab_size, seq_len, self.encoder_hidden_size, embedding_size)
        self.__encoder.build((None, seq_len))
        self.__attention = Attention(seq_len,
                                     attention_hidden_size=self.attention_hidden_size,
                                     encoder_hidden_size=self.encoder_hidden_size,
                                     decoder_hidden_size=self.decoder_hidden_size)
        # self.__attention.build([(None, seq_len, self.encoder_hidden_size), (None, seq_len), (None, self.decoder_hidden_size)])
        self.__decoder = Decoder(self.decoder_hidden_size, self.out_vocab_size)

    def translate(self, sentences):
        word_probs = self.predict(sentences)
        workd_idxs = tf.math.argmax(word_probs, axis=2)
        return self.idx_to_words(workd_idxs)

    def idx_to_words(self, sentences_idx):

        sentences_words = list()

        for i in range(sentences_idx.shape[1]):
            sentences_words.append([self.out_vocabulary[idx] for idx in sentences_idx[:, i]])

        return sentences_words

    def predict(self, sentences):

        batch_size = sentences.shape[0]

        # encode the sentences
        encoder_state, encoded_sentences = self.__encoder(sentences)

        # initial inference variables
        previous_attention = Variable(np.ones([batch_size, self.seq_len]), dtype=tf.float32)
        previous_decoder_state = encoder_state
        # TODO: add <start> to the vocabulary

        translated_sentences = list()

        # word by word to the translation
        for i in range(self.seq_len):

            # compute attention weights for the current step
            previous_attention, context = self.__attention(encoded_sentences, previous_attention, previous_decoder_state)

            # decode
            previous_decoder_state, previous_translation = self.__decoder(previous_decoder_state, context)

            previous_translation = tf.expand_dims(previous_translation, axis=1)
            # ugly
            if i == 0:
                translated_sentences = previous_translation
            else:
                translated_sentences = tf.concat([translated_sentences, previous_translation], axis=1)

        return translated_sentences

    def fit(self, source_data, target_data, epochs=100, batch_size=32):
        """

        :param source_data: [nr_samples, seq_len]
        :param target_data: [nr_samples, seq_len]
        :param epochs:
        :param batch_size:
        :return:
        """

        train_op = Adagrad(learning_rate=0.1)
        loss_op = CategoricalCrossentropy()

        trainable_variables = [self.__attention.trainable_variables,
                               self.__encoder.trainable_variables,
                               self.__decoder.trainable_variables]

        for epoch in range(epochs):
            for i in tqdm(range(int(len(source_data)/batch_size))):   # TODO: there is some leftover

                x = tf.cast(source_data[i * batch_size:(i + 1) * batch_size], dtype=tf.float32)
                y = target_data[i * batch_size:(i + 1) * batch_size]
                y = tf.one_hot(y, self.out_vocab_size)

                with GradientTape() as tape:

                    predictions = self.predict(x)

                    loss = loss_op(y, predictions)

                    gradients = tape.gradient(loss, trainable_variables)

                for grads, vars in zip(gradients, trainable_variables):
                    train_op.apply_gradients(zip(grads, vars))

                if i % (int(50/batch_size)) == 0:
                    print("loss=" + str(loss))

            print("Epoch " + str(epoch) + ", loss=" + str(loss))
